// =============================================================================
//
//      --- kvi_systray_clock.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviSysTrayClock"

#include <qpainter.h>
#include <qpixmap.h>

#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_string.h"
#include "kvi_systray.h"
#include "kvi_systray_clock.h"

static char *days_of_week[7] =
{
	_i19n_("Sun"), _i19n_("Mon"), _i19n_("Tue"), _i19n_("Wed"),
	_i19n_("Thu"), _i19n_("Fri"), _i19n_("Sat"),
};

static char *months_of_year[12] =
{
	_i19n_("Jan"), _i19n_("Feb"), _i19n_("Mar"),
	_i19n_("Apr"), _i19n_("May"), _i19n_("Jun"),
	_i19n_("Jul"), _i19n_("Aug"), _i19n_("Sep"),
	_i19n_("Oct"), _i19n_("Nov"), _i19n_("Dec")
};

KviSysTrayClock::KviSysTrayClock(KviSysTray *parent)
	: KviSysTrayWidget(parent, __tr("Local clock"), "systray_local_clock")
{
	m_timer = 0;
	setBackgroundMode(NoBackground);
}

KviSysTrayClock::KviSysTrayClock(KviSysTray *parent, const char *tooltip, const char *name)
	: KviSysTrayWidget(parent, tooltip, name)
{
	m_timer = 0;
	setBackgroundMode(NoBackground);
}

KviSysTrayClock::~KviSysTrayClock()
{
	if( m_timer )
		killTimer(m_timer);
}

void KviSysTrayClock::show()
{
	m_timer = startTimer(1000);
	KviSysTrayWidget::show();
}

void KviSysTrayClock::hide()
{
	if( m_timer ) {
		killTimer(m_timer);
		m_timer = 0;
	}
	KviSysTrayWidget::hide();
}

void KviSysTrayClock::timerEvent(QTimerEvent *e)
{
	if( e->timerId() == m_timer )
		paintEvent(0);
}

struct tm *KviSysTrayClock::getTime(const time_t *timep)
{
	return localtime(timep);
}

QString KviSysTrayClock::icon()
{
	return SYSTRAY_ICON_CLOCK;
}

void KviSysTrayClock::paintEvent(QPaintEvent *)
{
	if( !isVisible() )
		return;

	QPainter pa(m_pSysTray->getMemoryBuffer());
	int widgetWidth  = width();
	int widgetHeight = height();

	paintDefaultBackground(&pa);

	pa.setPen(m_pSysTray->getForeground());
	time_t tim = time(0);
	struct tm *t = getTime(&tim);
	KviStr timeStr(KviStr::Format,
		"%d%d:%d%d:%d%d",
		t->tm_hour / 10, t->tm_hour % 10, t->tm_min / 10, t->tm_min % 10, t->tm_sec / 10, t->tm_sec % 10
	);
	if( t->tm_isdst > 0 )
		timeStr += __tr(" (DST)");
	KviStr dateStr(KviStr::Format,
		__tr("%s %d %s %d (day %d)"),
		_i18n_translateNoText(days_of_week[t->tm_wday]), t->tm_mday,
		_i18n_translateNoText(months_of_year[t->tm_mon]), 1900 + t->tm_year, t->tm_yday
	);

	if( widgetHeight > 32 ) {
		// Big mode
		g_pOptions->m_fntSysTray.setPointSize(16);
		pa.setFont(g_pOptions->m_fntSysTray);
		pa.drawText(4, widgetHeight - 6, timeStr.ptr(), timeStr.len());
		g_pOptions->m_fntSysTray.setPointSize(10);
		pa.setFont(g_pOptions->m_fntSysTray);
		pa.drawText(4, 14, dateStr.ptr(), dateStr.len());
	} else {
		// Small mode
		timeStr.append(" - ");
		timeStr.append(dateStr);
		g_pOptions->m_fntSysTray.setPointSize(14);
		pa.setFont(g_pOptions->m_fntSysTray);
		pa.drawText(4, (widgetHeight + 12) >> 1, timeStr.ptr(), timeStr.len());
	}

	paintDefaultFrame(&pa);
	bitBlt(this, 0, 0, m_pSysTray->getMemoryBuffer(), 0, 0, widgetWidth, widgetHeight);
}

#include "m_kvi_systray_clock.moc"
