// =============================================================================
//
//   Charpicker plugin for the KVIrc IRC client
//   Copyright (C) 1999-2000 Stefanek Szymon (kvirc@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviCharPicker"

#include <qtooltip.h>

#define __KVIRC_PLUGIN__

#include "kvi_charpicker.h"
#include "kvi_charpicker_button.h"
#include "kvi_input.h"
#include "kvi_locale.h"
#include "kvi_window.h"
#include "kvirc_plugin.h"

static QPtrList<KviCharPicker> *m_pCharPickerList = 0;
static bool m_bInCleanup = false;

KviCharPicker::KviCharPicker(QSplitter *parent, KviWindow *wnd)
	: QScrollView(parent, "char_picker_widget")
{
	m_pWnd = wnd;
	setFont(g_pOptions->m_fntInput);
	m_pCharPickerList->append(this);
	setFrameStyle(QFrame::Sunken | QFrame::Panel);
	enableClipper(true);

	m_pCloseButton = new KviPushButton(__tr("X"), this);
	QToolTip::add(m_pCloseButton, __tr("Close"));
	setCornerWidget(m_pCloseButton);
	connect(m_pCloseButton, SIGNAL(clicked()), this, SLOT(closeClicked()));
	m_pCloseButton->show();

	m_pUpdateButton = new KviPushButton(__tr("Update font"), viewport());
	m_pUpdateButton->resize(m_pUpdateButton->sizeHint());
	m_pUpdateButton->show();
	addChild(m_pUpdateButton, 0, 0);
	connect(m_pUpdateButton, SIGNAL(clicked()), this, SLOT(updateFont()));

	int baseY = m_pUpdateButton->height();

	m_pBtn[0] = new KviCharPickerButton(viewport(), 'X');
	m_pBtn[0]->setEnabled(false);
	QSize min = m_pBtn[0]->sizeHint();
	m_pBtn[0]->resize(min);
	m_pBtn[0]->show();
	addChild(m_pBtn[0], 0, baseY);
	int maxY = baseY;
	int maxX = 0;
	for( int i = 1; i < 256; i++ ) {
		m_pBtn[i] = new KviCharPickerButton(viewport(), i);
		m_pBtn[i]->resize(min);
		int ax = (i % 16) * min.width();
		int ay = (i / 16) * min.height() + baseY;
		if( maxX < (ax + min.width())  ) maxX = ax + min.width();
		if( maxY < (ay + min.height()) ) maxY = ay + min.height();
		addChild(m_pBtn[i], ax, ay);
		m_pBtn[i]->show();
		KviStr tmp(KviStr::Format, "Ascii %d", i);
		QToolTip::add(m_pBtn[i], tmp.ptr());
		connect(m_pBtn[i], SIGNAL(charPicked(const QString &)), this, SLOT(charPicked(const QString &)));
	}
	setVScrollBarMode(QScrollView::Auto);
	setHScrollBarMode(QScrollView::Auto);
	setResizePolicy(QScrollView::Manual);
	resizeContents(maxX, maxY);
}

void KviCharPicker::updateFont()
{
	m_pUpdateButton->setFont(g_pOptions->m_fntInput);
	m_pUpdateButton->resize(m_pUpdateButton->sizeHint());
	int baseY = m_pUpdateButton->height();

	m_pCloseButton->setFont(g_pOptions->m_fntInput);

	m_pBtn[0]->setFont(g_pOptions->m_fntInput);
	QSize min = m_pBtn[0]->sizeHint();
	m_pBtn[0]->resize(min);
	moveChild(m_pBtn[0], 0, baseY);
	int maxY = baseY;
	int maxX = 0;
	for( int i = 1; i < 256; i++ ) {
		m_pBtn[i]->setFont(g_pOptions->m_fntInput);
		m_pBtn[i]->resize(min);
		int ax = (i % 16) * min.width();
		int ay = (i / 16) * min.height() + baseY;
		if( maxX < (ax + min.width())  ) maxX = ax + min.width();
		if( maxY < (ay + min.height()) ) maxY = ay + min.height();
		moveChild(m_pBtn[i], ax, ay);
	}
	resizeContents(maxX, maxY);
}

void KviCharPicker::closeClicked()
{
	delete this;
}

KviCharPicker::~KviCharPicker()
{
	if( !m_bInCleanup ) {
		m_pCharPickerList->removeRef(this); // <--autoDelete disabled
	}
}

void KviCharPicker::charPicked(const QString &str)
{
	KviStr s = str;
	if( m_pWnd->m_pInput && *(s.ptr()))
		m_pWnd->m_pInput->insertChar(*(s.ptr()));
}

static KviCharPicker *lookup_charpicker(KviWindow *wnd)
{
	for( KviCharPicker *p = m_pCharPickerList->first(); p; p = m_pCharPickerList->next()) {
		if( p->m_pWnd == wnd ) return p;
	}
	return 0;
}


bool charpicker_plugin_command_charpicker(KviPluginCommandStruct *cmd)
{
	if( kvirc_plugin_param(cmd, 1) ) {
		// Turn it off
		KviCharPicker *p = lookup_charpicker(cmd->window);
		if( !p ) {
			cmd->error = KVI_ERROR_InvalidOperation;
			cmd->errorstr = __tr("No charpicker to remove in this window");
			return false;
		}
		delete p;
		return true;
	}

	if( kvirc_plugin_mdi_window_can_add_widget(cmd->window) ) {
		if( !lookup_charpicker(cmd->window) ) {
			KviCharPicker *p = new KviCharPicker(kvirc_plugin_mdi_window_splitter(cmd->window), cmd->window);
			p->show();
		} else {
			cmd->error = KVI_ERROR_InvalidOperation;
			cmd->errorstr = __tr("A charpicker is already in this window");
			return false;
		}
	} else {
		cmd->error = KVI_ERROR_InvalidOperation;
		cmd->errorstr = __tr("Cannot add charpicker to this window");
		return false;
	}
	return true;
}

bool charpicker_plugin_init(KviPluginCommandStruct *cmd)
{
	m_pCharPickerList = new QPtrList<KviCharPicker>;
	m_pCharPickerList->setAutoDelete(false);
	kvirc_plugin_register_command(cmd->handle, "CHARPICKER", charpicker_plugin_command_charpicker);
	return true;
}

void charpicker_plugin_cleanup()
{
	// Remove all the char picker widgets
	m_bInCleanup = true;
	m_pCharPickerList->setAutoDelete(true);
	delete m_pCharPickerList;
	m_pCharPickerList = 0;
}

KviPlugin kvirc_plugin =
{
	"Charpicker",
	"An ASCII character picker",
	"0.1",
	"Szymon Stefanek <stefanek@tin.it>",
	"<html>"
		"Allows you to add an ASCII character"
		"picker to almost any MDI window in KVIrc.<br><br>"
		"Exports the <b>/CHARPICKER</b> command."
		"</html>",
	charpicker_plugin_init,
	charpicker_plugin_cleanup,
	0,
	0
};

#include "m_kvi_charpicker.moc"
