//=============================================================================
//
//   File : kvi_kvs_corecallbackcommands.cpp
//   Created on Fri 31 Oct 2003 04:07:58 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_corecallbackcommands.h"

#include "kvi_kvs_kernel.h"
#include "kvi_kvs_timermanager.h"
#include "kvi_kvs_aliasmanager.h"
#include "kvi_kvs_variantlist.h"

#include "kvi_locale.h"

namespace KviKvsCoreCallbackCommands
{
	/*
		@doc: alias
		@title:
			alias
		@type:
			command
		@short:
			Adds a new alias or modifies an existing one
		@syntax:
			alias [-q] (<alias_name>) <implementation>
			alias [-q] (<alias_name>){}
		@switches:
			!sw: -q | --quiet
			Causes the command to run quietly
		@description:
			Adds the alias <alias_name> with the specified <implementation> code.
			The implementation code can be either a single KVS instruction
			or an instruction block (instruction list enclosed in braces).[br]
			If the alias was already existing, it is replaced with the
			new implementation.[br]
			If the <implementation> is empty (eg. "{}" or just a ";")
			the alias <alias_name> is removed instead of being added.
			If the "remove" form is used but the specified <alias_name> is
			not existing in the alias store then a warning is printed unless
			the -q (--quiet) switch is used.
		@examples:
			[example]
				[comment]# Add the alias j[/comment]
				alias(j)
				{
					[cmd]join[/cmd] $0;
				}
				[comment]# Remove the alias j[/comment]
				alias(j){}
			[/example]
		@seealso:
			[doc:kvs_aliasesandfunctions]Aliases and functions[/doc]
	*/

	_KVS_CORECALLBACKCOMMAND(alias)
	{
		KviKvsVariant * vName = pParams->first();

		if(!vName || vName->isEmpty())
		{
			c->error(__tr2qs("Missing alias name"));
			return false;
		}

		QString szName;
		vName->asString(szName);

		if(pCallback->code().isEmpty())
		{
			if(!KviKvsAliasManager::instance()->remove(szName))
			{
				if(!pSwitches->find('q',"quiet"))
					c->warning(__tr2qs("The alias %Q is not existing"),&szName);
			}
		} else {
			KviKvsScript * pScript = new KviKvsScript(*pCallback);
			pScript->setName(szName);
			KviKvsAliasManager::instance()->add(szName,new KviKvsScript(*pCallback));
		}
		return true;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////

	/*
		@doc: timer
		@title:
			timer
		@type:
			command
		@short:
			Starts a timer
		@syntax:
			timer [-s] [-p] (<name>,<delay_in_msecs>[,<callback_param1>[,<callback_param2>[,...]]])
			{
				<callback_command>
			}
		@switches:
			!sw: -s | --single-shot
			Causes the timer to trigger only once (single shot timer)
			!sw: -p | --persistent
			Creates a persistent timer bound to any existing window
		@description:
			Starts a new timer named <name> with the specified delay (in milliseconds).[br]
			The timer periodically calls the specified <callback_command> code passing the
			eventual <callback_param> strings as positional parameters.[br]
			If a timer with the same name already exists, it is replaced by this one.[br]
			[b]The <callback_command> is evaluated at timer "shot" time and NOT while
			this command is being parsed. This means that the identifiers that you put
			inside <callback_command> will NOT have the current values.[/b]
			The values will be assigned at timer "shot" time.[br]
			This is a common scripters error and problem: if it is not clear, look at the examples below.[br]
			The timer is bound to the window in that this command is executed in.[br]
			If the window gets destroyed, the timer is stopped; unless the -p switch is used.[br]
			The -p switch causes the timer to be persistent across the application and exists until
			the last window has been closed: it is basically rebound to another (random) window when the
			original window is destroyed.[br]
			The -s switch cuases this timer to trigger only once: it will be automatically destroyed after that.[br]
			The time has an associated set of [doc:data_structures]extended scope variables[/doc]:
			the variables that begin with "%:" have their life extended to the whole "life" of the timer.[br]
			Using a very low delay is a common method to perform some background processing: you
			basically split a huge job in small slices and execute them when the timer is triggered
			until you run out of slices. A delay of 0 will cause the timer to be called whenever
			KVIrc has some "idle time" to spend.
			On the other hand, remember that timers are precious resources: many timers running
			with a very low delay will cause KVIrc to slow down.[br]
			Since all the kvirc timers share the same namespace it is a good idea to use
			descriptive timer names: a timer named "a" is likely to be used by two or more scripts
			at once causing one (or both) of them to fail.[br]
			A timer can be stopped at any time by using the [cmd]killtimer[/cmd] command.
		@seealso:
			[cmd]killtimer[/cmd]
		@examples:
			[example]
				[comment]# Just a plain timer[/comment]
				timer(test,1000){ echo "Hello!"; }
				[comment]# Now watch the timer running[/comment]
				killtimer test
				[comment]# Single shot timer[/comment]
				timer -s (test,1000){ echo "This will fire only once!"; }
				[comment]# The call above is equivalent to[/comment]
				timer(test,1000){ echo "This will file only once!"; killtimer test; }
				[comment]# Callback parameters: consider the following code[/comment]
				%parameter = "some string value"
				echo "Before calling /timer \%parameter is \"%parameter\""
				timer -s (test,1000,%parameter){ echo "inside the callback \%parameter is \"%parameter\" but \$0 is \"$0\""; }
				[comment]# watch the timer running , and note the behaviour of the %parameter variable[/comment]
				killtimer test
				[comment]# Use the extended scope timer variables[/comment]
				timer(test,1000)
				{
					[comment]# Use the extended scope %:count variable to keep track[/comment]
					[comment]# of the times that this timer has been called[/comment]
					[cmd]if[/cmd]("%:count" == "")%:count = 1
					else %:count++
					[cmd]echo[/cmd] "This timer has fired %:count times"
					if(%:count == 10)
					{
						# This will kill the current timer, we don't need to specify the name
						[cmd]killtimer[/cmd]
					}
				}
				[comment]# Use isTimer to check if the timer exists[/comment]
				[cmd]echo[/cmd] [fnc]$isTimer[/fnc](test)
				[comment]# Repeat the command above after the 10th timeout...[/comment]
			[/example]
	*/

	_KVS_CORECALLBACKCOMMAND(timer)
	{
		KviKvsVariant * vName = pParams->first();
		KviKvsVariant * vDelay = pParams->next();

		if(!vName || vName->isEmpty())
		{
			c->error(__tr2qs("Missing timer name"));
			return false;
		}

		QString szName;
		vName->asString(szName);

		if(!vDelay)
		{
			c->error(__tr2qs("Missing timeout delay"));
			return false;
		}

		int iDelay;
		if(!vDelay->asInteger(iDelay))
		{
			c->error(__tr2qs("The timeout delay didn't evaluate to an integer"));
			return false;
		}

		KviKvsTimer::Lifetime lt;

		if(pSwitches->find('s',"single-shot"))lt = KviKvsTimer::SingleShot;
		else if(pSwitches->find('p',"persistent"))lt = KviKvsTimer::Persistent;
		else lt = KviKvsTimer::WindowLifetime;

		// prepare the callback parameters
		KviKvsVariantList * l = new KviKvsVariantList();
		l->setAutoDelete(true);

		KviKvsVariant * v = pParams->next();
		while(v)
		{
			l->append(new KviKvsVariant(*v)); // copy
			v = pParams->next();
		}

		if(!KviKvsTimerManager::instance()->addTimer(szName,lt,c->window(),iDelay,new KviKvsScript(*pCallback),l))
		{
			c->error(__tr2qs("Unable to add the timer: insufficient system resources"));
			return false;
		}

		return true;
	}


	void init()
	{
		KviKvsKernel * pKern = KviKvsKernel::instance();

#define _REGCMD(__cmdName,__routine) \
		{ \
			KviKvsCoreCallbackCommandExecRoutine * r = new KviKvsCoreCallbackCommandExecRoutine; \
			r->proc = KVI_PTR2MEMBER(KviKvsCoreCallbackCommands::__routine); \
			pKern->registerCoreCallbackCommandExecRoutine(QString(__cmdName),r); \
		}
		
		_REGCMD("alias",alias);
		_REGCMD("timer",timer);

#undef _REGCMD
	}
};
