#ifndef _KVI_KVS_RUNTIMECONTEXT_H_
#define _KVI_KVS_RUNTIMECONTEXT_H_
//=============================================================================
//
//   File : kvi_kvs_runtimecontext.h
//   Created on Tue 07 Oct 2003 01:49:40 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "kvi_settings.h"

#include "kvi_window.h"

#include "kvi_kvs_variant.h"
#include "kvi_kvs_hash.h"
#include "kvi_kvs_variantlist.h"
#include "kvi_kvs_reporttarget.h"
#include "kvi_kvs_switchlist.h"

class KviKvsScript;
class KviConsole;
class KviIrcContext;
class KviIrcConnection;

class KviKvsExtendedRunTimeData
{
	friend class KviKvsRunTimeContext;
protected:
	KviKvsHash       * m_pExtendedScopeVariables; // extended scope vars, shallow, may be 0
	KviKvsSwitchList * m_pAliasSwitchList;        // switches for the aliases, shallow, may be 0
public:
	// all shallow data, all may be 0
	KviKvsExtendedRunTimeData()
		: m_pExtendedScopeVariables(0),
		m_pAliasSwitchList(0)
		{};
	KviKvsExtendedRunTimeData(KviKvsHash * pExtScopeVariables)
		: m_pExtendedScopeVariables(pExtScopeVariables),
		m_pAliasSwitchList(0)
		{};
	KviKvsExtendedRunTimeData(KviKvsSwitchList * pAliasSwitchList)
		: m_pExtendedScopeVariables(0),
		m_pAliasSwitchList(pAliasSwitchList)
		{};
public:
	KviKvsHash * extendedScopeVariables(){ return m_pExtendedScopeVariables; };
	KviKvsSwitchList * aliasSwitchList(){ return m_pAliasSwitchList; };
};


class KVIRC_API KviKvsRunTimeContext : public KviKvsReportTarget
{
	friend class KviKvsScript;
protected:
	// only KviKvsScript can instantiate this class
	KviKvsRunTimeContext(KviKvsScript * pScript,
				KviWindow * pWnd,
				KviKvsVariantList * pParams,
				KviKvsVariant * pRetVal,
				KviKvsExtendedRunTimeData * pExtData = 0);
public:
	~KviKvsRunTimeContext();
protected:
	// stuff that is fixed in the whole script context
	KviKvsScript              * m_pScript;                  // shallow, never 0 except for module event calls
	KviKvsHash                * m_pLocalVariables;          // owned, never 0
	KviKvsVariantList         * m_pParameterList;           // shallow, never 0
	KviKvsVariant             * m_pReturnValue;             // shallow, never 0 except for module event calls

	// stuff that is generally global but sometimes may change
	// during the execution of the script
	KviWindow                 * m_pWindow;                  // shallow, never 0

	enum RunTimeFlags { BreakPending = 1, HaltCalled = 2 };
	unsigned int                m_uRunTimeFlags;            // a combination of RunTimeFlags
	
	KviKvsExtendedRunTimeData * m_pExtendedData;            // shallow, may be 0
public:
	// the window that this script is bound to (it MAY change during the script parsing)
	KviWindow * window()
		{ return m_pWindow; };
	// quick access to the irc context (it MAY be zero!)
	KviIrcContext * context()
		{ return m_pWindow->context(); };
	// quick access to the console that this script is bound to (it MAY be zero, if there is no console)
	KviConsole * console()
		{ return m_pWindow->console(); };
	// quick access to the connection: it MAY be zero!
	KviIrcConnection * connection()
		{ return m_pWindow->connection(); };

	KviKvsScript * script()
		{ return m_pScript; };

	// the local variables of this script
	KviKvsHash * localVariables()
		{ return m_pLocalVariables; };
	// the global application-wide variables
	KviKvsHash * globalVariables();
	// the parameters passed to this script
	KviKvsVariantList * parameterList()
		{ return m_pParameterList; };

	KviKvsVariant * returnValue()
		{ return m_pReturnValue; };


	// the extended scope variables
	KviKvsHash * extendedScopeVariables()
		{ return m_pExtendedData ? m_pExtendedData->extendedScopeVariables() : 0; };
	KviKvsSwitchList * aliasSwitchList()
		{ return m_pExtendedData ? m_pExtendedData->aliasSwitchList() : 0; };

	// this MUST be called before any blocking call that might return to the main event loop
	// and eventually quit kvirc or close the window that this command is associated to
	// Actually this is a NO-OP but later may really do something that avoids kvirc to crash
	void enterBlockingSection();
	// this MUST be called after exiting the blocking section above
	// if this function returns false your parsing code MUST "return false" immediately:
	// it means that something rather critical happened and the script
	// execution cannot continue
	bool leaveBlockingSection();

//PRIVATE:
	// tricky
	// don't use it: it's only for the parser
	void setWindow(KviWindow * pWnd)
		{ m_pWindow = pWnd; };

	// returns the old pointer
	KviKvsVariant * swapReturnValuePointer(KviKvsVariant * pNewPointer);
	// the old pointer MUST be reset!
	
	// this is called by the parser when a break is encountered
	// the parser calls setBreakPending() and returns false
	// the contexts that can handle a break should check breakPending()
	// after an internal evaluation returned false.
	// if breakPending() they should handle it with handleBreak() and
	// return true instead.
	void setBreakPending()
		{ m_uRunTimeFlags |= BreakPending; };
	// this tells if a break command has been called
	// it may return true if an execution() command has returned false
	// (if breakPending() there should be no error()
	bool breakPending()
		{ return (m_uRunTimeFlags & BreakPending); };
	// this is called by the commands that can handle a break
	void handleBreak()
		{ m_uRunTimeFlags &= ~BreakPending; };

	// this is called by the parser when a halt is encountered
	// the parser then returns false and all the stack frames
	// above should do the same
	void setHaltCalled()
		{ m_uRunTimeFlags |= HaltCalled; };
	bool haltCalled()
		{ return (m_uRunTimeFlags & HaltCalled); };
};

#endif //!_KVI_KVS_RUNTIMECONTEXT_H_
