//=============================================================================
//
//   File : kvi_kvs_treenode_operation.cpp
//   Created on Thu 18 Dec 2003 22:36:00 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_treenode_operation.h"
#include "kvi_kvs_treenode_data.h"
#include "kvi_kvs_runtimecontext.h"
#include "kvi_locale.h"


#include <math.h>


KviKvsTreeNodeOperation::KviKvsTreeNodeOperation(const QChar * pLocation)
: KviKvsTreeNodeInstruction(pLocation)
{
	//m_pTargetData = 0; no need to set it
}

KviKvsTreeNodeOperation::~KviKvsTreeNodeOperation()
{
	delete m_pTargetData;
}

void KviKvsTreeNodeOperation::setTargetVariableReference(KviKvsTreeNodeData * r)
{
	m_pTargetData = r;
	m_pTargetData->setParent(this);
}

void KviKvsTreeNodeOperation::dump(const char * prefix)
{
	debug("%s Operation",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
}





KviKvsTreeNodeOperationAssignment::KviKvsTreeNodeOperationAssignment(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationAssignment::~KviKvsTreeNodeOperationAssignment()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationAssignment::dump(const char * prefix)
{
	debug("%s OperationAssignment",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationAssignment::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;

	target->result()->takeFrom(v);
	delete target;
	return true;
}






KviKvsTreeNodeOperationDecrement::KviKvsTreeNodeOperationDecrement(const QChar * pLocation)
: KviKvsTreeNodeOperation(pLocation)
{
}

KviKvsTreeNodeOperationDecrement::~KviKvsTreeNodeOperationDecrement()
{
}

void KviKvsTreeNodeOperationDecrement::dump(const char * prefix)
{
	debug("%s OperationDecrement",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationDecrement::execute(KviKvsRunTimeContext * c)
{
	KviKvsRWEvaluationResult * v = m_pTargetData->evaluateReadWrite(c);
	if(!v)return false;

	int iVal;
	if(v->result()->asInteger(iVal))
	{
		v->result()->setInteger(iVal - 1);
		delete v;
        v = 0;
		return true;
	}

	double dVal;
	if(v->result()->asReal(dVal))
	{
		v->result()->setReal(dVal - 1.0);
		delete v;
        v = 0;
		return true;
	}

	c->error(location(),__tr2qs("The target variable didn't evaluate to an integer or real value"));
	delete v;
	return false;
}







KviKvsTreeNodeOperationIncrement::KviKvsTreeNodeOperationIncrement(const QChar * pLocation)
: KviKvsTreeNodeOperation(pLocation)
{
}

KviKvsTreeNodeOperationIncrement::~KviKvsTreeNodeOperationIncrement()
{
}

void KviKvsTreeNodeOperationIncrement::dump(const char * prefix)
{
	debug("%s OperationIncrement",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationIncrement::execute(KviKvsRunTimeContext * c)
{
	KviKvsRWEvaluationResult * v = m_pTargetData->evaluateReadWrite(c);
	if(!v)return false;

	int iVal;
	if(v->result()->asInteger(iVal))
	{
		v->result()->setInteger(iVal + 1);
		delete v;
        v = 0;
		return true;
	}

	double dVal;
	if(v->result()->asReal(dVal))
	{
		v->result()->setReal(dVal + 1.0);
		delete v;
        v = 0;
		return true;
	}
	c->error(location(),__tr2qs("The target variable didn't evaluate to an integer or real value"));
	delete v;
	return false;
}




KviKvsTreeNodeOperationSelfAnd::KviKvsTreeNodeOperationSelfAnd(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfAnd::~KviKvsTreeNodeOperationSelfAnd()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfAnd::dump(const char * prefix)
{
	debug("%s OperationSelfAnd",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfAnd::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;

	int iRVal;

	if(!v.asInteger(iRVal))
	{
		c->error(location(),__tr2qs("The right side of operator '&=' didn't evaluate to an integer"));
		return false;
	}

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	int iLVal;
	
	if(!target->result()->asInteger(iLVal))
	{
		c->error(location(),__tr2qs("The left side of operator '&=' didn't evaluate to an integer"));
		delete target;
        target = 0;
		return false;
	}
	
	target->result()->setInteger(iLVal & iRVal);
	delete target;
	return true;
}



KviKvsTreeNodeOperationSelfDivision::KviKvsTreeNodeOperationSelfDivision(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfDivision::~KviKvsTreeNodeOperationSelfDivision()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfDivision::dump(const char * prefix)
{
	debug("%s OperationSelfDivision",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfDivision::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;

	KviKvsNumber rnum;

	if(!v.asNumber(rnum))
	{
		c->error(location(),__tr2qs("The right side of operator '/=' didn't evaluate to a number"));
		return false;
	}
	
	KviKvsNumber lnum;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asNumber(lnum))
	{
		c->error(location(),__tr2qs("The left side of operator '/=' didn't evaluate to a number"));
		delete target;
        target = 0;
		return false;
	}

	if(rnum.isInteger())
	{
		if(rnum.integer() == 0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			delete target;
            target = 0;
			return false;
		}
		if(lnum.isInteger())
			target->result()->setInteger(lnum.integer() / rnum.integer());
		else
			target->result()->setReal(lnum.real() / (double)(rnum.integer()));
	} else {
		if(rnum.real() == 0.0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			delete target;
            target = 0;
			return false;
		}
		if(lnum.isInteger())
			target->result()->setReal(((double)(lnum.integer())) / rnum.real());
		else
			target->result()->setReal(lnum.real() / rnum.real());
	}

	delete target;
	return true;
}

KviKvsTreeNodeOperationSelfModulus::KviKvsTreeNodeOperationSelfModulus(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfModulus::~KviKvsTreeNodeOperationSelfModulus()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfModulus::dump(const char * prefix)
{
	debug("%s OperationSelfModulus",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfModulus::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;
	KviKvsNumber rnum;

	if(!v.asNumber(rnum))
	{
		c->error(location(),__tr2qs("The right side of operator '%=' didn't evaluate to a number"));
		return false;
	}
	
	KviKvsNumber lnum;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asNumber(lnum))
	{
		c->error(location(),__tr2qs("The left side of operator '%=' didn't evaluate to a number"));
		delete target;
        target = 0;
		return false;
	}

	if(rnum.isInteger())
	{
		if(rnum.integer() == 0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			delete target;
            target = 0;
			return false;
		}
		if(lnum.isInteger())
			target->result()->setInteger(lnum.integer() % rnum.integer());
		else
			target->result()->setReal(fmod(lnum.real(),(double)(rnum.integer())));
	} else {
		if(rnum.real() == 0.0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			delete target;
            target = 0;
			return false;
		}
		if(lnum.isInteger())
			target->result()->setReal(fmod(((double)(lnum.integer())),rnum.real()));
		else
			target->result()->setReal(fmod(lnum.real(),rnum.real()));
	}

	return true;
}





KviKvsTreeNodeOperationSelfMultiplication::KviKvsTreeNodeOperationSelfMultiplication(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfMultiplication::~KviKvsTreeNodeOperationSelfMultiplication()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfMultiplication::dump(const char * prefix)
{
	debug("%s OperationSelfMultiplication",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfMultiplication::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;
	KviKvsNumber rnum;

	if(!v.asNumber(rnum))
	{
		c->error(location(),__tr2qs("The right side of operator '*=' didn't evaluate to a number"));
		return false;
	}
	
	KviKvsNumber lnum;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asNumber(lnum))
	{
		c->error(location(),__tr2qs("The left side of operator '*=' didn't evaluate to a number"));
		delete target;
		return false;
	}

	if(rnum.isInteger())
	{
		if(lnum.isInteger())
			target->result()->setInteger(lnum.integer() * rnum.integer());
		else
			target->result()->setReal(lnum.real() * (double)(rnum.integer()));
	} else {
		if(lnum.isInteger())
			target->result()->setReal(((double)(lnum.integer())) * rnum.real());
		else
			target->result()->setReal(lnum.real() * rnum.real());
	}
	return true;
}




KviKvsTreeNodeOperationSelfOr::KviKvsTreeNodeOperationSelfOr(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfOr::~KviKvsTreeNodeOperationSelfOr()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfOr::dump(const char * prefix)
{
	debug("%s OperationSelfOr",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfOr::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;

	int iRVal;
	if(!v.asInteger(iRVal))
	{
		c->error(location(),__tr2qs("The right side of operator '|=' didn't evaluate to an integer"));
		return false;
	}
	
	int iLVal;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asInteger(iLVal))
	{
		c->error(location(),__tr2qs("The left side of operator '|=' didn't evaluate to an integer"));
		delete target;
        target = 0;
		return false;
	}
	
	target->result()->setInteger(iLVal | iRVal);
	delete target;
	return true;
}




KviKvsTreeNodeOperationSelfShl::KviKvsTreeNodeOperationSelfShl(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfShl::~KviKvsTreeNodeOperationSelfShl()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfShl::dump(const char * prefix)
{
	debug("%s OperationSelfShl",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfShl::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;

	int iRVal;

	if(!v.asInteger(iRVal))
	{
		c->error(location(),__tr2qs("The right side of operator '<<=' didn't evaluate to an integer"));
		return false;
	}
	
	int iLVal;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asInteger(iLVal))
	{
		c->error(location(),__tr2qs("The left side of operator '<<=' didn't evaluate to an integer"));
		delete target;
        target = 0;
		return false;
	}
	
	target->result()->setInteger(iLVal << iRVal);
	delete target;
	return true;
}





KviKvsTreeNodeOperationSelfShr::KviKvsTreeNodeOperationSelfShr(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfShr::~KviKvsTreeNodeOperationSelfShr()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfShr::dump(const char * prefix)
{
	debug("%s OperationSelfShr",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfShr::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;

	int iRVal;

	if(!v.asInteger(iRVal))
	{
		c->error(location(),__tr2qs("The right side of operator '>>=' didn't evaluate to an integer"));
		return false;
	}
	
	int iLVal;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asInteger(iLVal))
	{
		c->error(location(),__tr2qs("The left side of operator '>>=' didn't evaluate to an integer"));
		delete target;
        target = 0;
		return false;
	}
	
	target->result()->setInteger(iLVal >> iRVal);
	delete target;
	return true;
}











KviKvsTreeNodeOperationSelfSubtraction::KviKvsTreeNodeOperationSelfSubtraction(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfSubtraction::~KviKvsTreeNodeOperationSelfSubtraction()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfSubtraction::dump(const char * prefix)
{
	debug("%s OperationSelfSubtraction",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfSubtraction::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;
	KviKvsNumber rnum;

	if(!v.asNumber(rnum))
	{
		c->error(location(),__tr2qs("The right side of operator '-=' didn't evaluate to a number"));
		return false;
	}
	
	KviKvsNumber lnum;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asNumber(lnum))
	{
		c->error(location(),__tr2qs("The left side of operator '-=' didn't evaluate to a number"));
		delete target;
        target = 0;
		return false;
	}

	if(rnum.isInteger())
	{
		if(lnum.isInteger())
			target->result()->setInteger(lnum.integer() - rnum.integer());
		else
			target->result()->setReal(lnum.real() - (double)(rnum.integer()));
	} else {
		if(lnum.isInteger())
			target->result()->setReal(((double)(lnum.integer())) - rnum.real());
		else
			target->result()->setReal(lnum.real() - rnum.real());
	}

	delete target;
	return true;
}





KviKvsTreeNodeOperationSelfSum::KviKvsTreeNodeOperationSelfSum(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfSum::~KviKvsTreeNodeOperationSelfSum()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfSum::dump(const char * prefix)
{
	debug("%s OperationSelfSum",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfSum::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;
	KviKvsNumber rnum;

	if(!v.asNumber(rnum))
	{
		c->error(location(),__tr2qs("The right side of operator '+=' didn't evaluate to a number"));
		return false;
	}
	
	KviKvsNumber lnum;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asNumber(lnum))
	{
		c->error(location(),__tr2qs("The left side of operator '+=' didn't evaluate to a number"));
		delete target;
        target = 0;
		return false;
	}

	if(rnum.isInteger())
	{
		if(lnum.isInteger())
			target->result()->setInteger(lnum.integer() + rnum.integer());
		else
			target->result()->setReal(lnum.real() + (double)(rnum.integer()));
	} else {
		if(lnum.isInteger())
			target->result()->setReal(((double)(lnum.integer())) + rnum.real());
		else
			target->result()->setReal(lnum.real() + rnum.real());
	}

	delete target;
	return true;
}







KviKvsTreeNodeOperationSelfXor::KviKvsTreeNodeOperationSelfXor(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationSelfXor::~KviKvsTreeNodeOperationSelfXor()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationSelfXor::dump(const char * prefix)
{
	debug("%s OperationSelfXor",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationSelfXor::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;

	int iRVal;

	if(!v.asInteger(iRVal))
	{
		c->error(location(),__tr2qs("The right side of operator '^=' didn't evaluate to an integer"));
		return false;
	}
	
	int iLVal;

	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	if(!target->result()->asInteger(iLVal))
	{
		c->error(location(),__tr2qs("The left side of operator '^=' didn't evaluate to an integer"));
		delete target;
        target = 0;
		return false;
	}
	
	target->result()->setInteger(iLVal ^ iRVal);
	delete target;
	return true;
}







KviKvsTreeNodeOperationStringAppend::KviKvsTreeNodeOperationStringAppend(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationStringAppend::~KviKvsTreeNodeOperationStringAppend()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationStringAppend::dump(const char * prefix)
{
	debug("%s OperationStringAppend",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationStringAppend::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;
	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	QString sz1;
	target->result()->asString(sz1);
	v.appendAsString(sz1);
	target->result()->setString(sz1);
	delete target;
	return true;
}







KviKvsTreeNodeOperationStringAppendWithComma::KviKvsTreeNodeOperationStringAppendWithComma(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationStringAppendWithComma::~KviKvsTreeNodeOperationStringAppendWithComma()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationStringAppendWithComma::dump(const char * prefix)
{
	debug("%s OperationStringAppendWithComma",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationStringAppendWithComma::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;
	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	QString sz1;
	target->result()->asString(sz1);
	if(sz1.isEmpty())
	{
		v.asString(sz1);
	} else {
		QString sz2;
		v.asString(sz2);
		sz1 += QChar(',');
		sz1 += sz2;
	}
	target->result()->setString(sz1);
	delete target;
	return true;
}






KviKvsTreeNodeOperationStringAppendWithSpace::KviKvsTreeNodeOperationStringAppendWithSpace(const QChar * pLocation,KviKvsTreeNodeData * pRightSide)
: KviKvsTreeNodeOperation(pLocation)
{
	m_pRightSide = pRightSide;
	m_pRightSide->setParent(this);
}

KviKvsTreeNodeOperationStringAppendWithSpace::~KviKvsTreeNodeOperationStringAppendWithSpace()
{
	delete m_pRightSide;
}

void KviKvsTreeNodeOperationStringAppendWithSpace::dump(const char * prefix)
{
	debug("%s OperationStringAppendWithSpace",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pTargetData->dump(tmp.latin1());
	m_pRightSide->dump(tmp.latin1());
}

bool KviKvsTreeNodeOperationStringAppendWithSpace::execute(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pRightSide->evaluateReadOnly(c,&v))return false;
	KviKvsRWEvaluationResult * target = m_pTargetData->evaluateReadWrite(c);
	if(!target)return false;
	
	QString sz1;
	target->result()->asString(sz1);
	if(sz1.isEmpty())
	{
		v.asString(sz1);
	} else {
		QString sz2;
		v.asString(sz2);
		sz1 += QChar(' ');
		sz1 += sz2;
	}
	target->result()->setString(sz1);
	delete target;
	return true;
}
