//=============================================================================
//
//   File : kvi_popupmenu.cpp
//   Creation date : Sun Sep 1 2000 15:02:21 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000-2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================
#define __KVIRC__
#define _KVI_DEBUG_CHECK_RANGE_


#include "kvi_debug.h"
#include "kvi_popupmenu.h"
#include "kvi_iconmanager.h"
#include "kvi_window.h"
#include "kvi_console.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_config.h"
#include "kvi_app.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_datacontainer.h"
#include "kvi_cmdformatter.h"
#include "kvi_popupmanager.h"
#include "kvi_menubar.h"
#ifdef COMPILE_ON_WINDOWS
	#include "kvi_malloc.h"
#endif

#include <qlabel.h>

KviPopupMenuItem::KviPopupMenuItem(const KviPopupMenuItem * src)
{
	m_type = src->m_type;
	m_szText = src->m_szText;
	m_szCode = src->m_szCode;
	m_szIcon = src->m_szIcon;
	m_szExpr = src->m_szExpr;
	m_szId = src->m_szId;

	if(src->m_pMenu && (m_type != ExtMenu))
	{
		m_pMenu = new KviPopupMenu(src->m_pMenu->name());
		m_pMenu->copyFrom(src->m_pMenu);
	} else m_pMenu = 0;

	m_pLabel = 0;
}

void KviPopupMenuItem::setMenu(KviPopupMenu * m)
{
	if(m_pMenu)delete m_pMenu;
	m_pMenu = m;
}







KviPopupMenuTopLevelData::KviPopupMenuTopLevelData(KviDataContainer * d,KviParameterList * l,KviWindow * wnd,bool bRunInTestMode)
{
	pDataContainer = d;
	pParamList = l;
	pWnd = wnd;
	bLocked = false;
	bTestMode = bRunInTestMode;
}

KviPopupMenuTopLevelData::~KviPopupMenuTopLevelData()
{
	if(pParamList)delete pParamList;
	if(pDataContainer)delete pDataContainer;
}








KviPopupMenu::KviPopupMenu(const QString & name)
:QPopupMenu(0,name.utf8().data())
{
	m_pItemList = new KviPtrList<KviPopupMenuItem>;
	m_pItemList->setAutoDelete(true);
	m_pPrologues = new KviPtrList<KviPopupMenuCodeSnippet>;
	m_pPrologues->setAutoDelete(true);
	m_pEpilogues = new KviPtrList<KviPopupMenuCodeSnippet>;
	m_pEpilogues->setAutoDelete(true);
	m_pParentPopup = 0;
	m_pTopLevelData = 0;
	m_pTempTopLevelData = 0;
	m_bSetupDone = false;
	connect(this,SIGNAL(activated(int)),this,SLOT(itemClicked(int)));
	connect(this,SIGNAL(aboutToShow()),this,SLOT(setupMenuContents()));
}


KviPopupMenu::~KviPopupMenu()
{
	delete m_pItemList;
	delete m_pPrologues;
	delete m_pEpilogues;
	if(m_pTopLevelData)delete m_pTopLevelData;
	if(m_pTempTopLevelData)delete m_pTempTopLevelData;
}

bool KviPopupMenu::removeItemById(const QString &szId,bool bRecursive)
{
	KviPopupMenuCodeSnippet * se;
	
	for(se = m_pEpilogues->first();se;se = m_pEpilogues->next())
	{
		if(KviQString::equalCI(szId,se->m_szId))
		{
			m_pEpilogues->removeRef(se);
			return true;
		}
	}

	for(se = m_pPrologues->first();se;se = m_pPrologues->next())
	{
		if(KviQString::equalCI(szId,se->m_szId))
		{
			m_pPrologues->removeRef(se);
			return true;
		}
	}

	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		if(KviQString::equalCI(it->id(),szId))
		{
			m_pItemList->removeRef(it); // bye :)
			return true;
		}
	}

	if(bRecursive)
	{
		for(KviPopupMenuItem * ii = m_pItemList->first();ii;ii = m_pItemList->next())
		{
			if(ii->menu())
			{
				bool bRet = ii->menu()->removeItemById(szId,true);
				if(bRet)return true;
			}
		}
	}

	return false;
}

void KviPopupMenu::copyFrom(const KviPopupMenu * src)
{
	doClear();

	for(KviPopupMenuCodeSnippet * se = src->m_pEpilogues->first();se;se = src->m_pEpilogues->next())
	{
		KviPopupMenuCodeSnippet * s = new KviPopupMenuCodeSnippet;
		s->m_szCode = se->m_szCode;
		s->m_szId = se->m_szId;
		m_pEpilogues->append(s);
	}

	for(KviPopupMenuCodeSnippet * sp = src->m_pPrologues->first();sp;sp = src->m_pPrologues->next())
	{
		KviPopupMenuCodeSnippet * s = new KviPopupMenuCodeSnippet;
		s->m_szCode = sp->m_szCode;
		s->m_szId = sp->m_szId;
		m_pPrologues->append(s);
	}


	for(const KviPopupMenuItem * it = src->m_pItemList->first();it;it = src->m_pItemList->next())
	{
		addItem(new KviPopupMenuItem(it));
	}
}

KviPopupMenuTopLevelData * KviPopupMenu::topLevelData()
{
	if(parentPopup())return parentPopup()->topLevelData();
	return m_pTopLevelData;
}

bool KviPopupMenu::isLocked()
{
	if(topLevelPopup()->isVisible())return true;
	KviPopupMenuTopLevelData * d = topLevelData();
	return d ? d->bLocked : false;
}

KviPopupMenu * KviPopupMenu::topLevelPopup()
{
	if(parentPopup())return parentPopup();
	return this;
}

void KviPopupMenu::addItem(KviPopupMenuItem * it)
{
	__range_invalid(isLocked());
	m_pItemList->append(it);
	if(it->menu())
	{
		it->menu()->setParentPopup(this);
		KviStr tmp(KviStr::Format,"_sub%d",m_pItemList->count());
		tmp.prepend(name());
		it->menu()->setName(tmp.ptr());
	}
}

void KviPopupMenu::doPopup(const QPoint & pnt,KviWindow * wnd,KviParameterList * params,bool bTestMode)
{
	__range_invalid(isLocked());
	__range_invalid(parentPopup());
	// This might be a compat problem later :(((((
	// it is also an ugly trick
	clearMenuContents();
	m_pTempTopLevelData = new KviPopupMenuTopLevelData(new KviDataContainer(false),params,wnd,bTestMode);
	QPopupMenu::popup(pnt);
}

void KviPopupMenu::clearMenuContents()
{
	m_bSetupDone = false;

	clear();

	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		if(it->isExtMenu())it->setMenu(0);
		else {
			if(it->menu())it->menu()->clearMenuContents();
		}
	}


	if(m_pTopLevelData)
	{
		delete m_pTopLevelData;
		m_pTopLevelData = 0;
	}
	if(m_pTempTopLevelData)
	{
		delete m_pTempTopLevelData;
		m_pTempTopLevelData = 0;
	}
}

void KviPopupMenu::doClear()
{
	clear();
	if(m_pTopLevelData)
	{
		delete m_pTopLevelData;
		m_pTopLevelData = 0;
	}
	if(m_pTempTopLevelData)
	{
		delete m_pTempTopLevelData;
		m_pTempTopLevelData = 0;
	}
	m_bSetupDone = false;
	m_pItemList->clear();

	m_pPrologues->clear();
	m_pEpilogues->clear();
}

void KviPopupMenu::lock(bool bLock)
{
	KviPopupMenuTopLevelData * d = topLevelData();
	if(!d)return;
	d->bLocked = bLock;
}

void KviPopupMenu::setupMenuContents()
{
	// This might be a compat problem later :((((

	if(parentPopup() == 0)
	{
		if(m_pTempTopLevelData == 0)
		{
			// We have been called by a KviMenuBar!
			// m_bSetupDone is not valid here
			clearMenuContents();
			m_pTopLevelData = new KviPopupMenuTopLevelData(new KviDataContainer(false),0,g_pActiveWindow);
		} else {
			if(m_bSetupDone)return;
			// we have been called by doPopup
			// the menu contents have been already cleared
			if(m_pTopLevelData)debug("Ops.. something got messed in KviPopupMenu activation system");
			// Swap the top level data from temporary to the permanent
			m_pTopLevelData = m_pTempTopLevelData;
			m_pTempTopLevelData = 0;
		}
	} else {
		if(m_bSetupDone)return;
	}

	m_bSetupDone = true;

	// HACK...this is to remove the separator inserted by Qt when popup() is called and the popup is empty
	clear();

	KviPopupMenuTopLevelData * d = topLevelData();

	if(!d)
	{
		debug("Ops...menu contents changed behind my back!");
		return;
	}

	lock(true);

	// FIXME: #warning "Maybe say that the window has changed"
	if(!g_pApp->windowExists(d->pWnd))d->pWnd = g_pApp->activeConsole();

	// Execute the prologue code
	for(KviPopupMenuCodeSnippet * s = m_pPrologues->first();s;s = m_pPrologues->next())
	{
		KviCommand cmdp(s->m_szCode,d->pWnd,0,d->pDataContainer);
		if(d->pParamList)cmdp.setParams(d->pParamList,false);
		if(!g_pUserParser->parseCommand(&cmdp))
		{
			if(cmdp.hasError())
			{
				d->pWnd->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken prologue code for menu '%s', error detail follows"),name());
				g_pUserParser->printError(&cmdp);
			}
		}
		cmdp.forgetExtendedScopeDataContainer();
	}

	// Fill this menu contents
	int idx = 0;
	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		if(it->isSeparator())
		{
			insertSeparator();
		} else {
			// Find the pixmap (if any)
			QPixmap * pix = g_pIconManager->getImage(it->icon());
			// Parse the text

// FIXME: #warning "Maybe first parse the condition, no ?"
			KviCommand cmd(it->text(),d->pWnd,0,d->pDataContainer);
			if(d->pParamList)cmd.setParams(d->pParamList,false);
			KviStr parsed;
			if(!g_pUserParser->parseCmdFinalPart(&cmd,parsed))parsed = it->text();
			// Parse the condition (if present)
			bool bInsertIt = true;
			if(it->hasExpression())
			{
				cmd.setCmdBuffer(it->expr());
				long result;
				if(!g_pUserParser->evaluateExpression(&cmd,&result,false))
				{
					// broken expression
					g_pUserParser->printError(&cmd);
					d->pWnd->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken expression for menu item '%Q', ignoring"),&(it->text()));
				} else {
					if(result == 0)bInsertIt = false;
				}
			}

			// Eventually insert the item
			if(bInsertIt)
			{
				int id;
				if(it->isLabel())
				{
					it->setLabel(new QLabel(parsed.ptr(),this));
					id = insertItem(it->label());
					it->label()->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
					if(pix)it->label()->setPixmap(*pix);
				} else if(it->isExtMenu())
				{
					it->setMenu(0);
					KviPopupMenu * source = g_pPopupManager->lookupPopup(it->code());
					if(source)
					{
						QString tmp = QString("ontheflycopyof_%1").arg(it->code());
						KviPopupMenu * onTheFly = new KviPopupMenu(tmp);
						onTheFly->copyFrom(source);
						onTheFly->setParentPopup(this);
						it->setMenu(onTheFly);
						if(pix)id = insertItem(*pix,parsed.ptr(),it->menu());
						else id = insertItem(parsed.ptr(),it->menu());
					} else {
						d->pWnd->output(KVI_OUT_PARSERWARNING,__tr2qs("Can't find the external popup '%Q', ignoring"),&(it->code()));
						bInsertIt = false;
					}
				} else if(it->menu())
				{
					if(pix)id = insertItem(*pix,parsed.ptr(),it->menu());
					else id = insertItem(parsed.ptr(),it->menu());
				} else {
					if(pix)id = insertItem(*pix,parsed.ptr());
					else id = insertItem(parsed.ptr());
				}
				if(bInsertIt)
				{
					//debug("Inserted and set the param");
					if(!setItemParameter(id,idx))debug("Ops...failed to set it ?");
				}
			}

			cmd.forgetExtendedScopeDataContainer();
		}
		++idx;
	}

	// Now the epilogue
	for(KviPopupMenuCodeSnippet * e = m_pEpilogues->first();e;e = m_pEpilogues->next())
	{
		KviCommand cmde(e->m_szCode,d->pWnd,0,d->pDataContainer);
		if(d->pParamList)cmde.setParams(d->pParamList,false);
		if(!g_pUserParser->parseCommand(&cmde))
		{
			if(cmde.hasError())
			{
				d->pWnd->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken epilogue code for menu '%s', error details follow"),name());
				g_pUserParser->printError(&cmde);
			}
		}
		cmde.forgetExtendedScopeDataContainer();
	}

	lock(false);

}

void KviPopupMenu::testModeItemClickedInternal(KviPopupMenuItem * it)
{
	if(!it)return;
	emit testModeItemClicked(it);
}


void KviPopupMenu::itemClicked(int itemId)
{
	lock(true);
	int param = itemParameter(itemId);
	KviPopupMenuItem * it = m_pItemList->at(param);
	KviPopupMenuTopLevelData * d = topLevelData();
	if(it && d)
	{
		if(d->bTestMode)
		{
			// find the toplevel popup
			KviPopupMenu * p = this;
			while(p->parentPopup())p = p->parentPopup();
			p->testModeItemClickedInternal(it);
		} else {
			// FIXME: #warning "Maybe tell that the window has changed"
			if(!g_pApp->windowExists(d->pWnd))d->pWnd = g_pApp->activeConsole();
			KviCommand cmd(it->code(),d->pWnd,0,d->pDataContainer);
			cmd.setParams(d->pParamList,false);
			if(!g_pUserParser->parseCommand(&cmd))
			{
				if(cmd.hasError())g_pUserParser->printError(&cmd);
			}
			cmd.forgetExtendedScopeDataContainer();
		}
	} else debug("oops....no menu item at position %d",param);
	lock(false);
	// UGLY Qt 3.0.0.... we can't clear menu contents here :(
	//#if QT_VERSION < 300
	//	topLevelPopup()->clearMenuContents();
	//#endif
		//debug("Cleared menu contents");
}


void KviPopupMenu::load(const QString &prefix,KviConfig * cfg)
{
	doClear();

	int cnt;
	int idx;

	QString tmp = prefix;
	tmp.append("_PrologueCount");

	cnt = cfg->readIntEntry(tmp,0);

	if(cnt > 0)
	{
		for(idx = 0;idx < cnt;idx++)
		{
			KviQString::sprintf(tmp,"%Q_Prologue%d",&(prefix),idx);
			QString pr = cfg->readQStringEntry(tmp,"");
			if(!pr.isEmpty())
			{
				KviPopupMenuCodeSnippet * snip = new KviPopupMenuCodeSnippet;
				snip->m_szCode = pr;
				KviQString::sprintf(tmp,"%Q_PrologueId%d",&(prefix),idx);
				snip->m_szId = cfg->readQStringEntry(tmp,"");
				m_pPrologues->append(snip);
			}
		}
	} else {
		// Might be old version!
		KviQString::sprintf(tmp,"%Q_Prologue",&(prefix));
		QString pr = cfg->readQStringEntry(tmp,"");
		if(!pr.isEmpty())
		{
			KviPopupMenuCodeSnippet * snip = new KviPopupMenuCodeSnippet;
			snip->m_szCode = pr;
			m_pPrologues->append(snip);
		}
	}

	KviQString::sprintf(tmp,"%Q_EpilogueCount",&prefix);
	cnt = cfg->readIntEntry(tmp,0);

	if(cnt > 0)
	{
		for(idx = 0;idx < cnt;idx++)
		{
			KviQString::sprintf(tmp,"%Q_Epilogue%d",&prefix,idx);
			QString ep = cfg->readQStringEntry(tmp,"");
			if(!ep.isEmpty())
			{
				KviPopupMenuCodeSnippet * snip = new KviPopupMenuCodeSnippet;
				snip->m_szCode = ep;
				KviQString::sprintf(tmp,"%Q_EpilogueId%d",&(prefix),idx);
				snip->m_szId = cfg->readQStringEntry(tmp,"");
				m_pEpilogues->append(snip);
			}
		}
	} else {
		// Might be old version!
		KviQString::sprintf(tmp,"%Q_Epilogue",&prefix);
		QString ep = cfg->readQStringEntry(tmp,"");
		if(!ep.isEmpty())
		{
			KviPopupMenuCodeSnippet * snip = new KviPopupMenuCodeSnippet;
			snip->m_szCode = ep;
			m_pEpilogues->append(snip);
		}
	}


	KviQString::sprintf(tmp,"%Q_Count",&prefix);

	cnt = cfg->readIntEntry(tmp,0);

	for(idx = 0;idx < cnt;idx++)
	{
		QString pre;
		KviQString::sprintf(pre,"%Q_%d",&prefix,idx);
		KviQString::sprintf(tmp,"%Q_Type",&pre);

		//if(!prefix.isEmpty())debug("PREFIX(%s),PRE(%s),TMP(%s)",prefix.latin1(),pre.latin1(),tmp.latin1());
		int type = cfg->readIntEntry(tmp,3);
		
		KviQString::sprintf(tmp,"%Q_Id",&pre);
		QString id = cfg->readQStringEntry(tmp,QString::null);
		
		switch(type)
		{
			case 0: // separator
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Separator,QString::null,QString::null,QString::null,QString::null,0,id));
			break;
			case 1: // item
			{
				QString text,icon,code,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,QString::null);
				KviQString::sprintf(tmp,"%Q_Code",&pre);
				code = cfg->readQStringEntry(tmp,QString::null);
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,QString::null);
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Item,text,icon,expr,code,0,id));
			}
			break;
			case 2: // menu
			{
				QString text,icon,nome,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,QString::null);
				KviQString::sprintf(tmp,"%Q_Name",&pre);
				nome = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,QString::null);
				KviPopupMenu * pop = new KviPopupMenu(nome);
				pop->load(pre,cfg);
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Menu,text,icon,expr,QString::null,pop,id));
			}
			break;
			case 3: // label
			{
				QString text,icon,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,QString::null);
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,QString::null);
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Label,text,icon,expr,QString::null,0,id));
			}
			break;
			case 4: // extmenu
			{
				QString text,icon,code,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,QString::null);
				KviQString::sprintf(tmp,"%Q_ExtName",&pre);
				code = cfg->readQStringEntry(tmp,QString::null);
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,QString::null);
				addItem(new KviPopupMenuItem(KviPopupMenuItem::ExtMenu,text,icon,expr,code,0,id));
			}
			break;
			default: // ignore
			break;
		}
	}
}

// FIXME: #warning "NOBODY EDITS THE POPUPS IN THE CONFIG!...A binary config would be faster and work better for sure here"

void KviPopupMenu::save(const QString & prefix,KviConfig * cfg)
{
	int idx;

	KviPopupMenuCodeSnippet * s;
	QString tmp;

	KviQString::sprintf(tmp,"%Q_PrologueCount",&prefix);
	cfg->writeEntry(tmp,m_pPrologues->count());

	idx = 0;
	for(s = m_pPrologues->first();s;s = m_pPrologues->next())
	{
		KviQString::sprintf(tmp,"%Q_Prologue%d",&prefix,idx);
		cfg->writeEntry(tmp,s->m_szCode);
		KviQString::sprintf(tmp,"%Q_PrologueId%d",&prefix,idx);
		cfg->writeEntry(tmp,s->m_szId);
		idx++;
	}

	KviQString::sprintf(tmp,"%Q_EpilogueCount",&prefix);
	cfg->writeEntry(tmp,m_pEpilogues->count());

	idx = 0;
	for(s = m_pEpilogues->first();s;s = m_pEpilogues->next())
	{
		KviQString::sprintf(tmp,"%Q_Epilogue%d",&prefix,idx);
		cfg->writeEntry(tmp,s->m_szCode);
		KviQString::sprintf(tmp,"%Q_EpilogueId%d",&prefix,idx);
		cfg->writeEntry(tmp,s->m_szId);
		idx++;
	}

	KviQString::sprintf(tmp,"%Q_Count",&prefix);
	cfg->writeEntry(tmp,m_pItemList->count());
	idx = 0;


	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		QString pre;
		KviQString::sprintf(pre,"%Q_%d",&prefix,idx);
		KviQString::sprintf(tmp,"%Q_Type",&pre);
		int typeCode = 0;
		switch(it->type())
		{
			case KviPopupMenuItem::Label:        typeCode = 3; break;
			case KviPopupMenuItem::Separator:    typeCode = 0; break;
			case KviPopupMenuItem::Menu:         typeCode = 2; break;
			case KviPopupMenuItem::Item:         typeCode = 1; break;
			case KviPopupMenuItem::ExtMenu:      typeCode = 4; break;
		}

		cfg->writeEntry(tmp,typeCode);

		KviQString::sprintf(tmp,"%Q_Id",&pre);
		cfg->writeEntry(tmp,it->id());

		if(!it->isSeparator())
		{
			KviQString::sprintf(tmp,"%Q_Text",&pre);
			cfg->writeEntry(tmp,it->text());
			if(it->hasIcon())
			{
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				cfg->writeEntry(tmp,it->icon());
			}
			if(it->hasExpression())
			{
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				cfg->writeEntry(tmp,it->expr());
			}
			if(it->isMenu())
			{
				KviQString::sprintf(tmp,"%Q_Name",&pre);
				cfg->writeEntry(tmp,it->menu()->name());
				it->menu()->save(pre,cfg);
			} else if(it->isExtMenu())
			{
				KviQString::sprintf(tmp,"%Q_ExtName",&pre);
				cfg->writeEntry(tmp,it->code());
			} else {
				KviQString::sprintf(tmp,"%Q_Code",&pre);
				cfg->writeEntry(tmp,it->code());
			}
		}
		++idx;
	}
}

void KviPopupMenu::generateDefPopupCore(QString &buffer)
{
	QString tmp;

	buffer = "";

	KviPopupMenuCodeSnippet * s;

	for(s = m_pPrologues->first();s;s = m_pPrologues->next())
	{
		buffer.append("prologue");
		if(!s->m_szId.isEmpty())
		{
			buffer.append("(\"");
			buffer.append(s->m_szId);
			buffer.append("\")\n");
		} else {
			buffer.append('\n');
		}
		s->m_szCode.stripWhiteSpace();
		tmp = s->m_szCode;
		KviCommandFormatter::blockFromBuffer(tmp);
		buffer.append(tmp);
		buffer.append('\n');
	}


	for(s = m_pEpilogues->first();s;s = m_pEpilogues->next())
	{
		buffer.append("epilogue");
		if(!s->m_szId.isEmpty())
		{
			buffer.append("(\"");
			buffer.append(s->m_szId);
			buffer.append("\")\n");
		} else {
			buffer.append('\n');
		}
		s->m_szCode.stripWhiteSpace();
		tmp = s->m_szCode;
		KviCommandFormatter::blockFromBuffer(tmp);
		buffer.append(tmp);
		buffer.append('\n');
	}

	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		switch(it->type())
		{
			case KviPopupMenuItem::Item:
				if(!(it->id().isEmpty()))
				{
					if(it->hasIcon())KviQString::appendFormatted(buffer,"item(%Q,\"%Q\",\"%Q\")",&(it->text()),&(it->icon()),&(it->id()));
					else KviQString::appendFormatted(buffer,"item(%Q,,\"%Q\")",&(it->text()),&(it->id()));
				} else {
					if(it->hasIcon())KviQString::appendFormatted(buffer,"item(%Q,\"%Q\")",&(it->text()),&(it->icon()));
					else KviQString::appendFormatted(buffer,"item(%Q)",&(it->text()));
				}
				if(it->hasExpression())KviQString::appendFormatted(buffer," (%Q)",&(it->expr()));
				buffer.append("\n");
				tmp = it->code();
				KviCommandFormatter::blockFromBuffer(tmp);
				buffer.append(tmp);
				buffer.append("\n");
			break;
			case KviPopupMenuItem::Menu:
				if(!(it->id().isEmpty()))
				{
					if(it->hasIcon())KviQString::appendFormatted(buffer,"popup(%Q,\"%Q\",\"%Q\")",&(it->text()),&(it->icon()),&(it->id()));
					else KviQString::appendFormatted(buffer,"popup(%Q,,\"%Q\")",&(it->text()),&(it->id()));
				} else {
					if(it->hasIcon())KviQString::appendFormatted(buffer,"popup(%Q,\"%Q\")",&(it->text()),&(it->icon()));
					else KviQString::appendFormatted(buffer,"popup(%Q)",&(it->text()));
				}
				if(it->hasExpression())KviQString::appendFormatted(buffer," (%Q)",&(it->expr()));
				buffer.append("\n");
				it->menu()->generateDefPopupCore(tmp);
				KviCommandFormatter::blockFromBuffer(tmp);
				buffer.append(tmp);
				buffer.append("\n");
			break;
			case KviPopupMenuItem::Separator:
				if(it->id().isEmpty())
					buffer.append("separator\n\n");
				else
					KviQString::appendFormatted(buffer,"separator(\"%Q\")\n\n",&(it->id()));
			break;
			case KviPopupMenuItem::Label:
				if(it->id().isEmpty())
					KviQString::appendFormatted(buffer,"label(%Q)",&(it->text()));
				else
					KviQString::appendFormatted(buffer,"label(%Q,\"%Q\")",&(it->text()),&(it->id()));
				if(it->hasExpression())KviQString::appendFormatted(buffer," (%Q)",&(it->expr()));
				buffer.append("\n\n");
			break;
			case KviPopupMenuItem::ExtMenu:
				if(!(it->id().isEmpty()))
				{
					if(it->hasIcon())KviQString::appendFormatted(buffer,"extpopup(%Q,\"%Q\",\"%Q\",\"%Q\")",&(it->text()),&(it->code()),&(it->icon()),&(it->id()));
					else KviQString::appendFormatted(buffer,"extpopup(%Q,\"%Q\",\"%Q\")",&(it->text()),&(it->code()),&(it->id()));
				} else {
					if(it->hasIcon())KviQString::appendFormatted(buffer,"extpopup(%Q,\"%Q\",\"%Q\")",&(it->text()),&(it->code()),&(it->icon()));
					else KviQString::appendFormatted(buffer,"extpopup(%Q,\"%Q\")",&(it->text()),&(it->code()));
				}
				if(it->hasExpression())KviQString::appendFormatted(buffer," (%Q)",&(it->expr()));
				buffer.append("\n\n");
			break;
		}
	}
}

void KviPopupMenu::generateDefPopup(QString &buffer)
{
	KviQString::sprintf(buffer,"defpopup(\"%s\")\n",name());

	QString core;

	generateDefPopupCore(core);

	KviCommandFormatter::blockFromBuffer(core);

	buffer.append(core);
}

#include "kvi_popupmenu.moc"
