//
//   File : aliaseditor.cpp
//   Creation date : Mon Dec 23 2002 14:36:18 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the linkss of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#include "aliaseditor.h"

#include "kvi_iconmanager.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_imagedialog.h"
#include "kvi_config.h"
#include "kvi_filedialog.h"
#include "kvi_fileutils.h"
#include "kvi_scripteditor.h"
#include "kvi_debug.h"
#include "kvi_app.h"
#include "kvi_qstring.h"
#include "kvi_kvs_aliasmanager.h"
#include "kvi_filedialog.h"
#include "kvi_cmdformatter.h"

#include <qsplitter.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qtooltip.h>
#include <qpushbutton.h>
#include <qdir.h>
#include <qmessagebox.h>

extern KviAliasEditorWindow * g_pAliasEditorWindow;


KviAliasListViewItem::KviAliasListViewItem(QListView * par,const QString & name,const QString & buffer)
: QListViewItem(par), m_szName(name), m_szBuffer(buffer)
{
	setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS)));
	setText(0,m_szName);
}

KviAliasEditor::KviAliasEditor(QWidget * par)
: QWidget(par)
{
	QGridLayout * l = new QGridLayout(this,1,1,2,2);

	QSplitter * spl = new QSplitter(QSplitter::Horizontal,this);

	l->addWidget(spl,0,0);
	
	QVBox * box = new QVBox(spl);
	m_pListView = new QListView(box);
	m_pListView->addColumn(__tr2qs("Alias"));
	m_pListView->setMultiSelection(false);
	m_pListView->setShowSortIndicator(true);

	QPushButton * pb = new QPushButton(__tr2qs("&Export All To..."),box);
	connect(pb,SIGNAL(clicked()),this,SLOT(exportAllAliases()));

	box = new QVBox(spl);
	m_pNameEditor = new QLineEdit(box);
	QToolTip::add(m_pNameEditor,__tr2qs("Edit the alias name."));
	m_pEditor = KviScriptEditor::createInstance(box);
	m_pEditor->setFocus();

	m_bOneTimeSetupDone = false;
	m_pLastEditedItem = 0;

	m_pContextPopup = new QPopupMenu(this);

	selectionChanged(0);
}

KviAliasEditor::~KviAliasEditor()
{
	KviScriptEditor::destroyInstance(m_pEditor);
}

void KviAliasEditor::oneTimeSetup()
{
	if(m_bOneTimeSetupDone)return;
	m_bOneTimeSetupDone = true;

	QDict<KviKvsScript> * a = KviKvsAliasManager::instance()->aliasDict();
	if(!a)return;

	QDictIterator<KviKvsScript> it(*a);
	KviAliasListViewItem * item;
	while(it.current())
	{
		KviKvsScript * alias = it.current();
		item = new KviAliasListViewItem(m_pListView,alias->name(),alias->code());
		++it;
	}
	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(selectionChanged(QListViewItem *)));
	connect(m_pListView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
		this,SLOT(itemPressed(QListViewItem *,const QPoint &,int)));

}

void KviAliasEditor::itemPressed(QListViewItem *it,const QPoint &pnt,int col)
{
	__range_valid(m_bOneTimeSetupDone);
	m_pContextPopup->clear();

	int id = m_pContextPopup->insertItem(
		*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS)),
		__tr2qs("&New Alias"),
		this,SLOT(newAlias()));

	id = m_pContextPopup->insertItem(
		*(g_pIconManager->getSmallIcon(KVI_SMALLICON_QUIT)),
		__tr2qs("Re&move Alias"),
		this,SLOT(removeCurrentAlias()));
	m_pContextPopup->setItemEnabled(id,it);
	
	id = m_pContextPopup->insertItem(
		*(g_pIconManager->getSmallIcon(KVI_SMALLICON_FOLDER)),
		__tr2qs("&Export Alias To..."),
		this,SLOT(exportCurrentAlias()));
	m_pContextPopup->setItemEnabled(id,it);


	m_pContextPopup->popup(pnt);
}

void KviAliasEditor::getExportAliasBuffer(QString &buffer,KviAliasListViewItem * it)
{
	QString szBuf = it->m_szBuffer;

	KviCommandFormatter::blockFromBuffer(szBuf);
	
	buffer = "alias(";
	buffer += it->m_szName;
	buffer += ")\n";
	buffer += szBuf;
	buffer += "\n";
}

void KviAliasEditor::exportCurrentAlias()
{
	if(!m_pLastEditedItem)return;
	saveLastEditedItem();
	if(!m_pLastEditedItem)return;

	QString szName = QDir::homeDirPath();
	if(!szName.endsWith(QString(KVI_PATH_SEPARATOR)))szName += KVI_PATH_SEPARATOR;
	szName += m_pLastEditedItem->m_szName;
	szName += ".kvs";
	
	QString szFile;
	
	if(!KviFileDialog::askForSaveFileName(szFile,__tr2qs("Choose a Filename - KVIrc"),szName,QString::null,true,true))return;

	QString szOut;
	getExportAliasBuffer(szOut,m_pLastEditedItem);
	
	if(!KviFileUtils::writeFile(szFile,szOut))
	{
		QMessageBox::warning(this,__tr2qs("Write Failed - KVIrc"),__tr2qs("Unable to write to the alias file."),__tr2qs("&OK"));
	}
}

void KviAliasEditor::exportAllAliases()
{
	saveLastEditedItem();

	KviAliasListViewItem * it = (KviAliasListViewItem *)m_pListView->firstChild();

	QString out;

	while(it)
	{
		QString tmp;
		getExportAliasBuffer(tmp,it);
		out += tmp;
		out += "\n";
		it = (KviAliasListViewItem *)it->nextSibling();
	}

	QString szName = QDir::homeDirPath();
	if(!szName.endsWith(QString(KVI_PATH_SEPARATOR)))szName += KVI_PATH_SEPARATOR;
	szName += "aliases.kvs";
	
	QString szFile;
	
	if(!KviFileDialog::askForSaveFileName(szFile,__tr2qs("Choose a Filename - KVIrc"),szName,QString::null,true,true))return;

	if(!KviFileUtils::writeFile(szFile,out))
	{
		QMessageBox::warning(this,__tr2qs("Write Failed - KVIrc"),__tr2qs("Unable to write to the alias file."),__tr2qs("Ok"));
	}
}

void KviAliasEditor::removeCurrentAlias()
{
	if(m_pLastEditedItem)
	{
		KviAliasListViewItem * it = m_pLastEditedItem;
		m_pLastEditedItem = 0;
		delete it;
		if(!m_pLastEditedItem)selectionChanged(0);
	}
}

void KviAliasEditor::newAlias()
{
	QString newName;
	getUniqueHandlerName(0,newName);
	KviAliasListViewItem * it = new KviAliasListViewItem(m_pListView,newName,"");
	m_pListView->setCurrentItem(it);
}

void KviAliasEditor::saveLastEditedItem()
{
	if(m_pLastEditedItem == 0)return;

	QString newName = m_pNameEditor->text();
	if(!KviQString::equalCI(newName,m_pLastEditedItem->m_szName))
	{
		getUniqueHandlerName(m_pLastEditedItem,newName);
	}

	m_pLastEditedItem->m_szName = newName;

	m_pEditor->getText(m_pLastEditedItem->m_szBuffer);
	m_pLastEditedItem->setText(0,newName);
	
}

void KviAliasEditor::selectionChanged(QListViewItem *it)
{
	saveLastEditedItem();

	m_pLastEditedItem = (KviAliasListViewItem *)it;
	if(it)
	{
		m_pNameEditor->setText(((KviAliasListViewItem *)it)->m_szName);
		m_pEditor->setText(((KviAliasListViewItem *)it)->m_szBuffer);
	}

	m_pNameEditor->setEnabled(it);
	m_pEditor->setEnabled(it);
}

void KviAliasEditor::showEvent(QShowEvent *e)
{
	oneTimeSetup();
	QWidget::showEvent(e);
}

void KviAliasEditor::commit()
{
	if(!m_bOneTimeSetupDone)return;

	saveLastEditedItem();

	KviKvsAliasManager::instance()->clear();

	KviAliasListViewItem * it = (KviAliasListViewItem *)m_pListView->firstChild();

	while(it)
	{
		// WARNING: On MSVC operator new here is valid ONLY because
		//          KviKvsScript has a non virtual detructor!
		KviKvsScript * a = new KviKvsScript(it->m_szName,new QString(it->m_szBuffer));
		KviKvsAliasManager::instance()->add(it->m_szName,a);
		it = (KviAliasListViewItem *)it->nextSibling();
	}

	g_pApp->saveAliases();
}


void KviAliasEditor::getUniqueHandlerName(KviAliasListViewItem *item,QString &buffer)
{
	__range_valid(m_bOneTimeSetupDone);

	if(buffer.isEmpty())buffer = "alias";

	QString newName = buffer;

	bool bFound = true;
	int idx = 1;

	while(bFound)
	{
		bFound = false;

		for(KviAliasListViewItem * ch = (KviAliasListViewItem *)(m_pListView->firstChild());ch;ch = (KviAliasListViewItem *)ch->nextSibling())
		{
			if(KviQString::equalCI(ch->m_szName,newName) && (ch != item))
			{
				bFound = true;
				newName.sprintf("%s.%d",buffer.utf8().data(),idx);
				idx++;
				break;
			}
		}

	}

	buffer = newName;
}




KviAliasEditorWindow::KviAliasEditorWindow(KviFrame * lpFrm)
: KviWindow(KVI_WINDOW_TYPE_SCRIPTEDITOR,lpFrm,"aliaseditor",0)
{
	g_pAliasEditorWindow = this;

	setFixedCaption(__tr2qs("Alias Editor"));

	m_pEditor = new KviAliasEditor(this);

	m_pBase = new QWidget(this);
	QGridLayout * g = new QGridLayout(m_pBase,1,4,4,4);

	QPushButton * btn = new QPushButton(__tr2qs("&OK"),m_pBase);
	connect(btn,SIGNAL(clicked()),this,SLOT(okClicked()));
	btn->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ACCEPT)));
	g->addWidget(btn,0,1);

	btn = new QPushButton(__tr2qs("&Apply"),m_pBase);
	connect(btn,SIGNAL(clicked()),this,SLOT(applyClicked()));
	btn->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ACCEPT)));
	g->addWidget(btn,0,2);

	btn = new QPushButton(__tr2qs("Cancel"),m_pBase);
	connect(btn,SIGNAL(clicked()),this,SLOT(cancelClicked()));
	btn->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DISCARD)));
	g->addWidget(btn,0,3);

	g->setColStretch(0,1);
}

KviAliasEditorWindow::~KviAliasEditorWindow()
{
	g_pAliasEditorWindow = 0;
}

void KviAliasEditorWindow::okClicked()
{
	m_pEditor->commit();
	close();
}

void KviAliasEditorWindow::applyClicked()
{
	m_pEditor->commit();
}

void KviAliasEditorWindow::cancelClicked()
{
	close();
}


QPixmap * KviAliasEditorWindow::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS);
}

void KviAliasEditorWindow::resizeEvent(QResizeEvent *e)
{
	int hght = m_pBase->sizeHint().height();
	m_pEditor->setGeometry(0,0,width(),height()- hght);
	m_pBase->setGeometry(0,height() - hght,width(),hght);
}


void KviAliasEditorWindow::getConfigGroupName(KviStr &szName)
{
	szName = "aliaseditor";
}


void KviAliasEditorWindow::saveProperties(KviConfig *cfg)
{
/*
#ifdef COMPILE_SCRIPTTOOLBAR
	cfg->writeEntry("Sizes",m_pEditor->sizes());
	cfg->writeEntry("LastAlias",m_pEditor->lastEditedAlias().ptr());
	//debug("LAST EDITED=%s",m_pEditor->lastEditedAlias().ptr());
#endif // COMPILE_SCRIPTTOOLBAR
*/
}

void KviAliasEditorWindow::loadProperties(KviConfig *cfg)
{
/*
#ifdef COMPILE_SCRIPTTOOLBAR
	QValueList<int> def;
	def.append(20);
	def.append(80);
	m_pEditor->setSizes(cfg->readIntListEntry("Sizes",def));
	KviStr tmp = cfg->readEntry("LastAlias","");
	m_pEditor->editAlias(tmp);
	//debug("LAST EDITED WAS %s",tmp.ptr());
#endif // COMPILE_SCRIPTTOOLBAR
*/
}
