//
//   File : libkviconfig.cpp
//   Creation date : Thu Jan 31 2002 22:50:12 GMT by Szymon Stefanek
//
//   This config is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma@kvirc.net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_string.h"
#include "kvi_config.h"
#include "kvi_fileutils.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_app.h"
#include "kvi_locale.h"

#include <qasciidict.h>

static QAsciiDict<KviConfig> * g_pConfigDict = 0;
static int g_iNextConfigId = 0;

/*
	@doc: config.open
	@type:
		function
	@title:
		$config.open
	@short:
		Opens a config file
	@syntax:
		$config.open(<filename>[,<flags>])
	@description:
		Opens a config file.
		If the file already exists , its contents
		are parsed and available for extracting by the [fnc]$config.read[/fnc]() function.[br]
		<filename> may be an absolute path or a relative path: if a relative path is used,
		KVIrc will complete it with a local kvirc directory suitable for writing script config files.[br]
		If the config file doesn't exist , it is opened as empty config file.[br]
		Flags can contain a combination of letters 'r' and 'w'.[br]
		If only 'r' is specified , the config file is opened in "read-only" mode: no changes will be written to disk.[br]
		If only 'w' is specified , the config file is opened in "write-only" mode: the contents of the file on disk
		are not readed.[br]
		If <flags> are not specified then 'rw' is assumed.[br]
		The function returns an identifier for the open config file. This identifier
		is a mandatory parameter in all the other config.* functions and commands.[br]
		The config section is set to the default section name: you can change it by using
		[cmd]config.setsection[/cmd].[br]
		After you have finished using the file , don't forget to call [cmd]config.close[/cmd]!
		KVIrc tries to not open a config file twice: if the file was already opened then
		the identifier of the open file is returned.[br]
		You can check if a config file is already open by using the
		[fnc]$config.id[/fnc]() function.[br]
	@examples:
		[example]
		# It is a good idea to store the returned id to a variable :)
		%cfg = $config.open(myscript.kvc)
		%num = [fnc]$config.read[/fnc](%cfg,Number,0)
		[cmd]echo[/cmd] Number of entries is %num
		%num++;
		[cmd]config.write[/cmd] %cfg Number %num
		[cmd]config.close[/cmd] %cfg
		[/example]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_open(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_open");
	KviStr szFile = parms->safeFirstParam();
	KviStr szR    = parms->safeNextParam();

	KviConfig::FileMode fm;

	if(szR.contains('r'))
	{
		if(szR.contains('w'))fm = KviConfig::ReadWrite;
		else fm = KviConfig::Read;
	} else {
		if(szR.contains('w'))fm = KviConfig::Write;
		else fm = KviConfig::ReadWrite;
	}

	kvi_adjustFilePath(szFile);

	KviStr szAbsFile;

	if(kvi_isAbsolutePath(szFile))szAbsFile = szFile;
	else g_pApp->getLocalKvircDirectory(szAbsFile,KviApp::ConfigScripts,szFile.ptr(),true);


	QAsciiDictIterator<KviConfig> it(*g_pConfigDict);
	while(it.current())
	{
		if(kvi_strEqualCI(it.current()->fileName(),szAbsFile.ptr()))
		{
			buffer.append(KviStr::Format,"%s",it.currentKey());
			if(it.current()->readOnly() && (fm & KviConfig::Write))
			{
				it.current()->setReadOnly(false);
			}
			return c->leaveStackFrame();
		}
		++it;
	}

	KviConfig * cfg = new KviConfig(szAbsFile.ptr(),fm);
	g_iNextConfigId++;
	KviStr tmp(KviStr::Format,"%d",g_iNextConfigId);
	g_pConfigDict->insert(tmp.ptr(),cfg);
	buffer.append(tmp);
	return c->leaveStackFrame();
}

/*
	@doc: config.id
	@type:
		function
	@title:
		$config.id
	@short:
		Returns the id of an open config file
	@syntax:
		$config.id(<filename>)
	@description:
		Returns the id of an open config file specified by <filename>.[br]
		(This is the same id that is returned by [fnc]$config.open[/fnc].[br]
		If no such file is open, 0 is returned (that is an invalid config identifier).[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_id(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_id");
	KviStr szFile = parms->safeFirstParam();

	kvi_adjustFilePath(szFile);

	KviStr szAbsFile;

	if(kvi_isAbsolutePath(szFile))szAbsFile = szFile;
	else g_pApp->getLocalKvircDirectory(szAbsFile,KviApp::ConfigScripts,szFile.ptr(),true);

	QAsciiDictIterator<KviConfig> it(*g_pConfigDict);
	while(it.current())
	{
		if(kvi_strEqualCI(it.current()->fileName(),szAbsFile.ptr()))
		{
			buffer.append(KviStr::Format,"%s",it.currentKey());
			return c->leaveStackFrame();
		}
	}

	buffer.append('0');
	return c->leaveStackFrame();
}

/*
	@doc: config.read
	@type:
		function
	@title:
		$config.read
	@short:
		Reads an entry from a config file
	@syntax:
		$config.read(<id>,<key>[,<default>])
	@description:
		Reads and returns the value associated to <key> in the current section of the config
		file identified by <id>.[br]
		<id> must be a valid config identifier returned by [fnc]$config.open[/fnc]()
		If no value is associated with <key> in the current section the return value
		is an empty string or the string <default> if it has been passed.[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_read(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_read");
	KviStr szId   = parms->safeFirstParam();
	KviStr szKey  = parms->safeNextParam();
	KviStr szDefault = parms->safeNextParam();

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		buffer.append(cfg->readEntry(QString::fromUtf8(szKey.ptr()),QString::fromUtf8(szDefault.ptr())));
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}

/*
	@doc: config.section
	@type:
		function
	@title:
		$config.section
	@short:
		Returns the name of the current section of a config file
	@syntax:
		$config.section(<id>)
	@description:
		Returns the name of the current section in the config file identifier by <id>.[br]
		<id> must be a valid config identifier returned by [fnc]$config.open[/fnc]()
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_section(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_section");
	KviStr szId   = parms->safeFirstParam();

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		buffer.append(cfg->group());
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: config.readonly
	@type:
		function
	@title:
		$config.readonly
	@short:
		Checks wheter a config file is opened in readonly mode
	@syntax:
		$config.readonly(<id>)
	@description:
		Returns 1 if the config file identified by <id> is opened in readonly mode,
		0 otherwise.[br]
		<id> must be a valid config identifier returned by [fnc]$config.open[/fnc]()
		If <id> does not identify an open config file, a warning is printed and 0 is returned.[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_readonly(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_readonly");
	KviStr szId   = parms->safeFirstParam();

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		buffer.append(cfg->readOnly() ? '1' : '0');
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
		buffer.append('0');
	}

	return c->leaveStackFrame();
}

/*
	@doc: config.filename
	@type:
		function
	@title:
		$config.filename
	@short:
		Returns the filename of an open config file
	@syntax:
		$config.filename(<id>)
	@description:
		Returns the filename of the [b]open[/b] config file identifier by <id>.[br]
		<id> must be a valid config identifier returned by [fnc]$config.open[/fnc]()
		If the <id> doesn't match any open file, and empty string is returned.[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_filename(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_filename");
	KviStr szId   = parms->safeFirstParam();
	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	if(cfg)buffer.append(cfg->fileName());
	return c->leaveStackFrame();
}

/*
	@doc: config.hassection
	@type:
		function
	@title:
		$config.hassection
	@short:
		Checks wheter a section is present in a config file
	@syntax:
		$config.hassection(<id>,<section_name>)
	@description:
		Returns 1 if the section <section_name> exists in the config file identifier by <id>,
		0 otherwise.[br]
		<id> must be a valid config identifier returned by [fnc]$config.open[/fnc]()
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_hassection(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_hassection");
	KviStr szId   = parms->safeFirstParam();
	KviStr szSect = parms->safeNextParam();

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		buffer.append(cfg->hasGroup(QString::fromUtf8(szSect.ptr())) ? '1' : '0');
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}

/*
	@doc: config.sectionlist
	@type:
		function
	@title:
		$config.sectionlist
	@short:
		Returns a list of the sections in a config file
	@syntax:
		$config.sectionlist(<id>)
	@description:
		Returns an array of sections present in the config file identified by <id>.[br]
		<id> must be a valid config identifier returned by [fnc]$config.open[/fnc]()
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_sectionlist(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_sectionlist");
	KviStr szId   = parms->safeFirstParam();

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	c->beginListArrayOrDictionaryReturnIdentifier();

	if(cfg)
	{
		KviConfigIterator it(*(cfg->dict()));
		int id=0;
		while(it.current())
		{
//			if(!bFirst)buffer.append(',');
//			else bFirst = false;
//			buffer.append(it.currentKey());
			c->addListArrayOrDictionaryReturnValue(id++,KviStr(it.currentKey()),buffer);
			++it;
		}
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}

/*
	@doc: config.keylist
	@type:
		function
	@title:
		$config.keylist
	@short:
		Returns a list of keys in a config file
	@syntax:
		$config.keylist(<id>)
	@description:
		Returns an array of keys present in the current section of the config file identified by <id>.[br]
		<id> must be a valid config identifier returned by [fnc]$config.open[/fnc]()
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_keylist(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_keylist");
	KviStr szId   = parms->safeFirstParam();

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());

	c->beginListArrayOrDictionaryReturnIdentifier();

	if(cfg)
	{
		KviConfigGroup * d = cfg->dict()->find(cfg->group());
		if(!d)return c->leaveStackFrame();

		KviConfigGroupIterator it(*d);

//		bool bFirst = true;
		int id=0;
		while(it.current())
		{
//			if(!bFirst)buffer.append(',');
//			else bFirst = false;
//			buffer.append(it.currentKey());
			c->addListArrayOrDictionaryReturnValue(id++,KviStr(it.currentKey()),buffer);
			++it;
		}
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: config.filelist
	@type:
		function
	@title:
		$config.filelist
	@short:
		Returns a list of open config files
	@syntax:
		$config.filelist
	@description:
		Returns an array of open config files.[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_fnc_filelist(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"config_module_fnc_filelist");
	QAsciiDictIterator<KviConfig> it(*g_pConfigDict);

	c->beginListArrayOrDictionaryReturnIdentifier();

//	bool bFirst = true;
	int id=0;
	while(it.current())
	{
//		if(!bFirst)buffer.append(',');
//		else bFirst = false;
//		buffer.append(it.currentKey());
		c->addListArrayOrDictionaryReturnValue(id++,KviStr(it.currentKey()),buffer);
		++it;
	}

	return c->leaveStackFrame();
}


/*
	@doc: config.close
	@type:
		command
	@title:
		config.close
	@keyterms:
		storing & reading configuration entries
	@short:
		Closes a config file
	@syntax:
		config.close [-q] <id>
	@description:
		Closes the config file identified by <id>.[br]
		<id> must be a a valid config file identifier returned by [fnc]$config.open[/fnc].[br]
		If the config file was opened as read-write (default), the changes will be stored
		to disk. If the config was opened as read-only , non changes will be written.[br]
		If the <id> does not match any open config file, a warning is printed unless
		the -q switch is used.[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_cmd_close(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"config_module_cmd_close");
	KviStr szId;
	if(!g_pUserParser->parseCmdFinalPart(c,szId))return false;

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		if(cfg->dirty() && cfg->readOnly())
		{
			if(!c->hasSwitch('q'))c->warning(__tr("The config file '%s' has been changed but is opened as read-only: changes will be lost"),szId.ptr());
		}
		
		g_pConfigDict->remove(szId.ptr()); // byez!
	} else {
		if(!c->hasSwitch('q'))c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}

/*
	@doc: config.flush
	@type:
		command
	@title:
		config.flush
	@keyterms:
		storing & reading configuration entries
	@short:
		Flushes a config file
	@syntax:
		config.flush <id>
	@description:
		Flushes the config file identified by <id>.[br]
		<id> must be a a valid config file identifier returned by [fnc]$config.open[/fnc].[br]
		If the config file was opened as read-write (default), the changes will be stored
		to disk. If the config was opened as read-only ,an error is printed.[br]
		If the <id> does not match any open config file, a warning is printed.
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_cmd_flush(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"config_module_cmd_flush");
	KviStr szId;
	if(!g_pUserParser->parseCmdFinalPart(c,szId))return false;

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		if(cfg->readOnly())c->warning(__tr("The config file with id '%s' is read only"),szId.ptr());
		else cfg->sync();
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}

/*
	@doc: config.clear
	@type:
		command
	@title:
		config.clear
	@keyterms:
		storing & reading configuration entries
	@short:
		Clears a config file
	@syntax:
		config.clear <id>
	@description:
		Clears the contents of the config file identified by <id>.[br]
		All the sections are removed (except for the default section) and all the key=value
		pairs are destroyed.[br]
		<id> must be a a valid config file identifier returned by [fnc]$config.open[/fnc].[br]
		If the <id> does not match any open config file, a warning is printed.
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_cmd_clear(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"config_module_cmd_clear");
	KviStr szId;
	if(!g_pUserParser->parseCmdFinalPart(c,szId))return false;

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		cfg->clear();
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: config.clearsection
	@type:
		command
	@title:
		config.clearsection
	@keyterms:
		storing & reading configuration entries
	@short:
		Clears a section in a config file
	@syntax:
		config.clearsection <id> <section_name>
	@description:
		Clears the contents of the section <section_name> in the config file identified by <id>.[br]
		All the key=value pairs in the section are destroyed.[br]
		<id> must be a a valid config file identifier returned by [fnc]$config.open[/fnc].[br]
		If the <id> does not match any open config file, a warning is printed.
		If <section_name> was also the current section, the current section is set to the default one.[br]
		No warning is printed if the section wasn't existing.[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_cmd_clearsection(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"config_module_cmd_clearsection");
	KviStr szId,szSect;
	if(!g_pUserParser->parseCmdSingleToken(c,szId))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szSect))return false;

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		cfg->clearGroup(QString::fromUtf8(szSect.ptr()));
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: config.write
	@type:
		command
	@title:
		config.write
	@keyterms:
		storing & reading configuration entries
	@short:
		Writes a data field to a config file
	@syntax:
		config.write <id> <key> <value>
	@description:
		Writes the <key>=<value> pair to the config file identifier by <id>.[br]
		<id> must be a valid config file id returned by [fnc]$config.open[/fnc]().[br]
		<key> and <value> can be any strings.[br]
		The <key>=<value> pair is written in the current section of the config file.[br]
		If <key> already exists in the current section of the config , the corresponding value is replaced with <value>.[br]
		If <value> is an empty string, the <key> is simply removed from the current section.[br]
		If a section remains empty (with no keys) at file write time, that section will be removed.[br]
		The changes are NOT written to disk: you must call [cmd]config.flush[/cmd] or (better) [cmd]config.close[/cmd]
		for the changes to be effectively written.[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_cmd_write(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"config_module_cmd_write");
	KviStr szId,szKey,szVal;
	if(!g_pUserParser->parseCmdSingleToken(c,szId))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,szKey))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szVal))return false;

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		cfg->writeEntry(QString::fromUtf8(szKey.ptr()),QString::fromUtf8(szVal.ptr()));
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: config.setsection
	@type:
		command
	@title:
		config.setsection
	@keyterms:
		storing & reading configuration entries
	@short:
		Changes the current section of a config file
	@syntax:
		config.setsection <id> <section_name>
	@description:
		Changes the current section of the config file <id> to <section_name>.[br]
		Sections are effectively namespaces for the keys: each section can contain
		a set of keys and equal keys in different sections do not collide.[br]
		<id> must be a valid config file id returned by [fnc]$config.open[/fnc]().[br]
	@seealso:
		[module:config]Config module documentation[/module]
*/

static bool config_module_cmd_setsection(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"config_module_cmd_setsection");
	KviStr szId,szSec;
	if(!g_pUserParser->parseCmdSingleToken(c,szId))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szSec))return false;

	KviConfig * cfg = g_pConfigDict->find(szId.ptr());
	
	if(cfg)
	{
		cfg->setGroup(QString::fromUtf8(szSec.ptr()));
	} else {
		c->warning(__tr("The config file with id '%s' is not open"),szId.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: config
	@type:
		module
	@short:
		Managing of configuration files
	@title:
		The config module
	@body:
		The config module is a powerful interface that manages text configuration files.[br]
		The files are arranged in named sections and the data is stored in key=value pairs
		belonging to each section.[br]
		[br]
		[section1_name][br]
		<key1>=<value1>[br]
		<key2>=<value2>[br]
		<key3>=<value3>[br]
		....[br]
		[section2_name][br]
		<key1>=<value1>[br]
		....[br]
		[br]
		Keys and values are plain text strings. Eventual newlines in values are encoded using
		a simple hexadecimal notation.[br]
		The config files can be also edited by using any text editor.[br]
		The config module works on config files loaded in memory. For this reason,
		you must first open a config file by the means of [fnc]$config.open[/fnc]().[br]
		Once you open the file , if it exists on disk , its contents are loaded into memory
		and you can access the data stored in it. If the file wasn't existing, and empty
		memory config file is created.[br]
		The config file is opened in read-write mode by default. You can also choose
		the "read-only" method: a read-only config file is never flushed to disk (but you can
		set the values in memory!).[br]
		It is not a good idea to keep a config file open forever: a good approach
		is to keep settings in memory variables and write them all at once when it comes to
		write settings: you open the config file , write all your values and close the file.[br]
		You can obviously keep the config file open for some time but remember that 
		no change is written to the disk until [cmd]config.flush[/cmd] or [cmd]config.close[/cmd] is called.[br]
		You write entries by using [cmd]config.write[/cmd] and read them by using [fnc]$config.read[/fnc]().[br]
		You can change the "current" config section by using [cmd]config.setsection[/cmd].[br]
		Here comes a complete list of commands and functions exported by this module:[br]
		[fnc]$config.open[/fnc]()[br]
		[fnc]$config.read[/fnc]()[br]
		[fnc]$config.section[/fnc]()[br]
		[fnc]$config.hassection[/fnc]()[br]
		[fnc]$config.sectionlist[/fnc]()[br]
		[fnc]$config.keylist[/fnc]()[br]
		[fnc]$config.filelist[/fnc]()[br]
		[fnc]$config.filename[/fnc]()[br]
		[fnc]$config.id[/fnc]()[br]
		[fnc]$config.readonly[/fnc]()[br]
		[cmd]config.close[/cmd][br]
		[cmd]config.flush[/cmd][br]
		[cmd]config.clear[/cmd][br]
		[cmd]config.clearsection[/cmd][br]
		[cmd]config.setsection[/cmd][br]
		[cmd]config.write[/cmd][br]
*/

static bool config_module_init(KviModule * m)
{


	g_pConfigDict = new QAsciiDict<KviConfig>;
	g_pConfigDict->setAutoDelete(true);

	m->registerFunction("open",config_module_fnc_open);
	m->registerFunction("read",config_module_fnc_read);
	m->registerFunction("section",config_module_fnc_section);
	m->registerFunction("hassection",config_module_fnc_hassection);
	m->registerFunction("sectionlist",config_module_fnc_sectionlist);
	m->registerFunction("keylist",config_module_fnc_keylist);
	m->registerFunction("filelist",config_module_fnc_filelist);
	m->registerFunction("filename",config_module_fnc_filename);
	m->registerFunction("id",config_module_fnc_id);
	m->registerFunction("readonly",config_module_fnc_readonly);

	m->registerCommand("close",config_module_cmd_close);
	m->registerCommand("flush",config_module_cmd_flush);
	m->registerCommand("clear",config_module_cmd_clear);
	m->registerCommand("clearsection",config_module_cmd_clearsection);
	m->registerCommand("setsection",config_module_cmd_setsection);
	m->registerCommand("write",config_module_cmd_write);


	return true;
}

static bool config_module_cleanup(KviModule *m)
{
	delete g_pConfigDict;
    g_pConfigDict = 0;
	return true;
}

static bool config_module_can_unload(KviModule *m)
{
	return g_pConfigDict->isEmpty();
}

KVIRC_MODULE(
	"Config",                                               // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Config file system for KVIrc",
	config_module_init,
	config_module_can_unload,
	0,
	config_module_cleanup
)
