//==================================================================================
//
//   File : httpfiletransfer.h
//   Creation date : Tue Apr 22 2003 02:00:12 GMT by Szymon Stefanek
//
//   This config is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//==================================================================================
#include "httpfiletransfer.h"

#include "kvi_app.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_window.h"
#include "kvi_iconmanager.h"
#include "kvi_netutils.h"

#include <qpopupmenu.h>
#include <qpainter.h>

static KviPtrList<KviHttpFileTransfer> * g_pHttpFileTransfers = 0;
static QPixmap * g_pHttpIcon = 0;


KviHttpFileTransfer::KviHttpFileTransfer()
: KviFileTransfer()
{
	init(); // ensure we're initialized
	g_pHttpFileTransfers->append(this);

	m_tStartTime = kvi_unixTime();
	m_tTransferStartTime = 0;
	m_tTransferEndTime = 0;

	m_pHttpRequest = new KviHttpRequest();

	connect(m_pHttpRequest,SIGNAL(status(const char *)),this,SLOT(statusMessage(const char *)));
	connect(m_pHttpRequest,SIGNAL(terminated(bool)),this,SLOT(transferTerminated(bool)));
	connect(m_pHttpRequest,SIGNAL(header(QAsciiDict<KviStr> *)),this,SLOT(headersReceived(QAsciiDict<KviStr> *)));
	connect(m_pHttpRequest,SIGNAL(resolvingHost(const char *)),this,SLOT(resolvingHost(const char *)));
	connect(m_pHttpRequest,SIGNAL(requestSent(const QStringList &)),this,SLOT(requestSent(const QStringList &)));
	connect(m_pHttpRequest,SIGNAL(contactingHost(const char *)),this,SLOT(contactingHost(const char *)));
	connect(m_pHttpRequest,SIGNAL(receivedResponse(const char *)),this,SLOT(receivedResponse(const char *)));
	connect(m_pHttpRequest,SIGNAL(connectionEstabilished()),this,SLOT(connectionEstabilished()));
	
	m_eGeneralStatus = Initializing;
	m_szStatusString = __tr2qs_ctx("Initializing","http");
}

KviHttpFileTransfer::~KviHttpFileTransfer()
{
	g_pHttpFileTransfers->removeRef(this);
	delete m_pHttpRequest;
}

void KviHttpFileTransfer::die()
{
	delete this;
}

QString KviHttpFileTransfer::localFileName()
{
	return m_pHttpRequest->fileName();
}

void KviHttpFileTransfer::abort()
{
	m_pHttpRequest->abort();
}

void KviHttpFileTransfer::fillContextPopup(QPopupMenu * m,int column)
{
	int id = m->insertItem(__tr2qs_ctx("Abort","http"),this,SLOT(abort()));
	if(!active())m->setItemEnabled(id,false);
}

bool KviHttpFileTransfer::active()
{
	return ((m_eGeneralStatus == Connecting) || (m_eGeneralStatus == Downloading));
}

void KviHttpFileTransfer::displayPaint(QPainter * p,int column,int width,int height)
{
	QString txt;
	bool bIsTerminated = ((m_eGeneralStatus == Success) || (m_eGeneralStatus == Failure));

	switch(column)
	{
		case COLUMN_TRANSFERTYPE:
		{
			int offset = 0;
			switch(m_eGeneralStatus)
			{
				case Initializing: offset = 0; break;
				case Connecting: offset = 0; break;
				case Downloading: offset = 48; break;
				case Success: offset = 96; break;
				case Failure: offset = 144; break;
			}
			p->drawPixmap(3,3,*g_pHttpIcon,offset,0,48,64);
		}
		break;
		case COLUMN_FILEINFO:
		{
			QFontMetrics fm(p->font());

			QString szFrom = __tr2qs_ctx("From: ","http");
			QString szTo   = __tr2qs_ctx("To: ","http");

			int daW1 = fm.width(szFrom);
			int daW2 = fm.width(szTo);
			if(daW1 < daW2)daW1 = daW2;
			int iLineSpacing = fm.lineSpacing();

			p->setPen(Qt::black);

			int iY = 4;

			p->drawText(4 + daW1,iY,width - (8 + daW1),height - 8,Qt::AlignTop | Qt::AlignLeft,m_pHttpRequest->url().url().ptr());
			iY += iLineSpacing;
			if(!(m_pHttpRequest->fileName().isEmpty()))
			{
				p->drawText(4 + daW1,iY,width - (8 + daW1),height - 8,Qt::AlignTop | Qt::AlignLeft,m_pHttpRequest->fileName());
			}
			iY += iLineSpacing;


			p->setPen(Qt::darkGray);

			p->drawText(4,4,width - 8,height - 8,Qt::AlignTop | Qt::AlignLeft,szFrom);
			p->drawText(4,4 + iLineSpacing,width - 8,height - 8,Qt::AlignTop | Qt::AlignLeft,szTo);

			p->setPen(QColor(180,180,200));

			iLineSpacing += 2;

			p->drawRect(4,height - (iLineSpacing + 4),width - 8,iLineSpacing);
			p->fillRect(5,height - (iLineSpacing + 3),width - 10,iLineSpacing - 2,bIsTerminated ? QColor(210,210,210) : QColor(190,190,240));
			p->setPen(Qt::black);
			p->drawText(7,height - (iLineSpacing + 4),width - 14,iLineSpacing,Qt::AlignVCenter | Qt::AlignLeft,m_szStatusString);
		}
		break;
		case COLUMN_PROGRESS:
		{
			QFontMetrics fm(p->font());

			unsigned int uTotal = m_pHttpRequest->totalSize();
			unsigned int uRecvd = m_pHttpRequest->receivedSize();
			int iW = width - 8;

			p->setPen(bIsTerminated ? Qt::lightGray : QColor(210,210,240));
			p->drawRect(4,4,iW,12);

			int iAvgSpeed = -1;
			int iEta = -1;

			if(m_tTransferStartTime > 0)
			{
				int tSpan = kvi_timeSpan(m_tTransferEndTime > 0 ? m_tTransferEndTime : kvi_unixTime(),m_tTransferStartTime);
				if(tSpan > 0)
				{
					//debug("SPAN: %d (%d - %d)",tSpan,m_tTransferEndTime > 0 ? m_tTransferEndTime : kvi_unixTime(),m_tTransferStartTime);
					iAvgSpeed = uRecvd / tSpan;
					if(!bIsTerminated && (uTotal >= uRecvd))
					{
						unsigned int uRemaining = uTotal - uRecvd;
						iEta = uRemaining / iAvgSpeed;
					}
				}
			}

			if(uTotal > 0)
			{
				double dPerc = (double)(((double)uRecvd) * 100.0) / (double)uTotal;
				iW -= 2;
				int iL = (int) ((((double)iW) * dPerc) / 100.0);
				//iR = iW - iL;
				p->fillRect(5,5,iL,10,bIsTerminated ? QColor(140,110,110) : QColor(200,100,100));

				txt = QString(__tr2qs_ctx("%1 of %2 bytes (%3 %)","http")).arg(uRecvd).arg(uTotal).arg(dPerc,0,'f',2);
			} else {
				txt = QString(__tr2qs_ctx("%1 bytes","http")).arg(m_pHttpRequest->receivedSize());
			}

			p->setPen(Qt::black);

			p->drawText(4,19,width - 8,height - 8,Qt::AlignTop | Qt::AlignLeft,txt);

			int iLeftHalf = (iW - 2) / 2;
			int iRightHalf = iW - (iLeftHalf + 1);
			int iLineSpacing = fm.lineSpacing() + 2;

			/*
			txt = __tr2qs_ctx("Spd:","dcc");
			txt += " ";
			if(iInstantSpeed >= 0)
			{
				QString tmpisp;
				KviNetUtils::formatNetworkBandwidthString(tmpisp,iAvgSpeed);
				txt += tmpisp;
			} else {
				txt += "? B/s";
			}
			*/

			txt = __tr2qs_ctx("Avg:","dcc");
			txt += " ";
			if(iAvgSpeed >= 0)
			{
				QString tmpspd;
				KviNetUtils::formatNetworkBandwidthString(tmpspd,iAvgSpeed);
				txt += tmpspd;
			} else {
				txt += "? B/s";
			}



			int iDaH = height - (iLineSpacing + 4);

			p->setPen(QColor(180,180,200));
			p->drawRect(4,iDaH,iLeftHalf,iLineSpacing);
			p->fillRect(5,iDaH + 1,iLeftHalf - 2,iLineSpacing - 2,bIsTerminated ? QColor(210,210,210) : QColor(190,190,240));
			p->setPen(bIsTerminated ? Qt::darkGray : Qt::black);
			p->drawText(6,iDaH,iLeftHalf - 4,iLineSpacing,Qt::AlignLeft | Qt::AlignVCenter,txt);

			unsigned int uD,uH,uM,uS;

			if(bIsTerminated)
			{
				KviTimeUtils::secondsToDaysHoursMinsSecs(kvi_timeSpan(m_tTransferEndTime,m_tTransferStartTime),&uD,&uH,&uM,&uS);
				txt = "TOT: ";
				if(uD > 0)txt += QString(__tr2qs_ctx("%1d %2h %3m %4s","http")).arg(uD).arg(uH).arg(uM).arg(uS);
				else if(uH > 0)txt += QString(__tr2qs_ctx("%2h %3m %4s","http")).arg(uH).arg(uM).arg(uS);
				else txt += QString(__tr2qs_ctx("%3m %4s","http")).arg(uM).arg(uS);
			} else {
				if(iEta >= 0)
				{
					KviTimeUtils::secondsToDaysHoursMinsSecs(iEta,&uD,&uH,&uM,&uS);
					txt = "ETA: ";
					if(uD > 0)txt += QString(__tr2qs_ctx("%1d %2h %3m %4s","http")).arg(uD).arg(uH).arg(uM).arg(uS);
					else if(uH > 0)txt += QString(__tr2qs_ctx("%2h %3m %4s","http")).arg(uH).arg(uM).arg(uS);
					else txt += QString(__tr2qs_ctx("%3m %4s","http")).arg(uM).arg(uS);
				} else {
					txt = "ETA: Unknown";
				}
			}

			p->setPen(QColor(180,180,200));
			p->drawRect(width - (4 + iRightHalf),iDaH,iRightHalf,iLineSpacing);
			p->fillRect(width - (3 + iRightHalf),iDaH + 1,iRightHalf - 2,iLineSpacing - 2,bIsTerminated ? QColor(210,210,210) : QColor(190,190,240));
			p->setPen(bIsTerminated ? Qt::darkGray : Qt::black);
			p->drawText(width - (2 + iRightHalf),iDaH,iRightHalf - 4,iLineSpacing,Qt::AlignLeft | Qt::AlignVCenter,txt);

		}
		break;
	}
}

int KviHttpFileTransfer::displayHeight(int iLineSpacing)
{
	int iH = (iLineSpacing * 3) + 10;
	return iH >= 70 ? iH : 70;
}

QString KviHttpFileTransfer::tipText()
{
	QString s;
	s = QString("<table><tr><td bgcolor=\"#000000\"><font color=\"#FFFFFF\"><b>HTTP Transfer (ID %1)</b></font></td></tr>").arg(id());

	if(m_lRequest.count() > 0)
	{
		s += "<tr><td bgcolor=\"#404040\"><font color=\"#FFFFFF\">Request Headers</font></td></tr>";
		s += "<tr><td bgcolor=\"#C0C0C0\">";
		for(QStringList::ConstIterator it = m_lRequest.begin();it != m_lRequest.end();++it)
		{
			s += "&nbsp; &nbsp;";
			s += *it;
			s += "<br>";
		}
		s += "</td></tr>";
	}

	if(m_lHeaders.count() > 0)
	{
		s += "<tr><td bgcolor=\"#404040\"><font color=\"#FFFFFF\">Response Headers</font></td></tr>";
		s += "<tr><td bgcolor=\"#C0C0C0\">";
		for(QStringList::ConstIterator it = m_lHeaders.begin();it != m_lHeaders.end();++it)
		{
			s += "&nbsp; &nbsp;";
			s += *it;
			s += "<br>";
		}
		s += "</td></tr>";
	}

	s += "<table>";

	return s;
}

void KviHttpFileTransfer::init()
{
	if(g_pHttpFileTransfers)return;
	g_pHttpFileTransfers = new KviPtrList<KviHttpFileTransfer>;
	g_pHttpFileTransfers->setAutoDelete(false);

	QPixmap * pix = g_pIconManager->getImage("kvi_httpicons.png");
	if(pix)g_pHttpIcon = new QPixmap(*pix);
	else g_pHttpIcon = new QPixmap(192,48);
}

void KviHttpFileTransfer::done()
{
	if(!g_pHttpFileTransfers)return;
	while(KviHttpFileTransfer * t = g_pHttpFileTransfers->first())
		delete t;
	delete g_pHttpFileTransfers;
	g_pHttpFileTransfers = 0;
	delete g_pHttpIcon;
	g_pHttpIcon = 0;
}

unsigned int KviHttpFileTransfer::runningTransfers()
{
	if(!g_pHttpFileTransfers)return 0;
	return g_pHttpFileTransfers->count();
}

void KviHttpFileTransfer::requestSent(const QStringList &requestHeaders)
{
	m_szStatusString = __tr2qs_ctx("Request sent, waiting for reply...","http");
	displayUpdate();

	KviWindow * out = transferWindow();
	if(!out)return;

	out->output(KVI_OUT_GENERICSTATUS,__tr2qs_ctx("[HTTP %d]: Request data sent:","http"),id());

	for(QStringList::ConstIterator it = requestHeaders.begin();it != requestHeaders.end();++it)
	{
		out->output(KVI_OUT_GENERICSTATUS,"[HTTP %d]:   %s",id(),(*it).latin1());
	}

	m_lRequest = requestHeaders;
}

void KviHttpFileTransfer::connectionEstabilished()
{
	m_szStatusString = __tr2qs_ctx("Connection established, sending request","http");
	displayUpdate();
}

void KviHttpFileTransfer::resolvingHost(const char *hostname)
{
	m_szStatusString = __tr2qs_ctx("Resolving host %1","http").arg(hostname);
	displayUpdate();
}

void KviHttpFileTransfer::contactingHost(const char *ipandport)
{
	m_szStatusString = __tr2qs_ctx("Contacting host %1","http").arg(ipandport);
	displayUpdate();
}

void KviHttpFileTransfer::receivedResponse(const char * response)
{
	m_lHeaders.clear();
	m_lHeaders.append(response);
	m_szStatusString = __tr2qs_ctx("Transferring data (%1)","http").arg(response);
	m_tTransferStartTime = kvi_unixTime();
	m_eGeneralStatus = Downloading;
	displayUpdate();
}

void KviHttpFileTransfer::statusMessage(const char *txt)
{
	KviWindow * out = transferWindow();
	if(out)out->output(KVI_OUT_GENERICSTATUS,"[HTTP %d]: %s",id(),txt);
}

void KviHttpFileTransfer::transferTerminated(bool bSuccess)
{
	KviWindow * out = transferWindow();

	m_tTransferEndTime = kvi_unixTime();

	if(bSuccess)
	{
		m_szStatusString = __tr2qs_ctx("Transfer completed","http");
		m_eGeneralStatus = Success;
		displayUpdate();
		if(out)out->output(KVI_OUT_GENERICSUCCESS,__tr2qs_ctx("[HTTP %d]: Transfer completed","http"),id());
		g_pApp->fileDownloadTerminated(true,m_pHttpRequest->url().url().ptr(),m_pHttpRequest->fileName());
	} else {
		m_szStatusString = __tr2qs_ctx("Transfer failed","http");
		m_szStatusString += ": ";
		m_szStatusString += m_pHttpRequest->lastError().ptr();
		m_eGeneralStatus = Failure;
		displayUpdate();
		if(out)out->output(KVI_OUT_GENERICERROR,__tr2qs_ctx("[HTTP %d]: Transfer failed: %s","http"),id(),m_pHttpRequest->lastError().ptr());
		g_pApp->fileDownloadTerminated(false,m_pHttpRequest->url().url().ptr(),m_pHttpRequest->fileName(),QString::null,m_pHttpRequest->lastError().ptr());
	}
}

void KviHttpFileTransfer::headersReceived(QAsciiDict<KviStr> *h)
{
	if(!h)return;
	KviWindow * out = transferWindow();

	if(out)out->output(KVI_OUT_GENERICSTATUS,__tr2qs_ctx("[HTTP %d]: Response headers:","http"),id());
	QAsciiDictIterator<KviStr> it(*h);
	while(KviStr * s = it.current())
	{
		QString szHeader = it.currentKey();
		szHeader += ": ";
		szHeader += s->ptr();
		m_lHeaders.append(szHeader);
		if(out)out->output(KVI_OUT_GENERICSTATUS,"[HTTP %d]:   %s: %s",id(),it.currentKey(),s->ptr());
		++it;
	}
}

bool KviHttpFileTransfer::startDownload()
{
	m_eGeneralStatus = Connecting;
	return m_pHttpRequest->start();
}

#include "m_httpfiletransfer.moc"
