//vim: ts=8
//   File : class_combobox.cpp
//   Creation date : Thu Mar 22 20:57:45 CET 2001 by Krzysztof Godlewski
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_debug.h"
#include "kvi_scriptobject.h"

#include "class_list.h"
#include "class_combobox.h"

static KviScriptObjectClass * g_pComboBoxClass = 0;

//extern KviScriptObjectController * g_pScriptObjectController;

static KviScriptObject * comboBoxClassCreateInstance(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n)
{
	return new KviScriptComboBoxObject(c, p, n);
}

KviScriptComboBoxObject::KviScriptComboBoxObject(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n) : KviScriptWidgetObject(c,p,n)
{
}

KviScriptComboBoxObject::~KviScriptComboBoxObject()
{
}

/*
	@doc:	combobox
	@keyterms:
		combobox object class, selection
	@title:
		combobox class
	@type:
		class
	@short:
		Combined button and popup list
	@inherits:
		[class]object[/class]
		[class]widget[/class]
	@description:
	@functions:
		!fn: $insertList(<list>, <index>)
		Inserts the items present in the <list> object starting from 
		index <index>, which is -1 by default (which means the items 
		will be appended).
		!fn: $insertItem(<str>, <index>)
		Inserts a text item at position <index>. If index is negative or
		not specified the item is appended.
		!fn: $changeItem(<str>, <index>)
		Changes text of item at <index> to <str>.
		!fn: $removeItem(<index>)
		Removes item at given index.
		!fn: $setMaxCount(<num>)
		Sets the maximum number of items the combo box can hold to 
		<num>. If <num> is smaller than the current number of items, 
		the list is truncated at the end. There is no limit by default.
		!fn: $maxCount()
		Returns the current maximum number of items in the combo box. 
		!fn: $count()
		Returns number of items in the widget.
		!fn: $current()
		Returns currently selected item.
		!fn: $currentItem()
		Returns index of currently selected item.
		!fn: $setEditable(<bool>)
		Make the input field editable, if <bool>. Otherwise the 
		user may only choose one of the items in the combo box.[br]
		If the parameter is ommited, it is assumed to be false.
		!fn: $editable()
		Returns whether the combobox is editable or not.
		!fn: $setEditText(<str>)
		Sets the text in the embedded line edit to newText without 
		changing the combo's contents. Does nothing if the combo 
		isn't editable.
		!fn: $textAt(<index>)
		Returns item at given index.
		!fn: $setCurrentItem(<index>)
		Sets the current combobox item. This is the item to be 
		displayed on the combobox button.
*/

bool KviScriptComboBoxObject::init(KviCommand *)
{
	setObject(new QComboBox(parentScriptWidget(), name()), true);
	return true;
}

#define comboboxFuncReg(__nam, __func) \
        g_pComboBoxClass->registerFunctionHandler(__nam, \
        (KviScriptObjectFunctionHandlerProc)( \
		KVI_PTR2MEMBER(KviScriptComboBoxObject::__func)), 0, true);

void KviScriptComboBoxObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("widget");
	__range_valid(base);

	g_pComboBoxClass = new KviScriptObjectClass(base, "combobox",
		comboBoxClassCreateInstance, true);

	comboboxFuncReg("insertList", functionInsertList);
	comboboxFuncReg("insertItem", functionInsertItem);
	comboboxFuncReg("changeItem", functionChangeItem);
	comboboxFuncReg("removeItem", functionRemoveItem);
	comboboxFuncReg("setMaxCount", functionSetMaxCount);
	comboboxFuncReg("maxCount", functionMaxCount);
	comboboxFuncReg("count", functionCount);
	comboboxFuncReg("current", functionCurrent);
	comboboxFuncReg("currentIndex", functionCurrentIndex);
	comboboxFuncReg("setEditable", functionEditable);
	comboboxFuncReg("editable", functionEditable);
	comboboxFuncReg("setEditText", functionSetEditText);
	comboboxFuncReg("textAt", functionTextAt);
	comboboxFuncReg("setCurrentItem", functionSetCurrentItem);
}

void KviScriptComboBoxObject::unregisterSelf()
{
	delete g_pComboBoxClass;
    g_pComboBoxClass = 0;
}

bool KviScriptComboBoxObject::functionInsertList(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "combobox::insertList");

	if(!widget())
		return c->leaveStackFrame();
		
	if(!p->count())
		return c->error(KviError_notEnoughParameters);
	
	KviScriptListObject * pList = (KviScriptListObject *)g_pScriptObjectController-> \
		lookupObject(p->first()->ptr());
	(void) (p->next()); //Noldor
	bool bOk = true;
	//int idx = (p->count() == 1 ? -1 : p->getInt(&bOk));

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	if(!pList)
		return c->error(KviError_noSuchObject);

	for(KviStr * str = pList->dataList()->first(); str; str = \
		pList->dataList()->next())
		((QComboBox *)widget())->insertItem(str->ptr());
	
	return c->leaveStackFrame();
}

bool KviScriptComboBoxObject::functionInsertItem(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "combobox::insertItem");
	
	if(!widget())
		return c->leaveStackFrame();

	if(!p->count())
		return c->error(KviError_notEnoughParameters);
		
	KviStr * pStr = p->safeFirst();
	(void) (p->next()); //Noldor
	bool bOk = true;
	int idx = (p->count() == 1 ? -1 : p->getInt(&bOk));

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	((QComboBox *)widget())->insertItem(pStr->ptr(), idx);
	
	return c->leaveStackFrame();
}

bool KviScriptComboBoxObject::functionChangeItem(KviCommand * c, KviParameterList * p,
	KviStr &)
{
	ENTER_STACK_FRAME(c, "combobox::changeItem");
	if(!widget())
		return c->leaveStackFrame();

	KviStr * pStr;
	
	if(!(pStr = p->safeFirst()))
		c->warning("No string parameter given - using empty string");
	(void) (p->next()); //Noldor
	bool bOk;
	int idx = p->getInt(&bOk);
	
	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	int cnt;
	if(idx >= (cnt = ((QComboBox *)widget())->count()))
	{
		c->warning("Item index [%d] is too big - defaulting to " \
			"$count() - 1 [%d]", idx, cnt);
		idx = cnt - 1;
	}

	((QComboBox *)widget())->changeItem(pStr->ptr(), idx);
		
	return c->leaveStackFrame();
}

bool KviScriptComboBoxObject::functionRemoveItem(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "combobox::removeItem");

	if(!widget())
		return c->leaveStackFrame();

	bool bOk;
	int idx = p->getInt(&bOk);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	int cnt;
	if(idx >= (cnt = ((QComboBox *)widget())->count()))
		c->warning("Item index [%d] is too big - defaulting to " \
			"$count() - 1 [%d]", idx, cnt);

	((QComboBox *)widget())->removeItem(idx);

	return c->leaveStackFrame();
}

bool KviScriptComboBoxObject::functionSetMaxCount(KviCommand * c, \
	KviParameterList * p, KviStr &)
{
	ENTER_STACK_FRAME(c, "combobox::setMaxCount");
	
	bool bOk;
	int cnt = p->getInt(&bOk);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	((QComboBox *)widget())->setMaxCount(cnt);
		
	return c->leaveStackFrame();
}

bool KviScriptComboBoxObject::functionMaxCount(KviCommand * c, KviParameterList *, \
	KviStr & b)
{
	if(!widget())
		b.append(KviStr::Format, "%d", \
			((QComboBox *)widget())->maxCount());
	return true;
}

bool KviScriptComboBoxObject::functionCount(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d",((QComboBox *)widget())->count());

	return true;
}

bool KviScriptComboBoxObject::functionCurrent(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%s", ((QComboBox *)widget())-> \
			currentText().local8Bit().data());
	return true;
}

bool KviScriptComboBoxObject::functionCurrentIndex(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QComboBox *)widget())-> \
			currentItem());
	return true;	
}

bool KviScriptComboBoxObject::functionSetEditable(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	if(widget())
		((QComboBox *)widget())->setEditable(p->getBool());

	return true;
}

bool KviScriptComboBoxObject::functionEditable(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(((QComboBox *)widget())->editable() ? '1' : '0');
	return true;
}

bool KviScriptComboBoxObject::functionSetEditText(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "combobox::setEditText");

	KviStr * pStr;
	if(!(pStr = p->safeFirst()))
		c->warning("No string parameter given - using empty string");

	((QComboBox *)widget())->setEditText(pStr->ptr());
		
	return c->leaveStackFrame();
}

bool KviScriptComboBoxObject::functionTextAt(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "combobox::textAt");
	
	if(widget())
	{
		bool bOk;
		int idx = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);
			
		b.append(((QComboBox *)widget())->text(idx));
	}

	return c->leaveStackFrame();
}

bool KviScriptComboBoxObject::functionSetCurrentItem(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "combobox::setCurrentItem");
	
	if(widget())
	{
		bool bOk;
		int idx = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);
			
		((QComboBox *)widget())->setCurrentItem(idx);
	}

	return c->leaveStackFrame();
}

#include "m_class_combobox.moc"
