//
//   File : class_layout.cpp
//   Creation date : Fri Now 22 2002 00:50:01 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_command.h"
#include "kvi_locale.h"

#include "class_layout.h"


static KviScriptObjectClass * g_pLayoutClass = 0;

static KviScriptObject * layoutClassCreateInstance(KviScriptObjectClass * cls,KviScriptObject *par,const char * nam)
{
	return new KviScriptLayoutObject(cls,par,nam);
}

KviScriptLayoutObject::KviScriptLayoutObject(KviScriptObjectClass * cla,KviScriptObject * par,const char * nam)
: KviScriptObject(cla,par,nam)
{
}

KviScriptLayoutObject::~KviScriptLayoutObject()
{
}


/*
	@doc: layout
	@keyterms:
		layout object class, child widgets
	@title:
		layout class
	@type:
		class
	@short:
		Manages child widget geometry
	@inherits:
		[class]object[/class]
	@description:
		The layout is a geometry management tool for child widgets.
		You create a layout , give it some widgets to manage and it will layout them
		automatically.[br]
		The parent of the layout must be the widget for which child widget geometries have to be managed.
		A layout is a grid of NxM cells in which you insert child widgets with [classfnc:layout]$addWidget[/classfnc]().[br]
		Widgets that must span multiple cells can be added to the layout with [classfnc:layout]$addMultiCellWidget[/classfnc]().[br]
	@functions:
		!fn: $addWidget(<widget>,<row>,<column>)
		Adds a widget to this layout placing it at position <row>,<column> in the grid
		!fn: $addMultiCellWidget(<widet>,<from_row>,<to_row>,<from_col>,<to_col>)
		Adds a widget to this layout spanning multiple grid cells
		!fn: $setRowStretch(<row>,<stretch_value>)
		Sets the stretch value for a particular row of this layout. The <stretch_value>
		must be a positive integer. The rows with bigger stretch values will take more space
		in the layout.
		!fn: $setColStretch(<column>,<stretch_value>)
		Sets the stretch value for a particular column in this layout. The <stretch_value>
		must be a positive integer. The rows with bigger stretch values will take more space
		in the layout.
		!fn: $addRowSpacing(<row>,<spacing>)
		Sets the minimum height of the specified <row> to <spacing> which must be a positive integer
		!fn: $addColSpacing(<column>,<spacing>)
		Sets the minimum width of the specigfied <column> to <spacing> which must be a positive integer
		!fn: $setSpacing(<value>)
		Sets the default spacing of the widgets in pixels
		!fn: $setMargin(<value>)
		Sets the dimension of the layout margin : the distance from the border to the outermost child widget edges.
		!fn: $setResizeMode(<mode>)
		Sets the resize mode of the parent widget in relation to this layout.
		<mode> can be one of:<br>
		Auto: this is the default<br>
		Fixed: the parent widget of this layout is resized to the "sizeHint" value and it cannot be resized by the user.<br>
		Minimum: the minimum size of the parent widget of this layout is set to minimumSize() and it cannot be smaller<br>
		FreeResize: the parent widget of this layout is not constrained at all<br>
*/

bool KviScriptLayoutObject::init(KviCommand *c)
{
	QWidget * w = parentScriptWidget();
	if(!w)
	{
		c->warning(__tr("The parent of a layout must be a widget!"));
		return false;
	}
	setObject(new QGridLayout(w));
	return true;
}

void KviScriptLayoutObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController->lookupClass("object");
	__range_valid(base);
	g_pLayoutClass = new KviScriptObjectClass(base,"layout",layoutClassCreateInstance,true);

	g_pLayoutClass->registerFunctionHandler("addWidget",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionAddWidget)),0,true);
	g_pLayoutClass->registerFunctionHandler("addMultiCellWidget",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionAddMultiCellWidget)),0,true);
	g_pLayoutClass->registerFunctionHandler("setRowStretch",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionSetRowStretch)),0,true);
	g_pLayoutClass->registerFunctionHandler("setColStretch",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionSetColStretch)),0,true);
	g_pLayoutClass->registerFunctionHandler("addRowSpacing",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionAddRowSpacing)),0,true);
	g_pLayoutClass->registerFunctionHandler("addColSpacing",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionAddColSpacing)),0,true);
	g_pLayoutClass->registerFunctionHandler("setMargin",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionSetMargin)),0,true);
	g_pLayoutClass->registerFunctionHandler("setSpacing",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionSetSpacing)),0,true);
	g_pLayoutClass->registerFunctionHandler("setResizeMode",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLayoutObject::functionSetResizeMode)),0,true);
}

void KviScriptLayoutObject::unregisterSelf()
{
	delete g_pLayoutClass; // this will delete all the objects of this class
    g_pLayoutClass = 0;
}

bool KviScriptLayoutObject::functionAddWidget(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::addWidget");

	KviStr * pWidget = params->safeFirst();
	KviStr * pRow = params->safeNext();
	KviStr * pCol = params->safeNext();

	KviScriptObject * ob = g_pScriptObjectController->lookupObject(pWidget->ptr());
	if(!ob)
	{
		c->warning(__tr("Can't find the widget object to add (%s)"),pWidget->ptr());
		return c->leaveStackFrame();
	}

	if(!ob->object())
	{
		c->warning(__tr("Invalid object"));
		return c->leaveStackFrame();
	}

	if(!ob->object()->isWidgetType())
	{
		c->warning(__tr("Can't add a non-widget object"));
		return c->leaveStackFrame();
	}

	bool bOk;
	unsigned int uRow = pRow->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid row parameter (%s)"),pRow->ptr());
		return c->leaveStackFrame();
	}

	unsigned int uCol = pCol->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid column parameter (%s)"),pCol->ptr());
		return c->leaveStackFrame();
	}

	((QGridLayout *)object())->addWidget(((QWidget *)(ob->object())),uRow,uCol);

	return c->leaveStackFrame();
}

bool KviScriptLayoutObject::functionSetResizeMode(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::setResizeMode");

	KviStr * pVal = params->safeFirst();

	QLayout::ResizeMode r = QLayout::Auto;
	if(kvi_strEqualCI(pVal->ptr(),"FreeResize"))r = QLayout::FreeResize;
	else if(kvi_strEqualCI(pVal->ptr(),"Minimum"))r = QLayout::Minimum;
	else if(kvi_strEqualCI(pVal->ptr(),"Fixed"))r = QLayout::Fixed;
	else c->warning(__tr("Invalid resize mode (%s): defaulting to Auto"),pVal->ptr());

	((QGridLayout *)object())->setResizeMode(r);

	return c->leaveStackFrame();
}

bool KviScriptLayoutObject::functionSetMargin(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::setMargin");

	KviStr * pVal = params->safeFirst();

	bool bOk;
	unsigned int uVal = pVal->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid margin value (%s)"),pVal->ptr());
		return c->leaveStackFrame();
	}

	((QGridLayout *)object())->setMargin(uVal);

	return c->leaveStackFrame();
}


bool KviScriptLayoutObject::functionSetSpacing(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::setSpacing");

	KviStr * pVal = params->safeFirst();

	bool bOk;
	unsigned int uVal = pVal->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid spacing value (%s)"),pVal->ptr());
		return c->leaveStackFrame();
	}

	((QGridLayout *)object())->setSpacing(uVal);

	return c->leaveStackFrame();
}

bool KviScriptLayoutObject::functionAddRowSpacing(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::addRowSpacing");

	KviStr * pRow = params->safeFirst();
	KviStr * pVal = params->safeNext();

	bool bOk;
	unsigned int uRow = pRow->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid row parameter (%s)"),pRow->ptr());
		return c->leaveStackFrame();
	}

	unsigned int uVal = pVal->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid spacing value (%s)"),pVal->ptr());
		return c->leaveStackFrame();
	}

	((QGridLayout *)object())->addRowSpacing(uRow,uVal);

	return c->leaveStackFrame();
}


bool KviScriptLayoutObject::functionSetRowStretch(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::setRowStretch");

	KviStr * pRow = params->safeFirst();
	KviStr * pVal = params->safeNext();

	bool bOk;
	unsigned int uRow = pRow->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid row parameter (%s)"),pRow->ptr());
		return c->leaveStackFrame();
	}

	unsigned int uVal = pVal->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid stretch value (%s)"),pVal->ptr());
		return c->leaveStackFrame();
	}

	((QGridLayout *)object())->setRowStretch(uRow,uVal);

	return c->leaveStackFrame();
}

bool KviScriptLayoutObject::functionSetColStretch(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::setColStretch");

	KviStr * pCol = params->safeFirst();
	KviStr * pVal = params->safeNext();

	bool bOk;
	unsigned int uCol = pCol->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid column parameter (%s)"),pCol->ptr());
		return c->leaveStackFrame();
	}

	unsigned int uVal = pVal->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid stretch value (%s)"),pVal->ptr());
		return c->leaveStackFrame();
	}

	((QGridLayout *)object())->setRowStretch(uCol,uVal);

	return c->leaveStackFrame();
}


bool KviScriptLayoutObject::functionAddColSpacing(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::addColSpacing");

	KviStr * pCol = params->safeFirst();
	KviStr * pVal = params->safeNext();

	bool bOk;
	unsigned int uCol = pCol->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid column parameter (%s)"),pCol->ptr());
		return c->leaveStackFrame();
	}

	unsigned int uVal = pVal->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid spacing value (%s)"),pVal->ptr());
		return c->leaveStackFrame();
	}

	((QGridLayout *)object())->addColSpacing(uCol,uVal);

	return c->leaveStackFrame();
}

bool KviScriptLayoutObject::functionAddMultiCellWidget(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"layout::addMulciCellWidget");

	KviStr * pWidget = params->safeFirst();
	KviStr * pFromRow = params->safeNext();
	KviStr * pToRow = params->safeNext();
	KviStr * pFromCol = params->safeNext();
	KviStr * pToCol = params->safeNext();

	KviScriptObject * ob = g_pScriptObjectController->lookupObject(pWidget->ptr());
	if(!ob)
	{
		c->warning(__tr("Can't find the widget object to add (%s)"),pWidget->ptr());
		return c->leaveStackFrame();
	}

	if(!ob->object())
	{
		c->warning(__tr("Invalid object"));
		return c->leaveStackFrame();
	}

	if(!ob->object()->isWidgetType())
	{
		c->warning(__tr("Can't add a non-widget object"));
		return c->leaveStackFrame();
	}

	bool bOk;
	unsigned int uFromRow = pFromRow->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid from_row parameter (%s)"),pFromRow->ptr());
		return c->leaveStackFrame();
	}


	unsigned int uToRow = pToRow->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid to_row parameter (%s)"),pToRow->ptr());
		return c->leaveStackFrame();
	}


	unsigned int uFromCol = pFromCol->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid from_column parameter (%s)"),pFromCol->ptr());
		return c->leaveStackFrame();
	}


	unsigned int uToCol = pToCol->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid to_column parameter (%s)"),pToCol->ptr());
		return c->leaveStackFrame();
	}


	((QGridLayout *)object())->addMultiCellWidget(((QWidget *)(ob->object())),uFromRow,uToRow,uFromCol,uToCol);

	return c->leaveStackFrame();
}



#include "m_class_layout.moc"
