//
//   File : class_socket.cpp
//   Creation date : Sun Nov 11 03:13:45 2001 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_settings.h"

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_command.h"
#include "kvi_locale.h"
#include "kvi_error.h"
#include "kvi_netutils.h"
#include "kvi_dns.h"
#include "kvi_error.h"
#include "kvi_socket.h"
#include "kvi_malloc.h"
#include "kvi_memmove.h"
#include "kvi_databuffer.h"

#include "class_socket.h"

//#include <stdlib.h>

#define KVI_IN_BUFFER_ALLOC_CHUNK 4096
#define KVI_READ_CHUNK 1024

static KviScriptObjectClass * g_pSocketClass = 0;

static KviScriptObject * socketClassCreateInstance(KviScriptObjectClass * cls,KviScriptObject *par,const char * nam)
{
	return new KviScriptSocketObject(cls,par,nam);
}

KviScriptSocketObject::KviScriptSocketObject(KviScriptObjectClass * cla,KviScriptObject * par,const char * nam)
: KviScriptObject(cla,par,nam)
{
	m_sock = KVI_INVALID_SOCKET;
	m_iStatus = KVI_SCRIPT_SOCKET_STATUS_DISCONNECTED;
	m_uRemotePort = 0;
	m_uSecondaryPort = 0;
	m_pSn = 0;
	m_pDns = 0;
	m_pDelayTimer = 0;
	m_uConnectTimeout = 60000;
	m_pInBuffer = 0;
	m_uInBufferLen = 0;
	m_uInDataLen = 0;
	m_pOutBuffer = new KviDataBuffer();
	m_pFlushTimer = new QTimer();
	m_bIpV6 = false;
	m_uLocalPort = 0;
	m_secondarySock = KVI_INVALID_SOCKET;
	connect(m_pFlushTimer,SIGNAL(timeout()),this,SLOT(tryFlush()));
}

KviScriptSocketObject::~KviScriptSocketObject()
{
	delete m_pOutBuffer;
	delete m_pFlushTimer;

	if(m_pInBuffer)kvi_free(m_pInBuffer);
	if(m_pDelayTimer)delete m_pDelayTimer;
	if(m_pDns)delete m_pDns;
	if(m_pSn)delete m_pSn;
	if(m_sock != KVI_INVALID_SOCKET)kvi_socket_close(m_sock);
	if(m_secondarySock != KVI_INVALID_SOCKET)kvi_socket_close(m_secondarySock);
}

void KviScriptSocketObject::reset()
{
	if(m_pDelayTimer)
	{
		delete m_pDelayTimer;
		m_pDelayTimer = 0;
	}
	if(m_pDns)
	{
		delete m_pDns;
		m_pDns = 0;
	}
	if(m_pSn)
	{
		delete m_pSn;
		m_pSn = 0;
	}
	if(m_sock != KVI_INVALID_SOCKET)
	{
		kvi_socket_close(m_sock);
		m_sock = KVI_INVALID_SOCKET;
	}
	if(m_pInBuffer)
	{
		kvi_free(m_pInBuffer);
		m_pInBuffer = 0;
	}
	if(m_secondarySock)
	{
		kvi_socket_close(m_secondarySock);
		m_secondarySock = KVI_INVALID_SOCKET;
	}
	m_uSecondaryPort = 0;
	m_szSecondaryIp.clear();
	delete m_pOutBuffer;
	if(m_pFlushTimer->isActive())m_pFlushTimer->stop();
	m_pOutBuffer = new KviDataBuffer();
	m_uInBufferLen = 0;
	m_uInDataLen = 0;
	m_iStatus = KVI_SCRIPT_SOCKET_STATUS_DISCONNECTED;
	m_uRemotePort = 0;
	m_szRemoteIp.clear();
	m_uLocalPort = 0;
	m_szLocalIp.clear();
	m_bIpV6 = false;
}

/*
	@doc: socket
	@keyterms:
		socket object class
	@title:
		socket class
	@type:
		class
	@short:
		A Ipv4/Ipv6 TCP socket
	@inherits:
		[class]object[/class]
	@description:
		This class provides a standard TCP/IP socket functionality.[br]
		It can be used either for connecting to a remote host or to listening for incoming connections.[br]
		If the KVIrc executable has been compiled with the IPV6 protocol support , this socket also supports it.[br]
	@functions:
		!fn: $status()
		Returns the status of the socket :[br]
		4=connected[br]
		3=listening[br]
		2=connecting[br]
		1=in dns call[br]
		0=disconnected[br]

		!fn: $connectTimeout()
		Returns the value of the connect attempt timeout in milliseconds.[br]
		This is the timeout after that a connection attempt will be considered as failed if the remote
		host is not responding. You can set it with [classfnc:socket]$setConnectTimeout[/classfnc]().[br]

		!fn: $setConnectTimeout(<timeout>)
		Sets the connect timeout for this socket. <timeout> must be a positive value in milliseconds.[br]
		The default timeout is 60000.[br]

		!fn: $connect(<host>,<port>)
		Attempts a connection to <host> on port <port>.[br]
		<host> can be a numeric internet address (either Ipv4 or Ipv6 (if supported)) or a hostname.[br]
		If a hostname is used, a DNS lookup is performed (the socket enters the "dns call" state).[br]
		This function returns 1 if the connect attempt can be succesfully initiated,
		0 otherwise.[br] In fact , this function returns 0 only if the supplied <port> parameter
		is invalid or the socket is in an incoherent state (already connected or listening):
		for a newly created socket and with a valid <port> number you can safely ignore
		the return value.[br]
		Please note that the connection is asynchronous: when this function returns
		the socket is NOT connected: it has just initiated the connect attempt
		and you will be notified of the attempt result by an asynchronous event call:
		in case of failure , $connectFailedEvent() will be called , in case of
		succes , $connectEvent() will be called.[br]

		!fn: $listen([<port>[,<interface>[,<force_ipv6>]]])
		Attempts to listen on the specified <port> and <interface>.[br]
		If <port> is not passed it is assumed to be 0 , if <interface> is not passed , it is assumed to be
		"any interface" (INADDR_ANY).[br] Port 0 means that the kernel should choose a "random" port to listen on.[br]
		If the <interface> is recognized as IPV6 address , and IPV6 is supported , the socket listens
		in IPV6 mode. If <interface> is an empty string and <force_ipv6> is 1 the socket listens
		on "any ipv6 interface".[br]
		This function returns '1' in case of success and '0' in case of failure.[br]
		On some systems listening in the IPV6 namespace allows to accept also IPV4 connections (this includes
		linux but not windows afaik).[br]
		When an incoming connection will arrive , $incomingConnectionEvent() will be called.[br]

		!fn: $accept(<socketId>)
		This function can be called only from inside $incomingConnectionEvent().[br]
		<socketId> must be a newly created socket sub-class instance object.[br]
		The passed object will be set up to handle the incoming connection and this object
		will remain in listening state (unless you explicitly close it).[br]
		This function returns '1' in case of success and '0' in case of failure.[br]

		!fn: $connectEvent()
		This function is called when a connection attempt has been succesfully completed.
		The socket is actually connected to [classfnc:socket]$remoteIp[/classfnc]() on
		[classfnc:socket]$remotePort[/classfnc](). You can start
		writing data and you may expect [classfnc:socket]$dataAvailableEvent[/classfnc]() to be
		triggered.[br]

		!fn: $incomingConnectionEvent()
		This function is called when an incoming connection arrives over a socket in listening state.[br]
		You must call [classfnc:socket]$accept[/classfnc]() passing a newly created socket object
		to accept and handle the connection.[br] If you don't call [classfnc:socket]$accept[/classfnc]()
		the incoming connection will be automatically terminated.[br]

		!fn: $connectFailedEvent(<reason>)
		This function is called when a connection attempt fails for some reason. <reason> contains
		the error string.[br]
		This function may be called only between a call to [classfnc:socket]$connect[/classfnc]() and
		the [classfnc:socket]$connectEvent[/classfnc]().[br]

		!fn: $disconnectEvent([error])
		This function is called when a connection is terminated either cleanly or because of an error.[br]
		[error] is an empty string in case of a "clean" termination (connection closed by the remote host)
		or is a message describing the socket error that caused the connection to be interrupted.[br]

		!fn: $dataAvailableEvent(<data_length>)
		This function is called when some data is available to be read: the <data_length> parameter specifies
		the length of the available data in bytes.[br]
		You can use one of the $read* functions to obtain the data.[br]

		!fn: $read(<length>)
		Reads at most <length> bytes of data from the socket. If <length> is anything "outside" the
		available data range (<length> < 0 or <length> > available_data_length), this function
		returns all the available data.[br]
		Please note that this function can't deal withi binary data: NULL characters are transformed to
		ASCII characters 255.[br]

		!fn: $readHex(<length>)
		Reads at most <length> bytes of data from the socket. If <length> is anything "outside" the
		available data range (<length> < 0 or <length> > available_data_length), this function
		returns all the available data.[br]
		Returns the data encoded as hexadecimal digit string: this function can deal with binary data too.[br]

		!fn: $write(<data>)
		Writes <data> to the socket.[br]
		This function can't deal with binary data (you can't send a NULL character)[br]
		Please note that when this function finishes it does not mean that the data has reached the remote end.[br]
		Basically it does not even mean that the data has been sent to the remote host.[br]
		The data is enqueued for sending and will be sent as soon as possible.[br]
		If you're going to [cmd]delete[/cmd] this object just after the $write call, you should
		call [classfnc:socket]$close[/classfnc]() just before [cmd]delete[/cmd] to ensure the data delivery.[br]

		!fn: $writeHex(<hex_data>)
		Writes <data> to the socket.[br]
		<data> is expected to be a hexadecimal rappresentation of the bytes to be sent (two HEX characters
		for each byte). This means that the length of <hex_data> string must be a multiple of 2.[br]
		Returns the length of the actually decoded and sent data in bytes or -1 in case of error (the string
		was not a valid hexadecimla rappresentation).[br]
		Please note that when this function finishes it does not mean that the data has reached the remote end.[br]
		Basically it does not even mean that the data has been sent to the remote host.[br]
		The data is enqueued for sending and will be sent as soon as possible.[br]
		If you're going to [cmd]delete[/cmd] this object just after the $writeHex call, you should
		call [classfnc:socket]$close[/classfnc]() just before [cmd]delete[/cmd] to ensure the data delivery.[br]

		!fn: $close()
		Resets this socket state: kills any pending or active connection. After a close() call
		the socket may be used for a new connection.[br]
		If there is an active connection, there is a last attempt to flush the pending outgoing data.[br]
		You don't need to call $close() if you [cmd]delete[/cmd] the socket: KVIrc will
		reset the socket state automatically and free the memory. But if you want to ensure data delivery
		after a $write call sequece and just before a [cmd]delete[/cmd], $close() is the only chance to do it.[br]

		!fn: $remoteIp()
		Returns the IP address of the remote end of this socket.[br]
		The return value is meaningful only if the socket is in connected or connecting state.[br]

		!fn: $remotePort()
		Returns the port of the remote end of this socket.[br]
		The return value is meaningful only if the socket is in connected or connecting state.[br]

		!fn: $localIp()
		Returns the IP address of the local end of this socket.[br]
		The return value is meaningful only if the socket is in connected , listening or connecting state.[br]

		!fn: $localPort()
		Returns the port of the local end of this socket.[br]
		The return value is meaningful only if the socket is in connected , listening or connecting state.[br]


*/


#define REG_FH(__name,__func) \
	g_pSocketClass->registerFunctionHandler(__name,(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptSocketObject::__func)),0,true)


void KviScriptSocketObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController->lookupClass("object");

	__range_valid(base);
	g_pSocketClass = new KviScriptObjectClass(base,"socket",socketClassCreateInstance,true);

	REG_FH("status",functionStatus);
	REG_FH("remotePort",functionRemotePort);
	REG_FH("remoteIp",functionRemoteIp);
	REG_FH("localIp",functionLocalIp);
	REG_FH("localPort",functionLocalPort);
	REG_FH("connect",functionConnect);
	REG_FH("connectTimeout",functionConnectTimeout);
	REG_FH("setConnectTimeout",functionSetConnectTimeout);
	REG_FH("close",functionClose);
	REG_FH("read",functionRead);
	REG_FH("readHex",functionReadHex);
	REG_FH("write",functionWrite);
	REG_FH("writeHex",functionWriteHex);
	REG_FH("listen",functionListen);
	REG_FH("accept",functionAccept);

	g_pSocketClass->registerEmptyFunctionHandler("connectEvent");
	g_pSocketClass->registerEmptyFunctionHandler("connectFailedEvent");
	g_pSocketClass->registerEmptyFunctionHandler("disconnectEvent");
	g_pSocketClass->registerEmptyFunctionHandler("dataAvailableEvent");
	g_pSocketClass->registerEmptyFunctionHandler("incomingConnectionEvent");
}

#undef REG_FH

void KviScriptSocketObject::unregisterSelf()
{
	delete g_pSocketClass; // this will delete all the objects of this class
    g_pSocketClass = 0;
}

bool KviScriptSocketObject::functionStatus(KviCommand *c,KviParameterList *,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%d",m_iStatus);
	return true;
}

bool KviScriptSocketObject::functionClose(KviCommand *c,KviParameterList *,KviStr &buffer)
{
	if((m_pOutBuffer->size() != 0) && (m_iStatus == KVI_SCRIPT_SOCKET_STATUS_CONNECTED))
		tryFlush();

	reset();
	return true;
}

bool KviScriptSocketObject::functionConnectTimeout(KviCommand *c,KviParameterList *,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%u",m_uConnectTimeout);
	return true;
}

bool KviScriptSocketObject::functionSetConnectTimeout(KviCommand *c,KviParameterList *params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"socket::setConnectTimeout");
	KviStr szTimeout = params->safeFirstParam();
	bool bOk;
	unsigned int uTimeout = szTimeout.toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid timeout (%s)"),szTimeout.ptr());
		return c->leaveStackFrame();
	}
	m_uConnectTimeout = uTimeout;
	return c->leaveStackFrame();
}

bool KviScriptSocketObject::functionRemotePort(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%u",m_uRemotePort);
	return true;
}

bool KviScriptSocketObject::functionRemoteIp(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_szRemoteIp);
	return true;
}

bool KviScriptSocketObject::functionLocalPort(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%u",m_uLocalPort);
	return true;
}

bool KviScriptSocketObject::functionLocalIp(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_szLocalIp);
	return true;
}

bool KviScriptSocketObject::functionListen(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"socket::listen");
	if((m_sock != KVI_INVALID_SOCKET) || (m_iStatus != KVI_SCRIPT_SOCKET_STATUS_DISCONNECTED))
	{
		c->warning(__tr("Another connection in progress"));
		buffer.append('0');
		return c->leaveStackFrame();
	}

	KviStr szPort = *(params->safeFirst());
	if(szPort.hasData())
	{
		bool bOk;
		m_uLocalPort = szPort.toUShort(&bOk);
		if(!bOk)
		{
			c->warning(__tr("Invalid port specified: the kernel will choose one"));
			m_uLocalPort = 0;
		}
	} else m_uLocalPort = 0;

	KviStr m_szLocalIp   = *(params->safeNext());

	params->safeNext();
	m_bIpV6 = params->getBool();

#ifndef COMPILE_IPV6_SUPPORT
	if(m_bIpV6)
	{
		c->warning(__tr("No IPV6 support in this executable"));
		buffer.append('0');
		reset();
		return c->leaveStackFrame();
	}
#endif

	bool bGotIp = false;

	if(m_szLocalIp.hasData())
	{
		// Check the address type
		if(kvi_isValidStringIp(m_szLocalIp.ptr()))bGotIp = true;
		else {
#ifdef COMPILE_IPV6_SUPPORT
			if(kvi_isValidStringIp_V6(m_szLocalIp.ptr()))
			{
				bGotIp = true;
				m_bIpV6 = true;
			} else {
#else
				c->warning(__tr("Invalid IP address specified (%s)"),m_szLocalIp.ptr());
				buffer.append('0');
				reset();
				return c->leaveStackFrame();
#endif
#ifdef COMPILE_IPV6_SUPPORT
			}
#endif
		}
	}


#ifdef COMPILE_IPV6_SUPPORT
	m_sock = kvi_socket_create(m_bIpV6 ? KVI_SOCKET_PF_INET6 : KVI_SOCKET_PF_INET,
								KVI_SOCKET_TYPE_STREAM,KVI_SOCKET_PROTO_TCP);
#else
	m_sock = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,KVI_SOCKET_PROTO_TCP);
#endif

	if(m_sock == KVI_INVALID_SOCKET)
	{
		c->warning(__tr("Socket creation failed"));
		buffer.append('0');
		reset();
		return c->leaveStackFrame();
	}

	if(m_szLocalIp.hasData())
	{
#ifdef COMPILE_IPV6_SUPPORT
		KviSockaddr sa(m_szLocalIp.ptr(),m_uLocalPort,m_bIpV6);
#else
		KviSockaddr sa(m_szLocalIp.ptr(),m_uLocalPort,false);
#endif
		if(!sa.socketAddress())
		{
			c->warning(__tr("Invalid socket address"));
			reset();
			buffer.append('0');
			return c->leaveStackFrame();
		}

		if(!kvi_socket_bind(m_sock,sa.socketAddress(),((int)(sa.addressLength()))))
		{
			c->warning(__tr("Bind failure"));
			reset();
			buffer.append('0');
			return c->leaveStackFrame();
		}
	} else {
#ifdef COMPILE_IPV6_SUPPORT
		KviSockaddr sa(m_uLocalPort,m_bIpV6);
#else
		KviSockaddr sa(m_uLocalPort,false);
#endif
		if(!sa.socketAddress())
		{
			c->warning(__tr("Invalid socket address"));
			reset();
			buffer.append('0');
			return c->leaveStackFrame();
		}

		if(!kvi_socket_bind(m_sock,sa.socketAddress(),((int)(sa.addressLength()))))
		{
			c->warning(__tr("Bind failure"));
			reset();
			buffer.append('0');
			return c->leaveStackFrame();
		}
	}


	if(!kvi_socket_listen(m_sock,5))
	{
		c->warning(__tr("Listen failure"));
		reset();
		buffer.append('0');
		return c->leaveStackFrame();
	}


	// Reread the port in case we're binding to a random one (0)

#ifdef COMPILE_IPV6_SUPPORT
	KviSockaddr sareal(0,m_bIpV6);
#else
	KviSockaddr sareal(0,false);
#endif

	int size = (int)sareal.addressLength();

	if(kvi_socket_getsockname(m_sock,sareal.socketAddress(),&size))
	{
		m_uLocalPort = sareal.port();
		sareal.getStringAddress(m_szLocalIp);
	}

	// and setup the READ notifier...
	m_pSn = new QSocketNotifier(m_sock,QSocketNotifier::Read);
	QObject::connect(m_pSn,SIGNAL(activated(int)),this,SLOT(incomingConnection(int)));
	m_pSn->setEnabled(true);

	m_iStatus = KVI_SCRIPT_SOCKET_STATUS_LISTENING;

	buffer.append('1');

	return c->leaveStackFrame();
}

void KviScriptSocketObject::incomingConnection(int)
{
#ifdef COMPILE_IPV6_SUPPORT
	struct sockaddr_in6 hostSockAddr6;
#endif
	struct sockaddr_in  hostSockAddr;

	int size = sizeof(hostSockAddr);
	struct sockaddr * addr = (struct sockaddr *)&hostSockAddr;

#ifdef COMPILE_IPV6_SUPPORT
	if(m_bIpV6)
	{
		addr = (struct sockaddr *)&hostSockAddr6;
		size = sizeof(hostSockAddr6);
	}
#endif

	// Incoming connection
	m_secondarySock = kvi_socket_accept(m_sock,addr,&size);
	if(m_secondarySock != KVI_INVALID_SOCKET)
	{
		// Connected
#ifdef COMPILE_IPV6_SUPPORT
		if(m_bIpV6)
		{
			m_uSecondaryPort = ntohs(((struct sockaddr_in6 *)addr)->sin6_port);
			if(!kvi_binaryIpToStringIp_V6(((struct sockaddr_in6 *)addr)->sin6_addr,m_szSecondaryIp))
				m_szSecondaryIp = __tr("unknown");
		} else {
#endif
			m_uSecondaryPort = ntohs(((struct sockaddr_in *)addr)->sin_port);
			if(!kvi_binaryIpToStringIp(((struct sockaddr_in *)addr)->sin_addr,m_szSecondaryIp))
				m_szSecondaryIp = __tr("unknown");
#ifdef COMPILE_IPV6_SUPPORT
		}
#endif
		callEventFunction("incomingConnectionEvent",0,0);
		if(m_secondarySock != KVI_INVALID_SOCKET)
		{
			kvi_socket_close(m_secondarySock);
			m_secondarySock = KVI_INVALID_SOCKET;
			m_uSecondaryPort = 0;
			m_szSecondaryIp.clear();
		}

	} // else..huh ?... wait for the next notifier call	
}

void KviScriptSocketObject::acceptConnection(kvi_socket_t s,unsigned short int uPort,const char * szIp)
{
	reset();
	m_sock = s;
	m_uRemotePort = uPort;
	m_szRemoteIp = szIp;
#ifdef COMPILE_IPV6_SUPPORT
	KviSockaddr sareal(0,m_bIpV6);
#else
	KviSockaddr sareal(0,false);
#endif

	int size = (int)sareal.addressLength();

	if(kvi_socket_getsockname(m_sock,sareal.socketAddress(),&size))
	{
		m_uLocalPort = sareal.port();
		sareal.getStringAddress(m_szLocalIp);
	}

	m_iStatus = KVI_SCRIPT_SOCKET_STATUS_CONNECTED;

	m_pSn = new QSocketNotifier((int)m_sock,QSocketNotifier::Read);
	QObject::connect(m_pSn,SIGNAL(activated(int)),this,SLOT(readNotifierFired(int)));
	m_pSn->setEnabled(true);
}


bool KviScriptSocketObject::functionAccept(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"socket::accept");

	KviScriptObject * ob = g_pScriptObjectController->lookupObject(params->safeFirstParam());

	if(!ob)
	{
		c->warning(__tr("No socket object specified"));
		return c->leaveStackFrame();
	}

	if(!ob->inherits("KviScriptSocketObject"))
	{
		c->warning(__tr("Invalid socket object specified (it doesn't inherit from socket)"));
		return c->leaveStackFrame();
	}

	if(m_secondarySock != KVI_INVALID_SOCKET)
	{
		((KviScriptSocketObject *)ob)->acceptConnection(m_secondarySock,m_uSecondaryPort,m_szSecondaryIp.ptr());

		m_secondarySock = KVI_INVALID_SOCKET;
		m_uSecondaryPort = 0;
		m_szSecondaryIp.clear();
	} else {
		c->warning(__tr("There is no connection to accept!"));
	}
	return c->leaveStackFrame();
}

bool KviScriptSocketObject::functionConnect(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"socket::connect");
	if((m_sock != KVI_INVALID_SOCKET) || (m_iStatus != KVI_SCRIPT_SOCKET_STATUS_DISCONNECTED))
	{
		c->warning(__tr("Another connection in progress"));
		buffer.append("0");
		return c->leaveStackFrame();
	}

	m_szRemoteIp = params->safeFirstParam();
	KviStr szRemotePort = params->safeNextParam();
	bool bOk;
	m_uRemotePort = szRemotePort.toUShort(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid port (%s)"),szRemotePort.ptr());
		buffer.append('0');
		return c->leaveStackFrame();
	}

#ifdef COMPILE_IPV6_SUPPORT
	if(kvi_isValidStringIp(m_szRemoteIp.ptr()) || kvi_isValidStringIp_V6(m_szRemoteIp.ptr()))
#else
	if(kvi_isValidStringIp(m_szRemoteIp.ptr()))
#endif
	{
		m_iStatus = KVI_SCRIPT_SOCKET_STATUS_CONNECTING;
		delayedConnect();
	} else {
		m_iStatus = KVI_SCRIPT_SOCKET_STATUS_DNS;
		delayedLookupRemoteIp();
	}
	buffer.append('1');
	return c->leaveStackFrame();
}

void KviScriptSocketObject::delayedConnect()
{
	if(m_pDelayTimer)delete m_pDelayTimer;
	m_pDelayTimer = new QTimer();
	connect(m_pDelayTimer,SIGNAL(timeout()),this,SLOT(doConnect()));
	m_pDelayTimer->start(0,true);
}

void KviScriptSocketObject::doConnect()
{
	if(m_pDelayTimer)delete m_pDelayTimer;
	m_pDelayTimer = 0;


	KviSockaddr sa(m_szRemoteIp.ptr(),m_uRemotePort,!kvi_isValidStringIp(m_szRemoteIp.ptr()));

	if(!sa.socketAddress())
	{
		callEventFunction("connectFailedEvent",0,new KviParameterList(
			new KviStr(KviStr::Format,__tr("Invalid ip address (%s)"),m_szRemoteIp.ptr())));
		reset();
		return;
	}


	// create the socket
#ifdef COMPILE_IPV6_SUPPORT
	m_bIpV6 = sa.isIpV6();
	m_sock = kvi_socket_create(sa.isIpV6() ? KVI_SOCKET_PF_INET6 : KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,KVI_SOCKET_PROTO_TCP);
#else
	m_bIpV6 = false;
	m_sock = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,KVI_SOCKET_PROTO_TCP);
#endif

	if(m_sock == KVI_INVALID_SOCKET)
	{
		callEventFunction("connectFailedEvent",0,new KviParameterList(
			new KviStr(__tr("Failed to create the socket"))));
		reset();
		return;
	}

	if(!kvi_socket_setNonBlocking(m_sock))
	{
		callEventFunction("connectFailedEvent",0,new KviParameterList(
			new KviStr(__tr("Failed to setup a nonblocking socket"))));
		reset();
		return;
	}

	if(!kvi_socket_connect(m_sock,sa.socketAddress(),((int)(sa.addressLength()))))
	{
		int err = kvi_socket_error();
		if(!kvi_socket_recoverableConnectError(err))
		{
			// Ops...
			int sockError=err;
			if(sockError==0)
			{
				// Zero error ?...let's look closer
				int iSize=sizeof(int);
				if(!kvi_socket_getsockopt(m_sock,SOL_SOCKET,SO_ERROR,
						(void *)&sockError,&iSize))sockError=0;
			}
			callEventFunction("connectFailedEvent",0,new KviParameterList(
				new KviStr(KviStr::Format,__tr("Connect failure: %s"),KviError::getDescription(KviError::translateSystemError(sockError)).latin1())));
			reset();
			return;
		}
	}

	m_pDelayTimer = new QTimer();
	connect(m_pDelayTimer,SIGNAL(timeout()),this,SLOT(connectTimeout()));
	m_pDelayTimer->start(m_uConnectTimeout,true);

	m_pSn = new QSocketNotifier((int)m_sock,QSocketNotifier::Write);
	QObject::connect(m_pSn,SIGNAL(activated(int)),this,SLOT(writeNotifierFired(int)));
	m_pSn->setEnabled(true);
}

void KviScriptSocketObject::connectTimeout()
{
	callEventFunction("connectFailedEvent",0,new KviParameterList(
			new KviStr(__tr("Connect attempt timed out"))));
	reset();
}

void KviScriptSocketObject::delayedLookupRemoteIp()
{
	if(m_pDelayTimer)delete m_pDelayTimer;
	m_pDelayTimer = new QTimer();
	connect(m_pDelayTimer,SIGNAL(timeout()),this,SLOT(lookupRemoteIp()));
	m_pDelayTimer->start(0,true);
}

void KviScriptSocketObject::lookupRemoteIp()
{
	if(m_pDelayTimer)delete m_pDelayTimer;
	m_pDelayTimer = 0;
	if(m_pDns)delete m_pDns;
	m_pDns = new KviDns();
	connect(m_pDns,SIGNAL(lookupDone(KviDns *)),this,SLOT(lookupDone(KviDns *)));
	if(!m_pDns->lookup(m_szRemoteIp.ptr(),KviDns::Any))
	{
		callEventFunction("connectFailedEvent",0,new KviParameterList(
			new KviStr(__tr("Can't start the DNS thread"))));
		reset();
	}
}

void KviScriptSocketObject::lookupDone(KviDns *pDns)
{
	if(pDns->state() != KviDns::Success)
	{
		callEventFunction("connectFailedEvent",0,new KviParameterList(
			new KviStr(KviError::getDescription(pDns->error()))));
		reset();
		return;
	}

	m_szRemoteIp = pDns->firstIpAddress();
	delete m_pDns;
	m_pDns = 0;
	doConnect();
}

void KviScriptSocketObject::writeNotifierFired(int)
{
	if(m_pSn)
	{
		delete m_pSn;
		m_pSn = 0;
	}
	if(m_pDelayTimer)
	{
		delete m_pDelayTimer;
		m_pDelayTimer = 0;
	}

	// Check for errors...
	int sockError;
	int iSize=sizeof(int);
	if(!kvi_socket_getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize))sockError = -1;
	//sockError = 0;
	if(sockError != 0)
	{
		//debug("Failed here %d",sockError);
		//failed
		if(sockError > 0)sockError = KviError::translateSystemError(sockError);
		else sockError = KviError_unknownError; //Error 0 ?
		callEventFunction("connectFailedEvent",0,new KviParameterList(
				new KviStr(KviError::getDescription(sockError))));
		reset();
	} else {
		// Succesfully connected
		// create the correct read notifier now...
		m_pSn = new QSocketNotifier((int)m_sock,QSocketNotifier::Read);
		QObject::connect(m_pSn,SIGNAL(activated(int)),this,SLOT(readNotifierFired(int)));
		m_pSn->setEnabled(true);

#ifdef COMPILE_IPV6_SUPPORT
		KviSockaddr sareal(0,m_bIpV6);
#else
		KviSockaddr sareal(0,false);
#endif
		int size = (int)sareal.addressLength();
		if(kvi_socket_getsockname(m_sock,sareal.socketAddress(),&size))
		{
			m_uLocalPort = sareal.port();
			sareal.getStringAddress(m_szLocalIp);
		}

		callEventFunction("connectEvent",0,0);
		m_iStatus = KVI_SCRIPT_SOCKET_STATUS_CONNECTED;
	}
}

void KviScriptSocketObject::readNotifierFired(int)
{
	//read data
	if((m_uInBufferLen - m_uInDataLen) < KVI_READ_CHUNK)
	{
		m_uInBufferLen += KVI_IN_BUFFER_ALLOC_CHUNK;
		m_pInBuffer = (char *)kvi_realloc(m_pInBuffer,m_uInBufferLen);
	}

	int readLength = kvi_socket_recv(m_sock,m_pInBuffer + m_uInDataLen,KVI_READ_CHUNK);

	if(readLength <= 0)
	{
		if(readLength==0)
		{
			callEventFunction("disconnectEvent",0,0);
			reset();
			return;
		} else {
			//check for transmission errors
			int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
			if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
			if((err != EAGAIN) && (err != EINTR))
#endif
			{
				if(err > 0)
				{
					callEventFunction("disconnectEvent",0,new KviParameterList(
						new KviStr(KviError::getDescription(KviError::translateSystemError(err)))));
				} else {
					callEventFunction("disconnectEvent",0,new KviParameterList(
						new KviStr(KviError::getDescription(KviError_remoteEndClosedConnection))));
				}
				reset();
				return;
			} //else transient error...wait again...
		}
		return; // can do nothing
	}
	// readLength > 0
	m_uInDataLen += readLength;

	KviStr * s = new KviStr();
	s->setNum(m_uInDataLen);

	callEventFunction("dataAvailableEvent",0,new KviParameterList(s));

	if(m_uInDataLen > (1024 * 1024)) // too much data in buffer (not reading)
	{
		callEventFunction("disconnectEvent",0,new KviParameterList(
			new KviStr(__tr("Too much unprocessed incoming data (you've left this socket unmanaged ?)"))));
		reset();
	}
}

void KviScriptSocketObject::eatInData(unsigned int uLen)
{
	if(uLen > m_uInDataLen)uLen = m_uInDataLen;

	m_uInDataLen -= uLen;

	if(m_uInDataLen > 0)
	{
		kvi_memmove(m_pInBuffer,m_pInBuffer + uLen,m_uInDataLen);
	}

	unsigned int uSpace = m_uInBufferLen - m_uInDataLen;

	if(uSpace > KVI_IN_BUFFER_ALLOC_CHUNK)
	{
		m_uInBufferLen -= KVI_IN_BUFFER_ALLOC_CHUNK;
		m_pInBuffer = (char *)kvi_realloc(m_pInBuffer,m_uInBufferLen);
	}
}

unsigned int KviScriptSocketObject::readGetLength(KviParameterList * params)
{
	KviStr * pszLen = params->safeFirst();
	unsigned int uLen = m_uInDataLen;
	if(pszLen)
	{
		bool bOk;
		uLen = pszLen->toInt(&bOk);
		if(!bOk)uLen = m_uInDataLen;
	}

	if(uLen > m_uInDataLen)uLen = m_uInDataLen;
	return uLen;
}

bool KviScriptSocketObject::functionRead(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	unsigned int uLen = readGetLength(params);

	if(uLen > 0)
	{
		// convert NULLS to char 255
		for(unsigned int i = 0;i < uLen;i++)
		{
			if(!m_pInBuffer[i])m_pInBuffer[i] = (char)(255);
		}

		buffer.append(m_pInBuffer,uLen);

		eatInData(uLen);
	}
	return true;
}

bool KviScriptSocketObject::functionReadHex(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	unsigned int uLen = readGetLength(params);

	if(uLen > 0)
	{
		KviStr szTmp;
		szTmp.bufferToHex(m_pInBuffer,uLen);
		buffer.append(szTmp);
		eatInData(uLen);
	}
	return true;
}

bool KviScriptSocketObject::functionWrite(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pszData = params->safeFirst();

	if(pszData)
	{
		if(pszData->len() > 0)
		{
			m_pOutBuffer->append((const unsigned char *)(pszData->ptr()),pszData->len());
			delayedFlush(0);
		}
	}
	return true;
}

bool KviScriptSocketObject::functionWriteHex(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pszData = params->safeFirst();
	int len = 0;
	if(pszData)
	{
		if(pszData->len() > 0)
		{
			char * ptr;
			len = pszData->hexToBuffer(&ptr);
			if(len > 0)
			{
				m_pOutBuffer->append((const unsigned char *)(ptr),len);
				delayedFlush(0);
				KviStr::freeBuffer(ptr);
			}
		}
	}
	buffer.append(KviStr::Format,"%d",len);
	return true;
}

void KviScriptSocketObject::delayedFlush(unsigned int uTimeout)
{
	if(m_pFlushTimer->isActive())m_pFlushTimer->stop();
	m_pFlushTimer->start(uTimeout);
}

void KviScriptSocketObject::tryFlush()
{
	if(m_pFlushTimer->isActive())m_pFlushTimer->stop();

	if(m_pOutBuffer->size() == 0)return;

	int result = kvi_socket_send(m_sock,m_pOutBuffer->data(),m_pOutBuffer->size());

	if(result >= 0)
	{
		if(result == m_pOutBuffer->size())
		{
			m_pOutBuffer->clear();
		} else {
			if(result > 0)m_pOutBuffer->remove(result);
			delayedFlush(500);
		}
	} else {
		// Oops...error ?
		int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
		if((err == EAGAIN) || (err == EINTR) || (err = WSAEWOULDBLOCK))
#else
		if((err == EAGAIN)||(err == EINTR))
#endif
		{
			// Transient error...partial send as before...
			// Async continue...
			delayedFlush(500);
			return;
		} else {
			// Disconnected... :(
			callEventFunction("disconnectEvent",0,new KviParameterList(
				new KviStr(KviError::getDescription(KviError::translateSystemError(err)))));
			reset();
			return;
		}
	}
}

#include "m_class_socket.moc"
