//=============================================================================
//
//   File : optw_theme.cpp
//   Creation date : Sun May 25 2003 18:00:00 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "optw_theme.h"

#include <qlayout.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qregexp.h>
#include <qmessagebox.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qdatetime.h>

#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_config.h"
#include "kvi_fileutils.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_iconmanager.h"
#include "kvi_internalcmd.h"

KviSaveThemeDialog::KviSaveThemeDialog(QWidget * par)
: QDialog(par)
{
		setCaption(__tr2qs("Save Current Theme"));

	QGridLayout * g = new QGridLayout(this,6,2,4,4);

	QLabel * l;

	l = new QLabel(__tr2qs("Theme name:"),this);
	g->addWidget(l,0,0);

	m_pThemeNameEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pThemeNameEdit,0,0,1,2);
	connect(m_pThemeNameEdit,SIGNAL(textChanged(const QString &)),this,SLOT(themeNameChanged(const QString &)));


	l = new QLabel(__tr2qs("Version:"),this);
	g->addWidget(l,1,0);

	m_pVersionEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pVersionEdit,1,1,1,2);


	l = new QLabel(__tr2qs("Author:"),this);
	g->addWidget(l,2,0);

	m_pAuthorEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pAuthorEdit,2,2,1,2);

	l = new QLabel(__tr2qs("Description:"),this);
	g->addWidget(l,3,0);

	m_pDescriptionEdit = new QTextEdit(this);
	g->addMultiCellWidget(m_pDescriptionEdit,3,4,1,2);

	QHBox * h = new QHBox(this);
	h->setSpacing(4);
	g->addWidget(h,5,2);
	m_pOkButton = new QPushButton(__tr2qs("&OK"),h);
	m_pOkButton->setEnabled(false);
	connect(m_pOkButton,SIGNAL(clicked()),this,SLOT(saveTheme()));
	m_pOkButton->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ACCEPT)));

	QPushButton * pb = new QPushButton(__tr2qs("Cancel"),h);
	connect(pb,SIGNAL(clicked()),this,SLOT(reject()));
	pb->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DISCARD)));

}

KviSaveThemeDialog::~KviSaveThemeDialog()
{

}

void KviSaveThemeDialog::themeNameChanged(const QString &txt)
{
	m_pOkButton->setEnabled(!txt.isEmpty());
}

void KviSaveThemeDialog::saveTheme()
{
	KviThemeInfo sto;
	sto.szName = m_pThemeNameEdit->text();
	if(sto.szName.isEmpty())
	{
		QMessageBox::critical(this,__tr2qs("Save Theme - KVIrc"),__tr2qs("You must choose a theme name!"),QMessageBox::Ok,
			QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}

	sto.szAuthor = m_pAuthorEdit->text();
	sto.szDescription = m_pDescriptionEdit->text();
	sto.szDate = QDateTime::currentDateTime().toString();
	sto.szVersion = m_pVersionEdit->text();
	sto.szKvircVersion = KVI_VERSION;

	if(sto.szVersion.isEmpty())sto.szVersion = "1.0.0";

	sto.szSubdirectory = sto.szName + QString("-") + sto.szVersion;
	sto.szSubdirectory.replace(QRegExp("[ \\\\/:][ \\\\/:]*"),"_");

	g_pApp->getLocalKvircDirectory(sto.szAbsoluteDirectory,KviApp::Themes,sto.szSubdirectory,true);

	if(!kvi_directoryExists(sto.szAbsoluteDirectory))
	{
		if(!kvi_makeDir(sto.szAbsoluteDirectory.latin1()))
		{
			QMessageBox::critical(this,__tr2qs("Save Theme - KVIrc"),__tr2qs("Unable to create theme directory."),
				QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
			return;
		}
	}

	g_pApp->saveTheme(sto);

	// write down the images
	
	QString szMsg = __tr2qs("Theme saved successfully to ");
	szMsg += sto.szAbsoluteDirectory;

	QMessageBox::information(this,__tr2qs("Save Theme - KVIrc"),szMsg,QMessageBox::Ok,
		 QMessageBox::NoButton,QMessageBox::NoButton);

	accept();
}




KviThemeListBoxItem::KviThemeListBoxItem(QListBox * box,KviThemeInfo * inf)
: QListBoxText(box)
{
	QString szTxt = inf->szName;
	szTxt += " ";
	szTxt += inf->szVersion;
	setText(szTxt);
	m_pThemeInfo = inf;
}

KviThemeListBoxItem::~KviThemeListBoxItem()
{
	delete m_pThemeInfo;
}



KviThemeOptionsWidget::KviThemeOptionsWidget(QWidget * parent)
: KviOptionsWidget(parent,"theme_options_widget")
{
	m_pSaveThemeDialog = 0;

	createLayout(9,1);

	QString szDir;
	g_pApp->getLocalKvircDirectory(szDir,KviApp::Themes);
	addLabel(0,0,1,0,__tr2qs("The list below shows the currently installed themes. To apply a theme " \
		"select it and click \"Ok\" or \"Apply\". Please note that the message text colors are applied " \
		"only to new messages in each window.<br>To install a new theme, unpack it to:<br><b>") + szDir +
		QString("<br></b>"));
	m_pListBox = new QListBox(this);
	layout()->addMultiCellWidget(m_pListBox,1,8,0,0);
	connect(m_pListBox,SIGNAL(highlighted(QListBoxItem *)),this,SLOT(loadThemeInfo(QListBoxItem *)));

	m_pNameLabel = new QLabel(this);
	m_pNameLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pNameLabel,1,1);

	m_pVersionLabel = new QLabel(this);
	m_pVersionLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pVersionLabel,2,1);

	m_pAuthorLabel = new QLabel(this);
	m_pAuthorLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pAuthorLabel,3,1);

	m_pDateLabel = new QLabel(this);
	m_pDateLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pDateLabel,4,1);

	m_pDescriptionLabel = new QLabel(this);
	m_pDescriptionLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pDescriptionLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
	layout()->addWidget(m_pDescriptionLabel,5,1);

	QPushButton * pb = new QPushButton(__tr2qs("Refresh Theme List"),this);
	layout()->addWidget(pb,6,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(fillThemeBox()));

	pb = new QPushButton(__tr2qs("Save Current Theme..."),this);
	layout()->addWidget(pb,7,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(saveCurrentTheme()));

	pb = new QPushButton(__tr2qs("Get More Themes..."),this);
	layout()->addWidget(pb,8,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(getMoreThemes()));


	layout()->setRowStretch(5,1);
	layout()->setColStretch(1,1);

	fillThemeBox();
}

KviThemeOptionsWidget::~KviThemeOptionsWidget()
{
	if(m_pSaveThemeDialog)
	{
		delete m_pSaveThemeDialog;
		m_pSaveThemeDialog = 0;
	}
}

void KviThemeOptionsWidget::getMoreThemes()
{
	if(!g_pFrame)return;
	g_pFrame->executeInternalCommand(KVI_INTERNALCOMMAND_OPENURL_KVIRC_THEMES);
}

void KviThemeOptionsWidget::loadThemeInfo(QListBoxItem * i)
{
	KviThemeListBoxItem * it = (KviThemeListBoxItem *)i;

	if(!it)
	{
		m_pNameLabel->setText("");
		m_pAuthorLabel->setText("");
		m_pVersionLabel->setText("");
		m_pDateLabel->setText("");
		m_pDescriptionLabel->setText("");
		return;
	}

	if(it->text().isEmpty())return; //hm, not set yet ?

	if(it->themeInfo()->szAbsoluteDirectory.isEmpty())
	{
		m_pNameLabel->setText(__tr2qs("Current Theme"));
		m_pAuthorLabel->setText("");
		m_pVersionLabel->setText("");
		m_pDateLabel->setText("");
		m_pDescriptionLabel->setText(__tr2qs("<p>The active theme settings</p>"));
		return;
	}

	m_pNameLabel->setText(QString(__tr2qs("<p>Theme: %1</p>")).arg(it->themeInfo()->szName));
	m_pVersionLabel->setText(QString(__tr2qs("<p>Version: %1</p>")).arg(it->themeInfo()->szVersion));
	m_pDateLabel->setText(QString(__tr2qs("<p>Date: %1</p>")).arg(it->themeInfo()->szDate));
	m_pAuthorLabel->setText(QString(__tr2qs("<p>Author: %1</p>")).arg(it->themeInfo()->szAuthor));
	m_pDescriptionLabel->setText(it->themeInfo()->szDescription);
}

void KviThemeOptionsWidget::saveCurrentTheme()
{
	if(m_pSaveThemeDialog)return;
	m_pSaveThemeDialog = new KviSaveThemeDialog(this);
	m_pSaveThemeDialog->exec();
	if(m_pSaveThemeDialog)
	{
		delete m_pSaveThemeDialog;
		m_pSaveThemeDialog = 0;
		fillThemeBox();
	}
}

void KviThemeOptionsWidget::fillThemeBox(const QString &szDir)
{
	QDir d(szDir);

	QStringList sl = d.entryList(QDir::Dirs);

	for(QStringList::Iterator it = sl.begin();it != sl.end();++it)
	{
		if(*it == ".")continue;
		if(*it == "..")continue;

		QString szTest = szDir;
		szTest += KVI_PATH_SEPARATOR_CHAR;
		szTest += *it;

		KviThemeInfo * inf = new KviThemeInfo;
		if(g_pApp->loadThemeInfo(szTest,*inf))
		{
			KviThemeListBoxItem * item = 0;
			item = new KviThemeListBoxItem(m_pListBox,inf);
		} else {
			delete inf;
		}
	}
}

void KviThemeOptionsWidget::fillThemeBox()
{
	m_pListBox->clear();

	KviThemeInfo * ti = new KviThemeInfo;
	ti->szName = __tr2qs("Current Theme");

	KviThemeListBoxItem * itewrq = new KviThemeListBoxItem(m_pListBox,ti);

	QString szDir;
	g_pApp->getGlobalKvircDirectory(szDir,KviApp::Themes);
	fillThemeBox(szDir);
	g_pApp->getLocalKvircDirectory(szDir,KviApp::Themes);
	fillThemeBox(szDir);

	m_pListBox->setCurrentItem(itewrq);
}

void KviThemeOptionsWidget::commit()
{
	KviOptionsWidget::commit();

	KviThemeListBoxItem * it = (KviThemeListBoxItem *)m_pListBox->item(m_pListBox->currentItem());
	if(!it)return;

	QString szPath = it->themeInfo()->szAbsoluteDirectory;
	if(szPath.isEmpty())return;

	g_pApp->loadTheme(szPath);

	// make sure that we do not re-commit again
	it = 0;
	QString szCur = __tr2qs("Current Theme");
	for(unsigned int i = 0;i< m_pListBox->count();i++)
	{
		KviThemeListBoxItem * daItem = (KviThemeListBoxItem *)m_pListBox->item(i);
		if(daItem)
		{
			if(daItem->themeInfo()->szName == szCur)
			{
				it = daItem;
				break;
			}
		}
	}
	
	if(!it)fillThemeBox();
	else m_pListBox->setCurrentItem(it);
}

#include "m_optw_theme.moc"
