//=================================================================================
//
//   File : kvi_window.cpp
//   Creation date : Sat Sep 01 2001 17:13:12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=================================================================================

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_console.h"
#include "kvi_options.h"
#include "kvi_ircsocket.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_ircview.h"
#include "kvi_input.h"
#include "kvi_iconmanager.h"

#include "userwindow.h"

#include <qasciidict.h>
#include <qtimer.h>

// kvi_app.cpp
extern KVIRC_API QAsciiDict<KviWindow> * g_pGlobalWindowDict;
KviPtrList<KviUserWindow> * g_pUserWindowList = 0;

// $window.caption $window.x $window.y $window.width $window.height $window.isActive $window.type
// $window.input.text $window.input.cursorpos $window.input.textlen

static KviWindow * get_window_by_first_param(KviCommand *c,KviParameterList *parms)
{
	KviStr * id = parms->safeFirst();
	if(id->isEmpty())return c->window();
	return g_pApp->findWindow(id->ptr());
}

#define GET_WINDOW_BY_FINAL_PART \
	KviStr windowid; \
	if(!g_pUserParser->parseCmdFinalPart(c,windowid))return false; \
	KviWindow * wnd = 0; \
	if(windowid.isEmpty()) \
		wnd = c->window(); \
	else \
	{ \
		wnd = g_pApp->findWindow(windowid.ptr()); \
		if(!wnd && !c->hasSwitch('q'))c->warning(__tr("The window with id '%s' does not exist"),windowid.ptr()); \
	}

#define GET_WINDOW_BY_FIRST_TOKEN \
	KviStr windowid; \
	if(!g_pUserParser->parseCmdSingleToken(c,windowid))return false; \
	KviWindow * wnd = 0; \
	if(windowid.isEmpty()) \
		wnd = c->window(); \
	else \
	{ \
		wnd = g_pApp->findWindow(windowid.ptr()); \
		if(!wnd && !c->hasSwitch('q'))c->warning(__tr("The window with id '%s' does not exist"),windowid.ptr()); \
	}

/*
	@doc: window.clearOutput
	@type:
		command
	@title:
		window.clearOutput
	@short:
		Clears the output a window
	@syntax:
		window.clearOutput [-q] [window_id]
	@description:
		Clears the text output of the window specified by window_id. If window_id is missing then
		the current window is used. If the window has no text output then no operations is performed.
		If the specified window does not exist a warning is printed unless the -q switch is used.
	@seealso:
		[fnc]$window.hasOutput[/fnc]()
*/

static bool window_module_cmd_clearOutput(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::clearOutput");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)
	{
		if(wnd->view())wnd->view()->emptyBuffer(true);
	}
	return c->leaveStackFrame();
}

/*
	@doc: window.close
	@type:
		command
	@title:
		window.close
	@short:
		Closes a window
	@syntax:
		window.close [-q] [window_id]
	@description:
		Closes the window specified by window_id. If window_id is missing then
		the current window is closed. The close operation is asynchronous: it is
		performed immediately after the script has terminated the execution and
		the control is returned to the main KVIrc core. If the specified window
		does not exist a warning is printed unless the -q switch is used.
*/

static bool window_module_cmd_close(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::close");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->delayedClose();
	return c->leaveStackFrame();
}

/*
	@doc: window.dock
	@type:
		command
	@title:
		window.dock
	@short:
		Docks a window
	@syntax:
		window.dock [-q] [window_id]
	@description:
		Docks the window specified by window_id. If window_id is missing then
		the current window is docked. If the specified window was already docked then
		no operation is performed. If the specified window
		does not exist a warning is printed unless the -q switch is used.
	@seealso:
		[cmd]window.undock[/cmd]
*/

static bool window_module_cmd_dock(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::dock");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->dock();
	return c->leaveStackFrame();
}

/*
	@doc: window.undock
	@type:
		command
	@title:
		window.undock
	@short:
		Undocks a window
	@syntax:
		window.undock [-q] [window_id]
	@description:
		Undocks the window specified by window_id. If window_id is missing then
		the current window is undocked. If the specified window was already undocked then
		no operation is performed. If the specified window
		does not exist a warning is printed unless the -q switch is used.
	@seealso:
		[cmd]window.dock[/cmd]
*/

static bool window_module_cmd_undock(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::undock");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->undock();
	return c->leaveStackFrame();
}


/*
	@doc: window.maximize
	@type:
		command
	@title:
		window.maximize
	@short:
		Maximizes a window
	@syntax:
		window.maximize [-q] [window_id]
	@description:
		Maximizes the window specified by window_id. If window_id is missing then
		the current window is maximized. If the specified window was already maximized then
		no operation is performed. If the specified window
		does not exist a warning is printed unless the -q switch is used.
	@seealso:
		[cmd]window.minimize[/cmd], [cmd]window.restore[/cmd], [fnc]$window.isMaxmimized[/cmd],
		[fnc]$window.isMinimized[/cmd]
*/

static bool window_module_cmd_maximize(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::maximize");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->maximize();
	return c->leaveStackFrame();
}

/*
	@doc: window.minimize
	@type:
		command
	@title:
		window.minimize
	@short:
		Minimizes a window
	@syntax:
		window.minimize [-q] [window_id]
	@description:
		Minimizes the window specified by window_id. If window_id is missing then
		the current window is minimized. If the specified window was already minimized then
		no operation is performed. If the specified window
		does not exist a warning is printed unless the -q switch is used.
	@seealso:
		[cmd]window.maximize[/cmd], [cmd]window.restore[/cmd], [fnc]$window.isMaxmimized[/cmd],
		[fnc]$window.isMinimized[/cmd]
*/

static bool window_module_cmd_minimize(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::minimize");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->minimize();
	return c->leaveStackFrame();
}

/*
	@doc: window.restore
	@type:
		command
	@title:
		window.restore
	@short:
		Restores a window
	@syntax:
		window.restore [-q] [window_id]
	@description:
		Restores the window specified by window_id. If window_id is missing then
		the current window is restored. If the specified window was already restored then
		no operation is performed. If the specified window
		does not exist a warning is printed unless the -q switch is used.
	@seealso:
		[cmd]window.maximize[/cmd], [cmd]window.minimize[/cmd], [fnc]$window.isMaxmimized[/cmd],
		[fnc]$window.isMinimized[/cmd]
*/

static bool window_module_cmd_restore(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::restore");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->restore();
	return c->leaveStackFrame();
}

/*
	@doc: window.activate
	@type:
		command
	@title:
		window.activate
	@short:
		Activates a window
	@syntax:
		window.activate [-q] [window_id]
	@description:
		Activates the window specified by window_id. If window_id is missing then
		the current window is activated. If the specified window
		does not exist a warning is printed unless the -q switch is used.
		Please note that if the window is currently docked to a frame then this
		command will NOT raise the frame window. If you're interested in
		the user's attention then you might be interested in [cmd]window.demandAttention[/cmd].
	@seealso:
		[cmd]window.demandAttention[/cmd]
*/

static bool window_module_cmd_activate(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::activate");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->autoRaise();
	return c->leaveStackFrame();
}

/*
	@doc: window.demandAttention
	@type:
		command
	@title:
		window.demandAttention
	@short:
		Flashes a window's system taskbar entry
	@syntax:
		window.demandAttention [-q] [window_id]
	@description:
		Flashes the system taskbar entry of the window
		specified by the window_id. If window_id is missing then
		the current window's system taskbar entry is flashed. If the specified window
		does not exist a warning is printed unless the -q switch is used.
		If the window is currently docked in a frame then the frame's
		system taskbar entry will be flashed.
		Please note that this command is highly system dependant:
		on systems that do not have a system taskbar or there
		is no way to flash an etry this command will do nothing.
		At the time of writing this command works flawlessly on
		Windows and in KDE compilations.
	@seealso:
		[cmd]window.demandAttention[/cmd]
*/

static bool window_module_cmd_demandAttention(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::demandAttention");
	GET_WINDOW_BY_FINAL_PART
	if(wnd)wnd->demandAttention();
	return c->leaveStackFrame();
}

/*
	@doc: window.isMaximized
	@type:
		function
	@title:
		$window.isMaximized
	@short:
		Checks if a window is currently maximized
	@syntax:
		$window.isMaximized
		$window.isMaximized(<window_id>)
	@description:
		Returns 1 if the window specified by <window_id> is currently maximized and 0 otherwise.
		The form with no parameters works on the current window. If the specified window
		doesn't exist then 0 is returned.
	@seealso:
		[fnc]$window.isMinimized[/fnc](), [cmd]window.maximize[/cmd],
		[cmd]window.minimize[/cmd], [cmd]window.restore[/cmd]
*/


static bool window_module_fnc_isMaximized(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.isMaximized");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->isMaximized() ? '1' : '0');
	else buffer.append('0');
	return c->leaveStackFrame();
}

/*
	@doc: window.activityLevel
	@type:
		function
	@title:
		$window.activityLevel
	@short:
		Returns the current activity level of a window
	@syntax:
		$window.activityLevel
		$window.activityLevel(<window_id>)
	@description:
		Returns the current activity level of the window specified by <window_id>.
		The form without parameters works on the current window.[br]
		The activity level is a number describing the level of traffic in the window
		and depends on the window type. On channels and queries it is dependant on the number
		and frequency of actions performed by the users.
	@seealso:
		[fnc]$window.activityTemperature[/fnc]
*/


static bool window_module_fnc_activityLevel(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.activityLevel");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)
	{
		unsigned int v,t;
		wnd->activityMeter(&v,&t);
		buffer.append(KviStr::Format,"%u",v);
	} else buffer.append('0');
	return c->leaveStackFrame();
}


/*
	@doc: window.activityTemperature
	@type:
		function
	@title:
		$window.activityTemperature
	@short:
		Returns the current activity temperature of a window
	@syntax:
		$window.activityTemperature
		$window.activityTemperature(<window_id>)
	@description:
		Returns the current activity temperature of the window specified by <window_id>.
		The form without parameters works on the current window.[br]
		The activity temperature describes the type of traffic in the window and is
		strictly related to the [fnc]$window.activityLevel[/fnc].[br]
		On channels and queries the temperature describes the type of the actions
		performed by the users. High temperatures denote more "human" behaviour (like
		speaking to the channel, changing the topic etc...),
		low temperatures denote automatic behaviours (like changing the channel limit
		or mode: actions often performed by bots).[br]
	@seealso:
		[fnc]$window.activityLevel[/fnc]
*/


static bool window_module_fnc_activityTemperature(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.activityTemperature");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)
	{
		unsigned int v,t;
		wnd->activityMeter(&v,&t);
		buffer.append(KviStr::Format,"%u",t);
	} else buffer.append('0');
	return c->leaveStackFrame();
}

/*
	@doc: window.isMinimized
	@type:
		function
	@title:
		$window.isMinimized
	@short:
		Checks if a window is currently minimized
	@syntax:
		$window.isMinimized
		$window.isMinimized(<window_id>)
	@description:
		Returns 1 if the window specified by <window_id> is currently minimized and 0 otherwise.
		The form with no parameters works on the current window. If the specified window
		doesn't exist then 0 is returned.
	@seealso:
		[fnc]$window.isMaximized[/fnc](), [cmd]window.maximize[/cmd],
		[cmd]window.minimize[/cmd], [cmd]window.restore[/cmd]
*/


static bool window_module_fnc_isMinimized(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.isMinimized");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->isMinimized() ? '1' : '0');
	else buffer.append('0');
	return c->leaveStackFrame();
}


/*
	@doc: window.hasInput
	@type:
		function
	@title:
		$window.hasInput
	@short:
		Checks if a window has an input field
	@syntax:
		$window.hasInput
		$window.hasInput(<window_id>)
	@description:
		Returns 1 if the window specified by <window_id> has an input field and 0 otherwise.
		The form with no parameters works on the current window. If the specified window
		doesn't exist then 0 is returned.
	@seealso:
		[fnc]$window.hasOutput[/fnc]
*/


static bool window_module_fnc_hasInput(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.hasInput");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->input() ? '1' : '0');
	else buffer.append('0');
	return c->leaveStackFrame();
}


/*
	@doc: window.hasUserFocus
	@type:
		function
	@title:
		$window.hasUserFocus
	@short:
		Checks if a window has the user focus
	@syntax:
		$window.hasUserFocus
		$window.hasUserFocus(<window_id>)
	@description:
		Returns 1 if the window specified by <window_id> has
		currently the user focus and 0 otherwise.
		The form with no parameters works on the current window.
		If the specified window doesn't exist then 0 is returned.
		A window has the user focus if it is the KVIrc's active
		window and has the user's input focus (i.e. typing
		on the keyboard will write in this window).
	@seealso:
*/

static bool window_module_fnc_hasUserFocus(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.hasUserFocus");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)
	{
		bool b = (wnd == g_pActiveWindow) && wnd->isActiveWindow();
		buffer.append(b ? '1' : '0');
	} else buffer.append('0');
	return c->leaveStackFrame();
}

/*
	@doc: window.console
	@type:
		function
	@title:
		$window.console
	@short:
		Returns the console that a window is attacched to
	@syntax:
		$window.console
		$window.console(<window_id>)
	@description:
		Returns the id of the console window that the window specified by window_id is attacched to.
		The console is the main (and only) console of the IRC context. If window_id is missing then
		the current window is used. If this window does not belong to an irc context (and thus has
		no attacched console) then 0 is returned.
	@seealso:
*/


static bool window_module_fnc_console(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"window::$console");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->console() ? wnd->console()->id() : "0");
	else buffer.append("0");
	return c->leaveStackFrame();
}

/*
	@doc: window.hasOutput
	@type:
		function
	@title:
		$window.hasOutput
	@short:
		Checks if a window has a text output widget
	@syntax:
		$window.hasOutput
		$window.hasOutput(<window_id>)
	@description:
		Returns 1 if the window specified by <window_id> has a text output widget and 0 otherwise.
		The form with no parameters works on the current window. If the specified window
		doesn't exist then 0 is returned.
	@seealso:
		[fnc]$window.hasInput[/fnc]
*/


static bool window_module_fnc_hasOutput(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"window::$hasOutput");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->view() ? '1' : '0');
	else buffer.append('0');
	return c->leaveStackFrame();
}

/*
	@doc: window.exists
	@type:
		function
	@title:
		$window.exists
	@short:
		Checks for the existence of a window
	@syntax:
		$window.exists(<window_id>)
	@description:
		Returns 1 if a specified window exists
	@seealso:
*/

static bool window_module_fnc_exists(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	buffer.append(g_pApp->findWindow(parms->safeFirstParam()) ? '1' : '0');
	return true;
}


/*
	@doc: window.type
	@type:
		function
	@title:
		$window.type
	@short:
		Returns the type of a window
	@syntax:
		$window.type
		$window.type(<window_id>)
	@description:
		Returns the type of the window with <window_id>.[br]
		The form with no parameters returns the type of the current window.[br]
		If the window with the specified id does not exist, an empty string is returned.[br]
	@seealso:
*/


static bool window_module_fnc_type(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.type");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->typeString());
	return c->leaveStackFrame();
}



/*
	@doc: window.caption
	@type:
		function
	@title:
		$window.caption
	@short:
		Returns the caption of a window
	@syntax:
		$window.caption
		$window.caption(<window_id>)
	@description:
		Returns the caption of the window with <window_id>.[br]
		The form with no parameters returns the caption of the current window.[br]
		If the window with the specified id does not exist, an empty string is returned.[br]
	@seealso:
*/


static bool window_module_fnc_caption(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.caption");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->plainTextCaption());
	return c->leaveStackFrame();
}

/*
	@doc: window.listtypes
	@type:
		command
	@title:
		window.listtypes
	@short:
		Lists available types of windows
	@syntax:
		window.listtypes
	@description:
		Lists the types of windows that are built in the current release of KVIrc.[br]
		This is actually a command and not a static list just because new window
		types may be added in subsequent releases.[br]
	@seealso:
		[cmd]window.listtypes[/cmd]
*/

static bool window_module_cmd_listtypes(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window.listtypes");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
	c->window()->listWindowTypes();
	return c->leaveStackFrame();
}


/*
	@doc: window.list
	@type:
		function
	@title:
		$window.list
	@short:
		Generates lists of windows
	@syntax:
		$window.list(<type>[,<irc_context_id>])
	@description:
		Returns an array of window identifiers with a speficied type and eventually belonging to a specified
		irc context.[br]
		<type> is a window type such as 'query' or 'channel'.[br]
		See [cmd]window.listtypes[/cmd] for a list of available window types in this KVIrc release.[br]
		If <type> is the special word 'all', all the window types are listed.[br]
		<irc_context_id> specifies the irc context in which the windows are searched.[br]
		If no <irc_context_id> is specified , the current one is used.[br]
		If <irc_context_id> is the special word 'all', all the irc context are searched.[br]
		If <irc_context_id> is the special word 'none' then only windows not belonging to any
		irc context are listed.[br]
		The special word 'any' used as <irc_context_id> merges the effects of 'all' and 'none'
		by searching all the irc contexts AND the windows not belonging to any irc context.[br]
		The windows that do not belong to any irc context (such as DCC windows), must be searched
		by using 'none' or 'any' as <irc_context_id>.
	@examples:
		[example]
			[comment]# List all the queries of the current irc context[/comment]
			[cmd]echo[/cmd] $window.list(query)
			[comment]# Equivalent to the above[/comment]
			[cmd]echo[/cmd] $window.list(query,[fnc]$ic[/fnc])
			[comment]# List all the channels in all the irc contexts[/comment]
			[cmd]echo[/cmd] $window.list(channel,all)
			[comment]# List all the windows in the current irc context[/comment]
			[cmd]echo[/cmd] $window.list(all)
			[comment]# List all the windows in all irc contexts[/comment]
			[cmd]echo[/cmd] $window.list(all,all)
			[comment]# List all the DCC Send windows: They don't belong to any irc context[/comment]
			[cmd]echo[/cmd] $window.list(dccsend,none)
			[comment]# List all the user windows created with $window.open[/comment]
			[comment]# They may either belong to an irc context or not[/comment]
			[cmd]echo[/cmd] $window.list(userwnd,any)
			[comment]# Ok , let's use it[/comment]
			[comment]# A nice alias that allows iterating commands through all the consoles[/comment]
			[comment]# Note the array returned by $window.list[/comment]
			[comment]# This is by LatinSuD :)[/comment]
			[cmd]alias[/cmd](iterate)
			{
				%ctxt[]=[fnc]$window.list[/fnc](console,all)
				[cmd]for[/cmd](%i=0;%i<%ctxt[]#;%i++)
				{
					[cmd]eval[/cmd] -r=%ctxt[%i] $0-
				}
			}
			iterate [cmd]echo[/cmd] Hi ppl! :)
			[comment]# The returned array works nicely also in [cmd]foreach[/cmd][/comment]
			[comment]# Say hi to all the channels :)[/comment]
			[cmd]alias[/cmd](sayallchans)
			{
				[cmd]foreach[/cmd](%x,[fnc]$window.list[/fnc](channel,all))
						[cmd]say[/cmd] -r=%x $0-;
			}
			sayallchans Hi ppl :)
		[/example]
	@seealso:
		[cmd]window.listtypes[/cmd]
*/

static bool window_module_fnc_list(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.list");

	// all , all
	// all , context
	// type , all
	// type , context

	KviStr * type = parms->safeFirst();
	KviStr * cntx = parms->safeNext();

	if(type->isEmpty())return c->error(KviError_notEnoughParameters,__tr("Window type or 'all' expected as first parameter"));

	c->beginListArrayOrDictionaryReturnIdentifier();

	int idx = 0;

	if(kvi_strEqualCI(cntx->ptr(),"all"))
	{
		// all contexts but no "no_context" windows
		bool bAllWindows = kvi_strEqualCI(type->ptr(),"all");
		QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

		while(KviWindow * wnd = it.current())
		{
			if(wnd->context())
			{
				if(bAllWindows)
				{
					c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
				} else {
					if(kvi_strEqualCI(wnd->typeString(),type->ptr()))
					{
						c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
					}
				}
			}
			++it;
		}
	} else if(kvi_strEqualCI(cntx->ptr(),"any"))
	{
		// all contexts and also "no_context" windows
		bool bAllWindows = kvi_strEqualCI(type->ptr(),"all");
		QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

		while(KviWindow * wnd = it.current())
		{
			if(bAllWindows)
			{
				c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
			} else {
				if(kvi_strEqualCI(wnd->typeString(),type->ptr()))
				{
					c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
				}
			}
			++it;
		}
	} else if(kvi_strEqualCI(cntx->ptr(),"none"))
	{
		// only "no_context" windows
		bool bAllWindows = kvi_strEqualCI(type->ptr(),"all");
		QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

		while(KviWindow * wnd = it.current())
		{
			if(!wnd->context())
			{
				if(bAllWindows)
				{
					c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
				} else {
					if(kvi_strEqualCI(wnd->typeString(),type->ptr()))
					{
						c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
					}
				}
			}
			++it;
		}

	} else {
		// some specified context
		unsigned int uId = 0;

		if(cntx->hasData())
		{
			// specific context
			bool bOk;
			uId = cntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidParameter,__tr("Invalid IRC context id '%s'"),cntx->ptr());
		} else {
			// current irc context
			if(!c->window()->console())return c->noIrcContext();
			uId = c->window()->console()->ircContextId();
		}

		bool bAllWindows = kvi_strEqualCI(type->ptr(),"all");
		QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

		while(KviWindow * wnd = it.current())
		{
			if(wnd->console())
			{
				if(wnd->console()->ircContextId() == uId)
				{
					if(bAllWindows)
					{
						c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
					} else {
						if(kvi_strEqualCI(wnd->typeString(),type->ptr()))
						{
							c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
						}
					}
				}
			}
			++it;
		}
	}

	return c->leaveStackFrame();
}


/*
	@doc: window.open
	@type:
		function
	@title:
		$window.open
	@short:
		Creates and opens a new window
	@syntax:
		$window.open([<flags>[,<caption>[,<irc_context>[,<icon>]]]])
	@description:
		Creates a new window, opens it and returns its window identifier.
		<flags> may be any combination of the following flag characters:[br]
		[b]i[/b]: Causes the window to have an input field in that
		the user can type commands or text. The text typed is reported
		by the [event:ontextinput]OnTextInput[/event] event.[br]
		[b]m[/b]: Creates a window that is initially minimized.[br]
		[b]q[/b]: Don't print warning messages during the creation.[br]
		If <caption> is given then the new window will have it as the initial plain text <caption>.
		You can change the caption later by calling [cmd]window.setCaption[/cmd].[br]
		If <irc_context> is given then the new window is bound to the specified irc context
		and will be destroyed when the attacched console closes.
		If <irc_context> is omitted or is 0 then the window will be context free (not bound
		to any context) and will exist until it is closed by the GUI, by a [cmd]window.close[/cmd]
		call or until KVIrc terminates. When <irc_context> is given but is not valid
		then a warning is printed (unless the q flag is used) and the created window is context free.[br]
		You will generally use the [fnc]$context[/fnc] function to retrieve the current IRC context id.[br]
		[br]
		<icon> is intepreted as the index of the internal icon to be used
		for the window. If <icon> is omitted then a default icon is used.[br]
	@examples:
		[example]
			%w = $window.open()
			[cmd]window.close[/cmd] %w
			%w = $window.open("m","My funky window")
			[cmd]window.close[/cmd] %w
			%w = $window.open("im","My funky window 2",$context,10)
		[/example]
	@seealso:
		[cmd]window.close[/cmd]
*/

static bool window_module_fnc_open(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$window.open");

	KviStr * flags = parms->safeFirst();
	KviStr * caption = parms->safeNext();
	KviStr * ctx = parms->safeNext();
	KviStr * icon = parms->safeNext();
	
	bool bOk = false;
	int iIcon = icon->toInt(&bOk) % KVI_NUM_SMALL_ICONS;
	if((!bOk) || (iIcon < 0))iIcon = KVI_SMALLICON_USERWINDOW;

	int iFlags = 0;
	if(flags->contains('i'))iFlags |= KviUserWindow::HasInput;


	bool bContextGiven = !ctx->isEmpty();
	KviConsole * pConsole = 0;
	if(bContextGiven)
	{
		unsigned int uCtx = ctx->toUInt(&bOk);
		if(bOk)
		{
			pConsole = g_pApp->findConsole(uCtx);
		}
		if(!pConsole)
		{
			if(!flags->contains('q'))c->warning(__tr2qs("The specified IRC context is not valid: creating a context free window"));
		}
	}

	KviUserWindow * wnd = new KviUserWindow(
						c->window()->frame(),
						caption->ptr(),
						iIcon,
						pConsole,
						iFlags);

	c->window()->frame()->addWindow(wnd,!flags->contains('m'));
	if(flags->contains('m'))wnd->minimize();

	buffer.append(wnd->id());
	return c->leaveStackFrame();
}

/*
	@doc: window.setCaption
	@type:
		command
	@title:
		window.setCaption
	@short:
		Sets the caption of an user window
	@syntax:
		window.setCaption [-q] <window_id> <plain_text_caption> [html_inactive_caption] [html_active_caption]
	@description:
		Sets the caption of the user window specified by <window_id> to <plain_text_caption>.[br]
		If <window_id> is an empty string then the current window is assumed.[br]
		The window must be of type userwnd and must have been created by [fnc]$window.open[/fnc]:
		it is not possible to change the caption of other window types.[br]
		If the window is not of the expected type then a warning is printed unless the -q switch is used.[br]
		If [html_inactive_caption] and/or [html_active_caption] are given then
		the html versions of the captions displayed in the window caption bars
		are set too. If one of these parameters is missing then <plain_text_caption> is
		used for the html versions too. The html captions can contain limited html code. In particular
		you're allowed to use the &lt;nobr&gt;,&lt;font&gt;,&lt;b&gt; and &lt;i&gt; html tags.
		It is better to avoid using colored fonts since you can't know which color scheme the
		user will have set.[br]
		If the specified window does not exist a warning is printed unless the -q switch is used.
	@seealso:
*/

static bool window_module_cmd_setCaption(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"window::setCaption");
	GET_WINDOW_BY_FIRST_TOKEN
	
	KviStr sz1,sz2,sz3;
	if(!g_pUserParser->parseCmdSingleToken(c,sz1))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,sz2))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,sz3))return false;

	if(sz2.isEmpty())sz2 = sz1;
	if(sz3.isEmpty())sz3 = sz1;

	if(wnd)
	{
		if(wnd->type() == KVI_WINDOW_TYPE_USERWINDOW)
		{
			((KviUserWindow *)wnd)->setCaptionStrings(sz1.ptr(),sz2.ptr(),sz3.ptr());
		} else {
			if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified window is not of type \"userwnd\""));
		}
	}
	return c->leaveStackFrame();
}

static bool window_module_init(KviModule *m)
{
	g_pUserWindowList = new KviPtrList<KviUserWindow>();
	g_pUserWindowList->setAutoDelete(false);

	m->registerFunction("list",window_module_fnc_list);
	m->registerCommand("listtypes",window_module_cmd_listtypes);

	m->registerFunction("open",window_module_fnc_open);
	m->registerFunction("exists",window_module_fnc_exists);
	m->registerFunction("type",window_module_fnc_type);
	m->registerFunction("caption",window_module_fnc_caption);
	m->registerFunction("isMaximized",window_module_fnc_isMaximized);
	m->registerFunction("isMinimized",window_module_fnc_isMinimized);
	m->registerFunction("hasInput",window_module_fnc_hasInput);
	m->registerFunction("hasOutput",window_module_fnc_hasOutput);
	m->registerFunction("hasUserFocus",window_module_fnc_hasUserFocus);
	m->registerFunction("console",window_module_fnc_console);
	m->registerFunction("activityLevel",window_module_fnc_activityLevel);
	m->registerFunction("activityTemperature",window_module_fnc_activityTemperature);
	
	m->registerCommand("close",window_module_cmd_close);
	m->registerCommand("activate",window_module_cmd_activate);
	m->registerCommand("dock",window_module_cmd_dock);
	m->registerCommand("undock",window_module_cmd_undock);
	m->registerCommand("maximize",window_module_cmd_maximize);
	m->registerCommand("minimize",window_module_cmd_minimize);
	m->registerCommand("restore",window_module_cmd_restore);
	m->registerCommand("clearOutput",window_module_cmd_clearOutput);
	m->registerCommand("demandAttention",window_module_cmd_demandAttention);
	m->registerCommand("setCaption",window_module_cmd_setCaption);

	// saveOutput (view()->saveBuffer())
/*
	m->registerFunction("geometry",window_module_fnc_geometry);
	m->registerCommand("setGeometry",window_module_cmd_setGeometry);

	m->registerFunction("isActive",window_module_fnc_isActive);

	// Input stuff
	m->registerFunction("inputText",window_module_fnc_inputText);
	m->registerCommand("setInputText",window_module_cmd_setInputText);

	// Output stuff
	m->registerFunction("outputBufferSize",window_module_fnc_outputBufferSize);
	m->registerCommand("setOutputBufferSize",window_module_cmd_setOutputBufferSize);

	m->registerFunction("outputShowsImages",window_module_fnc_outputShowsImages);
	m->registerCommand("setOutputShowsImages",window_module_cmd_setOutputShowsImages);
	
	m->registerFunction("outputTimestamp",window_module_fnc_outputTimestamp);
	m->registerCommand("setOutputTimestamp",window_module_cmd_setOutputTimestamp);

	m->registerFunction("hasOutputBackgroundPixmap",window_module_fnc_hasOutputBackgroundPixmap);
	m->registerCommand("setOutputBackgroundPixmap",window_module_cmd_setOutputBackgroundPixmap);

*/
	return true;
}

static bool window_module_cleanup(KviModule *m)
{
	while(KviUserWindow * w = g_pUserWindowList->first())
		w->close();
	delete g_pUserWindowList;
	return true;
}

static bool window_module_can_unload(KviModule *m)
{
	return g_pUserWindowList->isEmpty();
}

KVIRC_MODULE(
	"Window",                                               // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001-2004 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"KVIrc window management functions",
	window_module_init,
	window_module_can_unload,
	0,
	window_module_cleanup
)
