/*****************************************************************************
 * Copyright (C) 2004-2009 Christoph Thielecke <crissi99@gmx.de>             *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This package is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this package; if not, write to the Free Software               *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA *
 *****************************************************************************/

#include "importprofiledialog.h"

#include <QTextStream>
#include <QUrl>
#include <QtCore/QFile>
#include <QtCore/QFileInfo>
#include <QtGui/QCheckBox>

#include <kconfig.h>
#include <kconfiggroup.h>
#include <kdialog.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ktemporaryfile.h>
#include <kurl.h>
#include <kurlrequester.h>

#include <string>

#include "ciscopasswddecoder.h"
#include "utils.h"

ImportProfileDialog::ImportProfileDialog(KVpncConfig *GlobalConfig, QWidget *parent, const QString& caption, QString file)
        : KDialog(parent)
{
    Q_UNUSED(caption);

    QWidget *page = new QWidget(this);
    setMainWidget(page);
    setupUi(page);
    decodeEncPasswd = false;
    if (!file.isEmpty())
        filename = file;
    else
        filename = "";
    importOk = false;
    this->GlobalConfig = GlobalConfig;


    FilenameUrlrequester->setFilter("*.pcf");
    FilenameUrlrequester->setUrl(filename);

}


ImportProfileDialog::~ImportProfileDialog()
{
}

void ImportProfileDialog::accept()
{

    //filename="/etc/CiscoSystemsVPNClient/Profiles/hs_harz.pcf";
    filename = FilenameUrlrequester->url().toLocalFile();
    if (!filename.isEmpty()) {
        f = new QFile(filename);
        canAccept();
    }

    else {
        KMessageBox::sorry(0, i18n("Filename cannot be empty."), i18n("Empty Filename"));
    }
}


void ImportProfileDialog::canAccept()
{

    if (!f->exists()) {
        KMessageBox::information(0, i18n("File not found."), i18n("No File"));

        //  emit progress( 100 );
        return ;
    }

    KConfig confighandle(filename, KConfig::SimpleConfig);

    QStringList grouplist = confighandle.groupList();

    if (GlobalConfig->KvpncDebugLevel > 0) {
        QString groups = "";
        for (QStringList::Iterator group = grouplist.begin(); group != grouplist.end(); ++group)
            groups += QString(" " + *group);
        GlobalConfig->appendLogEntry(i18n("PCF import: groups found: [ %1 ]", groups), KVpncEnum::debug);
    }
    KConfigGroup config = confighandle.group("main");

    // sample config

    /*
    [main]
    Description=
    Host=192.168.13.1
    AuthType=1 // nur auslesen
    GroupName = hs_harz
    GroupPwd =
    Username = u15119
    SaveUserPassword = 0
    UserPassword =
    NTDomain =
    EnableBackup = 0
    BackupServer =
    EnableMSLogon = 1
    TunnelingMode = 0
    TcpTunnelingPort = 10000
    CertStore = 0
    CertName =
    CertPath =
    CertSubjectName =
    CertSerialHash = 00000000000000000000000000000000
    SendCertChain = 0
    VerifyCertDN =
    DHGroup = 2
    ForceKeepAlives = 0
    PeerTimeout = 90
    EnableLocalLAN = 1 // only reading because we do not want to do this
    EnableSplitDNS = 1
    */
    QString Description = config.readEntry("Description", "");

    if (Description.isEmpty())
        Description = config.readEntry("!Description", i18n("Profile imported from file %1.", filename));

    if (!Description.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: description found: %1", Description), KVpncEnum::debug);
    }

    QString Gateway = config.readEntry("Host", "");

    if (Gateway.isEmpty())
        Gateway = config.readEntry("!Host", "");

    if (!Gateway.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: gateway found: %1", Gateway), KVpncEnum::debug);
    }
    if (GlobalConfig->KvpncDebugLevel > 4) {
        GlobalConfig->appendLogEntry(i18n("PCF import: gateway: %1", Gateway), KVpncEnum::debug);
    }


    QString GroupName = config.readEntry("GroupName" , "");

    if (GroupName.isEmpty())
        GroupName = config.readEntry("!GroupName", "");

    if (GroupName.isEmpty())
        GroupName = "importedProfile";

    if (!GroupName.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: group name found: %1", GroupName), KVpncEnum::debug);
    }

    QString GroupPwd = config.readEntry("GroupPwd" , "");

    if (GroupName.isEmpty())
        GroupPwd = config.readEntry("!GroupPwd", "");

    if (!GroupName.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: group password found: %1", GroupPwd), KVpncEnum::debug);
    }

    QString enc_GroupPwd = config.readEntry("enc_GroupPwd", "");

    if (enc_GroupPwd.isEmpty())
        enc_GroupPwd = config.readEntry("!enc_GroupPwd", "");

    if (!enc_GroupPwd.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: encrypted group password found: %1", enc_GroupPwd), KVpncEnum::debug);
    }

    QString Username = config.readEntry("Username" , "");

    if (!Username.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: username found: %1", Username), KVpncEnum::debug);
    }

    bool saveUserPassword = (bool)config.readEntry("SaveUserPassword", false);

    if (saveUserPassword == false)
        (bool) config.readEntry("!SaveUserPassword", false);

    if (GlobalConfig->KvpncDebugLevel > 0) {
        if (saveUserPassword) {
            GlobalConfig->appendLogEntry(i18n("PCF import: save user pass : %1",
                                              i18n("yes")),
                                         KVpncEnum::debug);
        } else {
            GlobalConfig->appendLogEntry(i18n("PCF import: save user pass : %1",
                                              i18n("no")),
                                         KVpncEnum::debug);
        }
    }

    QString UserPassword = config.readEntry("UserPassword", "");

    if (UserPassword.isEmpty())
        UserPassword = config.readEntry("!UserPassword", "");

    if (!UserPassword.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: user password found: %1", UserPassword), KVpncEnum::debug);
    }

    QString enc_UserPassword = config.readEntry("enc_UserPassword", "");

    if (enc_UserPassword.isEmpty())
        enc_UserPassword = config.readEntry("!enc_UserPassword", "");

    if (!enc_UserPassword.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: encrypted user password found: %1", enc_UserPassword), KVpncEnum::debug);
    }

    QString NtDomain = config.readEntry("NTDomain");

    if (NtDomain.isEmpty())
        NtDomain = config.readEntry("!NTDomain", "");

    if (!NtDomain.isEmpty()) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: NT domain found: %1", NtDomain), KVpncEnum::debug);
    }

    // bool enableBackup = (bool) config.readEntry( "EnableBackup" , false );
    QString BackupServer = config.readEntry("BackupServer", "");
    bool enableMSLogon = (bool) config.readEntry("EnableMSLogon", false);
    // int TunnelingMode = (int) config.readEntry( "TunnelingMode", -1 );
    // int CertStore = (int) config.readEntry( "CertStore", -1 );
    QString CertName = config.readEntry("CertName", "");

    if (GlobalConfig->KvpncDebugLevel > 0)
        GlobalConfig->appendLogEntry(i18n("PCF import: certificate name found: %1", CertName), KVpncEnum::debug);

    QString CertPath = config.readEntry("CertPath", "");
    QString CertSubjectName = config.readEntry("CertSubjectName", "");
    QString CertSerialHash = config.readEntry("CertSerialHash", "");
    // bool SendCertChain = (bool) config.readEntry( "SendCertChain" , "" );
    // bool VerifyCertDN = (bool) config.readEntry( "VerifyCertDN", false );
    int DHGroup = (int) config.readEntry("DHGroup", -1);
    int LocalPort = 500;

    if (GlobalConfig->KvpncDebugLevel > 0)
        GlobalConfig->appendLogEntry(i18n("PCF import: Diffie Hellman group found: %1", QString().setNum(DHGroup)), KVpncEnum::debug);

    // bool ForceKeepAlives = (bool) config.readEntry( "ForceKeepAlives", false );

    // bool EnableLocalLAN = (bool) config.readEntry( "EnableLocalLAN", false ); // nur auslesen aber immer aus :)
    // bool EnableSplitDNS = (bool) config.readEntry( "EnableSplitDNS", false );

    bool useUdp;
    if ((int) config.readEntry("EnableNat", 0) == 1) {
        useUdp = true;
    } else {
        useUdp = false;
    }

    if (GlobalConfig->KvpncDebugLevel > 0) {
        if (useUdp) {
            GlobalConfig->appendLogEntry(i18n("PCF import: enable NAT mode : %1",
                                              i18n("yes")),
                                         KVpncEnum::debug);
        } else {
            GlobalConfig->appendLogEntry(i18n("PCF import: enable NAT mode : %1",
                                              i18n("no")),
                                         KVpncEnum::debug);
        }
    }

    bool saveGroupPwd = true;
    bool usePerfectSecrecy = false;
    bool useSingleDes = false;
    bool useLocalPort = false;

    bool useNtDomainName = false;
    if (enableMSLogon && !NtDomain.isEmpty())
        useNtDomainName = true;

//  if ( TunnelingPort != 10000 )
//  {
//   useUdp=true; //FIXME: is this right? I guess its only on udp
//   useUdpPort = true;
//  }

    bool useIkeGroup = false;
    QString IkeGroup;
    if (DHGroup != -1) {
        IkeGroup =  "dh" + QString().setNum(DHGroup) ;
        useIkeGroup = true;
    }

    int PeerTimeout = (int) config.readEntry("PeerTimeout", -1);

    if (PeerTimeout > -1) {
        if (GlobalConfig->KvpncDebugLevel > 0)
            GlobalConfig->appendLogEntry(i18n("PCF import: peer timeout found: %1", QString().setNum(PeerTimeout)), KVpncEnum::debug);
    }

    QString PerfectForwardSecrecy = ""; //QString("dh"+QString().setNum(DHGroup));
    // bool usePerfectForwardSecrety = false;

    CiscoPasswdDecoder dec(filename);
    QString userpasswd = "";
    QString grouppasswd = "";
    dec.decodePasswords(userpasswd, grouppasswd);

    UserPassword = userpasswd;
    GroupPwd = grouppasswd;

    if (GroupPwd.isEmpty())
        saveGroupPwd = false;

    if (UserPassword.isEmpty() && enc_UserPassword.isEmpty())
        saveUserPassword = false;

    if (Description.isEmpty()) {
        Description = (QFileInfo(filename).fileName());
        Description = Utils(this->GlobalConfig).removeSpecialCharsForFilename(Description.left(filename.section('/', -1).length() - 4));
    }

    if (GroupName.isEmpty()) {
        GroupName = Utils(this->GlobalConfig).removeSpecialCharsForFilename(GroupName.left(filename.section('/', -1).length() - 4));
    }

    bool useApplicationVersion = false;
    bool useGlobalIpsecSecret = false;


    VpnAccountData::ConnectionType ConnType = VpnAccountData::cisco;
    QString ProfileName = QFileInfo(f->fileName().trimmed().remove(".pcf").remove(".PCF")).fileName();
    acc = new VpnAccountData(ConnType, ProfileName);

    /*
    AuthType=
    The authentication type of the user:
    1 = Pre-shared keys (default)
    3 = Digital Certificate using an RSA signature.
    5 = Mutual authentication (hybrid)
    */
    int AuthType = (int) config.readEntry("AuthType" , -1);

    if (AuthType == -1)
        AuthType = (int) config.readEntry("!AuthType", -1);

    if (GlobalConfig->KvpncDebugLevel > 0)
        if (AuthType == 1)
            GlobalConfig->appendLogEntry(i18n("PCF import: authentication type found: %1", i18n("PSK")), KVpncEnum::debug);
    if (AuthType == 3)
        GlobalConfig->appendLogEntry(i18n("PCF import: authentication type found: %1", i18n("certificate")), KVpncEnum::debug);
    if (AuthType == 5)
        GlobalConfig->appendLogEntry(i18n("PCF import: authentication type found: %1", i18n("hybrid")), KVpncEnum::debug);

    if (AuthType == 3 || AuthType == 5) {
        // vpnc has no cert support :(
        acc->setConnectionType(VpnAccountData::ciscoorig);
        acc->setAuthType(VpnAccountData::cert);

        if (!CertName.isEmpty())
            acc->setX509Certificate(CertName);
    }
    if (AuthType == 1)
        acc->setAuthType(VpnAccountData::psk);



    acc->setDescription(Description);
    acc->setGateway(Gateway);
    acc->setID(GroupName);
    acc->setPreSharedKey(GroupPwd);
    acc->setUserName(Username);
    acc->setUserPassword(UserPassword);
    acc->setSaveUserPassword(saveUserPassword);
    acc->setSavePsk(saveGroupPwd);
    acc->setNtDomainName(NtDomain);
    acc->setPerfectForwardSecrety(PerfectForwardSecrecy);
    acc->setIkeGroup(IkeGroup);
    acc->setUseApplicationVersion(useApplicationVersion);
    acc->setUseGlobalIpsecSecret(useGlobalIpsecSecret);
    acc->setUseIkeGroup(useIkeGroup);
    acc->setUseLocalPort(useLocalPort);
    acc->setNtDomainName(NtDomain);
    acc->setUseSingleDes(useSingleDes);
    //acc->setUseAdvancedSettings( useAdvancedSettings );
    acc->setUseAdvancedSettings(true);
    acc->setUsePerfectForwardSecrety(usePerfectSecrecy);
    acc->setUseUdp(useUdp);
//  acc->setUdpPort( 10000 );

    if (useLocalPort) {
        acc->setLocalPort(LocalPort);
        acc->setUseLocalPort(true);
    }

//  if (useUdpPort){
//   acc->setUdpPort(TunnelingPort);
//   acc->setUseUdpPort(true);
//  }

    if (PeerTimeout > -1) {
        // read minutes but store seconds
        acc->setPeerTimeout(PeerTimeout*60);
    }

    /*
    acc.setName( Description );
    acc.setGateway( Gateway );
    acc.setID( GroupName );
    acc.setGroupPassword( GroupPwd );
    acc.setUserName( Username );
    acc.setUserPassword( UserPassword );
    acc.setSaveUserPassword( saveUserPassword );
    acc.setSaveGroupPassword( true );
    //acc.setIkeGroup( QString IkeGroup );
    acc.setPerfectForwardSecrety( QString PerfectForwardSecrecy );
    acc.setNtDomainName( QString Name );
    acc.setApplicationVersion( QString version );
    acc.setUseSingleDes( bool useSingleDes );
    acc.setLocalPort( int port );
    acc.setUseIkeGroup( bool useIkeGroup);
    acc.setUsePerfectForwardSecrety(bool usePerfectForwardSecrety);
    acc.setUseNtDomainName(bool useNtDomainName);
    acc.setUseApplicationVersion(bool useApplicationVersion);
    acc.setUseLocalPort(bool useLocalPort);
    acc.setUseAdvancedSettings(bool useAdvancedSettings);
    acc.setUseGlobalIpsecSecret(bool useGlobalIpsecSecret);
    */
    importOk = true;
    QDialog::accept();
}

#include "importprofiledialog.moc"
