/***************************************************************************
 *   Copyright (C) 2004 by Matthias Reif                                   *
 *   matthias.reif@informatik.tu-chemnitz.de                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kencryptiongroup.h"

KEncryptionGroup::KEncryptionGroup( QString title, QWidget *parent, const char *name ) : QButtonGroup( title, parent, name )
{
	offRadio = new QRadioButton( i18n( "off" ), this );
	openRadio = new QRadioButton( i18n( "open" ), this );
	restrictedRadio = new QRadioButton( i18n( "restricted" ), this );
	offRadio->adjustSize();
	openRadio->adjustSize();
	restrictedRadio->adjustSize();
	offRadio->move( KWaveControl::SPACING, 25 );

	openRadio->move( offRadio->x() + offRadio->width() + KWaveControl::SPACING, offRadio->y() );
	restrictedRadio->move( openRadio->x() + openRadio->width() + KWaveControl::SPACING, openRadio->y() );

	keyLabel = new QLabel( "Key:", this );
	keyLabel->adjustSize();
	keyLabel->move( offRadio->x(), offRadio->y() + offRadio->height() + KWaveControl::SPACING );
	
	keySelection = new KComboBox( this );
	keySelection->insertItem( "8888-8888-8888-8888" );
	keySelection->adjustSize();
	keySelection->clear();
	keySelection->move( keyLabel->x() + keyLabel->width() + KWaveControl::SPACING, keyLabel->y() );

	editButton = new QToolButton( this );
	editButton->setTextLabel( i18n( "edit" ), true );
	editButton->setIconSet( QIconSet( KGlobal::iconLoader()->loadIcon( "edit", KIcon::Small ) ) );
	editButton->move( keySelection->x() + keySelection->width() + KWaveControl::SPACING, keySelection->y() );
	editButton->adjustSize();
	
	adjustSize();

	errorLabel = new QLabel( this );
	errorLabel->hide();
	errorLabel->setAlignment( Qt::AlignCenter );

	connect( offRadio, SIGNAL( toggled( bool ) ), this, SLOT( offToggled( bool ) ) );
	connect( editButton, SIGNAL( clicked() ), this, SLOT( editSlot() ) );
	connect( keySelection, SIGNAL( activated( int ) ), this, SLOT( keyActivated( int ) ) );

	connect( this, SIGNAL( clicked( int ) ), this, SLOT( modeChangedSlot( int ) ) );
}


KEncryptionGroup::~KEncryptionGroup()
{
}


/*!
    \fn KEncryptionGroup::setCurrentKey( Encoding encoding )
 */
void KEncryptionGroup::setCurrentKey( Encoding encoding )
{
	offRadio->setChecked( encoding.mode == IW_ENCODE_DISABLED );
	openRadio->setChecked( encoding.mode == IW_ENCODE_OPEN );
	restrictedRadio->setChecked( encoding.mode == IW_ENCODE_RESTRICTED );
	keySelection->setCurrentItem( encoding.key - 1 );
}


/*!
    \fn KEncryptionGroup::setKeys( vector< EKey > keys )
 */
void KEncryptionGroup::setKeys( vector< EKey > keys )
{
	keySelection->clear();
	if( keys.size () > 0 )
	{
		hideError();
		unsigned int i;
		for( i = 0; i < keys.size(); i++ )
		{
			if( keys[i].length > 0 )
			{
				keySelection->insertItem( QString( keys[ i ].formatedKey ) );
				delete [] keys[ i ].key;
				delete [] keys[ i ].formatedKey;
			}
			else
			{
				keySelection->insertItem( "[off]" );
			}
		}
	}
	else
	{
		showError( QString( strerror( errno ) ) );
	}
}


/*!
    \fn KEncryptionGroup::showError( QString error )
 */
void KEncryptionGroup::showError( QString error )
{
	errorLabel->show();
	errorLabel->setText( error );
	errorLabel->adjustSize();
	errorLabel->move( ( this->width() - errorLabel->width() ) / 2, ( this->height() - errorLabel->height() ) / 2 );
	offRadio->hide();
	openRadio->hide();
	restrictedRadio->hide();
	keyLabel->hide();
	keySelection->hide();
	editButton->hide();
}


/*!
    \fn KEncryptionGroup::hideError()
 */
void KEncryptionGroup::hideError()
{
	offRadio->show();
	openRadio->show();
	restrictedRadio->show();
	keyLabel->show();
	keySelection->show();
	editButton->show();
	errorLabel->hide();
}


/*!
    \fn KEncryptionGroup::offToggled( bool on )
 */
void KEncryptionGroup::offToggled( bool on )
{
	keyLabel->setEnabled( ! on );
	keySelection->setEnabled( ! on );
	editButton->setEnabled( ! on );
}


/*!
    \fn KEncryptionGroup::editSlot()
 */
void KEncryptionGroup::editSlot()
{
	KKeyDialog* keyDialog = new KKeyDialog( keySelection->currentText(), this, "Key" );
	if( keyDialog->exec() == QDialog::Accepted )
	{
		unsigned char* key = NULL;
		int keyLength;
		QString temp;
		switch( keyDialog->getKeyType() )
		{
			case ASCII:
				key = (unsigned char*)keyDialog->getKey().ascii();
				keyLength = keyDialog->getKey().length();
				break;
			case HEX:
				key = hexToKey( keyDialog->getKey(), keyLength );
				break;
			case LOGIN:
				key = (unsigned char*)keyDialog->getKey().ascii();
				keyLength = keyDialog->getKey().length();
				char* p = strchr( (const char*)key, ':' );
				if( p == NULL )
				{
					KMessageBox::error( this, "Invalid login format", "Error" );
					return;
				}
				else
				{
					*p = '\0';
				}
				break;
		}
		changed( keySelection->currentItem() + 1, key, keyLength );
	}
	delete keyDialog;
}


/*!
    \fn KEncryptionGroup::getMode()
 */
int KEncryptionGroup::getMode()
{
	if( offRadio->isChecked() )
		return IW_ENCODE_DISABLED;
	else if( openRadio->isChecked() )
		return IW_ENCODE_OPEN;
	else
		return IW_ENCODE_RESTRICTED;
}


/*!
    \fn KEncryptionGroup::keyActivated( int index )
 */
void KEncryptionGroup::keyActivated( int index )
{
	changed( index + 1 );
}


/*!
    \fn KEncryptionGroup::modeChangedSlot( int )
 */
void KEncryptionGroup::modeChangedSlot( int )
{
	
	modeChanged( getMode() );
}


/*!
    \fn KEncryptionGroup::hexToKey( QString hex, int& length )
 */
unsigned char* KEncryptionGroup::hexToKey( QString hex, int& length )
{
	QString temp = hex.remove( QRegExp( "[^a-fA-F0-9]" ) );
	length = temp.length() / 2 + temp.length() % 2;
	unsigned char* key = new unsigned char[ length ];
	unsigned int i;
	for( i = 0; i < temp.length() ; i += 2 )
	{
		key[ i / 2 ] = temp.mid( i, 2 ).toInt( 0, 16 );
	}
	return key;
}


/*!
    \fn KEncryptionGroup::getCurrentKey()
 */
QString KEncryptionGroup::getCurrentKey()
{
	return keySelection->currentText();
}


/*!
    \fn KEncryptionGroup::getCurrentIndex()
 */
int KEncryptionGroup::getCurrentIndex()
{
	return keySelection->currentItem();
}


#include "kencryptiongroup.moc"
