//
// C++ Implementation: serenityclient
//
// Author: Remi Villatel <maxilys@tele2.fr>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include <kdeversion.h>
#ifndef KDE_IS_VERSION
#define KDE_IS_VERSION(a, b, c) 0
#endif

#include <klocale.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>

#include <qapplication.h>
#include <qcursor.h>
#include <qfontmetrics.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qtooltip.h>

#include "serenityclient.h"
#include "serenityhandler.h"
#include "serenitybutton.h"
#include "pixmaps.h"
#include "enums.h"


SerenityClient::SerenityClient(KDecorationBridge *bridge, KDecorationFactory *factory)
 : KDecoration(bridge, factory), m_titleBar(0), 
				captionBuffer(QPixmap(0, 0)), 
				captionStyle(-1), 
				captionActive(false)
{
}

SerenityClient::~SerenityClient()
{
	// Delete buttons from memory.
	for (int n=0; n<ButtonTypeCount; n++)
	{
		if (m_button[n]) delete m_button[n];
	}
}

void SerenityClient::init()
{
	createMainWidget(WResizeNoErase | WRepaintNoErase);
	widget()->setBackgroundMode(NoBackground);	// To avoid flicker
	widget()->installEventFilter(this);

	int position;
	if ( maximizeMode() != MaximizeFull 
	|| (SerenityHandler::maxCorner() == true && SerenityHandler::noMaxFrame() == false) ) 
	{
		position = SerenityHandler::titlePosition();
	}
	else
	{
		position = 0;
	}
	switch (position)
	{
		case 1:		// Minimum left
		{
			leftSpacerPolicy = QSizePolicy::Fixed;
			rightSpacerPolicy = QSizePolicy::Expanding;
			break;
		}
		case 2:		// Minimum right
		{
			leftSpacerPolicy = QSizePolicy::Expanding;
			rightSpacerPolicy = QSizePolicy::Fixed;
			break;
		}
		case 0:		// Maximum width
		default:
		{
			leftSpacerPolicy = QSizePolicy::Fixed;
			rightSpacerPolicy = QSizePolicy::Fixed;
			break;
		}
	}

	// Set up layout
	int borderSize = SerenityHandler::borderSize();
	int buttonSize = SerenityHandler::buttonSize();
	int spacing = SerenityHandler::buttonSpacing();

	mainLayout = new QVBoxLayout(widget());
	//
	QBoxLayout::Direction direction;
	if (QApplication::reverseLayout())
		direction = QBoxLayout::RightToLeft;
	else
		direction = QBoxLayout::LeftToRight;
	//
	m_topSpacer = new QSpacerItem(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed);
	//
	titleLayout = new QBoxLayout(0, direction, 0, 0, 0);
	m_titleBar = new QSpacerItem(buttonSize, buttonSize, QSizePolicy::Expanding, QSizePolicy::Fixed);
	m_leftButtonSpacer = new QSpacerItem(borderSize, buttonSize, (QSizePolicy::SizeType)leftSpacerPolicy, QSizePolicy::Fixed);
	m_rightButtonSpacer = new QSpacerItem(borderSize, buttonSize, (QSizePolicy::SizeType)rightSpacerPolicy, QSizePolicy::Fixed);
	m_decoSpacer = new QSpacerItem(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed);
	//
	midLayout = new QBoxLayout(0, direction, 0, 0, 0);
	//
	m_bottomSpacer = new QSpacerItem(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed);

	for (int n=0; n<ButtonTypeCount; n++) m_button[n] = 0; // Reset buttons

	leftButtonLayout = new QBoxLayout(0, direction, 0, 0, 0);
	leftButtonLayout->setSpacing(SerenityHandler::extraSpacing());
	rightButtonLayout = new QBoxLayout(0, direction, 0, 0, 0);
	rightButtonLayout->setSpacing(SerenityHandler::extraSpacing());
	// Add the left buttons
	leftButtonLayout->addItem(m_leftButtonSpacer); // Left button margin
	addButtons(leftButtonLayout, options()->customButtonPositions() ? options()->titleButtonsLeft() : QString(default_left));
	// Add the right buttons
	addButtons(rightButtonLayout, options()->customButtonPositions() ? options()->titleButtonsRight() : QString(default_right));
	rightButtonLayout->addItem(m_rightButtonSpacer); // Right button margin

	titleLayout->addLayout(leftButtonLayout); // Contains the left buttons
	titleLayout->addSpacing(spacing);
	titleLayout->addItem(m_titleBar); // The text
	titleLayout->addSpacing(spacing);
	titleLayout->addLayout(rightButtonLayout); // Contains the right buttons

	midLayout->addSpacing(borderSize);
	if (isPreview())
	{
		midLayout->addWidget(new QLabel(i18n("<center><b>Serenity 1.4 Preview</b></center>"), widget() ), 1);
	}
	else
	{
		midLayout->addItem(new QSpacerItem(0, 0));
	}
	midLayout->addSpacing(borderSize);

	// Layout order
	mainLayout->addItem(m_topSpacer);
	mainLayout->addLayout(titleLayout);
	mainLayout->addItem(m_decoSpacer);
	mainLayout->addLayout(midLayout);
	mainLayout->addItem(m_bottomSpacer);

	/// MXLS: More connections
	connect( this, SIGNAL( keepAboveChanged( bool )), SLOT( keepAboveChange( bool )));
	connect( this, SIGNAL( keepBelowChanged( bool )), SLOT( keepBelowChange( bool )));
}

bool SerenityClient::eventFilter(QObject *o, QEvent *e)
{
	if (o != widget())
		return false;

	switch (e->type())
	{
		case QEvent::Resize:
			resizeEvent(static_cast<QResizeEvent*>(e));
			return true;
		case QEvent::Paint:
			paintEvent(static_cast<QPaintEvent*>(e));
			return true;
		case QEvent::MouseButtonDblClick:
			mouseDoubleClickEvent(static_cast<QMouseEvent*>(e));
			return true;
		case QEvent::MouseButtonPress:
			processMousePressEvent(static_cast<QMouseEvent*>(e));
			return true;
		case QEvent::Show:
			showEvent(static_cast<QShowEvent*>(e));
			return true;
#if KDE_IS_VERSION(3, 5, 0)
		case QEvent::Wheel:
			wheelEvent( static_cast< QWheelEvent* >( e ));
			return true;
#endif
		default:
			return false;
	}
}

void SerenityClient::addButtons(QBoxLayout *layout, const QString &s)
{
	int wide = SerenityHandler::buttonSize() * 3/2;
	if (wide & 1)	// Odd size
		wide++;
	//
	if (s.length() > 0)
	{
		for (uint n=0; n<s.length(); n++)
		{
			switch (s[n])
			{
				case 'M': // Menu
					if (!m_button[ButtonMenu])
					{
						m_button[ButtonMenu] = new SerenityButton(this, "menu", i18n("Menu"), ButtonMenu);
						if (SerenityHandler::menuWide()) 
							m_button[ButtonMenu]->setFixedWidth(wide);
						connect(m_button[ButtonMenu], SIGNAL(pressed()), this, SLOT(menuButtonPressed()));
						layout->addWidget(m_button[ButtonMenu], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
					break;
				case 'H': // Help
					if (!m_button[ButtonHelp] && providesContextHelp())
					{
						m_button[ButtonHelp] = new SerenityButton(this, "help", i18n("Help"), ButtonHelp);
						if (SerenityHandler::helpWide()) 
							m_button[ButtonHelp]->setFixedWidth(wide);
						connect(m_button[ButtonHelp], SIGNAL(clicked()), this, SLOT(showContextHelp()));
						layout->addWidget(m_button[ButtonHelp], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
					break;
				case 'I': // Minimize
					if ((!m_button[ButtonMin]) && isMinimizable())
					{
						m_button[ButtonMin] = new SerenityButton(this, "minimize", i18n("Minimize"), ButtonMin);
						if (SerenityHandler::minWide()) 
							m_button[ButtonMin]->setFixedWidth(wide);
						connect(m_button[ButtonMin], SIGNAL(clicked()), this, SLOT(minimize()));
						layout->addWidget(m_button[ButtonMin], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
					break;
				case 'A': // Maximize
					if ((!m_button[ButtonMax]) && isMaximizable())
					{
						m_button[ButtonMax] = new SerenityButton(this, "maximize", (maximizeMode()!=MaximizeRestore)?i18n("Minimize"):i18n("Maximize"), ButtonMax);
						if (SerenityHandler::maxWide()) 
							m_button[ButtonMax]->setFixedWidth(wide);
						connect(m_button[ButtonMax], SIGNAL(clicked()), this, SLOT(maxButtonPressed()));
						layout->addWidget(m_button[ButtonMax], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
					break;
				case 'X': // Close
                  			if ((!m_button[ButtonClose]) && isCloseable())
					{
						m_button[ButtonClose] = new SerenityButton(this, "close", i18n("Close"), ButtonClose);
						if (SerenityHandler::closeWide()) 
							m_button[ButtonClose]->setFixedWidth(wide);
						connect(m_button[ButtonClose], SIGNAL(clicked()), this, SLOT(closeWindow()));
						layout->addWidget(m_button[ButtonClose], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
					break;
				case 'S': // OnAllDesktops
					if (!m_button[ButtonOnAllDesktops])
					{
						m_button[ButtonOnAllDesktops] = new SerenityButton(this, "onAllDesktops", isOnAllDesktops() ? i18n("Not On All Desktops") : i18n("On All Desktops"), ButtonOnAllDesktops);
						if (SerenityHandler::stickyWide()) 
							m_button[ButtonOnAllDesktops]->setFixedWidth(wide);
						m_button[ButtonOnAllDesktops]->setOnAllDesktops(isOnAllDesktops());
						connect(m_button[ButtonOnAllDesktops], SIGNAL(clicked()), this, SLOT(toggleOnAllDesktops()));
						layout->addWidget(m_button[ButtonOnAllDesktops], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
					break;
				case 'F': // Above all others
					if (!m_button[ButtonAbove])
					{
						m_button[ButtonAbove] = new SerenityButton(this, "above", i18n("Keep Above Others"), ButtonAbove);
						if (SerenityHandler::aboveWide()) 
							m_button[ButtonAbove]->setFixedWidth(wide);
						m_button[ButtonAbove]->setKeepAbove(isKeptAbove);
						connect(m_button[ButtonAbove], SIGNAL(clicked()), this, SLOT(aboveButtonPressed()));
						layout->addWidget(m_button[ButtonAbove], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
				break;
				case 'B': // Below all others
					if (!m_button[ButtonBelow])
					{
						m_button[ButtonBelow] = new SerenityButton(this, "below", i18n("Keep Below Others"), ButtonBelow);
						if (SerenityHandler::belowWide()) 
							m_button[ButtonBelow]->setFixedWidth(wide);
						m_button[ButtonBelow]->setKeepBelow(isKeptBelow);
						connect(m_button[ButtonBelow], SIGNAL(clicked()), this, SLOT(belowButtonPressed()));
						layout->addWidget(m_button[ButtonBelow], 0, Qt::AlignHCenter | Qt::AlignTop);
					}
				break;
				case '_': // Spacer
					if ( (n != 0) && (n != (s.length()-1)) )
						layout->addSpacing(SerenityHandler::buttonSpacing() 
								- SerenityHandler::extraSpacing());
			}
		}
	}
}

void SerenityClient::paintEvent(QPaintEvent*)
{
	if (!SerenityHandler::initialized()) return;

	bool active = isActive();

	QPainter p(widget());

	QRect titleRect(m_titleBar->geometry());
	QRect midRect(m_decoSpacer->geometry());
	QRect bottomRect(m_bottomSpacer->geometry());

	int borderSize = SerenityHandler::borderSize();
	bool shaded = isShade();
	int midY = midRect.top()-2;
	int rr = width()-1; //rightRect.right();
	int bb = bottomRect.bottom();
	int leftX = 0;
	int rightX = rr;
	int position;
	QRegion mask;
	if ( maximizeMode()!=MaximizeFull 
	|| (maximizeMode()==MaximizeFull && SerenityHandler::maxCorner() == true 
		&& SerenityHandler::noMaxFrame() == false ) ) 
	{
		position = SerenityHandler::titlePosition();
		switch (position)
		{
			case 1:		// Minimum left
			{
				leftX = 0;
				rightX = rr 
					- QRect(m_rightButtonSpacer->geometry()).width() 
					+ borderSize;
				if (rightX > rr-borderSize)
				{
					rightX = rr;
					position = 0;
				}
				break;
			}
			case 2:		// Minimum right
			{
				leftX = QRect(m_leftButtonSpacer->geometry()).width() 
					- borderSize;
				rightX = rr;
				if (leftX < borderSize)
				{
					leftX = 0;
					position = 0;
				}
				break;
			}
			case 0:		// Maximum width
			default:
			{
				leftX = 0;
				rightX = rr;
				break;
			}
		}
		if (shaded) midY += 7;
		//
		int botLeft = 0;
		int botRight = rr;
		mask = QRegion(leftX, 0, rightX-leftX+1, midY+1);
		mask += QRegion(0, midY+1, rr+1, bb-midY+1);
		// Radius 4
		// Remove top left corner
		mask -= QRegion(leftX, 0, 4, 1);
		mask -= QRegion(leftX, 1, 2, 1);
		mask -= QRegion(leftX, 2, 1, 2);
		if (position == 2)
		{
			mask -= QRegion(0, midY+1, 4, 1);
			mask -= QRegion(0, midY+2, 2, 1);
			mask -= QRegion(0, midY+3, 1, 2);
			if (shaded) botLeft = leftX;
		}
		// Remove top right corner
		mask -= QRegion(rightX-3, 0, 4, 1);
		mask -= QRegion(rightX-1, 1, 2, 1);
		mask -= QRegion(rightX, 2, 1, 2);
		if (position == 1)
		{
			mask -= QRegion(rr-3, midY+1, 4, 1);
			mask -= QRegion(rr-1, midY+2, 2, 1);
			mask -= QRegion(rr, midY+3, 1, 2);
			if (shaded) botRight = rightX;
		}
		// Remove bottom left corner
		mask -= QRegion(botLeft, bb, 4, 1);
		mask -= QRegion(botLeft, bb-1, 2, 1);
		mask -= QRegion(botLeft, bb-3, 1, 2);
		// Remove bottom right corner
		mask -= QRegion(botRight-3, bb, 4, 1);
		mask -= QRegion(botRight-1, bb-1, 2, 1);
		mask -= QRegion(botRight, bb-3, 1, 2);
	}
	else
	{
		position = 0;
		mask = QRegion(0, 0, width(), bottomRect.bottom()+1);
	}
	setMask(mask);

	// MXLS: Palette
	QPalette pal = QApplication::palette();
	QColor ground = pal.color(QPalette::Active, QColorGroup::Background);
	QColor textColor = KDecoration::options()->color(KDecoration::ColorFont, active);
	QColor frmBase;
	if (SerenityHandler::useFrameColor() == true)
		frmBase = options()->color(ColorFrame, active);
	else
		frmBase = ground;
	//
	int yy = titleRect.height()+borderSize*2;
	QRect frame;
	// The whole titlebar (almost)
	frame.setCoords(0, 0, rr, yy);
	QRegion clip = QRegion(frame) - QRegion(titleRect);
	p.setClipRegion(clip);
	p.fillRect(frame, ground);
	p.setClipping(false);
	bool definitelyFramed = false;
	if ( (maximizeMode() != MaximizeFull)
	|| (maximizeMode()==MaximizeFull && SerenityHandler::maxCorner() == true 
		&& SerenityHandler::noMaxFrame() == false) )
	{
		definitelyFramed = true;
	}
	if (active && SerenityHandler::zenBorder() && definitelyFramed) 
	{
		// Some colors...
		int contrast = SerenityHandler::gradientContrast();
		QColor gradientTopLeft = Pixmaps::colorMix(ground, frmBase, 96).light(100+contrast*3);
		QColor gradientBottomRight = frmBase.dark(100+contrast*4);
		// ...to draw some gradients.
		int width = rr - borderSize*2 + 1;
		int height = bb - borderSize*2 + 1;
		int yb = yy-borderSize-1;
		// Top
		Pixmaps::renderGradient(&p, 
					QRect(leftX+borderSize, 0, rightX-leftX, borderSize), 
					gradientTopLeft, ground, 
					Pixmaps::VerticalGradient);
		Pixmaps::renderDiagonalGradient(&p, 
					QRect(leftX, 0, borderSize, borderSize), 
					gradientTopLeft, gradientTopLeft, ground);
		Pixmaps::renderDiagonalGradient(&p, 
					QRect(rightX-borderSize+1, 0, borderSize, borderSize), 
					gradientTopLeft, ground, gradientBottomRight);
		// Bottom
		Pixmaps::renderDiagonalGradient(&p, 
					QRect(0, bb-borderSize+1, borderSize, borderSize), 
					gradientTopLeft, ground, gradientBottomRight);
		Pixmaps::renderGradient(&p, 
					QRect(borderSize, bb-borderSize+1, width, borderSize), 
					ground, gradientBottomRight, 
					Pixmaps::VerticalGradient);
		Pixmaps::renderDiagonalGradient(&p, 
					QRect(rr-borderSize+1, bb-borderSize+1, borderSize, borderSize), 
					ground, gradientBottomRight, gradientBottomRight);
		// Middle
		Pixmaps::renderGradient(&p, 
					QRect(0, borderSize, borderSize, height), 
					gradientTopLeft, ground, 
					Pixmaps::HorizontalGradient);
		Pixmaps::renderGradient(&p, 
					QRect(rr-borderSize+1, borderSize, borderSize, height), 
					ground, gradientBottomRight, 
					Pixmaps::HorizontalGradient);
		if (position == 1)
		{
			Pixmaps::renderGradient(&p, 
					QRect(rightX-borderSize+1, borderSize, borderSize, titleRect.height()), 
					ground, gradientBottomRight, 
					Pixmaps::HorizontalGradient);
			if (!shaded)
			{
				Pixmaps::renderGradient(&p, 
						QRect(rightX, yb, rr-rightX, borderSize), 
						gradientTopLeft, ground, 
						Pixmaps::VerticalGradient);
				Pixmaps::renderDiagonalGradient(&p, 
						QRect(rr-borderSize+1, yb, borderSize, borderSize), 
						gradientTopLeft, ground, gradientBottomRight);
				Pixmaps::renderSpecialCorner(&p, 
						rightX-borderSize+1, yb, borderSize, 
						gradientBottomRight, ground, gradientTopLeft);
			}
			else
			{
				Pixmaps::renderDiagonalGradient(&p, 
						QRect(rightX-borderSize+1, yb, borderSize, borderSize), 
						ground, gradientBottomRight, gradientBottomRight);
			}
		}
		else if (position == 2)
		{
			Pixmaps::renderGradient(&p, 
					QRect(leftX, borderSize, borderSize, titleRect.height()), 
					gradientTopLeft, ground, 
					Pixmaps::HorizontalGradient);
			if (!shaded)
			{
				Pixmaps::renderGradient(&p, 
						QRect(borderSize, yb, leftX-borderSize+1, borderSize), 
						gradientTopLeft, ground, 
						Pixmaps::VerticalGradient);
				Pixmaps::renderDiagonalGradient(&p, 
						QRect(0, yb, borderSize, borderSize), 
						gradientTopLeft, gradientTopLeft, ground);
				Pixmaps::renderDiagonalGradient(&p, 
						QRect(leftX, yb, borderSize, borderSize), 
						gradientTopLeft, ground, ground);
			}
			else
			{
				Pixmaps::renderDiagonalGradient(&p, 
						QRect(leftX, yb, borderSize, borderSize), 
						gradientTopLeft, ground, gradientBottomRight);
			}
		}
	}
	else	// No gradients.
	{
		// The two sides (almost)
		frame.setCoords(0, yy, 6, bb);
		p.fillRect(frame, ground);
		frame.setCoords(rr-6, yy, rr, bb);
		p.fillRect(frame, ground);
		// The bottom (almost)
		frame.setCoords(0, bb-6, rr, bb);
		p.fillRect(frame, ground);
	}
	/// Now we're going to draw the (external) frame...
	if (definitelyFramed)
	{
		// Radius = 4
		if (shaded)
		{
			uniframe(p, leftX, 0, rightX, bb, 4, frmBase, textColor);
		}
		else
		{
			uniframe(p, 0, 0, rr, bb, 4, frmBase, textColor);
			if (position == 1)
			{
				emptyCorner(p, rightX-1, 0, rr, midY+1, 
						true, frmBase, textColor);
			}
			else if (position == 2)
			{
				emptyCorner(p, 0, 0, leftX+1, midY+1, 
						false, frmBase, textColor);
			}
		}
	}
	else 	// (maximizeMode() == MaximizeFull)
	{
		// Square corners (on request)...
		// If we have to draw a frame.
		if (SerenityHandler::noMaxFrame() == false)
		{
			uniframe(p, 0, 0, rr, bb, 
					(SerenityHandler::maxCorner() == true ? 4 : 1), 
					frmBase, textColor);
		}
	}
	/// Draw title (caption).
	if (titleRect.width() > 0)	// Do we even have to paint the title?
	{
		int buttonStyle = SerenityHandler::buttonStyle();
		if ( (captionActive != active)
		|| (captionBuffer.size() != titleRect.size()) 
		|| (captionStyle != buttonStyle) )
		{
			QColor titlebar = KDecoration::options()->color(KDecoration::ColorTitleBar, active);
			QColor surface = Qt::white;
			if (buttonStyle == 2)
			{
				surface = Pixmaps::colorMix(ground, 
							pal.color(QPalette::Active, QColorGroup::Foreground), 
							216);
			}
			else if (buttonStyle == 3)
			{
				surface = titlebar;
			}
			else if (buttonStyle == 4)
			{
				surface = Pixmaps::colorMix(ground, titlebar, 160);
			}
			if (qGray(surface.rgb()) < 64)
				surface = surface.light(150);
			//
			captionBuffer.resize(titleRect.width(), titleRect.height());
			QPainter bufferPainter(&captionBuffer);
			//
			if ( SerenityHandler::noTitleFrame()	// No caption framing
			|| (buttonStyle == 1) || (buttonStyle == 7) )
			{
				captionBuffer.fill(ground);
			}
			else if ( (buttonStyle == 0) || (buttonStyle == 5) 
			|| (buttonStyle == 6) ) 
			{
				Pixmaps::renderSurface(&bufferPainter, 
					QRect( 0, 0, titleRect.width(), titleRect.height() ), 
					ground, titlebar, titlebar, 0);
			}
			else if ( (buttonStyle == 2) || (buttonStyle == 3) )
			{
				Pixmaps::renderFlatArea(&bufferPainter, 
					QRect( 0, 0, titleRect.width(), titleRect.height() ), 
					ground, surface);
			}
			else if (buttonStyle == 4)
			{
				captionBuffer.fill(ground);
				Pixmaps::renderCarving(&bufferPainter, 
					//titleRect, 
					QRect( 0, 0, titleRect.width(), titleRect.height() ), 
					ground, surface);
			}
			captionActive = active;
			captionStyle = buttonStyle;
		}
		QPixmap carbon = captionBuffer;
		QPainter cp(&carbon);
		// Draw caption text
		QRect caR = carbon.rect();
		caR.addCoords(2, 1, -2, -1);	// Leave some room around eventually.
		cp.setFont(options()->font(active, false));
		cp.setPen(options()->color(ColorFont, active));
		cp.drawText( caR, 
			(SerenityHandler::centerTitle() ? AlignHCenter : AlignAuto) | AlignVCenter, 
			reduced(caption(), caR.width(), 
				cp.fontMetrics()).stripWhiteSpace() );
		p.drawPixmap(titleRect.left(), titleRect.top(), carbon);
	}
}

void SerenityClient::uniframe(QPainter& current,
			int left, int top, int right, int bottom,
			int radius,
			QColor inner, QColor outer)
{
	// Useless since radius is always a constant.
	//if (radius<1) radius=1;
	//if (radius>4) radius=4;

	QColor contour;
	if (SerenityHandler::useFrameColor() == true)
		contour = options()->color( ColorFrame, isActive() );
	else
		contour = Pixmaps::colorMix(inner, outer, 184);
	QColor aliased = Pixmaps::colorMix(inner, contour);
	
	int width = (right-left)+1;
	int height = (bottom-top)+1;

	current.setPen(contour);
	current.drawRect(left, top, width, height);

	if (radius==4)
	{
		current.setPen(contour);
		current.drawLine(left+2, top+1, left+3, top+1);
		current.drawLine(left+1, top+2, left+1, top+3);
		//
		current.drawLine(right-2, top+1, right-3, top+1);
		current.drawLine(right-1, top+2, right-1, top+3);
		//
		current.drawLine(left+2, bottom-1, left+3, bottom-1);
		current.drawLine(left+1, bottom-2, left+1, bottom-3);
		//
		current.drawLine(right-2, bottom-1, right-3, bottom-1);
		current.drawLine(right-1, bottom-2, right-1, bottom-3);
		//
		current.setPen(aliased);
		current.drawPoint(left+1, top+4);
		current.drawPoint(left+2, top+2);
		current.drawPoint(left+4, top+1);
		//
		current.drawPoint(right-1, top+4);
		current.drawPoint(right-2, top+2);
		current.drawPoint(right-4, top+1);
		//
		current.drawPoint(left+1, bottom-4);
		current.drawPoint(left+2, bottom-2);
		current.drawPoint(left+4, bottom-1);
		//
		current.drawPoint(right-1, bottom-4);
		current.drawPoint(right-2, bottom-2);
		current.drawPoint(right-4, bottom-1);
	}
	else	// Radius == 1
	{
		current.setPen(aliased);
		current.drawPoint(left, top);
		current.drawPoint(right, top);
		current.drawPoint(left, bottom);
		current.drawPoint(right, bottom);
	}
}

void SerenityClient::emptyCorner(QPainter& current,
			int left, int top, int right, int bottom,
			bool onRight,
			QColor inner, QColor outer)
{
	QColor contour;
	if (SerenityHandler::useFrameColor() == true)
		contour = options()->color( ColorFrame, isActive() );
	else
		contour = Pixmaps::colorMix(inner, outer, 184);
	//
	QColor aliased = Pixmaps::colorMix(inner, contour);
	
	int width = (right-left)+1;
	int height = (bottom-top)+1;

	current.setPen(contour);
	if (onRight) 
	{
		current.drawRect(left+1, top, width-1, height);
		//
		current.drawLine(left-1, top+1, left-2, top+1);
		current.drawLine(left, top+2, left, top+3);
		//
		current.drawLine(right-2, bottom+1, right-3, bottom+1);
		current.drawLine(right-1, bottom+2, right-1, bottom+3);
		//
		current.setPen(aliased);
		current.drawPoint(left, top+4);
		current.drawPoint(left-1, top+2);
		current.drawPoint(left-3, top+1);
		//
		current.drawPoint(left+1, bottom);
		//
		current.drawPoint(right-1, bottom+4);
		current.drawPoint(right-2, bottom+2);
		current.drawPoint(right-4, bottom+1);
	}
	else
	{
		current.drawRect(left, top, width-1, height);
		//
		current.drawLine(right+1, top+1, right+2, top+1);
		current.drawLine(right, top+2, right, top+3);
		//
		current.drawLine(left+2, bottom+1, left+3, bottom+1);
		current.drawLine(left+1, bottom+2, left+1, bottom+3);
		//
		current.setPen(aliased);
		current.drawPoint(right, top+4);
		current.drawPoint(right+1, top+2);
		current.drawPoint(right+3, top+1);
		//
		current.drawPoint(right-1, bottom);
		//
		current.drawPoint(left+1, bottom+4);
		current.drawPoint(left+2, bottom+2);
		current.drawPoint(left+4, bottom+1);
	}
}

QString SerenityClient::reduced(QString str, int width, QFontMetrics fm)
{
	int wst = fm.width(str);
	int i;
	if (wst > width)	// To remove the name of the app.
	{
		i = str.findRev(" - ");
		if (i > 0)
			str = str.left(i);
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To remove "Something: "
	{
		i = str.find(": ");
		if (i > 0)
			str = str.mid(i+2);
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To change "file:///" into "/"
	{
		i = str.find(":///");
		if (i > 0)
			str = str.mid(i+3);
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To remove "protocol://"
	{
		i = str.find("://");
		if (i > 0)
			str = str.mid(i+3);
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To remove "anything:"
	{
		i = str.find(":");
		if (i > 0)
			str = str.mid(i+1);
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To shorten "(anything)"
	{
		i = str.findRev("(");
		if (i > 0)
			str = str.left(i) + "(...)";
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To shorten "[anything]"
	{
		i = str.findRev("[");
		if (i > 0)
			str = str.left(i) + "[...]";
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To remove "(...)"
	{
		i = str.findRev("(");
		if (i > 0)
			str = str.left(i);
	}
	else return str;
	//
	wst = fm.width(str);
	if (wst > width)	// To remove "[...]"
	{
		i = str.findRev("[");
		if (i > 0)
			str = str.left(i);
	}
	//			// To shorten a path
	while ( (fm.width(str) > width) && (str.find("/", 2) > 0) )
	{
		i = str.find("/", 2);
		if (i > 0)
			str = "..." + str.mid(i+1);
	}
	//
	return str;
}


void SerenityClient::showEvent(QShowEvent*)
{
	widget()->update();
}

void SerenityClient::resizeEvent(QResizeEvent*)
{
	if ((widget()->isVisibleToTLW()) && (!widget()->testWFlags(WStaticContents)))
	{
		QRegion region = widget()->rect();
		region.subtract(m_titleBar->geometry());
		widget()->erase(region);
	}
}

void SerenityClient::captionChange()
{
	widget()->repaint(m_titleBar->geometry(), false);
}

void SerenityClient::mouseDoubleClickEvent(QMouseEvent *e)
{
	if (m_titleBar->geometry().contains(e->pos()))
	{
		titlebarDblClickOperation();
	}
}

#if KDE_IS_VERSION(3, 5, 0)
void SerenityClient::wheelEvent(QWheelEvent *e)
{
	if (isSetShade() || titleLayout->geometry().contains( e->pos() ) )
	{
		titlebarMouseWheelOperation( e->delta());
	}
}
#endif

SerenityClient::Position SerenityClient::mousePosition(const QPoint &point) const
{
	const int corner = 21;
	const int borderSize = SerenityHandler::borderSize();
	SerenityClient::Position pos = PositionCenter;

	if (isShade() || !isResizable())
						pos=PositionCenter; 
	else if (point.y() <= borderSize)
	{
		// Inside top frame
		if (point.x() <= corner)	pos = PositionTopLeft;
		else if (point.x() >= (width()-corner))
						pos = PositionTopRight;
		else				pos = PositionTop;
	}
	else if (point.y() >= (height()-borderSize))
	{
		// Inside handle
		if (point.x() <= corner)	pos = PositionBottomLeft;
		else if (point.x() >= (width()-corner))
						pos = PositionBottomRight;
		else				pos = PositionBottom;
	}
	else if (point.x() <= borderSize)
	{
		// On left frame
		if (point.y() <= corner)	pos = PositionTopLeft;
		else if (point.y() >= (height()-corner))
						pos = PositionBottomLeft;
		else				pos = PositionLeft;
	}
	else if (point.x() >= width()-borderSize)
	{
		// On right frame
		if (point.y() <= corner)	pos = PositionTopRight;
		else if (point.y() >= (height()-corner))
						pos = PositionBottomRight;
		else				pos = PositionRight;
	}
	else
	{
		// Inside the frame
		pos = PositionCenter;
	}
	return pos;
}

void SerenityClient::iconChange()
{
	if (m_button[ButtonMenu])
	{
		m_button[ButtonMenu]->repaint(false);
	}
}

void SerenityClient::activeChange()
{
	// Repaint the buttons when state changes
	for (int n=0; n<ButtonTypeCount; n++)
		if (m_button[n]) m_button[n]->repaint(false);
	widget()->repaint(false);
}

void SerenityClient::maximizeChange()
{
	const bool m = (maximizeMode() != MaximizeRestore);
	if (m_button[ButtonMax])
	{
		m_button[ButtonMax]->setMaximized(m);
		m_button[ButtonMax]->setTipText(m ? i18n("Restore") : i18n("Maximize"));
	}
}

void SerenityClient::desktopChange()
{
	if (m_button[ButtonOnAllDesktops])
	{
		m_button[ButtonOnAllDesktops]->setOnAllDesktops(isOnAllDesktops());
		m_button[ButtonOnAllDesktops]->setTipText(isOnAllDesktops() ? i18n("Not On All Desktops") : i18n("On All Desktops"));
	}
}

void SerenityClient::keepAboveChange(bool a)
{
	if (m_button[ButtonAbove])
	{
		m_button[ButtonAbove]->setOn(a);
		m_button[ButtonAbove]->repaint(false);
	}
}

void SerenityClient::keepBelowChange(bool b)
{
	if (m_button[ButtonBelow])
	{
		m_button[ButtonBelow]->setOn(b);
		m_button[ButtonBelow]->repaint(false);
	}
}

void SerenityClient::maxButtonPressed()
{
	if (m_button[ButtonMax])
	{
		switch (m_button[ButtonMax]->lastMousePress())
		{
			case MidButton:
				maximize(maximizeMode() ^ MaximizeVertical);
				break;
			case RightButton:
				maximize(maximizeMode() ^ MaximizeHorizontal);
				break;
			default:
				maximize(maximizeMode() == MaximizeFull ? MaximizeRestore : MaximizeFull);
		}
	}
}

void SerenityClient::aboveButtonPressed()
{
	if (m_button[ButtonAbove])
		setKeepAbove(!keepAbove());
}

void SerenityClient::belowButtonPressed()
{
	if (m_button[ButtonBelow])
		setKeepBelow(!keepBelow());
}

void SerenityClient::menuButtonPressed()
{
	if (m_button[ButtonMenu])
	{
		static QTime* t = 0;
		static SerenityClient* lastClient = 0;
		if (t == 0) 
			t = new QTime;
		bool dbl = (lastClient == this && t->elapsed() <= QApplication::doubleClickInterval());
		lastClient = this;
		t->start();
		
		if (dbl)
		{	// Double Click on menu button...
			closeWindow();
			return;
		}
		else
		{
			QPoint pt(m_button[ButtonMenu]->rect().left(), m_button[ButtonMenu]->rect().bottom());
			showWindowMenu(m_button[ButtonMenu]->mapToGlobal(pt));
			m_button[ButtonMenu]->setDown(false);
		}
	}
}

QSize SerenityClient::minimumSize() const
{
	QString s1 = options()->customButtonPositions() ? options()->titleButtonsLeft() : QString(default_left) ;
	QString s2 = options()->customButtonPositions() ? options()->titleButtonsRight() : QString(default_right) ;
	int i = (s1.length()+s2.length()+2) * SerenityHandler::buttonSize();
	return QSize(i, i/3);
}

void SerenityClient::borders(int& left, int& right, int& top, int& bottom) const
{
	int leftPolicy = leftSpacerPolicy;
	int rightPolicy = rightSpacerPolicy;
	int borderSize = SerenityHandler::borderSize();
	int buttonSize = SerenityHandler::buttonSize();
	if (maximizeMode()==MaximizeFull)
	{
		if (SerenityHandler::noMaxFrame() == true || SerenityHandler::maxCorner() == false)
		{
			leftPolicy = QSizePolicy::Fixed;
			rightPolicy = QSizePolicy::Fixed;
		}
		if (SerenityHandler::noMaxFrame() == true)
			borderSize = 0;
	}
	m_titleBar->changeSize(10, buttonSize, QSizePolicy::Expanding, QSizePolicy::Fixed);
	m_leftButtonSpacer->changeSize(borderSize, buttonSize, (QSizePolicy::SizeType)leftPolicy, QSizePolicy::Fixed);
	m_rightButtonSpacer->changeSize(borderSize, buttonSize, (QSizePolicy::SizeType)rightPolicy, QSizePolicy::Fixed);
	
	left = right = borderSize;
	if (!isShade())
	{
		m_topSpacer->changeSize(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed );
		m_decoSpacer->changeSize(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed );
		m_bottomSpacer->changeSize(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed );
		top = borderSize + buttonSize + borderSize;
		bottom = borderSize;
	}
	else
	{
		m_topSpacer->changeSize(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed );
		m_decoSpacer->changeSize(10, 0, QSizePolicy::Expanding, QSizePolicy::Fixed );
		m_bottomSpacer->changeSize(10, borderSize, QSizePolicy::Expanding, QSizePolicy::Fixed );
		top = borderSize + buttonSize;
		bottom = borderSize;
	}
	// Activate updated layout
	widget()->layout()->activate();
}

void SerenityClient::reset(unsigned long) // unsigned long changed
{
	// TODO: Implementation
}

void SerenityClient::resize(const QSize& s)
{
	widget()->resize(s);
}


// Plugin Stuff
extern "C"
{
	KDecorationFactory *create_factory()
	{
		return new SerenityHandler();
	}
}

#include "serenityclient.moc"
