#include <qtimer.h>

#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpassdlg.h>

#include "configuration.h"
#include "kwlansuprocess.h"

KWlanSuProcess::KWlanSuProcess(QObject *parent, const char *name):
  SuProcessBase(parent, name)
{
  m_pwstate = 0;
  m_keep = 1; // Show "Keep password" checkbox
  m_restart = false;
  m_parent = parent;
}

KWlanSuProcess::~KWlanSuProcess()
{
  m_password.fill(0);
}

bool KWlanSuProcess::start(RunMode runmode, Communication comm)
{
  if (runmode == DontCare)
    kdWarning() << k_funcinfo << "KWlanSuProcess can be started only in NotifyOnExit or Block mode" << endl; 

  connect(this, SIGNAL( processExited(SuProcessBase *) ), 
    this, SLOT( slotProcessExited(SuProcessBase *) ));

  SuProcessBase::start(runmode, comm);
  return TRUE;
}

void KWlanSuProcess::setDescription(const QString &desc)
{
  m_desc = desc;
}

void KWlanSuProcess::setRestart(bool restart)
{
  m_restart = restart;
}

bool KWlanSuProcess::getPassword(QCString &pass)
{
  int result;

  //Presumably, when password is empty it is not required...
  if (!PasswordCache().password().isEmpty()) {
    pass = PasswordCache().password();
    return true;
  }

  QString prompt;
  if (m_useSudo) prompt= i18n("<qt>KWlan is going to"
                        "<p align=\"center\"><b>%1</b></p>"
			"and requires your password to continue. "
			"<p>If you choose to keep the password, you will not be asked "
			"to enter it again for some period of time.</p></qt>"
                       ).arg(m_desc);
  else prompt= i18n("<qt>KWlan is going to"
              "<p align=\"center\"><b>%1</b></p>"
              "and requires your root password to continue. "
              "<p>If you choose to keep the password, you will not be asked "
              "to enter it again for some period of time.</p></qt>"
                   ).arg(m_desc);
  m_pwstate = PasswordRequired;
  KPasswordDialog *passDialog = new KPasswordDialog(KPasswordDialog::Password, &m_keep, 0,0);
  passDialog->disableCoreDumps(); //We are keeping password in memory - it is bad idea to dump it.
  
  result = passDialog->getPassword(m_password, prompt, &m_keep);
  if (result == KPasswordDialog::Accepted) {
    pass = m_password;
    m_pwstate = PasswordEntered;
    return true;
  }
  else {
    m_pwstate = PasswordRefused;
    return false;  
  }
}

void KWlanSuProcess::slotProcessExited(SuProcessBase *proc)
{
  // First,check if auth is ok
  if (!proc->authFailed())
  {
    // Store the password, if needed
    if (m_keep && (m_pwstate == PasswordEntered)) {
      //Config().m_password = m_password;
      PasswordCache().setPassword(m_password);
    }  

    // Check exit status and restart/delete process if needed and permitted
    if (proc->exitStatus()) 
    {
      if (m_restart) {
        // We are allowed to restart - ask the user
        QString prompt = i18n("<qt>KWlan was unable to"
                              "<p align=\"center\"><b>%1</b></p>"
			      "This may be caused by temporary malfunction or system misconfiguration "
			      "(e.g. absence of the utility called). You may want to run this task again and see "
			      "if this helps. Do you want to restart the task?</qt>"
                             ).arg(m_desc); 

        int result = KMessageBox::questionYesNo(0, prompt, i18n("Error has occured"));
      
        // The user decided to try again
        if (result == KMessageBox::Yes) {
          m_pwstate = 0;
	  m_keep = 1;
	  QTimer::singleShot(0, proc, SLOT(restart()));
        } 
        // The user declined the request - just delete the process.
        else
          proc->deleteLater();    
      }
      else {
      // Error has occured but we are not allowed to restart - show message box and delete the process
        QString prompt = i18n("<qt>KWlan was unable to"
                              "<p align=\"center\"><b>%1</b></p>"
			      "This may be caused by system misconfiguration e.g. absence "
			      "of the utility called or incorrect parameters passed. "
			      "Please check this and try again."
                              ).arg(m_desc); 
			     
        KMessageBox::error(0, prompt, i18n("Error has occured"));
        proc->deleteLater();
      }
    }   
    // The process was finished successfully
    else
      proc->deleteLater(); 
      
  }
  // Authorization failed - check why
  else {
  
    // The user refused to enter the password - do not restart the process  
      
    if (m_pwstate == PasswordRefused) {
        /*
      QString prompt = i18n("KWlan requires root password to run some programs in priviledged mode. "
                            "Without this it will be unable to start WPA Supplicant or configure the network. "
			    "KWlan will continue to run but will not be able to do anything useful."
			    );
      KMessageBox::sorry(0, prompt, i18n("Password required"));
        */
      proc->deleteLater();
      
    }
      
    // The password was incorrect - complain and run again.
    // TODO: should we check m_restart here?
    else {
      // I don't think it's wise to show this dialog from UI perspective
      /*QString prompt = i18n("The password you entered was rejected by the system. "
                            "Please re-enter the password in the next dialog or cancel the task."
			   );    
      KMessageBox::error(0, prompt, i18n("Incorrect password"));*/
      
      m_pwstate = 0;
      m_keep = 1;
      QTimer::singleShot(0, proc, SLOT(restart()));      
    }
  }
}
