/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qpixmap.h>
#include <qpainter.h>
#include <qptrlist.h>
#include <qtabwidget.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qprogressdialog.h>
#include <qtoolbutton.h>
#include <knuminput.h>
#include <klineedit.h>
#include <kapplication.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <qevent.h>
#include "configuration.h"
#include "importimagedlg.h"
#include "kxstitch.h"
#include "flossscheme.h"
#include "floss.h"

ImportImageDlg::ImportImageDlg(const Magick::Image& image, QWidget *parent, const char *name, bool modal)
  : ImportImage(parent,name,modal),
    m_pixmap(new QPixmap),
    m_useFractionals(false),
    m_orig(image)
{
  m_originalSize = QSize(m_orig.columns(),m_orig.rows());
  QString caption = QString("Import Image - Image Size %1 x %2 pixels").arg(m_originalSize.width()).arg(m_originalSize.height());
  setCaption(caption);

  // get the users preferred pattern size in stitches
  double cc = KXSConfig().NewPattern_ClothCount;
  QString ccu = KXSConfig().NewPattern_ClothCountUnits;
  Import_ClothCount->setSuffix(ccu);
  Import_ClothCount->setRange(1,55,(ccu == "/cm")?0.01:1.0,false);
  Import_ClothCount->setValue(cc);
  m_preferredSize = QSize(KXSConfig().NewPattern_Width,KXSConfig().NewPattern_Height);
  int sw = m_preferredSize.width()*100/m_originalSize.width();
  int sh = m_preferredSize.height()*100/m_originalSize.height();
  int s = sw <? sh; // pick smallest scale

  Import_FlossScheme->insertStringList(((KXStitchApplication*)kapp)->schemes());
  Configuration::setComboBoxItem(Import_FlossScheme,KXSConfig().NewPattern_FlossScheme);
  Import_PatternScale->setValue(s);
  Import_UseMaxColors->setChecked(KXSConfig().Import_UseMaxColors);
  Import_MaxColors->setValue(KXSConfig().Import_MaxColors);

  connect(Import_FlossScheme, SIGNAL(activated(int)), this, SLOT(schemeChanged(int)));
  connect(Import_UseMaxColors, SIGNAL(stateChanged(int)), this, SLOT(maxColorsChanged(int)));
  connect(Import_MaxColors, SIGNAL(valueChanged(int)), this, SLOT(maxColorsChanged(int)));
  connect(Import_ClothCount, SIGNAL(valueChanged(double)), this, SLOT(clothCountChanged(double)));
  connect(Import_PatternScale, SIGNAL(valueChanged(int)), this, SLOT(scaleChanged(int)));
  connect(Import_UseFractionals, SIGNAL(stateChanged(int)), this, SLOT(useFractionalsChanged(int)));
  connect(ResetButton, SIGNAL(clicked()), this, SLOT(resetImage()));
  connect(HelpButton, SIGNAL(clicked()), this, SLOT(contextHelp()));

  createImageMap();
  renderPixmap();
}

ImportImageDlg::~ImportImageDlg()
{
  delete m_pixmap;
}

void ImportImageDlg::resetImage()
{
  m_image = m_orig;
  renderPixmap();
}

void ImportImageDlg::schemeChanged(int)
{
  createImageMap();
  renderPixmap();
}

void ImportImageDlg::calcSizes()
{
  m_image = m_orig;
  m_preferredSize = m_originalSize*Import_PatternScale->value()/100;
  QSize imageSize = m_preferredSize;
  if (m_useFractionals)
  {
    // double the size of the image
    imageSize *= 2;
  }
  Magick::Geometry geom(imageSize.width(), imageSize.height());
  geom.percent(false);
  m_image.sample(geom);
  clothCountChanged(Import_ClothCount->value());
}

void ImportImageDlg::maxColorsChanged(int)
{
  killTimers();
  startTimer(500);
}

void ImportImageDlg::clothCountChanged(double n)
{
  double wi = m_preferredSize.width();
  double hi = m_preferredSize.height();
  QString suffix;
  QString units = KXSConfig().NewPattern_Units;
  if (units != "Stitches")
  {
    wi /= n;
    hi /= n;
    suffix = Import_ClothCount->suffix().right(2);
  }
  Import_PatternSize->setText(QString(i18n("%1%2 x %3%4")).arg(wi,0,'g',3).arg(suffix).arg(hi,0,'g',3).arg(suffix));
}

void ImportImageDlg::scaleChanged(int)
{
  killTimers();
  startTimer(500);
}

void ImportImageDlg::useFractionalsChanged(int s)
{
  m_useFractionals = (s == QButton::On);
  killTimers();
  startTimer(500);
}

void ImportImageDlg::createImageMap()
{
  FlossScheme *scheme = ((KXStitchApplication*)kapp)->scheme(Import_FlossScheme->currentText());
  m_map = *(scheme->createImageMap());
}

void ImportImageDlg::renderPixmap()
{
  CropImageLabel->setCursor(waitCursor);
  calcSizes();
  m_image.modifyImage();
  m_pixmap->resize(m_image.columns(), m_image.rows());
  m_pixmap->fill();
  if (Import_UseMaxColors->isChecked())
  {
    // TODO add noise reduction routines to clean up converted images
    m_image.quantizeColorSpace(Magick::RGBColorspace);
    m_image.quantizeColors(Import_MaxColors->value());
    m_image.quantize();
  }
  m_image.map(m_map);
  m_image.modifyImage();

  QPainter p;
  p.begin(m_pixmap);
  int w = m_image.columns();
  int h = m_image.rows();
  int pixelCount = w*h;

  QProgressDialog progress(i18n("Rendering preview..."), i18n("Cancel"), pixelCount, this, i18n("Progress"), true);
  progress.setMinimumDuration(100);

  Magick::Pixels cache(m_image);
  const Magick::PixelPacket *pixels = cache.get(0,0,w,h);
  for (int cy = 0 ; cy < h ; cy++)
  {
    kapp->processEvents();
    progress.setProgress(cy*w);
    if (progress.wasCancelled()) break;
    for (int cx = 0 ; cx < w ; cx++)
    {
      Magick::PixelPacket packet = *pixels++;
      Magick::Color c = Magick::Color(packet.red, packet.green, packet.blue);
      if (!(packet.opacity))
      {
        QColor c(packet.red/257, packet.green/257, packet.blue/257);
        p.setPen(QPen(c));
        p.drawPoint(cx,cy);
      }
    }
  }
  p.end();
  CropImageLabel->setPixmap(*m_pixmap);
  CropImageLabel->setCursor(arrowCursor);
}

void ImportImageDlg::accept()
{
  if (Import_SetDefault->isChecked())
  {
    KXSConfig().Import_UseMaxColors = Import_UseMaxColors->isChecked();
    KXSConfig().Import_MaxColors = Import_MaxColors->value();
    KXSConfig().writeSettings();
  }
  QDialog::accept();
}

void ImportImageDlg::contextHelp()
{
  kapp->invokeHelp("ImportImageDlg");
}

void ImportImageDlg::timerEvent(QTimerEvent *e)
{
  killTimers();
  renderPixmap();
}