/***************************************************************************
 *   Copyright (C) 2005 by Roland Weigert   *
 *   roweigert@t-online.de   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "settingsmaindialog.h"
extern playerinterface *playerlink;

settingsmaindialog::settingsmaindialog(QWidget *parent)
{
    //only for debugging
    if (!parent)
        parent=this;
    mainsettingslayout=new QHBoxLayout(parent);
    mainsettingslayout->setSpacing(5);
    mainsettingslayout->setMargin(5);
    KIconLoader *loader=new KIconLoader();
    QPixmap pixmap;
    QString picname;
    splitfield=new QSplitter(QSplitter::Horizontal,parent);
    mainsettingslayout->addWidget(splitfield);
    settingslist=new QListView(splitfield);
    settingslayout=new QHBox(splitfield);
    QSize listviewsize(300,500);
    splitfield->setResizeMode(settingslist,QSplitter::Stretch);
    splitfield->setResizeMode(settingslayout,QSplitter::Stretch);
    settingslist->setSelectionMode(QListView::Single);
    settingslist->setRootIsDecorated(FALSE);
    settingslist->setResizeMode(QListView::AllColumns);
    settingslist->addColumn(i18n("Setting-Groups"),300);
    filesavingbutton=new QListViewItem(settingslist,i18n("Filesaving-settings"));
    ripperbutton=new QListViewItem(settingslist,i18n("CD-Ripper-settings"));
    directoriesbutton=new QListViewItem(settingslist,i18n("Diretory-settings"));
    databasebutton=new QListViewItem(settingslist,i18n("Database-settings"));
    internetbutton=new QListViewItem(settingslist,i18n("Internet-settings"));
    playerbutton=new QListViewItem(settingslist,i18n("Player-settings"));
    //set all pictures for the buttons
    picname="filesave.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    filesavingbutton->setPixmap(0,pixmap);
    picname="cdaudio_unmount.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    ripperbutton->setPixmap(0,pixmap);
    picname="folder.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    directoriesbutton->setPixmap(0,pixmap);
    picname="database.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    databasebutton->setPixmap(0,pixmap);
    picname="internet.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    internetbutton->setPixmap(0,pixmap);
    picname="player_play.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    playerbutton->setPixmap(0,pixmap);
    delete loader;
    //set the empty-settingspage on first start
    shownpage=new settingsempty(settingslayout);
    connect(settingslist,SIGNAL(selectionChanged(QListViewItem *)),SLOT(opensubmenu(QListViewItem *)));
    //settingslayout->adjustSize();
}

settingsmaindialog::~ settingsmaindialog()
{
    delete shownpage;
    delete filesavingbutton;
    delete ripperbutton;
    delete directoriesbutton;
    delete databasebutton;
    delete internetbutton;
    delete playerbutton;
    delete settingslayout;
    delete mainsettingslayout;
}

void settingsmaindialog::opensubmenu(QListViewItem *selectedmenu)
{
    delete shownpage;
    if(selectedmenu==filesavingbutton)
    {
        shownpage=new settingssaving(settingslayout);
    }
    else if(selectedmenu==ripperbutton)
    {
        shownpage=new settingsripper(settingslayout);
    }
    else if(selectedmenu==databasebutton)
    {
        shownpage=new settingsdatabase(settingslayout);
        connect(shownpage,SIGNAL(databasechangesignal()),this,SIGNAL(databasechangesignal()));
        connect(shownpage,SIGNAL(autocompletesignal(bool)),this,SIGNAL(autocompletesignal(bool)));
    }
    else if(selectedmenu==internetbutton)
    {
        shownpage=new settingsinternet(settingslayout);
    }
    else if(selectedmenu==directoriesbutton)
    {
        shownpage=new settingsdirectories(settingslayout);
        connect(shownpage,SIGNAL(directorychange()),SIGNAL(directorychange()));
    }
    else if(selectedmenu==playerbutton)
    {
        shownpage=new settingsplayer(settingslayout);
    }
    settingslayout->adjustSize();
}
//########################################################################################################################################

/*######################
 *# Emptysetting-class #
 *######################*/
settingsempty::settingsempty(QHBox *parent)
{
    infotext=new QLabel(i18n("Please select a sub-menu to change the settings for that category."),parent);
    infotext->show();
}

settingsempty::~settingsempty()
{
    delete infotext;
}
//########################################################################################################################################

/*##########################
 *# Directorysetting-class #
 *##########################*/
settingsdirectories::settingsdirectories(QHBox *parent)
{
    KIconLoader *loader=new KIconLoader();
    QPixmap pixmap;
    QString picname;
    pathgroupbox=new QGroupBox(2,Qt::Horizontal,i18n("Working directories"),parent);
    readingpathbutton=new QPushButton(i18n("Reading-path"),pathgroupbox);
    picname="folder.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    QIconSet buttoniconset(pixmap);
    readingpathbutton->setIconSet(buttoniconset);
    QToolTip::add
        (readingpathbutton,i18n("Set the directory, in wich the tagger looks for musicfiles."));
    readingpathledit=new QLineEdit(pathgroupbox);
    readingpathledit->setText(settingsfile.reading_dir);
    destinationpathbutton=new QPushButton(i18n("Destination-path"),pathgroupbox);
    QToolTip::add
        (destinationpathbutton,i18n("Set the directory, in wich cleanup moves the files, and ripper stores ready ripped files."));
    picname="folder_sound.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    buttoniconset=QIconSet(pixmap);
    destinationpathbutton->setIconSet(buttoniconset);
    destinationpathledit=new QLineEdit(pathgroupbox);
    destinationpathledit->setText(settingsfile.destination_dir);
    rejectpathbutton=new QPushButton(i18n("Reject-path"),pathgroupbox);
    QToolTip::add
        (rejectpathbutton,i18n("Set the directory, in wich cleanup moves not propper tagged files."));
    picname="folder_txt.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    buttoniconset=QIconSet(pixmap);
    rejectpathbutton->setIconSet(buttoniconset);
    rejectpathledit=new QLineEdit(pathgroupbox);
    rejectpathledit->setText(settingsfile.reject_dir);
    temppathbutton=new QPushButton(i18n("Temp-path"),pathgroupbox);
    QToolTip::add
        (temppathbutton,i18n("Set the directory, in wich ripper stores temporary files."));
    picname="folder_tar.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    buttoniconset=QIconSet(pixmap);
    temppathbutton->setIconSet(buttoniconset);
    temppathledit=new QLineEdit(pathgroupbox);
    temppathledit->setText(settingsfile.mp3_dir);
    pathgroupbox->show();
    connect(readingpathbutton,SIGNAL(clicked()),SLOT(getreadingpath()));
    connect(readingpathledit,SIGNAL(returnPressed()),SLOT(setreadingpath()));
    connect(destinationpathbutton,SIGNAL(clicked()),SLOT(getdestinationpath()));
    connect(destinationpathledit,SIGNAL(returnPressed()),SLOT(setdestinationpath()));
    connect(rejectpathbutton,SIGNAL(clicked()),SLOT(getrejectpath()));
    connect(rejectpathledit,SIGNAL(returnPressed()),SLOT(setrejectpath()));
    connect(temppathbutton,SIGNAL(clicked()),SLOT(gettemppath()));
    connect(temppathledit,SIGNAL(returnPressed()),SLOT(settemppath()));
    delete loader;
}

settingsdirectories::~settingsdirectories()
{
    delete readingpathbutton;
    delete readingpathledit;
    delete destinationpathbutton;
    delete destinationpathledit;
    delete rejectpathbutton;
    delete rejectpathledit;
    delete temppathbutton;
    delete temppathledit;
    delete pathgroupbox;
}

//###########		Memberslots		##############

void settingsdirectories::getreadingpath()
{
    QString path;
    if(path=QFileDialog::getExistingDirectory(settingsfile.reading_dir,this,i18n("choose directory"),i18n("Please choose reading directory")))
    {
        this->readingpathledit->setText(settingsfile.reading_dir);
    }
    if (path.compare("/"))
    {
        QFileInfo *dateiinfo=new QFileInfo(path);
        if (dateiinfo->exists())
        {
            settingsfile.reading_dir=path;
            this->readingpathledit->setText(settingsfile.reading_dir);
            emit directorychange();
        }
        delete dateiinfo;
    }
    else
    {
        //Throw out Error if user tries to use root-directory
        this->readingpathledit->setText(settingsfile.reading_dir);
        QMessageBox::warning(this,i18n("access error"),i18n("You can't use root-directory as reading-directory"),"&Ok",QString::null,QString::null,1);
    }
}

void settingsdirectories::setreadingpath()
{
    settingsfile.reading_dir=readingpathledit->text();
    emit directorychange();
}

void settingsdirectories::getdestinationpath()
{
    QString path;
    if(path=QFileDialog::getExistingDirectory(settingsfile.destination_dir,this,i18n("choose directory"),i18n("please choose destination-directory")))
    {
        this->destinationpathledit->setText(settingsfile.destination_dir);
    }
    if (path.compare("/"))
    {
        QFileInfo *dateiinfo=new QFileInfo(path);
        if (dateiinfo->exists())
        {
            settingsfile.destination_dir=path;
            this->destinationpathledit->setText(settingsfile.destination_dir);
        }
        delete dateiinfo;
    }
    else
    {
        //Throw out Error if user tries to use root-directory
        this->destinationpathledit->setText(settingsfile.destination_dir);
        QMessageBox::warning(this,i18n("access error"),i18n("You can't copy your files into root-directory"),"&Ok",QString::null,QString::null,1);
    }
}

void settingsdirectories::setdestinationpath()
{
    settingsfile.destination_dir=destinationpathledit->text();
}

void settingsdirectories::getrejectpath()
{
    QString path;
    if(path=QFileDialog::getExistingDirectory(settingsfile.reject_dir,this,i18n("choose directory"),i18n("please choose reject-directory")))
    {
        this->rejectpathledit->setText(settingsfile.reject_dir);
    }
    if (path.compare("/"))
    {
        QFileInfo *dateiinfo=new QFileInfo(path);
        if (dateiinfo->exists())
        {
            settingsfile.reject_dir=path;
            this->rejectpathledit->setText(settingsfile.reject_dir);
        }
        delete dateiinfo;
    }
    else
    {
        //Throw out Error if user tries to use root-directory
        this->rejectpathledit->setText(settingsfile.reject_dir);
        QMessageBox::warning(this,i18n("access error"),i18n("You can't move your reject-files to root-directory"),"&Ok",QString::null,QString::null,1);
    }
}

void settingsdirectories::setrejectpath()
{
    settingsfile.reject_dir=rejectpathledit->text();
}

void settingsdirectories::gettemppath()
{
    QString path;
    if(path=QFileDialog::getExistingDirectory(settingsfile.mp3_dir,this,i18n("choose directory"),i18n("please choose temp-directory")))
    {
        this->temppathledit->setText(settingsfile.mp3_dir);
    }
    if (path.compare("/"))
    {
        QFileInfo *dateiinfo=new QFileInfo(path);
        if (dateiinfo->exists())
        {
            settingsfile.mp3_dir=path;
            this->temppathledit->setText(settingsfile.mp3_dir);
        }
        delete dateiinfo;
    }
    else
    {
        //Throw out Error if user tries to use root-directory
        this->temppathledit->setText(settingsfile.mp3_dir);
        QMessageBox::warning(this,i18n("access error"),i18n("You can't use root-directory as temp"),"&Ok",QString::null,QString::null,1);
    }
}

void settingsdirectories::settemppath()
{
    settingsfile.mp3_dir=temppathledit->text();
}
//########################################################################################################################################


/*#########################
 *# Databasesetting-class #
 *#########################*/
settingsdatabase::settingsdatabase(QHBox *parent)
{
    databasegroupbox=new QGroupBox(1,Qt::Horizontal,i18n("Database Settings"),parent);
    databasetypecombo=new QComboBox(FALSE,databasegroupbox);
    databasetypecombo->insertItem("None");
    databasetypecombo->insertItem("MySQL");
    databasetypecombo->insertItem("PostgresQL");
    databasetypecombo->insertItem("SQL Lite");
    databasetypecombo->setCurrentItem(settingsfile.databasetype);
    autocompletebox=new QCheckBox(i18n("Autocomplete boxes with databasevalues"),databasegroupbox);
    autocompletebox->setChecked(settingsfile.autocomplete);
    editdatabasebutton=new QPushButton(i18n("Edit database"),databasegroupbox);
    KIconLoader *loader=new KIconLoader();
    QPixmap pixmap;
    QString picname;
    picname="edit.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    QIconSet buttonset(pixmap);
    editdatabasebutton->setIconSet(buttonset);
    delete loader;
    databasegroupbox->show();
    connect(databasetypecombo,SIGNAL(activated(int)),SLOT(databasechange(int)));
    connect(autocompletebox,SIGNAL(clicked()),SLOT(autocompleteslot()));
    connect(editdatabasebutton,SIGNAL(clicked()),SLOT(editdatabaseslot()));
}

settingsdatabase::~settingsdatabase()
{
    delete databasetypecombo;
    delete autocompletebox;
    delete editdatabasebutton;
    delete databasegroupbox;
}
//###########		Memberslots		##############

void settingsdatabase::autocompleteslot()
{
    settingsfile.autocomplete=autocompletebox->isChecked();
    emit autocompletesignal(settingsfile.autocomplete);
}

void settingsdatabase::databasechange(int newdb)
{
    if (newdb==2||newdb==3)
    {
        //not supported yet,set to no database instead
        databasetypecombo->setCurrentItem(0);
        newdb=0;
        QMessageBox::information(this,i18n("Databasesupport"),i18n("Only MySQL-Support is implemented till now, sorry!"));
    }
    settingsfile.databasetype=newdb;
    if (settingsfile.databasetype)
    {
        database=new databasewrapper();//initialise database and create if needed
        if(database->quit)
        {
            settingsfile.databasetype=0;
            databasetypecombo->setCurrentItem(0);
            delete database;
            database=NULL;
        }
    }
    else
    {
        delete database;
        database=NULL;
    }
    emit databasechangesignal();
    autocompletebox->setEnabled(settingsfile.databasetype);
}

void settingsdatabase::editdatabaseslot()
{
    QMessageBox::information(this,i18n("Database-Editor"),i18n("Not included yet, sorry!"));
}
//########################################################################################################################################

/*#######################
 *# Rippersetting-class #
 *#######################*/
settingsripper::settingsripper(QHBox *parent)
{
    rippergroupbox=new QGroupBox(1,Qt::Horizontal,i18n("CD-Ripping Settings"),parent);
    encodergroupbox=new QGroupBox(1,Qt::Horizontal,i18n("Used Encoderengine"),rippergroupbox);
    encodertypecombo=new QComboBox(FALSE,encodergroupbox);
    encodertypecombo->insertItem("Lame");
    encodertypecombo->insertItem("Gogo");
    encodertypecombo->insertItem("Bladeenc");
    encodertypecombo->insertItem("Oggenc");
    encodertypecombo->setCurrentItem(settingsfile.encodingengine);
    encoderpathgroupbox=new QGroupBox(2,Qt::Horizontal,i18n("path to executable"),encodergroupbox);
    encoderpathbutton=new QPushButton(i18n("Find executable"),encoderpathgroupbox);
    encoderpathledit=new QLineEdit(encoderpathgroupbox);
    bitrategroupbox=new QButtonGroup(1,Qt::Vertical,i18n("Bitrate"),rippergroupbox);
    bitrate128=new QRadioButton("128kBit",bitrategroupbox);
    bitrate160=new QRadioButton("160kBit",bitrategroupbox);
    bitrate192=new QRadioButton("192kBit",bitrategroupbox);
    bitrate256=new QRadioButton("256kBit",bitrategroupbox);
    bitrate320=new QRadioButton("320kBit",bitrategroupbox);
    bitratevbr=new QCheckBox("Vbr",bitrategroupbox);
    commandlinegroupbox=new QGroupBox(1,Qt::Vertical,i18n("Commandlineoptions"),rippergroupbox);
    commandlineledit=new QLineEdit(commandlinegroupbox);
    //use encoderchange to set the right bitrate stuff
    encoderchange(settingsfile.encodingengine);
    KIconLoader *loader=new KIconLoader();
    QPixmap pixmap;
    QString picname;
    picname="exec.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    QIconSet buttonset(pixmap);
    encoderpathbutton->setIconSet(buttonset);
    delete loader;
    rippergroupbox->show();
    connect(bitrategroupbox,SIGNAL(clicked(int)),this,SLOT(bitrateslot(int)));
    connect(encodertypecombo,SIGNAL(activated(int)),this,SLOT(encoderchange(int)));
    connect(encoderpathbutton,SIGNAL(clicked()),this,SLOT(executableselect()));
    connect(encoderpathledit,SIGNAL(textChanged(const QString &)),this,SLOT(executableleditslot(const QString &)));
    connect(commandlineledit,SIGNAL(textChanged(const QString &)),this,SLOT(commandlineleditslot(const QString &)));
}

settingsripper::~settingsripper()
{
    delete commandlineledit;
    delete commandlinegroupbox;
    delete bitrate128;
    delete bitrate160;
    delete bitrate192;
    delete bitrate256;
    delete bitrate320;
    delete bitratevbr;
    delete bitrategroupbox;
    delete encoderpathledit;
    delete encoderpathbutton;
    delete encoderpathgroupbox;
    delete encodertypecombo;
    delete encodergroupbox;
    delete rippergroupbox;
}
//###########		Memberslots		##############

void settingsripper::bitrateslot(int bitrate)
{
    //lame gogo behave the same way
    if (bitrate==5&&(encodertypecombo->currentItem()==0||encodertypecombo->currentItem()==1||encodertypecombo->currentItem()==2))
    {
        //deselect all bitratecheckboxes for lame and gogo if vbr is used(bladeenc does not know vbr)
        if (bitratevbr->isChecked())
        {
            bitrate128->setChecked(FALSE);
            bitrate160->setChecked(FALSE);
            bitrate192->setChecked(FALSE);
            bitrate256->setChecked(FALSE);
            bitrate320->setChecked(FALSE);
        }
        else
        {
            //set defaultbitrate if vbr is deselected
            bitrate128->setChecked(TRUE);
        }
    }
    else if (bitrate==5&&encodertypecombo->currentItem()==3)
    {
        if (!bitratevbr->isChecked()&&!bitrate128->isChecked()&&!bitrate160->isChecked()&&!bitrate192->isChecked()&&!bitrate256->isChecked()&&!bitrate320->isChecked())
        {
            //Check if any bitrate is selected, and select default if none is selected
            bitrate128->setChecked(TRUE);
        }
    }
    if (bitrate!=5&&(encodertypecombo->currentItem()==0||encodertypecombo->currentItem()==1||encodertypecombo->currentItem()==2))
        bitratevbr->setChecked(FALSE);
    //Check wich bitrate is selected, and set it for the right encoding-engine
    long newbitrate=0;
    bool vbr=FALSE;
    if (bitrate128->isChecked())
        newbitrate=128;
    if (bitrate160->isChecked())
        newbitrate=160;
    if (bitrate192->isChecked())
        newbitrate=192;
    if (bitrate256->isChecked())
        newbitrate=256;
    if (bitrate320->isChecked())
        newbitrate=320;
    if (bitratevbr->isChecked())
    {
        if (!encodertypecombo->currentItem()==3)
            newbitrate=0;
        vbr=TRUE;
    }
    if  (encodertypecombo->currentItem()==0)
        settingsfile.lamebitrate=newbitrate;
    if  (encodertypecombo->currentItem()==1)
        settingsfile.gogobitrate=newbitrate;
    if  (encodertypecombo->currentItem()==2)
        settingsfile.bladeencbitrate=newbitrate;
    if  (encodertypecombo->currentItem()==3)
    {
        settingsfile.oggencbitrate=newbitrate;
        settingsfile.oggencvbr=vbr;
    }
}

void settingsripper::encoderchange(int encodertype)
{
    settingsfile.encodingengine=encodertype;
    bitratevbr->setEnabled(TRUE);
    //deselect all, so we see what is in our settings for real
    bitratevbr->setChecked(FALSE);
    bitrate128->setChecked(FALSE);
    bitrate160->setChecked(FALSE);
    bitrate192->setChecked(FALSE);
    bitrate256->setChecked(FALSE);
    bitrate320->setChecked(FALSE);
    switch (encodertype)
    {
    case 0:
        {
            //lame
            encoderpathledit->setText(settingsfile.lameexe);
            commandlineledit->setText(settingsfile.lameoptions);
            if (settingsfile.lamebitrate==0)
                bitratevbr->setChecked(TRUE);
            else if(settingsfile.lamebitrate==128)
                bitrate128->setChecked(TRUE);
            else if(settingsfile.lamebitrate==160)
                bitrate160->setChecked(TRUE);
            else if(settingsfile.lamebitrate==192)
                bitrate192->setChecked(TRUE);
            else if(settingsfile.lamebitrate==256)
                bitrate256->setChecked(TRUE);
            else if(settingsfile.lamebitrate==320)
                bitrate320->setChecked(TRUE);
            break;
        }
    case 1:
        {
            //gogo
            encoderpathledit->setText(settingsfile.gogoexe);
            commandlineledit->setText(settingsfile.gogooptions);
            if (settingsfile.gogobitrate==0)
                bitratevbr->setChecked(TRUE);
            else if(settingsfile.gogobitrate==128)
                bitrate128->setChecked(TRUE);
            else if(settingsfile.gogobitrate==160)
                bitrate160->setChecked(TRUE);
            else if(settingsfile.gogobitrate==192)
                bitrate192->setChecked(TRUE);
            else if(settingsfile.gogobitrate==256)
                bitrate256->setChecked(TRUE);
            else if(settingsfile.gogobitrate==320)
                bitrate320->setChecked(TRUE);
            break;
        }
    case 2:
        {
            //bladeenc
            encoderpathledit->setText(settingsfile.bladeencexe);
            commandlineledit->setText(settingsfile.bladeencoptions);
            bitratevbr->setEnabled(FALSE);
            if(settingsfile.bladeencbitrate==128)
                bitrate128->setChecked(TRUE);
            else if(settingsfile.bladeencbitrate==160)
                bitrate160->setChecked(TRUE);
            else if(settingsfile.bladeencbitrate==192)
                bitrate192->setChecked(TRUE);
            else if(settingsfile.bladeencbitrate==256)
                bitrate256->setChecked(TRUE);
            else if(settingsfile.bladeencbitrate==320)
                bitrate320->setChecked(TRUE);
            break;
        }
    case 3:
        {
            //oggenc
            encoderpathledit->setText(settingsfile.oggencexe);
            commandlineledit->setText(settingsfile.oggencoptions);
            if (settingsfile.oggencvbr)
                bitratevbr->setChecked(TRUE);
            if(settingsfile.oggencbitrate==128)
                bitrate128->setChecked(TRUE);
            else if(settingsfile.oggencbitrate==160)
                bitrate160->setChecked(TRUE);
            else if(settingsfile.oggencbitrate==192)
                bitrate192->setChecked(TRUE);
            else if(settingsfile.oggencbitrate==256)
                bitrate256->setChecked(TRUE);
            else if(settingsfile.oggencbitrate==320)
                bitrate320->setChecked(TRUE);
            break;
        }
    }
}

void settingsripper::executableleditslot(const QString &newpath)
{
    switch (settingsfile.encodingengine)
    {
    case 0:
        {
            //lame
            settingsfile.lameexe=newpath;
            break;
        }
    case 1:
        {
            //gogo
            settingsfile.gogoexe=newpath;
            break;
        }
    case 2:
        {
            //bladeenc
            settingsfile.bladeencexe=newpath;
            break;
        }
    case 3:
        {
            //oggenc
            settingsfile.oggencexe=newpath;
            break;
        }
    }
}

void settingsripper::executableselect()
{
    QString exename;
    QString exepathold;
    switch (settingsfile.encodingengine)
    {
    case 0:
        {
            //lame
            exename=QString("lame");
            exepathold=settingsfile.lameexe;
            break;
        }
    case 1:
        {
            //gogo
            exename=QString("gogo");
            exepathold=settingsfile.gogoexe;
            break;
        }
    case 2:
        {
            //bladeenc
            exename=QString("bladeenc");
            exepathold=settingsfile.bladeencexe;
            break;
        }
    case 3:
        {
            //oggenc
            exename=QString("oggenc");
            exepathold=settingsfile.oggencexe;
            break;
        }
    }
    QString infotext=QString(i18n("Please choose %1-executable")).arg(exename);
    QString exepathnew;
    if(exepathnew=QFileDialog::getOpenFileName(exepathold,QString::null,this,i18n("Choose executable"),infotext))
    {
        //Check if we have the right executable or not
        QString newexe=exepathnew.right(exepathnew.length()-(exepathnew.findRev('/')+1));
        if (newexe.lower().compare(exename))
        {
            QString errortext=QString(i18n("This file does not look like the right %1-executable!")).arg(exename);
            QMessageBox::information(this,i18n("Wrong executable"),errortext);
        }
        else
        {
            //looks like it is the right file, set it for the new executable
            encoderpathledit->setText(exepathnew);
            switch (settingsfile.encodingengine)
            {
            case 0:
                {
                    //lame
                    settingsfile.lameexe=exepathnew;
                    break;
                }
            case 1:
                {
                    //gogo
                    settingsfile.gogoexe=exepathnew;
                    break;
                }
            case 2:
                {
                    //bladeenc
                    settingsfile.bladeencexe=exepathnew;
                    break;
                }
            case 3:
                {
                    //oggenc
                    settingsfile.oggencexe=exepathnew;
                    break;
                }
            }
        }
    }
}

void settingsripper::commandlineleditslot(const QString &newoptions)
{
    switch (settingsfile.encodingengine)
    {
    case 0:
        {
            //lame
            settingsfile.lameoptions=newoptions;
            break;
        }
    case 1:
        {
            //gogo
            settingsfile.gogooptions=newoptions;
            break;
        }
    case 2:
        {
            //bladeenc
            settingsfile.bladeencoptions=newoptions;
            break;
        }
    case 3:
        {
            //oggenc
            settingsfile.oggencoptions=newoptions;
            break;
        }
    }
}//########################################################################################################################################

/*#####################
 *#Playersetting-Class#
 *#####################*/
settingsplayer::settingsplayer(QHBox *parent)
{
    playergroupbox=new QGroupBox(1,Qt::Horizontal,i18n("Player-settings"),parent);
    playerbox=new QGroupBox(1,Qt::Horizontal,i18n("Player to use"),playergroupbox);
    playertypecombo=new QComboBox(FALSE,playerbox);
    playertypecombo->insertItem("MPlayer");
    playertypecombo->insertItem("None");
    playertypecombo->setCurrentItem(settingsfile.playertype);
    executablebox=new QGroupBox(1,Qt::Vertical,i18n("Select executable"),playerbox);
    playerpathbutton=new QPushButton(i18n("Find executable"),executablebox);
    playerpathledit=new QLineEdit(executablebox);
    playertypecombo->setCurrentItem(settingsfile.playertype);
    switch (settingsfile.playertype)
    {
    case 0:
        {
            playerpathledit->setText(settingsfile.mplayerexe);
            playerpathledit->setEnabled(TRUE);
            playerpathbutton->setEnabled(TRUE);
            break;
        }
    default:
        {
            playerpathledit->setText("");
            playerpathledit->setEnabled(FALSE);
            playerpathbutton->setEnabled(FALSE);
            break;
        }
    }
    KIconLoader *loader=new KIconLoader();
    QPixmap pixmap;
    QString picname;
    picname="exec.png";
    pixmap=loader->loadIcon(picname,KIcon::Desktop);
    QIconSet buttonset(pixmap);
    playerpathbutton->setIconSet(buttonset);
    delete loader;
    playergroupbox->show();
    connect(playertypecombo,SIGNAL(activated(int)),this,SLOT(playerchange(int)));
    connect(playerpathbutton,SIGNAL(clicked()),this,SLOT(executableselect()));
    connect(playerpathledit,SIGNAL(textChanged(const QString &)),this,SLOT(executableleditslot(const QString &)));
    //connect(commandlineledit,SIGNAL(textChanged(const QString &)),this,SLOT(commandlineleditslot(const QString &)));
}

settingsplayer::~settingsplayer()
{
    delete playertypecombo;
    delete playerpathbutton;
    delete playerpathledit;
    delete executablebox;
    delete playerbox;
    delete playergroupbox;
}

void settingsplayer::playerchange(int newplayer)
{
    int result=0;
    if (playerlink->isplaying)
        result=QMessageBox::warning(this,i18n("Playerchange"),i18n("This will stop the player and delete the playlist. Are you sure you want to do this"),QMessageBox::Ok,QMessageBox::Cancel);
    //Every playerchange needs the destruction of playerlink!
    if (result==1||result==0)
    {
        delete playerlink;
        switch (newplayer)
        {
        case 0:
            {
                settingsfile.playertype=0;
                playerpathledit->setText(settingsfile.mplayerexe);
                playerpathledit->setEnabled(TRUE);
                playerpathbutton->setEnabled(TRUE);
                break;
            }
        default:
            {
                settingsfile.playertype=1;
                playerpathledit->setText("");
                playerpathledit->setEnabled(FALSE);
                playerpathbutton->setEnabled(FALSE);
                break;
            }
        }
        playerlink=new playerinterface;
    }
    else
    {
        playertypecombo->setCurrentItem(settingsfile.playertype);
    }
}

void settingsplayer::executableselect()
{
    QString exename;
    QString exepathold;
    switch (settingsfile.playertype)
    {
    case 0:
        {
            //mplayer
            exename=QString("mplayer");
            exepathold=settingsfile.mplayerexe;
            break;
        }
    }
    QString infotext=QString(i18n("Please choose %1-executable")).arg(exename);
    QString exepathnew;
    if(exepathnew=QFileDialog::getOpenFileName(exepathold,QString::null,this,i18n("Choose executable"),infotext))
    {
        //Check if we have the right executable or not
        QString newexe=exepathnew.right(exepathnew.length()-(exepathnew.findRev('/')+1));
        if (newexe.lower().compare(exename))
        {
            QString errortext=QString(i18n("This file does not look like the right %1-executable!")).arg(exename);
            QMessageBox::information(this,i18n("Wrong executable"),errortext);
        }
        else
        {
            //looks like it is the right file, set it for the new executable
            playerpathledit->setText(exepathnew);
            switch (settingsfile.playertype)
            {
            case 0:
                {
                    //mplayer
                    settingsfile.mplayerexe=exepathnew;
                    break;
                }
            }
        }
    }
}

void settingsplayer::executableleditslot(const QString &newexe)
{
    switch (settingsfile.playertype)
    {
    case 0:
        {
            //mplayer
            settingsfile.mplayerexe=newexe;
            break;
        }
    }
}
//########################################################################################################################################


/*#########################
 *# Internetsetting-class #
 *#########################*/
settingsinternet::settingsinternet(QHBox *parent)
{
    freedbc=NULL;
    progresswindow=NULL;
    progress=NULL;
    internetgroupbox=new QGroupBox(1,Qt::Horizontal,i18n("Internet-settings"),parent);
    freedbgroupbox=new QGroupBox(2,Qt::Horizontal,i18n("Freedb-support for Cd-Ripper"),internetgroupbox);
    freedbcheckbox=new QCheckBox(i18n("Use freedb for discrecognising"),freedbgroupbox);
    freedbcheckbox->setChecked(settingsfile.freedbsupport);
    freedbgroupbox->addSpace(0);
    getserverlistbutton=new QPushButton(i18n("Select Server"),freedbgroupbox);
    freedbserverledit=new QLineEdit(freedbgroupbox);
    freedbserverledit->setText(settingsfile.freedbservername);
    loginlabel=new QLabel(i18n("Freedb Loginname"),freedbgroupbox);
    freedbloginledit=new QLineEdit(freedbgroupbox);
    freedbloginledit->setText(settingsfile.freedbloginname);
    musicbrainzgroupbox=new QGroupBox(1,Qt::Horizontal,i18n("Musicbrainz-support for tagguessing"),internetgroupbox);
    musicbrainzcheckbox=new QCheckBox(i18n("Use musicbrainz to fill tags not known"),musicbrainzgroupbox);
    musicbrainzcheckbox->setChecked(settingsfile.musicbrainzsupport);
    internetgroupbox->show();
    connect(getserverlistbutton,SIGNAL(clicked()),SLOT(serverlistslot()));
    connect(freedbloginledit,SIGNAL(textChanged(const QString &)),SLOT(freedbloginslot(const QString &)));
    connect(freedbserverledit,SIGNAL(textChanged(const QString &)),SLOT(freedbserverslot(const QString &)));
    connect(freedbcheckbox,SIGNAL(clicked()),SLOT(freedbcheckboxslot()));
    connect(musicbrainzcheckbox,SIGNAL(clicked()),SLOT(musicbrainzcheckboxslot()));
}

settingsinternet::~settingsinternet()
{
    delete musicbrainzcheckbox;
    delete musicbrainzgroupbox;
    delete freedbserverledit;
    delete getserverlistbutton;
    delete loginlabel;
    delete freedbloginledit;
    delete freedbcheckbox;
    delete freedbgroupbox;
    delete internetgroupbox;
}
//###########		Memberslots		##############

void settingsinternet::serverlistslot()
{
    disconnect(freedbserverledit,SIGNAL(textChanged(const QString &)),this,SLOT(freedbserverslot(const QString &)));
    progresswindow=new progressmainwindow(FALSE);
    progress=progresswindow->startInfoOnly(i18n("Fetching FreeDB-serverlist"));
    progresswindow->show();
    freedbc=new freedbconnect();
    freedbc->getservers();
    connect(freedbc,SIGNAL(selected(QString)),this,SLOT(serverselected(QString)));
    connect(freedbc,SIGNAL(cancelled()),this,SLOT(servercancelled()));
    connect(freedbc,SIGNAL(error(QString)),this,SLOT(errorreceived(QString)));
}

void settingsinternet::serverselected(QString server)
{
    freedbserverledit->setText(server);
    settingsfile.freedbservername=server;
    if (freedbc)
    {
        delete freedbc;
        freedbc=NULL;
    }
    if (progress)
    {
        delete progress;
        progress=NULL;
    }
    if (progresswindow)
    {
        delete progresswindow;
        progresswindow=NULL;
    }
    connect(freedbserverledit,SIGNAL(textChanged(const QString &)),SLOT(freedbserverslot(const QString &)));
}

void settingsinternet::servercancelled()
{
    if (settingsfile.freedbservername.isEmpty())
    {
        freedbserverledit->setText("freedb.freedb.org");
        settingsfile.freedbservername=QString("freedb.freedb.org");
    }
    if (freedbc)
    {
        delete freedbc;
        freedbc=NULL;
    }
    if (progress)
    {
        delete progress;
        progress=NULL;
    }
    if (progresswindow)
    {
        delete progresswindow;
        progresswindow=NULL;
    }
    connect(freedbserverledit,SIGNAL(textChanged(const QString &)),SLOT(freedbserverslot(const QString &)));
}

void settingsinternet::errorreceived(QString errortext)
{
    QMessageBox::information(this,i18n("Freedb-Problem"),errortext);
    if (freedbc)
    {
        delete freedbc;
        freedbc=NULL;
    }
    if (progress)
    {
        delete progress;
        progress=NULL;
    }
    if (progresswindow)
    {
        delete progresswindow;
        progresswindow=NULL;
    }
}

void settingsinternet::freedbloginslot(const QString &newtext)
{
    settingsfile.freedbloginname=newtext;
}

void settingsinternet::freedbserverslot(const QString &newtext)
{
    settingsfile.freedbservername=newtext;
}

void settingsinternet::freedbcheckboxslot()
{
    settingsfile.freedbsupport=freedbcheckbox->isChecked();
}

void settingsinternet::musicbrainzcheckboxslot()
{
    settingsfile.musicbrainzsupport=musicbrainzcheckbox->isChecked();
}

//########################################################################################################################################

/*#####################
 *# Filesave-settings #
 *#####################*/
settingssaving::settingssaving(QHBox *parent)
{
    filesavinggroupbox=new QGroupBox(1,Qt::Horizontal,i18n("Filesaving Settings"),parent);
    tagtypegroupbox=new QButtonGroup(1,Qt::Vertical,i18n("Save witch Tagtype"),filesavinggroupbox);
    tagtypegroupbox->setExclusive(FALSE);
    v1tagbox=new QCheckBox("Id3v1",tagtypegroupbox);
    v23tagbox=new QCheckBox("Id3v2.3 (Id3lib)",tagtypegroupbox);
    v24tagbox=new QCheckBox("Id3v2.4 (taglib)",tagtypegroupbox);
    if (settingsfile.tagversion&0x01||settingsfile.tagversion&0x04)
        v1tagbox->setChecked(TRUE);
    if (settingsfile.tagversion&0x02)
        v23tagbox->setChecked(TRUE);
    if (settingsfile.tagversion&0x08)
        v24tagbox->setChecked(TRUE);
    filtergroupbox=new QGroupBox(1,Qt::Horizontal,i18n("Filenamefilter for Cleanup and Ripper"),filesavinggroupbox);
    filterledit=new QLineEdit(filtergroupbox);
    filterledit->setText(settingsfile.filter);
    filterbuttonsgroupbox=new QGroupBox(4,Qt::Horizontal,i18n("Mnemnomics"),filtergroupbox);
    directorybutton=new QPushButton(i18n("Directory"),filterbuttonsgroupbox);
    yearbutton=new QPushButton(i18n("Year"),filterbuttonsgroupbox);
    artistbutton=new QPushButton(i18n("Artist"),filterbuttonsgroupbox);
    songbutton=new QPushButton(i18n("Song"),filterbuttonsgroupbox);
    trackbutton=new QPushButton(i18n("Track"),filterbuttonsgroupbox);
    albumbutton=new QPushButton(i18n("Album"),filterbuttonsgroupbox);
    genrebutton=new QPushButton(i18n("Genre"),filterbuttonsgroupbox);
    picturegroupbox=new QGroupBox(1,Qt::Vertical,i18n("Pictures in directories"),filesavinggroupbox);
    picturecheckbox=new QCheckBox(i18n("Save Picture"),picturegroupbox);
    picturecheckbox->setChecked(settingsfile.savepicture);
    pictureledit=new QLineEdit(picturegroupbox);
    pictureledit->setText(settingsfile.picturename);
    pictureformatcombo=new QComboBox(FALSE,picturegroupbox);
    pictureformatcombo->insertItem("jpg");
    pictureformatcombo->insertItem("gif");
    pictureformatcombo->insertItem("bmp");
    pictureformatcombo->insertItem("png");
    pictureformatcombo->setCurrentText(settingsfile.pictureformat);
    picturesizebox=new QSpinBox(25,150,5,picturegroupbox,i18n("Size"));
    picturesizebox->setValue(settingsfile.picturesize);
    overwritegroupbox=new QGroupBox(2,Qt::Horizontal,i18n("Delete Settings"),filesavinggroupbox);
    overwritecheckbox=new QCheckBox(i18n("Overwrite existing files"),overwritegroupbox);
    bettercheckbox=new QCheckBox(i18n("If bitrate is higher"),overwritegroupbox);
    deletecheckbox=new QCheckBox(i18n("Move files"),overwritegroupbox);
    deletecheckbox->setChecked(settingsfile.remove);
    overwritecheckbox->setChecked(settingsfile.overwrite);
    bettercheckbox->setChecked(settingsfile.better);
    bettercheckbox->setEnabled(settingsfile.overwrite);
    verbosegroupbox=new QGroupBox(1,Qt::Horizontal,(i18n("Verbosity")),filesavinggroupbox);
    verbosecheckbox=new QCheckBox(i18n("Only pop up status on error"),verbosegroupbox);
    verbosecheckbox->setChecked(!settingsfile.verbose);
    filesavinggroupbox->show();
    //filter
    connect(filterledit,SIGNAL(textChanged(const QString &)),SLOT(savefilter(const QString &)));
    connect(directorybutton,SIGNAL(clicked()),SLOT(dirslashslot()));
    connect(yearbutton,SIGNAL(clicked()),SLOT(yearslot()));
    connect(artistbutton,SIGNAL(clicked()),SLOT(artistslot()));
    connect(songbutton,SIGNAL(clicked()),SLOT(songslot()));
    connect(trackbutton,SIGNAL(clicked()),SLOT(trackslot()));
    connect(albumbutton,SIGNAL(clicked()),SLOT(albumslot()));
    connect(genrebutton,SIGNAL(clicked()),SLOT(genreslot()));
    //tagtype
    connect(tagtypegroupbox,SIGNAL(clicked(int)),SLOT(tagtypeslot(int)));
    //picturesave
    connect(picturecheckbox,SIGNAL(clicked()),SLOT(coversavecheckboxslot()));
    connect(pictureledit,SIGNAL(textChanged(const QString &)),SLOT(covernamechanged(const QString &)));
    connect(picturesizebox,SIGNAL(valueChanged(int)),SLOT(coversizechanged(int)));
    connect(pictureformatcombo,SIGNAL(highlighted(const QString &)),SLOT(coverformatchanged(const QString &)));
    //overwrite
    connect(overwritecheckbox,SIGNAL(toggled(bool)),SLOT(overwriteslot()));
    connect(bettercheckbox,SIGNAL(toggled(bool)),SLOT(betterslot()));
    connect(deletecheckbox,SIGNAL(toggled(bool)),SLOT(deleteslot()));
    connect(verbosecheckbox,SIGNAL(toggled(bool)),SLOT(verboseslot()));
}

settingssaving::~settingssaving()
{
    delete verbosecheckbox;
    delete deletecheckbox;
    delete bettercheckbox;
    delete overwritecheckbox;
    delete overwritegroupbox;
    delete pictureformatcombo;
    delete pictureledit;
    delete picturecheckbox;
    delete picturegroupbox;
    delete directorybutton;
    delete yearbutton;
    delete artistbutton;
    delete songbutton;
    delete trackbutton;
    delete albumbutton;
    delete genrebutton;
    delete filterledit;
    delete filterbuttonsgroupbox;
    delete filtergroupbox;
    delete v1tagbox;
    delete v23tagbox;
    delete v24tagbox;
    delete tagtypegroupbox;
    delete verbosegroupbox;
    delete filesavinggroupbox;
}

//###########		Memberslots		##############

void settingssaving::albumslot()
{
    QString text=filterledit->text();
    if (!text.contains("#ALBUM#"))
    {
        text=QString("%1#ALBUM#").arg(text);
        filterledit->setText(text);
    }
}

void settingssaving::artistslot()
{
    QString text=filterledit->text();
    if (!text.contains("#BAND#"))
    {
        text=QString("%1#BAND#").arg(text);
        filterledit->setText(text);
    }
}

void settingssaving::dirslashslot()
{
    QString text=filterledit->text();
    if ((!text.endsWith("/"))&&(!text.isEmpty()))
    {
        text=QString("%1/").arg(text);
        filterledit->setText(text);
    }
}

void settingssaving::genreslot()
{
    QString text=filterledit->text();
    if (!text.contains("#GENRE#"))
    {
        text=QString("%1#GENRE#").arg(text);
        filterledit->setText(text);
    }
}

void settingssaving::songslot()
{
    QString text=filterledit->text();
    if (!text.contains("#SONG#"))
    {
        text=QString("%1#SONG#").arg(text);
        filterledit->setText(text);
    }
}

void settingssaving::trackslot()
{
    QString text=filterledit->text();
    if (!text.contains("#TRACK#"))
    {
        text=QString("%1#TRACK#").arg(text);
        filterledit->setText(text);
    }
}

void settingssaving::yearslot()
{
    QString text=filterledit->text();
    if (!text.contains("#YEAR#"))
    {
        text=QString("%1#YEAR#").arg(text);
        filterledit->setText(text);
    }
}

void settingssaving::savefilter(const QString & string)
{
    settingsfile.filter=string;
}

void settingssaving::tagtypeslot(int typebox)
{
    //Check wich Checkbox was pressed, and take care, that always one tagtype is selected
    if (typebox==1&&v23tagbox->isChecked())
    {
        //Make sure that only tagtype v2.3 is selected
        v24tagbox->setChecked(FALSE);
    }
    else if (typebox==2&&v24tagbox->isChecked())
    {
        //Make sure that only tagtype v2.3 is selected
        v23tagbox->setChecked(FALSE);
    }
    if (tagtypegroupbox->selectedId()==-1&&typebox!=0)
    {
        //if nothing is selected set Checkbox for ID3v1
        v1tagbox->setChecked(TRUE);
    }
    else if(tagtypegroupbox->selectedId()==-1&&typebox==0&&!v24tagbox->isChecked())
    {
        //if nothing is selected set Checkbox for ID3v2.3 as default, when user deactivates ID3v1 as last
        v23tagbox->setChecked(TRUE);
    }
    //Construct new tagversion-number, out of selection, and set it in settingsfile
    int newtagtype=0;
    if (v1tagbox->isChecked())
    {
        if (v23tagbox->isChecked())
            newtagtype=3;
        else if (v24tagbox->isChecked())
            newtagtype=12;
        else
            newtagtype=1;
    }
    else if (v23tagbox->isChecked())
        newtagtype=2;
    else if (v24tagbox->isChecked())
        newtagtype=8;
    settingsfile.tagversion=newtagtype;
}

void settingssaving::coversavecheckboxslot()
{
    settingsfile.savepicture=picturecheckbox->isChecked();
}

void settingssaving::covernamechanged(const QString &text)
{
    if (!text.isEmpty())
        settingsfile.picturename=text;
}

void settingssaving::coversizechanged(int size)
{
    settingsfile.picturesize=size;
}

void settingssaving::coverformatchanged(const QString &text)
{
    settingsfile.pictureformat=text;
}

void settingssaving::deleteslot()
{
    settingsfile.remove=deletecheckbox->isChecked();
}

void settingssaving::overwriteslot()
{
    bettercheckbox->setEnabled(overwritecheckbox->isChecked());
    settingsfile.overwrite=overwritecheckbox->isChecked();
}

void settingssaving::betterslot()
{
    settingsfile.better=bettercheckbox->isChecked();
}

void settingssaving::verboseslot()
{
    settingsfile.verbose=!verbosecheckbox->isChecked();
}
