
#include <qcombobox.h>
#include <qlistbox.h>
#include <qtoolbutton.h>
#include <qaction.h>
#include <qfiledialog.h>

#include <rpackagelister.h>
#include <rconfiguration.h>

#include <rqmainwindow.h>
#include <rqcacheprogress.h>
#include <rqobservers.h>
#include <rqchangeswindow.h>
#include <rqsummarywindow.h>
#include <rqfetchwindow.h>
#include <rqinstallwindow.h>
#include <rqpixmaps.h>
#include <rqfilterswindow.h>
#include <rqpackageitem.h>
#include <rqfindwindow.h>

#include <apt-pkg/configuration.h>

#include <fstream>

#include <config.h>

RQMainWindow::RQMainWindow(RPackageLister *lister)
   : _listViewControl(lister, _packageListView), _lister(lister),
     _packagePopup(this), _userDialog(this), _packageTip(_packageListView)
{
   // Fix buttons in the toolbar. Right now qt-designer is
   // saving buttons in XPM format.
   _refreshButton->setPixmap(RQPixmaps::find("update.png"));
   _upgradeButton->setPixmap(RQPixmaps::find("distupgrade.png"));
   _commitButton->setPixmap(RQPixmaps::find("proceed.png"));

   // Plug progress system into the lister.
   _cacheProg = new RQCacheProgress(this);
   _lister->setProgressMeter(_cacheProg);
   _lister->setUserDialog(&_userDialog);

   _listViewControl.setClickable(true);

   // Connect menubar actions.
   connect(fileImportMarksAction, SIGNAL(activated()),
           this, SLOT(importMarks()));
   connect(fileExportMarksAction, SIGNAL(activated()),
           this, SLOT(exportMarks()));
   connect(fileCommitAction, SIGNAL(activated()), this, SLOT(commitChanges()));
   connect(fileUpgradeAction, SIGNAL(activated()), this, SLOT(distUpgrade()));
   connect(fileRefreshAction, SIGNAL(activated()), this, SLOT(refreshCache()));
   connect(fileFixBrokenAction, SIGNAL(activated()), this, SLOT(fixBroken()));
   connect(editUndoAction, SIGNAL(activated()), this, SLOT(undo()));
   connect(editRedoAction, SIGNAL(activated()), this, SLOT(redo()));
   connect(editFindAction, SIGNAL(activated()), this, SLOT(findPackage()));
   connect(editFiltersAction, SIGNAL(activated()), this, SLOT(editFilters()));

   // Connect toolbar buttons.
   connect(_commitButton, SIGNAL(clicked()), this, SLOT(commitChanges()));
   connect(_upgradeButton, SIGNAL(clicked()), this, SLOT(distUpgrade()));
   connect(_refreshButton, SIGNAL(clicked()), this, SLOT(refreshCache()));

   // Reload subviews after cache opening.
   connect(&cacheObserver, SIGNAL(cacheOpen()), this, SLOT(reloadViews()));
   connect(&cacheObserver, SIGNAL(cacheOpen()), this, SLOT(reloadPackages()));

   // Handle actions on views and filters.
   connect(_viewsComboBox, SIGNAL(activated(int)), this, SLOT(changedView(int)));
   connect(_subViewsListBox, SIGNAL(highlighted(int)), this, SLOT(changedSubView(int)));
#if 0
   connect(_filtersComboBox, SIGNAL(activated(int)), this, SLOT(changedFilter(int)));
#endif
   restoreState();

   // Finally, show main window.
   show();

}

RQMainWindow::~RQMainWindow()
{
   saveState();
   RWriteConfigFile(*_config);
   delete _cacheProg;
}

// Slots

void RQMainWindow::restoreState()
{
#if 0
   _lister->restoreFilters();
#endif

   reloadViews();

#if 0
   reloadFilters();
#endif
}

void RQMainWindow::saveState()
{
}

void RQMainWindow::reloadViews()
{
   // Reload combo. Only needed once, since views are hardcoded.
   if (_viewsComboBox->count() == 0) {
      vector<string> views = _lister->getViews();
      for (unsigned int i = 0; i != views.size(); i++)
         _viewsComboBox->insertItem(views[i].c_str());
   }
   _viewsComboBox->setCurrentItem(_config->FindI("Synaptic::ViewMode", 0));


   // Reload list from selected view.
   _subViewsListBox->clear();
   vector<string> subViews = _lister->getSubViews();
   for (unsigned int i = 0; i != subViews.size(); i++)
      _subViewsListBox->insertItem(subViews[i].c_str());
}

void RQMainWindow::changedView(int index)
{
   _lister->setView(index);
   _lister->setSubView("none");
   reloadViews();
   reloadPackages();
}

void RQMainWindow::changedSubView(int index)
{
   _lister->setSubView(_subViewsListBox->text(index).ascii());
   reloadPackages();
}

#if 0
void RQMainWindow::reloadFilters()
{
   _filtersComboBox->clear();
   _filtersComboBox->insertItem(tr("All Packages"));
   vector<string> filters = _lister->getFilterNames();
   for (unsigned int i = 0; i != filters.size(); i++)
      _filtersComboBox->insertItem(tr(filters[i].c_str()));
}

void RQMainWindow::changedFilter(int index)
{
   _lister->setFilter(index-1);
   reloadPackages();
}
#endif

void RQMainWindow::reloadPackages()
{
   _listViewControl.reloadViewPackages();
}

void RQMainWindow::distUpgrade()
{
   if (!_lister->check()) {
      _userDialog.error(tr("Operation not possible with broken packages.\n"
                           "Please fix them first."));
      return;
   }

   RPackageLister::pkgState state;
   _lister->saveState(state);

   _lister->distUpgrade();

   vector<RPackage *> selected;
   vector<RPackage *> changed;
   _lister->getStateChanges(state, changed, changed, changed,
                            changed, changed, changed,
			    changed, selected);

   if (!changed.empty()) {
      RQChangesWindow changes(this, changed);
      changes.exec();
      if (changes.result() == QDialog::Rejected) {
         _lister->restoreState(state);
         changed.clear();
      } else {
         _lister->saveUndoState(state);
      }
   } else {
      _userDialog.proceed(tr("Your system is up-to-date!"));
   }

   _packageListView->triggerUpdate();
}

void RQMainWindow::fixBroken()
{
   if (_lister->check()) {
      _userDialog.proceed(tr("There are no broken packages."));
      return;
   }

   RPackageLister::pkgState state;
   _lister->saveState(state);

   _lister->fixBroken();

   vector<RPackage *> selected;
   vector<RPackage *> changed;
   _lister->getStateChanges(state, changed, changed, changed,
                            changed, changed, changed,
			    changed, selected);

   if (!changed.empty()) {
      RQChangesWindow changes(this, changed);
      changes.exec();
      if (changes.result() == QDialog::Rejected) {
         _lister->restoreState(state);
         selected.clear();
         changed.clear();
      } else {
         _lister->saveUndoState(state);
      }
   } else {
      _userDialog.error(tr("Can't fix broken packages!"));
   }

   _packageListView->triggerUpdate();
}

void RQMainWindow::commitChanges()
{
   if (!_lister->check()) {
      _userDialog.error(tr("Operation not possible with broken packages.\n"
                           "Please fix them first."));
      return;
   }

   int installed;
   int broken;
   int toInstall;
   int toReInstall;
   int toRemove;
   double sizeChange;
   _lister->getStats(installed, broken, toInstall, toReInstall, toRemove,
                     sizeChange);
   if (!toInstall && !toRemove)
      return;

   RQSummaryWindow summary(this, _lister);
   summary.exec();
   if (summary.result() == QDialog::Rejected)
      return;

   setEnabled(false);

   // Pointers to packages will become invalid.
   _packageListView->clear();

   // XXX Save selections to a temporary file here.

   RQFetchWindow fetch(this);
#ifdef HAVE_RPM
   RQInstallWindow install(this, _lister);
#else
   RQDummyInstallWindow install;
#endif

   _lister->commitChanges(&fetch, &install);

   if (!_lister->openCache(true)) {
      _userDialog.showErrors();
      exit(1);
   }

   // XXX Restore selections from temporary file here.

   reloadViews();
   reloadPackages();

   setEnabled(true);
}

void RQMainWindow::refreshCache()
{
   // We don't want someone messing with the cache meanwhile.
   setEnabled(false);

   // Pointers to packages will become invalid.
   _packageListView->clear();

   RQFetchWindow fetch(this);

   // XXX Save selections to a temporary file here.

   string error;
   if (!_lister->updateCache(&fetch, error))
      _userDialog.showErrors();

   // Forget new packages here.

   if (!_lister->openCache(true))
      _userDialog.showErrors();

   // XXX Restore selections from temporary file here.

   reloadViews();
   reloadPackages();

   setEnabled(true);
}


void RQMainWindow::editFilters()
{
   RQFiltersWindow filters(this, _lister->filterView());
   filters.exec();
#if 0
   reloadFilters();
#endif
   _lister->filterView()->storeFilters();

   // FIXME: update list here, like in gtk: refreshTable()
   //        because after a filter change, the view should be updated as well
}

void RQMainWindow::undo()
{
   _lister->undo();
   _packageListView->triggerUpdate();
}

void RQMainWindow::redo()
{
   _lister->redo();
   _packageListView->triggerUpdate();
}

void RQMainWindow::importMarks()
{
   QFileDialog dialog;
   dialog.setMode(QFileDialog::ExistingFile);
   if (dialog.exec() == QDialog::Accepted) {
      QString filename = dialog.selectedFile();
      ifstream filestream(filename.ascii());
      if (!filestream) {
         _userDialog.error(tr("Can't read from %1!").arg(filename));
      } else {
         _lister->readSelections(filestream);
         _packageListView->triggerUpdate();
      }
   }
}

void RQMainWindow::exportMarks()
{
   QFileDialog dialog;
   dialog.setMode(QFileDialog::AnyFile);
   if (dialog.exec() == QDialog::Accepted) {
      QString filename = dialog.selectedFile();
      ofstream filestream(filename.ascii());
      if (!filestream) {
         _userDialog.error(tr("Can't write to %1!").arg(filename));
      } else {
         _lister->writeSelections(filestream, false);
      }
   }
}

void RQMainWindow::findPackage()
{
   RQFindWindow find(this, _lister);
   find.exec();
   _packageListView->triggerUpdate();
}

// vim:ts=3:sw=3:et
