/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *      Max Howell, Last.fm Ltd <max@last.fm>                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "aboutdialog.h"
#include "autoupdater.h"
#include "browserthread.h"
#include "configwizard.h"
#include "confirmdialog.h"
#include "container.h"
#include "containerutils.h"
#include "controlinterface.h"
#include "deleteuserdialog.h"
#include "interfaces/ExtensionInterface.h"
#include "iconshack.h"
#include "interfaces/InputInterface.h"
#include "itunesscript.h"
#include "logindialog.h"
#include "macstyleoverrides.h"
#include "mediadevices/mediadevice.h"
#include "metadata.h"
#include "interfaces/InputInterface.h"
#include "playerlistener.h"
#include "Radio.h"
#include "recommenddialog.h"
#include "settingsdialog.h"
#include "Scrobbler-1.2.h"
#include "Settings.h"
#include "systray.h"
#include "tagdialog.h"
#include "interfaces/TranscodeInterface.h"
#include "recommenddialog.h"
#include "RestStateWidget.h"
#include "updatewizard.h"
#include "volumeslider.h"
#include "WebService.h"
#include "WebService/Request.h"
#include "winstyleoverrides.h"
#include "winutils.h"
#include "utils.h"
#include "LastMessageBox.h"

#include "sidebarextension/sidebartree.h"
#include "searchextension/searchextension.h"
#include "metadataextension/metadataextension.h"

Container *Container::s_instance = 0;
static const int k_statusBarFontSizeMac = 10;

// Station bar grey bg colour
static const QColor k_stationBarGreyTop = QColor( 0xba, 0xba, 0xba, 0xff );
static const QColor k_stationBarGreyMiddle = QColor( 0xe2, 0xe2, 0xe2, 0xff );
static const QColor k_stationBarGreyBottom = QColor( 0xff, 0xff, 0xff, 0xff );

// Track bar blue bg colour
static const QColor k_trackBarBkgrBlueTop = QColor( 0xeb, 0xf0, 0xf2, 0xff );
static const QColor k_trackBarBkgrBlueMiddle = QColor( 0xe5, 0xe9, 0xec, 0xff );
static const QColor k_trackBarBkgrBlueBottom = QColor( 0xdc, 0xe2, 0xe5, 0xff );

// Track bar progress bar colour
static const QColor k_trackBarProgressTop = QColor( 0xd6, 0xde, 0xe6, 0xff );
static const QColor k_trackBarProgressMiddle = QColor( 0xd0, 0xd9, 0xe2, 0xff );
static const QColor k_trackBarProgressBottom = QColor( 0xca, 0xd4, 0xdc, 0xff );

// Track bar scrobbled colour
static const QColor k_trackBarScrobbledTop = QColor( 0xba, 0xc7, 0xd7, 0xff );
static const QColor k_trackBarScrobbledMiddle = QColor( 0xb8, 0xc4, 0xd5, 0xff );
static const QColor k_trackBarScrobbledBottom = QColor( 0xb5, 0xc1, 0xd2, 0xff );


Container::Container()
        : QMainWindow()
        , m_eUserInfo( 0 )
        , m_recommendDlg( new RecommendDialog( this ) )
        , m_userCheck( false )
        , m_sidebarEnabled( false )
        , m_stopped( true ) 
        , m_closingDown( false )
{
    //TODO REFACTOR! The size of this monster is scary.

    // HACK: this is just to make sure the web service gets created as part of
    // the main GUI thread. If we don't do this, it will get created as part
    // of the HttpInput initialisation and then it won't work because it's
    // done from a different thread.
    The::webService();

    s_instance = this;

    m_iControl = new ControlInterface( this );
    m_listener = new CPlayerListener( this );
    m_settingsDialog = new SettingsDialog( this );
    m_updater = new CAutoUpdater( this );

    m_scrobbler = new ScrobblerManager( this );
    connect( m_scrobbler, SIGNAL(status( int, QVariant )), SLOT(onScrobblerStatusChange( int, QVariant )) );

    m_scrobbleLabel = new QLabel( this );
    m_scrobbleToggle = new ImageButton( this );
  #ifdef Q_WS_MAC
    m_scrobbleGraphicMac = new ImageButton( this );
  #endif

  #if QT_VERSION >= 0x040300
    setUnifiedTitleAndToolBarOnMac( true );
  #endif

    setWindowTitle( "Last.fm" ); //don't translate this!
    m_radio = new Radio( this );

    ui.setupUi( this );
  #ifdef Q_WS_X11
    setWindowIcon( QIcon( dataPath( "icons/as.png" ) ) );
  #endif
    setMinimumSize( 551, 400 );
    setAcceptDrops( true );

  #ifdef Q_WS_MAC
    // Start helper daemon _before_ we bind any sockets, otherwise they might
    // end up being blocked by the child-process.
    installHelperApp();
  #endif

//////
    // don't use CTRL as that varies by platform, it is easier to have a single
    // shortcut to tell all users
    QObject *openL = new QShortcut( QKeySequence( "Alt+Shift+L" ), this );
    QObject* openH = new QShortcut( QKeySequence( "Alt+Shift+H" ), this );
    QObject* openF = new QShortcut( QKeySequence( "Alt+Shift+F" ), this );
    
    connect( openL, SIGNAL(activated()), SLOT(onAltShiftL()) );
    connect( openH, SIGNAL(activated()), SLOT(onAltShiftH()) );
    connect( openF, SIGNAL(activated()), SLOT(onAltShiftF()) );

//////
    loadExtensions();
    initListener();

    // <ToolBar>
    QToolBar* toolbar = addToolBar( "Last.fm" );
    toolbar->setObjectName( "Lastfm_Main_Toolbar" );
    toolbar->setMovable( false );
    toolbar->setContextMenuPolicy( Qt::CustomContextMenu );
    toolbar->setToolButtonStyle( Qt::ToolButtonTextUnderIcon );
  #if defined( Q_WS_MAC ) && QT_VERSION >= 0x040300
        // using unified toolbar so don't do this
  #else
        toolbar->setAutoFillBackground( true );
  #endif
  #ifdef WIN32
        // Can't use 32 for height as labels get truncated on Win classic
        toolbar->setIconSize( QSize( 50, 34 ) );
  #else
        toolbar->setIconSize( QSize( 32, 32 ) );
  #endif

    ui.actionSidebar->setShortcut( tr( "Ctrl+F" ) );
    ui.actionSidebar->setToolTip( tr( "Show/hide my profile" ) );
  #ifdef WIN32
    ui.actionSidebar->setCheckable( true );
  #endif
    connect( ui.actionSidebar, SIGNAL(triggered()), SLOT(toggleSidebar()) );

    // Copy the menu action and add the icon to it. Don't want the icon in the
    // menu as it will obscure the tick.
    m_sidebarActionWithIcon = new QAction( toolbar );
    m_sidebarActionWithIcon->setIcon( QIcon( dataPath( "buttons/myprofile.png" ) ) );
    m_sidebarActionWithIcon->setText( tr( "My Profile" ) );
    m_sidebarActionWithIcon->setToolTip( tr( "Show/hide my profile" ) );
  #ifdef WIN32
    m_sidebarActionWithIcon->setCheckable( true );
  #endif
    connect( m_sidebarActionWithIcon, SIGNAL(triggered()), SLOT(toggleSidebar()) );

    toolbar->addAction( m_sidebarActionWithIcon );
    toolbar->addSeparator();

    //m_recommendAction  = toolbar->addAction( QIcon( dataPath( "buttons/recommend.png" ) ), tr( "Recommend" ), this, SLOT( showRecommendDialog() ) );
    ui.actionRecommend->setIcon( QIcon( dataPath( "buttons/recommend.png" ) ) );
    ui.actionRecommend->setShortcut( tr( "Ctrl+R" ) );
    ui.actionRecommend->setToolTip( tr( "Recommend what's currently playing to someone" ) );
    toolbar->addAction( ui.actionRecommend );
    connect( ui.actionRecommend, SIGNAL(triggered()), SLOT(showRecommendDialog()) );

    //m_tagAction = toolbar->addAction( QIcon( dataPath( "buttons/tag.png" ) ), tr( "Tag" ), this, SLOT( showTagDialog() ) );
    ui.actionTag->setIcon( QIcon( dataPath( "buttons/tag.png" ) ) );
    ui.actionTag->setShortcut( tr( "Ctrl+T" ) );
    ui.actionTag->setToolTip( tr( "Tag what's currently playing" ) );
    toolbar->addAction( ui.actionTag );
    connect( ui.actionTag, SIGNAL(triggered()), SLOT(showTagDialog()) );

    toolbar->addSeparator();

    //m_loveAction = toolbar->addAction( QIcon( dataPath( "buttons/love.png" ) ), tr( "Love" ), this, SLOT( love() ) );
    ui.actionLove->setIcon( QIcon( dataPath( "buttons/love.png" ) ) );
    ui.actionLove->setShortcut( tr( "Ctrl+L" ) );
    ui.actionLove->setToolTip( tr( "Express your love for the track that's playing" ) );
    toolbar->addAction( ui.actionLove );
    connect( ui.actionLove, SIGNAL( triggered() ),
             this,          SLOT( love() ) );

    //m_banAction  = toolbar->addAction( QIcon( dataPath( "buttons/ban.png" ) ), tr( "Ban" ), this, SLOT( ban() ) );
    ui.actionBan->setIcon( QIcon( dataPath( "buttons/ban.png" ) ) );
    ui.actionBan->setShortcut( tr( "Ctrl+B" ) );
    ui.actionBan->setToolTip( tr( "Don't ever play me this again" ) );
    toolbar->addAction( ui.actionBan );
    connect( ui.actionBan, SIGNAL( triggered() ),
             this,         SLOT( ban() ) );

    //m_skipAction = new QAction( QIcon( dataPath( "buttons/skip.png" ) ), tr( "Skip" ), this );
    ui.actionSkip->setIcon( QIcon( dataPath( "buttons/skip.png" ) ) );
    ui.actionSkip->setShortcut( QKeySequence( Qt::CTRL + Qt::Key_Right ) ); //NOTE don't translate
    ui.actionSkip->setToolTip( tr( "Skip track" ) );

    //m_playAction  = new QAction( QIcon( dataPath( "buttons/play.png" ) ), tr( "Play" ), this );
    ui.actionPlay->setIcon( QIcon( dataPath( "buttons/play.png" ) ) );
    ui.actionPlay->setShortcut( Qt::Key_Space );
    ui.actionPlay->setToolTip( tr( "Resume last radio station" ) );

    //m_stopAction  = new QAction( QIcon( dataPath( "buttons/stop.png" ) ), tr( "Stop" ), this );
    ui.actionStop->setIcon( QIcon( dataPath( "buttons/stop.png" ) ) );
    ui.actionStop->setToolTip( tr( "Stop radio" ) );

    toolbar->addSeparator();

    toolbar->addAction( ui.actionPlay );
    toolbar->addAction( ui.actionStop );
    toolbar->addAction( ui.actionSkip );

  #ifdef Q_WS_MAC
    //ui.actionSettings->setText( tr( "Options" ) );
  #endif
    ui.actionSettings->setMenuRole( QAction::PreferencesRole );
    ui.actionSettings->setShortcut( tr( "Ctrl+O" ) );
    ui.actionToggleScrobbling->setShortcut( tr( "Ctrl+E" ) );

    // Help shortcut
  #ifdef Q_WS_MAC
    ui.actionFAQ->setShortcut( tr( "Ctrl+?" ) );
  #else
    ui.actionFAQ->setShortcut( tr( "F1") );
  #endif

    // Workaround for showing "Space" shortcut in the Mac menu    
  #ifdef Q_WS_MAC
    ui.actionPlay->setIconText( ui.actionPlay->text().replace( "&", "" ) );
    ui.actionStop->setIconText( ui.actionStop->text().replace( "&", "" ) );

    // For Systray we don't want to display the hotkey in the menu
    m_sysTrayActionPlay = new QAction( this );
    m_sysTrayActionStop = new QAction( this );
    m_sysTrayActions = new QActionGroup( this );

    m_sysTrayActions->addAction( m_sysTrayActionPlay );
    m_sysTrayActions->addAction( m_sysTrayActionStop );

    connect( m_sysTrayActionPlay,           SIGNAL( triggered() ),
             this,                            SLOT( play() ) );
    connect( m_sysTrayActionStop,           SIGNAL( triggered() ),
             this,                            SLOT( stop() ) );

    m_sysTrayActionPlay->setText( ui.actionPlay->text() ); 
    m_sysTrayActionStop->setText( ui.actionStop->text() );
    m_sysTrayActionPlay->setIcon( ui.actionPlay->icon() );
    m_sysTrayActionStop->setIcon( ui.actionStop->icon() );

    // the spaces align the word space to the right of the menu, there is apparently
    // no alternative on mac osx 10.4 and below. Without the extra text the space
    // shortcut is rendered as ' '. Sweet!
    ui.actionPlay->setText( ui.actionPlay->text() + "            " + tr("Space" ) );
    ui.actionStop->setText( ui.actionStop->text() + "            " + tr("Space" ) );
  #endif

    ui.actionCheckForUpdates->setMenuRole( QAction::ApplicationSpecificRole );
    ui.actionAboutLastfm->setMenuRole( QAction::AboutRole );

  #ifdef Q_WS_MAC
    ui.menuHelp->insertAction( ui.actionCheckForUpdates, ui.actionAboutLastfm ); // Change the order of these 2 entries on the mac
  #else
    // Additional seperators for menu entries for the other platforms
    ui.menuHelp->insertSeparator( ui.actionCheckForUpdates );
    ui.menuHelp->insertSeparator( ui.actionAboutLastfm );
  #endif

    //NOTE don't translate these! It breaks the shortcut due to the arrow key string getting translated
    ui.actionVolumeUp->setShortcut( QKeySequence( Qt::CTRL + Qt::Key_Up ) );
    ui.actionVolumeDown->setShortcut( Qt::CTRL + Qt::Key_Down );
  #ifdef Q_WS_MAC
    ui.actionMute->setShortcut( Qt::CTRL + Qt::ALT + Qt::Key_Down );
  #else
    ui.actionMute->setShortcut( tr( "Ctrl+M" ) );
  #endif

  #ifdef Q_WS_X11
    delete ui.actionCheckForUpdates;
  #endif

    QIcon scrobbleIcon( dataPath( "icons/scrobble16.png" ) );
    QString scrobbleTooltip = tr( "Click to enable/disable scrobbling" );
    m_scrobbleNormal = scrobbleIcon.pixmap( 16, 16, QIcon::Normal );
    m_scrobbleDisabled = scrobbleIcon.pixmap( 16, 16, QIcon::Disabled );
    m_scrobbleToggle->setImages( m_scrobbleNormal, m_scrobbleNormal, m_scrobbleNormal, m_scrobbleDisabled );
    m_scrobbleToggle->setToolTip( scrobbleTooltip );

    statusBar()->addPermanentWidget( m_scrobbleLabel );

  #ifdef Q_WS_MAC
    QPalette lp = m_scrobbleLabel->palette();
    lp.setColor( QPalette::Text, QColor( 0x59, 0x59, 0x59 ) );
    lp.setColor( QPalette::WindowText, QColor( 0x59, 0x59, 0x59 ) );
    m_scrobbleLabel->setPalette( lp );
    QPixmap scrobbleGraphicMac( dataPath( "icons/scrobbling_graphic.png" ) );
    m_scrobbleGraphicMac->setImages( scrobbleGraphicMac, scrobbleGraphicMac, scrobbleGraphicMac, scrobbleGraphicMac );
    m_scrobbleGraphicMac->setToolTip( scrobbleTooltip );
    statusBar()->addPermanentWidget( m_scrobbleGraphicMac );
  #endif

    statusBar()->addPermanentWidget( m_scrobbleToggle );

    ui.actionToggleScrobbling->setCheckable( true );
    ui.actionToggleDiscoveryMode->setCheckable( true );

    // <PlayControls>
    QWidget *playControls = new QWidget( toolbar );
    ui_playcontrols.setupUi( playControls );

    QAction *playControlsAction = toolbar->addWidget( playControls );
    playControlsAction->setVisible( true );

    // </PlayControls>
    // </ToolBar>

  #ifndef Q_WS_MAC
    QMenu* fileMenu = new QMenu( tr( "&File" ), this );
    QAction* a = fileMenu->addAction( tr( "&Send to Tray" ), this, SLOT( close() ) );
    a->setToolTip( tr( "Send application to system tray" ) );
    fileMenu->addSeparator();
    QAction* exitAction = fileMenu->addAction( tr( "E&xit" ), this, SLOT( actualClose() ) );
  #ifdef Q_WS_X11
    exitAction->setShortcut( tr( "CTRL+Q" ) );
    exitAction->setText( tr("&Quit") );
  #endif
    menuBar()->insertMenu( menuBar()->actions().at( 0 ), fileMenu );
  #endif

    QShortcut *s = new QShortcut( this );
    s->setKey( QKeySequence( "CTRL+W" ) );
    connect( s, SIGNAL(activated()), SLOT(hide()) );

    ui.actionTag->setEnabled( false );
    ui.actionRecommend->setEnabled( false );
    ui.actionLove->setEnabled( false );
    ui.actionBan->setEnabled( false );
    ui.actionSkip->setEnabled( false );
    ui.actionStop->setEnabled( false );
    ui.actionStop->setVisible( false );

    ui.stack->setAcceptDrops( false );

  #ifdef Q_WS_MAC
    {
        QApplication::setStyle( new MacStyleOverrides() );

        QPalette p = ui.centralwidget->palette();
        p.setColor( QPalette::Window, QColor( 0xe9, 0xe9, 0xe9 ) );
        ui.centralwidget->setPalette( p );
        p.setColor( QPalette::Button, QColor( 0xe9, 0xe9, 0xe9 ) );

        // Hacky hackety McHackerson
        QLinearGradient grad( 0.5, 0.0, 0.5, 20.0 );
        grad.setColorAt( 0.0, QColor( 0x8c, 0x8c, 0x8c ) );
        grad.setColorAt( 0.05, QColor( 0xf7, 0xf7, 0xf7 ) );
        grad.setColorAt( 1.0, QColor( 0xe4, 0xe4, 0xe4 ) );
        QBrush b( grad );
        p = statusBar()->palette();
        p.setBrush( QPalette::Window, b );
        p.setBrush( QPalette::Base, b );
        p.setColor( QPalette::Text, QColor( 0x59, 0x59, 0x59 ) );
        p.setColor( QPalette::WindowText, QColor( 0x59, 0x59, 0x59 ) );
        statusBar()->setPalette( p );

        // mac specific mainwindow adjustments
        QFrame *hline = new QFrame;
        hline->setFrameStyle( QFrame::HLine | QFrame::Plain );
        p.setColor( QPalette::WindowText, QColor( 140, 140, 140 ) );
        hline->setPalette( p );
        ui.vboxLayout1->insertWidget( ui.vboxLayout1->indexOf( ui.stack ), hline );
        ui.stack->setFrameStyle( QFrame::NoFrame );

        //alter the spacings and that
        ui.vboxLayout2->setMargin( 14 );
        for (int x = 0; x < ui.vboxLayout1->count(); ++x)
            if (ui.vboxLayout1->itemAt( x )->spacerItem())
                delete ui.vboxLayout1->takeAt( x );

        QFont f = statusBar()->font();
        f.setPixelSize( k_statusBarFontSizeMac );
        statusBar()->setFont( f );
        m_scrobbleLabel->setFont( f );
    }

  #elif defined WIN32
    // This is in order to remove the borders around the status bar widgets
    QString styleName = QApplication::style()->objectName();
    if( styleName == "windowsxp" )
    {
        statusBar()->setStyle( new WinXPStyleOverrides() );
    }
    else
    {
        statusBar()->setStyle( new WinStyleOverrides() );

        ui.stack->layout()->setMargin( 1 );
    }

  #elif defined Q_WS_X11
    statusBar()->setStyle( new WinXPStyleOverrides() );

    //aesthetics, separates the statusbar and central widget slightly
    static_cast<QVBoxLayout*>(ui.centralwidget->layout())->addSpacing( 2 );
  #endif

    ui.songTimeBar->setEnabled( false );
    ui.songTimeBar->setItemType( ItemTrack );

    QLinearGradient trackBarBkgrGradient( 0, 0, 0, ui.songTimeBar->height() );
    trackBarBkgrGradient.setColorAt( 0, k_trackBarBkgrBlueTop );
    trackBarBkgrGradient.setColorAt( 0.5, k_trackBarBkgrBlueMiddle );
    trackBarBkgrGradient.setColorAt( 0.51, k_trackBarBkgrBlueBottom );
    trackBarBkgrGradient.setColorAt( 1, k_trackBarBkgrBlueBottom );

    QLinearGradient trackBarProgressGradient( 0, 0, 0, ui.songTimeBar->height() );
    trackBarProgressGradient.setColorAt( 0, k_trackBarProgressTop );
    trackBarProgressGradient.setColorAt( 0.5, k_trackBarProgressMiddle );
    trackBarProgressGradient.setColorAt( 0.51, k_trackBarProgressBottom );
    trackBarProgressGradient.setColorAt( 1, k_trackBarProgressBottom );

    QLinearGradient trackBarScrobbledGradient( 0, 0, 0, ui.songTimeBar->height() );
    trackBarScrobbledGradient.setColorAt( 0, k_trackBarScrobbledTop );
    trackBarScrobbledGradient.setColorAt( 0.5, k_trackBarScrobbledMiddle );
    trackBarScrobbledGradient.setColorAt( 0.51, k_trackBarScrobbledBottom );
    trackBarScrobbledGradient.setColorAt( 1, k_trackBarScrobbledBottom );

    ui.songTimeBar->setBackgroundGradient( trackBarBkgrGradient );
    ui.songTimeBar->setForegroundGradient( trackBarProgressGradient );
    ui.songTimeBar->setScrobbledGradient( trackBarScrobbledGradient );

    connect( this,           SIGNAL( newSong( MetaData ) ),
             ui.songTimeBar, SLOT( setNewTrack( const MetaData& ) ) );

    ui.stationTimeBar->setProgressEnabled( false );
    ui.stationTimeBar->setVisible( false );
    ui.stationTimeBar->setItemType( ItemStation );

    QLinearGradient stationBarGradient( 0, 0, 0, ui.stationTimeBar->height() );
    stationBarGradient.setColorAt( 0, k_stationBarGreyTop );
    stationBarGradient.setColorAt( 0.07, k_stationBarGreyMiddle );
    stationBarGradient.setColorAt( 0.14, k_stationBarGreyBottom );
    stationBarGradient.setColorAt( 1, k_stationBarGreyBottom );

    if ( qApp->arguments().contains( "--sanity" ) )
        stationBarGradient = trackBarBkgrGradient;
    ui.stationTimeBar->setBackgroundGradient( stationBarGradient );
    ui.stationTimeBar->setStopWatch( &m_radio->stationStopWatch() );

//////
    restoreGeometry( The::settings().containerGeometry() );
    setWindowState( The::settings().containerWindowState() );

//////
  #ifdef Q_WS_MAC
    ui.splitter->setHandleWidth( 1 );
  #endif
      
    ui.splitter->setCollapsible( 0, false );
    ui.splitter->setCollapsible( 1, false );
    ui.splitter->setStretchFactor( 0, 0 );
    ui.splitter->setStretchFactor( 1, 1 );
    QByteArray state = The::settings().splitterState();
    if (state.size())
        ui.splitter->restoreState( state );
        
    #ifdef Q_WS_MAC
    //HACK for osx since the splitter is undetectable for most users
    QList<int> sizes = ui.splitter->sizes();
    if (sizes.count() == 2 && sizes[0] < 100)
    {
        sizes[1] -= 190 - sizes[0];
        sizes[0] = 190;
        ui.splitter->setSizes( sizes );
    }
    #endif


////// force a white background as we don't support colour variants, soz 
    QPalette p = centralWidget()->palette();
    p.setBrush( QPalette::Base, Qt::white );
    centralWidget()->setPalette( p );

//////
    initTray();

////// GUI slots
    connect( ui.actionDashboard,            SIGNAL( triggered() ),
             this,                            SLOT( gotoProfile() ) );
    connect( ui.actionSettings,             SIGNAL( triggered() ),
             this,                            SLOT( showSettingsDialog() ) );
    connect( ui.actionGetPlugin,            SIGNAL( triggered() ),
             this,                            SLOT( getPlugin() ) );
  #ifndef Q_WS_X11
    connect( ui.actionCheckForUpdates,      SIGNAL( triggered() ),
             this,                            SLOT( checkForUpdates() ) );
  #endif
    connect( ui.actionAddUser,              SIGNAL( triggered() ),
             this,                            SLOT( addUser() ) );
    connect( ui.actionDeleteUser,           SIGNAL( triggered() ),
             this,                            SLOT( deleteUser() ) );
    connect( ui.actionToggleScrobbling,     SIGNAL( triggered() ),
             this,                            SLOT( toggleScrobbling() ) );
    connect( ui.actionToggleDiscoveryMode,  SIGNAL( triggered() ),
             this,                            SLOT( toggleDiscoveryMode() ) );
    connect( ui.actionAboutLastfm,          SIGNAL( triggered() ),
             this,                            SLOT( about() ) );
    connect( ui.menuUser,                   SIGNAL( aboutToShow() ),
             this,                            SLOT( showUserMenu() ) );
    connect( ui.menuUser,                   SIGNAL( triggered( QAction* ) ),
             this,                            SLOT( userSelected( QAction* ) ) );
    connect( ui.actionSkip,                 SIGNAL( triggered() ),
             this,                            SLOT( skip() ) );
    connect( ui.actionStop,                 SIGNAL( triggered() ),
             this,                            SLOT( stop() ) );
    connect( ui.actionPlay,                 SIGNAL( triggered() ),
             this,                            SLOT( play() ) );
    connect( ui_playcontrols.volume,        SIGNAL( valueChanged( int ) ),
             m_radio,                         SLOT( setVolume( int ) ) );
    connect( m_scrobbleToggle,              SIGNAL( clicked() ),
             this,                            SLOT( toggleScrobbling() ) );
    connect( ui.actionVolumeUp,             SIGNAL( triggered() ),
             this,                            SLOT( volumeUp() ) );
    connect( ui.actionVolumeDown,           SIGNAL( triggered() ),
             this,                            SLOT( volumeDown() ) );
    connect( ui.actionMute,                 SIGNAL( triggered() ),
             this,                            SLOT( mute() ) );
    connect( ui.actionFAQ,                  SIGNAL( triggered() ),
             this,                            SLOT( showFAQ() ) );
    connect( ui.actionForums,               SIGNAL( triggered() ),
             this,                            SLOT( showForums() ) );
    connect( ui.actionInviteAFriend,        SIGNAL( triggered() ),
             this,                            SLOT( inviteAFriend() ) );

  #ifdef Q_WS_MAC
    connect( m_scrobbleGraphicMac,          SIGNAL( clicked() ),
             this,                            SLOT( toggleScrobbling() ) );
  #endif

    // Services & Extensions
    //connect( The::webService(), SIGNAL(handshakeResult( Handshake* )), SLOT(handshakeResult( Handshake* )) );
    connect( The::webService(), SIGNAL(success( Request* )), SLOT(webServiceSuccess( Request* )) );
    connect( The::webService(), SIGNAL(failure( Request* )), SLOT(webServiceFailure( Request* )), Qt::QueuedConnection );

    connect( m_radio,      SIGNAL( buffering( int, int ) ),
             this,           SLOT( onRadioBuffering( int, int ) ) );
    connect( m_radio,      SIGNAL( stateChanged( RadioState ) ),
             this,           SLOT( onRadioStateChange( RadioState ) ) );
    connect( m_radio,      SIGNAL( error( RadioError, const QString& ) ),
             this,           SLOT( onRadioError( RadioError, const QString& ) ), Qt::QueuedConnection );

    connect( &The::settings(), SIGNAL( userSettingsChanged( UserSettings& ) ),
             this,               SLOT( updateUserStuff( UserSettings& ) ) );
    connect( &The::settings(), SIGNAL( doReconnect() ),
             this,               SLOT( initServices() ) );

    // Auto updater
    connect( m_updater, SIGNAL( updateCheckDone( bool, bool, QString ) ),
             this,        SLOT( updateCheckDone( bool, bool, QString ) ) );

    // update volume _after_ signals have been connected
    ui_playcontrols.volume->setValue( The::settings().volume() );

  #ifdef HIDE_RADIO
    ui_playcontrols.volume->setVisible( false );

    ui.menuControls->setVisible( false );
    ui.actionPlay->setVisible( false );
    ui.actionStop->setVisible( false );
    ui.actionSkip->setVisible( false );
    ui.actionBan->setVisible( false );
    ui.actionToggleDiscoveryMode->setVisible( false );

    ui.tabWidget->removeTab( 0 );

    #ifdef Q_WS_MAC
        m_sysTrayActions->setVisible( false );
    #endif
  #endif

  #ifndef WIN32
    ui.actionGetPlugin->setEnabled( false );
    ui.actionGetPlugin->setVisible( false );
  #else
    ui.menuTools->insertSeparator( ui.actionTag );
  #endif

  #ifdef Q_WS_MAC
    new ITunesScript( this, m_listener );
  #endif

    if ( qApp->arguments().contains( "--debug" ) )
    {
        // Add crash option to menu!
        ui.menuHelp->addAction( "Crash please", this, SLOT( crash() ) );
    }

  #ifdef MONITOR_STREAMING
    ui.menuHelp->addAction( "Stream monitor", m_radio, SLOT( activateStreamMonitor() ) );
  #endif
  
    connect( ui.stack, SIGNAL(currentChanged( int )), SIGNAL(stackIndexChanged( int )) );
  
//////
    m_showRestStateTimer = new QTimer( this );
    m_showRestStateTimer->setSingleShot( true );
    m_showRestStateTimer->setInterval( 200 );
    connect( m_showRestStateTimer, SIGNAL(timeout()), SLOT(onRestStateTimerTimeout()) );

    // Remove the path to the app and parse
    QStringList params = qApp->arguments().mid( 1 );
    m_iControl->parseCommands( params );
}


void
Container::initServices()
{
    Q_DEBUG_BLOCK;

    UserSettings& user = The::settings().currentUser();
    LOG( 3, "Starting app for user: " << user.username() << "\n" );

    // we must restore state here as we save it in toggleSidebar in order to get
    // round the bug in Qt where saveState for the splitter is lost for hidden widgets
    m_sidebarEnabled = !user.sidebarEnabled();
    toggleSidebar();
    
    updateUserStuff( user );
    statusBar()->showMessage( tr( "Connecting to server..." ) );

//////// We now have a current user
    QString username = user.username();
    QString password = user.password();
    QString version = The::settings().version();

    m_radio->init( username, password, version );

////// shut down the current listener, this will cause a scrobble if required
    // also it makes sense as the new user should start with the chance to scrobble
    // the current track if applicable
    if (m_listener->GetActivePlayer())
    {
        CPlayerCommand command( PCMD_STOP, m_listener->GetActivePlayer()->GetID(), m_listener->GetNowPlaying() );
        m_listener->Handle( command );
        
        command.mCmd = PCMD_START;
        m_listener->Handle( command );
    }

//////
    // initialise the scrobbler to the new user, handshake is delayed until current users scrobble completes
    Scrobbler::Init init;
    init.username = username;
    init.client_version = version;
    init.password = password;

    m_scrobbler->handshake( init );

//////
    // enable again when handshake succeeds
    ui.actionPlay->setEnabled( false );
    m_rest_state_widget->setPlayEnabled( false );

    m_rest_state_widget->updatePlayerNames();
}


Container::~Container()
{
    Q_DEBUG_BLOCK;
    
    delete m_metaDataExtension;
}


void
Container::initListener()
{
    connect( m_listener, SIGNAL( trackChanged( TrackInfo, bool ) ),
             this,         SLOT( setNewSong( TrackInfo, bool ) ), Qt::QueuedConnection );

    connect( m_listener, SIGNAL( trackScrobbled( const TrackInfo& ) ),
             this,         SLOT( onTrackScrobbled( const TrackInfo& ) ), Qt::QueuedConnection );

    connect( m_listener, SIGNAL( exceptionThrown( QString ) ),
             this,         SLOT( listenerException( QString ) ) );

    // Start listener worker thread
    m_listener->start();
}


void
Container::dragMoveEvent( QDragMoveEvent* event )
{
  #ifndef HIDE_RADIO
    QString url = event->mimeData()->urls().value( 0 ).toString();
    if ( url.startsWith( "lastfm://" ) )
    {
        event->acceptProposedAction();
    }
    else
        event->ignore();
  #endif // HIDE_RADIO
}


void
Container::dragEnterEvent( QDragEnterEvent* event )
{
  #ifndef HIDE_RADIO
   QString url = event->mimeData()->urls().value( 0 ).toString();
    if ( url.startsWith( "lastfm://" ) )
    {
        event->acceptProposedAction();
    }
    else
        event->ignore();
  #endif // HIDE_RADIO
}


void
Container::dropEvent( QDropEvent* event )
{
  #ifndef HIDE_RADIO
    QString url = event->mimeData()->urls().value( 0 ).toString();
    if ( url.startsWith( "lastfm://" ) )
    {
        m_radio->playStation( StationUrl( url ) );
    }
  #endif // HIDE_RADIO
}


void
Container::closeEvent( QCloseEvent *event )
{
  #ifdef Q_WS_MAC
    if ( !event->spontaneous() )
    {
        actualClose();
        event->ignore();
        return;
    }
  #endif

//  This code needs to be tested on Mac.
//     #ifndef WIN32
//     if ( event->spontaneous() )
//     {
//         actualClose( false );
//         event->ignore();
//         return;
//     }
//     #endif

    // Just minimise to tray
    minimiseToTray();
    event->ignore();
}


bool
Container::event( QEvent* e )
{
    if (e->type() == QEvent::Move || e->type() == QEvent::Resize)
    {
        // Again, Qt is broken, if maximised the saveGeometry function fails to
        // save the geometry for the non-maximised state. So instead we must save it
        // for *every* resize and move event. Yay!

        if (windowState() != Qt::WindowMaximized)
            The::settings().setContainerGeometry( saveGeometry() );
    }

    return QMainWindow::event( e );
}


void
Container::actualClose( bool confirm )
{
    Q_DEBUG_BLOCK;

    bool okToClose = true;

    if ( confirm )
    {
        QString m = tr( "Really quit Last.fm? Any music you listen to will not be scrobbled "
                        "to your profile." );
        ConfirmDialog dlg( ConfirmDialog::Quit, m, this );
        okToClose = dlg.exec() == QDialog::Accepted;
    }

    if ( okToClose )
    {
        hide(); //makes it look like we have closed already to user

        m_closingDown = true;

        // This allows any extensions (i.e. Skype) to clear its stuff. Bit of a hack.
        emit newSong( MetaData() );

        // EJ TODO FIXME: Can't do this as the radio's stopping is asynchronous
        // and it will lead to the radio calling the listener after it's died.
        // If we're not doing this however, we will get an assert about a thread
        // quitting (the StopWatch inside the radio) improperly.
        // Need to solve the whole shutdown thing in a more elegant way.
        m_radio->stop();

        LOGL( Logger::Debug, "Radio state at shutdown: " <<
            CUtils::radioState2String( m_radio->state() ) );

        // Bottom line here is that we must wait until the radio thread has stopped
        // and it has dealt with all outstanding events.
        do
        {
            QApplication::processEvents();
            #ifdef WIN32
                Sleep( 10 );
            #else
                usleep( 100 );
            #endif
        }
        while ( m_radio->state() != State_Stopped &&
                m_radio->state() != State_Uninitialised &&
                m_radio->state() != State_Handshaking &&
                m_radio->state() != State_Handshaken );

        LOG( 3, "Saving appState\n" );

        // don't save if hidden as Qt is shit and this doesn't work, we save 
        // just before hiding too to ensure we always have the state saved correctly
        if (m_sidebarEnabled)
            The::settings().setSplitterState( ui.splitter->saveState() );

        The::settings().setContainerWindowState( windowState() );

        The::settings().setVolume( ui_playcontrols.volume->value() );
        The::currentUser().setSidebarEnabled( m_sidebarEnabled );

        LOG( 3, "Saving config\n" );
        The::settings().save();

        LOG( 3, "Shutting down listener\n" );
        qDebug() << "Shutting down listener";
        m_listener->Stop();

        LOG( 3, "Calling exit\n" );
        QApplication::exit( 0 );

        qApp->sendPostedEvents( &The::scrobbler(), 0 /*all event types*/ );
    }
}


void
Container::minimiseToTray()
{
    hide();

  #ifdef WIN32
    // Do animation and fail gracefully if not possible to find systray
    RECT rectFrame;    // animate from
    RECT rectSysTray;  // animate to

    ::GetWindowRect( (HWND)winId(), &rectFrame );

    // Get taskbar window
    HWND taskbarWnd = ::FindWindow( L"Shell_TrayWnd", NULL );
    if ( taskbarWnd == NULL )
        return;

    // Use taskbar window to get position of tray window
    HWND trayWnd = ::FindWindowEx( taskbarWnd, NULL, L"TrayNotifyWnd", NULL );
    if ( trayWnd == NULL )
        return;

    ::GetWindowRect( trayWnd, &rectSysTray );
    ::DrawAnimatedRects( (HWND)winId(), IDANI_CAPTION, &rectFrame, &rectSysTray );

    // Make it release memory as when minimised
    HANDLE h = ::GetCurrentProcess();
    SetProcessWorkingSetSize( h, -1 ,-1 );
  #endif // WIN32
}


void
Container::toggleDiscoveryMode()
{
    bool enabled = ui.actionToggleDiscoveryMode->isChecked();
    m_radio->setDiscoveryMode( enabled );
}


void
Container::loadExtensions()
{
    m_metaDataExtension = 0;

    m_sidebar = new SideBarTree;
    ui.sidebarFrame->layout()->addWidget( m_sidebar );
    connect( m_sidebar, SIGNAL(statusMessage( QString )), SLOT(statusMessage( QString )) );
    connect( m_sidebar, SIGNAL(plsShowRestState()), SLOT(showRestState()) );
    connect( m_sidebar, SIGNAL(plsShowNowPlaying()), SLOT(showMetaDataWidget()) );

    LOGL( 3, "Added sidebar extension to GUI." );

//////
    ui.stack->setBackgroundRole( QPalette::Base );
    ui.stack->addWidget( m_rest_state_widget = new RestStateWidget );
    m_rest_state_widget->setFocus();

    connect( this,              SIGNAL(playStarted()),
             m_rest_state_widget, SLOT(clear()) );

    connect( &The::settings(),  SIGNAL(userSwitched( UserSettings& )),
             m_rest_state_widget, SLOT(clear()) );

//////

    m_metaDataExtension = new MetaDataExtension;
    m_metaDataExtension->setOwner( ui.stack );
    ui.stack->addWidget( m_metaDataExtension->gui() );

    connect( this,                SIGNAL( newSong( MetaData ) ),
             m_metaDataExtension, SLOT( setMetaData( MetaData ) ) );

    connect( m_metaDataExtension, SIGNAL( trackModerated( MetaData ) ),
             ui.songTimeBar,      SLOT( setNewTrack( MetaData ) ) );

    connect( m_metaDataExtension, SIGNAL( trackModerated( MetaData ) ),
             this,                SLOT( updateWindowTitle( MetaData ) ) );

    connect( m_metaDataExtension, SIGNAL( tagButtonClicked() ),
             this,                SLOT( showTagDialogMD() ) );

    connect( m_metaDataExtension, SIGNAL( urlHovered( QString ) ),
             this,                SLOT( displayUrlInStatusBar( QString ) ) );

    LOGL( 3, "Added metadata extension to GUI." );

    QString dirPath;
  #ifdef WIN32
    // Hack to get it working with VS2005
    dirPath = qApp->applicationDirPath();
  #else
    dirPath = qApp->applicationDirPath() + "/extensions";
  #endif

  #ifndef QT_NO_DEBUG
    dirPath += "/debug";
  #endif

    QDir extensionsDir( dirPath );

    foreach ( QString fileName, extensionsDir.entryList( QDir::Files ) )
    {
        if ( !fileName.startsWith( EXTENSION_PREFIX ) || !QLibrary::isLibrary( extensionsDir.absoluteFilePath( fileName ) ) )
            continue;

        LOGL( 3, "Trying to load extension: " << fileName );
        QObject* plugin = QPluginLoader( extensionsDir.absoluteFilePath( fileName ) ).instance();
        if ( plugin )
        {
            LOGL( 3, "Extension loaded." );
            ExtensionInterface *iExtension = qobject_cast<ExtensionInterface *>( plugin );
            if ( iExtension )
            {
                if ( iExtension->name() == "UserInfo Extension" )
                {
                    m_eUserInfo = iExtension;
                }

                else if ( iExtension->name() == "Growl Notifier Extension" )
                {
                    // Add it to Options
                    LOGL( 3, "Loaded a settings extension" );
                    m_settingsDialog->addExtension( iExtension );

                    connect( m_metaDataExtension, SIGNAL( metadataFetched( MetaData ) ),
                             iExtension, SLOT( notify( MetaData ) ) );

                    m_metaDataExtension->setAlwaysFetchMetaData( true );
                }

                else if ( iExtension->hasSettingsPane() )
                {
                    // Add it to Options
                    LOGL( 3, "Loaded a settings extension" );
                    m_settingsDialog->addExtension( iExtension );

                    connect( this,     SIGNAL( newSong( MetaData ) ),
                             iExtension, SLOT( notify( MetaData ) ) );
                }
            }
        } // endif plugin
        else
        {
            LOGL( 1, "Failed to load " << fileName );
        }
    }
}


void
Container::displayUrlInStatusBar( const QString& url )
{
    QUrl realUrl( QUrl::fromEncoded( url.toLatin1() ) );
    QString msg = realUrl.toString();

    if ( msg != statusBar()->currentMessage() )
    {
        statusBar()->showMessage( msg );
    }
}


void
Container::onRadioError( RadioError error, const QString& message )
{
  #ifndef HIDE_RADIO

    QString dialogTitle = tr( "Last.fm" );

    switch( error )
    {
        // TODO: do we really want a statusbar message for all of these?
        case Request_Aborted:
        case Request_BadResponseCode:
        case Request_HostNotFound:
        case Request_NoResponse:
            statusBar()->showMessage( message );
            break;

        case Handshake_WrongUserNameOrPassword:
            LastMessageBox::critical( dialogTitle, message );
            statusBar()->showMessage( message );

            showSettingsDialog();
            break;

        case Handshake_Banned:
            //TODO make a nicer shutdown (or start upgrade process) via a signal to the container
            LastMessageBox::critical( dialogTitle, message );
            qApp->quit();
            break;

        // Critical box
        case Handshake_SessionFailed:
        case Radio_PluginLoadFailed:
        case Radio_NoSoundcard:
        case Radio_PlaybackError:
        case Radio_UnknownError:
            statusBar()->showMessage( "" );
            LastMessageBox::critical( dialogTitle, message );

            break;

        // Info box
        case Radio_SkipLimitExceeded:
        case ChangeStation_NotEnoughContent:
        case ChangeStation_TooFewGroupMembers:
        case ChangeStation_TooFewFans:
        case ChangeStation_TooFewNeighbours:
        case ChangeStation_Unavailable:
        case ChangeStation_SubscribersOnly:
        case ChangeStation_StreamerOffline:
        case ChangeStation_UnknownError:
        case Radio_BadPlaylist:
        case Radio_IllegalResume:
        case Radio_InvalidUrl:
        case Radio_InvalidAuth:
        case Radio_TrackNotFound:
        case Radio_OutOfPlaylist:
        case Radio_TooManyRetries:
        case Radio_ConnectionRefused:
        case Playlist_RecSysDown:
            QApplication::setOverrideCursor( Qt::ArrowCursor );
            statusBar()->showMessage( "" );
            LastMessageBox::information( dialogTitle, message );
            QApplication::restoreOverrideCursor();
            break;

        default:
            Q_ASSERT( !"Is it correct to leave some unhandled?" );
    }

    LOGL( 2, "Radio error " << error << ": " << message );

  #endif // HIDE_RADIO
}


void
Container::toggleSidebar()
{
    QIcon icon;
    m_sidebarEnabled = !m_sidebarEnabled;

    if (!m_sidebarEnabled) 
    {
        // save before the rest as Qt bug means a hidden splitter-child causes 
        // corrupt saveState() result, don't do if mainWindow isHidden, ie
        // the application is starting up, as the splitter sizes haven't gotten
        // finalised yet and consequently we save corrupted values
        if (isVisible())
            The::settings().setSplitterState( ui.splitter->saveState() );

        icon = IconShack::instance().GetGoodUserIconExpanded( The::currentUser().icon() );

        #ifndef Q_WS_MAC
        centralWidget()->setContentsMargins( 5, 0, 5, 0 ); //aesthetics
        #endif
    }
    else {
        #ifndef Q_WS_MAC
        centralWidget()->setContentsMargins( 0, 0, 5, 0 ); //aesthetics
        #endif
        icon = IconShack::instance().GetGoodUserIconCollapsed( The::currentUser().icon() );
    }

    ui.actionSidebar->setChecked( m_sidebarEnabled );
    m_sidebarActionWithIcon->setChecked( m_sidebarEnabled  );
    m_sidebar->parentWidget()->setVisible( m_sidebarEnabled );
    m_sidebarActionWithIcon->setIcon( icon );
}


void
Container::getPlugin()
{
    ConfigWizard( this, ConfigWizard::Plugin ).exec();
    m_rest_state_widget->updatePlayerNames();
}


void
Container::addMediaDevice( const QString& uid )
{
    ConfigWizard cw( NULL, ConfigWizard::MediaDevice, uid );
    if ( !cw.isWizardRunning() )
        cw.exec();
}


void
Container::checkForUpdates( bool invokedByUser )
{
    bool userAllowed = true;
  #ifdef WIN32
    if ( ( QSysInfo::WindowsVersion & QSysInfo::WV_VISTA ) > 0 )
    {
        // On Vista, we always return true as UAC will request admin privileges
        // to perform the install
        userAllowed = true;
    }
    else
    {
        userAllowed = !CWinUtils::IsLimitedUser();
    }
  #endif

    m_userCheck = invokedByUser;
    if ( m_userCheck )
    {
        if ( userAllowed )
        {
            QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );
        }
        else
        {
            LastMessageBox::information(
                tr( "Update Error" ),
                tr( "An update installs file on your computer, therefore you "
                    "must be logged in as an Administrator to do it. Please "
                    "get an Administrator to run this for you." ),
                QMessageBox::Ok);
        }
    }

    if ( userAllowed )
    {
        m_updater->checkForUpdates();
    }
}


void
Container::updateCheckDone( bool updatesAvailable, bool error, QString errorMsg )
{
    QApplication::restoreOverrideCursor();

    if ( error )
    {
        // Can't connect to the internet
        LOG( 2, "Update check failed. Error: " << errorMsg << "\n" );
        if ( m_userCheck )
        {
            LastMessageBox::critical( tr( "Connection Problem" ),
                tr( "Last.fm couldn't connect to the Internet to check "
                    "for updates.\n\nError: %1" ).arg( errorMsg ) );
        }
        return;
    }

    // No connection error, let's see if we have updates
    if ( !updatesAvailable )
    {
        LOG( 3, "Update check said no updates available.\n" );
        if ( m_userCheck )
        {
            LastMessageBox::information( tr( "Up To Date" ),
                tr( "No updates available. All your software is up to date!\n" ) );
        }
        return;
    }

    // Go ahead and launch update wizard
    LOG( 3, "New updates available. Launching update wizard.\n" );

    UpdateWizard( *m_updater, this ).exec();
}


void
Container::listenerException( QString msg )
{
  #ifndef LASTFM_MULTI_PROCESS_HACK
    // Can't do much else than fire up a dialog box of doom at this stage
    LastMessageBox::critical( tr( "Plugin Listener Error" ), msg );
  #else
    // don't make debugging a pita, clearly this isn't meant for release builds though
    statusMessage( msg );
  #endif
}


void
Container::showFAQ()
{
    new BrowserThread( "http://www.last.fm/help/faq/" );
}


void
Container::showForums()
{
    new BrowserThread( "http://www.last.fm/forum/34905/" );
}


void
Container::inviteAFriend()
{
    QByteArray user = QUrl::toPercentEncoding( The::settings().currentUsername() );

    new BrowserThread( "http://www.last.fm/user/" + user + "/inviteafriend/" );
}



void
Container::about()
{
    AboutDialog( this ).exec();
}


void
Container::initTray()
{
    m_sysTray = new SysTray( this );
    m_sysTray->show();

    // Connect user change to tray
    connect( &The::settings(), SIGNAL(userSettingsChanged( UserSettings& )),
             m_sysTray,          SLOT(setUser( UserSettings& )) );
    connect( this,             SIGNAL(newSong( MetaData )),
             m_sysTray,          SLOT(setTrack( MetaData )) );
}


void
Container::showUserMenu()
{
    // Work out whether we have old items to delete
    QList<QAction*> actions = ui.menuUser->actions();
    if ( actions.size() != 3 )
    {
        // Delete old items
        for ( int i = actions.size() - 1; i >= 3; --i )
        {
            ui.menuUser->removeAction( actions.at( i ) );
        }
    }

    // For each user, add them and put a check next to the current
    QString currentUser = The::settings().currentUsername();
    QStringList users = The::settings().allUsers();
    for ( int i = 0; i < users.size(); ++i )
    {
        QString username = users.at( i );
        QAction* action = new QAction( ui.menuUser );
        action->setData( "user" );
        action->setText( username );
        action->setCheckable( true );
        if ( username == currentUser )
        {
            action->setChecked( true );
        }

        ui.menuUser->addAction( action );
    }
}


void
Container::userSelected( QAction* action )
{
    The::settings().currentUser().setSidebarEnabled( m_sidebarEnabled );

    if ( action->data().toString() == "user" )
    {
        QString username = action->text();
        LOGL( 3, "Switching user to: " << username );

        if (!The::settings().user( username ).rememberPass())
        {
            LoginDialog( this, LoginWidget::LOGIN, username ).exec();
        }
        else
        {
            The::settings().setCurrentUsername( username );
            The::settings().save( true, false );
        }
    }
}


void
Container::addUser()
{
    LoginDialog( this, LoginWidget::ADD_USER ).exec();
}


void
Container::deleteUser()
{
    DeleteUserDialog( this ).exec();
}


void
Container::toggleScrobbling()
{
    The::settings().currentUser().toggleLogToProfile();
}


void
Container::updateWindowTitle( const MetaData& data )
{
    QString title;
    QString track = data.toString();

    if ( track.isEmpty() )
        title += "Last.fm";
    else
        title += track;

    if ( The::settings().allUsers().count() > 1 )
        title += " | " + The::currentUsername();

    if ( qApp->arguments().contains( "--debug" ) )
        title += " [debug]";

    setWindowTitle( title );
}


void
Container::updateUserStuff( UserSettings& user )
{
    Q_DEBUG_BLOCK;

    //FIXME called twice on startup!

    updateWindowTitle( m_metaData );

    QPixmap pix = m_sidebarEnabled
            ? IconShack::instance().GetGoodUserIconCollapsed( user.icon() )
            : IconShack::instance().GetGoodUserIconExpanded( user.icon() );
    m_sidebarActionWithIcon->setIcon( QIcon( pix ) );

    bool const enabled = user.isLogToProfile();

    ui.songTimeBar->setScrobblingEnabled( enabled );
    ui.actionToggleScrobbling->setChecked( enabled );
    ui.actionToggleDiscoveryMode->setChecked( user.isDiscovery() );
    //m_scrobbleAction->setChecked( enabled );
    m_scrobbleLabel->setText( " " + tr("Scrobbling %1").arg( enabled ? tr("on") : tr("off") ) + " " );

    if ( enabled )
        m_scrobbleToggle->setImages( m_scrobbleNormal, m_scrobbleNormal, m_scrobbleNormal, m_scrobbleDisabled );
    else
        m_scrobbleToggle->setImages( m_scrobbleDisabled, m_scrobbleDisabled, m_scrobbleDisabled, m_scrobbleDisabled );

}


void
Container::onRadioBuffering( int size,
                             int total )
{
  #ifndef HIDE_RADIO

    bool finished = size == total;

    if ( finished )
    {
        statusBar()->clearMessage();
    }
    else
    {
        int percent = (int)(((float)size / total) * 100);
        statusBar()->showMessage( tr( "Buffering... (%1%)", "%1 is the percentage buffering is complete" ).arg( percent ) );
    }

  #endif // HIDE_RADIO
}


void
Container::setNewSong( TrackInfo data, bool songStarted )
{
    // The assumption that a new track has just started can't be made here.
    // It could just be that the radio was stopped and an underlying local
    // track became visible.

    m_metaData = data;

    // Reset GUI
    ui.songTimeBar->clear();
    ui.songTimeBar->setEnabled( false );
    ui.songTimeBar->setClockText( "" );

    ui.actionTag->setEnabled( false );
    ui.actionRecommend->setEnabled( false );
    ui.actionLove->setEnabled( false );

    if ( !songStarted )
    {
        // Playback stopped
        m_stopped = true;
        emit newSong( m_metaData ); //empty TrackInfo object

        // we use a timer as some players say they stop before starting a new
        // track, which results in us flickering between the two widgets, and 
        // that looks shite
        if (ui.stack->currentIndex() != 0)
            m_showRestStateTimer->start();
    }
    else if ( m_metaData.scrobblableStatus() != TrackInfo::OkToScrobble &&
              m_metaData.source() != TrackInfo::Radio )
    {
        // Invalid track, all we do is update the song bar
        displayUnscrobblableStatus( m_metaData.scrobblableStatus() );
        m_metaDataExtension->clear();
    }
    else
    {
        // Playback started

        if (The::currentUser().isLogToProfile())
            The::scrobbler().nowPlaying( data );

        // The reason for these extra checks is that one user had
        // crashes here (bug #56). This could happen if the player listener
        // receives a START, emits a queued signal, then immediately
        // receives a STOP. If the first queued signal doesn't get here
        // until after the STOP, GetActivePlayer will return NULL.
        CPlayerConnection* plyr = m_listener->GetActivePlayer();
        if ( plyr == NULL )
        {
            LOG( 1, "Caught a NULL player in setNewSong, not setting stopwatch" );
        }
        else
        {
            StopWatch& watch = plyr->GetStopWatch();
            ui.songTimeBar->setStopWatch( &watch );
            ui.songTimeBar->setReverse( true );
            //ui.songTimeBar->setMaximum( data.scrobbleTime() );
            if ( plyr->IsScrobbled() )
            {
                setTrackScrobbled();
            }
            else
            {
                ui.songTimeBar->setClockText( "" );
            }
        }

        ui.actionTag->setEnabled( true );
        ui.actionRecommend->setEnabled( true );
        ui.actionLove->setEnabled( true );

        QHash<QString, QString> itemdata;
        itemdata.insert( "artist", data.artist() );
        itemdata.insert( "track", data.track() );
        ui.songTimeBar->setItemData( itemdata );

        // Only display scrobble timer if we're logging to profile and the track
        // is scrobblable.
        bool showScrobbleTimer = The::settings().currentUser().isLogToProfile()
                         && m_metaData.scrobblableStatus() == TrackInfo::OkToScrobble;
        ui.songTimeBar->setProgressEnabled( showScrobbleTimer );
        ui.songTimeBar->setClockEnabled( true );
        ui.songTimeBar->setMode( m_metaData.source() );

        if ( m_stopped )
        {
            m_stopped = false;
            emit playStarted();

            ui.stack->setCurrentIndex( 1 );
            m_showRestStateTimer->stop();
        }

        // Clear any status messages on the start of a new track or it will look odd
        if ( data.source() != TrackInfo::Radio )
        {
            statusBar()->clearMessage();
        }

        emit newSong( m_metaData );
    }

    updateWindowTitle( m_metaData );
}


void
Container::displayUnscrobblableStatus( TrackInfo::ScrobblableStatus status )
{
    Q_ASSERT( status != TrackInfo::OkToScrobble );

    QString text = tr( "Can't scrobble: %1" );

    switch ( status )
    {
        case TrackInfo::NoTimeStamp:
        {
            text = text.arg( tr( "missing start time" ) );
        }
        break;

        case TrackInfo::TooShort:
        {
            text = text.arg( tr( "track too short" ) );
        }
        break;

        case TrackInfo::ArtistNameMissing:
        case TrackInfo::TrackNameMissing:
        {
            text = text.arg( tr( "artist or title missing from ID3 tag" ) );
        }
        break;

        case TrackInfo::ExcludedDir:
        {
            text = tr( "Won't scrobble: track is in directory set to not scrobble" );
        }
        break;

        case TrackInfo::ArtistInvalid:
        {
            text = text.arg( tr( "invalid artist name" ) );
        }
        break;

        default:
            Q_ASSERT( !"Shouldn't happen" );
    }

    ui.songTimeBar->setText( text );
}


void
Container::onScrobblerStatusChange( int code, const QVariant& data )
{
    switch (code)
    {
        case Scrobbler::ErrorBadAuthorisation:
            //TODO this can prolly be removed, but is untested and NOW is bug fix branch
            m_sysTray->setUser( The::currentUser() );

            //NOTE radio is likely to fail auth too, so don't show 2 dialogs
            statusBar()->showMessage( tr("Your username and password are incorrect") );
            break;

        case Scrobbler::Connecting:
            statusBar()->showMessage( tr("Connecting to Last.fm...") );
            break;

        case Scrobbler::Handshaken:
            //FIXME we have to to do this due to BadAuthorisation state
            //TODO see above, prolly can be removed?
            m_sysTray->setUser( The::currentUser() );
            break;

        case Scrobbler::Scrobbling:
            statusBar()->showMessage( tr( "Scrobbling %n track(s)...", "", data.toInt() ) );
            break;

        case Scrobbler::TracksScrobbled:
            statusBar()->showMessage( tr( "%n track(s) scrobbled", "", data.toInt() ) );
            break;

        case Scrobbler::TracksNotScrobbled:
            statusBar()->showMessage( tr( "%n track(s) will be scrobbled later", "", data.toInt() ) );
            break;

        case Scrobbler::ErrorBannedClient:
            LastMessageBox::critical( tr("Error"), tr( "This software is too old, please upgrade." ) );
            break;

        case Scrobbler::ErrorBadTime:
            LastMessageBox::critical( tr("Error"), tr( "Your computer's clock is inaccurate. Please correct it, or Last.fm can not authorise any Scrobbling." ) );
            break;
    }
}


void
Container::onTrackScrobbled( const TrackInfo& track )
{
    Q_DEBUG_BLOCK;

    //TODO move into sidebar itself

    setTrackScrobbled();

    Track t;
    t.setArtist( track.artist() );
    t.setTitle( track.track() );

    m_sidebar->addRecentlyPlayedTrack( t );
}


void
Container::setTrackScrobbled()
{
    ui.songTimeBar->pushClockText( tr( "scrobbled" ), 5 );
}


void
Container::showTagDialog( int defaultTagType )
{
    TagDialog d( defaultTagType, this );
    d.setSong( m_metaData );
    d.exec();
}


void
Container::showTagDialogMD()
{
    showTagDialog( 0 );
}


void
Container::play()
{
    m_radio->resumeStation();
}


void
Container::stop()
{
    m_radio->stop();
}


void
Container::volumeUp()
{
    if ( ui_playcontrols.volume->value() != 100 )
    {
        if ( ui_playcontrols.volume->value() + 5 > 100 )
            ui_playcontrols.volume->setValue( 100 );
        else
            ui_playcontrols.volume->setValue( ui_playcontrols.volume->value() + 5 );
    }
}


void
Container::volumeDown()
{
    if ( ui_playcontrols.volume->value() != 0 )
    {
        if ( ui_playcontrols.volume->value() - 5 < 0 )
            ui_playcontrols.volume->setValue( 0 );
        else
            ui_playcontrols.volume->setValue( ui_playcontrols.volume->value() - 5 );
    }
}


void
Container::mute()
{
    if ( ui_playcontrols.volume->value() != 0 )
    {
        m_lastVolume = ui_playcontrols.volume->value();
        ui_playcontrols.volume->setValue( 0 );
    }
    else
        ui_playcontrols.volume->setValue( m_lastVolume );
}


void
Container::love()
{
    // Make copy in case dialog stays on screen across track change
    MetaData track = m_metaData;

    QString confirmMsg = QString( tr( "Are you sure you want to add %1 to your loved tracks?" ) )
        .arg( track.track() );
    ConfirmDialog dlg( ConfirmDialog::Love, confirmMsg, this );

    if ( dlg.exec() == QDialog::Accepted )
    {
        ui.actionLove->setEnabled( false );

        (new LoveRequest( track ))->start();
    }
}


void
Container::ban()
{
    // Make copy in case dialog stays on screen across track change
    MetaData track = m_metaData;

    QString confirmMsg = QString( tr( "Are you sure you want to ban %1?" ) )
        .arg( track.track() );
    ConfirmDialog dlg( ConfirmDialog::Ban, confirmMsg, this );

    if ( dlg.exec() == QDialog::Accepted )
    {
        ui.actionSkip->setEnabled( false );
        ui.actionLove->setEnabled( false );
        ui.actionBan->setEnabled( false );

        (new BanRequest( track ))->start();

        // Listener needs to know about banned tracks so that we can prevent
        // them from getting submitted.
        CPlayerConnection* player = m_listener->GetActivePlayer();
        if ( player != NULL )
        {
            player->ban();
        }

        m_radio->skip();
    }
}


void
Container::skip()
{
    ui.actionSkip->setEnabled( false );
    ui.actionLove->setEnabled( false );
    ui.actionBan->setEnabled( false );

    ui.songTimeBar->clear();
    ui.songTimeBar->setEnabled( false );
    ui.songTimeBar->setClockText( "" );
    ui.songTimeBar->setText( tr( "Skipping..." ) );

    m_radio->skip();
}


void
Container::switchPlayButton( bool enabled )
{
    if ( enabled )
    {
        ui.actionStop->setVisible( false );
        ui.actionPlay->setVisible( true );

        ui.actionPlay->setShortcut( Qt::Key_Space );
        ui.actionStop->setShortcut( QKeySequence() );
    }
    else
    {
        ui.actionPlay->setVisible( false );
        ui.actionStop->setVisible( true );

        ui.actionPlay->setShortcut( QKeySequence() );
        ui.actionStop->setShortcut( Qt::Key_Space );
    }

    #ifdef Q_WS_MAC
        m_sysTrayActionPlay->setVisible( ui.actionPlay->isVisible() );
        m_sysTrayActionStop->setVisible( ui.actionStop->isVisible() );
    #endif
}


void
Container::onRadioStateChange( RadioState newState )
{
  #ifndef HIDE_RADIO

    switch ( newState )
    {
        case State_Uninitialised:
        {
            // Nothing
        }
        break;

        case State_Handshaking:
        {
            statusBar()->showMessage( tr( "Contacting radio service..." ) );
        }
        break;

        case State_Handshaken:
        {
            LOGL( 3, "Radio streamer handshake successful." );
            statusBar()->showMessage( tr( "Radio service initialised" ) );

            // Play is disabled in ctor so enable it when handshake is complete
            // to indicate we can now start radio.
            switchPlayButton( true );
            ui.actionPlay->setEnabled( true );
            m_rest_state_widget->setPlayEnabled( true );

            // do we have to load a station since we got started with a cli-arg?
            m_iControl->setHandshaked();
            if ( m_preloadStation.contains( "lastfm://" ) )
            {
                m_radio->playStation( m_preloadStation );
                m_preloadStation.clear();
            }
            else if ( The::settings().currentUser().resumePlayback() &&
                     !The::settings().currentUser().resumeStation().isEmpty() )
            {
                m_radio->playStation( The::settings().currentUser().resumeStation() );
            }

            #ifndef Q_WS_X11
            if ( !The::settings().isFirstRun() )
                // Don't do this until here as we won't have received a base host
                // earlier.
                checkForUpdates( false );
            #endif
        }
        break;

        case State_ChangingStation:
        {
            statusBar()->showMessage( tr("Starting station %1...").arg( m_radio->stationUrl() ) );

            showMetaDataWidget();
            m_metaDataExtension->displayTuningIn();

            if ( m_radio->stationUrl().isPlaylist() )
                ui.stationTimeBar->setText( tr( "Connecting to playlist..." ) );
            else
                ui.stationTimeBar->setText( tr( "Starting station..." ) );

            ui.stationTimeBar->setClockText( "" );
            ui.stationTimeBar->setEnabled( true );
            ui.stationTimeBar->setVisible( true );

            switchPlayButton( false );
            ui.actionStop->setEnabled( true );
        }
        break;

        case State_FetchingPlaylist:
        {
            // Need to apply this again in case we ran out of playlist
            switchPlayButton( false );
            ui.actionStop->setEnabled( true );

            showMetaDataWidget();
            m_metaDataExtension->displayTuningIn();

            statusBar()->showMessage( tr( "Retrieving playlist..." ) );
        }
        break;

        case State_FetchingStream:
        {
            statusBar()->showMessage( tr( "Retrieving stream..." ) );

            // We now have a station name. Bit ugly this, maybe change to a signal?
            QString stationText = tr( "Station: %1" ).arg( m_radio->stationName() );
            ui.stationTimeBar->setText( stationText );

            // Need to apply this again in case we resumed
            switchPlayButton( false );
            ui.actionStop->setEnabled( true );

            ui.stationTimeBar->setEnabled( true );
            ui.stationTimeBar->setVisible( true );
        }
        break;

        case State_StreamFetched:
        {
            // Resuming not possible until we've fetched a stream for the first time
            // TODO: this is actually bollocks because the stream could have been a preview
            ui.actionPlay->setEnabled( true );
            m_rest_state_widget->setPlayEnabled( true );
        }
        break;

        case State_Buffering:
        {
            // Status bar message handled in onRadioBuffering
        }
        break;

        case State_Streaming:
        {
            ui.actionSkip->setEnabled( true );
            ui.actionBan->setEnabled( true );
        }
        break;

        case State_Skipping:
        {
            ui.actionSkip->setEnabled( false );
            ui.actionBan->setEnabled( false );
        }
        break;

        case State_Stopping:
        {
            ui.stationTimeBar->clear();
            ui.stationTimeBar->setEnabled( false );
            ui.stationTimeBar->setVisible( false );

            ui.actionStop->setEnabled( false );
            ui.actionSkip->setEnabled( false );
            ui.actionBan->setEnabled( false );

            ui.stack->setCurrentIndex( 0 );
        }
        break;

        case State_Stopped:
        {
            switchPlayButton( true );
            m_stopped = true;
            statusBar()->showMessage( "" );
        }
        break;

        default:
            Q_ASSERT( !"Undefined state case reached in onRadioStateChanged!" );
    }

  #endif // HIDE_RADIO
}


void
Container::gotoProfile()
{
    QByteArray user = QUrl::toPercentEncoding( The::settings().currentUsername() );

    // This will redirect to the language version the user last used.
    QString dashUrl = "http://www.last.fm/user/" + user;

    new BrowserThread( dashUrl );
}


void
Container::crash()
{
    int n = doCrash();
    Q_UNUSED( n );
}


// This will get optimised out unless we put it in a function that returns
// something.
int
Container::doCrash()
{
    QString* s = NULL;
    return s->length();
}


void
Container::showRecommendDialog()
{
    m_recommendDlg->setSong( m_metaData );
    m_recommendDlg->exec();
}


void
Container::showSettingsDialog()
{
    m_settingsDialog->exec();
}


#ifdef Q_WS_X11
    // includes only relevent to this function - please leave here :)
    #include <QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
#endif

void
Container::toggleWindowVisibility() 
{
    //TODO really we should check to see if the window itself is obscured?
    // hard to say as exact desire of user is a little hard to predict.
    // certainly we should raise the window if it isn't active as chances are it
    // is behind other windows

    if (isVisible() && isActiveWindow())
        hide();
    else {
        #ifndef Q_WS_X11
            showNormal(), activateWindow(), raise();
        #else
            showNormal();

            //NOTE don't raise, as this won't work with focus stealing prevention
            //raise();

            QX11Info const i;
            Atom const _NET_ACTIVE_WINDOW = XInternAtom( i.display(), "_NET_ACTIVE_WINDOW", False);

            // this sends the correct demand for window activation to the Window 
            // manager. Thus forcing window activation.
            ///@see http://standards.freedesktop.org/wm-spec/wm-spec-1.3.html#id2506353
            XEvent e;
            e.xclient.type = ClientMessage;
            e.xclient.message_type = _NET_ACTIVE_WINDOW;
            e.xclient.display = i.display();
            e.xclient.window = winId();
            e.xclient.format = 32;
            e.xclient.data.l[0] = 1; // we are a normal application
            e.xclient.data.l[1] = i.appUserTime();
            e.xclient.data.l[2] = qApp->activeWindow() ? qApp->activeWindow()->winId() : 0;
            e.xclient.data.l[3] = 0l;
            e.xclient.data.l[4] = 0l;

            // we send to the root window per fdo NET spec
            XSendEvent( i.display(), i.appRootWindow(), false, SubstructureRedirectMask | SubstructureNotifyMask, &e );
        #endif
    }
}


//TODO mxcl, would be better to have this handled by the Requests themselves in some generic error handler

void
Container::webServiceSuccess( Request *r )
{
    switch (r->type()) 
    {
        case TypeSetTag: 
            statusBar()->showMessage( tr( "You tagged %1 successfully" ).arg( static_cast<SetTagRequest*>(r)->title() ) );
            break;

        case TypeRecommend:
            statusBar()->showMessage( tr( "Recommendation sent." ) );
            break;

        case TypeLove:
            statusBar()->showMessage( tr( "You added this track to your Loved tracks." ) );
            break;

        case TypeUnLove:
            statusBar()->showMessage( tr( "You removed this track from your Loved tracks." ) );
            break;

        case TypeBan:
            statusBar()->showMessage( tr( "You banned this track." ) );
            break;

        case TypeUnBan:
            statusBar()->showMessage( tr( "You unbanned this track." ) );
            break;

        default:
            break;
    }
}


void
Container::webServiceFailure( Request *r )
{
    switch (r->type())
    {
        case TypeSkip:
            LastMessageBox::information( tr( "Sorry" ), tr( "We couldn't skip this track." ) );
            break;

        case TypeSetTag:
            statusBar()->showMessage( tr( "Tagging %1 failed" ).arg( static_cast<SetTagRequest*>(r)->title() ) );
            break;

        default:
            break;
    }
}


void
Container::onRestStateTimerTimeout()
{
    if (!m_metaDataExtension->isTuningIn())
        showRestState();
}


void
Container::showRestState()
{
    ui.stack->setCurrentIndex( 0 );
}


void
Container::onAltShiftL()
{
    //FIXME get path from Logger instance, DRY!
    QDesktopServices::openUrl( QUrl::fromLocalFile( savePath( "container.log" ) ) );
}


void
Container::onAltShiftH()
{
    //FIXME get path from Helper, DRY!
    QDesktopServices::openUrl( QUrl::fromLocalFile( savePath( "lastfmhelper.log" ) ) );
}


void
Container::onAltShiftF()
{
    QDesktopServices::openUrl( QUrl::fromLocalFile( savePath( "" ) ) );
}


bool
Container::isTuningInOrPlaying() const //TODO REMOVE
{
    if (!m_metaData.isEmpty())
        return true;

    if (!m_radio)
        return false;

    switch (m_radio->state())
    {
        case State_Uninitialised:
        case State_Handshaking:
        case State_Handshaken:
        case State_Stopping:
        case State_Stopped:
            return false;
        
        case State_ChangingStation:
        case State_FetchingPlaylist:
        case State_FetchingStream:
        case State_StreamFetched:
        case State_Buffering:
        case State_Streaming:
        case State_Skipping:
            return true;
    }
}


namespace The
{
    /// these exist so you don't have to include container.h and radio.h to use
    /// the radio, etc. only radio.h

    Radio &radio() { return Container::instance().radio(); }
    ScrobblerManager& scrobbler() { return Container::instance().scrobbler(); }
    RecommendDialog& recommendDialog() { return Container::instance().recommendDialog(); }
}

#undef CAST
