/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include <QtGui>

#include "skypenotifyextension.h"
#include "containerutils.h"
#include "logger.h"
#include "Settings.h"

#define SKYPE_SEND_URL (0)

static const QString kSetMoodMsgCmd = "SET PROFILE MOOD_TEXT ";
static const QString kGetMoodMsgCmd = "GET PROFILE MOOD_TEXT";

// Note about saving of mood messages
//
// We're trying best we can to store the user's previous mood message before
// we start sending notifications through so that we can restore it when
// the user disables our plugin. This involves reading it at startup and
// also watching for manual modifications of it while we're active.
// There is however one sequence of events where a user's mood message might
// get lost but it's an edge case and it would require more effort than
// it's worth to fix. It goes a little something like:
//     Enable plugin in Options
//     Quit app
//     Change mood msg in Skype
//     Launch Last.fm and disable plugin straight away
//     Whatever old mood message we had stored would be restored

SkypeNotifyExtension::SkypeNotifyExtension() :
    m_parent( NULL ),
    m_settingsPanel( NULL ),
    m_enabledByDefault( false ),
    m_defaultFormat( tr( "Currently listening to '%2' by %1" ) ),
    m_haveWaiting( false )
{
    LOG( 3, "Initialising Skype Extension\n" );
}


QString
SkypeNotifyExtension::name() const
{
    return QString( "Skype Notifier Extension" );
}


QString
SkypeNotifyExtension::version() const
{
    return QString( "1.0.0.0" );
}

QWidget*
SkypeNotifyExtension::settingsPane()
{
    if ( !m_settingsPanel )
    {
        initSettingsPanel();
    }

    return m_settingsPanel;
}

void
SkypeNotifyExtension::initSettingsPanel()
{
    LOG( 3, "Initialising Skype GUI\n" );

    m_settingsPanel = new QWidget( m_parent );
    ui.setupUi( m_settingsPanel );

    // Loaded from resources to avoid having to ship a separate file
    bool t = m_icon.load( ":/options_skype.png" );

    connect( ui.enabledCheck,     SIGNAL( stateChanged( int ) ),
        this,                SIGNAL( settingsChanged() ) );
    connect( ui.nowPlayingFormat, SIGNAL( textChanged( QString ) ),
        this,                SIGNAL( settingsChanged() ) );
}

QPixmap*
SkypeNotifyExtension::settingsIcon()
{
    Q_ASSERT( !m_icon.isNull() );

    return &m_icon;
}

void
SkypeNotifyExtension::populateSettings()
{
    // This won't get called until the options pane is displayed for 
    // the first time.

    // The client is initialised so that the default QSettings ctor will
    // create an object with the right path.
    CurrentUserSettings config;
    config.beginGroup( name() );

    ui.enabledCheck->setChecked(
        config.value( "Enabled", m_enabledByDefault ).toBool() );
    ui.nowPlayingFormat->setText(
        config.value( "Format", m_defaultFormat ).toString() );
    config.endGroup();
}

void
SkypeNotifyExtension::saveSettings()
{
    CurrentUserSettings config;
    config.beginGroup( name() );

    bool switchedOff =
        !ui.enabledCheck->isChecked() &&
        config.value( "Enabled", m_enabledByDefault ).toBool();
    bool switchedOn =
        ui.enabledCheck->isChecked() &&
        !config.value( "Enabled", m_enabledByDefault ).toBool();

    config.setValue( "Enabled", ui.enabledCheck->isChecked() );
    config.setValue( "Format", ui.nowPlayingFormat->text() );

    if ( switchedOff )
    {
        QString oldMsg = config.value( "SavedMoodMessage", "" ).toString();
        QString fullCmd = kSetMoodMsgCmd + oldMsg;

        LOGL( 3, "Disabled Skype plugin, restoring old mood message: " << oldMsg );

        m_lastNotification = oldMsg;
        sendToSkype( fullCmd.toUtf8() );
    }

    // Must call endGroup before we do the notify below, otherwise the newly
    // enabled state won't be persisted in time.
    config.endGroup();

    if ( switchedOn )
    {
        LOGL( 3, "Enabled Skype plugin, sending current track");

        notify( m_cachedMetadata );
    }
}

void
SkypeNotifyExtension::notify( MetaData metaData )
{
    // Always store this so that we can send it immediately if we get switched
    // from disabled to enabled during a track.
    m_cachedMetadata = metaData;

    if ( !isEnabled() )
    {
        return;
    }
    
    if ( !isSkypeAttached() )
    { 
        // This is asynchronous so we flag that we have waiting metadata
        // for transmission once we're connected.
        initSkypeCommunication();
        m_haveWaiting = true;
        return;
    }

    QString cmd = kSetMoodMsgCmd;
    
    QByteArray utfToSend;
    if ( metaData.isEmpty() )
    {
        utfToSend = cmd.toUtf8();
    }
    else
    {
        QString format = nowPlayingFormat();
        
        QString nowPlaying = cmd + format
            .replace( QString("%1"), metaData.artist() )
            .replace( QString("%2"), metaData.track() )
            .replace( QString("%3"), metaData.album() );

        #if SKYPE_SEND_URL
            QString url = metaData.trackPageUrl();
            if ( !url.isEmpty() )
            {
                if ( url.startsWith( "http://" ) ) { url.remove( 0, 7 ); }
                if ( url.endsWith( "/" ) ) { url.chop( 1 ); }
                url.append( "/?s" ); // identifies link as coming from Skype

                nowPlaying += " " + url;
            }
        #endif // SKYPE_SEND_URL                
        
        utfToSend = nowPlaying.toUtf8();
    }

    sendToSkype( utfToSend );

    LOGL( 4, "Sending to Skype: " << QString::fromUtf8( utfToSend ) );

    m_haveWaiting = false;
}

void
SkypeNotifyExtension::notifyWaiting()
{
    // This is always called by the subclass after a successful connect.

    // First, query for old mood message. As soon as we get the reply,
    // we'll send the waiting notification.
    QByteArray utfToSend = kGetMoodMsgCmd.toUtf8();
    sendToSkype( utfToSend );
}

void
SkypeNotifyExtension::receiveFromSkype( QByteArray utfResp )
{
    QString resp = QString::fromUtf8( utfResp );

    LOGL( 4, "Message from Skype: " << resp );

    // Parse response here and look for confirmation of mood message changes
    QString moodChangedCmd( "PROFILE MOOD_TEXT" );
    int cmdLen = moodChangedCmd.size();
    if ( resp.startsWith( moodChangedCmd ) )
    {
        QString msg = resp.right( resp.size() - ( cmdLen + 1 ) );
        if ( !msg.isEmpty() )
        {
            // Compare and store if different to last one sent
            if ( msg != m_lastNotification )
            {
                LOGL( 3, "Skype old mood message found, saving: " << msg );

                // Save out user's old message to config
                CurrentUserSettings config;
                config.beginGroup( name() );
                config.setValue( "SavedMoodMessage", msg );
                config.endGroup();                
            }
        }

        if ( m_haveWaiting )
        {
            // Now that we've stored any existing mood msg, we can send the
            // waiting notification
            LOGL( 3, "Sending waiting notification to Skype" );
            notify( m_cachedMetadata );
        }
    }
    
}
    
bool
SkypeNotifyExtension::isEnabled()
{
    CurrentUserSettings config;
    config.beginGroup( name() );
    bool enabled = config.value( "Enabled", m_enabledByDefault ).toBool();
    config.endGroup();
    
    return enabled;
}

QString
SkypeNotifyExtension::nowPlayingFormat()
{
    CurrentUserSettings config;
    config.beginGroup( name() );
    QString format = config.value( "Format", m_defaultFormat ).toString();
    config.endGroup();
    
    return format;
}


