package latexDraw.ui;

import java.awt.*;
import java.awt.event.*;
import java.awt.print.PrinterJob;
import java.io.*;
import java.net.URL;
import java.util.Vector;

import javax.print.*;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;

import latexDraw.figures.*;
import latexDraw.figures.properties.Arrowable;
import latexDraw.filters.*;
import latexDraw.generators.svg.SVGDocumentGenerator;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.parsers.pstricks.PSTricksParser;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.components.*;
import latexDraw.ui.components.progressbars.PSTProgressBarManager;
import latexDraw.ui.components.progressbars.SVGProgressBarManager;
import latexDraw.ui.dialog.*;
import latexDraw.ui.listeners.MenusListener;
import latexDraw.ui.listeners.RecentFilesListener;
import latexDraw.ui.listeners.ShortcutsListener;
import latexDraw.ui.listeners.ToolbarListener;
import latexDraw.util.*;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/** 
 * This class contains all the elements of the graphic interface.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2009 Arnaud BLOUIN<br>
 * <br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 * <br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 * <br>
 * 07/01/09<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.3<br>
 */
public final class LaTeXDrawFrame extends JFrame implements ActionListener, ItemListener, ChangeListener, WindowStateListener
{
	private static final long serialVersionUID = 1L;
	
	/** The version of the application */
	public final static String VERSION   = "2.0.5";//$NON-NLS-1$
	
	public final static String VERSION_STABILITY = ""; //$NON-NLS-1$
	
	/** To change if update is needed or not. */
	public static final boolean WITH_UPDATE = true;
	
	public static final Insets INSET_BUTTON = new Insets(1,1,1,1);
	
	/** The path by default */
	public static final String DEFAULT_PATH = System.getProperty("user.home");//$NON-NLS-1$
	
	protected ShortcutsFrame shortcutsFrame;
	
	protected PSTProgressBarManager progressbarPST = null;
	
	protected SVGProgressBarManager progressbarSVG = null;
	
	protected transient MenusListener menusListener;
	
	protected transient RecentFilesListener recentFilesListener;
	
	private String latexIncludes = null;
	
	private String pathDistribLatex = null;
	
	/** The path of the location of the LaTeX editor. */
	private String pathTexEditor = null;
	
	/** This frame allows the user to insert PSTricks code in the drawing. */
	private InsertPSTricksCodeFrame insertCodeFrame;
	
	/** The panel containing the generated code*/
	protected CodePanel codePanel;
	
	/** The panel containing the drawing made by the user*/
	protected DrawPanel drawPanel;
	
	/** The toolbar of the application*/
	protected LToolbar toolbar;
	
	/** The progress bar for the parsing */
	protected JProgressBar progressBar;
	
	/** The button stop for the parsing */
	protected JButton stopButton;
	
	/** The menu bar of the application */
	protected LMenuBar menuBar;
	
	/** The identifier of the build */
	private static final String ID_BUILD = "20091002";//$NON-NLS-1$
	
	/** Allows to realize undo and redo */
	protected transient UndoRedoManager undoManager;
	
	/** This frame allows to set the parameters of a line */
	private ParametersLineFrame paramLineFrame;
	
	/** This frame allows to set the parameters of axes. */
	private ParametersAxeFrame paramAxesFrame;
	
	/** This frame allows to set the parameters of a circle */
	private ParametersCircleSquareFrame paramCircleFrame;
	
	/** This frame allows to set the parameters of an ellipse */
	private ParametersEllipseRectangleFrame paramEllipseFrame;
	
	/** This frame allows to set the parameters of a Bézier curve */
	private ParametersBezierCurveFrame paramBezierCurveFrame;
	
	/** This frame allows to set the parameters of akin points */
	private ParametersAkinPointsFrame paramAkinPointsFrame;
	
	/** This frame allows to set the parameters of a dot */
	private ParametersDotFrame paramDotFrame;
	
	/** This frame allows to set the parameters of a text */
	private ParametersTextFrame paramTextFrame;
	
	/** This frame allows to set the parameters of a triangle */
	private ParametersTriangleFrame paramTriangleFrame;
	
//	/** This frame allows to set the parameters of a drawing */
//	private ParametersDrawFrame paramDrawFrame;
	
	/** This frame allows to set the parameters of a grid */
	private ParametersGridFrame paramGridFrame;

	/** This frame allows to set the parameters of a rhombus */
	private ParametersRhombusFrame paramRhombusFrame;
	
	/** This frame allows to set the parameters of an arc */
	private ParametersArcFrame paramArcFrame;
	
	/** This frame allows to set the parameters of a polygon */
	private ParametersPolygonJoinedLinesFrame paramPolygonFrame;
	
	/** This frame allows to set the parameters of several joined lines */
	private ParametersPolygonJoinedLinesFrame paramJoinedLinesFrame;
	
	/** This frame allows the user to add comments to his drawing */
	private AddCommentsFrame addCommentsFrame;	
	
	/** This frame allows to set the parameters of the drawing (not the
	 * Figure.Draw */
	private DrawPropertiesFrame drawPropertiesFrame;
	
	/** Allows to display a frame displaying the LaTeXDraw help */
	private JFrame helpFrame;
	
	/** The current file of the project */
	private File currentFile;	

	/** The path used on open/save actions */
	private static String pathOpen = DEFAULT_PATH;
	
	/** The path used on exportation actions */
	private static String pathExport = DEFAULT_PATH;
	
	/** The frame which allows to convert pictures */
	public final static BatchConvertFrame convertFrame;
	
	/** Allows to know if the program must check new version */
	private boolean checkNewVersion;
	
    /** The About LaTeXDraw box */
    protected JFrame aboutTeXDraw;
    
    /** This frame allows the user to set his preferences */
    private PreferencesFrame preferencesFrame;
    
    /** Correspond to the current choice of the user (draw a line, ...). */
    private static String currentChoice;
    
    /** Corresponds to the name of the current project */
    private String projectName;
    
    /** The fileChooser used to save projects. */
    private JFileChooser fileChooserSave;

    /** The fileChooser used to open projects. @since 2.0.0 */
    private JFileChooser fileChooserOpen;
    
    /** The PictureChooser for 'insert picture' actions */
    private PictureChooser fileChooserPicture;
    
    /** The fileChooser for save/open actions */
    private ExportDialog fileChooserExport;
    
    /** The PictureChooser for import actions */
    private PictureChooser fileChooserImport;
    
	/** Allows to know if by default the grid must be displayed */
	public static final boolean DEFAULT_DISPLAY_GRID = true;
	
	/** Allows to know if by default the XScale must be displayed */
	public static final boolean DEFAULT_DISPLAY_XSCALE = true;
	
	/** Allows to know if by default the YScaleC must be displayed */
	public static final boolean DEFAULT_DISPLAY_YSCALE = true;
	
	/** Allows to know if by default the codePane must be displayed */
	public static final boolean DEFAULT_DISPLAY_CODEPANEL = true;
	
	/** Allows to know if by default the borders of the drawing  must be displayed */
	public static final boolean DEFAULT_DISPLAY_BORDERS = false;
	
	/** Allows to know if by default the program must check new version on start-up */
	public static final boolean DEFAULT_CHECK_VERSION = true;
	
	/** The splitpane of the program */
	protected JSplitPane splitPane;
	
	/** The field which allows to change the thickness of figures */
	private JSpinner thicknessField;
	
	/** Allows to set the colour of the borders of a figure */
	private ColorButton buttonBordersColor;
	
	/** Allows the attribute "filled" in Figure class */
	private JCheckBox filledCheckBox;
	
	/** Set/unset a shadow to a figure. */
	private JCheckBox shadowCheckBox;
	
	/** Allows have double boundaries on a figure */
	private JCheckBox dbleBoundCheckBox;
	
	/** Allows to set the colour of the interior of a figure */
	private ColorButton buttonInteriorColor;
	
	/** Allows to set the colour of the shadow of a figure */
	private ColorButton buttonShadowColor;
	
	/** Allows to change the colour of the hatch */
	private ColorButton buttonHatchColor;
	
	/** Contains buttons allowing to rotate figures. */
	private ListJToggleButton rotationButtons;
	
	/** Contains the buttons allowing to return figures (mirrors). */
	private ListJToggleButton mirrorButtons;
	
	private ListJToggleButton alignButtons;
	
	private ListJToggleButton distribButtons;
	
	/** Allows to select the type of the dot */
	private LaTeXDrawComboBox dotChoice;
	
	/** Allows to select the position of the borders of the figure (if possible) */
	private LaTeXDrawComboBox bordersPositionChoice;
	
	/** Allows to change the colour of the space between the double boundaries */
	private ColorButton buttonDbleBoundColor;
	
	/** This button allows to join several figures */
	private JButton joinButton;
	
	/** This button allows to separate several figures */
	private JButton separateButton;
	
	/** Allows to change the style of the left-end of the line */
	private LaTeXDrawComboBox lineArrowLChoice;
	
	/** Allows to change the style of the right-end of the line */
	private LaTeXDrawComboBox lineArrowRChoice;
	
	/** Allows to change the style of the line */
	private LaTeXDrawComboBox lineChoice;
	
	/** Allows to change the style of hatch */
	private LaTeXDrawComboBox hatchChoice;
	
	/** Allows to choose the first colour of a gradient. */
	private ColorButton buttonGradientStartC;
	
	/** Allows to choose the second colour of a gradient. */
	private ColorButton buttonGradientEndC;
	
	/** The former location of the divider location (used to show or hide the codepanel) */
	private int formerDividerLocation;
	
	private ListJToggleButton locationButtons;
	
	/** The version of the LaTeXDraw file loaded */
	private static String versionOfFile = "";//$NON-NLS-1$
	
	/** The X-scale of the drawing */
	private XScale xScale;
	
	/** The Y-scale of the drawing */	
	private YScale yScale;

	/** This label is behind the field thickness in the paramsToolbar */
	private JLabel labelThickness;
	
	/** The older version of java that this program can used */
	public static final String  VERSION_MIN = "1.5"; //$NON-NLS-1$
	
	/** This attribute is used to save the value of the "updateCode" checkBox
	 * of the code panel when we hide to code panel (we disactivate the auto-update when to code panel is hidden */
	private boolean formerIsInAutoUpdate;

	/** The status-bar of the frame */
	private JTextField statusBar;

	/** The look and feel of the program. */
	private static String lookAndFeel;

	protected transient ShortcutsListener shortcutsListener;
	
	protected JToggleButton rotateButton;
	
	/* Save of the general parameters. */
	protected ColorButton genGradientEndCB;
	protected ColorButton genGradientStartCB;
	protected ColorButton genLineCB;
	protected ColorButton genInteriorCB;
	protected ColorButton genDbleBordCB;
	protected ColorButton genHatchCB;
	protected ColorButton genShadowCB;
	protected Color genGradientEndC;
	protected Color genGradientStartC;
	protected Color genLineC;
	protected Color genInteriorC;
	protected Color genDbleC;
	protected Color genHatchC;
	protected Color genShadowC;
	protected float genThickness;
	protected float genDotSize;
	protected boolean genIsFilled;
	protected boolean genHasDbleBord;
	protected boolean genHasShadow;
	protected String genHatchS;
	protected String genDotS;
	protected String genLineStyleS;
	protected String genLeftArrS;
	protected String genRightArrS;
	protected String genBordPosS;
	protected ListJToggleButton genColorL;
	
	protected JSpinner dotSizeField;
	protected JLabel labelDotSize;
	
	/** The position of the divider in ]0,1[. @since 2.0.0 */
	protected double dividerPosition;
	
	
	static
	{
		LaTeXDrawPath.checkDirectories();
		
		try
		{
			lookAndFeel = PreferencesFrame.readTheme();
     		UIManager.setLookAndFeel(lookAndFeel);
		}
		catch(Exception ex) { /* Dommage ! */ }
		
		convertFrame = new BatchConvertFrame();
	}
	
	
	
	/** The constructor by default. */
	public LaTeXDrawFrame(String[] args, boolean splash)
	{
		dividerPosition = 0.8;
		
		SplashScreen splashScreen = new SplashScreen(lookAndFeel);

		if(splash)
			splashScreen.setVisible(true);
		
		try
		{
			splashScreen.addToProgressBar(5);
			statusBar = new JTextField("");//$NON-NLS-1$
			statusBar.setEditable(false);
			
			progressBar = new JProgressBar();
			progressBar.setMaximumSize(new Dimension(300, 20));
			progressBar.setPreferredSize(new Dimension(300, 20));
			progressBar.setVisible(false);
			
			menusListener = new MenusListener(this);
			
			splashScreen.addToProgressBar(15);
			setIconImage(LaTeXDrawResources.latexdrawIcon.getImage());
			
			checkNewVersion = DEFAULT_CHECK_VERSION && WITH_UPDATE;
	     	codePanel  = new CodePanel(this);
	     	codePanel.setVisible(DEFAULT_DISPLAY_CODEPANEL);
	     	splashScreen.addToProgressBar(5);
	     	DrawContainer draw = new DrawContainer(DEFAULT_DISPLAY_BORDERS);
	     	splashScreen.addToProgressBar(10);
	     	drawPanel  = new DrawPanel(draw, codePanel, this);
	     	splashScreen.addToProgressBar(5);
	     	
	        JPanel paramsToolbar = new JPanel(new FlowLayout(FlowLayout.LEFT));	
	     	JPanel p= new JPanel();
	     	
			projectName       = null;
			fileChooserSave   = null;
			fileChooserExport = null;
			currentFile       = null;		
			
			splashScreen.addToProgressBar(5);
			menuBar = new LMenuBar(this);
			splashScreen.addToProgressBar(10);
			createToolbar();
			splashScreen.addToProgressBar(10);
			
	     	undoManager = new UndoRedoManager(draw.getFigures(), draw, this);
	     	splashScreen.addToProgressBar(5);
	
	     	yScale = new YScale(drawPanel, xScale);
	     	yScale.setVisible(DEFAULT_DISPLAY_YSCALE);
	     	xScale = new XScale(drawPanel, yScale);
	     	yScale.setXScale(xScale);
	     	xScale.setVisible(DEFAULT_DISPLAY_XSCALE);
	     	
	     	p.setLayout(new BorderLayout());
	     	p.add(yScale, BorderLayout.WEST);
	     	p.add(xScale, BorderLayout.NORTH);
	     	p.add(drawPanel, BorderLayout.CENTER);
	        setJMenuBar(menuBar);
	        updateUndoRedo();
	        splashScreen.addToProgressBar(10);
	
	    	splitPane =  new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, p, codePanel);
	    	splitPane.setContinuousLayout(true); 
	     	splitPane.setOneTouchExpandable(true); 
	     	splitPane.setDividerSize(6);
	     	
	        getContentPane().setLayout(new BorderLayout());
	        getContentPane().add(toolbar, BorderLayout.NORTH);
	     	getContentPane().add(splitPane, BorderLayout.CENTER);
	     	
			addCommentsFrame = new AddCommentsFrame(this);
			
	     	SpinnerModel model = new SpinnerNumberModel(Figure.DEFAULT_THICKNESS,0.1,1000,0.1);
	     	thicknessField = new JSpinner(model);
	     	thicknessField.setEditor(new JSpinner.NumberEditor(thicknessField, "0.0"));//$NON-NLS-1$
			thicknessField.addChangeListener(this);
	     	thicknessField.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.65")); //$NON-NLS-1$
	     	thicknessField.setName(LaTeXDrawResources.NAME_THICKNESS_FIELD);
	     	thicknessField.setMaximumSize(new Dimension(55, 30));
	     	thicknessField.setMinimumSize(new Dimension(55, 30));
	     	thicknessField.setPreferredSize(new Dimension(55, 30));
	     	
	     	model = new SpinnerNumberModel(Dot.DEFAULT_WIDTH,0.1,1000,0.1);
	     	dotSizeField = new JSpinner(model);
	     	dotSizeField.setEditor(new JSpinner.NumberEditor(dotSizeField, "0.0"));//$NON-NLS-1$
	     	dotSizeField.addChangeListener(this);
	     	dotSizeField.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.0")); //$NON-NLS-1$
	     	dotSizeField.setName(LaTeXDrawResources.NAME_DOT_SIZE_FIELD);
	     	dotSizeField.setMaximumSize(new Dimension(55, 30));
	     	dotSizeField.setMinimumSize(new Dimension(55, 30));
	     	dotSizeField.setPreferredSize(new Dimension(55, 30));
	     	labelDotSize = new JLabel(LaTeXDrawResources.dotStyleNoneIcon);
	     	
	     	buttonBordersColor = new ColorButton(LaTeXDrawResources.LABEL_COLOR_BUTTON, new ButtonIcon(Color.BLACK));
	     	buttonBordersColor.setMargin(INSET_BUTTON);
	     	buttonBordersColor.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.66")); //$NON-NLS-1$
	     	buttonBordersColor.setActionCommand(LaTeXDrawResources.LABEL_COLOR_BUTTON);
	     	buttonBordersColor.addActionListener(this);
	     	genLineCB = new ColorButton(LaTeXDrawResources.LABEL_COLOR_BUTTON, new ButtonIcon(Color.BLACK));
	     	genLineCB.setMargin(INSET_BUTTON);
	     	genLineCB.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.66")); //$NON-NLS-1$
	     	genLineCB.setActionCommand(LaTeXDrawResources.LABEL_COLOR_BUTTON);
	     	genLineCB.addActionListener(this);
	     	
	     	filledCheckBox = new JCheckBox(LaTeXDrawResources.LABEL_FILLED_CHECKBOX);
	     	filledCheckBox.setMargin(LaTeXDrawFrame.INSET_BUTTON);
	     	filledCheckBox.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.67")); //$NON-NLS-1$
	     	filledCheckBox.setActionCommand(LaTeXDrawResources.LABEL_FILLED_CHECKBOX);
	     	filledCheckBox.addActionListener(this);
	     	
	     	buttonInteriorColor = new ColorButton(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON, new ButtonIcon(Figure.DEFAULT_INTERIOR_COL));
	     	buttonInteriorColor.setMargin(INSET_BUTTON);
	     	buttonInteriorColor.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.68")); //$NON-NLS-1$
	     	buttonInteriorColor.setActionCommand(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON);
	     	buttonInteriorColor.addActionListener(this);
	     	genInteriorCB = new ColorButton(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON, new ButtonIcon(Figure.DEFAULT_INTERIOR_COL));
	     	genInteriorCB.setMargin(INSET_BUTTON);
	     	genInteriorCB.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.68")); //$NON-NLS-1$
	     	genInteriorCB.setActionCommand(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON);
	     	genInteriorCB.addActionListener(this);
	     	
	     	buttonHatchColor = new ColorButton(LaTeXDrawResources.LABEL_COLOR_HATCH, new ButtonIcon(Figure.DEFAULT_BORDERS_COL));
	     	buttonHatchColor.setMargin(INSET_BUTTON);
	     	buttonHatchColor.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.69")); //$NON-NLS-1$
	     	buttonHatchColor.setActionCommand(LaTeXDrawResources.LABEL_COLOR_HATCH);
	     	buttonHatchColor.addActionListener(this);
	     	genHatchCB = new ColorButton(LaTeXDrawResources.LABEL_COLOR_HATCH, new ButtonIcon(Figure.DEFAULT_BORDERS_COL));
	     	genHatchCB.setMargin(INSET_BUTTON);
	     	genHatchCB.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.69")); //$NON-NLS-1$
	     	genHatchCB.setActionCommand(LaTeXDrawResources.LABEL_COLOR_HATCH);
	     	genHatchCB.addActionListener(this);
	     	
	     	dotChoice = Dot.createDotStyleChoice();
	     	dotChoice.setName(LaTeXDrawResources.CHOICE_DOT_NAME);
	     	dotChoice.addItemListener(this);
	     	dotChoice.setPreferredSize(new Dimension(55,30));
	     	dotChoice.setMaximumSize(new Dimension(55,30));
	     	
	     	lineChoice = Figure.createStyleLineChoice();
	     	lineChoice.setName(LaTeXDrawResources.CHOICE_LINE_NAME);
	     	lineChoice.addItemListener(this);
	     	lineChoice.setPreferredSize(new Dimension(70,30));
	     	lineChoice.setMaximumSize(new Dimension(70,30));
	     	
	     	hatchChoice = Figure.createFillChoice();
	     	hatchChoice.addItemListener(this);
	     	hatchChoice.setPreferredSize(new Dimension(65,30));
	     	hatchChoice.setMaximumSize(new Dimension(65,30));
	     	
	     	lineArrowLChoice = ArrowHead.createLeftArrowStyleList();
	     	lineArrowLChoice.setName(LaTeXDrawResources.CHOICE_ARROW_LEFT_NAME);
	     	lineArrowLChoice.addItemListener(this);
	     	lineArrowLChoice.setPreferredSize(new Dimension(80,30));
	     	lineArrowLChoice.setMaximumSize(new Dimension(80,30));
	     	
	     	lineArrowRChoice = ArrowHead.createRightArrowStyleList();
	     	lineArrowRChoice.setName(LaTeXDrawResources.CHOICE_ARROW_RIGHT_NAME);
	     	lineArrowRChoice.addItemListener(this); 
	     	lineArrowRChoice.setPreferredSize(new Dimension(80,30));
	     	lineArrowRChoice.setMaximumSize(new Dimension(80,30));
	        splashScreen.addToProgressBar(5);
	        JButton buttonImageForeground = new JButton(LaTeXDrawResources.foregroundIcon);
	        buttonImageForeground.setMargin(INSET_BUTTON);
	        buttonImageForeground.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.70")); //$NON-NLS-1$
	        buttonImageForeground.addActionListener(this);
	        buttonImageForeground.setActionCommand(LaTeXDrawResources.LABEL_FOREGROUND);
	        
	        JButton buttonImageBackground = new JButton(LaTeXDrawResources.backgroundIcon);
	        buttonImageBackground.setMargin(INSET_BUTTON);
	        buttonImageBackground.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.71")); //$NON-NLS-1$
	        buttonImageBackground.addActionListener(this);
	        buttonImageBackground.setActionCommand(LaTeXDrawResources.LABEL_BACKGROUND);
	     	
	        JButton buttonImageFront = new JButton(LaTeXDrawResources.inFrontOfSelIcon);
	        buttonImageFront.setMargin(INSET_BUTTON);
	     	buttonImageFront.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.72")); //$NON-NLS-1$
	     	buttonImageFront.addActionListener(this);
	     	buttonImageFront.setActionCommand(LaTeXDrawResources.LABEL_FRONT_SEL);

	     	JButton buttonImageBehind = new JButton(LaTeXDrawResources.behindSelIcon);
	     	buttonImageBehind.setMargin(INSET_BUTTON);
	        buttonImageBehind.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.73")); //$NON-NLS-1$
	        buttonImageBehind.addActionListener(this);
	        buttonImageBehind.setActionCommand(LaTeXDrawResources.LABEL_BEHIND_SEL);
	     	
	        rotationButtons = new ListJToggleButton(this, LaTeXDrawResources.rotateIcon, ListJToggleButton.LOCATION_NORTH);
	        rotationButtons.setToolTipText(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.2")); //$NON-NLS-1$
	        
	        rotateButton = new JToggleButton(LaTeXDrawResources.rotateIcon);
	     	rotateButton.setMargin(INSET_BUTTON);
	     	rotateButton.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.74")); //$NON-NLS-1$
	     	rotateButton.addActionListener(this);
	     	rotateButton.setActionCommand(LaTeXDrawResources.LABEL_ROTATE);
	     	
	     	JButton rotate90Button = new JButton(LaTeXDrawResources.rotate90Icon);
	     	rotate90Button.setMargin(INSET_BUTTON);
	     	rotate90Button.setToolTipText(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.3")); //$NON-NLS-1$
	     	rotate90Button.addActionListener(this);
	     	rotate90Button.setActionCommand(LaTeXDrawResources.LABEL_ROTATE_90);
	     	
	     	JButton rotate180Button = new JButton(LaTeXDrawResources.rotate180Icon);
	     	rotate180Button.setMargin(INSET_BUTTON);
	     	rotate180Button.setToolTipText(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.4")); //$NON-NLS-1$
	     	rotate180Button.addActionListener(this);
	     	rotate180Button.setActionCommand(LaTeXDrawResources.LABEL_ROTATE_180);
	     	
	     	JButton rotate270Button = new JButton(LaTeXDrawResources.rotate270Icon);
	     	rotate270Button.setMargin(INSET_BUTTON);
	     	rotate270Button.setToolTipText(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.5")); //$NON-NLS-1$
	     	rotate270Button.addActionListener(this);
	     	rotate270Button.setActionCommand(LaTeXDrawResources.LABEL_ROTATE_270);
	     	
	     	rotationButtons.addComponent(rotateButton);
	     	rotationButtons.addComponent(rotate90Button);
	     	rotationButtons.addComponent(rotate180Button);
	     	rotationButtons.addComponent(rotate270Button);
	     	rotationButtons.addSeparator();
	     	
	     	mirrorButtons = new ListJToggleButton(this, LaTeXDrawResources.mirrorHIcon, ListJToggleButton.LOCATION_NORTH);
	     	mirrorButtons.setToolTipText(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.6")); //$NON-NLS-1$
	     	
	     	JButton button = new JButton(LaTeXDrawResources.mirrorHIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.7")); //$NON-NLS-1$
	     	button.addActionListener(this);
	     	button.setActionCommand(LaTeXDrawResources.LABEL_MIRROR_H);
	     	mirrorButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.mirrorVIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.8")); //$NON-NLS-1$
	     	button.addActionListener(this);
	     	button.setActionCommand(LaTeXDrawResources.LABEL_MIRROR_V);
	        splashScreen.addToProgressBar(5);
	     	mirrorButtons.addComponent(button);
	     	mirrorButtons.addSeparator();
	     	
	     	alignButtons = new ListJToggleButton(this, LaTeXDrawResources.alignLeftIcon, ListJToggleButton.LOCATION_NORTH);
	     	alignButtons.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.1")); //$NON-NLS-1$
	     	ToolbarListener tl = new ToolbarListener(this);
	     	
	     	button = new JButton(LaTeXDrawResources.alignLeftIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.2")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_ALIGN_LEFT);
	     	alignButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.alignRightIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.3")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_ALIGN_RIGHT);
	     	alignButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.alignTopIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.4")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_ALIGN_TOP);
	     	alignButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.alignBottomIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.5")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_ALIGN_BOTTOM);
	     	alignButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.alignMiddleHorizIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.6")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_ALIGN_MIDDLE_H);
	     	alignButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.alignMiddleVertIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.7")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_ALIGN_MIDDLE_V);
	     	alignButtons.addComponent(button);
	     	alignButtons.addSeparator();
	     	
	     	distribButtons = new ListJToggleButton(this, LaTeXDrawResources.distVertBottomIcon, ListJToggleButton.LOCATION_NORTH);
	     	distribButtons.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.8")); //$NON-NLS-1$
	     	
	     	button = new JButton(LaTeXDrawResources.distVertBottomIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.9")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_V_BOTTOM);
	     	distribButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.distVertEqualIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.10")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_V_EQ);
	     	distribButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.distVertMiddleIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.11")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_V_MID);
	     	distribButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.distVertTopIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.12")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_V_TOP);
	     	distribButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.distHorizLeftIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.13")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_H_LEFT);
	     	distribButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.distHorizEqualIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.14")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_H_EQ);
	     	distribButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.distHorizMiddleIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.15")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_H_MID);
	     	distribButtons.addComponent(button);
	     	
	     	button = new JButton(LaTeXDrawResources.distHorizRightIcon);
	     	button.setMargin(INSET_BUTTON);
	     	button.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.16")); //$NON-NLS-1$
	     	button.addActionListener(tl);
	     	button.setActionCommand(ToolbarListener.LABEL_DISTRIB_H_RIGHT);
	     	distribButtons.addComponent(button);
	     	
	     	joinButton = new JButton(LaTeXDrawResources.joinIcon);
	     	joinButton.setMargin(INSET_BUTTON);
	     	joinButton.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.75")); //$NON-NLS-1$
	     	joinButton.addActionListener(this);
	     	joinButton.setActionCommand(LaTeXDrawResources.LABEL_JOIN);
	        
	        separateButton = new JButton(LaTeXDrawResources.separateIcon);
	        separateButton.setMargin(INSET_BUTTON);
	        separateButton.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.76")); //$NON-NLS-1$
	        separateButton.addActionListener(this);
	        separateButton.setActionCommand(LaTeXDrawResources.LABEL_SEPARATE);
	     	
	        bordersPositionChoice = Figure.createBordersPositionChoice();
	        bordersPositionChoice.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.77")); //$NON-NLS-1$
	        bordersPositionChoice.addItemListener(this);
	        bordersPositionChoice.setPreferredSize(new Dimension(45,30));
	        bordersPositionChoice.setMaximumSize(new Dimension(45,30));
	        
	        dbleBoundCheckBox = new JCheckBox(LaTeXDrawResources.LABEL_DBLE_BOUND_CHECKBOX);
	        dbleBoundCheckBox.setMargin(INSET_BUTTON);
	        dbleBoundCheckBox.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.78")); //$NON-NLS-1$
	        dbleBoundCheckBox.setActionCommand(LaTeXDrawResources.LABEL_DBLE_BOUND_CHECKBOX);
	        dbleBoundCheckBox.addActionListener(this);
	     	
	        buttonDbleBoundColor = new ColorButton(LaTeXDrawResources.LABEL_DBLE_BOUND_CHECKBOX, new ButtonIcon(Figure.DEFAULT_DOUBLE_COLOR));
	        buttonDbleBoundColor.setMargin(INSET_BUTTON);
	        buttonDbleBoundColor.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.79")); //$NON-NLS-1$
	        buttonDbleBoundColor.setActionCommand(LaTeXDrawResources.LABEL_COLOR_DBLE_BOUND);
	        buttonDbleBoundColor.addActionListener(this);
	        genDbleBordCB = new ColorButton(LaTeXDrawResources.LABEL_DBLE_BOUND_CHECKBOX, new ButtonIcon(Figure.DEFAULT_DOUBLE_COLOR));
	        genDbleBordCB.setMargin(INSET_BUTTON);
	        genDbleBordCB.setToolTipText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.79")); //$NON-NLS-1$
	        genDbleBordCB.setActionCommand(LaTeXDrawResources.LABEL_COLOR_DBLE_BOUND);
	        genDbleBordCB.addActionListener(this);

	        stopButton = new JButton(LaTeXDrawResources.stopIcon);
	        stopButton.setMargin(INSET_BUTTON);
	        stopButton.setActionCommand(LaTeXDrawResources.LABEL_STOP);
	        stopButton.addActionListener(this);
	        stopButton.setName(LaTeXDrawResources.LABEL_STOP);
	        stopButton.setToolTipText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.9")); //$NON-NLS-1$
	        stopButton.setVisible(false);
	        
	        shadowCheckBox = new JCheckBox(LaTeXDrawResources.LABEL_SHADOW_CHECKBOX);
	        shadowCheckBox.setMargin(INSET_BUTTON);
	        shadowCheckBox.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.4")); //$NON-NLS-1$
	        shadowCheckBox.setActionCommand(LaTeXDrawResources.LABEL_SHADOW_CHECKBOX);
	        shadowCheckBox.addActionListener(this); 
	        
	        buttonShadowColor = new ColorButton(LaTeXDrawResources.LABEL_COLOR_SHADOW, new ButtonIcon(Figure.DEFAULT_SHADOW_COLOR));
	        buttonShadowColor.setMargin(INSET_BUTTON);
	        buttonShadowColor.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.5")); //$NON-NLS-1$
	        buttonShadowColor.setActionCommand(LaTeXDrawResources.NAME_COLOR_SHADOW);
	        buttonShadowColor.addActionListener(this);
	        genShadowCB = new ColorButton(LaTeXDrawResources.LABEL_COLOR_SHADOW, new ButtonIcon(Figure.DEFAULT_SHADOW_COLOR));
	        genShadowCB.setMargin(INSET_BUTTON);
	        genShadowCB.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.5")); //$NON-NLS-1$
	        genShadowCB.setActionCommand(LaTeXDrawResources.NAME_COLOR_SHADOW);
	        genShadowCB.addActionListener(this);
	        
     		labelThickness = new JLabel(LaTeXDrawResources.thicknessIcon); 
     		locationButtons = new ListJToggleButton(this, LaTeXDrawResources.foregroundIcon, ListJToggleButton.LOCATION_NORTH);
     		locationButtons.addComponent(buttonImageForeground);
     		locationButtons.addComponent(buttonImageBackground);
     		locationButtons.addComponent(buttonImageFront);
     		locationButtons.addComponent(buttonImageBehind);
     		locationButtons.addSeparator();
     		locationButtons.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.6")); //$NON-NLS-1$
     		
	        buttonGradientStartC = new ColorButton(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST, new ButtonIcon(PSTricksConstants.DEFAULT_GRADIENT_START_COLOR)); 
	        buttonGradientStartC.setMargin(INSET_BUTTON);
	        buttonGradientStartC.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.7")); //$NON-NLS-1$
	        buttonGradientStartC.setActionCommand(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST);
	        buttonGradientStartC.addActionListener(this);
	        genGradientStartCB = new ColorButton(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST, new ButtonIcon(PSTricksConstants.DEFAULT_GRADIENT_START_COLOR)); 
	        genGradientStartCB.setMargin(INSET_BUTTON);
	        genGradientStartCB.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.7")); //$NON-NLS-1$
	        genGradientStartCB.setActionCommand(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST);
	        genGradientStartCB.addActionListener(this);
	        
	        buttonGradientEndC = new ColorButton(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND, new ButtonIcon(PSTricksConstants.DEFAULT_GRADIENT_END_COLOR));
	        buttonGradientEndC.setMargin(INSET_BUTTON);
	        buttonGradientEndC.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.8")); //$NON-NLS-1$
	        buttonGradientEndC.setActionCommand(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND);
	        buttonGradientEndC.addActionListener(this);
	        genGradientEndCB = new ColorButton(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND, new ButtonIcon(PSTricksConstants.DEFAULT_GRADIENT_END_COLOR));
	        genGradientEndCB.setMargin(INSET_BUTTON);
	        genGradientEndCB.setToolTipText(LaTeXDrawLang.getString1_7("LaTeXDrawFrame.8")); //$NON-NLS-1$
	        genGradientEndCB.setActionCommand(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND);
	        genGradientEndCB.addActionListener(this);
	        splashScreen.addToProgressBar(5);
	        
			initializeGeneralFields();
			genColorL = new ListJToggleButton(this, LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.17"), ListJToggleButton.LOCATION_NORTH); //$NON-NLS-1$
			genColorL.addComponent(genLineCB);
			genColorL.addComponent(genInteriorCB);
			genColorL.addComponent(genHatchCB);
			genColorL.addComponent(genDbleBordCB);
			genColorL.addComponent(genShadowCB);
			genColorL.addComponent(genGradientStartCB);
			genColorL.addComponent(genGradientEndCB);
			
	     	paramsToolbar.add(progressBar);
	     	paramsToolbar.add(stopButton);
	    	paramsToolbar.add(rotationButtons);
	    	paramsToolbar.add(locationButtons);
	    	paramsToolbar.add(mirrorButtons);
	    	paramsToolbar.add(alignButtons);
	    	paramsToolbar.add(distribButtons);
	    	paramsToolbar.add(joinButton);
	    	paramsToolbar.add(separateButton);
	        paramsToolbar.add(labelThickness);
     		paramsToolbar.add(thicknessField);
     		paramsToolbar.add(labelDotSize);
     		paramsToolbar.add(dotSizeField);
     		paramsToolbar.add(genColorL);
     		paramsToolbar.add(buttonBordersColor);
     		paramsToolbar.add(filledCheckBox);
     		paramsToolbar.add(buttonInteriorColor);
     		paramsToolbar.add(hatchChoice);
     		paramsToolbar.add(buttonGradientStartC);
     		paramsToolbar.add(buttonGradientEndC);
     		paramsToolbar.add(buttonHatchColor);
	     	paramsToolbar.add(dotChoice);
	     	paramsToolbar.add(lineChoice);
	     	paramsToolbar.add(lineArrowLChoice);
	     	paramsToolbar.add(lineArrowRChoice);
	     	paramsToolbar.add(bordersPositionChoice);
	     	paramsToolbar.add(dbleBoundCheckBox);
	     	paramsToolbar.add(buttonDbleBoundColor);
	     	paramsToolbar.add(shadowCheckBox);
	     	paramsToolbar.add(buttonShadowColor);
	     	
	     	JPanel southPanel = new JPanel();
	     	southPanel.setLayout(new BorderLayout());
	     	southPanel.add(paramsToolbar, BorderLayout.CENTER);  
	     	southPanel.add(statusBar, BorderLayout.SOUTH);
	     	getContentPane().add(southPanel, BorderLayout.SOUTH);
	        splashScreen.addToProgressBar(5);
	     	preferencesFrame = new PreferencesFrame(draw, this);
	     	formerIsInAutoUpdate = preferencesFrame.isCodeInAutoUpdate();
	     	convertFrame.setPathSelect(pathOpen);
	     	convertFrame.setPathOutput(pathExport);
	     	codePanel.setIsInAutoUpdate(formerIsInAutoUpdate);
			recentFilesListener = new RecentFilesListener(preferencesFrame, this);
			shortcutsListener = new ShortcutsListener(this);
			addKeyListener(shortcutsListener);

	     	drawPanel.isSelection(false);
	     	setTitle(LaTeXDrawResources.LABEL_APP);
	     	setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
	     	
	     	addWindowListener(
	                new WindowAdapter() 
	                {
						@Override
						public void windowClosing(WindowEvent e) 
	                    {
	                    	onWindowClosing();
	                    }
	                });  
	     	
	     	setCurrentChoice(LaTeXDrawResources.LABEL_SELECT_FIGURE);
	     	setSelection(false);
	     	
	     	boolean fileIsOk = true;
	     	/* Management of command line */
	     	String fName = getFileName(args);
	     	
	     	if(args.length>1)
     			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.81")); //$NON-NLS-1$
	     	
	     	if(fName!=null)
	     		if(fName.toLowerCase().endsWith(TeXFilter.TEX_EXTENSION))
	     			fileIsOk = importPSTricksFile(fName);
	     		else 
	     			fileIsOk = open(fName, false);
	     	
	     	menuBar.updateRecentFilesMenu();
	     	drawPanel.setIsModified(false);
	        splashScreen.addToProgressBar(5);
	        
	     	if(checkNewVersion && WITH_UPDATE)
	     	{
	     		VersionChecker vc = new VersionChecker(this);
	     		vc.start();
	     	}
	     	
			addWindowStateListener(this);
	     	splashScreen.setVisible(false);	
			setVisible(true);
			updateSplitSeparatorPosition();
			toolbar.getSliderDelimitor().setValue(Delimitor.getOpacity());
	     	
			if(!fileIsOk) {
				projectName = null;
				setTitle();
				JOptionPane.showMessageDialog(this, LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.18"), //$NON-NLS-1$
			 		    LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"), JOptionPane.WARNING_MESSAGE); //$NON-NLS-1$
			}
		}
		catch(Exception e1) 
		{
			splashScreen.setVisible(false);
			JOptionPane.showMessageDialog(this, LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.82") + //$NON-NLS-1$
		 		    e1.toString(), LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.83"),  //$NON-NLS-1$
		 		    JOptionPane.WARNING_MESSAGE); 
			e1.printStackTrace(); 
			LaTeXDrawFrame.this.windowClosed();
		}
	}
	

	
	/**
	 * deduces the filename that will be open, from the set of parameters.
	 * @param args The parameters given to latexdraw.
	 * @return The filename that will be open.
	 * @since 2.0.2
	 */
	public static String getFileName(String[] args)
	{
		if(args==null || args.length==0)
			return null;
		
		String name = args[0];
		
		if(args.length!=1)
		{
			File f = new File(name);
			boolean ok = f.exists();
			int i = 1;
			
			while(!ok && i<args.length)
			{
				name += " " + args[i];
				i++;
				f = new File(name);
				ok = f.exists();
			}
			
			f = null;
			
			if(!ok)
				name = args[0];
		}
		
		return name;
	}
	
	
	
	/**
	 * @return Returns the versionOfFile.
	 */
	public static String getVersionOfFile()
	{
		return versionOfFile;
	}

	
	public static void setVersionOfFile(String v)
	{
		if(v!=null)
			versionOfFile = v;
	}


	/**
	 * Defines actions to do on window closing
	 */
	public void onWindowClosing()
	{
		getPreferencesFrame().writeXMLPreferences(false);
		
		try
    	{
	    	if(drawPanel.isModified())
	    	{
	    		int id = dialogConfirmSave();
	    		
	    		switch(id)
	    		{
	    			case JOptionPane.YES_OPTION : 
	    				if(save(false))
	    					LaTeXDrawFrame.this.windowClosed();
	    				break;
	    			case JOptionPane.NO_OPTION :  
                        LaTeXDrawFrame.this.windowClosed();
                        break;
	    			case JOptionPane.CANCEL_OPTION : return;
	    			default : return;
	    		}              		
	    	}
	    	else LaTeXDrawFrame.this.windowClosed();
    	}catch(Exception ex)
    	{
    		ex.printStackTrace(); 
    		ExceptionFrameDialog.showExceptionDialog(ex);
    	}
	}
	
	

	
	/**
	 * Allows to show the frame which allows to change the parameters of
	 * a figure
	 * @param f The figure we want to display its parameters
	 */
	public void showParamFrame(Figure f, boolean deleteOnCancel)
	{
		try
		{
			if(f==null)
				return ;
			
			if(f instanceof BezierCurve)
			{ 
				if(paramBezierCurveFrame==null)
					paramBezierCurveFrame = new ParametersBezierCurveFrame(this, drawPanel, true);
				
				paramBezierCurveFrame.setVisible(true, f, deleteOnCancel, false);
			}else
			if(f instanceof AkinPoints)
			{
				if(paramAkinPointsFrame==null)
					paramAkinPointsFrame = new ParametersAkinPointsFrame(this, drawPanel, true);
			
				paramAkinPointsFrame.setVisible(true, f, deleteOnCancel, false);    			
			}else
			if(f instanceof Line)
			{
				if(paramLineFrame==null)
					paramLineFrame = new ParametersLineFrame(this, drawPanel, true);
			
				paramLineFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Arc)
			{
				if(paramArcFrame==null)
					paramArcFrame = new ParametersArcFrame(this, drawPanel, true);
	    		
				paramArcFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Axe)
			{
				if(paramAxesFrame==null)
					paramAxesFrame = new ParametersAxeFrame(this, drawPanel, true);
				
				paramAxesFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Circle || f instanceof Square)
			{
				if(paramCircleFrame==null)
					paramCircleFrame = new ParametersCircleSquareFrame(this, drawPanel, true);
	    		
				paramCircleFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Ellipse || f instanceof LaTeXDrawRectangle)
			{
				if(paramEllipseFrame==null)
					paramEllipseFrame = new ParametersEllipseRectangleFrame(this, drawPanel, true);
	    		
				paramEllipseFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Dot)
			{
				if(paramDotFrame==null)
					paramDotFrame = new ParametersDotFrame(this, drawPanel, true);
	    		
				paramDotFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Triangle)
			{
				if(paramTriangleFrame==null)
					paramTriangleFrame = new ParametersTriangleFrame(this, drawPanel, true);
	    		
				paramTriangleFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Rhombus)
			{
				if(paramRhombusFrame==null)
					paramRhombusFrame = new ParametersRhombusFrame(this, drawPanel, true);
	    		
				paramRhombusFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof JoinedLines)
			{
				if(paramJoinedLinesFrame==null)
					paramJoinedLinesFrame = new ParametersPolygonJoinedLinesFrame(this, drawPanel, false, true);
	    		
				paramJoinedLinesFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof LaTeXDrawPolygon)
			{
				if(paramPolygonFrame==null)
					paramPolygonFrame = new ParametersPolygonJoinedLinesFrame(this, drawPanel, true, true);
	    		
				paramPolygonFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Text)
			{
				if(paramTextFrame==null)
					paramTextFrame = new ParametersTextFrame(this, drawPanel, true);
				
				paramTextFrame.setVisible(true, f, deleteOnCancel, false);
			}
			else if(f instanceof Grid)
			{ 
				if(paramGridFrame==null)
					paramGridFrame = new ParametersGridFrame(this, drawPanel, true);
				
				paramGridFrame.setVisible(true, f, deleteOnCancel, false);
			}
//			else if(f instanceof Draw)
//			{ 
//				if(paramDrawFrame==null)
//					paramDrawFrame = new ParametersDrawFrame(this, drawPanel, true);
//				
//				paramDrawFrame.setVisible(true, f, deleteOnCancel, false);
//			}
			
		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	

	
	
	/**
	 * Allows to get the undo/redo manager
	 * @return The undo/redo manager
	 */
	public UndoRedoManager getUndoManager()
	{
		return undoManager;
	}
	
	
	
	/**
	 * Allows to set the name of the application
	 */
	public void setTitle()
	{
		if(projectName==null)
			  super.setTitle(LaTeXDrawResources.LABEL_APP);
		else  super.setTitle(LaTeXDrawResources.LABEL_APP + " - " + projectName + (drawPanel.isModified() ? "*" : ""));//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
	}
	
	
	
	/**
	 * Allows to update some fields when a figure is selected or not.
	 * @param onSelection True : a figure is selected.
	 */
	public void setSelection(boolean onSelection)
	{
		Draw listSelected = drawPanel.getDraw().getSelected();
		boolean empty = listSelected==null || listSelected.isEmpty();
		
		menuBar.getExportTemplateMenu().setEnabled(onSelection && !empty);
		menuBar.getDeleteFigure().setEnabled(onSelection && !empty);
		menuBar.getPropertiesFigureMenu().setEnabled(onSelection && !empty);
		toolbar.getButtonImageProperties().setEnabled(onSelection && !empty);
		toolbar.getButtonImageDel().setEnabled(onSelection && !empty);
		locationButtons.setVisible(onSelection && !empty);
		rotationButtons.setVisible(onSelection && !empty);
		mirrorButtons.setVisible(onSelection && !empty);
		alignButtons.setVisible(onSelection && listSelected!=null && listSelected.size()>1);
		distribButtons.setVisible(onSelection && listSelected!=null && listSelected.size()>2);
		menuBar.getCopyMenu().setEnabled(onSelection && !empty);
		menuBar.getCutMenu().setEnabled(onSelection && !empty);
		toolbar.getCopyCodeSelButton().setEnabled(onSelection && !empty);

		if(onSelection && listSelected!=null && listSelected.size()==1 && listSelected.getFigureAt(0).isCustomizable() &&
			!(listSelected.getFigureAt(0) instanceof Draw))
		{
			toolbar.getButtonImageProperties().setEnabled(true);
			joinButton.setVisible(false);
			buttonBordersColor.setVisible(true);
			separateButton.setVisible(listSelected.getFigureAt(0) instanceof Draw ? true : false);
			
			Figure selected = listSelected.getFigureAt(0);
			
			if(selected != null)
			{
				if(selected instanceof Dot)
				{
					dotChoice.setVisible(true);
					// A dot can always filled, or it cannot be filled.
					filledCheckBox.setVisible(false);
					buttonInteriorColor.setVisible(selected.canBeFilled());
					labelDotSize.setVisible(true);
					dotSizeField.setVisible(true);
				}
				else
				{					
					labelDotSize.setVisible(false);
					dotSizeField.setVisible(false);
					dotChoice.setVisible(false);
					filledCheckBox.setVisible(selected.canBeFilled() && !selected.hasGradient());
				} //else
				
				buttonInteriorColor.setVisible(selected.canBeFilled() && (selected.isFilled() || selected.hasShadow()) && 
												!selected.hasGradient() && selected.shadowFillsShape());
				lineArrowLChoice.setVisible(selected.canHaveArrow());
				lineArrowRChoice.setVisible(selected.canHaveArrow());	
				shadowCheckBox.setVisible(selected.canHaveShadow());
				buttonShadowColor.setVisible(selected.hasShadow());
				dbleBoundCheckBox.setVisible(selected.isDoubleBoundaryable());
				buttonDbleBoundColor.setVisible(dbleBoundCheckBox.isSelected());
				bordersPositionChoice.setVisible(selected.isBordersMovable());
				labelThickness.setVisible(selected.isThicknessable());
				thicknessField.setVisible(selected.isThicknessable());
				lineChoice.setVisible(selected.isDashableOrDotable());
				hatchChoice.setVisible(selected.canBeHatched());
				buttonGradientEndC.setVisible(selected.canBeHatched() && selected.hasGradient());
				buttonGradientStartC.setVisible(buttonGradientEndC.isVisible());
				buttonHatchColor.setVisible(selected.canBeHatched() && selected.isHatched());
			}
			genColorL.setVisible(false);
		}// if(onSelection)
		else 		
		{
			if(listSelected!=null && (listSelected.size()!=1 || (!listSelected.getFigureAt(0).isCustomizable() &&
					!(listSelected.getFigureAt(0) instanceof Draw))))
			{
				toolbar.getButtonImageProperties().setEnabled(false);
				menuBar.getPropertiesFigureMenu().setEnabled(false);
			}
			
			toolbar.getButtonImageProperties().setEnabled(false);
			dbleBoundCheckBox.setVisible(true);
			buttonDbleBoundColor.setVisible(false);
			bordersPositionChoice.setVisible(true);
			buttonBordersColor.setVisible(false);
			labelThickness.setVisible(true);
			buttonHatchColor.setVisible(false);
			buttonInteriorColor.setVisible(false);
			hatchChoice.setVisible(true);
			buttonHatchColor.setVisible(false);
			dotChoice.setVisible(true);
			lineChoice.setVisible(true);
			lineArrowLChoice.setVisible(true);
			lineArrowRChoice.setVisible(true);
			thicknessField.setVisible(true);
			buttonShadowColor.setVisible(false);
			shadowCheckBox.setVisible(true);
			buttonGradientEndC.setVisible(false);
			buttonGradientStartC.setVisible(false);
			filledCheckBox.setVisible(true);
			genColorL.setVisible(true);
			dotSizeField.setVisible(true);
			labelDotSize.setVisible(true);
			
			joinButton.setVisible(listSelected!=null && listSelected.size()>1);
			separateButton.setVisible(listSelected!=null && listSelected.size()==1 && listSelected.getFigureAt(0) instanceof Draw);
		}
		
		updateFrameFields();
	}
	
	
	
    
    /**
	 * Initialises the toolbar of the application.
	 */
	private void createToolbar()
	{
		toolbar = new LToolbar(this);		
 		requestFocus();
	}
	
	
	
	/**
	 * Allows to get the path for open/save actions
	 * @return The path for open/save actions
	 */
	public static String getPathExport()
	{
		return pathExport;
	}
	
	

	
	/**
	 * Allows to get the path for exportation actions
	 * @return The path for exportation actions
	 */
	public static String getPathOpen()
	{
		return pathOpen;
	}
	
	
	
	
	/**
	 * Allows to get the current choice of the user
	 * @return The current choice of the user
	 */
	public String getCurrentChoice()
	{
		return currentChoice;
	}
	
	
	
	/**
	 * Allows to get the codePanel
	 * @return The codePanel
	 */
	public CodePanel getCodePanel()
	{
		return codePanel;
	}
	
	
	
	/**
	 * Allows to get the drawPanel
	 * @return The drawPanel
	 */
	public DrawPanel getDrawPanel()
	{
		return drawPanel;
	}
	
	
	
	/**
	 * Allows to set if the grid must be displayed.
	 * @param display True : the grid must be displayed.
	 * @param classicGrid Define if it is the classic grid which must be drawn (only if display=true).
	 */
	public void displayGrid(boolean display, boolean classicGrid)
	{
		if(display)
		{
			menuBar.getDisplayGrid().setSelected(classicGrid);
			toolbar.getPersoGridGapField().setEnabled(!classicGrid);
			menuBar.getDisplayPersoGrid().setSelected(!classicGrid);
			drawPanel.draw.setClassicGridDrawn(classicGrid);
		}
		else
		{
			menuBar.getDisplayGrid().setSelected(false);
			toolbar.getPersoGridGapField().setEnabled(false);
			menuBar.getDisplayPersoGrid().setSelected(false);
		}
		
		drawPanel.setIsGridDisplay(display);
		drawPanel.setIsModified(true);
	}
	

	
	/**
	 * Allows to set if the XScale must be displayed
	 * @param display True : the XScale must be displayed
	 */
	public void displayXScale(boolean display)
	{
		menuBar.getDisplayXScale().setSelected(display);
		xScale.setVisible(display);
		
		try {drawPanel.setIsModified(true);} 
		catch (Exception e) 
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
		
		menuBar.getDisplayXScale().setSelected(display);
	}
	
	
	
	/**
	 * Allows to set if the YScale must be displayed
	 * @param display True : the YScale must be displayed
	 */
	public void displayYScale(boolean display)
	{
		yScale.setVisible(display);
		
		try {drawPanel.setIsModified(true);} 
		catch (Exception e) 
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
		
		menuBar.getDisplayYScale().setSelected(display);
	}
	

	
	
	/**
	 * Allows to set if the borders of the drawing must be displayed
	 * @param display True : the borders of the drawing must be displayed
	 */
	public void displayBorders(boolean display)
	{
		try 
		{
			drawPanel.displayBorders(display);
			menuBar.getDisplayBorders().setSelected(display);
			toolbar.getButtonImageBorders().setSelected(display);
			drawPanel.setIsModified(true);
		}
		catch (Exception e) 
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	
	/**
	 * Allows to set if the code panel must be displayed
	 * @param display True: the code panel must be displayed
	 */
	public void displayCodePanel(boolean display)
	{
		if(display!=isCodePanelDisplayed())
			drawPanel.setIsModified(true);
		
		menuBar.getDisplayCodePanel().setSelected(display);
		toolbar.getCopyCodeButton().setVisible(!display);
		
		if(display)
		{
			codePanel.setVisible(true);
			codePanel.setIsInAutoUpdate(formerIsInAutoUpdate);
			splitPane.setDividerLocation(formerDividerLocation);
		}
		else
		{
			formerIsInAutoUpdate = codePanel.isInAutoUpdate();
			codePanel.setIsInAutoUpdate(false);
			formerDividerLocation = splitPane.getDividerLocation();	
			codePanel.setVisible(false);
		}
		
		splitPane.revalidate();
	}
	
	
	
	/**
	 * Allows to set the path for open/save actions
	 * @param path The new path
	 */
	public void setPathOpen(String path)
	{
		pathOpen = path;
		if(fileChooserSave!=null)
			fileChooserSave.setCurrentDirectory(new File(path));
	}
	
	
	
	
	/**
	 * Allows to set the path for exportations actions
	 * @param path The new path
	 */
	public void setPathExport(String path)
	{
		pathExport = path;
		if(fileChooserExport!=null)
			fileChooserExport.setCurrentDirectory(new File(path));
	}
	
	
	
	
	/**
	 * Allows to set if the menu "paste" must be enable or not
	 * @param mustBeEnable True : the menu "paste" must be activated
	 */
	public void setMenuPaste(boolean mustBeEnable)
	{
		getLMenuBar().getPasteMenu().setEnabled(mustBeEnable);
	}
	
	
	
	
	/**
	 * Allows to set if the menu "copy" must be enable or not
	 * @param mustBeEnable True : the menu "copy" must be activated
	 */
	public void setMenuCopy(boolean mustBeEnable)
	{
		getLMenuBar().getCopyMenu().setEnabled(mustBeEnable);
	}
	
	

	
	/**
	 * Allows to set the selected field of the choice lineChoice
	 * @param choice The new selected choice
	 */
	public void setLineChoice(String choice)
	{
		if(!Figure.isValidStyle(choice))
			throw new IllegalArgumentException();
		
		lineChoice.removeItemListener(this);
		lineChoice.setSelectedItem(choice);
		lineChoice.addItemListener(this);
	}
	
	
	
	/**
	 * Allows to set the selected field of the choice <code>bordersPositionChoice</code>.
	 * @param choice The new choice.
	 * @since 1.9.1
	 */
	public void setBorderChoice(String choice)
	{
		bordersPositionChoice.removeItemListener(this);
		bordersPositionChoice.setSelectedItem(choice);
		bordersPositionChoice.addItemListener(this);
	}
	
	
	
	/**
	 * Allows to set the selected field of the choice <code>hatchChoice</code>.
	 * @param choice The new choice.
	 * @since 1.9.1
	 */
	public void setHatchChoice(String choice)
	{
		hatchChoice.removeItemListener(this);
		hatchChoice.setSelectedItem(choice);
		hatchChoice.addItemListener(this);
	}
	
	
	
	/**
	 * Allows to set the selected field of the choice lineArrowRChoice
	 * @param choice The new selected choice
	 */
	public void setLineArrowRChoice(String choice)
	{
		if(!PSTricksConstants.isValidArrowStyle(choice))
			throw new IllegalArgumentException();	
		
		lineArrowRChoice.removeItemListener(this);
		lineArrowRChoice.setSelectedItem(choice);
		lineArrowRChoice.addItemListener(this);
	}
	
	
	
	
	/**
	 * Allows to set the selected field of the choice lineArrowLChoice
	 * @param choice The new selected choice
	 */
	public void setLineArrowLChoice(String choice)
	{
		if(!PSTricksConstants.isValidArrowStyle(choice))
			throw new IllegalArgumentException();		
		
		lineArrowLChoice.removeItemListener(this);
		lineArrowLChoice.setSelectedItem(choice);
		lineArrowLChoice.addItemListener(this);
	}
	
	
		
	
	/**
	 * Allows to update the selected field in the Choice component
	 * dotChoice
	 * @param newChoice The choice which must be selected
	 */
	public void setDotChoice(String newChoice)
	{
		if(!PSTricksConstants.isValidDotStyle(newChoice))
			throw new IllegalArgumentException();
		
		dotChoice.removeItemListener(this);
		dotChoice.setSelectedItem(newChoice);
		dotChoice.addItemListener(this);
	}
	
	
	
	
	/**
	 * Allows to set the menu exportAs able or not
	 * @param state It's new state
	 */
	public void setExportAsMenu(boolean state)
	{
		getLMenuBar().getExportAsMenu().setEnabled(state);
		getToolbar().getButtonPdfExport().setEnabled(state);
	}
	
	
	
	/**
	 * Allows to set the value of the field thickness
	 * @param thickness The new value of the field thickness
	 */
	public void setThicknessFieldValue(float thickness)
	{
		thicknessField.removeChangeListener(this);
		thicknessField.setValue(Float.valueOf(thickness).doubleValue());
		thicknessField.addChangeListener(this);
	}
	
	
	
	public void setDotFieldValue(float width)
	{
		dotSizeField.removeChangeListener(this);
		dotSizeField.setValue(Float.valueOf(width).doubleValue());
		dotSizeField.addChangeListener(this);
	}
	
	
	/**
	 * Allows to set the attribute "isModified" in the drawPanel
	 * @param modified The new value of the attribute "isModified"
	 */
	public void setIsModified(boolean modified)
	{
		drawPanel.setIsModified(modified);
	}
	
	
	
	
	/**
	 * Allows to reinitialise some buttons
	 */
	public void reinitializeButtons()
	{
		updateUndoRedo();
		codePanel.newProject();
		Delimitor.setOpacity(Delimitor.DEFAULT_OPACITY);
		toolbar.getSliderDelimitor().setValue(Delimitor.DEFAULT_OPACITY);
		toolbar.getPersoGridGapField().setValue(MagneticGrid.DEFAULT_PERSONAL_GRID_GAP);
		menuBar.getDisplayGrid().setSelected(false);
		menuBar.getDisplayPersoGrid().setSelected(false);
		toolbar.getPersoGridGapField().setEnabled(false);
		toolbar.getMagneticCB().setSelected(false);
		rotationButtons.setSelected(false);
		mirrorButtons.setSelected(false);
		drawPanel.setOnRotation(false);
		preferencesFrame.readXMLPreferences();
		if(drawPropertiesFrame!=null)
			drawPropertiesFrame.setProperties();
		toolbar.getButtonImageSelect().setSelected(true);
		menuBar.getMenuSelect().setSelected(true);
		menuBar.getAutoAdjustBorders().setSelected(true);
		drawPanel.setMenuSelectedPopupGeneral(menuBar.getMenuSelect().getActionCommand(),true);
		rotateButton.setSelected(false);
		drawPanel.setOnRotation(false);
		setCurrentChoice(LaTeXDrawResources.LABEL_SELECT_FIGURE);
	}
	
	
	
	
	/**
	 * Allows to update buttons undo/redo
	 * (change their titles, ...)
	 */
	public void updateUndoRedo()
	{
		if(undoManager.isEmptyRedo())
		{
			toolbar.getButtonImageRedo().setToolTipText(UndoRedoManager.EMPTY_REDO);
			menuBar.getRedoMenu().setText(UndoRedoManager.EMPTY_REDO);
		}
		else
		{
			toolbar.getButtonImageRedo().setToolTipText(LaTeXDrawResources.LABEL_REDO +" " +undoManager.getTextNextRedo());//$NON-NLS-1$
			menuBar.getRedoMenu().setText(LaTeXDrawResources.LABEL_REDO +" " +undoManager.getTextNextRedo());//$NON-NLS-1$
		}
		
		if(undoManager.isEmptyUndo())
		{
			menuBar.getUndoMenu().setText(UndoRedoManager.EMPTY_UNDO);
			toolbar.getButtonImageUndo().setToolTipText(UndoRedoManager.EMPTY_UNDO);
		}
		else
		{
			menuBar.getUndoMenu().setText(LaTeXDrawResources.LABEL_UNDO +" " +undoManager.getTextNextUndo());//$NON-NLS-1$
			toolbar.getButtonImageUndo().setToolTipText(LaTeXDrawResources.LABEL_UNDO +" " +undoManager.getTextNextUndo());//$NON-NLS-1$
		}
		
		menuBar.getUndoMenu().setEnabled(!undoManager.isEmptyUndo());
		menuBar.getRedoMenu().setEnabled(!undoManager.isEmptyRedo());
		toolbar.getButtonImageRedo().setEnabled(!undoManager.isEmptyRedo());
		toolbar.getButtonImageUndo().setEnabled(!undoManager.isEmptyUndo());
	}
	
	
	
	
	/**
	 * Allows to update fields of the frame : if a figure is selected, its parameters are updated, else general parameters are used
	 * to set the fields.
	 */
	public void updateFrameFields()
	{	
		try
		{
			Draw listSelected = drawPanel.getDraw().getSelected();
			
			if(listSelected!=null && listSelected.size()==1)
			{
				Figure f = listSelected.getFigureAt(0);
				
				if(f!=null)
				{
					if(f instanceof Dot)
					{
						setDotFieldValue(((Dot)f).getWidth());
						setDotChoice(((Dot)f).getCurrentStyle());
						buttonInteriorColor.setVisible(f.canBeFilled());
					}
					else
					{
						if(f.canHaveArrow())
						{
		    				setLineArrowLChoice(((Arrowable)f).getArrow1Style());
		    				setLineArrowRChoice(((Arrowable)f).hasTwoLeftArrows() ? 
		    						ArrowHead.invertArrowStyle(((Arrowable)f).getArrow2Style()) : ((Arrowable)f).getArrow2Style());
						}
						
						setThicknessFieldValue(f.getThickness());	
						setLineChoice(f.getLineStyle());
						filledCheckBox.setSelected(f.canBeFilled() && f.isFilled());
					}
					
					buttonBordersColor.setColor(f.getLinesColor());
					setHatchChoice(f.getHatchingStyle());
					dbleBoundCheckBox.setSelected(f.isDoubleBoundaryable() && f.hasDoubleBoundary());
					shadowCheckBox.setSelected(f.hasShadow());
					updateFillCB();
					
					if(f.canHaveShadow())
						buttonShadowColor.setColor(f.getShadowColor());
					
					if(f.canBeFilled())
					{
						buttonBordersColor.setColor(f.getLinesColor());
						buttonInteriorColor.setColor(f.getInteriorColor());
						buttonGradientEndC.setColor(f.getGradientEndColor());
						buttonGradientStartC.setColor(f.getGradientStartColor());
					}
					
					if(f.canBeHatched())
						buttonHatchColor.setColor(f.getHatchingColor());
					
					if(f.isDoubleBoundaryable())
						buttonDbleBoundColor.setColor(f.getDoubleColor());
					
					if(f.isBordersMovable())
						setBorderChoice(f.getBordersPosition());
				}//if
			}
			else 
				if(listSelected!=null && listSelected.size()>1)
				{// We try to identify the figures' attributes that are equals.
					boolean ok;
					int i, size = listSelected.size();
					Color linesC = listSelected.getFigureAt(0).getLinesColor();
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i).isDoubleBoundaryable())
							ok = false;
					
					if(!ok)
					{
						boolean hasDbleBound = listSelected.getFigureAt(i-1).hasDoubleBoundary();
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).isDoubleBoundaryable() && 
								listSelected.getFigureAt(i).hasDoubleBoundary()!=hasDbleBound)
								ok = false;
						
						dbleBoundCheckBox.setSelected(ok ? hasDbleBound : genHasDbleBord);
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).hasDoubleBoundary())
								ok = false;
						
						if(!ok)
						{
							Color dbleC = listSelected.getFigureAt(i-1).getDoubleColor();
							
							for(i=0, ok=true; i<size && ok; i++)
								if(listSelected.getFigureAt(i).hasDoubleBoundary() && 
									!listSelected.getFigureAt(i).getDoubleColor().equals(dbleC))
									ok = false;
							
							genDbleBordCB.setColor(ok ? dbleC : genDbleC);
						}
						else
							genDbleBordCB.setColor(genDbleC);
					}
					else
						dbleBoundCheckBox.setSelected(genHasDbleBord);
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i) instanceof Arrowable)
							ok = false;
					
					if(!ok)
					{
						Arrowable arr = ((Arrowable)listSelected.getFigureAt(i-1));
						String arrow1S = arr.getArrow1Style();
						String arrow2S = arr.hasTwoLeftArrows() ? ArrowHead.invertArrowStyle(arr.getArrow2Style()) : arr.getArrow2Style();
						boolean ok2 = true;
						String str;
						
						for(i=0, ok=true; i<size; i++)
							if(listSelected.getFigureAt(i) instanceof Arrowable)
							{
								Arrowable a = (Arrowable)listSelected.getFigureAt(i);
								str = a.getArrow2Style();
								
								if(a.hasTwoLeftArrows())
									str = ArrowHead.invertArrowStyle(str);
								
								if(!a.getArrow1Style().equals(arrow1S))
									ok = false;
								
								if(!str.equals(arrow2S))
									ok2 = false;
							}
						
						setLineArrowLChoice(ok ? arrow1S : genLeftArrS);
						setLineArrowRChoice(ok2 ? arrow2S : genRightArrS);
					}
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i) instanceof Dot)
							ok = false;
				
					if(!ok)
					{
						String dotStyle = ((Dot)listSelected.getFigureAt(i-1)).getCurrentStyle();
						float width = ((Dot)listSelected.getFigureAt(i-1)).getWidth();
						boolean ok2 = true;
						
						for(i=0, ok=true; i<size; i++)
							if(listSelected.getFigureAt(i) instanceof Dot)
							{
								Dot d = (Dot)listSelected.getFigureAt(i);
								
								if(!d.getCurrentStyle().equals(dotStyle))
									ok = false;
							
								if(d.getWidth()!=width)
									ok2 = false;
							}
						
						setDotChoice(ok ? dotStyle : genDotS);
						setDotFieldValue(ok2 ? width : genDotSize);
					}
					else setLineChoice(genLineStyleS);
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i).isDashableOrDotable())
							ok = false;
				
					if(!ok)
					{
						String lineStyle = listSelected.getFigureAt(i-1).getLineStyle();
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).isDashableOrDotable() && 
								!listSelected.getFigureAt(i).getLineStyle().equals(lineStyle))
									ok = false;
						
						setLineChoice(ok ? lineStyle : genLineStyleS);
					}
					else setLineChoice(genLineStyleS);
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i).isBordersMovable())
							ok = false;
					
					if(!ok)
					{
						String bordPos = listSelected.getFigureAt(i-1).getBordersPosition();
					
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).isBordersMovable() && 
								!listSelected.getFigureAt(i).getBordersPosition().equals(bordPos))
									ok = false;
						
						setBorderChoice(ok ? bordPos : genBordPosS);
					}
					else setBorderChoice(genBordPosS);
					
					for(i=1; i<size && ok; i++)
						if(!listSelected.getFigureAt(i).getLinesColor().equals(linesC))
							ok = false;
					
					genLineCB.setColor(ok ? linesC : genLineC);
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i).isThicknessable())
							ok = false;
					
					if(!ok)
					{
						Float thickness = listSelected.getFigureAt(i-1).getThickness();
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).isThicknessable() && listSelected.getFigureAt(i).getThickness()!=thickness)
								ok = false;
					
						setThicknessFieldValue(ok ? thickness : genThickness);
					}
					else setThicknessFieldValue(genThickness);
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i).canHaveShadow())
							ok = false;
					
					if(!ok)
					{
						boolean hasShadow = listSelected.getFigureAt(i-1).hasShadow();
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).canHaveShadow() && listSelected.getFigureAt(i).hasShadow()!=hasShadow)
								ok = false;
						
						shadowCheckBox.setSelected(ok ? hasShadow : genHasShadow);
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).hasShadow())
								ok = false;
						
						if(!ok)
						{
							Color shadC = listSelected.getFigureAt(i-1).getShadowColor();
							
							for(i=0, ok=true; i<size && ok; i++)
								if(listSelected.getFigureAt(i).hasShadow() && !listSelected.getFigureAt(i).getShadowColor().equals(shadC))
									ok = false;
							
							genShadowCB.setColor(ok ? shadC : genShadowC);
						}
						else
							genShadowCB.setColor(genShadowC);
					}
					else
						shadowCheckBox.setSelected(genHasShadow);
					
					for(i=0, ok=true; i<size && ok; i++)
						if(listSelected.getFigureAt(i).canBeFilled())
							ok = false;
					
					if(!ok)
					{
						int j = i-1;
						boolean isFilled = listSelected.getFigureAt(j).isFilled();
						String hatchS = listSelected.getFigureAt(j).getHatchingStyle();
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).canBeFilled() && listSelected.getFigureAt(i).isFilled()!=isFilled)
								ok = false;
						
						filledCheckBox.setSelected(ok ? isFilled : genIsFilled);
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).canBeFilled() && 
								(listSelected.getFigureAt(i).hasGradient() || listSelected.getFigureAt(i).isHatched()) &&
								!listSelected.getFigureAt(i).getHatchingStyle().equals(hatchS))
								ok = false;
						
						setHatchChoice(ok ? hatchS : genHatchS);
						
						if(ok)
							if(listSelected.getFigureAt(j).hasGradient())
							{
								Color gradEndC = listSelected.getFigureAt(j).getGradientEndColor();
								Color gradStartC = listSelected.getFigureAt(j).getGradientStartColor();
								boolean okEnd = true;
								
								for(i=0, ok=true; i<size; i++)
									if(listSelected.getFigureAt(i).canBeHatched())
									{
										if(!listSelected.getFigureAt(i).getGradientEndColor().equals(gradEndC))
											okEnd = false;
										if(!listSelected.getFigureAt(i).getGradientStartColor().equals(gradStartC))
											ok = false;
									}
								
								genGradientStartCB.setColor(ok ? gradStartC : genGradientStartC);
								genGradientEndCB.setColor(okEnd ? gradEndC : genGradientEndC);
								genHatchCB.setColor(genHatchC);
							}
							else
							{
								Color hatchC = listSelected.getFigureAt(j).getHatchingColor();
								
								for(i=0, ok=true; i<size; i++)
									if(listSelected.getFigureAt(i).canBeHatched() && 
										!listSelected.getFigureAt(i).getHatchingColor().equals(hatchC))
										ok = false;
								
								genGradientStartCB.setColor(genGradientStartC);
								genGradientEndCB.setColor(genGradientEndC);
								genHatchCB.setColor(ok ? hatchC : genHatchC);
							}
						else
						{
							genGradientStartCB.setColor(genGradientStartC);
							genGradientEndCB.setColor(genGradientEndC);
							genHatchCB.setColor(genHatchC);
						}
						
						for(i=0, ok=true; i<size && ok; i++)
							if(listSelected.getFigureAt(i).isFilled())
								ok = false;
						
						if(!ok)
						{
							Color fillC = listSelected.getFigureAt(i-1).getInteriorColor();
							
							for(ok=true, i=0; i<size && ok; i++)
								if(listSelected.getFigureAt(i).isFilled() && !listSelected.getFigureAt(i).getInteriorColor().equals(fillC) &&
									!listSelected.getFigureAt(i).hasGradient())
									ok = false;
							
							genInteriorCB.setColor(ok ? fillC : genInteriorC);
						}
						else genInteriorCB.setColor(genInteriorC);
					}
					else 
					{
						genInteriorCB.setColor(genInteriorC);
						setHatchChoice(genHatchS);
						filledCheckBox.setSelected(genIsFilled);
						buttonGradientStartC.setColor(genGradientStartC);
						buttonGradientEndC.setColor(genGradientEndC);
						buttonHatchColor.setColor(genHatchC);
					}
				}
				else// In this case, the general values are used.
				{
					setBorderChoice(genBordPosS);
					setLineChoice(genLineStyleS);
					setDotChoice(genDotS);
					dbleBoundCheckBox.setSelected(genHasDbleBord);
					shadowCheckBox.setSelected(genHasShadow);
					setHatchChoice(genHatchS);
					filledCheckBox.setSelected(genIsFilled);
					setLineArrowLChoice(genLeftArrS);
					setLineArrowRChoice(genRightArrS);
					setThicknessFieldValue(genThickness);
					genDbleBordCB.setColor(genDbleC);
					genGradientEndCB.setColor(genGradientEndC);
					genGradientStartCB.setColor(genGradientStartC);
					genInteriorCB.setColor(genInteriorC);
					genLineCB.setColor(genLineC);
					genHatchCB.setColor(genHatchC);
					genShadowCB.setColor(genShadowC);
				}
		}catch(Exception ex)
		{
			ex.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	
	
	/**
	 * Allows to know if the user wants to auto adjust the 
	 * borders of the drawing.
	 * @return True if the user wants it
	 */
	public boolean isAutoAdjustBorders()
	{
		return getLMenuBar().getAutoAdjustBorders().isSelected();
	}
	
	
	
	
	/**
	 * Allows to know if the X-scale must be displayed
	 * @return True if the X-scale must be displayed
	 */
	public boolean isXScaleDisplayed()
	{
		return getLMenuBar().getDisplayXScale().isSelected();
	}
	
	

	/**
	 * Allows to know if the Y-scale must be displayed
	 * @return True if the Y-scale must be displayed
	 */
	public boolean isYScaleDisplayed()
	{
		return getLMenuBar().getDisplayYScale().isSelected();
	}
	
	
	
	
	/**
	 * Allows to know if the the borders of the drawing must be displayed
	 * @return True if the the borders of the drawing must be displayed
	 */
	public boolean isBordersDisplayed()
	{
		return getLMenuBar().getDisplayBorders().isSelected();
	}
	
	
	
	/**
	 * Allows to know if the codePanel must be displayed
	 * @return True if the codePanel must be displayed
	 */
	public boolean isCodePanelDisplayed()
	{
		if(codePanel==null)
			return false;
		return codePanel.isVisible();
	}
	
	
	
	
	
	public void actionPerformed(ActionEvent e) 
	{ 
    	Object src = e.getSource();

    	if(src instanceof JMenuItem || src instanceof JButton || src instanceof JToggleButton || src instanceof JCheckBox)
    	{
    		AbstractButton ab = ((AbstractButton)src);
    		String label = ab.getActionCommand();
			Draw listSelected = drawPanel.getDraw().getSelected();
			boolean ok = ab.isSelected();

			try
			{
	    		if(label.equals(LaTeXDrawResources.LABEL_FILLED_CHECKBOX))
	    		{
					if(listSelected!=null && listSelected.size()>0)
						addToUndoTransformation(listSelected);
								
					if(listSelected!=null && listSelected.size()==1)
					{
						Figure selected = listSelected.getFigureAt(0);
						selected.setIsFilled(ok && selected.canBeFilled());
						buttonInteriorColor.setVisible(((ok && selected.canBeFilled()) || selected.hasShadow()));
						buttonInteriorColor.setColor(selected.getInteriorColor());
					}
					else
					{
						genIsFilled = ok;
						
						if(listSelected!=null)
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.canBeFilled())
									f.setIsFilled(ok);
					}
					
					drawPanel.updateDraw(true);
					drawPanel.setIsModified(true);
					requestFocus();
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_DBLE_BOUND_CHECKBOX))
	    		{
					if(listSelected!=null && listSelected.size()>0)
						addToUndoTransformation(listSelected);
								
					if(listSelected!=null && listSelected.size()==1)
					{
						Figure selected = listSelected.getFigureAt(0);
						selected.setHasDoubleBoundary(ok && selected.isDoubleBoundaryable());
						buttonDbleBoundColor.setVisible(ok && selected.isDoubleBoundaryable());
					}
					else
					{
						genHasDbleBord = ok;
						
						if(listSelected!=null)
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.isDoubleBoundaryable())
									f.setHasDoubleBoundary(ok);
					}
					
					if(listSelected!=null) listSelected.updateBorders();
					drawPanel.updateDraw(true);
					drawPanel.setIsModified(true);
					requestFocus();
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_SHADOW_CHECKBOX))
	    		{
					if(listSelected!=null && listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(listSelected!=null && listSelected.size()==1)
					{
						Figure selected = listSelected.getFigureAt(0);
						selected.setHasShadow(ok && selected.canHaveShadow());
						buttonShadowColor.setVisible(ok && selected.canHaveShadow());
						buttonInteriorColor.setVisible((ok && selected.canHaveShadow() && 
								selected.canBeFilled() && selected.shadowFillsShape() && 
								!selected.hasGradient()) || (!ok && selected.isFilled()));
						updateFillCB();
					}
					else
					{
						genHasShadow = ok;
						
						if(listSelected!=null)
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.canHaveShadow())
									f.setHasShadow(ok);
					}
					
					if(listSelected!=null) listSelected.updateBorders();
					drawPanel.updateDraw(true);
					drawPanel.setIsModified(true);
					requestFocus();
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_ROTATE))
	    		{	
	    			drawPanel.setOnRotation(!drawPanel.isOnRotation());
	    			rotateButton.setSelected(drawPanel.isOnRotation());
	    			drawPanel.
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_MAGNETIC_CB))
	    		{	
	    			drawPanel.setGridIsMagnetic(toolbar.getMagneticCB().isSelected());
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.compareTo(LaTeXDrawResources.LABEL_COLOR_HATCH)==0)
	    		{
	    			 if(listSelected==null)
	    				 return ;
	    			 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.34"), buttonHatchColor.getColor()); //$NON-NLS-1$

					if(c!=null)
					{
						if(listSelected.size()>0)
							addToUndoTransformation(listSelected);
						
						if(listSelected.size()==1)
						{
							buttonHatchColor.setColor(c);
							listSelected.getFigureAt(0).setHatchingColor(c);
						}
						else
						{
							genHatchCB.setColor(c);
							genHatchC = c;
							
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.canBeHatched())
									f.setHatchingColor(c);
						}
						
						drawPanel.updateDraw(true);
						drawPanel.setIsModified(true);
					}
					requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.compareTo(LaTeXDrawResources.LABEL_COLOR_GRADIENT_FIRST)==0)
	    		{
	    			 if(listSelected==null)
	    				 return ;
	    			 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getString1_7("LaTeXDrawFrame.7"), buttonGradientStartC.getColor()); //$NON-NLS-1$

					if(c!=null)
					{
						if(listSelected.size()>0)
							addToUndoTransformation(listSelected);
							
						if(listSelected.size()==1)
						{
							buttonGradientStartC.setColor(c);
							listSelected.getFigureAt(0).setGradientStartColor(c);
						}
						else
						{
							genGradientStartCB.setColor(c);
							genGradientStartC = c;
							
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.canBeHatched())
									f.setGradientStartColor(c);
						}
						
						drawPanel.updateDraw(true);
						drawPanel.setIsModified(true);
					}
					
					requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.compareTo(LaTeXDrawResources.LABEL_COLOR_GRADIENT_SECOND)==0)
	    		{
	    			 if(listSelected==null)
	    				 return ;
	    			 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getString1_7("LaTeXDrawFrame.8"), buttonGradientEndC.getColor()); //$NON-NLS-1$
					
					if(c!=null)
					{
						if(listSelected.size()>0)
							addToUndoTransformation(listSelected);
						
						if(listSelected.size()==1)
						{
							buttonGradientEndC.setColor(c);
							listSelected.getFigureAt(0).setGradientEndColor(c);
						}
						else
						{
							genGradientEndCB.setColor(c);
							genGradientEndC = c;
							
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.canBeHatched())
									f.setGradientEndColor(c);
						}
						
						drawPanel.updateDraw(true);
						drawPanel.setIsModified(true);
					}
					
					requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.compareTo(LaTeXDrawResources.LABEL_COLOR_DBLE_BOUND)==0)
	    		{
	    			 if(listSelected==null)
	    				 return ;
	    			 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.151"), buttonDbleBoundColor.getColor()); //$NON-NLS-1$
					
					if(c!=null)
					{
						if(listSelected.size()>0)
							addToUndoTransformation(listSelected);
						
						if(listSelected.size()==1)
						{
							buttonDbleBoundColor.setColor(c);
							Figure selected = listSelected.getFigureAt(0);
							selected.setDoubleColor(c);
						}
						else
						{
							genDbleBordCB.setColor(c);
							genDbleC = c;
							
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.isDoubleBoundaryable())
									f.setDoubleColor(c);
						}
						drawPanel.updateDraw(true);
						drawPanel.setIsModified(true);
					}
					
					requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.compareTo(LaTeXDrawResources.NAME_COLOR_SHADOW)==0)
	    		{
	    			 if(listSelected==null)
	    				 return ;
	    			 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
								LaTeXDrawLang.getString1_7("LaTeXDrawFrame.9"), buttonShadowColor.getColor());  //$NON-NLS-1$
					
					if(c!=null)
					{
						if(listSelected.size()>0)
							addToUndoTransformation(listSelected);
						
						if(listSelected.size()==1)
						{
							buttonShadowColor.setColor(c);
							listSelected.getFigureAt(0).setShadowColor(c);
						}
						else
						{
							genShadowCB.setColor(c);
							genShadowC = c;
							
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.canHaveShadow())
									f.setShadowColor(c);
						}
						
						drawPanel.updateDraw(true);
						drawPanel.setIsModified(true);
					}
					requestFocus();
	    			return ;
	    		}
	    		
	    		 if(label.compareTo(LaTeXDrawResources.LABEL_COLOR_BUTTON)==0)
	    		 {
	    			 if(listSelected==null)
	    				 return ;
	    			 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
									LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.36"), ((ColorButton)src).getColor()); //$NON-NLS-1$

					if(listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(c!=null)
					{
						if(listSelected.size()==1)
							listSelected.getFigureAt(0).setLinesColor(c);
						else
						{
							genLineC = c;
							
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable())
									f.setLinesColor(c);
						}
						
						drawPanel.updateDraw(true);
						drawPanel.setIsModified(true);
						((ColorButton)src).setColor(c);
					}
					
					requestFocus();
					return ;
				 }
	    		 
	    		 
	    		 if(label.compareTo(LaTeXDrawResources.LABEL_COLOR_INTERIOR_BUTTON)==0)
	    		 {
	    			 if(listSelected==null)
	    				 return ;
	    			 
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
								LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.37"), ((ColorButton)src).getColor()); //$NON-NLS-1$
						
					if(c!=null)
					{
						if(listSelected.size()>0)
							addToUndoTransformation(listSelected);
							
						if(listSelected.size()==1)
							listSelected.getFigureAt(0).setInteriorColor(c);
						else
						{
							genInteriorC = c;
							
							for(Figure f : listSelected.getFigures())
								if(f.isCustomizable() && f.canBeFilled())
									f.setInteriorColor(c);
						}
						
						drawPanel.updateDraw(true);
						drawPanel.setIsModified(true);
						((ColorButton)src).setColor(c);
					}
					requestFocus();
					return ;
				 }
	    		 
	    		 if(label.equals(LaTeXDrawResources.LABEL_JOIN))
	    		 {
	    			 drawPanel.joinSelection(undoManager);
	    			 updateUndoRedo();
	    			 setSelection(true);
	    			 setIsModified(true);
	    			 setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.154")); //$NON-NLS-1$
	    			 requestFocus();
	    			 return;
	    		 }
	    		
	    		 if(label.equals(LaTeXDrawResources.LABEL_SEPARATE))
	    		 {
	    			 drawPanel.separateSelection(undoManager);
	    			 updateUndoRedo();
	    			 setSelection(true);
	    			 setIsModified(true);
	    			 setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.155")); //$NON-NLS-1$
	    			 requestFocus();
	    			 return ;
	    		 }
	    		 
	    		if(label.equals(LaTeXDrawResources.LABEL_BEHIND_SEL))
	    		{
	    			setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
	    			drawPanel.setOnWaitingClicDispositionBehind(true);
	    			requestFocus();
	    			return;
	    		}   	
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_FRONT_SEL))
	    		{
	    			setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
	    			drawPanel.setOnWaitingClicDispositionFront(true);
	    			requestFocus();
	    			return;
	    		}   	
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_BACKGROUND))
	    		{
	    			drawPanel.selectedInBackground();
	    			setIsModified(true);
	    			requestFocus();
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_FOREGROUND))
	    		{
	    			drawPanel.selectedInForeground();
	    			setIsModified(true);
	    			requestFocus();
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_ROTATE_90))
	    		{
	    			drawPanel.draw.rotateSelection(Math.PI/2.);
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_ROTATE_180))
	    		{
	    			drawPanel.draw.rotateSelection(Math.PI);
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_ROTATE_270))
	    		{
	    			drawPanel.draw.rotateSelection(3*Math.PI/2.);
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_MIRROR_H))
	    		{
	    			drawPanel.draw.mirrorHorizontal();
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_MIRROR_V))
	    		{
	    			drawPanel.draw.mirrorVertical();
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_TEX_EDITOR))
	    		{
	    			String title = LaTeXDrawLang.getString1_8("LaTeXDrawFrame.19"); //$NON-NLS-1$
	    			
	    			if(pathTexEditor==null || pathTexEditor.length()==0)
	    				JOptionPane.showMessageDialog(this, LaTeXDrawLang.getString1_8("LaTeXDrawFrame.20"),  //$NON-NLS-1$
	    			 		   						title, JOptionPane.WARNING_MESSAGE); 
	    			else
		    			try{Runtime.getRuntime().exec(pathTexEditor);}
		    			catch(Exception ex)
		    			{
		    				JOptionPane.showMessageDialog(this, 
	    			 		   "<html>"+LaTeXDrawLang.getString1_8("LaTeXDrawFrame.21")+"<br>"+//$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
	    			 		   LaTeXDrawLang.getString1_8("LaTeXDrawFrame.22")+"</html>", //$NON-NLS-1$//$NON-NLS-2$
	    			 		   title, JOptionPane.WARNING_MESSAGE); 
		    			}
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_TEMPLATE_UPDATE))
	    		{
	    			menuBar.getMenuTemplate().updateTemplates();
	    			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.160")); //$NON-NLS-1$
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_COMMENTS))
	    		{
    				addCommentsFrame.setVisible(true);
	    			return;
	    		}
	    		
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_PREFERENCES))
	    		{
	    			if(!preferencesFrame.isVisible())
	    				preferencesFrame.setVisible(true);
	    			
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_INSERT_CODE))
	    		{
	    			if(insertCodeFrame==null)
	    				insertCodeFrame = new InsertPSTricksCodeFrame(this);
	    			
	    			insertCodeFrame.setVisible(true);
	    			requestFocus();
	    			return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_DRAW_PROPERTIES))
	    		{
	    			if(drawPropertiesFrame==null)
	    				drawPropertiesFrame = new DrawPropertiesFrame(this);
	    			
	    			drawPropertiesFrame.setVisible(true);
					return ;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_HELP))
	    		{
	    			if(helpFrame==null)
	    				createHelpFrame();
	    			
	    			if(helpFrame!=null)
	    				helpFrame.setVisible(true);
	    			
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_DISPLAY_GRID))
	    		{
	    			if(menuBar.getDisplayGrid().isSelected())
	    			{
	    				menuBar.getDisplayPersoGrid().setSelected(false);
	    				toolbar.getPersoGridGapField().setEnabled(false);
	    			}
	    			
	    			displayGrid(menuBar.getDisplayGrid().isSelected(), menuBar.getDisplayGrid().isSelected());
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_DISPLAY_PERSO_GRID))
	    		{
	    			if(menuBar.getDisplayPersoGrid().isSelected())
	    			{
	    				menuBar.getDisplayGrid().setSelected(false);
	    				toolbar.getPersoGridGapField().setEnabled(true);
	    			}
	    			
	    			displayGrid(menuBar.getDisplayPersoGrid().isSelected(), !menuBar.getDisplayPersoGrid().isSelected());
	    			return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_ABOUT))
	    		{
	    			if(aboutTeXDraw==null)
	    				createAboutLaTeXDrawFrame();
	    			
	    			aboutTeXDraw.setVisible(true);
		    		return;
	    		}
	    		
	    		if(label.equals(LaTeXDrawResources.LABEL_DISPLAY_CODE_PANEL))
	    		{
	    			if(!(e.getSource() instanceof JCheckBoxMenuItem))
	    				menuBar.getDisplayCodePanel().setSelected(!menuBar.getDisplayCodePanel().isSelected());
	    			
	    			displayCodePanel(menuBar.getDisplayCodePanel().isSelected());
	    			requestFocus();
	    			return;
	    		}
	    		
	    		if(label.compareTo(LaTeXDrawResources.TEMPLATE_START_ACTION_CMD)>0)
	    		{
	    			String str = label.substring(LaTeXDrawResources.TEMPLATE_START_ACTION_CMD.length());
	    			boolean fromShared = str.charAt(0)=='0' ? false : true;
	    			String path;
	    			String name = str.substring(1) + SVGFilter.SVG_EXTENSION;
	    			
	    			if(fromShared)
	    				path = LaTeXDrawPath.PATH_TEMPLATES_SHARED==null ? null : LaTeXDrawPath.PATH_TEMPLATES_SHARED+File.separator+name;
	    			else
	    				path = LaTeXDrawPath.PATH_TEMPLATES_DIR_USER+File.separator+name;
	    			
	    			Draw template = SVGDocumentGenerator.toLatexdraw(new File(path), null, null);
	    			
	    			if(template!=null)
	    			{
		    			template.setDrawFigures(true);
		    			template.setSelected(false);
		    			
		    			if(template.size()==0)
		    				return ;
		    			
	    				drawPanel.getDraw().addFigure(template.size()>1 ? template : template.getFigureAt(0), undoManager);
		    			updateUndoRedo();
		    			drawPanel.updateDraw(true);
	    			}
	    			else
	    			{
	    				JOptionPane.showMessageDialog(this, 
		    			 		   "<html>"+LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.19")+"<br>"+LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.20")+"</html>",//$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$//$NON-NLS-5$
		    			 		   LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.21"), JOptionPane.WARNING_MESSAGE);  //$NON-NLS-1$
	    			}
	    			requestFocus();
	    			return ;
	    		}
			}
			catch(Exception ex)
			{
				ex.printStackTrace(); 
				ExceptionFrameDialog.showExceptionDialog(ex);
			}
    	} // if(src instanceof JMenuItem || src instanceof JButton)
	}
	
	
	
	
	/**
	 * To launch when the selected figures are transformed : add an undo.
	 * @param listSelected THe list of the selected figures.
	 * @since 1.9.1
	 */
	private void addToUndoTransformation(Draw listSelected)
	{
			try
			{
				Object clone = listSelected.clone();
				Vector<Figure> figures = drawPanel.draw.getFigures();
				int i, size = listSelected.size();
				int[] id = new int[size];
				
				for(i=0; i<size; i++)
					id[i] = figures.indexOf(listSelected.getFigureAt(i));
				
				undoManager.add(UndoRedoManager.LABEL_TRANSFORM, id, clone, false);
				updateUndoRedo();
				
			}catch(CloneNotSupportedException e)
			{
				e.printStackTrace();
				ExceptionFrameDialog.showExceptionDialog(e);
			}
	}


	
	/**
	 * Allows to print the draw
	 */
	public void printDraw()
	{
		PrinterJob printJob = PrinterJob.getPrinterJob();
		printJob.setPrintable(drawPanel.getDraw());

		System.setProperty("sun.java2d.print.pipeline", "pdl");//$NON-NLS-1$ //$NON-NLS-2$

		if (printJob.printDialog()) 
		{
			try 
			{ 
				setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.162")); //$NON-NLS-1$
				printJob.print(); 
				setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.163")); //$NON-NLS-1$
			}
			catch(Exception e) 
			{ 
				setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.164")); //$NON-NLS-1$
				e.printStackTrace();
				ExceptionFrameDialog.showExceptionDialog(e);
			}
		}
	}
	

	
	
	/**
	 * Allows to print the code generated.
	 */
	public void printCode()
	{
		drawPanel.updateCode();
		
		PrintRequestAttributeSet pras = new HashPrintRequestAttributeSet();
		DocFlavor flavor =  DocFlavor.BYTE_ARRAY.AUTOSENSE;
		PrintService printService[] = PrintServiceLookup.lookupPrintServices(null, null);
		PrintService defaultService = PrintServiceLookup.lookupDefaultPrintService();
		
		if(printService!=null && printService.length!=0)
		{
			if(defaultService==null)
				defaultService = printService[0];
			
			PrintService service = ServiceUI.printDialog(null, 200, 200, printService, defaultService, null, pras);
			
			if(service != null)
			{
				DocPrintJob job = service.createPrintJob();
				Doc doc = new SimpleDoc(codePanel.getText().getBytes(), flavor, null);
				
				try { job.print(doc, pras); }
				catch(PrintException ex) { JOptionPane.showMessageDialog(this, LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.22")); } //$NON-NLS-1$
			}
		}
		else
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.23")); //$NON-NLS-1$
	}
	
	
	
	/**
	 * Allows to export the selected figures as a template in a file.
	 */
	public void exportAsTemplate()
	{
		int res;
		String templateName = null;
		boolean ok;
		File file = null;
		
		setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.168")); //$NON-NLS-1$
		
		try
		{
			do
			{
				templateName = JOptionPane.showInputDialog(this, 
						LaTeXDrawLang.getOthersString("DrawContainer.nameTemplate"));  //$NON-NLS-1$
				
				if(templateName==null || templateName.equals(""))//$NON-NLS-1$
					ok = false;
				else
				{
					file = new File(LaTeXDrawPath.PATH_TEMPLATES_DIR_USER + File.separator + templateName + SVGFilter.SVG_EXTENSION);
				
					if(!file.exists())
						ok = true;
					else
					{
						res = JOptionPane.showConfirmDialog(this, 
								LaTeXDrawLang.getOthersString("DrawContainer.overwriteTemplate"), //$NON-NLS-1$
								LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.42"), JOptionPane.YES_NO_OPTION); //$NON-NLS-1$
					
						ok = res==JOptionPane.YES_OPTION ? true : false;
					}
				}
			}while(!ok);
			
			Draw selection = getDrawPanel().getDraw().getSelected();
			
			if(file!=null)
				SVGDocumentGenerator.exportAsSVG(file.getPath(), selection);
			else
				templateName = null;
		}
		catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
		
		if(templateName!=null)
		{
			menuBar.getMenuTemplate().addItem(templateName);
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.169")); //$NON-NLS-1$
		}
		else 
			setStatusBarText("");//$NON-NLS-1$
	}
	
	
	
	/**
	 * Allows to insert several figures from PSTricks code into the current draw
	 * @param code The code to parse and to convert in figures
	 */
	public void insertPSTricksCode(String code)
	{
		try
		{
			String code2 = code;
			
			if(code2!=null && code2.length()!=0)
			{
				if(PSTricksParser.containsBeginPSPicture(code2)==0)
				{
					String eol = System.getProperty("line.separator"); //$NON-NLS-1$
					code2 = "\\begin{pspicture}(10,10)"+eol+code2+eol+"\\end{pspicture}";//$NON-NLS-1$ //$NON-NLS-2$
				}
				
				if(progressbarPST==null)
					progressbarPST = new PSTProgressBarManager(this);
				
				progressbarPST.setCodeToParse(code2);
				progressbarPST.startProgressBar();
			}
		}catch (Exception e) 
		{ 
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	/**
	 * Imports a TeXFile.
	 */
	public void importPSTricksFile()
	{
		currentFile = null;
		projectName = null;
		setTitle();
		try
		{
			if(fileChooserImport==null)	
			{
				fileChooserImport = new PictureChooser(this, LaTeXDrawLang.getString1_6("LaTeXDrawFrame.16"), //$NON-NLS-1$
						false, LaTeXDrawLang.getString1_6("LaTeXDrawFrame.17")); //$NON-NLS-1$

				JFileChooser fileChooser = fileChooserImport.getFileChooser();
				if(pathExport!=null)
					fileChooser.setCurrentDirectory(new File(pathExport));
		 		fileChooser.setApproveButtonText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.199")); //$NON-NLS-1$
		 		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		 		fileChooser.setAcceptAllFileFilterUsed(false);
		 		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		 		fileChooser.addChoosableFileFilter(new TeXFilter());
		 		fileChooser.setFileHidingEnabled(true);
		 		fileChooser.setApproveButtonText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.171")); //$NON-NLS-1$
		 		fileChooser.setMultiSelectionEnabled(false);
			}
			
			File file = fileChooserImport.displayFrame();
			
			if(file==null) return;

			importPSTricksFile(file.getAbsolutePath());
			
		}catch (Exception e) 
		{ 
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	/**
	 * Allows to import a TeXFile with the filename given.
	 */
	public boolean importPSTricksFile(String path)
	{
		try
		{
			File file = new File(path);
			
			if(!file.exists() || !file.canRead() || !file.isFile())
				return false;
			
			setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.15")); //$NON-NLS-1$
	
			reinitializeButtons();
			preferencesFrame.addRecentFile(file.getAbsolutePath());
			menuBar.updateRecentFilesMenu();
			setIsModified(false);
			newProject();
			undoManager = new UndoRedoManager(drawPanel.getDraw().getFigures(), drawPanel.getDraw(), this);
			updateUndoRedo();
			
			if(progressbarPST==null)
				progressbarPST = new PSTProgressBarManager(this);
			
			progressbarPST.setFileToParse(file);
			progressbarPST.startProgressBar();
			
			return true;
		}
		catch (Exception e) 
		{ 
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
			return false;
		}
	}
	
	
	
	
	/**
	 * Allows to export the pstricks code as a .tex file
	 */
	public void exportAsPsTricks()
	{
		setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.170")); //$NON-NLS-1$
		
		if(fileChooserExport==null)	
			fileChooserExport = new ExportDialog(currentFile==null ? pathExport : currentFile.getPath()); 	
		
		fileChooserExport.removeChoosableFileFilter(fileChooserExport.getFileFilter());
		fileChooserExport.setFileFilter(fileChooserExport.getAcceptAllFileFilter());
		fileChooserExport.setFileFilter(new TeXFilter());
		
		if(currentFile==null)
			fileChooserExport.setSelectedFile(null);
		else
		{
			String fPath = currentFile.getPath();
			
			if(fPath.endsWith(SVGFilter.SVG_EXTENSION))
				fPath = fPath.substring(0, fPath.indexOf(SVGFilter.SVG_EXTENSION));
			
			fileChooserExport.setSelectedFile(new File(fPath));
		}
		
		
		fileChooserExport.setDialogTitle(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.172"));    //$NON-NLS-1$

		int res = fileChooserExport.showSaveDialog(this);
		File file = fileChooserExport.getSelectedFile();
		
		if(res != JFileChooser.APPROVE_OPTION || file==null) 
		{
			setStatusBarText("");//$NON-NLS-1$
			return;
		}
		
		projectName = file.getName();
		String ex = TeXFilter.TEX_EXTENSION;
		int id = projectName.toLowerCase().indexOf(ex.toLowerCase());
		
		if(id ==-1)
			file = new File(file.getPath() + TeXFilter.TEX_EXTENSION);
			
		if(file.exists())
		{ 
			int replace = JOptionPane.showConfirmDialog(this,
  					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.173"), //$NON-NLS-1$
					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.174"), JOptionPane.YES_NO_OPTION); //$NON-NLS-1$
				
			
			if(replace == JOptionPane.NO_OPTION) 
				return; // The user doesn't want to replace the file
		}
		
		PrintWriter out=null;
		
		try // We export the code
		{
			FileWriter fw = new FileWriter(file);
			BufferedWriter bw = new BufferedWriter(fw);
			out  = new PrintWriter(bw);

			drawPanel.updateCode();
			out.println(codePanel.getText());
			out.close();
			bw.close();
			fw.close();
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.175")); //$NON-NLS-1$
		}
		catch (Exception e) 
		{ 
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.176")); //$NON-NLS-1$
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
		
		if(out!=null)
			out.close();
		setTitle();	
	}
	
	
	
	
	public void exportAsPicture(String format)
	{
		if(format==null)
			return ;
		
		boolean crop = format.equals(LaTeXDrawResources.LABEL_EXPORT_PDF_LATEX_CROP);
		setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.177")); //$NON-NLS-1$
		String titleFramesExport = "Exportation of the drawing";
		
		if(fileChooserExport==null)	
			fileChooserExport = new ExportDialog(currentFile==null ? pathExport : currentFile.getPath());
		
		fileChooserExport.removeChoosableFileFilter(fileChooserExport.getFileFilter());
		fileChooserExport.setFileFilter(fileChooserExport.getAcceptAllFileFilter());
		
		if(format.equals(LaTeXDrawResources.LABEL_EXPORT_EPS))
			fileChooserExport.setFileFilter(new EPSFilter());
		else
			if(format.equals(LaTeXDrawResources.LABEL_EXPORT_JPG))
				fileChooserExport.setFileFilter(new JPGFilter());
			else
				if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PNG))
					fileChooserExport.setFileFilter(new PNGFilter());
				else
					if(format.equals(LaTeXDrawResources.LABEL_EXPORT_BMP))
						fileChooserExport.setFileFilter(new BMPFilter());
					else
						if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PPM))
							fileChooserExport.setFileFilter(new PPMFilter());
						else
							if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PDF_LATEX) || crop)
								fileChooserExport.setFileFilter(new PDFFilter());
							else
								if(format.equals(LaTeXDrawResources.LABEL_EXPORT_EPS_LATEX))
									fileChooserExport.setFileFilter(new PSFilter());
								else
									return ;
		
		if(currentFile==null)
			fileChooserExport.setSelectedFile(null);
		else {
			String fPath = currentFile.getPath();
			
			if(fPath.endsWith(SVGFilter.SVG_EXTENSION))
				fPath = fPath.substring(0, fPath.indexOf(SVGFilter.SVG_EXTENSION));
			
			fileChooserExport.setSelectedFile(new File(fPath));
		}
		
		fileChooserExport.setDialogTitle(titleFramesExport);
		
		int response = fileChooserExport.showSaveDialog(this);
		File file = fileChooserExport.getSelectedFile();
		
		if(response != JFileChooser.APPROVE_OPTION || file==null)
			return;
		
		String name = file.getName(), ex;
		
		if(format.equals(LaTeXDrawResources.LABEL_EXPORT_EPS)) 
			ex = LaTeXDrawResources.EPS_EXTENSION;
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_JPG)) 
			ex = LaTeXDrawResources.JPG_EXTENSION;
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_BMP))
			ex = LaTeXDrawResources.BMP_EXTENSION;
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PNG))
			ex = LaTeXDrawResources.PNG_EXTENSION;
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PPM))
			ex = LaTeXDrawResources.PPM_EXTENSION;
		else if(format.equals(LaTeXDrawResources.LABEL_EXPORT_PDF_LATEX) || crop)
			ex = PDFFilter.PDF_EXTENSION;
		else
			ex = PSFilter.PS_EXTENSION;
		
		int id = name.toLowerCase().indexOf(ex.toLowerCase());
		
		if(id ==-1)
			file = new File(file.getPath() + ex);			
			
		if(file.exists()) { 
			int replace = JOptionPane.showConfirmDialog(this,
  					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.173"), //$NON-NLS-1$
  					titleFramesExport, JOptionPane.YES_NO_OPTION);
			
			if(replace == JOptionPane.NO_OPTION) 
				return; // The user doesn't want to replace the file
		}
		
		
		try {
			int compressionRate = fileChooserExport.getCompressionRate();
			drawPanel.exportAsPicture(file, format, compressionRate==-1 ? ExportDialog.DEFAULT_COMPRESSION_RATE :
													1.f-compressionRate/100.f);
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.184")); //$NON-NLS-1$
		}
		catch(Exception e) {
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.185")); //$NON-NLS-1$
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
    /** 
     * Shutdown procedure when run as an application.
     */
    protected void windowClosed() 
    {
    	System.exit(0);
    }


    

	public void stateChanged(ChangeEvent e)
	{
	   	Object src = e.getSource();
	   	String name = ((Component)src).getName();   

    	if(src instanceof JSpinner)
    	{
    		Draw selection = drawPanel.getDraw().getSelected();

    		if(name.equals(LaTeXDrawResources.NAME_THICKNESS_FIELD))
    		{
    			float thick = Float.valueOf(thicknessField.getValue().toString()).floatValue();

    			if(selection!=null && selection.size()==1)
    				selection.getFigureAt(0).setThickness(thick);
    			else
    			{
    				if(selection!=null)
	    				for(Figure f : selection.getFigures())
	    					if(f.isCustomizable() && f.isThicknessable())
	    						f.setThickness(thick);

    				genThickness = thick;
    			}

    			if(selection!=null) selection.updateBorders();
				drawPanel.updateDraw(true);
				drawPanel.setIsModified(true);
				thicknessField.requestFocus();
				
    			return;
    		}

    		if(name.equals(LaTeXDrawResources.NAME_DOT_SIZE_FIELD))
    		{
    			float size = Float.valueOf(dotSizeField.getValue().toString()).floatValue();

    			if(selection!=null && selection.size()==1 && selection.getFigureAt(0) instanceof Dot)
    				((Dot)selection.getFigureAt(0)).setWidth(size);
    			else
    			{
    				if(selection!=null)
	    				for(Figure f : selection.getFigures())
	    					if(f instanceof Dot)
	    						((Dot)f).setWidth(size);

    				genDotSize = size;
    			}

    			if(selection!=null) selection.updateBorders();
				drawPanel.updateDraw(true);
				drawPanel.setIsModified(true);
				dotSizeField.requestFocus();
				
    			return;
    		}
    		
    		if(name.equals(LaTeXDrawResources.LABEL_PERSO_GRID_GAP_FIELD))
    		{
    			drawPanel.getDraw().setPersonalGridGap(Integer.valueOf(toolbar.getPersoGridGapField().getValue().toString()).intValue());
    			drawPanel.setIsModified(true);
    			return ;
    		}
    		
    		if(name.equals(LaTeXDrawResources.LABEL_ZOOM_CHANGE))
    		{
    			int zoom = Integer.MAX_VALUE;
    				
    			try { zoom = Double.valueOf(((JSpinner)src).getValue().toString()).intValue(); }
    			catch(Exception ex) { ex.printStackTrace(); }
    			
    			if(zoom!=Integer.MAX_VALUE)
    				drawPanel.getDraw().setZoom(zoom/100.);
    			
    			return ;
    		}
    		
    		return ;
    	}
    	
    	if(src instanceof JSlider)
    	{
    		JSlider slider = (JSlider)src;
    		
    		if(name.equals(LaTeXDrawResources.NAME_SLIDER_DELIMITOR))
    		{
    			if(Delimitor.getOpacity()!=slider.getValue())
    			{
	    			Delimitor.setOpacity(slider.getValue());
	    			drawPanel.setIsModified(true);
	    			drawPanel.draw.repaint();
    			}
    			
    			return ;
    		}
    		return ;
    	}
	}
	
	
	
	
	
	/**
	 * Allows to save the current project in as an SVG picture.
	 */
	public boolean save(boolean saveAs)
	{
		if(!saveAs && !drawPanel.isModified())
			return true;
		
		File file;
		boolean addToRecent = false;
		
		// If the user want to "save as" or if he hasn't ever
		// save his project, we display a dialog box in order to select a file
		if(saveAs || currentFile==null)
		{
			if(fileChooserSave==null)		
			{
				fileChooserSave = new JFileChooser(pathOpen);		    	
				fileChooserSave.setFileSelectionMode(JFileChooser.FILES_ONLY);		   	
				fileChooserSave.setDialogTitle(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.188"));//$NON-NLS-1$
				fileChooserSave.addChoosableFileFilter(new SVGFilter());
				fileChooserSave.setMultiSelectionEnabled(false);
			} 	
	    	
			fileChooserSave.setSelectedFile(currentFile);
			int response = fileChooserSave.showSaveDialog(this);

			if(response != JFileChooser.APPROVE_OPTION)
				return false;
			
			file = fileChooserSave.getSelectedFile();
			
			if(file==null) 
				return false;
			
			projectName = file.getName();
			String ex = SVGFilter.SVG_EXTENSION;
			int id = projectName.toLowerCase().indexOf(ex.toLowerCase());
			
			if(id ==-1)
				file = new File(file.getPath() + SVGFilter.SVG_EXTENSION);			
			else
				projectName = projectName.substring(0, id);

			if(file.exists())
			{ 
				int replace = JOptionPane.showConfirmDialog(this,
	  					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.173"), //$NON-NLS-1$
						LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.188"), JOptionPane.YES_NO_OPTION); //$NON-NLS-1$
						
				if(replace == JOptionPane.NO_OPTION || replace == JOptionPane.CLOSED_OPTION)
				{
					projectName = null;
					
					return false;
				}
			}
			
			currentFile = file;
			addToRecent = true;
		}
		else file = currentFile;
		
		
		try
		{
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.186")); //$NON-NLS-1$
			SVGDocumentGenerator.exportAsSVG(file.getPath(), getDrawPanel().getDraw());
			setTitle();
			
			if(addToRecent)
			{
				preferencesFrame.addRecentFile(file.getAbsolutePath());
				menuBar.updateRecentFilesMenu();
			}
			
			drawPanel.setIsModified(false);
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.191")); //$NON-NLS-1$
		}catch(Exception e) 
		{  
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.192"));//$NON-NLS-1$
			e.printStackTrace(); 
			JOptionPane.showMessageDialog(this, 
		 		    LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.193") +e.toString(),  //$NON-NLS-1$
		 		   LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"), JOptionPane.WARNING_MESSAGE);//$NON-NLS-1$
		}

		return true;
	}
	
	
	
	
	public void newProject()
	{
		try
		{
	    	if(drawPanel.isModified())
	    		switch(JOptionPane.showConfirmDialog(this,
						LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.156"), //$NON-NLS-1$
						LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.110"),  //$NON-NLS-1$
						JOptionPane.YES_NO_CANCEL_OPTION))
				{
	    			case JOptionPane.YES_OPTION:
	    					boolean saved = save(false);
	    					if(!saved) return;
	    					break;
	    			case JOptionPane.NO_OPTION:
							break;
					case JOptionPane.CANCEL_OPTION:
							return;				
				}

	    	setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.197")); //$NON-NLS-1$
	    	projectName = null;
			setTitle();
    		drawPanel.newProject();
    		setZoom(DrawContainer.DEFAULT_ZOOM);
        	undoManager = new UndoRedoManager(drawPanel.getDraw().getFigures(), drawPanel.getDraw(), this);
			reinitializeButtons();
			setSelection(false);
			currentFile = null;
			drawPanel.getDraw().setIsModified(false);
			xScale.repaint();
			yScale.repaint();
			setExportAsMenu(false);
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.198")); //$NON-NLS-1$

		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	
	/**
	 * Allows to load a saved project
	 */
	public boolean open(String fileName, boolean withDialog)
	{
		File file;
		
		if(fileName==null)
		{
			if(fileChooserOpen==null)	
			{
				fileChooserOpen = new JFileChooser(pathOpen);
				fileChooserOpen.setFileSelectionMode(JFileChooser.FILES_ONLY);
				fileChooserOpen.addChoosableFileFilter(new LaTeXDrawFilter());
				fileChooserOpen.addChoosableFileFilter(new SVGFilter());
				fileChooserOpen.setMultiSelectionEnabled(false);
				fileChooserOpen.setDialogTitle(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.200")); //$NON-NLS-1$
			}
			
			int id = fileChooserOpen.showOpenDialog(this);
			
			if(id != JFileChooser.APPROVE_OPTION)
				return false;
			
			file = fileChooserOpen.getSelectedFile();
			
			if(file==null) 
				return false;
		}
		else
			file = new File(fileName);
		
		setIsModified(false);
		newProject();
		currentFile = file;
		reinitializeButtons();
		setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.201")); //$NON-NLS-1$
		projectName = file.getName();
		setExportAsMenu(true);
		
		String ext = projectName.endsWith(LaTeXDrawResources.PROJECT_EXTENSION) ? LaTeXDrawResources.PROJECT_EXTENSION : SVGFilter.SVG_EXTENSION;
		int id = projectName.toLowerCase().indexOf(ext.toLowerCase());
		
		if(id !=-1)			
			projectName = projectName.substring(0, id);
	    
	    if(file.exists() && file.isFile() && file.canRead()) 
	    {
	    	try 
	    	{
		    	if(ext.equals(SVGFilter.SVG_EXTENSION))
		    	{
		    		if(progressbarSVG==null)
		    			progressbarSVG = new SVGProgressBarManager(this);
		    		
		    		progressbarSVG.setFileToParse(file);
		    		progressbarSVG.startProgressBar();
		    	}
		    	else
		    	{
					ObjectInputStream in = new ObjectInputStream(new FileInputStream(file));
	                DrawContainer draw;
	                
					if(!LaTeXDrawResources.LABEL_APP.equals(in.readObject()))
						new LaTeXDrawException(LaTeXDrawException.NOT_TEXDRAW_FILE);
	
					versionOfFile = (String)in.readObject();
					boolean autoUpd = in.readBoolean();
					formerIsInAutoUpdate = in.readBoolean();
					boolean displayB = in.readBoolean();
					menuBar.getAutoAdjustBorders().setSelected(in.readBoolean());
					
	                if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.9.3")>=0) //$NON-NLS-1$
					{
						draw = new DrawContainer();
						draw.open(in);
					}
					else
						draw = (DrawContainer)in.readObject(); 
	                
					drawPanel.setDraw(draw);
					draw = drawPanel.getDraw();
					draw.isSelection(false);
					toolbar.getSliderDelimitor().setValue(Delimitor.getOpacity());
					menuBar.getDisplayBorders().setSelected(displayB);
					draw.setParentPanel(drawPanel);
					draw.setAutoAdjustement(menuBar.getAutoAdjustBorders().isSelected());				
					draw.displayBorders(displayB);	
					undoManager.setFigures(draw.getFigures());
					setPreferredSize((Dimension)in.readObject());
					Point location = (Point)in.readObject();
					setLocation(new Point(Math.max(0, location.x), Math.max(0, location.y)));
					splitPane.setDividerLocation(in.readInt());
					displayCodePanel(in.readBoolean());
					codePanel.setIsInAutoUpdate(autoUpd);
					Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
					formerDividerLocation=(int)(dim.width*0.7);
					formerIsInAutoUpdate = !isCodePanelDisplayed();
					toolbar.getButtonImageBorders().setSelected(menuBar.getDisplayBorders().isSelected());
					drawPanel.setIsGridDisplay(in.readBoolean());
					menuBar.getDisplayXScale().setSelected(in.readBoolean());
					menuBar.getDisplayYScale().setSelected(in.readBoolean());
					codePanel.setComments((String) in.readObject());
					
					if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")>=0) //$NON-NLS-1$
					{
						codePanel.setLabel((String)in.readObject());
						codePanel.setCaption((String)in.readObject());
						codePanel.setPositionHoriToken((String)in.readObject());
						codePanel.setPositionVertToken((String)in.readObject());
					}
					
					if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.8")>=0) //$NON-NLS-1$
					{
						drawPanel.setGridIsMagnetic(in.readBoolean());
						toolbar.getMagneticCB().setSelected(drawPanel.isGridMagnetic());
						boolean isClassicGrid = in.readBoolean();
						menuBar.getDisplayGrid().setSelected(isClassicGrid && drawPanel.isGridDisplayed());
						menuBar.getDisplayPersoGrid().setSelected(!isClassicGrid && drawPanel.isGridDisplayed());
						toolbar.getPersoGridGapField().setEnabled(menuBar.getDisplayPersoGrid().isSelected());
						drawPanel.draw.setClassicGridDrawn(isClassicGrid);
						drawPanel.draw.setPersonalGridGap(in.readInt());
						toolbar.getPersoGridGapField().setValue(drawPanel.draw.getPersonalGridGap());
					}
					else
					{
						menuBar.getDisplayGrid().setSelected(drawPanel.isGridDisplayed());
						menuBar.getDisplayPersoGrid().setSelected(false);
						toolbar.getPersoGridGapField().setEnabled(false);
					}
					
					if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.9")>=0) //$NON-NLS-1$
					{
						Delimitor.setOpacity(in.readInt());
						toolbar.getSliderDelimitor().setValue(Delimitor.getOpacity());
					}
					
					currentFile = new File(currentFile.getPath().substring(0, 
							currentFile.getPath().length()-LaTeXDrawResources.PROJECT_EXTENSION.length())+SVGFilter.SVG_EXTENSION);
					drawPanel.updateDraw(true);
					in.close();	
					
					setTitle();
					undoManager = new UndoRedoManager(drawPanel.getDraw().getFigures(), drawPanel.getDraw(), this);
					updateUndoRedo();
					setSelection(false);
					setIsModified(false);
					preferencesFrame.addRecentFile(file.getAbsolutePath());
					menuBar.updateRecentFilesMenu();
					setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.202")); //$NON-NLS-1$
		    	}
			}
	    	catch(InvalidClassException ex)
	    	{
	    		currentFile = null;
	    		setIsModified(false);
	    		
	    		if(withDialog)
		    		JOptionPane.showMessageDialog(this, LaTeXDrawLang.getString1_9("LaTeXDrawFrame.12"),  //$NON-NLS-1$
				 		    LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"), JOptionPane.WARNING_MESSAGE);  //$NON-NLS-1$
				return false;
	    	}
			catch(Exception ex) 
			{
				setIsModified(false);
				currentFile = null;
				setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.203")); //$NON-NLS-1$
				ex.printStackTrace(); 
				
				if(withDialog)
					JOptionPane.showMessageDialog(this, 
				 		    LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.204") + ex.toString(), //$NON-NLS-1$
				 		    LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"), JOptionPane.WARNING_MESSAGE);   //$NON-NLS-1$
				return false;
			}
	    }
	    else
	    {
	    	setIsModified(false);
			currentFile = null;
			setStatusBarText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.203")); //$NON-NLS-1$
			
			if(withDialog)
				JOptionPane.showMessageDialog(this, LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.18"), //$NON-NLS-1$
			 		    LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"), JOptionPane.WARNING_MESSAGE); //$NON-NLS-1$
			return false;
	    }

	    return true;
	}
	
	
	
	/**
	 * Allows to create and display a dialog box which ask to
	 * the user if he wants to save his project or not
	 * @return JOptionPane.YES_OPTION or JOptionPane.NO_OPTION
	 */
  	private int dialogConfirmSave()
  	{  	
  		return (JOptionPane.showConfirmDialog(this,
  					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.156"), //$NON-NLS-1$
					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.110"), JOptionPane.YES_NO_CANCEL_OPTION)); //$NON-NLS-1$
  	}

  	
  	
  	
  	/**
  	 * Allows to create the frame which contains to LaTeXDraw help
  	 */
  	private void createHelpFrame()
  	{
  		try
  		{
	  		JPanel pHelp = new JPanel(new BorderLayout());
			final JEditorPane htmlPageEditor = new JEditorPane(); 
			htmlPageEditor.setEditable(false);
			URL helpURL = getClass().getResource("/help/index.html");//$NON-NLS-1$
			htmlPageEditor.setPage(helpURL);
	  		helpFrame = new JFrame();
			Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
			
			htmlPageEditor.addHyperlinkListener(new HyperlinkListener() 
					{
						public void hyperlinkUpdate(HyperlinkEvent event) 
						{
							if (event.getEventType() == HyperlinkEvent.EventType.ACTIVATED) 
								try { htmlPageEditor.setPage(event.getURL()); }
								catch(IOException ioe) 
								{ 
									ioe.printStackTrace();
									ExceptionFrameDialog.showExceptionDialog(ioe);
								}
						}
					}); 
			
			JScrollPane scrollPane = new JScrollPane (htmlPageEditor);
			scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
			
			pHelp.add(scrollPane);
			helpFrame.getContentPane().add(pHelp, BorderLayout.CENTER);
			
			helpFrame.setSize(dim.width,dim.height);
	  		helpFrame.setLocation(0, 0);
	  		helpFrame.setResizable(true);
	  		helpFrame.setTitle(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.215")); //$NON-NLS-1$
	  		helpFrame.setVisible(false); 	
	  		
	  		helpFrame.setIconImage(Toolkit.getDefaultToolkit().getImage(
					getClass().getClassLoader().getResource("res/help-browser.png")));//$NON-NLS-1$

  		}catch(Exception ex)
  		{
			JOptionPane.showMessageDialog(this,
					LaTeXDrawLang.getString1_9("LaTeXDrawFrame.13"),  //$NON-NLS-1$
					LaTeXDrawLang.getString1_9("LaTeXDrawFrame.14"), //$NON-NLS-1$
		 		    JOptionPane.WARNING_MESSAGE); 
  		}

  	}
  	
  	
  	
	
	
	/**
	 * Allows to create the "About LaTeXDraw" box
	 */
	private void createAboutLaTeXDrawFrame()
	{
		Dimension dim 	= Toolkit.getDefaultToolkit().getScreenSize();
		aboutTeXDraw 	= new JFrame();
		JTabbedPane tabbedPane = new JTabbedPane();
		JPanel p2 		= new JPanel(), panel = new JPanel();
		JButton okB 	= new JButton(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.23")); //$NON-NLS-1$
		
		aboutTeXDraw.setTitle(LaTeXDrawResources.LABEL_ABOUT);
		aboutTeXDraw.setLocation(dim.width*1/4, dim.height*1/16);

		Image i = Toolkit.getDefaultToolkit().getImage(
						getClass().getClassLoader().getResource("res/LaTeXDrawSmall.png"));//$NON-NLS-1$
		MediaTracker tracker=new MediaTracker(this);
	    tracker.addImage(i,0);
	    try { tracker.waitForID(0); } 
	    catch(InterruptedException e) { e.printStackTrace(); }
	    
		p2.add(new DisplayCanvas(i));
		
		try
		{
			JEditorPane editorPane = new JEditorPane();
			editorPane.setEditable(false);
			editorPane.setDragEnabled(true);
			
			editorPane.setContentType("text/html");//$NON-NLS-1$
			editorPane.setBackground(tabbedPane.getBackground());
			editorPane.setText("<html><body><div style=\"text-align: center; \"><font size=\"-1\"><br>"+ //$NON-NLS-1$
					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.218")+"<br>"+//$NON-NLS-1$//$NON-NLS-2$
					LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.219")+ " " + //$NON-NLS-1$
					VERSION + (VERSION_STABILITY.length()==0 ? "" : " " + VERSION_STABILITY) + LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.220") + " " + ID_BUILD+"<br><br>"+//$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
					LaTeXDrawResources.LABEL_APP+ LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.221")+"<br>"+//$NON-NLS-1$//$NON-NLS-2$
					"Copyright(c) 2005-2008 - Arnaud BLOUIN - arno_b@users.sourceforge.net<br><br>"+//$NON-NLS-1$
					"http://latexdraw.sourceforge.net/<br></div></body></html>");//$NON-NLS-1$
			
			tabbedPane.add(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.general"), new JScrollPane(editorPane)); //$NON-NLS-1$
		}
		catch(Exception e) { System.out.println("About tabular cannot be created."); } //$NON-NLS-1$
		
		try
		{
			JEditorPane editorPane = new JEditorPane();
			editorPane.setEditable(false);
			editorPane.setDragEnabled(true);
			
			editorPane.setContentType("text/html");//$NON-NLS-1$
			editorPane.setBackground(tabbedPane.getBackground());
			editorPane.setText("<html><body><div style=\"text-align: center; \"><font size=\"-1\"><br><br>"+ //$NON-NLS-1$
					LaTeXDrawLang.getStringLaTeXDrawFrame("LFrame2.25")+"<br><br>"+ //$NON-NLS-1$//$NON-NLS-2$
					LaTeXDrawLang.getString1_6("LaTeXDrawFrame.18")+"<br>"+//$NON-NLS-1$//$NON-NLS-2$
					LaTeXDrawLang.getString1_6("LaTeXDrawFrame.19")+"<br>"+//$NON-NLS-1$//$NON-NLS-2$
					"<br></font></div></body></html>");//$NON-NLS-1$

			tabbedPane.add(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.25"), new JScrollPane(editorPane)); //$NON-NLS-1$
		}
		catch(Exception e) { System.out.println("Library tabular cannot be created."); } //$NON-NLS-1$
		
		try
		{
			JEditorPane editorPane = new JEditorPane(getClass().getResource("/res/translators.txt"));//$NON-NLS-1$
			editorPane.setEditable(false);
			editorPane.setDragEnabled(true);
			tabbedPane.add(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.26"), new JScrollPane(editorPane)); //$NON-NLS-1$
		}
		catch(Exception e) { System.out.println("Translators tabular cannot be created."); } //$NON-NLS-1$
		
		try
		{
			JEditorPane editorPane = new JEditorPane(getClass().getResource("/release_note.txt"));//$NON-NLS-1$
			editorPane.setEditable(false);
			editorPane.setDragEnabled(true);
			tabbedPane.add(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.27"), new JScrollPane(editorPane)); //$NON-NLS-1$
		}
		catch(Exception e) { System.out.println("Release note tabular cannot be created."); } //$NON-NLS-1$
		
		try
		{
			JEditorPane editorPane = new JEditorPane(getClass().getResource("/license.txt"));//$NON-NLS-1$
			editorPane.setEditable(false);
			editorPane.setDragEnabled(true);
			tabbedPane.add(LaTeXDrawLang.getString1_8("LaTeXDrawFrame.28"), new JScrollPane(editorPane)); //$NON-NLS-1$
		}
		catch(Exception e) { System.out.println("License tabular cannot be created."); } //$NON-NLS-1$
		
		tabbedPane.setPreferredSize(new Dimension(490, 290));
		p2.add(tabbedPane);

 		aboutTeXDraw.setIconImage(LaTeXDrawResources.aboutIcon.getImage());
 		
		okB.addActionListener(new ActionListener()
				{
					public void actionPerformed(ActionEvent e)
					{
						aboutTeXDraw.setVisible(false);
					}
				});
		panel.add(okB);
 		
 		aboutTeXDraw.getContentPane().setLayout(new BorderLayout());
 		aboutTeXDraw.getContentPane().add(p2, BorderLayout.CENTER);
 		aboutTeXDraw.getContentPane().add(panel, BorderLayout.SOUTH);
		
 		aboutTeXDraw.setSize(500, 620);
 		aboutTeXDraw.setResizable(false);
	}
	
	
	
	private void readObject(@SuppressWarnings("unused") ObjectInputStream ois)
	{		
		/* Nothing to do. */
	}



	/**
	 * Actions to do when an item change.
	 */
	public void itemStateChanged(ItemEvent e) 
	{
		if(e.getSource() instanceof LaTeXDrawComboBox)
		{
			LaTeXDrawComboBox c = (LaTeXDrawComboBox)e.getSource();
			String label = c.getName();
		
			try
			{
				DrawContainer draw = drawPanel.getDraw();
				Draw listSelected = draw.getSelected();
				
				if(label.compareTo(Figure.LABEL_HATCH_CHOICE)==0)
				{
					String style = ((JLabel) hatchChoice.getSelectedItem()).getText();
					Object item = e.getItem();
					
					if(listSelected==null ||
						(item instanceof JLabel && ((JLabel)item).getName().compareTo(style)!=0 && listSelected.size()<2))
						return ;
					
					if(listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(listSelected.size()==1)
					{
						Figure selected = listSelected.getFigureAt(0);
						selected.setHatchingStyle(style);
						
						if(style.compareTo(PSTricksConstants.TOKEN_FILL_GRADIENT)==0)
						{
							buttonGradientEndC.setVisible(true);
							buttonGradientStartC.setVisible(true);
							buttonInteriorColor.setVisible(false);
							filledCheckBox.setVisible(false);
							buttonHatchColor.setVisible(false);
						}
						else
						{
							buttonHatchColor.setVisible(style.compareTo(PSTricksConstants.TOKEN_FILL_SOLID)!=0 &&
														style.compareTo(PSTricksConstants.TOKEN_FILL_NONE)!=0);
							filledCheckBox.setVisible(true);
							buttonGradientEndC.setVisible(false);
							buttonGradientStartC.setVisible(false);
							buttonInteriorColor.setVisible(filledCheckBox.isSelected());
						}
					}
					else
					{
						genHatchS = style;
						
						for(Figure f : listSelected.getFigures())
							if(f.isCustomizable() && f.canBeHatched())
								f.setHatchingStyle(style);
					}
						
					updateFillCB();
					drawPanel.updateDraw(true);
					setIsModified(true);
					return ;
				}				
				
				if(label.compareTo(Figure.LABEL_BORDERS_POSITION_CHOICE)==0)
				{
					String style = ((JLabel) bordersPositionChoice.getSelectedItem()).getText();
					Object item = e.getItem();
					
					if(listSelected==null ||
						(item instanceof JLabel && ((JLabel)item).getName().compareTo(style)!=0 && listSelected.size()<2))
						return ;
					
					if(listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(listSelected.size()==1)
						listSelected.getFigureAt(0).setBordersPosition(style);
					else
					{
						genBordPosS = style;
						
						for(Figure f : listSelected.getFigures())
							if(f.isCustomizable() && f.isBordersMovable())
								f.setBordersPosition(style);
					}
						
					listSelected.updateBorders();
					drawPanel.updateDraw(true);
					setIsModified(true);
					return ;
				}		
				
				if(label.compareTo(LaTeXDrawResources.CHOICE_ARROW_LEFT_NAME)==0)
				{
					String arrowS = ((JLabel)lineArrowLChoice.getSelectedItem()).getText();
					Object item = e.getItem();
					
					if(listSelected==null ||
						(item instanceof JLabel && ((JLabel)item).getName().compareTo(arrowS)!=0 && listSelected.size()<2))
							return ;
					
					if(listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(listSelected.size()==1)
					{
						Figure selected = listSelected.getFigureAt(0);
						
						if(selected instanceof Arrowable)
							((Arrowable)selected).setArrow1Style(arrowS);
					}
					else
					{
						genLeftArrS = arrowS;
						
						for(Figure f : listSelected.getFigures())
							if(f.isCustomizable() && f instanceof Arrowable)
								((Arrowable)f).setArrow1Style(arrowS);
					}
						
					drawPanel.updateDraw(true);
					setIsModified(true);
					return;
				}
				
				if(label.compareTo(LaTeXDrawResources.CHOICE_ARROW_RIGHT_NAME)==0)
				{
					String arrowS = ((JLabel)lineArrowRChoice.getSelectedItem()).getText();
					Object item = e.getItem();
					
					if(listSelected==null ||
						(item instanceof JLabel && ((JLabel)item).getName().compareTo(arrowS)!=0 && listSelected.size()<2))
								return ;
					
					if(listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(listSelected.size()==1)
					{
						Figure selected = listSelected.getFigureAt(0);
						
						if(selected instanceof Arrowable)
						{
							if(((Arrowable)selected).hasTwoLeftArrows())
								arrowS = ArrowHead.invertArrowStyle(arrowS);
							
							((Arrowable)selected).setArrow2Style(arrowS);
						}
					}
					else
					{
						genRightArrS = arrowS;
						
						for(Figure f : listSelected.getFigures())
							if(f.isCustomizable() && f instanceof Arrowable)
								((Arrowable)f).setArrow2Style(((Arrowable)f).hasTwoLeftArrows() ? 
																ArrowHead.invertArrowStyle(arrowS) : arrowS);
					}
						
					drawPanel.updateDraw(true);
					setIsModified(true);
					return;
				}
				
				
				if(label.compareTo(LaTeXDrawResources.CHOICE_LINE_NAME)==0)
				{				
					String str = ((JLabel) lineChoice.getSelectedItem()).getText();
					Object item = e.getItem();
					
					if(listSelected==null ||
						(item instanceof JLabel && ((JLabel)item).getName().compareTo(str)!=0 && listSelected.size()<2))
								return ;
					
					if(listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(listSelected.size()==1)
						listSelected.getFigureAt(0).setLineStyle(str);
					else
					{
						genLineStyleS = str;
						
						for(Figure f : listSelected.getFigures())
							if(f.isCustomizable() && f.isDashableOrDotable())
								f.setLineStyle(str);
					}
						
					drawPanel.updateDraw(true);
					setIsModified(true);
					return ;
				}
				
				
				if(label.compareTo(LaTeXDrawResources.CHOICE_DOT_NAME)==0)
				{
					String str = ((JLabel) dotChoice.getSelectedItem()).getText();
					Object item = e.getItem();
					
					if(listSelected==null ||
						(item instanceof JLabel && ((JLabel)item).getName().compareTo(str)!=0 && listSelected.size()<2))
							return ;
					
					if(listSelected.size()>0)
						addToUndoTransformation(listSelected);
					
					if(listSelected.size()==1)
					{
						Figure selected = listSelected.getFigureAt(0);
						
						if(selected instanceof Dot)
						{
							((Dot)selected).setCurrentStyle(str);
							filledCheckBox.setVisible(false);
							buttonInteriorColor.setVisible(((Dot)selected).canBeFilled() && selected.isFilled());
						}
					}
					else
					{
						genDotS = str;
						
						for(Figure f : listSelected.getFigures())
							if(f instanceof Dot)
								((Dot)f).setCurrentStyle(str);
					}
					
					listSelected.updateBorders();
					setIsModified(true);
					drawPanel.updateDraw(true);
					return;
				}
			
			}catch(Exception ex)
			{
				ex.printStackTrace(); 
				ExceptionFrameDialog.showExceptionDialog(ex);
			}
		}
	}



	/**
	 * @return Returns the xScale.
	 */
	public XScale getXScale()
	{
		return xScale;
	}




	/**
	 * @return Returns the yScale.
	 */
	public YScale getYScale()
	{
		return yScale;
	}




	/**
	 * @return the checkNewVersion
	 */
	public boolean isCheckNewVersion()
	{
		return checkNewVersion;
	}




	/**
	 * @param checkNewVersion the checkNewVersion to set
	 */
	public void setCheckNewVersion(boolean checkNewVersion)
	{
		this.checkNewVersion = checkNewVersion;
	}




	/**
	 * @return the text of statusBar
	 */
	public String getStatusBarText()
	{
		return statusBar.getText();
	}




	/**
	 * @param text The text to put in the statusbar
	 */
	public void setStatusBarText(String text)
	{
		statusBar.setText(text);
	}

	
	
	
	/**
	 * @param unitForPixelsPerCm The unitForPixelsPerCm to set.
	 */
	public void setUnitForPixelsPerCm(String unitForPixelsPerCm)
	{
		drawPanel.getDraw().setUnitForPixelsPerCm(unitForPixelsPerCm);
		xScale.setUnitForPixelsPerCm(unitForPixelsPerCm);
	}

	
	
	
	/**
	 * Allows to choose and to insert a picture in the draw
	 */
	public void insertPicture()
	{
		try
		{
			if(fileChooserPicture==null)	
				fileChooserPicture = new PictureChooser(this, LaTeXDrawLang.getString1_6("LaTeXDrawFrame.23"),  //$NON-NLS-1$
														true, LaTeXDrawLang.getString1_6("LaTeXDrawFrame.1")); //$NON-NLS-1$
	 		
			File file = fileChooserPicture.displayFrame();
			
			if(file!=null && file.canRead() && file.isFile()) 
			{
				setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.25")); //$NON-NLS-1$
	
				boolean state = drawPanel.insertPicture(undoManager, file.getAbsolutePath());
				if(state) setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.28")); //$NON-NLS-1$
				else	  setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.27")); //$NON-NLS-1$
			}
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			JOptionPane.showMessageDialog(this, 
		 		    LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.204") + ex.toString(), //$NON-NLS-1$
		 		   LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"), JOptionPane.WARNING_MESSAGE);  //$NON-NLS-1$
		}
	}

	
	
	
	/**
	 * Allows to convert several figures in an other format.
	 */
	public void batchConvertion()
	{
		convertFrame.setVisible(true);
	}




	/**
	 * @param state Define if the grid must be magnetic.
	 */
	public void setMagneticGrid(boolean state)
	{
		toolbar.getMagneticCB().setSelected(state);
		drawPanel.setGridIsMagnetic(state);
	}




	/**
	 * @param value Define the interval between two lines of the personal grid.
	 */
	public void setPersoGridGap(int value)
	{
		toolbar.getPersoGridGapField().setValue(value);
		drawPanel.draw.setPersonalGridGap(value);
	}





	/**
	 * @return the pathTexEditor.
	 */
	public String getPathTexEditor()
	{
		return pathTexEditor;
	}




	/**
	 * @param pathTexEditor the pathTexEditor to set.
	 */
	public void setPathTexEditor(String pathTexEditor)
	{
		this.pathTexEditor = pathTexEditor;
	}
	
	
	
	/**
	 * Selects all the figures of the current drawing and put the current action to "selection".
	 * @since 1.9
	 */
	public void selectAllFigures()
	{
		menuBar.getMenuSelect().setSelected(true);
		toolbar.getButtonImageSelect().setSelected(true);
		actionPerformed(new ActionEvent(menuBar.getMenuSelect(), 0, LaTeXDrawResources.LABEL_SELECT_FIGURE));
		drawPanel.getDraw().selectAll();
		setSelection(true);
		setCurrentChoice(LaTeXDrawResources.LABEL_SELECT_FIGURE);
	}




	public void showShortcutsFrame()
	{
		if(shortcutsFrame==null)
			shortcutsFrame = new ShortcutsFrame();
		
		shortcutsFrame.setVisible(true);
	}




	public static void setCurrentChoice(String currentChoice)
	{
		LaTeXDrawFrame.currentChoice = currentChoice;
	}




	public void setFormerIsInAutoUpdate(boolean formerIsInAutoUpdate)
	{
		this.formerIsInAutoUpdate = formerIsInAutoUpdate;
	}




	/**
	 * @return the lookAndFeel.
	 * @since 1.9
	 */
	public static String getLookAndFeel()
	{
		return lookAndFeel;
	}




	/**
	 * @param lookAndFeel the lookAndFeel to set.
	 * @since 1.9
	 */
	public static void setLookAndFeel(String lookAndFeel)
	{
		if(lookAndFeel==null)
			throw new IllegalArgumentException();
		
    		LaTeXDrawFrame.lookAndFeel = lookAndFeel;
	}

	
	/**
	 * @return True if the selection of figures is activated.
	 * @since 1.9
	 */
	public boolean isOnSelection()
	{
		return getLMenuBar().getMenuSelect().isSelected();
	}


	
	
	/**
	 * Defines if the CheckBox FillCB must be enable or not considering if the current selected figure.
	 * has a shadow or not (in pstricks, all shape having a shadow is filled but joined-lines, akin-points and Bézier curves). 
	 * @since 1.9
	 */
	protected void updateFillCB()
	{
		Draw selection = drawPanel.draw.getSelected();
		
		if(selection!=null && selection.size()==1)
		{
			Figure f = selection.getFigureAt(0);
			
			if(!f.canBeFilled())
				return ;
			
			if(f instanceof JoinedLines || f instanceof BezierCurve || f instanceof AkinPoints)
			{
				String txt = f.getHatchingStyle();
				
				filledCheckBox.setEnabled(!(shadowCheckBox.isSelected() && f.canHaveShadow() &&
						(txt.equals(PSTricksConstants.TOKEN_FILL_CROSSHATCH) ||
						txt.equals(PSTricksConstants.TOKEN_FILL_CROSSHATCH_F) ||
						txt.equals(PSTricksConstants.TOKEN_FILL_HLINES) ||
						txt.equals(PSTricksConstants.TOKEN_FILL_HLINES_F) ||
						txt.equals(PSTricksConstants.TOKEN_FILL_VLINES) ||
						txt.equals(PSTricksConstants.TOKEN_FILL_VLINES_F))));
			}
			else
				filledCheckBox.setEnabled(!shadowCheckBox.isSelected() || !f.canHaveShadow());
		}
	}
	
	
	
	
	/**
	 * Initialise the values of the general parameters (do not change the fields).
	 * @since 1.9.1
	 */
	protected void initializeGeneralFields()
	{
		genBordPosS = Figure.DEFAULT_BORDERS_POSITION;
		genDbleBordCB.setColor(Figure.DEFAULT_DOUBLE_COLOR);
		genDotS = Dot.DEFAULT_DOT_STYLE;
		genGradientEndCB.setColor(PSTricksConstants.DEFAULT_GRADIENT_END_COLOR);
		genGradientStartCB.setColor(PSTricksConstants.DEFAULT_GRADIENT_START_COLOR);
		genHasDbleBord = Figure.DEFAULT_HAS_DOUBLE_BOUNDARY;
		genHatchCB.setColor(Figure.DEFAULT_HATCH_COL);
		genHatchS = Figure.DEFAULT_HATCH_STYLE;
		genInteriorCB.setColor(Figure.DEFAULT_INTERIOR_COL);
		genIsFilled = Figure.DEFAULT_IS_FILLED;
		genLeftArrS = genRightArrS = ArrowHead.DEFAULT_STYLE;
		genLineCB.setColor(Figure.DEFAULT_BORDERS_COL);
		genLineStyleS = Figure.DEFAULT_LINE_STYLE;
		genThickness = Figure.DEFAULT_THICKNESS;
		genDotSize = Dot.DEFAULT_WIDTH;
		genGradientEndC = PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
		genGradientStartC = PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
		genDbleC = Figure.DEFAULT_DOUBLE_COLOR;
		genHatchC = Figure.DEFAULT_HATCH_COL;
		genInteriorC = Figure.DEFAULT_INTERIOR_COL;
		genLineC = Figure.DEFAULT_BORDERS_COL;
		genShadowC = Figure.DEFAULT_SHADOW_COLOR;
	}
	

	
	/**
	 * @return The frame of preferences.
	 * @since 2.0.0
	 */
	public PreferencesFrame getPreferencesFrame() 
	{
		return preferencesFrame;
	}



	public void windowStateChanged(WindowEvent e)
	{
		updateSplitSeparatorPosition();
	}
	
	
	
	/**
	 * Update the position of the divider.
	 * @since 2.0.0
	 */
	public void updateSplitSeparatorPosition()
	{
		if(isCodePanelDisplayed())
		{
			int widthFrame, dividerLocation;
			
	     	formerDividerLocation = splitPane.getDividerLocation();
	     	widthFrame = getExtendedState()==Frame.MAXIMIZED_BOTH ? Toolkit.getDefaultToolkit().getScreenSize().width : getSize().width;
	     	dividerLocation = (int)(widthFrame*getDividerPosition());
	     	splitPane.setDividerLocation(dividerLocation); 
	     	codePanel.editor.setPreferredSize(new Dimension(widthFrame-dividerLocation, codePanel.editor.getPreferredSize().height));
	     	codePanel.editor.revalidate();
		}
	}



	/**
	 * @return the splitPane.
	 * @since 2.0.0
	 */
	public JSplitPane getSplitPane()
	{
		return splitPane;
	}


	
	
	public double getDividerCurrentLocation()
	{
		return isCodePanelDisplayed() ? splitPane.getDividerLocation() : formerDividerLocation; 
	}
	
	

	/**
	 * @return the dividerPosition read in the preferences file.
	 * @since 2.0.0
	 */
	public double getDividerPosition()
	{
		return dividerPosition;
	}



	/**
	 * @param dividerPosition the dividerPosition to set in ]0;1[.
	 * @since 2.0.0
	 */
	public void setDividerPosition(double dividerPosition)
	{
		if(dividerPosition>0 && dividerPosition<=1)
		{
			this.dividerPosition = dividerPosition;
			updateSplitSeparatorPosition();
		}
	}
	
	
	
	
	/**
	 * Sets the XML element "meta" with the parameters of the code panel.
	 * @param document The XML document.
	 * @param meta The Element to fill.
	 * @throws IllegalArgumentException If document or meta is null.
	 * @since 2.0.0
	 */
	public void getXMLMetadata(Document document, Element meta)
	{
		if(document==null || meta==null)
			throw new IllegalArgumentException();
		
    	Element elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_VERSION), elt2;

    	elt.appendChild(document.createTextNode(VERSION));
    	meta.appendChild(elt);
        
        elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_SIZE);
        meta.appendChild(elt);
        
        elt2 = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_WIDTH);
        elt2.setTextContent(String.valueOf(getWidth()));
        elt.appendChild(elt2);
        
        elt2 = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_HEIGHT);
        elt2.setTextContent(String.valueOf(getHeight()));
        elt.appendChild(elt2);
        
        elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POSITION);
        meta.appendChild(elt);
        
        elt2 = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POSITION_X);
        elt2.setTextContent(String.valueOf(getLocation().x));
        elt.appendChild(elt2);
        
        elt2 = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POSITION_Y);
        elt2.setTextContent(String.valueOf(getLocation().y));
        elt.appendChild(elt2);
        
        elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_DIVIDER_POSITION);
        double divLoc = isCodePanelDisplayed() ? getDividerCurrentLocation() : 1;
        elt.setTextContent(String.valueOf(divLoc<getWidth() && divLoc>1 ? divLoc/getWidth() : 0.8));
        meta.appendChild(elt);
        
        elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_DISPLAY_CODE_PANEL);
        elt.setTextContent(String.valueOf(menuBar.getDisplayCodePanel().isSelected()));
        meta.appendChild(elt);
        
        elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_DISPLAY_X);
        elt.setTextContent(String.valueOf(menuBar.getDisplayXScale().isSelected()));
        meta.appendChild(elt);
        
        elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_DISPLAY_Y);
        elt.setTextContent(String.valueOf(menuBar.getDisplayYScale().isSelected()));
        meta.appendChild(elt);
        
        elt = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_DELIMITOR_OPACITY);
        elt.setTextContent(String.valueOf(Delimitor.getOpacity()));
        meta.appendChild(elt);
	}



	/**
	 * Sets the latexdraw parameters from the <code>metadata</code> SVG tag.
	 * @param nl The list of the parameters.
	 * @since 2.0.0
	 */
	public void setXMLMetadata(NodeList nl)
	{
		if(nl==null)
			return ;
		
		Node n, n2;
		String name;
		double dividerPos = -1;
		
		for(int i=0, size = nl.getLength(); i<size; i++)
		{
			n = nl.item(i);
			
			if(n!=null && LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI.equals(n.getNamespaceURI()))
			{
				name = n.getNodeName();
				
				try
				{
					if(name.endsWith(LaTeXDrawNamespace.XML_SIZE))
					{
						NodeList nl2 = n.getChildNodes();
						
						for(int j=0, size2 = nl2.getLength(); j<size2; j++)
						{
							n2 = nl2.item(j);
							
							if(n2.getNodeName().endsWith(LaTeXDrawNamespace.XML_WIDTH))
								setSize(Double.valueOf(n2.getTextContent()).intValue(), getHeight());
							else
								if(n2.getNodeName().endsWith(LaTeXDrawNamespace.XML_HEIGHT))
									setSize(getWidth(), Double.valueOf(n2.getTextContent()).intValue());
						}
					}
					else if(name.endsWith(LaTeXDrawNamespace.XML_POSITION))
					{
						NodeList nl2 = n.getChildNodes();
						
						for(int j=0, size2 = nl2.getLength(); j<size2; j++)
						{
							n2 = nl2.item(j);
							
							if(n2.getNodeName().endsWith(LaTeXDrawNamespace.XML_POSITION_X))
								setLocation(Math.max(0, Double.valueOf(n2.getTextContent()).intValue()), getLocation().y);
							else
								if(n2.getNodeName().endsWith(LaTeXDrawNamespace.XML_POSITION_Y))
									setLocation(getLocation().x, Math.max(0, Double.valueOf(n2.getTextContent()).intValue()));
						}
					}
					else if(name.endsWith(LaTeXDrawNamespace.XML_DIVIDER_POSITION))
						dividerPos = Double.valueOf(n.getTextContent()).doubleValue();
					else if(name.endsWith(LaTeXDrawNamespace.XML_DISPLAY_CODE_PANEL))
						displayCodePanel(Boolean.valueOf(n.getTextContent()).booleanValue());
					else if(name.endsWith(LaTeXDrawNamespace.XML_DISPLAY_X))
						displayXScale(Boolean.valueOf(n.getTextContent()).booleanValue());
					else if(name.endsWith(LaTeXDrawNamespace.XML_DISPLAY_Y))
						displayYScale(Boolean.valueOf(n.getTextContent()).booleanValue());
					else if(name.endsWith(LaTeXDrawNamespace.XML_DELIMITOR_OPACITY)) {
						toolbar.getSliderDelimitor().setValue(Delimitor.getOpacity());
						Delimitor.setOpacity(Double.valueOf(n.getTextContent()).intValue());
					}
					else if(name.endsWith(LaTeXDrawNamespace.XML_MAGNETIC_GRID))
						setMagneticGrid(Boolean.valueOf(n.getTextContent()).booleanValue());
					else if(name.endsWith(LaTeXDrawNamespace.XML_ZOOM))
						setZoom(Double.valueOf(n.getTextContent()).doubleValue());
					else if(name.endsWith(LaTeXDrawNamespace.XML_GRID_GAP))
						toolbar.getPersoGridGapField().setValue(Double.valueOf(n.getTextContent()).intValue());
					else if(name.endsWith(LaTeXDrawNamespace.XML_DRAW_BORDERS))
						displayBorders(Boolean.valueOf(n.getTextContent()).booleanValue());
				}
				catch(Exception e) { e.printStackTrace(); System.out.println(name + ": invalid value."); }//$NON-NLS-1$
			}
		}
		
		if(dividerPos>0)
		{
			formerDividerLocation = (int)(getWidth()*dividerPos);
			setDividerPosition(dividerPos);
		}
	}
	
	
	public void setZoom(double zoom) {
		drawPanel.getDraw().setZoom(zoom);
		toolbar.getZoomField().setValue(drawPanel.getDraw().getZoom()*100.);
	}
	
	
	/**
	 * @return The menu listener.
	 * @since 2.0.0
	 */
	public MenusListener getMenuListener()
	{
		return menusListener;
	}

	/**
	 * @return the toolbar.
	 * @since 2.0.0
	 */
	public LToolbar getToolbar()
	{
		return toolbar;
	}
	
	
	/**
	 * @return The recent files Listener.
	 * @since 2.0.0
	 */
	public RecentFilesListener getRecentFilesListener()
	{
		return recentFilesListener;
	}
	
	
	public LMenuBar getLMenuBar()
	{
		return menuBar;
	}


	/**
	 * @return the progressBar.
	 * @since 2.0.0
	 */
	public JProgressBar getProgressBar()
	{
		return progressBar;
	}


	/**
	 * @return the stopButton.
	 * @since 2.0.0
	 */
	public JButton getStopButton()
	{
		return stopButton;
	}



	/**
	 * @return the currentFile.
	 * @since 2.0.0
	 */
	public File getCurrentFile()
	{
		return currentFile;
	}



	/**
	 * @param currentFile the currentFile to set.
	 * @since 2.0.0
	 */
	public void setCurrentFile(File currentFile)
	{
		this.currentFile = currentFile;
	}


	/**
	 * @return the formerDividerLocation.
	 * @since 2.0.0
	 */
	public int getFormerDividerLocation()
	{
		return formerDividerLocation;
	}



	public void setPathDistribLatex(String path)
	{
		pathDistribLatex = path;
	}
	
	
	public String getPathDistribLatex() {
		return pathDistribLatex;
	}



	public void setLatexIncludes(String includes)
	{
		latexIncludes = includes;
	}
	
	public String getLatexIncludes() {
		return latexIncludes;
	}
	
	public void setIsInAutoUpdate(boolean auto) {
		menuBar.getMenuAutoUpdate().setSelected(auto);
		codePanel.setIsInAutoUpdate(auto);
	}
}
