package latexDraw.figures;

import java.awt.*;
import java.awt.geom.Line2D;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.ObjectInputStream;

import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines a grid.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 06/06/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Grid extends GridShape 
{
	private static final long serialVersionUID = 1L;

	/** The colour of the sub-grid. */
	protected Color subGridColor;
	
	/** The size of the labels. */
	protected int gridLabelsSize;
	
	/** The number of division in a sub-grid. */
	protected int subGridDiv;
	
	/** The thickness of the main borders of the grid. */
	protected float gridWidth;
	
	/** The colour of the labels */
	protected Color gridLabelsColor;
	
	/** The number of dots in the lines of the grid 
	 * ( if >0, replace a plain line) */
	protected int gridDots;
	
	/** The thickness of the lines of the sub-grid */
	protected float subGridWidth;
	
	/** The number of dots in the lines of the sub-grid
	 * ( if >0, replace a plain line) */
	protected int subGridDots;
	
	/** The unit of the grid */
	protected double unit;
	
	/** The minimum possible size of the labels */
	public static final int MIN_LABELS_SIZE = 5;

	/** The size of the label by default */
	public static final int DEFAULT_LABELSSIZE = (int)PSTricksConstants.DEFAULT_GRID_LABEL;

	
	
	/**
	 * The constructor using two arguments.
	 * @param pos The current position of the south-west point of the grid.
	 */
	public Grid(LaTeXDrawPoint2D pos, boolean increaseMeter)
	{
		super(pos, increaseMeter);
		
		gridDots    		= PSTricksConstants.DEFAULT_GRIDDOTS;
		gridLabelsColor 	= PSTricksConstants.DEFAULT_LABELGRIDCOLOR;
		gridLabelsSize  	= DEFAULT_LABELSSIZE;
		gridWidth       	= (float)PSTricksConstants.DEFAULT_GRID_WIDTH*PPC;
		subGridColor = PSTricksConstants.DEFAULT_SUB_GRID_COLOR;
		subGridDiv   = PSTricksConstants.DEFAULT_SUBGRIDDIV;
		subGridDots  = PSTricksConstants.DEFAULT_SUBGRIDDOTS;
		subGridWidth = (float)PSTricksConstants.DEFAULT_SUB_GRID_WIDTH*PPC;
		unit 		 = PSTricksConstants.DEFAULT_UNIT;
		
		BufferedImage bufferImage = new BufferedImage(2, 2, BufferedImage.TYPE_INT_RGB);
		Graphics2D g = bufferImage.createGraphics();
		font = new Font(g.getFont().getName(), Font.PLAIN, gridLabelsSize);
		g.dispose();
		bufferImage.flush();
		updateFonts();
		updateShape();
	}
	
	

	/**
	 * @return Returns the gridDots.
	 */
	public synchronized int getGridDots()
	{
		return gridDots;
	}



	/**
	 * @param gridDots The gridDots to set.
	 */
	public synchronized void setGridDots(int gridDots)
	{
		if(gridDots>=0)
			this.gridDots = gridDots;
	}



	/**
	 * @return Returns the gridLabelsColor.
	 */
	public synchronized Color getGridLabelsColor()
	{
		return gridLabelsColor;
	}



	/**
	 * @param gridLabelsColor The gridLabelsColor to set.
	 */
	public synchronized void setGridLabelsColor(Color gridLabelsColor)
	{
		if(gridLabelsColor!=null)
			this.gridLabelsColor = gridLabelsColor;
	}



	/**
	 * @return Returns the gridLabelsSize.
	 */
	public synchronized int getGridLabelsSize()
	{
		return gridLabelsSize;
	}



	/**
	 * @param gridLabelsSize The gridLabelsSize to set.
	 */
	public synchronized void setGridLabelsSize(int gridLabelsSize)
	{
		this.gridLabelsSize = gridLabelsSize;
		
		if(gridLabelsSize>0)
			updateFonts();
	}



	/**
	 * @return Returns the gridWidth.
	 */
	public synchronized float getGridWidth()
	{
		return gridWidth;
	}



	/**
	 * @param gridWidth The gridWidth to set.
	 */
	public synchronized void setGridWidth(float gridWidth)
	{
		if(gridWidth>0)
			this.gridWidth = gridWidth;
	}




	/**
	 * @return Returns the subGridColor.
	 */
	public synchronized Color getSubGridColor()
	{
		return subGridColor;
	}



	/**
	 * @param subGridColor The subGridColor to set.
	 */
	public synchronized void setSubGridColor(Color subGridColor)
	{
		if(subGridColor!=null)
			this.subGridColor = subGridColor;
	}



	/**
	 * @return Returns the subGridDiv.
	 */
	public synchronized int getSubGridDiv()
	{
		return subGridDiv;
	}



	/**
	 * @param subGridDiv The subGridDiv to set.
	 */
	public synchronized void setSubGridDiv(int subGridDiv)
	{
		if(subGridDiv>=0)
			this.subGridDiv = subGridDiv;
	}



	/**
	 * @return Returns the subGridDots.
	 */
	public synchronized int getSubGridDots()
	{
		return subGridDots;
	}



	/**
	 * @param subGridDots The subGridDots to set.
	 */
	public synchronized void setSubGridDots(int subGridDots)
	{
		if(subGridDots>=0)
			this.subGridDots = subGridDots;
	}



	/**
	 * @return Returns the subGridWidth.
	 */
	public synchronized float getSubGridWidth()
	{
		return subGridWidth;
	}



	/**
	 * @param subGridWidth The subGridWidth to set.
	 */
	public synchronized void setSubGridWidth(float subGridWidth)
	{
		if(subGridWidth>0)
			this.subGridWidth = subGridWidth;
	}



	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Grid g = (Grid)super.clone();
		g.gridDots = gridDots;
		g.gridLabelsColor = gridLabelsColor;
		g.gridLabelsSize = gridLabelsSize;
		g.gridWidth = gridWidth;
		g.subGridColor = subGridColor;
		g.subGridDiv = subGridDiv;
		g.subGridDots = subGridDots;
		g.subGridWidth = subGridWidth;
		g.unit 	= unit;

		return g;
	}
	

	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		if(g==null || antiAlias==null || rendering==null || alphaInter==null || colorRendering==null)
			throw new IllegalArgumentException();
		
		Font formerFont = g.getFont();
		Color formerCol = g.getColor();
		g.setFont(getFont());
		
		updateBorders();

		double XStep = PPC*unit, i, j, XSubStep, k, l, m, n;
		double YStep = XStep, YSubStep;
		XStep *= gridEnd.x<gridStart.x ? -1 : 1 ;
		YStep *= gridEnd.y<gridStart.y ? -1 : 1 ;
		XSubStep = XStep/subGridDiv;
		YSubStep = YStep/subGridDiv;
		
		LaTeXDrawPoint2D NW = borders.getTheNWPoint();
		LaTeXDrawPoint2D SE = borders.getTheSEPoint();	
		double minX, maxX, minY, maxY;
		double absStep = Math.abs(XStep);

		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx - Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx + Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x) - Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x) + Math.cos(-rotationAngle)*(cy-c2y);
		double posX = position.x+Math.min(gridStart.x, gridEnd.x)*PPC*unit;
		double posY = position.y-Math.min(gridStart.y, gridEnd.y)*PPC*unit;
		
		if(rotationAngle%(Math.PI*2)!=0)
		{		
			g.rotate(rotationAngle);
			g.translate(c3x,c3y);
		}
		
		if(gridEnd.x<gridStart.x)
		{
			minX = gridEnd.x;
			maxX = gridStart.x;
		}else
		{
			minX = gridStart.x;
			maxX = gridEnd.x;
		}
		
		if(gridEnd.y<gridStart.y)
		{
			minY = gridEnd.y;
			maxY = gridStart.y;
		}else
		{
			minY = gridStart.y;
			maxY = gridEnd.y;
		}
		
		g.setColor(subGridColor);
		
		// We draw the sub-grid
		if(subGridDots>0)
		{
			double dotStep = (unit*PPC)/(subGridDots*subGridDiv);
			double nbX = (maxX-minX)*subGridDiv;
			double nbY = (maxY-minY)*subGridDiv;
			
			g.setStroke(new BasicStroke(subGridWidth, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER));
			
			for(i=0, n=NW.x; i<nbX; i++, n+=XSubStep)
				for(j=0, m=NW.y; j<=nbY; j++, m+=YSubStep)
					for(k=0; k<subGridDots; k++)
						g.draw(new Line2D.Double(n+k*dotStep, m, n+k*dotStep, m));
			
			for(j=0, n=NW.y; j<nbY; j++, n+=YSubStep)
				for(i=0, m=NW.x; i<=nbX; i++, m+=XSubStep)
					for(k=0; k<subGridDots; k++)
						g.draw(new Line2D.Double(m, n+k*dotStep, m, n+k*dotStep));
			
			g.draw(new Line2D.Double(SE.x, SE.y, SE.x, SE.y));
		}
		else
			if(subGridDiv>1)
			{
				g.setStroke(new BasicStroke(subGridWidth, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
				
				for(k=minX, i=posX; k<maxX; i+=XStep, k++)
					for(j=0; j<=subGridDiv; j++)
						g.draw(new Line2D.Double(i+XSubStep*j, SE.y, i+XSubStep*j, NW.y));
				
				for(k=minY, i=posY; k<maxY; i-=YStep, k++)
					for(j=0; j<=subGridDiv; j++)
						g.draw(new Line2D.Double(NW.x, i-YSubStep*j, SE.x, i-YSubStep*j));
			}
		
		g.setColor(linesColor);
		
		if(gridDots>0)
		{
			double dotStep = (unit*PPC)/gridDots;
			
			g.setStroke(new BasicStroke(gridWidth, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER));
			
			for(k=minX, i=posX; k<=maxX; i+=XStep, k++)
				for(m=NW.y, n=minY; n<maxY; n++, m+=absStep)
					for(l=0, j=m; l<gridDots; l++, j+=dotStep)
						g.draw(new Line2D.Double(i, j, i, j));
			
			for(k=minY, i=posY; k<=maxY; i-=YStep, k++)
				for(m=NW.x, n=minX; n<maxX; n++, m+=absStep)
					for(l=0, j=m; l<gridDots; l++, j+=dotStep)
						g.draw(new Line2D.Double(j, i, j, i));
			
			g.draw(new Line2D.Double(SE.x, SE.y, SE.x, SE.y));
		}
		else
		{
			g.setStroke(new BasicStroke(gridWidth, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
			
			for(k=minX, i=posX; k<=maxX; i+=XStep, k++)
				g.draw(new Line2D.Double(i, SE.y, i, NW.y));
			
			for(k=minY, i=posY; k<=maxY; i-=YStep, k++)
				g.draw(new Line2D.Double(NW.x, i, SE.x, i));
		}

		if(gridLabelsSize>0)
		{
			// We draw the labels
			float labelHeight = fontMetrics.getAscent();
			float labelWidth = fontMetrics.stringWidth(String.valueOf((int)maxX));
			String label;
			double xorigin = position.x+XStep*origin.x;
			double yorigin = isXLabelSouth  ? position.y-YStep*origin.y+labelHeight : position.y-YStep*origin.y-2;
			double width=gridWidth/2., tmp = isXLabelSouth ? width : -width;
		
			g.setColor(gridLabelsColor);
			
			for(i=NW.x + (isYLabelWest ? width+gridLabelsSize/4. : -width-labelWidth-gridLabelsSize/4.), j=minX; j<=maxX; i+=absStep, j++)
				g.drawString(String.valueOf((int)j), (int)i, (int)(yorigin+tmp));
	
			if(isYLabelWest)
				for(i=NW.y + (isXLabelSouth ? -width-gridLabelsSize/4. : width+labelHeight), j=maxY ; j>=minY; i+=absStep, j--)
				{
					label = String.valueOf((int)j);
					g.drawString(label, (int)(xorigin-fontMetrics.stringWidth(label)-gridLabelsSize/4.-width), (int)i);
				}
			else
				for(i=NW.y + (isXLabelSouth ? -width-gridLabelsSize/4. : width+labelHeight), j=maxY; j>=minY; i+=absStep, j--)
				{
					label = String.valueOf((int)j);
					g.drawString(label, (int)(xorigin+gridLabelsSize/4.+width), (int)i);
				}
		}

		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);
		}
		
		g.setFont(formerFont);
		g.setColor(formerCol);
		updateBorders(fontMetrics);
		
		if(isSelected)
			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
	}

	
	
	
	/**
	 * Allows to update the borders of the grid (without labels)
	 */
	@Override
	public synchronized void updateBorders()
	{
		if(borders==null)
			borders = new LaTeXDrawRectangle(false);
		
		double step = PPC*unit;
		double posX = position.x;
		double posY = position.y;
		double minX = gridEnd.x<gridStart.x ? gridEnd.x : gridStart.x;
		double minY = gridEnd.y<gridStart.y ? gridEnd.y : gridStart.y;
		
		posX+=minX*step;
		posY-=minY*step;
	
		if(gridStart.x<gridEnd.x)
			if(gridStart.y<gridEnd.y)
			{
				borders.setFirstPoint(posX, posY-step*Math.abs(gridEnd.y-gridStart.y));
				borders.setLastPoint(posX+step*Math.abs(gridEnd.x-gridStart.x), posY);
			}
			else
			{
				borders.setFirstPoint(posX, posY);
				borders.setLastPoint(posX+step*Math.abs(gridEnd.x-gridStart.x), 
									 posY+step*Math.abs(gridEnd.y-gridStart.y));
			}
		else
			if(gridStart.y<gridEnd.y)
			{
				borders.setFirstPoint(posX-step*Math.abs(gridEnd.x-gridStart.x), 
									  posY-step*Math.abs(gridEnd.y-gridStart.y));
				borders.setLastPoint(posX, posY);
			}
			else
			{
				borders.setFirstPoint(posX-step*Math.abs(gridEnd.x-gridStart.x), posY);
				borders.setLastPoint(posX, posY+step*Math.abs(gridEnd.y-gridStart.y));
			}	
		gravityCenter = borders.getGravityCenter();
		shape = createShape2D();
	}
	
	
	
	
	
	/**
	 * Allows to update the borders of the grid
	 * @param fontMet The fontMetrics of the labels (If null, the fontMetrics of the grid will be used).
	 */
	@Override
	public synchronized void updateBorders(FontMetrics fontMet)
	{
		if(fontMet==null)
			fontMet = fontMetrics;
		
		updateBorders();
		LaTeXDrawPoint2D NW = borders.getTheNWPoint(), SE = borders.getTheSEPoint();
		String labelMax = String.valueOf(gridEnd.y), labelMin = String.valueOf(gridStart.y);
		String maxLgth = labelMax.length()>labelMin.length() ? labelMax : labelMin;
		double NWxGap=0;
		double NWyGap=0;
		double SExGap=0;
		double SEyGap=0;
		double step = PPC*unit;
		
		if(gridStart.x>origin.x)
			NWxGap=origin.x-gridStart.x;
		else 
			if(gridEnd.x<origin.x)
				SExGap=origin.x-gridEnd.x;
		
		if(gridStart.y>origin.y)
			SEyGap=gridStart.y-origin.y;
		else 
			if(gridEnd.y<origin.y)
				NWyGap=gridEnd.y-origin.y;
		
		borders.setFirstPoint(NW.x+NWxGap*step-fontMet.stringWidth(maxLgth), NW.y+NWyGap*step-fontMet.getHeight());
		borders.setLastPoint(SE.x+SExGap*step+fontMet.stringWidth(String.valueOf(gridEnd.x)), SE.y+SEyGap*step+fontMet.getHeight());
		gravityCenter = borders.getGravityCenter();
		shape = createShape2D();
	}
	
	
	
	
	
	
	@Override
	public synchronized String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		String start = "", end = "", code = "";//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		int startX, startY, endX, endY;
		
		if(isXLabelSouth)
		{
			startY = (int)gridStart.y;
			endY   = (int)gridEnd.y;
		}else
		{
			startY = (int)gridEnd.y;
			endY   = (int)gridStart.y;

		}
		
		if(isYLabelWest)
		{
			startX = (int)gridStart.x;
			endX   = (int)gridEnd.x;
		}else
		{
			startX = (int)gridEnd.x;
			endX   = (int)gridStart.x;
		}
		
		String coord = "("+startX+","+startY+")("+ //$NON-NLS-1$ //$NON-NLS-3$ //$NON-NLS-2$
						endX+","+endY+")"; //$NON-NLS-1$ //$NON-NLS-2$
		
		coord = "(" + (int)origin.x + "," + (int)origin.y + ")" + coord; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		
		//add options
		code+="[gridwidth="+LaTeXDrawNumber.getCutNumber(gridWidth/ppc,0.0001); //$NON-NLS-1$
		code+=",subgridwidth="+LaTeXDrawNumber.getCutNumber(subGridWidth/ppc,0.0001); //$NON-NLS-1$
		
		code+=",gridlabels="+(float)(gridLabelsSize*0.6)+"pt"; //$NON-NLS-1$ //$NON-NLS-2$
		
		if(subGridDiv!=PSTricksConstants.DEFAULT_SUBGRIDDIV)
			code+=",subgriddiv="+subGridDiv; //$NON-NLS-1$
		
		if(gridDots!=PSTricksConstants.DEFAULT_GRIDDOTS)
			code+=",griddots="+gridDots; //$NON-NLS-1$
		
		if(subGridDots!=PSTricksConstants.DEFAULT_SUBGRIDDOTS)
			code+=",subgriddots="+subGridDots; //$NON-NLS-1$
		
		if(!gridLabelsColor.equals(PSTricksConstants.DEFAULT_LABELGRIDCOLOR))
		{
			String name = DviPsColors.getColourName(gridLabelsColor);
			if(name==null)
			{
				name = "color"+number+'b';//$NON-NLS-1$
				DviPsColors.addUserColour(gridLabelsColor, name); 
			}
			code +=",gridlabelcolor="+name; //$NON-NLS-1$
		}
		
		if(unit!=PSTricksConstants.DEFAULT_UNIT)
			code+=",unit="+(float)unit+"cm"; //$NON-NLS-1$ //$NON-NLS-2$
		
		if(!linesColor.equals(PSTricksConstants.DEFAULT_GRIDCOLOR))
		{
			String name = DviPsColors.getColourName(linesColor);
			if(name==null)
			{
				name = "color"+number;//$NON-NLS-1$
				DviPsColors.addUserColour(linesColor, name); 
			}
			code+=",gridcolor="+name; //$NON-NLS-1$
		}
		
		// subgridcolor
		String name = DviPsColors.getColourName(subGridColor);
		if(name==null)
		{
			name = "color"+number+'c';//$NON-NLS-1$
			DviPsColors.addUserColour(subGridColor, name); 
		}
		
		code+=",subgridcolor="+name; //$NON-NLS-1$
		code+="]"; //$NON-NLS-1$
		if(unit!=PSTricksConstants.DEFAULT_UNIT)
			end+="\n\\psset{unit="+PSTricksConstants.DEFAULT_UNIT+"cm}";//$NON-NLS-1$ //$NON-NLS-2$
		
		double x=0, y=0;
	
		// add position
		if(position.x!=0 || position.y!=0)
		{
			double posX = (position.x-d.x)/ppc, posY = (d.y-position.y)/ppc;
			if(Math.abs(posX) < 0.001) posX = 0;
			if(Math.abs(posY) < 0.001) posY = 0;
			
			end +="}"; //$NON-NLS-1$
			start = "\\rput(" + (float)posX + "," + (float)posY + "){" + start; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		
		if(rotationAngle%(Math.PI*2)!=0.)
		{
			updateBorders();
			updateGravityCenter();
			double angle = -Math.toDegrees(rotationAngle);
			double cx = (gravityCenter.x - position.x) / ppc;
			double cy = (position.y - gravityCenter.y) / ppc;
			x = -Math.cos(-rotationAngle) * cx + Math.sin(-rotationAngle) * cy + cx;
			y = -Math.sin(-rotationAngle) * cx - Math.cos(-rotationAngle) * cy + cy;

			if(Math.abs(x) < 0.001) x = 0;
			if(Math.abs(y) < 0.001) y = 0;
			
			start = start+"\\rput{" + (float)angle + "}(" + (float)x + ',' + (float)y + "){"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			end = end+"}"; //$NON-NLS-1$
		}
		return start +"\\psgrid"+ code + coord + end;//$NON-NLS-1$
	}



	
	
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		gridDots 		= ois.readInt();
		gridLabelsSize 	= ois.readInt();
		subGridDiv 		= ois.readInt();
		subGridDots 	= ois.readInt();
		subGridWidth 	= ois.readFloat();
		gridWidth		= ois.readFloat();
		gridEnd 		= (LaTeXDrawPoint2D)ois.readObject();
		gridStart 		= (LaTeXDrawPoint2D)ois.readObject();
		gridLabelsColor = (Color)ois.readObject();
		linesColor 		= (Color)ois.readObject();
		subGridColor 	= (Color)ois.readObject();
		position 		= (LaTeXDrawPoint2D)ois.readObject();
		origin 			= (LaTeXDrawPoint2D)ois.readObject();
		unit 			= ois.readDouble();
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")<=0) //$NON-NLS-1$
		{
			isYLabelWest 	= ois.readBoolean();
			isXLabelSouth 	= ois.readBoolean();
		}
		else
		{
			isXLabelSouth 	= ois.readBoolean();
			isYLabelWest 	= ois.readBoolean();
		}
		borders 		= (LaTeXDrawRectangle)ois.readObject();
		font 			= (Font)ois.readObject();
		
		updateFonts();
		shape = createShape2D();
	}

	


	public synchronized void updateFont()
	{
		font = new Font(font.getName(), Font.PLAIN, gridLabelsSize);
	}



	
	
	@Override
	public synchronized void updateFonts()
	{
		updateFont();
		super.updateFonts();
	}

	
	
	/**
	 * @param unit The unit to set.
	 */
	public synchronized void setUnit(double unit)
	{
		this.unit = unit;
	}
	
	
	
	/**
	 * @return Returns the unit.
	 */
	public synchronized double getUnit()
	{
		return unit;
	}




	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		position.setLocation(grid.getTransformedPointToGrid(position, false));
		updateShape();
	}	
	
	
	
	@Override
	public int hashCode()
	{
		return (int)(super.hashCode()*2.5);
	}
}



