/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_log.c
    \brief  Logging routines for LCMAPS
    \author Martijn Steenbakkers for the EU DataGrid.
*/

#define _XOPEN_SOURCE	500

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <syslog.h>
#include <time.h>
#include <ctype.h>
#include <sys/types.h>
#include <unistd.h>
#include "_lcmaps_log.h"

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
#ifndef CONF_LCMAPS_DEBUG_LEVEL
    #define CONF_LCMAPS_DEBUG_LEVEL 4 /*!< default log level */
#endif /* CONF_LCMAPS_DEBUG_LEVEL */

static int lcmaps_debuglevel_to_sysloglevel(int);
static const char * lcmaps_syslog_level_name_to_string(int);
static char * lcmaps_get_time_string(void);


/******************************************************************************
                       Define module specific variables
******************************************************************************/
static FILE *  lcmaps_logfp=NULL; /*!< stream associated with logfile \internal */
static int     logging_usrlog=0; /*!< flag to do user logging \internal */
static int     logging_syslog=0; /*!< flag to use syslog \internal */
static int     lcmaps_debug_level=LOG_INFO; /*!< log level \internal */
static char *  extra_logstr = NULL; /*!< string to be included in every log statement \internal */ 
static int     should_close_lcmaps_logfp = 0; /*!< Flag to check if the log stream should be closed \internal */ 
static int     detected_old_plugin = 0; /*!< Flag that indicates that an old plugin has been detected */


/******************************************************************************
Function:       lcmaps_debuglevel_to_sysloglevel()
Description:    Translation function from LCMAPS debug levels to syslog levels
Parameters:
                configuration_level
Returns:        (int) sysloglevels
******************************************************************************/
static int lcmaps_debuglevel_to_sysloglevel(int configuration_level)
{
    if (configuration_level < 0)
        return LOG_ERR;

    switch (configuration_level)
    {
        case 0  :
        case 1  : return LOG_ERR;
        case 2  : return LOG_WARNING;
        case 3  : return LOG_NOTICE;
        case 4  : return LOG_INFO;
        case 5  : return LOG_DEBUG;
        default : return LOG_DEBUG;
    }
}

/******************************************************************************
Function:       lcmaps_syslog_level_name_to_string()
Description:    Syslog level name to string
Parameters:
                syslog level
Returns:        char * name or UNKNOWN
******************************************************************************/
static const char * lcmaps_syslog_level_name_to_string(int syslog_level)
{
    switch (syslog_level) {
        case LOG_EMERG   : return "LOG_EMERG";
        case LOG_ALERT   : return "LOG_ALERT";
        case LOG_CRIT    : return "LOG_CRIT";
        case LOG_ERR     : return "LOG_ERR";
        case LOG_WARNING : return "LOG_WARNING";
        case LOG_NOTICE  : return "LOG_NOTICE";
        case LOG_INFO    : return "LOG_INFO";
        case LOG_DEBUG   : return "LOG_DEBUG";
        default          : return "UNKNOWN";
    }
}


/******************************************************************************
Function:       lcmaps_log_open()
Description:    Start logging
Parameters:
                path:    path of logfile
                fp:      file pointer to already opened file (or NULL)
                logtype: DO_USRLOG, DO_SYSLOG
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_log_open(
        char * path,
        FILE * fp,
        unsigned short logtype
        )
    \brief Start logging

    This function should only be used by the LCMAPS itself.
    It opens the logfile and tries to set the log level in the following order:
    -# Try if CONF_LCMAPS_DEBUG_LEVEL > 0
    -# Try if environment variable LCMAPS_DEBUG_LEVEL is set and if it is an integer > 0
    -# Otherwise set lcmaps_debug_level = LOG_INFO;

    \param path    path of logfile.
    \param fp      file pointer to already opened file (or NULL)
    \param logtype DO_USRLOG, DO_SYSLOG

    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
int
lcmaps_log_open(char * path, FILE * fp, unsigned short logtype )
{
    size_t j = 0;
    long   longval;
    char * logfile;
    int    conf_lcmaps_debug_level = 4; /* The LCMAPS debug messages setting in a range of 0-5, needs to be mapped onto Syslog levels. */
    char * debug_env = NULL;
    char * logstr_env = NULL;

    detected_old_plugin = 0; /* Initializer */

    /* syslog part */
    if ((logtype & DO_SYSLOG) == DO_SYSLOG) {
        logging_syslog=1;
    }

    /* logfile part: either fp is given or need filename */
    if ((logtype & DO_USRLOG) == DO_USRLOG) {
	if (lcmaps_logfp!=NULL)	{
	    lcmaps_log(LOG_DEBUG,
		    "%s() has already been called before.\n", __func__);
	    return 0;
	}
   
	/* filepointer or filename ? */
	if (fp != NULL) {
	    /* File already opened, should not be closed at the end */
	    lcmaps_logfp=fp;
	    should_close_lcmaps_logfp = 0;
	    logging_usrlog=1;
	} else { /* Need a filename: check argument and env variable */
	    logfile=(path ? path : getenv("LCMAPS_LOG_FILE"));
	    if (logfile != NULL) {
		/* Try to append to file */
		if ((lcmaps_logfp = fopen(logfile, "a")) != NULL) {
		    should_close_lcmaps_logfp = 1;
		    logging_usrlog=1;
		} else {
		    logging_usrlog=0;
		    logging_syslog=1;
		    syslog(LOG_ERR,
			"%s(): Cannot open logfile %s, will use syslog: %s\n",
			__func__, logfile, strerror(errno));
		}
	    } else {
		logging_usrlog=0;
		logging_syslog=1;
		/* This is not an error: better not log at all as we haven't yet
		 * set loglevel */
	    }
	}
    }

    /* Get the configuration for LCMAPS log leveling
     * 1. Try to get the LCMAPS_DEBUG_LEVEL value, if any.
     * 2. Set the default value for the CONF_LCMAPS_DEBUG_LEVEL, which can
     *    typically come from the ./configure(.ac) */
    if ( (debug_env = getenv("LCMAPS_DEBUG_LEVEL")) )
    {
        /* convert into integer */
        for (j = 0; j < strlen(debug_env); j++) {
            if (!isdigit((debug_env)[j])) {
                syslog(LOG_ERR, "%s(): found non-digit in environment variable in \"LCMAPS_DEBUG_LEVEL\" = %s\n", __func__, debug_env);
                return 1;
            }
        }
	errno=0;
	longval=strtol(debug_env, NULL, 10);
	if (errno!=0 || longval<0 || longval>5)	{
	    syslog(LOG_ERR, "%s(): environment variable value in \"LCMAPS_DEBUG_LEVEL\" should be 0 <= x <= 5.\n", __func__);
            return 1;
        }
	conf_lcmaps_debug_level=(int)longval;
    } else {
        /* Default debug_level, even when nothing else is provided */
        conf_lcmaps_debug_level = (int) CONF_LCMAPS_DEBUG_LEVEL;
    }

    lcmaps_debug_level = lcmaps_debuglevel_to_sysloglevel(conf_lcmaps_debug_level);
    lcmaps_log(LOG_DEBUG,"%s(): setting log level to %d (LCMAPS_DEBUG_LEVEL), which translates to Syslog level \"%s\".\n", 
	    __func__, conf_lcmaps_debug_level,
	    lcmaps_syslog_level_name_to_string(lcmaps_debug_level));

    /*
     * Check if there is an extra log string
     * These environment variables are checked: JOB_REPOSITORY_ID and GATEKEEPER_JM_ID
     */
    if ((logstr_env = getenv("LCMAPS_LOG_STRING")) != NULL ) {
	if ( (extra_logstr = strdup(logstr_env))==NULL)	{
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    return 1;
	}
    }

    return 0;
}

/******************************************************************************
Function:       lcmaps_log()
Description:    Log information to file and or syslog
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_log(
        int prty,
        char * fmt,
        ...
        )
    \brief log information

    This function logs information for LCMAPS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.

    \param prty syslog priority (if 0 don't syslog).
    \param fmt  string format followed by variable argument list

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcmaps_log(int prty, const char * fmt, ...)
{
    va_list pvar;
    char    buf[MAX_LOG_BUFFER_SIZE];
    int     res,i;
    char *  datetime = NULL;
    char *  lcmaps_log_ident = NULL;


    /* Early cut-off from logging for both file and syslog */
    if (prty > lcmaps_debug_level)
        return 1;

    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);

    /* Replace unprintable characters with ? */
    for (i=0; buf[i]!='\0'; i++)    {
	if (buf[i]!='\n' && !isprint(buf[i]))
	    buf[i]='?';
    }

    /* Check return value */
    if ( res < 0 )  {
	lcmaps_log(LOG_ERR, "lcmaps_log() error: %s\n",strerror(errno));
	return 1;
    }
	
    /* truncate too long entries */
    if ((size_t)res >= MAX_LOG_BUFFER_SIZE)
	snprintf(buf+MAX_LOG_BUFFER_SIZE-5,(size_t)5,"...\n");

    /* Log to an opened file pointer */
    if (logging_usrlog)
    {
        if (lcmaps_logfp == NULL)   {
            syslog(LOG_ERR, "lcmaps_log() error: cannot open file descriptor");
	    /* switch over to syslog */
	    logging_usrlog=0;
	    logging_syslog=1;
	}
        else
        {
            /* Get the (mandatory) prefixed datetime stamps */
            datetime = lcmaps_get_time_string(); /* Must be free'd */

            /* Prepend the loglines with a constant & significant string
             * that identifies the LCMAPS calling executable - Example:
             * "glexec" */
            lcmaps_log_ident = getenv("LCMAPS_LOG_IDENT");

            if (extra_logstr == NULL) {
                if (lcmaps_log_ident) {
                    fprintf(lcmaps_logfp,"%s:lcmaps[%ld] %11s: %s: %s", lcmaps_log_ident, (long)getpid(), lcmaps_syslog_level_name_to_string(prty), datetime, buf);
                } else {
                    fprintf(lcmaps_logfp,"lcmaps[%ld] %11s: %s: %s", (long)getpid(), lcmaps_syslog_level_name_to_string(prty), datetime, buf);
                }
            } else {
                if (lcmaps_log_ident) {
                    fprintf(lcmaps_logfp,"%s:lcmaps[%ld] %11s: %s: %s: %s", lcmaps_log_ident, (long)getpid(), lcmaps_syslog_level_name_to_string(prty), datetime, extra_logstr, buf);
                } else {
                    fprintf(lcmaps_logfp,"lcmaps[%ld] %11s: %s: %s: %s", (long)getpid(), lcmaps_syslog_level_name_to_string(prty), datetime, extra_logstr, buf);
                }
            }
            fflush(lcmaps_logfp);
            free(datetime); /* Liberation */
        }
    }
    /* Log to SysLog */
    if (logging_syslog)
    {
        /* Syslog broadcast protection - All LOG_EMERG, LOG_ALERT and LOG_CRIT
         * will be down-played to a LOG_ERR. */
        if (prty < LOG_ERR) {
            prty = LOG_ERR;
            if (detected_old_plugin == 0) {
                detected_old_plugin = 1; /* Flag up */
                lcmaps_log(LOG_WARNING, "Warning: detected an old plug-in based on its verbose output.\n");
            }
        }
#if 0
        /* Mention the log priority explicitly, for debugging purposes */
        if (extra_logstr == NULL) {
            syslog(prty, "lcmaps[%ld] %11s: %s",      (long)getpid(), lcmaps_syslog_level_name_to_string(prty), buf);
        } else {
            syslog(prty, "lcmaps[%ld] %11s: %s: %s",  (long)getpid(), lcmaps_syslog_level_name_to_string(prty), extra_logstr, buf);
        }
#else
        if (extra_logstr == NULL) {
            syslog(prty, "lcmaps: %s",      buf);
        } else {
            syslog(prty, "lcmaps: %s: %s",  extra_logstr, buf);
        }
#endif
    }

    return 0;
}

/******************************************************************************
Function:       lcmaps_log_debug()
Description:    Print debugging information
Parameters:
                debug_lvl: debugging level
                fmt:       string format
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_log_debug(
        int debug_lvl,
        char * fmt,
        ...
        )
    \brief Print debugging information

    This function prints debugging information (using lcmaps_log with priority 1)
    provided debug_lvl <= DEBUG_LEVEL (default is 0).

    \param debug_lvl debugging level
    \param fmt       string format followed by variable argument list

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcmaps_log_debug(int debug_lvl, const char * fmt, ...)
{
    va_list pvar;
    char    buf[MAX_LOG_BUFFER_SIZE];
    int     res;

#if 0
    /* If the LCMAPS definition of LOG_DEBUG (level 5)
     * is *not* explicitly set, do not even bother to
     * log anything */
    if (debug_level < 5)
    {
        /* Cutting off LOG_DEBUG messages */
        return 1;
    }
#endif

    /* Early cut-off from logging for both file and syslog */
    /* LOG_DEBUG, no exceptions */
    if (LOG_DEBUG > lcmaps_debug_level)
        return 1;

    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);
    
    /* Check return value */
    if ( res < 0 )  {
	lcmaps_log(LOG_ERR, "lcmaps_log_debug() error: %s\n",strerror(errno));
	return 1;
    }

    /* truncate too long entries */
    if ((size_t)res >= MAX_LOG_BUFFER_SIZE)
	snprintf(buf+MAX_LOG_BUFFER_SIZE-5,(size_t)5,"...\n");

    /* LOG_DEBUG, no exceptions */
    /* NOTE: Make sure NOT to use buf directly, but do via a format string */
    return lcmaps_log(LOG_DEBUG,"%s",buf);
}

/******************************************************************************
Function:       lcmaps_log_a_string()
Description:    Log a string according to the passed format to file and or syslog
Parameters:
                prty:       syslog priority (if 0 don't syslog)
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_log_a_string(
        int prty,
        char * fmt,
        char * the_string
        )
    \brief log information

    This function logs information for LCMAPS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.

    \param prty         syslog priority (if 0 don't syslog).
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcmaps_log_a_string(int prty, const char * fmt, const char * the_string)
{
    return lcmaps_log(prty, fmt, the_string);
}

/******************************************************************************
Function:       lcmaps_log_a_string_debug()
Description:    Print debugging information
Parameters:
                debug_lvl:  debugging level
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_log_a_string_debug(
        int debug_lvl,
        char * fmt,
        char * the_string
        )
    \brief Print debugging information

    \param debug_lvl    debugging level
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcmaps_log_a_string_debug(int debug_lvl, const char * fmt, const char * the_string)
{
    return lcmaps_log_debug(debug_lvl, fmt, the_string);
}


/******************************************************************************
Function:       lcmaps_log_close()
Description:    Stop logging
Parameters:
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_log_close()
    \brief Stop logging.

    \internal
    This function should only be used by the LCMAPS itself.

    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
int
lcmaps_log_close()
{
    if (extra_logstr != NULL)
    {
        free(extra_logstr);
        extra_logstr = NULL;
    }

    if (should_close_lcmaps_logfp)
    {
        fclose(lcmaps_logfp);
        lcmaps_logfp=NULL;
    }

    return 0;
}


/******************************************************************************
Function:       lcmaps_log_time()
Description:    Log information to file and or syslog with a timestamp
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_log_time(
        int prty,
        char * fmt,
        ...
        )
    \brief log information with timestamp
 
    This function logs information with a timestamp for LCMAPS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.
 
    \param prty syslog priority (if 0 don't syslog).
    \param fmt  string format followed by variable argument list
 
    \retval 0 succes.
    \retval 1 failure.
*/
int
lcmaps_log_time(int prty, const char * fmt, ...)
{
    va_list     pvar;
    char        buf[MAX_LOG_BUFFER_SIZE];
    int         res;
    int         rc = 0;

    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);
    
    /* Check return value */
    if ( res < 0 )  {
	lcmaps_log(LOG_ERR, "lcmaps_log_time() error: %s\n",strerror(errno));
	return 1;
    }

    /* truncate too long entries */
    if ((size_t)res >= MAX_LOG_BUFFER_SIZE)
	snprintf(buf+MAX_LOG_BUFFER_SIZE-5,(size_t)5,"...\n");

    /* Do regular logging, time prefix is added by lcmaps_log() when logging to
     * file */
    /* NOTE: make sure not to print buf directly, it might contain % */
    rc = lcmaps_log(prty, "%s", buf);

    return rc;
}



/******************************************************************************
Function:       lcmaps_get_time_string()
Description:    Log time information string in GMT
                The string is always 19 readable characters 
                followed by a newline
Parameters:
Returns:       !NULL succes
                NULL failure
******************************************************************************/
/*!
    \fn lcmaps_get_time_string(void)
    \brief Log time information string in GMT
           The string is always 19 readable characters 
           followed by a newline

    \retval !NULL succes.
    \retval NULL failure.
*/
static char * lcmaps_get_time_string(void)
{
    char      * datetime;
    time_t      mclock;
    struct tm * tmp;

    time(&mclock);
    /* Log in GMT, no exceptions */
    tmp = gmtime(&mclock);
    if (!tmp) {
        return NULL;
    }

    datetime = malloc(sizeof(char) * 21);

    /* This string will always return 19 characters and \0 */
    snprintf(datetime, (size_t)21, "%04d-%02d-%02d.%02d:%02d:%02dZ",
                       tmp->tm_year + 1900, tmp->tm_mon + 1, tmp->tm_mday,
                       tmp->tm_hour, tmp->tm_min, tmp->tm_sec);

    return datetime;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/pluginmanager/lcmaps_log.c,v $
    $Date: 2015-02-02 11:46:42 +0100 (Mon, 02 Feb 2015) $
    $Revision: 18216 $
    $Author: msalle $
******************************************************************************/
