{-# OPTIONS_GHC -optc-D__HUGS__ #-}
{-# INCLUDE <ldap.h> #-}
{-# LINE 1 "Modify.hsc" #-}
{- -*- Mode: haskell; -*-
{-# LINE 2 "Modify.hsc" #-}
Haskell LDAP Interface
Copyright (C) 2005 John Goerzen <jgoerzen@complete.org>

This code is under a 3-clause BSD license; see COPYING for details.
-}

{- |
   Module     : LDAP.Modify
   Copyright  : Copyright (C) 2005 John Goerzen
   License    : BSD

   Maintainer : John Goerzen,
   Maintainer : jgoerzen@complete.org
   Stability  : provisional
   Portability: portable

LDAP changes

Written by John Goerzen, jgoerzen\@complete.org
-}

module LDAP.Modify (LDAPModOp(..), LDAPMod(..),
                    ldapAdd, ldapModify, ldapDelete
                   )
where

import LDAP.Utils
import LDAP.Types
import LDAP.TypesLL
import LDAP.Data
import Foreign
import Foreign.C.String
import LDAP.Result
import Control.Exception(finally)
import Data.Bits


{-# LINE 39 "Modify.hsc" #-}

data LDAPMod = LDAPMod {modOp :: LDAPModOp -- ^ Type of operation to perform
                       ,modType :: String -- ^ Name of attribute to edit
                       ,modVals :: [String] -- ^ New values
                       }
             deriving (Eq, Show)

ldapModify :: LDAP              -- ^ LDAP connection object
           -> String            -- ^ DN to modify
           -> [LDAPMod]         -- ^ Changes to make
           -> IO ()
ldapModify = genericChange "ldapModify" ldap_modify_s

ldapAdd :: LDAP                 -- ^ LDAP connection object
        -> String               -- ^ DN to add
        -> [LDAPMod]            -- ^ Items to add
        -> IO ()
ldapAdd = genericChange "ldapAdd" ldap_add_s

genericChange name func ld dn changelist =
    withLDAPPtr ld (\cld ->
    withCString dn (\cdn ->
    withCLDAPModArr0 changelist (\cmods ->
    do checkLE name ld $ func cld cdn cmods
       return ()
            )))

{- | Delete the specified DN -}
ldapDelete :: LDAP -> String -> IO ()
ldapDelete ld dn =
    withLDAPPtr ld (\cld ->
    withCString dn (\cdn ->
    do checkLE "ldapDelete" ld $ ldap_delete_s cld cdn
       return ()
                   ))

data CLDAPMod

newCLDAPMod :: LDAPMod -> IO (Ptr CLDAPMod)
newCLDAPMod lm =
    do (ptr::(Ptr CLDAPMod)) <- mallocBytes (12)
{-# LINE 80 "Modify.hsc" #-}
       cmodtype <- newCString (modType lm)
       let (cmodop::LDAPInt) = 
               (fromIntegral . fromEnum . modOp $ lm) .|. 
               128
{-# LINE 84 "Modify.hsc" #-}
       bervals <- mapM newBerval (modVals lm)
       (arrptr::Ptr (Ptr Berval)) <- newArray0 nullPtr bervals 
       ( (\hsc_ptr -> pokeByteOff hsc_ptr 0) ) ptr cmodop
{-# LINE 87 "Modify.hsc" #-}
       ( (\hsc_ptr -> pokeByteOff hsc_ptr 4) ) ptr cmodtype
{-# LINE 88 "Modify.hsc" #-}
       ( (\hsc_ptr -> pokeByteOff hsc_ptr 8) ) ptr arrptr
{-# LINE 89 "Modify.hsc" #-}
       return ptr

freeCLDAPMod :: Ptr CLDAPMod -> IO ()
freeCLDAPMod ptr =
    do -- Free the array of Bervals
       (arrptr::Ptr (Ptr Berval)) <- ( (\hsc_ptr -> peekByteOff hsc_ptr 8) ) ptr
{-# LINE 95 "Modify.hsc" #-}
       (arr::[Ptr Berval]) <- peekArray0 nullPtr arrptr
       mapM_ freeHSBerval arr
       free arrptr
       -- Free the modtype
       (cmodtype::CString) <- ( (\hsc_ptr -> peekByteOff hsc_ptr 4) ) ptr
{-# LINE 100 "Modify.hsc" #-}
       free cmodtype
       -- mod_op is an int and doesn't need freeing
       -- free the LDAPMod itself.
       free ptr
       
withCLDAPModArr0 :: [LDAPMod] -> (Ptr (Ptr CLDAPMod) -> IO a) -> IO a
withCLDAPModArr0 = withAnyArr0 newCLDAPMod freeCLDAPMod

foreign import ccall unsafe "ldap.h ldap_modify_s"
  ldap_modify_s :: LDAPPtr -> CString -> Ptr (Ptr CLDAPMod) -> IO LDAPInt

foreign import ccall unsafe "ldap.h ldap_delete_s"
  ldap_delete_s :: LDAPPtr -> CString -> IO LDAPInt

foreign import ccall unsafe "ldap.h ldap_add_s"
  ldap_add_s :: LDAPPtr -> CString -> Ptr (Ptr CLDAPMod) -> IO LDAPInt
