#############################################################################
#
#  Linux Desktop Testing Project http://ldtp.freedesktop.org
# 
#  Author:
#     Veerapuram Varadhan <v.varadhan@gmail.com>
#     Prashanth Mohan <prashmohan@gmail.com>
#     Venkateswaran S <wenkat.s@gmail.com>
#     Nagappan Alagappan <nagappan@gmail.com>
# 
#  Copyright 2004 - 2007 Novell, Inc.
#  Copyright 2008 - 2009 Nagappan Alagappan
# 
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Library General Public
#  License as published by the Free Software Foundation; either
#  version 2 of the License, or (at your option) any later version.
# 
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Library General Public License for more details.
# 
#  You should have received a copy of the GNU Library General Public
#  License along with this library; if not, write to the
#  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
#  Boston, MA 02110, USA.
#
#############################################################################

__author__ = "Nagappan Alagappan <nagappan@gmail.com>, "\
    "Veerapuram Varadhan <v.varadhan@gmail.com>, "\
    "Prashanth Mohan <prashmohan@gmail.com>, Venkateswaran S <wenkat.s@gmail.com>"
# __maintainer__ = "Nagappan Alagappan <nagappan@gmail.com>"
__version__ = "1.7.0"

import os
import re
import sys
import xml
import time
import types
import base64
import select
import socket
import struct
import random
import thread
import commands
import traceback
import threading
import datetime
from xml.sax import saxutils
from xml.parsers.expat import ExpatError
from xml.dom.minidom import parse, parseString

# Let us not register our application under at-spi application list
os.environ ['GTK_MODULES'] = ''

import ldtplib
import ldtputils
from ldtplib.ldtplibutils import *

_ldtpDebug = os.getenv ('LDTP_DEBUG')
_userName = 'LDTP'
if os.getenv ('USER'):
    _userName = os.getenv ('USER')
elif os.getenv ('LOGNAME'):
    _userName = os.getenv ('LOGNAME')
_screenshotDir = _logDir = '/tmp/ldtp-%s' % _userName
commands.getstatusoutput ('mkdir -p %s' % _logDir)

commands.getstatusoutput ('rm -f /tmp/LDTP-XML-*')

# Init callback functions dictionary
_callbackFunctions = {}

# Init callback functions dictionary
_ltfxCallbackFunctions = {}

# Create read flag
_readFlag = threading.Event ()
# Clear the flag by default
_readFlag.clear ()

# Create notification flag
_notificationFlag = threading.Event ()
# Set the flag by default
_notificationFlag.set ()

# Contains poll fd's
_serverPoll = None

_mainSock, _ldtpUseTcp = connect2LdtpExecutionEngine ()

_sockFdPool = ldtplib.ldtplibutils.sockFdPool
_sockFdPool [threading.currentThread ()] = _mainSock

logger  = None
xmlhdlr = None

# Default GUI timeout 30 seconds
_ldtpGuiTimeout = 30

__timeDelayBetweenCmds = os.getenv ('LDTP_DELAY_CMD')

def logFailures (error, screenshot = True, logStatus = "error"):
    log (error, logStatus)
    if screenshot and not _ldtpUseTcp:
        _logFile = "%s/screenshot-%s.png" % (_screenshotDir,
                                             time.strftime ("%m-%d-%Y-%H-%M-%s"))
        log (ldtputils.imagecapture (outFile = _logFile), "screenshot")

def generatexml (commandId, *args):
    """This Function will parse some information into an LDTP packet
    INPUT: commandId --> command_id
    args --> [,window_name[,object_name[,args...]]]

    OUTPUT: returns a string in the LDTP packet (XML lib2) format"""
    # Argument length
    argsLen = len (args)
    _argDebug = ''
    
    # Fill action name
    _xml = "<?xml version=\"1.0\"?><REQUEST><ACTION>%ld</ACTION>" % commandId
    if argsLen >= 1:
        _xml = "%s<ID>%s</ID>" % (_xml, saxutils.escape (args [0]))
    if commandId != command.INITAPPMAP and argsLen >= 2:
        _xml = "%s<CONTEXT>%s</CONTEXT>" % (_xml, saxutils.escape (args [1]))
	_argDebug = "'%s'" % args [1]
    if argsLen >= 3:
        _xml = "%s<COMPONENT>%s</COMPONENT>" % (_xml, saxutils.escape (args [2]))
	_argDebug = "%s, '%s'" % (_argDebug, args [2])
    if argsLen >= 4:
        _xml = "%s<ARGUMENTS>" % _xml
        # Fixme: Dirty hack :( start
        if type (args [3]) == list:
            args = args [3]
            args.insert (0, '')
            args.insert (1, '')
            args.insert (2, '')
            argsLen = len (args)
        # Fixme: Dirty hack :( end
        for index in range (3, argsLen):
            if args [index] != None and args [index] != '':
                if type (args [index]) is type (int):
                    _xml = "%s<ARGUMENT><![CDATA[%d]]></ARGUMENT>" % (_xml, args [index])
                    _argDebug = "%s, %d" % (_argDebug, args [index])
                else:
                    _xml = "%s<ARGUMENT><![CDATA[%s]]></ARGUMENT>" % (_xml, args [index])
                    _argDebug = "%s, '%s'" % (_argDebug, args [index])
        _xml = "%s</ARGUMENTS>" % _xml
    if commandId == command.INITAPPMAP:
        _xml = "%s<ARGUMENTS><ARGUMENT><![CDATA[%s]]></ARGUMENT></ARGUMENTS>" % (_xml, args [1])
	_argDebug = "'%s'" % args [1]
    _cmd = dir (command)[commandId]
    if re.search ('exist', _cmd, re.I) == None and \
            re.search ('verify', _cmd, re.I) == None and \
            re.search ('wait', _cmd, re.I) == None and \
            re.search ('list', _cmd, re.I) == None and \
            re.search ('getobject', _cmd, re.I) == None and \
            re.search ('getchild', _cmd, re.I) == None and \
            re.search ('has', _cmd) == None:
        # Let us delay the command execution by 1 second, as the at-cspi really hogs
        if __timeDelayBetweenCmds != None:
            try:
                # Delay each command by '__timeDelayBetweenCmds' seconds
                time.sleep (int (__timeDelayBetweenCmds))
            except IndexError:
                # Default to 5 seconds delay if LDTP_DELAY_CMD
                # env variable is set
                time.sleep (5)
        else:
            time.sleep (1)

    if _ldtpDebug:
        try:
            print "%s (%s)" % (dir (command)[commandId].lower (), _argDebug)
        except:
            if hasattr (traceback, 'format_exc'):
                print traceback.format_exc ()
            else:
                print traceback.print_exc ()
    _xml = '%s</REQUEST>' % _xml
    return _xml

def parseobjectlist (xmldata):
    """Returns the object list"""
    try:
        _objList = None
        try:
            dom = parseString (xmldata)
        except UnicodeEncodeError:
            dom = parseString (xmldata.encode ('utf-8'))
        try:
            _objList = dom.getElementsByTagName ('OBJECTLIST')
        except IndexError:
            raise LdtpExecutionError ('Invalid Object List')
        if _objList == None:
            raise LdtpExecutionError ('Invalid Object List')
        taglist = []
        for dataelements in _objList:
            for data in dataelements.getElementsByTagName ('OBJECT'):
                taglist.append (getText (data.childNodes))
        return taglist
    except ExpatError, msg:
        if _ldtpDebug:
            if hasattr (traceback, 'format_exc'):
                print traceback.format_exc ()
            else:
                print traceback.print_exc ()
        raise LdtpExecutionError ('Parsing XML error: ' + str (msg))

def parsexml (xmlpacket):
    """Returns the value obtained from the server's return LDTP packet"""

    _statusMsg      = None
    _statusCode     = None
    _serverResponse = None
    _responseType   = None
    _requestId      = None
    _serverResponse = None
    _responseObj    = None
    _serverResponseLen = 0

    try:
        dom = parseString (xmlpacket)
        try:
            _responseObj  = dom.getElementsByTagName ('RESPONSE')[0]
            _responseType = 'response'
        except IndexError:
            try:
                _responseObj   = dom.getElementsByTagName ('NOTIFICATION')[0]
                _responseType = 'notification'
            except IndexError, msg:
                raise LdtpExecutionError ('Invalid Response')
        try:
            _responseStatusObj = _responseObj.getElementsByTagName ('STATUS')[0]
            _statusCode = int (getText (_responseStatusObj.getElementsByTagName ('CODE')[0].childNodes))
            _statusMsg  = getText (_responseStatusObj.getElementsByTagName ('MESSAGE')[0].childNodes)
        except ValueError:
            raise LdtpExecutionError ('Invalid Status')
        except IndexError:
            raise LdtpExecutionError ('Invalid Status')
        try:
            data = _responseObj.getElementsByTagName ('DATA')[0]
            _serverResponse    = getCData (data.getElementsByTagName ('VALUE').item (0).childNodes)#.encode ('utf-8')
            _serverResponseLen = int (getText (data.getElementsByTagName ('LENGTH')[0].childNodes))
        except ValueError:
            raise LdtpExecutionError ('Invalid Data Length')
        except IndexError:
            # Data tag may not be present
            pass
        try:
            data = _responseObj.getElementsByTagName ('FILE')[0]
            fileName = getText (data.getElementsByTagName ('NAME').item (0).childNodes)#.encode ('utf-8')
            if not os.path.exists (fileName):
                raise LdtpExecutionError ('File %s does not exist' % fileName)
            fp = open (fileName)
            _serverResponse = ''
            for line in fp.readlines ():
                _serverResponse = _serverResponse + line
            _serverResponseLen = len (_serverResponse)
            #commands.getstatusoutput ('rm -f ' + fileName)
        except IndexError:
            # Data tag may not be present
            pass
        try:
            _requestId  = getText (_responseObj.getElementsByTagName ('ID')[0].childNodes)
        except IndexError:
            # On notification _requestId will be empty
            pass
    except ExpatError, msg:
        if msg.code == xml.parsers.expat.errors.XML_ERROR_NO_ELEMENTS:
            return None
        if xml.parsers.expat.ErrorString (msg.code) == xml.parsers.expat.errors.XML_ERROR_NO_ELEMENTS:
            return None
        raise LdtpExecutionError ('Parsing XML error: ' + str (msg))

    # Return all the respective values, let the calling function decide what to do with the values
    return _responseType, (_statusCode, _statusMsg, _requestId), (_serverResponseLen, _serverResponse)

def getresponse (packetId = None, sockfd = None, timeOut = True):
    global _ldtpGuiTimeout
    count = 0
    # Let us initialize with empty string, as we are going to check for None later
    # returned by recvpacket later
    _responsePacket = ''
    while count < _ldtpGuiTimeout:
        try:
            _responsePacket = recvpacket (sockfd = sockfd)
            break
        except LdtpExecutionError, msg:
            if msg.value == 'Timeout':
                if timeOut is False:
                    # don't timeout, so don't increase the count !
                    continue
                time.sleep (1)
                # If timeout, let us retry for _ldtpGuiTimeout times
                count += 1
                continue
            raise
    if _responsePacket == None:
        if _ldtpDebug != None and _ldtpDebug == '2':
            print 'Response None'
        _mainSock, _ldtpUseTcp = connect2LdtpExecutionEngine ()
        _sockFdPool = ldtplib.ldtplibutils.sockFdPool
        _sockFdPool [threading.currentThread ()] = _mainSock
        raise LdtpExecutionError ('Server aborted')
    try:
        _responseType, _responseStatus, _responseData = parsexml (_responsePacket)
        if _responseType == 'response':
            # If response type is response, then let us enforce checking packetId
            if _responseStatus [2] == packetId:
                return _responseStatus, _responseData
            else:
                if _ldtpDebug:
                    print 'Invalid response packet', _responseStatus [2], packetId
                return (-1, "Invalid response packet", packetId), (0, None)
        # If notification, then return directly, don't check
        return _responseStatus, _responseData
    except TypeError, msg:
        if _ldtpDebug != None and _ldtpDebug == '2':
            print 'TypeError', msg
        return (-1, "Invalid response packet", packetId), (0, None)

def invokecallback (windowTitle, *args):
    _clientFd, _ldtpUseTcp = connect2LdtpExecutionEngine ()

    _sockFdPool = ldtplib.ldtplibutils.sockFdPool
    _sockFdPool [threading.currentThread ()] = _clientFd
    # Since the LDTP engine sends notification packet to the socket fd
    # who initiate the onwindowcreate call
    # and so, the registering for window event should be in this function
    _requestId  = threading.currentThread ().getName () + str (command.ONWINDOWCREATE)
    _message = generatexml (command.ONWINDOWCREATE, _requestId, windowTitle)
    sendpacket (_message)
    _responseStatus, _responseData = getresponse (_requestId)
    if _responseStatus [0] != 0:
        raise LdtpExecutionError ('onwindowcreate failed: ' + _responseStatus [1])
    # Once we receive the response, let us go into this thread getresponse call
    # If we receive any notification from LDTP engine, let us invoke the callback function
    # and go back to receive mode
    while True:
        # NOTE: The script writer has to use removecallback function to stop this never
        # ending while loop
        _responseStatus, _responseData = getresponse (sockfd = _clientFd, timeOut = False)
        if _ldtpDebug:
            print _responseStatus, _responseData
        # Get the callback function
        callback = _callbackFunctions.get (_responseData [1])[0]
        # Check whether its callable
        if callable(callback):
            # If callable then call the function
            if len(args) and args[0]:
                callback(*args[0])
            else:
                callback()
        else:
            if _ldtpDebug:
                print 'callback function is not callable, ignoring'

def invokeltfxcallback (clientsocket, callback):
    clientsock, _ldtpUseTcp = connect2LdtpExecutionEngine ()

    global _serverPoll
    # Register newly created socket for polling
    _serverPoll.register (clientsock, select.POLLIN)

    # Add new client socket to socket fd pool
    _sockFdPool[threading.currentThread ()] = clientsock
    _notificationFlag.set ()
    if (callable (callback)):
        callback ()
    # Unregister newly created socket from polling once its completed
    _serverPoll.unregister (clientsock)

def shutdown ():
    if threading.activeCount () > 1:
        thread.exit ()
    stoplog ()
    #sys.exit ()

def selectevent (windowName, componentName, calendarEventName):
    """Selects the row from the table of calendar events based on the calendar event name specified
    INPUT: selectevent ('<window name>', '<Calendar_view name>', '<calendar event name>')

    OUTPUT: Returns 1 on success and 0 on error."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTEVENT)
        _message = generatexml (command.SELECTEVENT, _requestId, windowName, componentName, calendarEventName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectevent failed: %s' % msg.value)
        raise
            
def selecteventindex (windowName, componentName, eventNumber):
    """Select an event from a calendar table using its index. Index for a calendar event starts from 1.
    INPUT: selecteventindex ('<window name>', '<component name>', <event number>)

    OUTPUT: Returns 1 on success and 0 on error."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTEVENTINDEX)
        _message = generatexml (command.SELECTEVENTINDEX, _requestId, windowName, componentName, str (eventNumber))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selecteventindex failed: %s' % msg.value)
        raise
            
def verifyeventexist (windowName, componentName):
    """verifies whether any events are present in a calendar table
    INPUT: verifyeventexist ('<window name>', '<component name>')

    OUTPUT: Returns 1 on success and 0 on no events."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYEVENTEXIST)
        _message = generatexml (command.VERIFYEVENTEXIST, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifyeventexist failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifyeventexist failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def ischildselected (windowName, componentName, item):
    """
    INPUT: ischildselected ('<window name>', '<component name>', '<item>')

    OUTPUT: Returns 1 on success and 0 on no events."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ISCHILDSELECTED)
        _message = generatexml (command.ISCHILDSELECTED, _requestId, windowName,
                                componentName, item)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('ischildselected failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('ischildselected failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def ischilditemselected (windowName, componentName, item):
    """
    INPUT: ischilditemselected ('<window name>', '<component name>', '<item>')

    OUTPUT: Returns 1 on success and 0 on no events."""
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.ISCHILDITEMSELECTED)
        _message = generatexml (command.ISCHILDITEMSELECTED, _requestId,
                                windowName, componentName, item)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('ischilditemselected failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('ischilditemselected failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def ischildindexselected (windowName, componentName, index):
    """
    INPUT: ischildindexselected ('<window name>', '<component name>', index)

    OUTPUT: Returns 1 on success and 0 on no events."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ISCHILDINDEXSELECTED)
        _message = generatexml (command.ISCHILDINDEXSELECTED, _requestId, windowName,
                                componentName, str (index))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('ischildindexselected failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('ischildindexselected failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def ischilditemindexselected (windowName, componentName, index):
    """
    INPUT: ischilditemindexselected ('<window name>', '<component name>', index)

    OUTPUT: Returns 1 on success and 0 on no events."""
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.ISCHILDITEMINDEXSELECTED)
        _message = generatexml (command.ISCHILDITEMINDEXSELECTED, _requestId,
                                windowName, componentName, str (index))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('ischilditemindexselected failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('ischilditemindexselected failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def check (windowName, componentName):
    """Check (tick) the check box state. 
    INPUT: check ('<window name>', '<component name>') 

    OUTPUT: Returns 1 if state is checked, else 0."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.CHECK)
        _message = generatexml (command.CHECK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('check failed: %s' % msg.value)
        raise
            
def uncheck (windowName, componentName):
    """Uncheck (un-tick) the check state.
    INPUT: uncheck ('<window name>', '<component name>')

    OUTPUT: Returns 1 if state is unchecked, else 0."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.UNCHECK)
        _message = generatexml (command.UNCHECK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('uncheck failed: %s' % msg.value)
        raise

def click (windowName, componentName):
    """click on radio button / check box / push button/ combo box/ radio menu item/ toggle button/ radio button.
    INPUT: click ('<window name>', '<component name>')

    OUTPUT: Clicks the component_name."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.CLICK)
        _message = generatexml (command.CLICK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('click failed: %s' % msg.value)
        raise

def press (windowName, componentName):
    """tell the button to press itself
    INPUT: press ('<window name>', '<component name>')

    OUTPUT: Presses the component_name."""
    try:
        _requestId = threading.currentThread ().getName () + str (command.PRESS)
        _message = generatexml (command.PRESS, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('press failed: %s' % msg.value)
        raise

def menucheck (windowName, componentName):
    """menucheck (tick) the menu item check state. 
    INPUT: menucheck ('<window name>', '<component name>') 

    OUTPUT: Returns 1 if state is checked, else 0."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.MENUCHECK)
        _message = generatexml (command.MENUCHECK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('menucheck failed: %s' % msg.value)
        raise

def menuuncheck (windowName, componentName):
    """menuuncheck (un-tick) the check state.
    INPUT: menuuncheck ('<window name>', '<component name>')

    OUTPUT: Returns 1 if state is unchecked, else 0."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.MENUUNCHECK)
        _message = generatexml (command.MENUUNCHECK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('menuuncheck failed: %s' % msg.value)
        raise

def verifycheck (windowName, componentName):
    """Checks the state of check box.
    INPUT: verifycheck ('<window name>', '<component name>')

    OUTPUT: If check box state is checked, then returns 1, else 0."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYCHECK)
        _message = generatexml (command.VERIFYCHECK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifycheck failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifycheck failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifyuncheck (windowName, componentName):
    """Checks the state of check box.
    INPUT: verifyuncheck ('<window name>', '<component name>')

    OUTPUT: If check box state is un-checked, then returns 1, else 0."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYUNCHECK)
        _message = generatexml (command.VERIFYUNCHECK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifyuncheck failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifyuncheck failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def menuitemenabled (windowName, menuItem):
    """
    INPUT: menuitemenabled ('<window name>', '<menu item>')

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.MENUITEMENABLED)
        _message = generatexml (command.MENUITEMENABLED, _requestId, windowName, menuItem)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('menuitemenabled failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('menuitemenabled failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifymenucheck (windowName, menuItem):
    """
    INPUT: verifymenucheck ('<window name>', '<menu item>')

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYMENUCHECK)
        _message = generatexml (command.VERIFYMENUCHECK, _requestId, windowName, menuItem)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifymenucheck failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifymenucheck failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifymenuuncheck (windowName, menuItem):
    """
    INPUT: verifymenuuncheck ('<window name>', '<menu item>')

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYMENUUNCHECK)
        _message = generatexml (command.VERIFYMENUUNCHECK, _requestId, windowName, menuItem)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifymenuuncheck failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifymenuuncheck failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifydropdown (windowName, componentName):
    """
    INPUT: verifydropdown ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYDROPDOWN)
        _message = generatexml (command.VERIFYDROPDOWN, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifydropdown failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifydropdown failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

# CheckMenuItem Functions

def selectmenuitem (windowName, menuHierarchy):
    """Selects the menu item specified.
    INPUT: selectmenuitem ('<window name>', '<menu hierarchy>')

    OUTPUT: 1 on success, LdtpExecutionError on failure."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTMENUITEM)
        _message = generatexml (command.SELECTMENUITEM, _requestId, windowName, menuHierarchy)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectmenuitem failed: %s' % msg.value)
        raise

#  Combobox Functions

def hidelist (windowName, componentName):
    """ Hides combo box drop down list in the current dialog. 
    Suppose in previous operation one testcase has clicked on combo box, 
    its drop down list will be displayed. If further no any operation has 
    been done on that combo box then to close that drop down list 'HideList' 
    action is required.
    INPUT: hidelist ('<window name>', '<component name>') 

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.HIDELIST)
        _message = generatexml (command.HIDELIST, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('hidelist failed: %s' % msg.value)
        raise

def selectindex (windowName, componentName, index):
    """ SelectIndex action will select an item from combo box where value of index is pointing to its position in list/menu.
    INPUT: selectindex ('<window name>', '<component name>', <index>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTINDEX)
        _message = generatexml (command.SELECTINDEX, _requestId, windowName, componentName, str (index))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectindex failed: %s' % msg.value)
        raise

def unselectindex (windowName, componentName, index):
    """ Unselectindex action will unselect an item from layered pane where value
    of index is pointing to its position in list.
    INPUT: unselectindex ('<window name>', '<component name>', <index>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.UNSELECTINDEX)
        _message = generatexml (command.UNSELECTINDEX, _requestId,
                                windowName, componentName, str (index))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('unselectindex failed: %s' % msg.value)
        raise

def unselectitemindex (windowName, componentName, index):
    """ Unselectitemindex action will unselect an item from layered pane
    where value of index is from the previously selected items
    INPUT: unselectitemindex ('<window name>', '<component name>', <index>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.UNSELECTITEMINDEX)
        _message = generatexml (command.UNSELECTITEMINDEX, _requestId,
                                windowName, componentName, str (index))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('unselectitemindex failed: %s' % msg.value)
        raise

def settextvalue (windowName, componentName, text):
    """puts the text into the component given by the component name
    INPUT: settextvalue ('<window name>', '<component name>', '<text>')

    OUTPUT: returns 1 on success and 0 otherwise"""
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.SETTEXTVALUE)
        _message = generatexml (command.SETTEXTVALUE, _requestId,
                                windowName, componentName, text)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('settextvalue failed: %s' % msg.value)
        raise

def gettextvalue (windowName, componentName, startPosition = None, endPosition = None):
    """
    INPUT: gettextvalue ('<window name>', '<component name>', [start position], [end position])

    OUTPUT: returns 1 on success and 0 otherwise"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.GETTEXTVALUE)
        _message = generatexml (command.GETTEXTVALUE, _requestId, windowName,
                                componentName, startPosition, endPosition)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('gettextvalue failed: %s' % msg.value)
        raise

def activatetext (windowName, componentName):
    """
    INPUT: activatetext ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ACTIVATETEXT)
        _message = generatexml (command.ACTIVATETEXT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('activatetext failed: %s' % msg.value)
        raise

def appendtext (windowName, componentName, text):
    """
    INPUT: appendtext ('<window name>', '<component name'>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.APPENDTEXT)
        _message = generatexml (command.APPENDTEXT, _requestId, windowName, componentName, text)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('appendtext failed: %s' % msg.value)
        raise

def getcursorposition (windowName, componentName):
    """
    INPUT: windowName --> Name in the title bar
           componentName --> Name of the object (TextBox)

    OUTPUT: postion of the text cursor in the object (integer)"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETCURSORPOSITION)
        _message = generatexml (command.GETCURSORPOSITION, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return long (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getcursorposition failed: %s' % msg.value)
        raise

def setcursorposition (windowName, componentName, position):
    """
    INPUT: windowName --> Name in the title bar
           componentName --> Name of the object (TextBox)
           position --> Position in the TextBox where cursor is to be moved to

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETCURSORPOSITION)
        _message = generatexml (command.SETCURSORPOSITION, _requestId, windowName, componentName, str (position))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('setcursorposition failed: %s' % msg.value)
        raise
    
def capturetofile (windowName, componentName, fileName = None):
    """
    INPUT: capturetofile ('<window name>', '<component name>'[, '<file name>'])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.CAPTURETOFILE)
        _message = generatexml (command.CAPTURETOFILE, _requestId, windowName, componentName, fileName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('capturetofile failed: %s' % msg.value)
        raise

def showlist (windowName, componentName):
    """ Displays combo box drop down list in the current dialog.
    INPUT: showlist ('windowName', 'componentName')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SHOWLIST)
        _message = generatexml (command.SHOWLIST, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('showlist failed: %s' % msg.value)
        raise

def verifyhidelist (windowName, componentName):
    """ Verifies if combo box drop down list in the current dialog is not visible.
    INPUT: verifyhidelist ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYHIDELIST)
        _message = generatexml (command.VERIFYHIDELIST, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifyhidelist failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifyhidelist failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifyselect (windowName, componentName, selectArgument):
    """ VerfySelect action will verify if combo box is set to value given in argument.
    INPUT: verifyselect ('<window name>', '<component name>', '<argument>')

    OUTPUT: VerifySelect function will try to find if text box associated with combo box
    is set to value specified in the argument."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSELECT)
        _message = generatexml (command.VERIFYSELECT, _requestId, windowName, componentName, selectArgument)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifyselect failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifyselect failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifysettext (windowName, componentName, text):
    """ checks if the text is inserted into the component given by the component name
    INPUT: verifysettext ('<window name>', '<component name>', '<text>')

    OUTPUT: returns 1 if the text is inserted into the specified component else returns 0 """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSETTEXT)
        _message = generatexml (command.VERIFYSETTEXT, _requestId, windowName, componentName, text)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifysettext failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifysettext failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifyshowlist (windowName, componentName):
    """ Verifies if combo box drop down list in the current dialog is visible.
    INPUT: verifyshowlist ('<window name>', '<component name>')

    OUTPUT: Combo box will generally have a list as its child or a menu as its child. 
    So this function gets object handle of list or menu object, checks if list or 
    menu items of combo box is visible, if yes then return zero else minus one."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSHOWLIST)
        _message = generatexml (command.VERIFYSHOWLIST, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifyshowlist failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifyshowlist failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def comboselect (windowName, componentName, menuItemName):
    """Select a menu item or list item in a combo box
    INPUT: comboselect ('<window name>', '<component name>', '<menu item name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.COMBOSELECT)
        _message = generatexml (command.COMBOSELECT, _requestId, windowName, componentName, menuItemName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('comboselect failed: %s' % msg.value)
        raise

def comboselectindex (windowName, componentName, index):
    """
    INPUT: comboselectindex ('<window name>', '<component name>', <index>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.COMBOSELECTINDEX)
        _message = generatexml (command.COMBOSELECTINDEX, _requestId, windowName,
                                componentName, str (index))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('comboselectindex failed: %s' % msg.value)
        raise

def comparetextproperty (windowName, componentName, textProperty,
                         startPosition = None, endPosition = None):
    """
    INPUT: comparetextproperty ('<window name>', '<component name>', <text property>[, <start position>[, <end position>]])

    OUTPUT: 1 on success, 0 on failure"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.COMPARETEXTPROPERTY)
        _message = generatexml (command.COMPARETEXTPROPERTY, _requestId, windowName,
                                componentName, textProperty, startPosition, endPosition)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('comparetextproperty failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('comparetextproperty failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def containstextproperty (windowName, componentName, textProperty,
                          startPosition = None, endPosition = None):
    """
    INPUT: containstextproperty ('<window name>', '<component name>',
                <text property>[, <start position>[, <end position>]])

    OUTPUT: 1 on success, 0 on failure"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.CONTAINSTEXTPROPERTY)
        _message = generatexml (command.CONTAINSTEXTPROPERTY, _requestId, windowName,
                                componentName, textProperty, startPosition, endPosition)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('containstextproperty failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('containstextproperty failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def gettextproperty (windowName, componentName, startPosition = None, endPosition = None):
    """
    INPUT: gettextproperty ('<window name>', '<component name>'[, <start position>[, <end position>]])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETTEXTPROPERTY)
        _message = generatexml (command.GETTEXTPROPERTY, _requestId, windowName, componentName, startPosition, endPosition)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('gettextproperty failed: %s' % msg.value)
        raise

def copytext (windowName, componentName, startPosition = None, endPosition = None):
    """
    INPUT: copytext ('<window name>', '<component name>'[, <start position>[, <end position>]])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.COPYTEXT)
        _message = generatexml (command.COPYTEXT, _requestId, windowName, componentName, startPosition, endPosition)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('copytext failed: %s' % msg.value)
        raise

def cuttext (windowName, componentName, startPosition = None, endPosition = None):
    """
    INPUT: cuttext ('<window name>', '<component name>'[, <start position>[, <end position>]])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.CUTTEXT)
        _message = generatexml (command.CUTTEXT, _requestId, windowName, componentName, startPosition, endPosition)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('cuttext failed: %s' % msg.value)
        raise

def inserttext (windowName, componentName, position, text):
    """
    INPUT: inserttext ('<window name>', '<component name>', <position>, '<text to be inserted>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.INSERTTEXT)
        _message = generatexml (command.INSERTTEXT, _requestId, windowName, componentName, str (position), text)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('inserttext failed: %s' % msg.value)
        raise

def deletetext (windowName, componentName, startPosition = None, endPosition = None):
    """
    INPUT: deletetext ('<window name>', '<component name>'[, <start position>[, <end position>]])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.DELETETEXT)
        _message = generatexml (command.DELETETEXT, _requestId, windowName, componentName, startPosition, endPosition)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('deletetext failed: %s' % msg.value)
        raise

def selecttextbyindexandregion (windowName, componentName,
                startPosition = None, endPosition = None, selectionNum = None):
    """
    INPUT: selecttextbyindexandregion ('<window name>', '<component name>'[, <start position>[, <end position>[, <selection number>]]])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    if startPosition != None:
        startPosition = str (startPosition)
    if endPosition != None:
        endPosition = str (endPosition)
    if selectionNum != None:
        selectionNum = str (selectionNum)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTTEXTBYINDEXANDREGION)
        _message = generatexml (command.SELECTTEXTBYINDEXANDREGION, _requestId, windowName, \
                        componentName, startPosition, endPosition, selectionNum)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selecttextbyindexandregion failed: %s' % msg.value)
        raise

def selecttextbyname (windowName, componentName):
    """
    INPUT: selecttextbyname ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTTEXTBYNAME)
        _message = generatexml (command.SELECTTEXTBYNAME, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selecttextbyname failed: %s' % msg.value)
        raise

def pastetext (windowName, componentName, position = None):
    """
    INPUT: pastetext ('<window name>', '<component name>'[, <position>])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    if position != None:
        position = str (position)
    try:
        _requestId  = threading.currentThread ().getName () + str (command.PASTETEXT)
        _message = generatexml (command.PASTETEXT, _requestId, windowName, componentName, position)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('pastetext failed: %s' % msg.value)
        raise

def expandtablecell (windowName, componentName, position):
    """
    INPUT: expandtablecell ('<window name>', '<component name>', <position>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.EXPANDTABLECELL)
        _message = generatexml (command.EXPANDTABLECELL, _requestId, windowName, componentName, str (position))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('expandtablecell failed: %s' % msg.value)
        raise

def getcharcount (windowName, componentName):
    """
    INPUT: getcharcount ('<window name>', '<component name>')

    OUTPUT: Character count on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETCHARCOUNT)
        _message = generatexml (command.GETCHARCOUNT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return int (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getcharcount failed: %s' % msg.value)
        raise

def getlabel (windowName, componentName):
    """
    INPUT: getlabel ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETLABEL)
        _message = generatexml (command.GETLABEL, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('getlabel failed: %s' % msg.value)
        raise

def getlabelatindex (windowName, componentName, index):
    """
    INPUT: getlabelatindex ('<window name>', '<component name>', <index>)

    OUTPUT: Label from the given index on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETLABELATINDEX)
        _message = generatexml (command.GETLABELATINDEX, _requestId, windowName, componentName, index)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('getlabelatindex failed: %s' % msg.value)
        raise

def selectlabelspanelbyname (windowName, labelName):
    """
    INPUT: selectlabelspanelbyname ('<window name>', '<label name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTLABELSPANELBYNAME)
        _message = generatexml (command.SELECTLABELSPANELBYNAME, _requestId, windowName, labelName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectlabelspanelbyname failed: %s' % msg.value)
        raise

def getpanelchildcount (windowName, componentName):
    """
    INPUT: getpanelchildcount ('<window name>', '<component name>')

    OUTPUT: Panel's child count on success in int type, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETPANELCHILDCOUNT)
        _message = generatexml (command.GETPANELCHILDCOUNT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return int (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getpanelchildcount failed: %s' % msg.value)
        raise

def selecteditemcount (windowName, componentName):
    """
    INPUT: selecteditemcount ('<window name>', '<component name>')

    OUTPUT: Layered panes selected item count on success in int type,
    LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.SELECTEDITEMCOUNT)
        _message = generatexml (command.SELECTEDITEMCOUNT, _requestId,
                                windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return int (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('selecteditemcount failed: %s' % msg.value)
        raise

def getslidervalue (windowName, componentName):
    """
    INPUT: getslidervalue ('<window name>', '<component name>')

    OUTPUT: slider current value on success in float type, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETSLIDERVALUE)
        _message = generatexml (command.GETSLIDERVALUE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return float (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getslidervalue failed: %s' % msg.value)
        raise

def getstatusbartext (windowName, componentName):
    """
    INPUT: getstatusbartext ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETSTATUSBARTEXT)
        _message = generatexml (command.GETSTATUSBARTEXT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('getstatusbartext failed: %s' % msg.value)
        raise

def gettreetablerowindex (windowName, componentName, rowText):
    """
    INPUT: gettreetablerowindex ('<window name>', '<component name>', '<row text>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETTREETABLEROWINDEX)
        _message = generatexml (command.GETTREETABLEROWINDEX, _requestId, windowName, componentName, rowText)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return int (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('gettreetablerowindex failed: %s' % msg.value)
        raise

def grabfocus (windowName, componentName = None):
    """
    INPUT: grabfocus ('<window name>'[, '<component name>'])

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GRABFOCUS)
        if (componentName != None):
            _message = generatexml (command.GRABFOCUS, _requestId, windowName, componentName)
        else:
            _message = generatexml (command.GRABFOCUS, _requestId, windowName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('grabfocus failed: %s' % msg.value)
        raise

def istextstateenabled (windowName, componentName):
    """
    INPUT: istextstateenabled ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ISTEXTSTATEENABLED)
        _message = generatexml (command.ISTEXTSTATEENABLED, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('istextstateenabled failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('istextstateenabled failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def listsubmenus (windowName, componentName):
    """
    INPUT: listsubmenus ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.LISTSUBMENUS)
        _message = generatexml (command.LISTSUBMENUS, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('listsubmenus failed: %s' % msg.value)
        raise

def invokemenu (windowName, componentName):
    """
    INPUT: invokemenu ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.INVOKEMENU)
        _message = generatexml (command.INVOKEMENU, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('invokemenu failed: %s' % msg.value)
        raise

def rightclick (windowName, componentName, data = None):
    """
    INPUT: rightclick ('<window name>', '<component name>', [, '<data>'])
    data will be used to select the row in table / tree table

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.RIGHTCLICK)
        _message = generatexml (command.RIGHTCLICK, _requestId, windowName, componentName, data)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('rightclick failed: %s' % msg.value)
        raise

def selectcalendardate (windowName, componentName, day, month, year):
    """
    INPUT: selectcalendardate ('<window name>', '<component name>', <day>, <month>, <year>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTCALENDARDATE)
        _message = generatexml (command.SELECTCALENDARDATE, _requestId, windowName, componentName, str (day), str (month), str (year))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectcalendardate failed: %s' % msg.value)
        raise

def selectitem (windowName, componentName, item):
    """
    INPUT: selectitem ('<window name>', '<component name>', '<list item to be selected in combo box>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTITEM)
        _message = generatexml (command.SELECTITEM, _requestId, windowName, componentName, item)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectitem failed: %s' % msg.value)
        raise

def unselectitem (windowName, componentName, item):
    """
    INPUT: unselectitem ('<window name>', '<component name>', '<list item to be selected in combo box>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.UNSELECTITEM)
        _message = generatexml (command.UNSELECTITEM, _requestId, windowName, componentName, item)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('unselectitem failed: %s' % msg.value)
        raise

def selectall (windowName, componentName):
    """
    INPUT: selectall ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTALL)
        _message = generatexml (command.SELECTALL, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectall failed: %s' % msg.value)
        raise

def unselectall (windowName, componentName):
    """
    INPUT: unselectall ('<window name>', '<component name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.UNSELECTALL)
        _message = generatexml (command.UNSELECTALL, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('unselectall failed: %s' % msg.value)
        raise

def selecttextitem (windowName, componentName, item):
    """
    INPUT: selecttextitem ('<window name>', '<component name>', '<text item to be selected in combo box>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTTEXTITEM)
        _message = generatexml (command.SELECTTEXTITEM, _requestId, windowName, componentName, item)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selecttextitem failed: %s' % msg.value)
        raise

def setcellvalue (windowName, componentName, row, column, item):
    """
    INPUT: setcellvalue ('<window name>', '<component name>', <row number>, <column number>, '<text to set>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETCELLVALUE)
        _message = generatexml (command.SETCELLVALUE, _requestId, windowName,
                                componentName, str (row), str (column), item)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('setcellvalue failed: %s' % msg.value)
        raise

def sortcolumn (windowName, componentName, item):
    """
    INPUT: sortcolumn ('<window name>', '<component name>', '<sort based on column name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SORTCOLUMN)
        _message = generatexml (command.SORTCOLUMN, _requestId, windowName, componentName, item)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('sortcolumn failed: %s' % msg.value)
        raise

def sortcolumnindex (windowName, componentName, index):
    """
    INPUT: sortcolumnindex ('<window name>', '<component name>', <sort based on column index>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SORTCOLUMNINDEX)
        _message = generatexml (command.SORTCOLUMNINDEX, _requestId, windowName, componentName, str (index))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('sortcolumnindex failed: %s' % msg.value)
        raise

def verifybuttoncount (windowName, componentName, count):
    """
    INPUT: verifybuttoncount ('<window name>', '<component name>', <button count>)

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYBUTTONCOUNT)
        _message = generatexml (command.VERIFYBUTTONCOUNT, _requestId, windowName, componentName, str (count))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifybuttoncount failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifybuttoncount failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

#  General Functions

def waittillguiexist (windowName, componentName = '', guiTimeOut = None):
    """ If the given window name exist, this function returns 1. If window doesnot exist, 
    then this function returns 0. Difference between guiexist and waitguiexist is, 
    waitguiexist waits for maximum 30 seconds. Still the window doesn't appear, 
    then 0 is returned. We can set the environment variable 'GUI_TIMEOUT' to change 
    the default waiting time.
    INPUT: waittillguiexist ('<window name>')

    OUTPUT: returns 1 on success and 0 on no existing window """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.WAITTILLGUIEXIST)
        if guiTimeOut:
            guiTimeOut = str (guiTimeOut)
        else:
            guiTimeOut = os.getenv ('GUI_TIMEOUT')
        _message = generatexml (command.WAITTILLGUIEXIST, _requestId, windowName,
                                componentName, guiTimeOut)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            # Window does not exist
            logFailures ("Window '%s' does not exist" % windowName.encode ('utf-8'),
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('waittillguiexist failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def waittillguinotexist (windowName, componentName = '', guiTimeOut = None):
    """ If the given window name does not exist, this function returns 1. If window exist, 
    then this function returns 0. Difference between guiexist and waitguinotexist is, 
    waitguinotexist waits for maximum 30 seconds. Still the window does not disappear, 
    then 0 is returned. We can set the environment variable 'GUI_TIMEOUT' to change the 
    default waiting time.
    INPUT: waittillguinotexist ('<window name>')

    OUTPUT: returns 1 on success and 0 on no existing window """
    try:
        if guiTimeOut:
            guiTimeOut = str (guiTimeOut)
        else:
            guiTimeOut = os.getenv ('GUI_TIMEOUT')
        _requestId  = threading.currentThread ().getName () + str (command.WAITTILLGUINOTEXIST)
        _message = generatexml (command.WAITTILLGUINOTEXIST, _requestId, windowName,
                                componentName, guiTimeOut)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ("Window '%s' exist" % windowName.encode ('utf-8'),
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('waittillguinotexist failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def guiexist (windowName, componentName = ''):
    """ If the given window name exist, this function returns 1. If window doesnot exist, 
    then this function returns 0.
    INPUT: guiexist ('<window name>') 

    OUTPUT: returns 1 on success and 0 on no existing window """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GUIEXIST)
        _message = generatexml (command.GUIEXIST, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('guiexist failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('guiexist failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def guitimeout (guiTimeOut = 30):
    """ Set gui time out period, If this function called with out arguments, then the default timeout is 30 seconds
    INPUT: guitimeout (30) 

    OUTPUT: returns 1 on success and 0 on failure"""
    try:
        global _ldtpGuiTimeout
        if type (guiTimeOut) == int:
            guiTimeOut = str (guiTimeOut)
        else:
            guiTimeOut = os.getenv ('GUI_TIMEOUT')
            if guiTimeOut is None:
                guiTimeOut = str (30)
        _ldtpGuiTimeout = int (guiTimeOut)
        _requestId  = threading.currentThread ().getName () + str (command.GUITIMEOUT)
        _message = generatexml (command.GUITIMEOUT, _requestId, guiTimeOut)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('guitimeout failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('guitimeout failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def objtimeout (objTimeOut = 5):
    """ Set obj time out period, If this function called with out arguments, then the default timeout is 5 seconds
    INPUT: objtimeout (5) 

    OUTPUT: returns 1 on success and 0 on failure"""
    try:
        if type (objTimeOut) == int:
            objTimeOut = str (objTimeOut)
        else:
            objTimeOut = os.getenv ('OBJ_TIMEOUT')
            if objTimeOut is None:
                objTimeOut = str (5)
        _requestId  = threading.currentThread ().getName () + str (command.OBJTIMEOUT)
        _message = generatexml (command.OBJTIMEOUT, _requestId, objTimeOut)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('objtimeout failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('objtimeout failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def objectexist (windowName, componentName):
    """ Check that the given componentName exists in the hierarchy under
    windowName. Returns 1 if componentName exists, 0 otherwise
    INPUT: objectexist ('<window name>', '<component name>')

    OUTPUT: returns 1 if component exists, 0 otherwise """
    try:
        objinfo = getobjectinfo (windowName, componentName)
        if objinfo:
            return 1
    except LdtpExecutionError:
        return 0

def initappmap (appmapFileName):
    """ Application map will be loaded
    INPUT: initappmap ('<application map name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.INITAPPMAP)
        _message = generatexml (command.INITAPPMAP, _requestId, appmapFileName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('initappmap failed: %s' % msg.value)
        raise

def setappmap (appmapFileName):
    """ A new application map will be loaded. Existing appmap will be unloaded.
    INPUT: setappmap ('<new application map name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETAPPMAP)
        _message = generatexml (command.SETAPPMAP, _requestId, appmapFileName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('setappmap failed: %s' % msg.value)
        raise

def setcontext (oldContext, newContext):
    """ Set context of window
    INPUT: setcontext ('<old window title>', '<new window title>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""

    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETCONTEXT)
        _message = generatexml (command.SETCONTEXT, _requestId, oldContext, newContext)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('setcontext failed: %s' % msg.value)
        raise

def releasecontext (oldContext = "", newContext = ""):
    """ Release context of window
    INPUT: releasecontext (oldContext, newContext)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""

    try:
        _requestId  = threading.currentThread ().getName () + str (command.RELEASECONTEXT)
        _message = generatexml (command.RELEASECONTEXT, _requestId, oldContext, newContext)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('releasecontext failed: %s' % msg.value)
        raise

def onwindowcreate (windowTitle, callbackFuncName, *args):
    """ On new window creation event, notify
    If registered with same title more than once, only the last title will
    be used, previously registered will be ignored
    INPUT: onwindowcreate ('<window title to watch>',
                <callback function to be called on window create event>)

    OUTPUT: 1 on success, LdtpExecutionError on failure"""

    try:
        # Since LDTP engine returns the response to the socket who initiated the
        # onwindowcreate call, let us initiate the call in the new thread
        threadId = thread.start_new_thread (invokecallback, (windowTitle, args))
        if _ldtpDebug:
            print windowTitle
        # Check for previous instance of the window title
        if _callbackFunctions.has_key (windowTitle):
            # delete the previous instance
            del _callbackFunctions [windowTitle]
        _callbackFunctions [windowTitle] = [callbackFuncName, threadId]
        return 1
    except LdtpExecutionError, msg:
        logFailures ('onwindowcreate failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def removecallback (windowTitle):
    """ remove reigstered callback
    INPUT: removecallback ('<window title to watch>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""

    try:
        # Check for window title in the list of callback
        if _callbackFunctions.has_key (windowTitle):
            # Exit last thread registered with same window title
            _callbackFunctions [windowTitle][1].exit ()
            # delete the previous instance
            del _callbackFunctions [windowTitle]
            _requestId = threading.currentThread ().getName () + str (command.REMOVECALLBACK)
            _message = generatexml (command.REMOVECALLBACK, _requestId, windowTitle)
            sendpacket (_message)
            _responseStatus, _responseData = getresponse (_requestId)
            if _responseStatus [0] != 0:
                raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('removecallback failed: %s' % msg.value)
        raise

def reinitldtp ():
    """ Reinitialize LDTP, which will close existing at-spi connection
    and reestablish it.
    INPUT: reinitldtp ()

    OUTPUT: 1 on success, LdtpExecutionError on failure"""

    try:
        _requestId  = threading.currentThread ().getName () + str (command.REINITLDTP)
        _message = generatexml (command.REINITLDTP, _requestId)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('reinitldtp failed: %s' % msg.value)
        raise

def remap (windowName, componentName = ''):
    """ We can handle dynamically created widgets (meaning widgets created at run time) using 
    this remap function. Calling remap will generate appmap for the given dialog at run 
    time and update the hash table. Then we can access the new widgets.

    Please note that the <application-name> should be same as the one given as the commmand-line 
    argument for appmap generation. 
    INPUT: remap ('<application-name>', '<window name>')

    OUTPUT: It uses the same logic that appmap module uses to generate appmap."""

    try:
        _requestId  = threading.currentThread ().getName () + str (command.REMAP)
        _message = generatexml (command.REMAP, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('remap failed: %s' % msg.value)
        raise

# Deprecated undoremap function as it is no longer needed

def undoremap (applicationName, windowName):
    """ This function is deprecated and no longer needs to be called.
    This function is a no-op and immediately returns, but exists to
    maintain compatibility with scripts which call it.

    INPUT: undoremap ('<application-name>', 'window name>')
    """
    return 1

#  Log Function

def startldtplog (logFileName, fileOverWrite = 1):
    """ Start logging on the specified file. second arugment is optional and 1 is default value
    0 - Append log to an existing file
    1 - Write log to a new file. If file already exist, 
    then erase existing file content and start log

    INPUT: startldtplog ('<log file name>', [0 or 1])

    OUTPUT: Log file will be created if log file is not present in any case. If second argument is 1, 
    then existing file content will be erased. If second argument is 0, then new logs will be 
    append to existing log.

    It returns 1 on Success and 0 on error  """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.STARTLOG)
        _message = generatexml (command.STARTLOG, _requestId, logFileName, str (fileOverWrite))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            pass
        return 1
    except LdtpExecutionError:
        pass

def ldtplog (message, priority = ''):
    """ Logs the message in the log.xml with the tag which can be viewed after the execution of scripts
    INPUT: ldtplog('<Message to be logged>','<tag>')

    OUTPUT: The required message will be logged into the log.xml on execution of scripts
    It returns 1 on Success and 0 on error  """

    try:
        _requestId  = threading.currentThread ().getName () + str (command.LOG)
        _message = generatexml (command.LOG, _requestId, message, priority)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            pass
        return 1
    except ConnectionLost:
        pass
    except LdtpExecutionError:
        pass

def stopldtplog ():
    """ Stop logging.
    INPUT: stopldtplog()

    OUTPUT: If a log file has been previously opened for logging, that file pointer will be closed. 
    So that the new logging will not be appened to the log file. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.STOPLOG)
        _message = generatexml (command.STOPLOG, _requestId)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            pass
        return 1
    except ConnectionLost:
        pass
    except LdtpExecutionError:
        pass

def setloglevel (level):
    return setInternalLogLevel (level, logger)

def startlog (logFileName, fileOverWrite = 1, screenshotDir = None):
    """ Start logging on the specified file. second arugment is optional and 1 is default value
    0 - Append log to an existing file
    1 - Write log to a new file. If file already exist, 
    then erase existing file content and start log

    INPUT: startlog ('<log file name>', [0 or 1])

    OUTPUT: Log file will be created if log file is not present in any case.
    If second argument is 1, then existing file content will be erased.
    If second argument is 0, then new logs will be append to existing log.

    It returns 1 on Success and 0 on error  """
    if logFileName is None:
        return 0
    global logger, xmlhdlr, _logDir, _screenshotDir
    _index = logFileName.rfind ('/')
    if screenshotDir:
        _screenshotDir = screenshotDir
        _status = commands.getstatusoutput ('mkdir -p %s' % _screenshotDir)
        if _ldtpDebug and _status [0] != 0:
            print _status [1]
    if _index != -1:
        _logDir = logFileName [ : _index]
    else:
        _logDir = os.getcwd ()
    logger, xmlhdlr = startInternalLog (logFileName, fileOverWrite, 'XML')
    if logger is not None:
        return 1
    else:
        return 0

def log (message, priority = None):
    """ Logs the message in the log.xml with the tag which can be viewed after the execution of scripts
    INPUT: log('<Message to be logged>','<tag>')

    OUTPUT: The required message will be logged into the log.xml on execution of scripts
    It returns 1 on Success and 0 on error  """
    if logger == None:
        _logFile = "%s/log-%s.xml" % (_logDir, time.strftime ("%m-%d-%Y-%H-%M"))
        startlog (_logFile)
    return internalLog (message, priority, logger)

def addlogger (confFileName):
    addInternalLogger (confFileName)
    return 1

def stoplog (handler = None):
    """ Stop logging.
    INPUT: stoplog()

    OUTPUT: If a log file has been previously opened for logging, that file pointer will be closed. 
    So that the new logging will not be appened to the log file. """
    return internalStopLog (handler, xmlhdlr, logger)

def selectrow (windowName, tableName, rowText, nthMatch = 0):
    """Selects the row with the text in the table
    INPUT: selectrow ('<window name>', '<table name>', '<row text to be selected>')

    OUTPUT: Returns 1 on success and 0 on error."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTROW)
        _message = generatexml (command.SELECTROW, _requestId, windowName, tableName, rowText, str (nthMatch))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectrow failed: %s' % msg.value)
        raise
    
def selectrowindex (windowName, tableName, rowIndex):
    """Selects the row with the index in the table
    INPUT: selectrowindex ('<window name>', '<table name>', <row index>)

    OUTPUT: Returns 1 on success and 0 on error."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTROWINDEX)
        _message = generatexml (command.SELECTROWINDEX, _requestId, windowName, tableName, str (rowIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectrowindex failed: %s' % msg.value)
        raise

#  MenuItem Functions

def doesmenuitemexist (windowName, menuHierarchy):
    """ checks if the specified menuitem specified in the menu hierarchy is present or not
    INPUT: doesmenuitemexist ('<window name>', '<menu hierarchy>')

    OUTPUT: Returns 1 if the menuitem is present and 0 otherwise """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.DOESMENUITEMEXIST)
        _message = generatexml (command.DOESMENUITEMEXIST, _requestId, windowName, menuHierarchy)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

#  Panel Functions

def selectpanel (windowName, componentName, panelNumber):
    """ Select a panel using the panel number in a list of panels
    INPUT: selectpanel ('<window name>', '<component name>', <panel number>)

    OUTPUT: Returns 1 on success and 0 on error """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTPANEL)
        _message = generatexml (command.SELECTPANEL, _requestId, windowName, componentName, str (panelNumber))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectpanel failed: %s' % msg.value)
        raise

def selectpanelname (windowName, componentName, panelName):
    """ Select a panel using the panel name in a list of panels
    INPUT: selectpanelname ('<window name>', '<component name>', <panel name>)

    OUTPUT: Returns 1 on success and 0 on error """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTPANELNAME)
        _message = generatexml (command.SELECTPANELNAME, _requestId, windowName, componentName, panelName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectpanelname failed: %s' % msg.value)
        raise

#  PushButton Functions

def verifypushbutton (windowName, componentName):
    """ Verify whether the given object is push button or not.
    INPUT: verifypushbutton ('<window name>', '<component name>')

    OUTPUT: Returns 1 if object is push button, else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYPUSHBUTTON)
        _message = generatexml (command.VERIFYPUSHBUTTON, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def stateenabled (windowName, componentName):
    """ Checks the radio button object state enabled or not
    INPUT: stateenabled ('<window name>', '<component name>')

    OUTPUT: Returns 1 if state is enabled, else 0."""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.STATEENABLED)
        _message = generatexml (command.STATEENABLED, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

#  ScrollBar Functions

def onedown (windowName, componentName, iterations):
    """ Move the scroll bar down 'n' times, where 'n' is the number of iterations 
    specified in  the argument field.
    INPUT: onedown ('<window name>', '<scroll component name>', '<number of iterations>')

    OUTPUT: Scrolls down if value does not exceed the maximum limit, else fails. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ONEDOWN)
        _message = generatexml (command.ONEDOWN, _requestId, windowName, componentName, str (iterations))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('onedown failed: %s' % msg.value)
        raise

def oneleft (windowName, componentName, iterations):
    """ Move the (horizontal) scroll bar left 'n' times,
    where 'n' is the number of iterations specified in the argument field.
    INPUT: oneleft ('<window name>', '<scroll component name>', '<number of iterations>')

    OUTPUT: Scrolls left if value does not exceed the maximum limit, else fails. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ONELEFT)
        _message = generatexml (command.ONELEFT, _requestId, windowName, componentName, str (iterations))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('oneleft failed: %s' % msg.value)
        raise

def oneright (windowName, componentName, iterations):
    """ Move the (horizontal) scroll bar right 'n' times,
    where 'n' is the number of iterations specified in the argument field.
    INPUT: oneright ('<window name>', '<scroll component name>', '<number of iterations>')

    OUTPUT: Scrolls right if value does not exceed the maximum limit, else fails. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ONERIGHT)
        _message = generatexml (command.ONERIGHT, _requestId, windowName, componentName, str (iterations))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('oneright failed: %s' % msg.value)
        raise

def oneup (windowName, componentName, iterations):
    """ Move the scroll bar up 'n' times, where 'n' is the number of iterations 
    specified in  the argument field.
    INPUT: oneup ('<window name>', '<scroll component name>', '<number of iterations>')

    OUTPUT: Scrolls up if value does not exceed the maximum limit, else fails. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.ONEUP)
        _message = generatexml (command.ONEUP, _requestId, windowName, componentName, str (iterations))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('oneup failed: %s' % msg.value)
        raise
    
def scrolldown (windowName, componentName):
    """ Move the (vertical) scroll bar to the bottom.
    INPUT: scrolldown ('<window name>', '<scroll component name>')

    OUTPUT: Returns 1 if action is performed, else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SCROLLDOWN)
        _message = generatexml (command.SCROLLDOWN, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('scrolldown failed: %s' % msg.value)
        raise

def scrollleft (windowName, componentName):
    """ Move the (horizontal) scroll bar to the left.
    INPUT: scrollleft ('<window name>', '<scroll component name>')

    OUTPUT: Returns 1 if action is performed, else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SCROLLLEFT)
        _message = generatexml (command.SCROLLLEFT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('scrollleft failed: %s' % msg.value)
        raise

def scrollright (windowName, componentName):
    """ Move the (horizontal) scroll bar to the right.
    INPUT: scrollright ('<window name>', '<scroll component name>')

    OUTPUT: Returns 1 if action is performed, else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SCROLLRIGHT)
        _message = generatexml (command.SCROLLRIGHT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('scrollright failed: %s' % msg.value)
        raise
    
def scrollup (windowName, componentName):
    """ Move the (vertical) scroll bar to the top.
    INPUT: scrollup ('<window name>', '<scroll component name>')

    OUTPUT: Returns 1 if action is performed, else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SCROLLUP)
        _message = generatexml (command.SCROLLUP, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('scrollup failed: %s' % msg.value)
        raise

def verifyscrollbar (windowName, componentName):
    """
    INPUT: verifyscrollbar ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSCROLLBAR)
        _message = generatexml (command.VERIFYSCROLLBAR, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifyscrollbarvertical (windowName, componentName):
    """
    INPUT: verifyscrollbarvertical ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSCROLLBARVERTICAL)
        _message = generatexml (command.VERIFYSCROLLBARVERTICAL, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifyscrollbarhorizontal (windowName, componentName):
    """
    INPUT: verifyscrollbarhorizontal ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSCROLLBARHORIZONTAL)
        _message = generatexml (command.VERIFYSCROLLBARHORIZONTAL, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifyslider (windowName, componentName):
    """
    INPUT: verifyslider ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSLIDER)
        _message = generatexml (command.VERIFYSLIDER, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifyslidervertical (windowName, componentName):
    """
    INPUT: verifyslidervertical ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSLIDERVERTICAL)
        _message = generatexml (command.VERIFYSLIDERVERTICAL, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifysliderhorizontal (windowName, componentName):
    """
    INPUT: verifysliderhorizontal ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSLIDERHORIZONTAL)
        _message = generatexml (command.VERIFYSLIDERHORIZONTAL, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifystatusbar (windowName, componentName):
    """
    INPUT: verifystatusbar ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSTATUSBAR)
        _message = generatexml (command.VERIFYSTATUSBAR, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifystatusbarvisible (windowName, componentName):
    """
    INPUT: verifystatusbarvisible ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSTATUSBARVISIBLE)
        _message = generatexml (command.VERIFYSTATUSBARVISIBLE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifytoggled (windowName, componentName):
    """
    INPUT: verifytoggled ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYTOGGLED)
        _message = generatexml (command.VERIFYTOGGLED, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifyvisiblebuttoncount (windowName, componentName, count):
    """
    INPUT: verifyvisiblebuttoncount ('<window name>', '<component name>', <button count>)

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYVISIBLEBUTTONCOUNT)
        _message = generatexml (command.VERIFYVISIBLEBUTTONCOUNT, _requestId, windowName, componentName, str (count))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

#  Slider Functions

def decrease (windowName, componentName, iterations):
    """ Decrease the value of the slider 'n' times, where 'n' is the number of 
        iterations specified in the argument field.
    INPUT: decrease ('<window name>', '<slider name>', '<number of iterations>')

    OUTPUT: Decreases the value if it does not fall below the minimum limit, else fails. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.DECREASE)
        _message = generatexml (command.DECREASE, _requestId, windowName, componentName, str (iterations))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('decrease failed: %s' % msg.value)
        raise

def increase (windowName, componentName, iterations):
    """ Increase the value of the slider 'n' times, where 'n' is the number of 
        iterations specified in the argument field.
    INPUT: increase ('<window name>', '<slider name>', '<number of iterations>')

    OUTPUT: Increases the value if it does not fall below the minimum limit, else fails. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.INCREASE)
        _message = generatexml (command.INCREASE, _requestId, windowName, componentName, str (iterations))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('increase failed: %s' % msg.value)
        raise

def setmax (windowName, componentName):
    """ Set the slider to the maximum value.
    INPUT: setmax ('<window name>', '<slider name>')

    OUTPUT: Returns 1 if action is performed, else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETMAX)
        _message = generatexml (command.SETMAX, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('setmax failed: %s' % msg.value)
        raise

def setmin (windowName, componentName):
    """ Set the slider to the minimum value.
    INPUT: setmin ('<window name>', '<slider name>')

    OUTPUT: Returns 1 if action is performed, else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETMIN)
        _message = generatexml (command.SETMIN, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('setmin failed: %s' % msg.value)
        raise

#  SpinButton Functions

def setvalue (windowName, componentName, value):
    """ Sets the value of the spin button.
    INPUT: setvalue ('<window name>', '<spinbutton name>', <value>)

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETVALUE)
        _message = generatexml (command.SETVALUE, _requestId, windowName, componentName, str (value))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError:
        raise

def getvalue (windowName, componentName):
    """ Gets the value in the spin button.
    INPUT: getvalue ('<window name>', '<spinbutton name>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETVALUE)
        _message = generatexml (command.GETVALUE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return float (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getvalue failed: %s' % msg.value)
        raise

def getminvalue (windowName, componentName):
    """ Gets the min value of the spin button.
    INPUT: getminvalue ('<window name>', '<spinbutton name>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETMINVALUE)
        _message = generatexml (command.GETMINVALUE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return float (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getminvalue failed: %s' % msg.value)
        raise

def getminincrement (windowName, componentName):
    """ Gets the min increment value in the spin button.
    INPUT: getminincrement ('<window name>', '<spinbutton name>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETMININCREMENT)
        _message = generatexml (command.GETMININCREMENT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return float (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getminincrement failed: %s' % msg.value)
        raise

def getmaxvalue (windowName, componentName):
    """ Gets the max value of the spin button.
    INPUT: getmaxvalue ('<window name>', '<spinbutton name>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETMAXVALUE)
        _message = generatexml (command.GETMAXVALUE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return float (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getmaxvalue failed: %s' % msg.value)
        raise

def verifysetvalue (windowName, componentName, value) :
    """ Verifies the value set in spin button.
    INPUT: verifysetvalue ('<window name>', '<spinbutton name>', '<value>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYSETVALUE)
        _message = generatexml (command.VERIFYSETVALUE, _requestId, windowName, componentName, str (value))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

#  TabControl Functions

def selecttab (windowName, componentName, tabName):
    """ Select the given tab name in the tab list
    INPUT: selecttab ('<window name>', '<tab list name>', '<tab name>')

    OUTPUT: Returns 1 if the tab is selected, otherwise exception will be thrown """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTTAB)
        _message = generatexml (command.SELECTTAB, _requestId, windowName, componentName, tabName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selecttab failed: %s' % msg.value)
        raise

def selecttabindex (windowName, componentName, tabIndex):
    """ Select a particular tab in the list of tabs
    INPUT: selecttabindex ('<window name>', '<tab list name>', <index of the tab>)

    OUTPUT: Returns 1 if the tab is selected, otherwise exception will be thrown """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTTABINDEX)
        _message = generatexml (command.SELECTTABINDEX, _requestId, windowName, componentName, str (tabIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selecttabindex failed: %s' % msg.value)
        raise

def gettabname (windowName, componentName, tabIndex):
    """ Get name of given tab index
    INPUT: gettabname ('<windwo name>', '<tab list name>', <index of the tab>)

    OUTPUT: Returns tab name if the tab is found, otherwise exception will be thrown """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETTABNAME)
        _message = generatexml (command.GETTABNAME, _requestId, windowName, componentName, str (tabIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('gettabname failed: %s' % msg.value)
        raise

def gettabcount (windowName, componentName):
    """ Gets the number of tabs in an object
    INPUT: gettabcount ('<window name>', '<object name>')
           e.g. gettabcount ('terminal','ptl0')

    OUTPUT: Returns the number of tabs in the object """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETTABCOUNT)
        _message = generatexml (command.GETTABCOUNT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return long (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('gettabcount failed: %s' % msg.value)
        raise

def verifytabname (windowName, componentName, tabName):
    """ Select the given tab name in the tab list
    INPUT: verifytabname ('<window name>', '<tab list name>', '<tab name>')

    OUTPUT: Returns 1 if the tab name is available, otherwise 0"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYTABNAME)
        _message = generatexml (command.VERIFYTABNAME, _requestId, windowName, componentName, tabName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifytabname failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifytabname failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def checkrow (windowName, componentName, rowIndex, columnIndex = 0):
    """ checks the row with the given index value in table. This can take an 
    optional column index and perform the action on that particular column. 
    If the column index is not given, 0 is taken as the default value.
    Index value starts from 0.
    INPUT: checkrow ('<window name>', '<table name>', <row index>, <col index>)

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.CHECKROW)
        _message = generatexml (command.CHECKROW, _requestId, windowName, componentName, str (rowIndex), str (columnIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('checkrow failed: %s' % msg.value)
        raise

def verifycheckrow (windowName, componentName, rowIndex, columnIndex = 0):
    """
    INPUT: verifycheckrow ('<window name>', '<component name>', <row index>[, <column index>])

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYCHECKROW)
        _message = generatexml (command.VERIFYCHECKROW, _requestId, windowName, componentName, str (rowIndex), str (columnIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def doesrowexist (windowName, componentName, rowText, partialMatch = False):
    """ Checks whether the table contains any row with any of its cell containing 
    the given string as its value.Please note that it checks for an exact match.
    INPUT: doesrowexist ('<window name>', '<table name>', '<string to be matched>' \
    [, do partial check])

    OUTPUT: Returns 1 if there are rows with the given string in any of its cell else 0. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.DOESROWEXIST)
        _message = generatexml (command.DOESROWEXIST, _requestId, windowName,
                                componentName, rowText, str(int(partialMatch)))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0
    
def doubleclick (windowName, componentName):
    """
    INPUT: doubleclick ('<window name>', '<table name>')

    OUTPUT: Returns 1 on success and exception on error"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.DOUBLECLICK)
        _message = generatexml (command.DOUBLECLICK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('doubleclick failed: %s' % msg.value)
        raise

def doubleclickrow (windowName, componentName, rowText):
    """ Double clicks the row in table whose first column's (0th column) value 
    is same as the contents of the third argument in the function call.
    INPUT: doubleclickrow ('<window name>', '<table name>', '<value of row in first column>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.DOUBLECLICKROW)
        _message = generatexml (command.DOUBLECLICKROW, _requestId, windowName, componentName, rowText)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('doubleclickrow failed: %s' % msg.value)
        raise

def doubleclickrowindex (windowName, componentName, rowIndex):
    """ Double clicks the row index in table.
    INPUT: doubleclickrow ('<window name>', '<table name>', <index of row>)

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.DOUBLECLICKROWINDEX)
        _message = generatexml (command.DOUBLECLICKROWINDEX, _requestId, windowName, componentName, str (rowIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('doubleclickrowindex failed: %s' % msg.value)
        raise

def singleclickrow (windowName, componentName, rowText):
    """ Single clicks the row in the table whose first column's (0th column) value is the
    the same as the contents of the 'rowText' or third arguement in the function call.
    INPUT: singleclickrow ('<window name>', '<table name>', '<value of the header/top column>')
    
    OUTPUT: Returns 1 on success and 0 on error."""
    try:
        _requestId = threading.currentThread ().getName () + str (command.SINGLECLICKROW)
        _message = generatexml (command.SINGLECLICKROW, _requestId, windowName, componentName, rowText)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError(_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('singleclickrow failed: %s' % msg.value)
        raise

def getcellvalue (windowName, componentName, row, column = 0):
    """ returns the text in a cell at given row and column of a tree table
    INPUT: getcellvalue ('<window name>', '<component name>', '<row>', '<column>')

    OUTPUT: returns the string """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETCELLVALUE)
        _message = generatexml (command.GETCELLVALUE, _requestId, windowName, componentName, str (row), str (column))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('getcellvalue failed: %s' % msg.value)
        raise

def getrowcount (windowName, componentName):
    """ returns the number of rows in the table
    INPUT: getrowcount ('<window name>', '<table name>' )

    OUTPUT: Returns the number of rows present in the table mentioned """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETROWCOUNT)
        _message = generatexml (command.GETROWCOUNT, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return int (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getrowcount failed: %s' % msg.value)
        raise
    
def gettablerowindex (windowName, componentName, rowText):
    """ Returns the id of the row containing the given <cellvalue>
    INPUT: gettablerowindex ('<window name>','<tablename>','<cellvalue>')

    OUTPUT: Return id of the row containing the given cell value, if it is found else return -1 """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETTABLEROWINDEX)
        _message = generatexml (command.GETTABLEROWINDEX, _requestId, windowName, componentName, rowText)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('gettablerowindex failed: %s' % _responseStatus [1])
            return -1L
        return long (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('gettablerowindex failed: %s' % msg.value)
        raise

def selectlastrow (windowName, componentName):
    """ Selects the last row of a table.
    INPUT: selectlastrow ('<window name>', '<table name>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTLASTROW)
        _message = generatexml (command.SELECTLASTROW, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectlastrow failed: %s' % msg.value)
        raise

def selectrowpartialmatch (windowName, componentName, rowPartialText):
    """ selects the row having cell that contains the given text.
    INPUT: selectrowpartialmatch ('<window name>', '<tree table name>', '<texttobesearchedfor>')

    OUTPUT: returns 1 on success and throws an exception on error """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SELECTROWPARTIALMATCH)
        _message = generatexml (command.SELECTROWPARTIALMATCH, _requestId, windowName, componentName, rowPartialText)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('selectrowpartialmatch failed: %s' % msg.value)
        raise

def verifypartialmatch (windowName, componentName, rowPartialText):
    """ 
    INPUT: verifypartialmatch ('<window name>', '<component name>', '<partial text to match>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYPARTIALMATCH)
        _message = generatexml (command.VERIFYPARTIALMATCH, _requestId, windowName, componentName, rowPartialText)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifypartialtablecell (windowName, componentName, rowIndex, columnIndex, rowPartialText):
    """ 
    INPUT: verifypartialtablecell ('<window name>', '<component name>', <row index>, <column index>, '<partial text>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYPARTIALTABLECELL)
        _message = generatexml (command.VERIFYPARTIALTABLECELL, _requestId, windowName, componentName,
                    str (rowIndex), str (columnIndex), rowPartialText)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def uncheckrow (windowName, componentName, rowIndex, columnIndex = 0):
    """ Unchecks the row with the given index value in table. This can take an 
    optional column index and perform the action on that particular column. 
    If the column index is not given, 0 is taken as the default value. Index 
    value starts from 0.
    INPUT: uncheckrow ('<window name>', '<table name>', <row index>, [<col index>])

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.UNCHECKROW)
        _message = generatexml (command.UNCHECKROW, _requestId, windowName, componentName, str (rowIndex), str (columnIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('uncheckrow failed: %s' % msg.value)
        raise

def verifyuncheckrow (windowName, componentName, rowIndex, columnIndex = 0):
    """ 
    INPUT: verifyuncheckrow ('<window name>', '<component name>', <row index>[, <column index>])

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYUNCHECKROW)
        _message = generatexml (command.VERIFYUNCHECKROW, _requestId, windowName, componentName, str (rowIndex), str (columnIndex))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifyuncheckrow failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifyuncheckrow failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifytablecell (windowName, componentName, rowIndex, columnIndex, textToCompare):
    """ Verifies the tablecell value with the String Passed ie., fifth argument
    INPUT: verifytablecell ('<window name>', '<table name>', '<row no>','<column no>','<string to be compared>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYTABLECELL)
        _message = generatexml (command.VERIFYTABLECELL, _requestId,
                                windowName, componentName, str (rowIndex),
                                str (columnIndex), textToCompare)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifytablecell failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifytablecell failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def verifypartialcellvalue (windowName, componentName, rowIndex, columnIndex, partialTextToCompare):
    """ Verifies the tablecell value with the sub String Passed ie., fifth argument
    INPUT: verifypartialtablecell ('<window name>', '<table name>', '<row no>','<column no>','< sub string to be compared>')

    OUTPUT: Returns 1 on success and 0 on error. """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYPARTIALTABLECELL)
        _message = generatexml (command.VERIFYPARTIALTABLECELL, _requestId, windowName, componentName,
                    str (rowIndex), str (columnIndex), partialTextToCompare)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('verifypartialcellvalue failed: %s' % _responseStatus [1],
                         screenshot = False, logStatus = "warning")
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('verifypartialcellvalue failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0
    
def getwindowsize (windowName):
    """
    INPUT: getwindowsize ('<window name>')

    OUTPUT: Get size of window on success, LdtpExecutionError on failure
    Window size format: [x, y, width, height] - list of int value
    """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETWINDOWSIZE)
        _message = generatexml (command.GETWINDOWSIZE, _requestId, windowName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        tmpList = parseobjectlist (_responseData [1])
        for i in range (0, len (tmpList)):
            tmpList [i] = int (tmpList [i])
        return tmpList
    except LdtpExecutionError, msg:
        logFailures ('getwindowsize failed: %s' % msg.value,
                     screenshot = False)
        raise

def getobjectsize (windowName, componentName):
    """
    INPUT: getobjectsize ('<window name>', '<component name>')

    OUTPUT: Get size of object on success, LdtpExecutionError on failure
    Window size format: [x, y, width, height] - list of int value
    """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETOBJECTSIZE)
        _message = generatexml (command.GETOBJECTSIZE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        tmpList = parseobjectlist (_responseData [1])
        for i in range (0, len (tmpList)):
            tmpList [i] = int (tmpList [i])
        return tmpList
    except LdtpExecutionError, msg:
        logFailures ('getobjectsize failed: %s' % msg.value,
                     screenshot = False)
        raise

def getwindowlist ():
    """
    INPUT: getwindowlist ()

    OUTPUT: list of windows on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETWINDOWLIST)
        _message = generatexml (command.GETWINDOWLIST, _requestId)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return parseobjectlist (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getwindowlist failed: %s' % msg.value,
                     screenshot = False)
        raise

def getapplist ():
    """
    INPUT: getapplist ()

    OUTPUT: list of applications on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETAPPLIST)
        _message = generatexml (command.GETAPPLIST, _requestId)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return parseobjectlist (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getapplist failed: %s' % msg.value,
                     screenshot = False)
        raise

def getallstates (windowName, componentName = ''):
    """
    INPUT: getallstates ('<window name>'[, '<component name>]')

    OUTPUT: list of aall states of given object on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETALLSTATES)
        _message = generatexml (command.GETALLSTATES, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        _tmpList = parseobjectlist (_responseData [1])
        index = 0
        for i in _tmpList:
            _tmpList [index] = int (i)
            index += 1
        return _tmpList
    except LdtpExecutionError, msg:
        logFailures ('getallstates failed: %s' % msg.value,
                     screenshot = False)
        raise

def getobjectlist (windowName):
    """
    INPUT: getobjectlist ('<window name>')

    OUTPUT: List of objects of the window on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETOBJECTLIST)
        _message = generatexml (command.GETOBJECTLIST, _requestId, windowName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return parseobjectlist (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getobjectlist failed: %s' % msg.value)
        raise

def getchild (windowName, componentName = '', role = '', parent = ''):
    """
    INPUT: getchild ('<window name>'[, '<component name>'[, '<role type>']])

    OUTPUT: List of children on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETCHILD)
        _message = ''
        if (role != None and role != '') or (parent != None and parent != ''):
            if role:
                role = re.sub (' ', '_', role)
            if role == '' and parent != '':
                role = 'BY PASS GENERATEXML BUG'
            _message = generatexml (command.GETCHILD, _requestId, windowName, componentName, role, parent)
        else:
            _message = generatexml (command.GETOBJECTINFO, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return parseobjectlist (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getchild failed: %s' % msg.value)
        raise

def getobjectinfo (windowName, componentName):
    """
    INPUT: getobjectinfo ('<window name>', '<component name>')

    OUTPUT: List of object informations on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETOBJECTINFO)
        _message = generatexml (command.GETOBJECTINFO, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError ('%s Window: %s Object: %s' % \
                                          (_responseStatus [1], windowName,
                                           componentName))
        return parseobjectlist (_responseData [1])
    except LdtpExecutionError, msg:
        logFailures ('getobjectinfo failed: %s' % msg.value)
        raise

def getobjectproperty (windowName, componentName, propertyName):
    """
    INPUT: getobjectproperty ('<window name>', '<component name>', '<property name>')

    OUTPUT: property value on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETOBJECTPROPERTY)
        _message = generatexml (command.GETOBJECTPROPERTY, _requestId, windowName, componentName, propertyName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError ('%s Window: %s Object: %s Property: %s' % \
                                          (_responseStatus [1], windowName,
                                           componentName, propertyName))
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('getobjectproperty failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        raise

def setlocale (locale = None):
    try:
        _requestId  = threading.currentThread ().getName () + str (command.SETLOCALE)
        _message = None
        if locale == None:
            _message = generatexml (command.SETLOCALE, _requestId, os.getenv ('LANG'))
        else:
            os.environ ['LANG'] = locale
            _message = generatexml (command.SETLOCALE, _requestId, locale)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
    except LdtpExecutionError, msg:
        logFailures ('setlocale failed: %s' % msg.value,
                     screenshot = False)
        raise

def bindtext (packageName, localeDir, mode = 'mo'):
    """
    INPUT: TODO

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.BINDTEXT)
        _message = generatexml (command.BINDTEXT, _requestId, packageName, localeDir, mode)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('bindtext failed: %s' % msg.value,
                     screenshot = False)
        raise

def generatemouseevent (x, y, eventType = 'b1c'):
    """
    INPUT:
    X - Coordinate
    Y - Coordinate
    eventType

    OUTPUT: None"""
    try:
        _requestId  = threading.currentThread ().getName () + \
            str (command.GENERATEMOUSEEVENT)
        _message = generatexml (command.GENERATEMOUSEEVENT, _requestId,
                                str (x), str (y), eventType)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('generatemouseevent failed: %s' % msg.value)
        raise

def mouseleftclick (windowName, componentName):
    """
    INPUT: windowName --> Window containing object
               componentName --> object's name

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.MOUSELEFTCLICK)
        _message = generatexml (command.MOUSELEFTCLICK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('mouseleftclick failed: %s' % msg.value)
        raise

def mouserightclick (windowName, componentName):
    """
    INPUT: windowName --> Window containing object
               componentName --> object's name

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.MOUSERIGHTCLICK)
        _message = generatexml (command.MOUSERIGHTCLICK, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('mouserightclick failed: %s' % msg.value)
        raise

def mousemove (windowName, componentName):
    """
    INPUT: windowName --> Window containing object
               componentName --> object's name

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.MOUSEMOVE)
        _message = generatexml (command.MOUSEMOVE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('mousemove failed: %s' % msg.value)
        raise

def enterstring (windowName, componentName = None, data = None):
    """
    INPUT: windowName --> Window containing object
               componentName --> object's name
           data --> string to be simulated as entered from keyboard
           Special non printing characters are enter within '<' and '>'
           e.g. <capslock>,<ctrl>, etc while other characters are entered
           in a straight forward fashion. e.g. '<ctrl>Nwww.google.com'
           If only first argument is passed then it will be considered as 'data' argument

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.KBDENTER)
        _message = ''
        if componentName is not None and data is not None:
            _message = generatexml (command.KBDENTER, _requestId, windowName, componentName, str (data))
        elif componentName is None and data is None:
            # Window name will hold the data
            _message = generatexml (command.GENERATEKEYEVENT, _requestId, windowName)
        else:
            raise LdtpExecutionError ('Invalid syntax')
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('enterstring failed: %s' % msg.value)
        raise

def generatekeyevent (data):
    """
    INPUT: data --> string to be simulated as entered from keyboard
    Special non printing characters are enter within '<' and '>'
    e.g. <capslock>,<ctrl>, etc while other characters are entered
    in a straight forward fashion. e.g. '<ctrl>Nwww.google.com'    

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GENERATEKEYEVENT)
        _message = generatexml (command.GENERATEKEYEVENT, _requestId, str (data))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('generatekeyevent failed: %s' % msg.value)
        raise

def keypress (data):
    """
    INPUT: data --> string to be simulated as entered from keyboard
    Special non printing characters are enter within '<' and '>'
    e.g. <capslock>,<ctrl>, etc while other characters are entered
    in a straight forward fashion. e.g. '<ctrl>Nwww.google.com'    

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.KEYPRESS)
        _message = generatexml (command.KEYPRESS, _requestId, str (data))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('keypress failed: %s' % msg.value)
        raise

def keyrelease (data):
    """
    INPUT: data --> string to be simulated as entered from keyboard
    Special non printing characters are enter within '<' and '>'
    e.g. <capslock>,<ctrl>, etc while other characters are entered
    in a straight forward fashion. e.g. '<ctrl>Nwww.google.com'    

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.KEYRELEASE)
        _message = generatexml (command.KEYRELEASE, _requestId, str (data))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('keyrelease failed: %s' % msg.value)
        raise

def stopscriptengine ():
    """
    INPUT: stopscriptengine ()

    OUTPUT: None"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.STOPSCRIPTENGINE)
        _message = generatexml (command.STOPSCRIPTENGINE, _requestId)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
    except LdtpExecutionError, msg:
        logFailures ('stopscriptengine failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        pass

def hasstate (windowName, componentName, *state):
    """
    INPUT: hasstate ('<window name>', '<component name>', <verify the object has this state>)

    OUTPUT: 1 on success, 0 on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.HASSTATE)

        _message = generatexml (command.HASSTATE, _requestId, windowName,
                                componentName, *state)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError, msg:
        logFailures ('hasstate failed: %s' % msg.value,
                     screenshot = False, logStatus = "warning")
        return 0

def launchapp (appName, arg = [''], delay = 5, env = 1):
    """
    INPUT: launchapp ('<application name>'[, <'argument'>][, 5][, 1 or 0])
    Currently argument is used, only when you are trying to run from local machine
    If 1, then set GTK_MODULES and GNOME_ACCESSIBILITY environment variables
    If 0, don't set the environment variables
    NOTE: Incase of remote session, the command line arguments should be part of
    application name

    OUTPUT: LdtpExecutionError on failure"""
    if _ldtpUseTcp:
        _requestId  = threading.currentThread ().getName () + str (command.LAUNCHAPP)
        # NOTE: Incase of remote session, the command line arguments should be part of
        # application name
        _message = generatexml (command.LAUNCHAPP, _requestId, appName, str (delay), str (env))
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            logFailures ('File ' + str (appName) + ' does not exist in the PATH.',
                         screenshot = False)
            raise LdtpExecutionError ('File ' + str (appName) + ' does not exist in the PATH.')
        if _responseData [1]:
            return str (_responseData [1]).split ('\n')
    else:
        _status = commands.getstatusoutput ('which ' + str (appName))
        if _status [0] != 0:
            logFailures ('File ' + str (appName) + ' does not exist in the PATH.',
                         screenshot = False)
            raise LdtpExecutionError ('File ' + str (appName) + ' does not exist in the PATH.')
        pid = os.fork ()
        if pid == 0:
            envVar = os.environ
            if env == 1:
                envVar ['GTK_MODULES'] = 'gail:atk-bridge'
                envVar ['GNOME_ACCESSIBILITY'] = '1'
            fd = _sockFdPool.get (threading.currentThread ())
            fd.close ()
            # If we don't insert the appName, we get '.' as application name
            arg.insert (0, appName)
            if arg [1] == '':
                arg.remove ('')
            os.execvpe (appName, arg, envVar)
        # Let us wait so that the application launches
        time.sleep (delay)

def titlebarselectmenuitem (windowName, componentName, *menupath):
    """Select a menu item from a titlebar object
    INPUT: titlebarselectmenuitem ('<window name>', '<component name>', '<*menupath>')

    OUTPUT: TODO"""
    try:
        _requestId  = str (random.randint (0, sys.maxint))
        _message = generatexml (command.TITLEBARSELECTMENUITEM, _requestId, windowName, componentName, *menupath)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus[0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('titlebarselect failed: %s' % msg.value)
        raise

def getfile (pathtoget, pathtosave):
    """
    INPUT: getfile ('<pathtoget>', '<pathtosave>')

    OUTPUT: a binary file
    pathtoget is the path of where the binary file placed in (server side)
    pathtosave is the file which you want to saved in (client side)"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GETFILE)
        _message = generatexml (command.GETFILE, _requestId, pathtoget)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        data = base64.b64decode (_responseData[1])

        fp = file (pathtosave, "wb")
        if fp is None:
            raise LdtpExecutionError, "Unable to open file for writing %s" % pathtosave
        fp.write (data)
        fp.close ()
    except LdtpExecutionError, msg:
        logFailures ('getfile failed: %s' % msg.value, screenshot = False)
        raise

def generatemd5 (filepath):
    try:
        _requestId  = threading.currentThread ().getName () + str (command.GENERATEMD5)
        _message = generatexml (command.GENERATEMD5, _requestId, filepath)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return _responseData [1]
    except LdtpExecutionError, msg:
        logFailures ('generatemd5 failed: %s' % msg.value, screenshot = False)
        raise

def appundertest (appName):
    """
    INPUT: appundertest ('<application name>')

    OUTPUT: 1 on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread ().getName () + str (command.APPUNDERTEST)
        _message = generatexml (command.APPUNDERTEST, _requestId, appName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus[0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        return 1
    except LdtpExecutionError, msg:
        logFailures ('appundertest failed: %s' % msg.value)
        raise

def launchapp2 (appname, aut = None, callAut = False):
    """Calls appundertest for the user.

    Implicit call to appundertest.
    Because:
    * This is usually what a user mean when he calls launchapp.
    * A newbie user could call appunder very far away from launchapp, and we don't know
      what could be the consequences of that.
      * Less typing.
    """
    if callAut == False:
        return launchapp (appname)
    if aut:
        appundertest (aut)
    else:
        appundertest (appname)
    launchapp (appname)
    
def verifyprogressbar (windowName, componentName):
    """
    INPUT: verifyprogressbar ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYPROGRESSBAR)
        _message = generatexml (command.VERIFYPROGRESSBAR, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def verifyprogressbarvisible (windowName, componentName):
    """
    INPUT: verifyprogressbarvisible ('<window name>', '<component name>')

    OUTPUT: 1 on success, 0 on failure """
    try:
        _requestId  = threading.currentThread ().getName () + str (command.VERIFYPROGRESSBARVISIBLE)
        _message = generatexml (command.VERIFYPROGRESSBARVISIBLE, _requestId, windowName, componentName)
        sendpacket (_message)
        _responseStatus, _responseData = getresponse (_requestId)
        if _responseStatus [0] != 0:
            return 0
        return 1
    except LdtpExecutionError:
        return 0

def windowuptime(windowName):
    """
    INPUT: windowuptime('<window name>')

    OUTPUT: start time, end time on success, LdtpExecutionError on failure"""
    try:
        _requestId  = threading.currentThread().getName() + str(command.WINDOWUPTIME)
        _message = generatexml(command.WINDOWUPTIME, _requestId, windowName)
        sendpacket(_message)
        _responseStatus, _responseData = getresponse(_requestId)
        if _responseStatus [0] != 0:
            raise LdtpExecutionError (_responseStatus [1])
        if not _responseData[1]:
            raise LdtpExecutionError("Window '%s' never poped up" % windowName)
        tmpTime = _responseData[1].split('-')
        endTime = tmpTime[0].split(' ')
        startTime = tmpTime[1].split(' ')
        _startTime = datetime.datetime(int(startTime[0]), int(startTime[1]),
                                       int(startTime[2]), int(startTime[3]),
                                       int(startTime[4]), int(startTime[5]))
        _endTime = datetime.datetime(int(endTime[0]), int(endTime[1]),
                                     int(endTime[2]),int(endTime[3]),
                                     int(endTime[4]), int(endTime[5]))
        return _startTime, _endTime
    except LdtpExecutionError, msg:
        logFailures ('windowuptime failed: %s' % msg.value)
        raise
