/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * Linux Desktop Testing Project http://ldtp.freedesktop.org
 *
 * Author:
 *    Nagappan A <nagappan@gmail.com>
 *
 * Copyright 2004 - 2006 Novell, Inc.
 * Copyright 2009 Nagappan Alagappan
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110, USA.
 */

#include "ldtp.h"
#include "ldtp-gui.h"
#include "ldtp-error.h"
#include "ldtp-logger.h"
#include "ldtp-command.h"
#include "ldtp-gui-comp.h"

static LDTPErrorCode
select_item (Accessible *object, char *item_name, FILE *log_fp)
{
	long i;
	gboolean flag = FALSE;
	LDTPErrorCode error;
	long child_count;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_SELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	child_count = Accessible_getChildCount (object);

	if (child_count > 0) {
		char *name = NULL;
		Accessible *child;

		for (i = 0; i < child_count; i++) {
			child = Accessible_getChildAtIndex (object, i);
			if (child) {
				name = Accessible_getName (child);
#ifdef ENABLE_LOCALIZE
				if (g_utf8_collate (name, _(item_name)) == 0) {
#else
					if (g_utf8_collate (name, item_name) == 0) {
#endif
						flag = TRUE;
						g_print ("Child name: %s\n", name);
						SPI_freeString (name);
						Accessible_unref (child);
						break;
					}
				SPI_freeString (name);
				Accessible_unref (child);
			}
		}
	}
	if (flag == TRUE) {
		SPIBoolean flag = FALSE;
		AccessibleSelection *selection;
		selection = Accessible_getSelection (object);
		if (selection) {
			flag = AccessibleSelection_selectChild (selection, i);
			Accessible_unref (selection);
		}
		g_print ("Selected: %s\n", item_name);
		if (flag == TRUE)
			return LDTP_ERROR_SUCCESS;
	}
	error = LDTP_ERROR_UNABLE_TO_SELECT_LAYERED_PANE_ITEM;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

#ifdef ENABLE_NEWROLES
static LDTPErrorCode
unselect_item (Accessible *object, char *item_name, FILE *log_fp)
{
	long i;
	gboolean flag = FALSE;
	LDTPErrorCode error;
	long child_count;

	if (!object || !item_name) {
		error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	child_count = Accessible_getChildCount (object);

	if (child_count > 0) {
		char *name = NULL;
		Accessible *child;

		for (i = 0; i < child_count; i++) {
			child = Accessible_getChildAtIndex (object, i);
			if (child) {
				name = Accessible_getName (child);
#ifdef ENABLE_LOCALIZE
				if (g_utf8_collate (name, _(item_name)) == 0) {
#else
					if (g_utf8_collate (name, item_name) == 0) {
#endif
						flag = TRUE;
						g_print ("Child name: %s\n", name);
						SPI_freeString (name);
						Accessible_unref (child);
						break;
					}
				SPI_freeString (name);
				Accessible_unref (child);
			}
		}
	}
	if (flag == TRUE) {
		SPIBoolean flag = FALSE;
		AccessibleSelection *selection;
		selection = Accessible_getSelection (object);
		if (selection) {
			flag = AccessibleSelection_deselectChild (selection, i);
			Accessible_unref (selection);
		}
		g_print ("Selected: %s\n", item_name);
		if (flag == TRUE)
			return LDTP_ERROR_SUCCESS;
	}
	error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}
#endif

static LDTPErrorCode
select_all (Accessible *object, FILE *log_fp)
{
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleSelection *selection;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_SELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		flag = AccessibleSelection_selectAll (selection);
		Accessible_unref (selection);
	}

	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;

	error = LDTP_ERROR_UNABLE_TO_SELECT_LAYERED_PANE_ITEM;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
unselect_all (Accessible *object, FILE *log_fp)
{
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleSelection *selection;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		flag = AccessibleSelection_clearSelection (selection);
		Accessible_unref (selection);
	}

	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;

	error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
select_index (Accessible *object, long index, FILE *log_fp)
{
	long child_count;
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleSelection *selection;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_SELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	child_count = Accessible_getChildCount (object);

	if (child_count == -1 || index < 0 || index > child_count) {
		error = LDTP_ERROR_LIST_INDEX_INCORRECT;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		flag = AccessibleSelection_selectChild (selection, index);
		Accessible_unref (selection);
	}

	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;

	error = LDTP_ERROR_UNABLE_TO_SELECT_LAYERED_PANE_ITEM;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

#ifdef ENABLE_NEWROLES
static LDTPErrorCode
unselect_index (Accessible *object, long index, FILE *log_fp)
{
	long child_count;
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleSelection *selection;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	child_count = Accessible_getChildCount (object);

	if (child_count == -1 || index < 0 || index > child_count) {
		error = LDTP_ERROR_LIST_INDEX_INCORRECT;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		flag = AccessibleSelection_deselectChild (selection, index);
		Accessible_unref (selection);
	}

	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;

	error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}
#endif

static LDTPErrorCode
unselect_item_index (Accessible *object, long index, FILE *log_fp)
{
	long child_count;
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleSelection *selection;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		child_count = AccessibleSelection_getNSelectedChildren (selection);
		if (index < 0 || child_count < 0 || index > child_count) {
			error = LDTP_ERROR_LIST_INDEX_INCORRECT;
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error),
				 log_fp);
			return error;
		}
		flag = AccessibleSelection_deselectSelectedChild (selection,
								  index);
		Accessible_unref (selection);
	}

	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;

	error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
is_child_selected (Accessible *object, gchar *item, FILE *log_fp)
{
	long i, child_count;
	Accessible *child;
	LDTPErrorCode error;
	gboolean flag = FALSE;
	AccessibleStateSet *state_set;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	child_count = Accessible_getChildCount (object);

	if (child_count < 0) {
		error = LDTP_ERROR_LIST_INDEX_INCORRECT;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error),
			 log_fp);
		return error;
	}

	for (i = 0; i < child_count; i++) {
		child = Accessible_getChildAtIndex (object, i);
		if (child) {
			char *name = Accessible_getName (child);
			if (name) {
				if (g_utf8_collate (item, name) == 0) {
					state_set = Accessible_getStateSet (object);
					if (state_set) {
						flag = AccessibleStateSet_contains (state_set,
										    SPI_STATE_SELECTED);
						AccessibleStateSet_unref (state_set);
					}
					Accessible_unref (child);
					SPI_freeString (name);
					break;
				}
				SPI_freeString (name);
			}
			Accessible_unref (child);
		}
	}
	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;
	error = LDTP_ERROR_CHILD_NOT_SELECTED;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
is_child_item_selected (Accessible *object, gchar *item, FILE *log_fp)
{
	long i, child_count;
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleSelection *selection;

	if (!object || !item) {
		error = LDTP_ERROR_CHILD_NOT_SELECTED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		char *name;
		Accessible *child;
		child_count = AccessibleSelection_getNSelectedChildren (selection);
		for (i = 0; i < child_count; i++) {
			child = AccessibleSelection_getSelectedChild (selection,
								      i);
			if (child) {
				name = Accessible_getName (child);
				if (name) {
					if (g_utf8_collate (item, name) == 0) {
						flag = AccessibleSelection_isChildSelected (selection,
											    i);
						SPI_freeString (name);
						Accessible_unref (child);
						break;
					}
					SPI_freeString (name);
				}
				Accessible_unref (child);
			}
		}
		Accessible_unref (selection);
	}

	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;

	error = LDTP_ERROR_CHILD_NOT_SELECTED;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
is_child_index_selected (Accessible *object, long index, FILE *log_fp)
{
	long child_count;
	Accessible *child;
	LDTPErrorCode error;
	gboolean flag = FALSE;
	AccessibleStateSet *state_set;

	if (!object) {
		error = LDTP_ERROR_UNABLE_TO_DESELECT_LAYERED_PANE_ITEM;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	child_count = Accessible_getChildCount (object);

	if (child_count < 0 || index < 0 || index > child_count) {
		error = LDTP_ERROR_LIST_INDEX_INCORRECT;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error),
			 log_fp);
		return error;
	}

	child = Accessible_getChildAtIndex (object, index);
	if (child) {
		state_set = Accessible_getStateSet (object);
		if (state_set) {
			flag = AccessibleStateSet_contains (state_set,
							    SPI_STATE_SELECTED);
			AccessibleStateSet_unref (state_set);
		}
		Accessible_unref (child);
	}
	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;
	error = LDTP_ERROR_CHILD_NOT_SELECTED;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
is_child_item_index_selected (Accessible *object, long index, FILE *log_fp)
{
	long child_count;
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleSelection *selection;

	if (!object) {
		error = LDTP_ERROR_CHILD_NOT_SELECTED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		child_count = AccessibleSelection_getNSelectedChildren (selection);
		if (index < 0 || child_count < 0 || index > child_count) {
			error = LDTP_ERROR_LIST_INDEX_INCORRECT;
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error),
				 log_fp);
			return error;
		}
		flag = AccessibleSelection_isChildSelected (selection,
							    index);
		Accessible_unref (selection);
	}

	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;

	error = LDTP_ERROR_CHILD_NOT_SELECTED;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static long
get_selected_item_count (Accessible *object, FILE *log_fp)
{
	long selected_children = -1;
	AccessibleSelection *selection;

	if (!object) {
		g_print ("Argument NULL\n");
		return -1;
	}

	selection = Accessible_getSelection (object);
	if (selection) {
		selected_children = AccessibleSelection_getNSelectedChildren (selection);
		Accessible_unref (selection);
	}

	return selected_children;
}

static LDTPErrorCode
right_click (Accessible *object, FILE *log_fp)
{
	long action_count, i;
	SPIBoolean flag = FALSE;
	LDTPErrorCode error;
	AccessibleAction *action;
	
	action = Accessible_getAction (object);
	action_count = AccessibleAction_getNActions (action);

	for (i = 0; i < action_count; i++) {
		char *name, *desc;
		name = AccessibleAction_getName (action, i);
		desc = AccessibleAction_getDescription (action, i);
		g_print ("name = %s, desc = %s\n", name, desc);
		SPI_freeString(name);
		SPI_freeString(desc);
	}
	/*
	  FIXME: Take action based on dynamic values
	  To execute menu action
	*/
	flag = AccessibleAction_doAction (action, 1);
	Accessible_unref (action);
	if (flag)
		return LDTP_ERROR_SUCCESS;
	else {
		error = LDTP_ERROR_RIGHT_CLICK_FAILED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

LDTPErrorCode
layered_pane_main (LDTPClientContext* cctxt, int command)
{
	LDTPErrorCode error;
	switch (command) {
	case LDTP_CMD_SELECTITEM: {
		char *item_name = NULL;
		item_name = cctxt->req->arg_list->data;
		error = select_item (cctxt->gui_handle->handle,
				     item_name, cctxt->log_fp);
		break;
	}
#ifdef ENABLE_NEWROLES
	case LDTP_CMD_UNSELECTITEM: {
		char *item_name = NULL;
		item_name = cctxt->req->arg_list->data;
		error = unselect_item (cctxt->gui_handle->handle,
				       item_name, cctxt->log_fp);
		break;
	}
#endif
	case LDTP_CMD_SELECTINDEX: {
		long index = -1;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			index = atol (cctxt->req->arg_list->data);
		error = select_index (cctxt->gui_handle->handle,
				      index, cctxt->log_fp);
		break;
	}
#ifdef ENABLE_NEWROLES
	case LDTP_CMD_UNSELECTINDEX: {
		long index = -1;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			index = atol (cctxt->req->arg_list->data);
		error = unselect_index (cctxt->gui_handle->handle,
					index, cctxt->log_fp);
		break;
	}
#endif
	case LDTP_CMD_UNSELECTITEMINDEX: {
		long index = -1;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			index = atol (cctxt->req->arg_list->data);
		error = unselect_item_index (cctxt->gui_handle->handle,
					     index, cctxt->log_fp);
		break;
	}
	case LDTP_CMD_SELECTALL:
		error = select_all (cctxt->gui_handle->handle,
				    cctxt->log_fp);
		break;
	case LDTP_CMD_UNSELECTALL:
		error = unselect_all (cctxt->gui_handle->handle,
				      cctxt->log_fp);
		break;
	case LDTP_CMD_ISCHILDSELECTED: {
		gchar *item = NULL;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			item = cctxt->req->arg_list->data;
		// Check the item is selected from the complete item list
		if (is_child_selected (cctxt->gui_handle->handle, item,
				       cctxt->log_fp))
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_CHILD_NOT_SELECTED;
		break;
	}
	case LDTP_CMD_ISCHILDITEMSELECTED: {
		gchar *item = NULL;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			item = cctxt->req->arg_list->data;
		// Check the item is selected from the existing selected item list
		if (is_child_item_selected (cctxt->gui_handle->handle, item,
					    cctxt->log_fp))
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_CHILD_NOT_SELECTED;
		break;
	}
	case LDTP_CMD_ISCHILDINDEXSELECTED: {
		long index = -1;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			index = atol (cctxt->req->arg_list->data);
		// Check the item is selected from the complete item list
		if (is_child_index_selected (cctxt->gui_handle->handle, index,
					     cctxt->log_fp))
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_CHILD_NOT_SELECTED;
		break;
	}
	case LDTP_CMD_ISCHILDITEMINDEXSELECTED: {
		long index = -1;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			index = atol (cctxt->req->arg_list->data);
		// Check the item is selected from the existing selected item list
		if (is_child_item_index_selected (cctxt->gui_handle->handle, index,
						  cctxt->log_fp))
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_CHILD_NOT_SELECTED;
		break;
	}
	case LDTP_CMD_SELECTEDITEMCOUNT: {
		long count;
		count = get_selected_item_count (cctxt->gui_handle->handle,
						 cctxt->log_fp);
		if (count == -1)
			error = LDTP_ERROR_UNABLE_TO_GET_SELECTED_ITEM_COUNT;
		else {
			cctxt->resp->data = g_strdup_printf ("%ld", count);
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
			error = LDTP_ERROR_SUCCESS;
		}
		break;
	}
	case LDTP_CMD_RIGHTCLICK:
		error = right_click (cctxt->gui_handle->handle, cctxt->log_fp);
		break;
	default:
		error = LDTP_ERROR_COMMAND_NOT_IMPLEMENTED;
	}
	return error;
}
