/*
 *  Copyright 1994-2012 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "context.h"
#include <jack/jack.h>

u_long id = 1330135598;
u_long options = BEQ_THREAD;

#define INSIZE 256

/* JACK data */
static jack_port_t **input_ports;
static jack_client_t *client;
static char *source_names[2] = { NULL, NULL };


static void
jack_shutdown(__attribute__ ((unused)) void *arg)
{
  /* TODO: do not xerror */
  xerror("JACK shut down\n");
}


static int
process(jack_nframes_t nframes, void *arg)
{
  unsigned int chn;
  size_t i;
  Context_t *ctx = (Context_t *)arg;
  jack_default_audio_sample_t *in;

  nframes = (nframes > INSIZE) ? INSIZE : nframes;

  for (chn = 0; chn < 2; chn++) {
    in = jack_port_get_buffer(input_ports[chn], nframes);
    for (i = 0; i < nframes; i++)
      ctx->input->data[A_LEFT+chn][i] = in[i];
  }
  Input_set(ctx->input, A_STEREO);

  return 0;
}


static void
setup_ports(char *source_names[])
{
  unsigned int i;
  const char **ports;

  input_ports = xcalloc(2, sizeof(jack_port_t *));

  for (i = 0; i < 2; i++) {
    char name[64];

    sprintf(name, "input_%d", i);

    if ((input_ports[i] = jack_port_register(client, name, JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0)) == 0) {
      fprintf(stderr, "[!] JACK: cannot register input port \"%s\" !\n", name);
      jack_client_close(client);
      exit(1);
    }
    else printf("[i] JACK: registered %s\n", name);
  }

  ports = jack_get_ports(client, NULL, NULL, JackPortIsPhysical|JackPortIsOutput);
  if (NULL == ports)
    xerror("JACK: no physical capture ports\n");

  for (i = 0; i < 2; i++) {
    if (jack_connect(client, ports[i], jack_port_name(input_ports[i]))) {
      fprintf(stderr, "[!] JACK: cannot connect input port %s to %s\n", jack_port_name(input_ports[i]), source_names[i]);
      jack_client_close(client);
      exit(1);
    }
    else printf("[i] JACK: connected %s to %s\n", source_names[i], jack_port_name(input_ports[i]));
  }
  xfree(ports);
}


void
create(Context_t *ctx)
{
  ctx->input = Input_new(INSIZE);

  if ((client = jack_client_open(PACKAGE, JackNullOption, NULL)) == 0)
    xerror("JACK server not running ?\n");

  jack_set_process_callback(client, process, ctx);
  jack_on_shutdown(client, jack_shutdown, NULL);

  if (jack_activate(client))
    xerror("JACK: cannot activate client\n");

  if (NULL == (source_names[0] = getenv("BINIOU_JACK_LEFT")))
    source_names[0] = "system:capture_1";
  if (NULL == (source_names[1] = getenv("BINIOU_JACK_RIGHT")))
    source_names[1] = "system:capture_2";

  printf("[i] JACK: left  capture from %s\n", source_names[0]);
  printf("[i] JACK: right capture from %s\n", source_names[1]);

  setup_ports(source_names);
}


void
destroy(Context_t *ctx)
{
  jack_client_close(client);
  Input_delete(ctx->input);
  printf("[+] JACK plugin exiting\n");
}
