/***********************************************************************************

	Copyright (C) 2007-2008 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <iostream>
#include <string>
#include <cstdlib>
#include <sstream>
#include <ctime>
#include <unistd.h>

#include "lifeograph.hpp"


// GLOBAL FUNCTIONS 
void g_open_url (Gtk::AboutDialog&, const Glib::ustring& link) {
	std::string l_command ("x-www-browser ");
	l_command += link;
    //TODO: what else can be made with the return value?
	std::cout << system (l_command.c_str()) << std::endl;
}

void g_mail_to (Gtk::AboutDialog&, const Glib::ustring& link) {
	//~ std::string l_command ("x-www-browser ");
	//~ l_command += link;
	//~ system (l_command.c_str());
	// TO BE IMPLEMENTED...
}

// CONSTRUCTOR
LIFEOGRAPH::LIFEOGRAPH (int l_argc, char *l_argv[])
:	FilechoosedialogSave(NULL), DialogPassword(NULL), ProgramPath(l_argv[0]),
    InternalOperation(INTOP_COMPLETE), LoggingStatus(LOGGED_OUT), EntryCount(0),
    EntryChanged(false) {

    MenuEntry = NULL;

    GconfClient = Gnome::Conf::Client::get_default_client();

	WindowMain = new Gtk::Window;
	WindowMain->signal_delete_event().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_event_delete));

	
	Gtk::AboutDialog::set_url_hook(&g_open_url);
	Gtk::AboutDialog::set_email_hook(&g_mail_to);

	// COMMANDLINE OPTIONS
    bool  l_force_salutation(false);
	for (int i = 1; i < l_argc; i++) {
		if (!strcmp(l_argv[i], "--open") || !strcmp(l_argv[i], "-o")) {
			if (i<l_argc) {
                if (Glib::file_test(l_argv[ ++i ], Glib::FILE_TEST_EXISTS))
				    Database.set_path(l_argv[i]);
            }
			else
				LIFEOHELPERS::ERROR("No path provided");
		}
        else if (!strcmp(l_argv[i], "--purge-conf")) {
	        if (GconfClient->dir_exists(LIFEO::CONFIG_PATH)) {
                std::vector<Gnome::Conf::Entry> l_vectorEntries=
                        GconfClient->all_entries(LIFEO::CONFIG_PATH);
                for (   std::vector<Gnome::Conf::Entry>::iterator l_begin=
                                l_vectorEntries.begin();
                        l_begin != l_vectorEntries.end();
                        l_begin++)
	                GconfClient->unset((*l_begin).get_key());
            }
        }
        else if (!strcmp(l_argv[i], "--set-bg")) {
            if (i<l_argc) {
                if (Glib::file_test(l_argv[ ++i ], Glib::FILE_TEST_EXISTS)) {
                    PixbufBackground = Gdk::Pixbuf::create_from_file(l_argv[i]);
                    GconfClient->set(LIFEO::CONFIG_PATH+"/bg_picture_welcome",
                            Glib::ustring(l_argv[i]));
                }
            }
			else
				LIFEOHELPERS::ERROR("No path provided");

        }
        else if (!strcmp(l_argv[i], "--force-welcome")) {
            l_force_salutation=true;
        }
        // TODO: when read only mode implemented
// 		else if (!strcmp(l_argv[i], "--read-only") || !strcmp(l_argv[i], "-R")) { 
// 		}
		else if (Glib::file_test(l_argv[i], Glib::FILE_TEST_EXISTS)) {
            Database.set_path(l_argv[i]);
		}
	}

	ListstoreEntries = Gtk::ListStore::create(ColrecEntry);
	ListstoreEntries->set_sort_func(ColrecEntry.Date,
            sigc::mem_fun(*this, &LIFEOGRAPH::sort_date_column));
    ListstoreEntries->set_sort_func(ColrecEntry.Favored,
            sigc::mem_fun(*this, &LIFEOGRAPH::sort_favored_column));

	if (!l_force_salutation && read_conf())
        draw_loginscreen();
    else // first time user
        draw_welcomescreen();

    InternalOperation=INTOP_USER;
}

void LIFEOGRAPH::run (void) {
	Gtk::Main::run (*WindowMain);
}

bool LIFEOGRAPH::read_conf (void) {
    if (GconfClient->dir_exists(LIFEO::CONFIG_PATH) == false) {
        std::cout << "first time user!!" << std::endl;
        return false;
    }
    if (Database.is_path_set() == false)  // if not set in commandline
        Database.set_path(GconfClient->get_string(LIFEO::CONFIG_PATH+"/db_path"));
	WindowMain->set_default_size(GconfClient->get_int(LIFEO::CONFIG_PATH+"/width"),
            GconfClient->get_int(LIFEO::CONFIG_PATH+"/height"));
	WindowMain->move(GconfClient->get_int(LIFEO::CONFIG_PATH+"/position_x"),
            GconfClient->get_int(LIFEO::CONFIG_PATH+"/position_y"));

    if (bool (PixbufBackground) == false) {  // if not set in commandline
        if (Glib::file_test(GconfClient->get_string(
                    LIFEO::CONFIG_PATH+"/bg_picture_welcome"),
                    Glib::FILE_TEST_EXISTS)) {
            PixbufBackground = Gdk::Pixbuf::create_from_file(
                    GconfClient->get_string(LIFEO::CONFIG_PATH+"/bg_picture_welcome"));
        }
        else {
            LIFEOHELPERS::ERROR l_error1("bg file could not be found!");
        }
    }

	//~ gconfClient->notify_add(

    return true;
}

int l_countdown = 60; // for testing in final version there won't be a count down

bool LIFEOGRAPH::handle_event (GdkEvent*) {
    l_countdown = 60;
    SigconnectionTimeout.disconnect();
    SigconnectionTimeout = Glib::signal_timeout().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_idle), 1000); 

    return false;
}

bool LIFEOGRAPH::handle_event_expose (GdkEventExpose* l_event) {
    WindowMain->get_window()->draw_pixbuf(
            WindowMain->get_style()->get_bg_gc(Gtk::STATE_NORMAL),
            PixbufBackground,
            0, 0, 0, 0, -1, -1,
            Gdk::RGB_DITHER_NONE, 0, 0 );

    if(WindowMain->get_child())
        WindowMain->propagate_expose(*WindowMain->get_child(), l_event);

    return true;
}

bool LIFEOGRAPH::handle_event_delete (GdkEventAny*) {
	if (LoggingStatus != LOGGED_IN)
		return false;
    
    return (!logout());
}

bool LIFEOGRAPH::handle_idle () {
    if (l_countdown > 0) {
        l_countdown--;
        return true;
    }
    else {
        if (DialogPassword)
            DialogPassword->hide();
        if (FilechoosedialogSave)
            FilechoosedialogSave->hide();
        handle_button_logout_clicked();
        LoggingStatus=LOGGED_TIME_OUT;
        return false;
    }
}

void LIFEOGRAPH::draw_welcomescreen (void) {
    Gtk::Alignment *l_alignment = Gtk::manage( new Gtk::Alignment(0.5, 0.5, 0.2, 0) );
    l_alignment->set_padding(100, 100, 100, 100);
    WindowMain->add(*l_alignment);

    Gtk::Label *l_labelSalutation = Gtk::manage(
            new Gtk::Label( _(LIFEO::STRING::SALUTATION), 0, 0.5));
    l_labelSalutation->set_use_markup(true);

    Gtk::Button *l_buttonStart = Gtk::manage( new Gtk::Button(Gtk::Stock::OK) );
    l_buttonStart->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::draw_loginscreen));

    Gtk::VBox *l_vbox = Gtk::manage(new Gtk::VBox);
    l_vbox->set_spacing(30);

    l_vbox->pack_start(*l_labelSalutation);
    l_vbox->pack_start(*l_buttonStart);

    l_alignment->add(*l_vbox);

	WindowMain->show_all();

}

void LIFEOGRAPH::draw_loginscreen (void) {
    INTERNALOPERATION l_intop=InternalOperation;
    InternalOperation=INTOP_COMPLETE;
    WindowMain->remove();
	
    Gtk::VBox *l_vbox = Gtk::manage( new Gtk::VBox );
    
    WindowMain->add(*l_vbox);

	Gtk::Alignment *l_align1 = Gtk::manage( new Gtk::Alignment(0.5, 0.5, 0.3, 0.2) );
    l_align1->set_padding(110, 90, 50, 50);
	l_vbox->pack_start(*l_align1, Gtk::PACK_EXPAND_WIDGET);

    Gtk::Table *l_table1 = Gtk::manage( new Gtk::Table );
    l_align1->add(*l_table1);
    l_table1->set_spacings(5);

	ButtonOpenDatabase = Gtk::manage( new Gtk::Button(Gtk::Stock::OK) );

    Gtk::HSeparator *l_hseparator = Gtk::manage( new Gtk::HSeparator );
    Gtk::Button *l_buttonNewDB = Gtk::manage( new Gtk::Button(Gtk::Stock::NEW) );
	l_buttonNewDB->set_relief(Gtk::RELIEF_NONE);

	ButtonOpenDatabase->set_flags(Gtk::CAN_DEFAULT | Gtk::RECEIVES_DEFAULT);
	WindowMain->set_default(*ButtonOpenDatabase);

    FilebuttonDB = Gtk::manage( new Gtk::FileChooserButton );

	EntryPassword = Gtk::manage( new Gtk::Entry );
	EntryPassword->set_visibility(false);
	EntryPassword->set_activates_default(true);

    Gtk::Label *l_labelDBName =
            Gtk::manage( new Gtk::Label(_("Diary:"), 0, 0.5));
    Gtk::Label *l_labelPassword =
            Gtk::manage( new Gtk::Label(_("Password:"), 0, 0.5));
    LabelStartupMessage = Gtk::manage( new Gtk::Label("", 0, 0.5) );
    LabelStartupMessage->set_line_wrap(true);
    //LabelStartupMessage->set_single_line_mode(false);   // is this necessary?
    LabelStartupMessage->set_size_request(300, -1);

    l_table1->attach(*l_labelDBName, 0, 1, 0, 1, Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK);
    l_table1->attach(*l_labelPassword, 0, 1, 1, 2, Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK);

    l_table1->attach(*FilebuttonDB, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK);
    l_table1->attach(*EntryPassword, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK);

    l_table1->attach(*ButtonOpenDatabase, 2, 3, 0, 2,
            Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK | Gtk::FILL);

    l_table1->attach(*l_hseparator, 0, 3, 2, 3, Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK);

    l_table1->attach(*LabelStartupMessage, 0, 2, 3, 4, Gtk::EXPAND | Gtk::FILL, Gtk::FILL);
    l_table1->attach(*l_buttonNewDB, 2, 3, 3, 4, Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK);

	Gtk::Alignment *l_align2 = Gtk::manage( new Gtk::Alignment(1.0, 0.5, 0, 1.0) );

	l_vbox->pack_start(*l_align2, Gtk::PACK_SHRINK);

	Gtk::HBox *l_hbox4 = Gtk::manage( new Gtk::HBox );

	l_align2->add(*l_hbox4);

	Gtk::Button *l_buttonAbout = Gtk::manage( new Gtk::Button(Gtk::Stock::ABOUT) );
	l_buttonAbout->set_relief(Gtk::RELIEF_NONE);
	Gtk::Button *l_buttonQuit = Gtk::manage( new Gtk::Button(Gtk::Stock::QUIT) );
	l_buttonQuit->set_relief(Gtk::RELIEF_NONE);

	l_hbox4->pack_start(*l_buttonAbout, Gtk::PACK_SHRINK);
	l_hbox4->pack_start(*l_buttonQuit, Gtk::PACK_SHRINK);

    // SETTING DB FILE
    if (Database.is_path_set()) {
        FilebuttonDB->set_filename(Database.get_path());
    }
    else {
        EntryPassword->set_sensitive(false);
        ButtonOpenDatabase->set_sensitive(false);
        LabelStartupMessage->set_label(_(LIFEO::STRING::SELECT_DB));
    }

    // SIGNALS
    FilebuttonDB->signal_selection_changed().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_filebutton_db_selectionchanged));
    l_buttonNewDB->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_newdb_clicked));
	EntryPassword->signal_changed().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_entry_password_changed));
	ButtonOpenDatabase->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_opendb_clicked));
	l_buttonQuit->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_quit_clicked));
	l_buttonAbout->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_about_clicked));

    if (bool (PixbufBackground))
        SigconnectionEventExpose = WindowMain->signal_expose_event().connect(
                sigc::mem_fun(*this, &LIFEOGRAPH::handle_event_expose));

	WindowMain->show_all();
	WindowMain->set_title(LIFEO::PROGRAM_NAME);
    InternalOperation=l_intop;
}

void LIFEOGRAPH::draw_editscreen (void) {
    SigconnectionEventExpose.disconnect();
	WindowMain->remove();
	Gtk::VBox *l_vbox = Gtk::manage( new Gtk::VBox );
	WindowMain->add(*l_vbox);

	Gtk::Toolbar *l_toolbar = Gtk::manage( new Gtk::Toolbar );

	ToolbuttonPrevious = Gtk::manage( new Gtk::ToolButton );
	ToolbuttonNext = Gtk::manage( new Gtk::ToolButton );
    ToolbuttonToday = Gtk::manage( new Gtk::ToolButton );
	ToolbuttonFilter = Gtk::manage( new Gtk::ToggleToolButton );
    ToolbuttonLogout = Gtk::manage(
            new Gtk::ToolButton(Gtk::Stock::DIALOG_AUTHENTICATION) );
	Gtk::ToolItem *l_toolitemOptions = Gtk::manage( new Gtk::ToolItem );
	Gtk::ToolItem *l_toolitemSpace = Gtk::manage( new Gtk::ToolItem );
	LIFEOHELPERS::MENUBUTTON *l_menubuttonOptions = Gtk::manage(
			new LIFEOHELPERS::MENUBUTTON(Gtk::Stock::PREFERENCES,
            _("Options")) );

    // OPTIONS MENU
	Gtk::ImageMenuItem  *l_menuitemPassword = Gtk::manage(
            new Gtk::ImageMenuItem(_("_Change Password..."), true));
	Gtk::MenuItem       *l_menuitemBackup = Gtk::manage(
            new Gtk::MenuItem(_("_Save a Copy of Current Diary..."), true));
    Gtk::Image          *l_imageQuitWithoutSaving = Gtk::manage(
            new Gtk::Image(Gtk::Stock::STOP, Gtk::ICON_SIZE_MENU));
	Gtk::ImageMenuItem  *l_menuitemQuitWithoutSaving = Gtk::manage(
            new Gtk::ImageMenuItem(*l_imageQuitWithoutSaving,
            _("_Quit without Saving"),
            true));
	Gtk::ImageMenuItem  *l_menuitemAbout = Gtk::manage(
            new Gtk::ImageMenuItem(Gtk::Stock::ABOUT) );
    Gtk::SeparatorMenuItem *l_menuitemSeparator1 = Gtk::manage(
            new Gtk::SeparatorMenuItem );
    Gtk::SeparatorMenuItem *l_menuitemSeparator2 = Gtk::manage(
            new Gtk::SeparatorMenuItem );

	l_menubuttonOptions->get_menu()->attach(*l_menuitemPassword, 0, 1, 0, 1);
	l_menubuttonOptions->get_menu()->attach(*l_menuitemBackup, 0, 1, 1, 2);
	l_menubuttonOptions->get_menu()->attach(*l_menuitemSeparator1, 0, 1, 2, 3);
    l_menubuttonOptions->get_menu()->attach(*l_menuitemQuitWithoutSaving, 0, 1, 3, 4);
    l_menubuttonOptions->get_menu()->attach(*l_menuitemSeparator2, 0, 1, 4, 5);
	l_menubuttonOptions->get_menu()->attach(*l_menuitemAbout, 0, 1, 5, 6);
	l_menubuttonOptions->get_menu()->show_all_children();

	ToolbuttonLogout->set_label(_("Log out"));
	ToolbuttonLogout->set_is_important(true);
    ToolbuttonLogout->set_tooltip_text(Database.get_path());

    //TODO: will be deleted after moving to action:
    ToolbuttonToday->set_label(_("Today"));
    ToolbuttonToday->set_icon_name("x-office-calendar");
    ToolbuttonToday->set_tooltip_text(
            _("Go to today (creates a new entry if there isn't any)"));

	l_toolitemOptions->add(*l_menubuttonOptions);
    l_toolitemSpace->set_expand(true);

	Gtk::SeparatorToolItem *l_toolbarseparator1 = Gtk::manage(
            new Gtk::SeparatorToolItem );
	Gtk::SeparatorToolItem *l_toolbarseparator2 = Gtk::manage(
            new Gtk::SeparatorToolItem );

	l_toolbar->append(*ToolbuttonPrevious);
	l_toolbar->append(*ToolbuttonNext);
	l_toolbar->append(*ToolbuttonToday);
	l_toolbar->append(*l_toolbarseparator1);
	l_toolbar->append(*ToolbuttonFilter);
	l_toolbar->append(*l_toolbarseparator2);
	l_toolbar->append(*l_toolitemOptions);
	l_toolbar->append(*l_toolitemSpace);
	l_toolbar->append(*ToolbuttonLogout);

	l_vbox->pack_start(*l_toolbar, Gtk::PACK_SHRINK);

    // TEXTVIEW
	TextviewEntry = Gtk::manage( new Gtk::TextView );
	TextviewEntry->set_wrap_mode(Gtk::WRAP_WORD);
	TextviewEntry->set_left_margin(10);

    // TEXT STYLES
	Glib::RefPtr<Gtk::TextBuffer::TagTable> l_tagTable
            =TextviewEntry->get_buffer()->get_tag_table();
	Glib::RefPtr<Gtk::TextBuffer::Tag> l_tag
            =Gtk::TextBuffer::Tag::create("TextTagHeading");
	l_tag->property_foreground()="blue";
    l_tag->property_weight()=Pango::WEIGHT_BOLD;
	l_tag->property_scale()=1.5;
	l_tagTable->add(l_tag);
	Glib::RefPtr<Gtk::TextBuffer::Tag> l_tagSubheading =
            Gtk::TextBuffer::Tag::create("TextTagSubheading");
    l_tagSubheading->property_foreground()="violet";
    l_tagSubheading->property_weight()=Pango::WEIGHT_BOLD;
	l_tagSubheading->property_scale()=1.2;
    l_tagTable->add(l_tagSubheading);

	Gtk::ScrolledWindow *l_scrolledwindow1 = Gtk::manage( new Gtk::ScrolledWindow );
	l_scrolledwindow1->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
    l_scrolledwindow1->set_size_request(300, -1);
	l_scrolledwindow1->add(*TextviewEntry);
	l_vbox->pack_start(*l_scrolledwindow1);

	HboxNavigation = Gtk::manage( new Gtk::HBox );
	HboxFilter = Gtk::manage( new Gtk::HBox );

	l_vbox->pack_start(*HboxNavigation, Gtk::PACK_SHRINK);
	l_vbox->pack_start(*HboxFilter, Gtk::PACK_SHRINK);

	Calendar = Gtk::manage( new Gtk::Calendar );
	TreeviewEntries = Gtk::manage( new Gtk::TreeView );

	Gtk::ScrolledWindow *l_scrolledwindow2 = Gtk::manage( new Gtk::ScrolledWindow );
	l_scrolledwindow2->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	l_scrolledwindow2->add(*TreeviewEntries);
	l_scrolledwindow2->set_shadow_type(Gtk::SHADOW_IN);
    l_scrolledwindow2->set_size_request(250, -1);
	HboxNavigation->pack_start(*Calendar, Gtk::PACK_SHRINK);
	HboxNavigation->pack_start(*l_scrolledwindow2);

	TreeviewEntries->set_rules_hint(true);
	TreeviewEntries->set_model(ListstoreEntries);

    Glib::RefPtr<Gtk::IconTheme> l_theme = Gtk::IconTheme::get_default();
    PixbufFavored = l_theme->load_icon("emblem-favorite", 16, Gtk::ICON_LOOKUP_FORCE_SVG);
    // ordinary (not favored) entry icon. it supposed to be empty:
    //TODO: make sure that it is good practice to pass an unitialized RefPtr
    //      to column. seems ok for now but sounds strange.
//     PixbufUnfavored = Gdk::Pixbuf::create(Gdk::COLORSPACE_RGB, false, 8, 1, 1);
    // favorites column header:
    Gtk::Image *l_imageFavored= Gtk::manage( new Gtk::Image(PixbufFavored) );
    l_imageFavored->set_alignment(0, 0);

	TreeviewEntries->append_column("", ColrecEntry.Favored);
	TreeviewEntries->append_column(_("Date"), ColrecEntry.Date);
	TreeviewEntries->append_column(_("Title"), ColrecEntry.Title);
	TreeviewEntries->set_search_column(-1);     // disable built-in search

    TreeviewEntries->get_column(0)->set_sort_column(ColrecEntry.Favored);
    TreeviewEntries->get_column(1)->set_sort_column(ColrecEntry.Date);

    TreeviewEntries->get_column(0)->set_widget(*l_imageFavored);
    // initially sort entries by date
	ListstoreEntries->set_sort_column(ColrecEntry.Date, Gtk::SORT_ASCENDING);

    // FILTER BAR
    HboxFilter->set_spacing(5);
    HboxFilter->set_border_width(5);
    EntryFilter = Gtk::manage( new Gtk::Entry );
    ButtonFilterClear = Gtk::manage( new Gtk::Button(Gtk::Stock::CLEAR) );
    Gtk::Label * l_labelFilter = Gtk::manage(
            new Gtk::Label(_("Filter:")) );
    HboxFilter->pack_start(*l_labelFilter, Gtk::PACK_SHRINK);
    HboxFilter->pack_start(*EntryFilter);
    HboxFilter->pack_start(*ButtonFilterClear, Gtk::PACK_SHRINK);
    
    // ACCELERATORS
    Glib::RefPtr<Gtk::ActionGroup> l_actiongroup =
            Gtk::ActionGroup::create("lifeograph actions");

    ActionToggleFilter = Gtk::Action::create(
            "Filter",
            Gtk::Stock::FIND,
            // TRANSLATORS: this is the label of the toggle filter bar action
            _("Filter"),
            _("Toggle filter bar"));
    ActionToggleFilter->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::toggle_filterbar));
    l_actiongroup->add(ActionToggleFilter, Gtk::AccelKey(GDK_f, Gdk::CONTROL_MASK));
    ActionToggleFilter->set_accel_group(WindowMain->get_accel_group());
    ActionToggleFilter->connect_accelerator();
    ActionToggleFilter->connect_proxy(*ToolbuttonFilter);
    ActionToggleFilter->property_is_important() = true;

    ActionPrevious = Gtk::Action::create(
            "Previous",
            Gtk::Stock::MEDIA_REWIND, 
            _("Previous"), _("Go to the previous entry in the list"));
    ActionPrevious->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_toolbutton_previous_clicked));
    l_actiongroup->add(ActionPrevious, Gtk::AccelKey(GDK_Left, Gdk::MOD1_MASK));
    ActionPrevious->set_accel_group(WindowMain->get_accel_group());
    ActionPrevious->connect_accelerator();
    ActionPrevious->connect_proxy(*ToolbuttonPrevious);

    ActionNext = Gtk::Action::create(
            "Next",
            Gtk::Stock::MEDIA_FORWARD,
            _("Next"), _("Go to the next entry in the list"));
    ActionNext->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_toolbutton_next_clicked));
    l_actiongroup->add(ActionNext, Gtk::AccelKey(GDK_Right, Gdk::MOD1_MASK));
    ActionNext->set_accel_group(WindowMain->get_accel_group());
    ActionNext->connect_accelerator();
    ActionNext->connect_proxy(*ToolbuttonNext);

    /*ActionToday = Gtk::Action::create_with_icon_name( //TODO: after gtkmm 2.14
            "Today",
             "x-office-calendar",
            _("Today"), _("Go to today (creates a new entry if there isn't any)"));
    ActionToday->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::show_today));
    l_actiongroup->add(ActionToday, Gtk::AccelKey(GDK_t, Gdk::CONTROL_MASK));
    ActionToday->set_accel_group(WindowMain->get_accel_group());
    ActionToday->connect_accelerator();
    ActionToday->connect_proxy(*ToolbuttonToday);*/

    ToolbuttonLogout->add_accelerator(
            "clicked",
            WindowMain->get_accel_group(),
            GDK_Escape,
            Gdk::CONTROL_MASK,
            Gtk::ACCEL_VISIBLE);

    // SIGNALS
	l_menuitemPassword->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::start_dialog_password));
	l_menuitemBackup->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_backup_clicked));
    l_menuitemQuitWithoutSaving->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::quit_withoutsaving));
	l_menuitemAbout->signal_activate().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_about_clicked));

	ToolbuttonToday->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::show_today));
	ToolbuttonLogout->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_logout_clicked));
	TextviewEntry->get_buffer()->signal_changed().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_textview_entry_changed));
	TreeviewEntries->get_selection()->signal_changed().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_treeview_entries_selectionchanged));
	TreeviewEntries->signal_button_press_event().connect_notify(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_treeview_entries_event_buttonpress));
    ListstoreEntries->signal_sort_column_changed().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_treeview_sortcolumnchanged));

	Calendar->signal_day_selected_double_click().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_calendar_doubleclicked));
	Calendar->signal_month_changed().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_calendar_monthchanged));
	Calendar->signal_day_selected().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_calendar_dayselected));

    EntryFilter->signal_changed().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_entry_filter_changed));
    ButtonFilterClear->signal_clicked().connect(
            sigc::mem_fun(*this, &LIFEOGRAPH::handle_button_filter_clicked));

    if (Database.OptionPlainDB == false)
        SigconnectionEvent = WindowMain->signal_event().connect(
                sigc::mem_fun(*this, &LIFEOGRAPH::handle_event));

	WindowMain->show_all();
    HboxFilter->hide(); 
    l_imageFavored->show();
}

void LIFEOGRAPH::handle_button_quit_clicked (void) {
	Gtk::Main::quit();
}

void LIFEOGRAPH::handle_button_about_clicked (void) {
	std::list<Glib::ustring> l_authors;
	l_authors.push_back("Ahmet Öztürk <aoz_2@yahoo.com>");

	Gtk::AboutDialog *l_aboutdialog = new Gtk::AboutDialog;    
	l_aboutdialog->set_name(LIFEO::PROGRAM_NAME);
	l_aboutdialog->set_version(LIFEO::PROGRAM_VERSION_STRING);
    if (Glib::file_test(ProgramPath + ".svg", Glib::FILE_TEST_EXISTS)) {
	    l_aboutdialog->set_logo(Gdk::Pixbuf::create_from_file(ProgramPath + ".svg"));
    }
	l_aboutdialog->set_copyright(_("Copyright (C) 2007-2008 Ahmet Öztürk"));
	l_aboutdialog->set_comments(_(LIFEO::STRING::SLOGAN));
	l_aboutdialog->set_website("http://blhps.dizayn.de/");
	l_aboutdialog->set_website_label(_("Author's Homepage"));
	l_aboutdialog->set_authors(l_authors);
	l_aboutdialog->set_transient_for(*WindowMain);
	l_aboutdialog->run();
    delete l_aboutdialog;
}

bool LIFEOGRAPH::initialize_new_database (void) {
    FilechoosedialogSave = new Gtk::FileChooserDialog(
    // TRANSLATORS: this is the title of filechooser dialog for saving a new diary
    // for the first time
            _("Where to Save The New Diary?"), Gtk::FILE_CHOOSER_ACTION_SAVE);
	FilechoosedialogSave->add_button (Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	FilechoosedialogSave->add_button (Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
    bool l_returnValue;

	if (FilechoosedialogSave->run() == Gtk::RESPONSE_ACCEPT) {
		std::string l_path (FilechoosedialogSave->get_filename());
        //TODO: check for validity?
	    Database.set_path(l_path);
        l_returnValue = true;
	}
    else
        l_returnValue = false;

    delete FilechoosedialogSave;
    FilechoosedialogSave = NULL;
    return l_returnValue;
}

void LIFEOGRAPH::quit_withoutsaving (void) {
    Gtk::MessageDialog l_messagedialog(*WindowMain,
                                       "",
                                       false,
                                       Gtk::MESSAGE_WARNING,
                                       Gtk::BUTTONS_CANCEL,
                                       true);
    l_messagedialog.set_message(_(LIFEO::STRING::QUIT_WITHOUT_SAVING));
    l_messagedialog.set_secondary_text(_(LIFEO::STRING::QUIT_WITHOUT_SAVING_SUB));
    l_messagedialog.add_button (_("_QUIT WITHOUT SAVING"), Gtk::RESPONSE_ACCEPT);

    if (l_messagedialog.run() ==  Gtk::RESPONSE_ACCEPT) {
        LoggingStatus=LOGGED_OUT;   // in case...
        //Gtk::Main::quit();    // which one is to go with? this one or next one?
        WindowMain->hide();
    }
}

bool LIFEOGRAPH::logout (void) {
    //if (Database.Entry.size() > 0)
		close_entry(false);

	if (Database.write() == LIFEO::SUCCESS) {
		int l_x, l_y;
		WindowMain->get_size(l_x, l_y);
		GconfClient->set(LIFEO::CONFIG_PATH+"/width", l_x);
		GconfClient->set(LIFEO::CONFIG_PATH+"/height", l_y);
		WindowMain->get_position(l_x, l_y);
		GconfClient->set(LIFEO::CONFIG_PATH+"/position_x", l_x);
		GconfClient->set(LIFEO::CONFIG_PATH+"/position_y", l_y);
		GconfClient->set(LIFEO::CONFIG_PATH+"/db_path", Database.get_path());

        LoggingStatus=LOGGED_OUT;
        Database.clear();    
		return true;
	}
	else {
		Gtk::MessageDialog l_messagedialog(*WindowMain,
                                           "",
                                           false,
                                           Gtk::MESSAGE_WARNING,
                                           Gtk::BUTTONS_OK,
                                           true);
		l_messagedialog.set_message(_(LIFEO::STRING::CANNOT_WRITE));
		l_messagedialog.set_secondary_text(_(LIFEO::STRING::CANNOT_WRITE_SUB));

        l_messagedialog.run();
		
		return false;
	}

}

void LIFEOGRAPH::handle_button_logout_clicked (void) {
    if (logout()) {
        SigconnectionEvent.disconnect();
        SigconnectionTimeout.disconnect();
        draw_loginscreen();
    }
}

void LIFEOGRAPH::handle_button_backup_clicked (void) {
	FilechoosedialogSave = new Gtk::FileChooserDialog(
            _("Save a Copy of the Diary"), Gtk::FILE_CHOOSER_ACTION_SAVE);
	FilechoosedialogSave->add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	FilechoosedialogSave->add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);

	Gtk::ComboBoxText *l_combobox = Gtk::manage( new Gtk::ComboBoxText );
	l_combobox->append_text (_("Encrypt The Diary"));
	l_combobox->append_text (_("Do NOT Encrypt The Diary"));
	l_combobox->set_active(0);	// defaults to encrypted diary
	l_combobox->show();

	FilechoosedialogSave->set_extra_widget(*l_combobox);
	//TODO: append dates of first and last entries to the name?

	int l_result = FilechoosedialogSave->run();

	if (l_result == Gtk::RESPONSE_ACCEPT) {
		std::string l_path (FilechoosedialogSave->get_filename());
		// unencrypted diary is selected:
		if (l_combobox->get_active_row_number() == 1) {
			Database.write(false, l_path);
		}
		// encrypted diary is selected:
		else {
			if (!Database.is_passphrase_set()) {
				DIALOGPASSWORD l_dialogPassword(
                        Database.get_passphrase(),
                        // TRANSLATORS: this is a title for the password dialog
                        _("Choose a Password"),
                        true);
				l_dialogPassword.set_transient_for(*WindowMain);

				l_dialogPassword.show_all();
				l_dialogPassword.run();
				if (l_dialogPassword.is_passphrase_set()) {
					Database.set_passphrase(l_dialogPassword.get_passphrase());
                    Database.OptionPlainDB = false;
					Database.write(true, l_path);
                    Database.OptionPlainDB = true;
					Database.set_passphrase("");	// unset it again
				}
			}
			// if current diary has already a passphrase, the copy is saved with it:
			else
				Database.write(true, l_path);
		}
	}
    delete FilechoosedialogSave;
    FilechoosedialogSave = NULL;
}

void LIFEOGRAPH::handle_filebutton_db_selectionchanged (void) {
    if (InternalOperation != INTOP_USER) return;

    // to get around gtk who nastily issues this signal
    // a few times before it actually sets the file name:
    if (FilebuttonDB->get_filename().size() < 1) return;

    // when nothing is actually selected it defaults to the current directory
    EntryPassword->set_sensitive(false);
    ButtonOpenDatabase->set_sensitive(false);

    if (Glib::file_test(FilebuttonDB->get_filename(), Glib::FILE_TEST_IS_DIR)) return;

    Database.set_path(FilebuttonDB->get_filename());
    FilebuttonDB->set_tooltip_text(Database.get_path());
    LIFEO::RESULT l_result = Database.read_header();
    if (l_result == LIFEO::SUCCESS) {
        if (Database.OptionPlainDB) {
            LabelStartupMessage->set_text(_(LIFEO::STRING::DB_IS_NOT_ENCRYPTED));
            ButtonOpenDatabase->set_sensitive(true);
        }
        else {
            EntryPassword->set_sensitive(true);
            if (LoggingStatus == LOGGED_TIME_OUT)
                LabelStartupMessage->set_text(_(LIFEO::STRING::ENTER_PASSWORD_TIMEOUT));
            else
                LabelStartupMessage->set_text(_(LIFEO::STRING::ENTER_PASSWORD));
            ButtonOpenDatabase->set_sensitive(EntryPassword->get_text_length() > 0);
        }
    }
    else if (l_result == LIFEO::INCOMPATIBLE_FILE) {
        LabelStartupMessage->set_text(_(LIFEO::STRING::INCOMPATIBLE_DB));
    }
    else {
        LabelStartupMessage->set_text(_(LIFEO::STRING::FAILED_TO_OPEN_DB));
    }

    // FOCUS ADJUSTMENT
    if (Database.is_path_set() == false)
        FilebuttonDB->grab_focus();
    else if (Database.OptionPlainDB)
        ButtonOpenDatabase->grab_focus();
    else
        EntryPassword->grab_focus();
}

void LIFEOGRAPH::handle_button_newdb_clicked (void) {
    if (initialize_new_database()) {
        Database.add_today();
        draw_editscreen();
        populate_listview_entries();
        show_entry((*ListstoreEntries->children().begin()), UI_BOTH);
        LoggingStatus=LOGGED_IN;
        Database.OptionPlainDB = true;
    }
}

void LIFEOGRAPH::handle_entry_password_changed (void) {
    if (Database.is_path_set())
	    ButtonOpenDatabase->set_sensitive(EntryPassword->get_text_length() > 0);
}

void LIFEOGRAPH::handle_button_opendb_clicked (void) {
    if (!Database.OptionPlainDB) {
        std::string t_string = EntryPassword->get_text();
        Database.set_passphrase(t_string);
    }

	switch (Database.read_body()) {
        case LIFEO::EMPTY_DATABASE:
            Database.add_today();
            // continues:
		case LIFEO::SUCCESS:
			draw_editscreen();
			populate_listview_entries();
			show_entry((*ListstoreEntries->children().begin()), UI_BOTH);
			LoggingStatus=LOGGED_IN;
		break;
		case LIFEO::WRONG_PASSWORD:
            LabelStartupMessage->set_text(_(LIFEO::STRING::WRONG_PASSWORD));
			EntryPassword->set_text("");
			EntryPassword->set_sensitive(false);
			sleep(1);
			EntryPassword->set_sensitive(true);
			EntryPassword->grab_focus();
		break;
		default:

		break;
	}
}

void LIFEOGRAPH::handle_toolbutton_previous_clicked (void) {
	if (CurrentEntryRow != (*ListstoreEntries->children().begin())) {
        Gtk::TreeModel::Row l_row(CurrentEntryRow);
        l_row--;
		show_entry(l_row, UI_BOTH);
	}
}

void LIFEOGRAPH::handle_toolbutton_next_clicked (void) {
    Gtk::TreeRow l_rowLast = *ListstoreEntries->children().end();
    l_rowLast--;
	if (CurrentEntryRow != l_rowLast) {
        Gtk::TreeRow l_row(CurrentEntryRow);
        l_row++;
		show_entry(l_row, UI_BOTH);
	}
}

void LIFEOGRAPH::toggle_filterbar (void) {
    // prevent recursive calls:
    ActionToggleFilter->disconnect_proxy(*ToolbuttonFilter);

    if (HboxFilter->is_visible()) {
        EntryFilter->set_text("");
		HboxFilter->hide();
        ToolbuttonFilter->set_active(false);    // in case signal did not come from button
    }
	else {
		HboxFilter->show();
        ToolbuttonFilter->set_active(true);    // in case signal did not come from button
        EntryFilter->grab_focus();
    }

    ActionToggleFilter->connect_proxy(*ToolbuttonFilter);
}

void LIFEOGRAPH::handle_treeview_entries_selectionchanged (void) {
	if (InternalOperation != INTOP_USER) return;

	Glib::RefPtr<Gtk::TreeSelection> l_selection = TreeviewEntries->get_selection();
	if (l_selection->count_selected_rows() > 0) {
		Gtk::TreeModel::Row l_row = (*l_selection->get_selected());

		if (l_row != CurrentEntryRow) {
			show_entry (l_row, UI_CAL);
		}
	}
}

void LIFEOGRAPH::handle_treeview_entries_event_buttonpress (GdkEventButton * l_event) {
    if (l_event->button != 3)
        return;

	Glib::RefPtr<Gtk::TreeSelection> l_selection = TreeviewEntries->get_selection();

	if (l_selection->count_selected_rows() <= 0)
		return;

    // treeview issues selection chaned signal after button press signal.
    // this causes context menus to be linked to the previously selected item
    // when a row is selected with right click.
    // to get aroud this  we use a somewhat dirty hack and select the row
    // from within the button press event handler
    // p.s.: i dunno but there might be a simpler way of doing this
    Gtk::TreeModel::Path l_path;
    Gtk::TreeViewColumn  *l_column;
    int i, j;
    TreeviewEntries->get_path_at_pos((int) l_event->x,
                                     (int) l_event->y,
                                     l_path, l_column,
                                     i, j);
    TreeviewEntries->set_cursor(l_path);
    // end of dirty hack

	if (MenuEntry) {
		delete MenuEntry;
		MenuEntry = NULL;
	}
    
    MenuEntry = new Gtk::Menu;
    Gtk::TreeRow l_row = *(l_selection->get_selected());
    Gtk::Image *l_imageDismiss= Gtk::manage(
            new Gtk::Image(Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU) );
    Gtk::Image *l_imageFavored= Gtk::manage(
            new Gtk::Image(PixbufFavored) );
    MenuEntry->items().push_back(
        Gtk::Menu_Helpers::ImageMenuElem(_("_Dismiss"), *l_imageDismiss,
            sigc::bind(sigc::mem_fun(*this, &LIFEOGRAPH::delete_entry), l_row)) );

    MenuEntry->items().push_back(
        Gtk::Menu_Helpers::ImageMenuElem(
            Database.get_entry(l_row[ColrecEntry.Order]).is_favored() ?
                    _("Remove from _Favorites") : _("Add to _Favorites"),
            *l_imageFavored,
            sigc::bind(sigc::mem_fun(*this, &LIFEOGRAPH::toggle_entryfavoredness), l_row)) );

    MenuEntry->accelerate(*WindowMain); 
    MenuEntry->popup(l_event->button, l_event->time);
}

void LIFEOGRAPH::handle_treeview_sortcolumnchanged () {
    //FIXME: dirty hack: this signal is issued before list is actually sorted
    // so we delay the evaluation a little bit..
    Glib::signal_timeout().connect(sigc::mem_fun(*this, &LIFEOGRAPH::update_toolbar), 100); 
}

bool LIFEOGRAPH::update_toolbar () {
    Gtk::TreeRow l_rowEnd(*ListstoreEntries->children().end());
    l_rowEnd--;
	ActionNext->set_sensitive(CurrentEntryRow != l_rowEnd);
    ActionPrevious->set_sensitive(CurrentEntryRow != (
            *ListstoreEntries->children().begin()));
    return false;
}

void LIFEOGRAPH::handle_calendar_doubleclicked (void) {
    close_entry(false);
	guint l_year, l_month, l_day;
	Calendar->get_date(l_year, l_month, l_day);
    int l_order = Database.create_entry(l_year, l_month+1, l_day);

	populate_listview_entries();
    update_calendar();
    
    for (   Gtk::TreeIter l_itr = ListstoreEntries->children().begin(); 
            l_itr != ListstoreEntries->children().end();
            l_itr++) {
        if ((*l_itr)[ColrecEntry.Order] == l_order) {
            show_entry(*l_itr, UI_LIST);
            break;
        }
    }
    //TODO: give focus to the entry editor
    //TextviewEntry->grab_focus();  // does not work becasue it is probably 
                                    // not the last signal
}

void LIFEOGRAPH::handle_calendar_dayselected (void) {
	if (InternalOperation != INTOP_USER) return;

	guint l_year, l_month, l_day;
	Calendar->get_date(l_year, l_month, l_day);
	int l_entry = Database.get_entry(l_year, l_month+1, l_day);
                            
	if (l_entry >= 0) {
        for (Gtk::TreeModel::iterator l_itr = ListstoreEntries->children().begin();
                l_itr != ListstoreEntries->children().end(); l_itr++) {
            if (l_entry == (*l_itr)[ColrecEntry.Order]) {
		        show_entry ((*l_itr), UI_LIST);
                break;
            }
        }
	}
}

void LIFEOGRAPH::handle_calendar_monthchanged (void) {
    if (InternalOperation != INTOP_COMPLETE) {
	    update_calendar();
    }
}

void LIFEOGRAPH::handle_textview_entry_changed (void) {
    if (InternalOperation == INTOP_COMPLETE) return;
    // shortcut
    Glib::RefPtr<Gtk::TextBuffer> l_textbuffer = TextviewEntry->get_buffer();
	// beginning & end
	Gtk::TextIter l_iterBegin = l_textbuffer->begin();
    Gtk::TextIter l_iterEnd = l_textbuffer->end();
    // remove all tags: heading and subheadings
	l_textbuffer->remove_all_tags(l_iterBegin, l_iterEnd);

	size_t l_pos = l_textbuffer->get_text().find('\n', 0);
	if (l_pos == std::string::npos)
        l_pos = l_textbuffer->get_text().size();
	l_iterEnd=l_textbuffer->get_iter_at_offset (l_pos);

	l_textbuffer->apply_tag_by_name("TextTagHeading", l_iterBegin, l_iterEnd);

    if (InternalOperation == INTOP_USER) {
	    CurrentEntryRow[ColrecEntry.Title]
                =l_textbuffer->get_slice(l_iterBegin, l_iterEnd);
        EntryChanged=true;
    }

    // look for three consecutive empty lines for subheadings
    //TODO(0.5.X): make number of empty lines a variable that is set at 
    //      database file header
    for (   l_pos = 0;
            (l_pos=l_textbuffer->get_text().find("\n\n\n", l_pos))
                    != std::string::npos;
            l_pos+=3) {
        if (l_pos+4 < l_textbuffer->get_text().size()) {
            if (l_textbuffer->get_text()[l_pos+3] != '\n') {
                l_iterBegin=l_textbuffer->get_iter_at_offset(l_pos+3);
                l_iterEnd=l_textbuffer->get_iter_at_offset(
                        l_textbuffer->get_text().find('\n', l_pos+4));
                l_textbuffer->apply_tag_by_name("TextTagSubheading", l_iterBegin, l_iterEnd);
            }
        }
        else
            break;    // to not lose time
    }
}

void LIFEOGRAPH::handle_entry_filter_changed (void) {
    const std::string l_filterString (EntryFilter->get_text());
    if (l_filterString.size() > 0) {
        Database.set_filter(l_filterString);
    }
    else {
        Database.remove_filter();
    }

    close_entry(false);  // current entry must be closed here because it may get filtered out of the liststore
    update_calendar();
    populate_listview_entries();

    if (EntryCount > 0) {
        show_entry(*ListstoreEntries->children().begin(), UI_BOTH);
        TextviewEntry->set_sensitive(true);
    }
    else {
        INTERNALOPERATION l_intop=InternalOperation;    
        InternalOperation=INTOP_COMPLETE;
        TextviewEntry->get_buffer()->set_text("");
        TextviewEntry->set_sensitive(false);
        InternalOperation=l_intop;
    }
            
    std::stringstream l_ss;
    l_ss << "Found in <b>" << EntryCount << "</b> of <b>" << Database.get_size() << "</b>";
    EntryFilter->set_tooltip_markup(l_ss.str());
}

void LIFEOGRAPH::handle_button_filter_clicked (void) {
    EntryFilter->set_text("");
}

void LIFEOGRAPH::show_entry (const Gtk::TreeModel::Row& l_row, UPDATEINDEX l_indices) {
    INTERNALOPERATION l_intop=InternalOperation;
	InternalOperation=INTOP_PARTIAL;
    const int l_entry = l_row[ColrecEntry.Order];

    // check validity of CurrentEntryRow:
    if (CurrentEntryRow != *ListstoreEntries->children().end())
        close_entry();

    // this is necessary because if the last entry was empty it may have been deleted
    // by close_entry(), changing list order:
    for (   Gtk::TreeIter l_itr = ListstoreEntries->children().begin();
            l_itr != ListstoreEntries->children().end();
            l_itr++)
        if ((*l_itr)[ColrecEntry.Order] == l_entry) {
            CurrentEntryRow = (*l_itr);
            break;
        }

	TextviewEntry->get_buffer()->set_text(Database.Entry[l_entry].get_text());
	update_title();

	if (l_indices & UI_CAL) {
        // FIXME: strangely, the only way i figured out to avoid day invalidation warning
        // is first selecting a day which is in every month:
        Calendar->select_day(1);
		Calendar->select_month(Database.Entry[l_entry].get_month()-1,
                               Database.Entry[l_entry].get_year());
		Calendar->select_day(Database.Entry[l_entry].get_day());
	}
	InternalOperation=INTOP_COMPLETE;
	if (l_indices & UI_LIST) {
		TreeviewEntries->get_selection()->select(CurrentEntryRow);
        TreeviewEntries->scroll_to_row(ListstoreEntries->get_path(
                TreeviewEntries->get_selection()->get_selected()));
	}

    TextviewEntry->set_sensitive(true);

    update_toolbar();
	InternalOperation=l_intop;
}

bool LIFEOGRAPH::close_entry (bool l_visualize) {
    if (EntryCount < 1) return false;
    INTERNALOPERATION l_intop=InternalOperation;
	InternalOperation=INTOP_COMPLETE;
    bool l_returnValue(false);
    // delete entry if empty
	if (TextviewEntry->get_buffer()->get_text().size() < 1) {
		Database.delete_entry(CurrentEntryRow[ColrecEntry.Order]); // delete entry
        if (l_visualize) {
		    ListstoreEntries->erase(CurrentEntryRow);
            populate_listview_entries();
		    update_calendar();
        }
        l_returnValue=true;
	}
	else if (EntryChanged) {
		Database.Entry[ CurrentEntryRow[ColrecEntry.Order] ].set_text(
                TextviewEntry->get_buffer()->get_text());
        if (l_visualize) {
		    CurrentEntryRow[ColrecEntry.Title] =
                    Database.Entry[ CurrentEntryRow[ColrecEntry.Order] ].get_title();
        }
	}
	InternalOperation=l_intop;
    EntryChanged=false;
    return l_returnValue;
}

void LIFEOGRAPH::delete_entry (Gtk::TreeRow& l_row) {
	if (Database.Entry.size() > 0) {
		Gtk::MessageDialog l_msgDlg	(*WindowMain,
		        "",	false, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true);
		l_msgDlg.set_message (_("Are you sure, you want to dismiss?"));
		l_msgDlg.set_secondary_text (_("This operation cannot be undone!"));
		l_msgDlg.add_button (Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
		l_msgDlg.add_button (_("_DISMISS!"), Gtk::RESPONSE_ACCEPT);
		int l_result = l_msgDlg.run();
		if (l_result != Gtk::RESPONSE_ACCEPT)
			return;

        INTERNALOPERATION l_intop=InternalOperation;
        InternalOperation=INTOP_COMPLETE;
        EntryChanged=false;

		Database.delete_entry(l_row[ColrecEntry.Order]); // delete entry

		update_calendar();
		populate_listview_entries();

		if (Database.Entry.size() > 0) {
            show_entry(*ListstoreEntries->children().begin(), UI_BOTH);
        }
        else {
			TextviewEntry->get_buffer()->set_text("");
			TextviewEntry->set_sensitive(false);
			WindowMain->set_title(_("Empty Database"));
		}
        InternalOperation=l_intop;
	}
}

void LIFEOGRAPH::toggle_entryfavoredness (Gtk::TreeRow& l_row) {
    if (Database.get_entry(l_row[ColrecEntry.Order]).is_favored()) {
        Database.get_entry(l_row[ColrecEntry.Order]).make_unfavored();
        l_row[ColrecEntry.Favored] = PixbufUnfavored;
    }
    else {
        Database.get_entry(l_row[ColrecEntry.Order]).make_favored();
        l_row[ColrecEntry.Favored] = PixbufFavored;
    }
}

void LIFEOGRAPH::show_today (void) {
    int l_index(Database.get_todays_first_entry());
    if (l_index < 0) {
        close_entry(false);
        l_index = Database.add_today();
        populate_listview_entries();
        show_entry((*ListstoreEntries->children()[l_index]), UI_BOTH);
    }
    else if (l_index != CurrentEntryRow[ColrecEntry.Order]) {
        if (close_entry(false)) {
            if (l_index > CurrentEntryRow[ColrecEntry.Order])
                l_index--;
            populate_listview_entries();
        }
        show_entry((*ListstoreEntries->children()[l_index]), UI_BOTH);
    }
}

void LIFEOGRAPH::populate_listview_entries (void) {
    INTERNALOPERATION l_intop=InternalOperation;
    InternalOperation=INTOP_COMPLETE;
	std::vector<LIFEOENTRY>::iterator l_itrEntry;
	Gtk::TreeModel::Row l_row;

	ListstoreEntries->clear();
    EntryCount=0;
	int i = 0;
	for (l_itrEntry = Database.Entry.begin();
            l_itrEntry != Database.Entry.end(); l_itrEntry++) {
        if ((*l_itrEntry).get_filtered_out() == false) {
		    l_row = *(ListstoreEntries->append());
		    l_row[ColrecEntry.Date] = (*l_itrEntry).get_date_str();
            l_row[ColrecEntry.Favored] =
                    (*l_itrEntry).is_favored() ? PixbufFavored : PixbufUnfavored;
		    l_row[ColrecEntry.Title] = (*l_itrEntry).get_title();
		    l_row[ColrecEntry.Order] = i;
            EntryCount++;
        }
        i++;
	}
    // invalidate CurrentEntryRow
    CurrentEntryRow = *ListstoreEntries->children().end();

    InternalOperation=l_intop;
}

void LIFEOGRAPH::update_calendar (void) {
	guint l_year, l_month, l_day;
	Calendar->get_date(l_year, l_month, l_day);
	std::vector<LIFEOENTRY>::iterator l_itr;
	Calendar->clear_marks();

	for (l_itr = Database.Entry.begin(); l_itr != Database.Entry.end(); l_itr++) {
        if ((*l_itr).get_filtered_out()) {
            continue;
        }
		if ((*l_itr).get_year() == l_year) {
			if ((*l_itr).get_month() == l_month+1) {
				Calendar->mark_day((*l_itr).get_day());
			}
			else if ((*l_itr).get_month() < l_month+1)
				break;
		}
		else if ((*l_itr).get_year() < l_year)
			break;
	}
}

void LIFEOGRAPH::update_title (void) {
	WindowMain->set_title(
            Database.Entry[CurrentEntryRow[ColrecEntry.Order]].get_date_str() +
            ": " +
			Database.Entry[CurrentEntryRow[ColrecEntry.Order]].get_title());
}

void LIFEOGRAPH::start_dialog_password() {
	if (!DialogPassword) {
		DialogPassword = new DIALOGPASSWORD (Database.get_passphrase(),
                // TRANSLATORS: this is the title of the password dialog
                                             _("Change Password"),
                                             false);
		if (!DialogPassword) {
			// TODO: print error msg
			return;
		}
		DialogPassword->set_transient_for(*WindowMain);
		DialogPassword->show_all();

        DialogPassword->run();

        if (DialogPassword->is_passphrase_set()) { //TODO: if (DialogPassword->run())
		    Database.set_passphrase(DialogPassword->get_passphrase());
            Database.OptionPlainDB = false;
        }

	    delete DialogPassword;
	    DialogPassword = NULL;
	}
}

int LIFEOGRAPH::sort_date_column (const Gtk::TreeModel::iterator& l_itr1,
                                  const Gtk::TreeModel::iterator& l_itr2) {
    if ( (*l_itr1)[ColrecEntry.Date] > (*l_itr2)[ColrecEntry.Date] )
        return -1;
    else if  ( (*l_itr1)[ColrecEntry.Date] < (*l_itr2)[ColrecEntry.Date] )
        return 1;
    else
        return 0;

}

int LIFEOGRAPH::sort_favored_column (const Gtk::TreeModel::iterator& l_itr1,
                                     const Gtk::TreeModel::iterator& l_itr2) {
    if ( Database.get_entry((*l_itr1)[ColrecEntry.Order]).is_favored() >
              Database.get_entry((*l_itr2)[ColrecEntry.Order]).is_favored() )
        return -1;
    else if  ( Database.get_entry((*l_itr1)[ColrecEntry.Order]).is_favored() <
              Database.get_entry((*l_itr2)[ColrecEntry.Order]).is_favored() )
        return 1;
    else
        return 0;

}

