/***********************************************************************************

	Copyright (C) 2007-2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <cstdio>	// for remove() and rename()
#include <string>
#include <sstream>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <gcrypt.h>

#include "lifeograph.hpp"
#include "helpers.hpp"


using namespace LIFEO;


Entry::Entry()
	:	m_date( 0 ), m_text( "" ),
	m_option_favored( false ), m_option_spellcheck( true ), m_flag_filteredout( false )
{
}

Entry::Entry( Date date, const Glib::ustring &text )
	:	m_date( date ), m_text( text ),
	m_option_favored( false ), m_option_spellcheck( true ), m_flag_filteredout( false )
{
}

std::string
Entry::get_date_str( void ) const
{
	std::ostringstream oss;

	oss << isolate_year( m_date ) << ".";
	oss << std::setfill( '0' ) << std::setw( 2 ) << isolate_month( m_date ) << ".";
	oss << std::setfill( '0' ) << std::setw( 2 ) << isolate_day( m_date );
	return( oss.str() );
}

std::string
Entry::get_year_str( void ) const
{
	std::ostringstream oss;
	oss << isolate_year( m_date );

	return oss.str();
}

std::string
Entry::get_yearmonth_str( void ) const
{
	std::ostringstream oss;

	oss << isolate_year( m_date ) << ".";
	oss << std::setfill( '0' ) << std::setw( 2 ) << isolate_month( m_date );

	return oss.str();
}

std::string
Entry::get_monthday_str( void ) const
{
	std::ostringstream oss;

	oss << std::setfill( '0' ) << std::setw( 2 ) << isolate_month( m_date ) << "/";
	oss << std::setfill( '0' ) << std::setw( 2 ) << isolate_day( m_date );
	return oss.str();
}

std::string
Entry::get_day_str( void ) const
{
	std::ostringstream oss;

	oss << std::setfill( '0' ) << std::setw( 2 ) << isolate_day( m_date );
	return oss.str();
}

void
Entry::set_tags( const Tagset &tagset )
{
	m_tags = tagset;
}

const Tagset&
Entry::get_tags( void ) const
{
	return m_tags;
}

Tagset&
Entry::get_tags( void )
{
	return m_tags;
}

// DATABASE
Database::Database()
	:	m_passphrase( "" ), m_filter( "" ), m_flag_encrypted( false ),
	m_option_spellcheck( false ), m_option_entryliststyle( 0 ),
	m_theme( NULL ), m_flag_old( false )
{

}

void
Database::set_path( const std::string &path )
{
	m_path = path;
}

const std::string&
Database::get_path( void ) const
{
	return m_path;
}

bool
Database::is_path_set( void ) const
{
	return ( (bool) m_path.size() );
}

//TODO: this function should return the result of the operation
void
Database::set_passphrase( const std::string &passphrase )
{
	// TODO: resetting the password should be carried out by a dedicated function
	if( passphrase.size() >= PASSPHRASE_MIN_SIZE )
	{
		m_passphrase = passphrase;
		m_flag_encrypted = true;
	}
	else
	if( passphrase.size() == 0 )
		m_flag_encrypted = false;
}

std::string
Database::get_passphrase( void ) const
{
	return m_passphrase;
}

bool
Database::compare_passphrase( const std::string &passphrase ) const
{
	return( m_passphrase == passphrase );
}

bool
Database::is_passphrase_set( void ) const
{
	return (bool) m_passphrase.size();
}

LIFEO::Result
Database::read_body()
{
	if( m_flag_encrypted )
		return read_encrypted();
	else
		return read_plain();
}

LIFEO::Result
Database::read_plain()
{
	std::ifstream file( m_path.c_str() );

	if( ! file.is_open() )
	{
		LIFEO::Error( "Failed to open database file " + m_path );
		return LIFEO::COULD_NOT_START;
	}

	file.seekg( 0, std::ios::end );

	int readsize = file.tellg() - BodyPosition - 1;

	if( readsize <= 0 )
	{
		std::cout << "empty database" << std::endl;
		file.close();
		return LIFEO::EMPTY_DATABASE;
	}

	file.seekg( BodyPosition );

	char *readbuffer = new char[ readsize + 1 ];
	// TODO: check validity

	file.read( readbuffer, readsize );

	readbuffer[ readsize ] = 0;   // terminating zero

	file.close();

	std::stringstream stream( readbuffer );
	LIFEO::Result result = parse_db_body_text( stream );
	delete[] readbuffer;

	return result;
}

LIFEO::Result
Database::read_header()
{
	m_flag_old = false;
	std::ifstream file( m_path.c_str() );

	if( ! file.is_open() )
	{
		LIFEO::Error( "Failed to open database file " + m_path );
		return LIFEO::COULD_NOT_START;
	}
	std::string line;

	getline( file, line );

	if( line != LIFEO::DB_FILE_HEADER )
	{
		file.close();
		return read_header37();
	}

	while( getline( file, line ) )
	{
		switch( line[ 0 ] )
		{
			case 'V':
			{
				int version = convert_string( line.substr( 2 ) );
				if( version < LIFEO::DB_FILE_VERSION_INT_MIN ||
					version > LIFEO::DB_FILE_VERSION_INT )
				{
					std::cout << "Incompatible file version" << std::endl;
					file.close();
					return LIFEO::INCOMPATIBLE_FILE;
				}
				break;
			}
			case 'E':
				m_flag_encrypted = ( line[ 2 ] != 'n' );
				break;
			case 0: // end of header
				BodyPosition = file.tellg();
				file.close();
				return LIFEO::SUCCESS;
			default:
				std::cout << "Unrecognized line: " << line << std::endl;
				break;
		}
	}

	file.close();
	m_path = "";
	return LIFEO::CORRUPT_FILE;
}

LIFEO::Result
Database::parse_db_body_text( std::stringstream &stream )
{
	if( m_flag_old )
		return parse_db_body_text37( stream );
	std::string		line( "" );
	Glib::ustring	content( "" );
	Date			date( 0 );
	bool			favored( false );
	Tagset			tags;
	Chapter			*ptr2chapter = NULL;

	// TAG DEFINITIONS & CHAPTERS
	while( getline( stream, line ) )
	{
		if( line[ 0 ] == 0 )
			break;
		else
		if( line.size() < 3 )
			continue;
		else
		{
			switch( line[ 0 ] )
			{
				case 'T':	// tag
					try
					{
						Tag::create( line.substr( 2 ) );
					}
					catch( std::exception &ex )
					{
						throw ex;
					}
					break;
				case 'L':	// current entryliststyle
					m_option_entryliststyle = LIFEO::convert_string( line.substr( 2 ) );
					break;
				case 'S':	// chapterset
					try
					{
						Chapterset *chapterset = new Chapterset( line.substr( 2 ) );
						m_chaptersets.push_back( chapterset );
					}
					catch( std::exception &ex )
					{
						throw ex;
					}
					break;
				case 'C':	// chapter
					if( m_chaptersets.size() > 0 )	// chapter
					{
						try
						{
							ptr2chapter = m_chaptersets.back()->add_chapter(
									line.substr( 2 ) );
						}
						catch( std::exception &ex )
						{
							throw ex;
						}
					}
					break;
				case 'd':	// begin date
					if( ptr2chapter != NULL )	// chapter begin date
					{
						m_chaptersets.back()->set_chapter_date(
								ptr2chapter, line.substr( 2 ) );
					}
					break;
				case 'O':	// options
					m_option_spellcheck = ( line[2] == 's' );
					break;
				case 'M':
					if( m_theme == NULL )
						m_theme = new Theme;

					switch( line[1] )
					{
						case 'f':	// font
							m_theme->font = line.substr( 2 );
							break;
						case 'b':	// base color
							m_theme->color_base = line.substr( 2 );
							break;
						case 't':	// text color
							m_theme->color_text = line.substr( 2 );
							break;
						case 'h':	// heading color
							m_theme->color_heading = line.substr( 2 );
							break;
						case 's':	// subheading color
							m_theme->color_subheading = line.substr( 2 );
							break;
						case 'l':	// highlight color
							m_theme->color_highlight = line.substr( 2 );
							break;
					}
					break;
				default:
					LIFEO::Error error1( "Unrecognized line:" );
					LIFEO::Error error2( line );
					return LIFEO::FAILURE;
			}
		}
	}

	// ENTRIES
	while( getline( stream, line ) )
	{
		switch( line[ 0 ] )
		{
			case 'Y':    // year
				date += make_year( LIFEO::convert_string( line.substr( 2 ) ) );
				break;
			case 'M':    // month
				date += make_month( LIFEO::convert_string( line.substr( 2 ) ) );
				break;
			case 'D':    // day
				date += LIFEO::convert_string( line.substr( 2 ) );
				break;
			case 'A':
				date = LIFEO::convert_string( line.substr( 2 ) );
				break;
			case 'C':    // content
				// when writing a database file by hand, one can easily forget to put
				// a blank space after C, so we have to check
				if( line.size() < 2 )
					continue;

				content += line.substr( 2 );
				content += "\n";
				break;
			case 'T':    // tags
			{
				Tag::Tagpool::const_iterator iter_tag =
					Tag::get_pool().begin();
				for( std::string::size_type i = 2; i < line.size(); i++ )
				{
					if( iter_tag == Tag::get_pool().end() )
						//TODO: issue a warning
						break;
					else
					if( line[ i ] == '#' )
						tags.add_tag( *iter_tag );
					else
					if( line[ i ] == ' ' )
						continue;

					iter_tag++;
				}
			}
			break;
			case '}':    // end of Entry
			{
				// erase redundant new line:
				content.erase( content.size() - 1 );
				Entry *entry_new = new Entry( date, content );
				date = 0; // to be compatible with older versions (Y/M/D style)
				if( favored )
					entry_new->make_favored();
				entry_new->set_tags( tags );
				// TODO: per entry spellcheck options
				//~ entry_new->set_spellcheck( opt_spell );
				m_entries.push_back( entry_new );
				content.clear();
				tags.clear();
				break;
			}
			case '{':
				favored = false;
				break;
			case '[':
				favored = true;
				break;
			case 0:
			case '\n':
				break;
			default:
				LIFEO::Error error1( "Unrecognized line:" );
				LIFEO::Error error2( line );
				return LIFEO::FAILURE;
				break;
		}
	}

	if( m_entries.size() )
	{
		return LIFEO::SUCCESS;
	}
	else
	{
		std::cout << "empty database" << std::endl;
		return LIFEO::EMPTY_DATABASE;
	}
}

bool
Database::create_db_header_text( std::stringstream &output, bool encrypted )
{
	output << LIFEO::DB_FILE_HEADER;
	output << "\nV " << LIFEO::DB_FILE_VERSION_INT;
	output << "\nE " << ( encrypted ? "yes" : "no" );
	output << "\n\n"; // end of header

	return true;
}

bool
Database::create_db_body_text( std::stringstream &output )
{
	std::string::size_type	pt_start, pt_end;
	std::string				content_std;

	// TAG DEFINITIONS
	for(	Tag::Tagpool::const_iterator iter = Tag::get_pool().begin();
			iter != Tag::get_pool().end();
			iter++ )
	{
		output << "T " << ( *iter )->get_name() << '\n';
	}

	// CHAPTERS
	output << "L " << m_option_entryliststyle << '\n';

	for(	Chapterset::Set::iterator iter = m_chaptersets.begin();
			iter != m_chaptersets.end();
			iter++ )
	{
		output << "S " << ( *iter )->get_name() << '\n';
		for(	Chapterset::iterator iter2 = ( *iter )->begin();
				iter2 != ( *iter )->end();
				iter2++ )
		{
			output << "C " << ( *iter2 )->get_name() << '\n';
			if( ( *iter2 )->is_initialized() )
				output << "d " << ( *iter2 )->get_date_str() << '\n';
		}
	}
	// OPTIONS (SPELLCHECKING)
	output << "O " << ( m_option_spellcheck ? 's' : '-' ) << '\n';
	
	// THEME
	if( m_theme != NULL )
	{
		output << "Mf" << m_theme->font << '\n';
		output << "Mb" << m_theme->color_base << '\n';
		output << "Mt" << m_theme->color_text << '\n';
		output << "Mh" << m_theme->color_heading << '\n';
		output << "Ms" << m_theme->color_subheading << '\n';
		output << "Ml" << m_theme->color_highlight << '\n';
	}

	output << '\n'; // end of section

	// ENTRIES
	for(	Entryiter itr_entry = m_entries.begin();
			itr_entry != m_entries.end();
			itr_entry++ )
	{
		// PURGE EMPTY ENTRIES
		if( ( *itr_entry )->get_text().size() < 1 ) continue;

		output << ( ( *itr_entry )->is_favored() ? "[" : "{" );
		output << "\nA " << ( *itr_entry )->get_date() << '\n';

		// TAGS
		if( ( *itr_entry )->get_tags().size() > 0 )
		{
			output << "T ";
			for(    Tag::Tagpool::const_iterator itr_tags =
						Tag::get_pool().begin();
					itr_tags != Tag::get_pool().end();
					itr_tags++ )
			{
				output <<
				( ( *itr_entry )->get_tags().checkfor_member( *itr_tags ) ? '#' : '-' );
			}
			output << '\n';
		}
		// CONTENT
		// NOTE: for some reason, implicit conversion from Glib:ustring...
		// ...fails while substr()ing when LANG=C
		// we might reconsider storing text of entries as std::string.
		// for now we convert entry text to std::string here:
		content_std = ( *itr_entry )->get_text();
		pt_start = 0;

		while( true )
		{
			pt_end = content_std.find( '\n', pt_start );
			if( pt_end == std::string::npos )
			{
				pt_end = content_std.size();
				output	<< "C "
						<< content_std.substr( pt_start, pt_end - pt_start );
				break; // end of while( true )
			}
			else
			{
				pt_end++;
				output	<< "C "
						<< content_std.substr( pt_start, pt_end - pt_start );
				pt_start = pt_end;
			}
		}

		output << "\n}\n";

	}
	return true;
}

LIFEO::Result
Database::read_encrypted()
{
	std::ifstream file( m_path.c_str(), std::ios::in | std::ios::binary );

	if( ! file.is_open() )
	{
		LIFEO::Error( "Failed to open database file" + m_path );
		return LIFEO::COULD_NOT_START;
	}
	file.seekg( BodyPosition );

	std::string line, content;
	unsigned char *buffer;
	try {
		// Allocate memory for salt
		unsigned char *salt = new unsigned char[ LIFEO::Cipher::cSALT_SIZE ];
		if( ! salt )
			throw LIFEO::Error( "Unable to allocate memory for salt" );
		// Read salt value
		file.read( (char*) salt, LIFEO::Cipher::cSALT_SIZE );

		unsigned char *iv = new unsigned char[ LIFEO::Cipher::cIV_SIZE ];
		if( ! iv )
			throw LIFEO::Error( "Unable to allocate memory for IV" );
		// Read IV
		file.read( (char*) iv, LIFEO::Cipher::cIV_SIZE );

		unsigned char *key;
		LIFEO::Cipher::expand_key( m_passphrase.c_str(), salt, &key );

		// Calulate bytes of data in file
		size_t size = LIFEO::get_file_size( file ) - file.tellg();
		if( size <= 3 )
		{
			file.close();
			return LIFEO::EMPTY_DATABASE;
		}
		buffer = new unsigned char[ size + 1 ];
		// TODO(<0.4.0): ex handling here!!!

		file.read( (char*) buffer, size );
		LIFEO::Cipher::decrypt_buffer( buffer, size, key, iv );
		file.close();

		// passphrase check
		if( buffer[ 0 ] != m_passphrase[ 0 ] && buffer[ 1 ] != '\n' )
		{
			return LIFEO::WRONG_PASSWORD;
		}

		buffer[ size ] = 0;   // terminating zero
	}
	catch( LIFEO::Error error )
	{
		return LIFEO::COULD_NOT_START;
	}

	std::stringstream stream;
	buffer += 2;    // ignore first two chars which are for passphrase checking
	stream << buffer;
	buffer -= 2;    // restore pointer to the start of the buffer before deletion
	delete[] buffer;

	return parse_db_body_text( stream );
}

LIFEO::Result
Database::write( void )
{
	std::string path_old( m_path + ".~previousversion~" );

	if( Glib::file_test( m_path, Glib::FILE_TEST_EXISTS ) )
		rename( m_path.c_str(), path_old.c_str() );

	return m_flag_encrypted ? write_encrypted( m_path ) : write_plain( m_path );
}

bool
Database::write_backup()
{
	std::string		path_backup( m_path + ".~backup~" );
	Result result = m_flag_encrypted ?
			write_encrypted( path_backup ) : write_plain( path_backup );

	return( result == SUCCESS );
}

LIFEO::Result
Database::write_copy( const std::string &path, const std::string &passphrase )
{
	Result result;

	if( passphrase.empty() )
		result = write_plain( path );
	else
	{	
		std::string passphrase_actual( m_passphrase );
		m_passphrase = passphrase;
		result = write_encrypted( path );
		m_passphrase = passphrase_actual;
	}

	return result;
}

LIFEO::Result
Database::write_txt( const std::string &path )
{
	std::ofstream file( path.c_str(), std::ios::out | std::ios::trunc );
	if( ! file.is_open() )
	{
		std::cout << "i/o error!: " << path << std::endl;
		return LIFEO::COULD_NOT_START;
	}

	// ENTRIES
	for(	Entryiter itr_entry = m_entries.begin();
			itr_entry != m_entries.end();
			itr_entry++ )
	{
		// purge empty entries:
		if( ( *itr_entry )->get_text().size() < 1 ) continue;

		file << ( "---------------------------------------------\n\n\n" );

		// DATE AND FAVOREDNESS
		if( ( *itr_entry )->is_favored() )
			file << "*** ";
		file << ( *itr_entry )->get_date_str();
		if( ( *itr_entry )->is_favored() )
			file << " ***";

		// TAGS
		Tagset *tagset = &( ( *itr_entry )->get_tags() );
		for(	Tag::Tagpool::const_iterator itr_tags = tagset->begin();
				itr_tags != tagset->end();
				itr_tags++ )
		{
			if( itr_tags == tagset->begin() )
				file << "\n>>> "; // FIXME: use a more meaningful sign
			else
				file << ", ";

			file << ( *itr_tags )->get_name();
		}

		// CONTENT
		file << "\n\n" << ( *itr_entry )->get_text();

		file << "\n\n\n";
	}
	file.close();
	return SUCCESS;
}

LIFEO::Result
Database::write_plain( const std::string &path, bool flag_headeronly )
{
	std::ofstream file( path.c_str(), std::ios::out | std::ios::trunc );
	if( ! file.is_open() )
	{
		std::cout << "i/o error!: " << path << std::endl;
		return LIFEO::COULD_NOT_START;
	}

	std::stringstream output;
	create_db_header_text( output, flag_headeronly );
	// header only mode is for encrypted databases
	if( ! flag_headeronly )
	{
		create_db_body_text( output );
	}

	file << output.str();
	file.close();

	return LIFEO::SUCCESS;
}

LIFEO::Result
Database::write_encrypted( const std::string &path )
{
	// writing header:
	write_plain( path, true );
	std::ofstream file( path.c_str(), std::ios::out | std::ios::app | std::ios::binary );
	if( ! file.is_open() )
	{
		std::cout << "i/o error!: " << path << std::endl;
		return LIFEO::COULD_NOT_START;
	}
	std::stringstream output;
	// first char of passphrase for validity checking
	output << m_passphrase[ 0 ] << '\n';
	create_db_body_text( output );

	// encryption
	try {
		size_t size =  output.str().size() + 1;

		unsigned char *key, *salt;
		LIFEO::Cipher::create_new_key( m_passphrase.c_str(), &salt, &key );

		unsigned char *iv;
		LIFEO::Cipher::create_iv( &iv );

		unsigned char *buffer = new unsigned char[ size ];
		memcpy( buffer, output.str().c_str(), size );

		LIFEO::Cipher::encrypt_buffer( buffer, size, key, iv );

		file.write( (char*) salt, LIFEO::Cipher::cSALT_SIZE );
		file.write( (char*) iv, LIFEO::Cipher::cIV_SIZE );
		file.write( (char*) buffer, size );

	}
	catch( LIFEO::Error error )
	{
		return LIFEO::FAILURE;
	}

	file.close();

	return LIFEO::SUCCESS;
}

void
Database::clear()
{
	m_entries.clear();
	Tag::clear_pool();
	m_chaptersets.clear();
	m_passphrase.clear();
	m_filter.clear();
	// NOTE: only reset body options here:
	m_option_spellcheck = true;
	m_option_entryliststyle = 0;
	if ( m_theme != NULL )
	{
		delete m_theme;
		m_theme = NULL;
	}
}

void
Database::delete_tag( Tag *tag )
{
	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); iter++ )
		( * iter )->get_tags().remove_tag( tag );

	Tag::delete_tag( tag );
}

Entry*
Database::get_entry_today( void )
{
	Glib::Date date;
	date.set_time_current();

	return get_entry( combine_date(	date.get_year(),
									date.get_month(),
									date.get_day() ) );
}

Entry*
Database::add_today( void )
{
	Glib::Date date;
	date.set_time_current();

	return create_entry( combine_date(	date.get_year(),
										date.get_month(),
										date.get_day() ) );
}

Entry*
Database::get_entry( Date date )
{
	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); iter++ )
	{
		if( ( *iter )->get_filtered_out() == false )
			if( ( *iter )->get_date() == date )
				return( *iter );
	}
	return NULL;
}

Entry*
Database::get_entry( Date date, int dayorder )
{
	int i = 0;
	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); iter++ )
	{
		if( ( *iter )->get_filtered_out() == false )
			if( ( *iter )->get_date() == date )
			{
				if( i == dayorder )
					return( *iter );
				else
				if( i < dayorder )
					i++;
				else
					break;
			}
	}
	return NULL;
}

Entryvector*
Database::get_entries( Date date )
{
	Entryvector *entries = new Entryvector;

	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); iter++ )
	{
		if( ( *iter )->get_filtered_out() == false )
			if( ( *iter )->get_date() == date )
			{
				entries->push_back( *iter );
			}
	}
	return entries;
}

Entry*
Database::get_entry_before( const Entry *entry )
{
	Entryiter iter = m_entries.begin();

	iter++;
	for( ; iter != m_entries.end(); iter++ )
		if( *iter == entry )
		{
			iter--;
			return( *iter );
		}

	return NULL;
}

Entry*
Database::get_entry_after( const Entry *entry )
{
	Entryiter iter_last = m_entries.end();

	iter_last--;

	for( Entryiter iter = m_entries.begin(); iter != iter_last; iter++ )
		if( *iter == entry )
		{
			iter++;
			return( *iter );
		}

	return NULL;
}

Entry*
Database::get_entry_first( void )
{
	// return first unfiltered entry
	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); iter++ )
	{
		if( ( *iter )->get_filtered_out() == false )
			return( *iter );
	}
	return NULL;
}

Entry*
Database::get_entry_nextinday( Date date, const Entry *entry )
{
	Entryiter	iter_last			= m_entries.end();
	Entryiter	iter_firstinday		= iter_last;
	bool		flag_found			= false;

	for( Entryiter iter = m_entries.begin(); iter != iter_last; iter++ )
	{
		if( ( *iter )->get_date() == date )
		{
			if( iter_firstinday == iter_last )
				iter_firstinday = iter;
			if( *iter == entry )
				flag_found = true;
			else
			if( flag_found )
				return( *iter );
		}
		else
		if( iter_firstinday != iter_last )
			break;
	}

	if( iter_firstinday != iter_last && *iter_firstinday != entry )
		return( * iter_firstinday );
	else
		return NULL;
}

int
Database::get_entrydayorder( Entry const*const entry ) const
{
	int dayorder = 0;
	for( Entryciter iter = m_entries.begin(); iter != m_entries.end(); iter++ )
		if( entry->get_date() == ( *iter )->get_date() )
		{
			if( *iter == entry )
			{
				return dayorder;
			}
			else
				dayorder++;
		}

	return -1;
}

bool
Database::is_first( Entry const*const entry ) const
{
	return( entry == m_entries.front() );
}

bool
Database::is_last( Entry const*const entry ) const
{
	return( entry == m_entries.back() );
}

Entry*
Database::create_entry( Date date, const Glib::ustring &content )
{
	Entry *entry = new Entry( date, content );

	// find out the place:
	Entryiter iter = m_entries.begin();
	for( ; iter != m_entries.end(); iter++ )
	{
		if( ( *iter )->get_date() < date )
			break;
	}
	m_entries.insert( iter, entry );

	return( entry );
}

bool
Database::delete_entry( Entry *entry )
{
	for( Entryiter iter = m_entries.begin(); iter != m_entries.end(); iter++ )
		if( *iter == entry )
		{
			m_entries.erase( iter );
			return true;
		}

	return false;
}

int
Database::list_entries( void )
{
	int t_number_of_entries( 0 );
	for( Entryiter t1_itr = m_entries.begin(); t1_itr != m_entries.end(); t1_itr++ )
	{
		std::cout << ( *t1_itr )->get_year() << "/" <<
		( *t1_itr )->get_month() << "/" << ( *t1_itr )->get_day() << std::endl;
		t_number_of_entries++;
	}
	std::cout << t_number_of_entries << " entries total." << std::endl;
	return( t_number_of_entries );
}

int
Database::set_filter( const Glib::ustring &string )
{
	Entryiter itr_entry;

	for( itr_entry = m_entries.begin(); itr_entry != m_entries.end(); itr_entry++ )
	{
		( *itr_entry )->set_filtered_out(
			( *itr_entry )->get_text().lowercase().find( string ) ==
			std::string::npos );
	}

	m_filter = string;

	return 0;   // reserved
}

void
Database::remove_filter( void )
{
	Entryiter itr_entry;

	for( itr_entry = m_entries.begin(); itr_entry != m_entries.end(); itr_entry++ )
	{
		( *itr_entry )->set_filtered_out( false );
	}

	m_filter.clear();
}

bool
Database::is_filter_set( void )
{
	return( ! m_filter.empty() );
}

int
Database::replace_filter( const Glib::ustring& newtext )
{
	Glib::ustring::size_type iter_str;
	const int chardiff = newtext.size() - m_filter.size();

	for(	Entryiter iter_entry = m_entries.begin();
			iter_entry != m_entries.end();
			iter_entry++ )
	{
		if( ( *iter_entry )->get_filtered_out() )
			continue;

		Glib::ustring entrytext = ( *iter_entry )->get_text().lowercase();
		iter_str = 0;
		int count = 0;
		while(	( iter_str = entrytext.find( m_filter, iter_str ) ) !=
				std::string::npos )
		{
			( *iter_entry )->get_text().erase(
				iter_str + ( chardiff * count ), m_filter.size() );
			( *iter_entry )->get_text().insert(
				iter_str + ( chardiff * count ), newtext );
			count++;
			iter_str += m_filter.size();
		}
	}
	
	return 0;	// reserved
}

Chapterset*
Database::add_chapterset( const Glib::ustring &name )
{
	Chapterset *chapterset = new Chapterset( name );

	m_chaptersets.push_back( chapterset );
	return chapterset;
}

void
Database::remove_chapterset( Chapterset *chapterset )
{
	delete chapterset;
	m_chaptersets.remove( chapterset );
}

bool
Database::rename_chapterset( Chapterset *set, const Glib::ustring &name )
{
	for( Chapterset::Set::iterator iter = m_chaptersets.begin();
		 iter != m_chaptersets.end();
		 iter++ )
	{
		if( ( *iter )->get_name() == name )
			return false;
	}
	set->set_name( name );
	return true;
}

Glib::ustring
Database::find_uniquename_chapterset( const Glib::ustring &name )
{
	unsigned int i = 0;
	Glib::ustring name_unique = name;
	Chapterset::Set::iterator iter = m_chaptersets.begin();

	while( iter != m_chaptersets.end() )
	{
		for( iter = m_chaptersets.begin(); iter != m_chaptersets.end(); iter++ )
		{
			if( ( *iter )->get_name() == name_unique )
			{
				i++;
				name_unique = Glib::ustring::compose( "%1(%2)", name, i );
				break;
			}
		}
	}
	return name_unique;
}

// TAG
// static member declarations
Tag::Tagpool Tag::m_pool;

Tag::Tag( const Glib::ustring &name, unsigned int usercount )
	: m_name( name ), m_usercount( usercount )
{
	m_pool.push_back( this );
}

Tag::~Tag( void )
{
	m_pool.remove( this );
}

Tag*
Tag::create( const Glib::ustring &name )
{
	for(    Tag::Tagpool::iterator iter = m_pool.begin();
			iter != m_pool.end();
			iter++ )
	{
		if( name == ( *iter )->m_name )
			return( *iter );
	}
	return new Tag( name, 0 );
}

void
Tag::delete_tag( Tag *tag )
{
	m_pool.remove( tag );
	delete tag;
}

Tag*
Tag::get_tag( unsigned int tagorder )
{
	if( tagorder >= m_pool.size() )
		return NULL;

	Tag::Tagpool::const_iterator iter = m_pool.begin();
	advance( iter, tagorder );
	return( *iter );
}

Tag*
Tag::get_tag( const Glib::ustring &name )
{
	for(    Tag::Tagpool::iterator iter = m_pool.begin();
			iter != m_pool.end();
			iter++ )
	{
		if( name == ( *iter )->m_name )
		{
			return( *iter );
		}
	}
	return NULL;
}

const Tag::Tagpool&
Tag::get_pool( void )
{
	return m_pool;
}

void
Tag::clear_pool( void )
{
	m_pool.clear();
}

const Glib::ustring&
Tag::get_name( void ) const
{
	return m_name;
}

void
Tag::set_name( const Glib::ustring &name )
{
	m_name = name;
}

// TAGSET
Tagset::~Tagset( void )
{
	for(    Tag::Tagpool::iterator iter = this->begin();
			iter != this->end();
			iter++ )
	{
		( *iter )->m_usercount--;
	}
}

bool
Tagset::add_tag( const Glib::ustring &name )
{
	Tag *tag;

	// create a new tag if there is no one with this name
	try {
		tag = Tag::create( name );
	}
	catch( std::exception & ex )
	{
		//TODO: add an explanation of the error
		return false;
	}

	// ignore the tag if it is already a member of the set
	for(    Tag::Tagpool::iterator iter = this->begin();
			iter != this->end();
			iter++ )
	{
		if( tag == ( *iter ) )
			return false;
	}

	tag->m_usercount++;
	this->push_back( tag );
	return true;
}

bool
Tagset::add_tag( Tag *tag )
{
	for(    Tag::Tagpool::iterator iter = this->begin();
			iter != this->end();
			iter++ )
	{
		if( tag == ( *iter ) )
		{
			return false;
		}
	}

	tag->m_usercount++;
	this->push_back( tag );
	return true;
}

bool
Tagset::checkfor_member( const Tag *tag ) const
{
	for(    Tag::Tagpool::const_iterator iter = this->begin();
			iter != this->end();
			iter++ )
	{
		if( tag == ( *iter ) )
		{
			return true;
		}
	}
	return false;
}

const Tag*
Tagset::get_tag( unsigned int tagorder ) const
{
	unsigned int i = 0;

	for(    Tag::Tagpool::const_iterator iter = this->begin();
			iter != this->end();
			iter++ )
	{
		if( i == tagorder )
		{
			return( *iter );
		}
		i++;
	}
	return NULL;
}

bool
Tagset::remove_tag( const Glib::ustring &name )
{
	for(
		Tag::Tagpool::iterator iter = this->begin();
		iter != this->end();
		iter++ )
	{
		if( name == ( *iter )->get_name() )
		{
			( *iter )->m_usercount--;
			this->remove( *iter );
			return true;
		}
	}
	return false;
}

void
Tagset::remove_tag( Tag *ptr2tag )
{
	if( ptr2tag != NULL )
	{
		ptr2tag->m_usercount--;
		remove( ptr2tag );
	}
}

// CHAPTERS
Chapter::Chapter( const Glib::ustring &name, const Glib::Date &date )
	: m_name( name ), m_date_begin( date ), m_flag_initialized( true )
{

}

Chapter::Chapter( const Glib::ustring &name )
	: m_name( name ), m_flag_initialized( false )
{

}

//~ Chapter*
//~ Chapter::create(const Glib::ustring& name, const Glib::Date& date)
//~ {
//~ return new Chapter( name, date );
//~ }

Chapter*
Chapter::create( const Glib::ustring &name )
{
	return new Chapter( name );
}

const Glib::ustring&
Chapter::get_name( void ) const
{
	return m_name;
}

void
Chapter::set_name( const Glib::ustring &name )
{
	m_name = name;
}

const Glib::Date&
Chapter::get_date_glib( void ) const
{
	return m_date_begin;
}

Glib::ustring
Chapter::get_date_str( void ) const
{
	if( m_flag_initialized )
		return m_date_begin.format_string( "%Y.%m.%d" );
	else
		return "";
}

Date
Chapter::get_date( void ) const
{
	if( m_flag_initialized )
	{
		return combine_date(	m_date_begin.get_year(),
								m_date_begin.get_month(),
								m_date_begin.get_day() );
	}
	else
		return 0;
}

void
Chapter::set_date( const Glib::Date &date )
{
	m_date_begin = date;
	m_flag_initialized = true;
}

Chapterset::Chapterset( const Glib::ustring &name )
	: m_name( name )
{

}

const Glib::ustring&
Chapterset::get_name( void ) const
{
	return m_name;
}

void
Chapterset::set_name( const Glib::ustring &name )
{
	m_name = name;
}

Chapter*
Chapterset::add_chapter( const Glib::ustring &name )
{
	Chapter *chapter = Chapter::create( name );

	push_back( chapter );
	return chapter;
}

void
Chapterset::remove_chapter( Chapter* chapter )
{
	delete chapter;
	remove( chapter );
}

bool
Chapterset::rename_chapter( Chapter *chapter, const Glib::ustring &name )
{
	for( Chapterset::iterator iter = begin(); iter != end(); iter++ )
	{
		if( ( *iter )->get_name() == name )
			return false;
	}
	chapter->set_name( name );
	return true;
}

bool
Chapterset::set_chapter_date( Chapter *chapter, const Glib::ustring &str_date )
{
	Glib::Date date;

	date.set_parse( str_date );
	Chapterset::iterator iter_insert = end();

	if( ! date.valid() )
		return false;

	if( size() > 1 )
	{
		for( Chapterset::iterator iter = begin(); iter != end(); iter++ )
		{
			if( ( *iter )->is_initialized() )
			{
				if( ( *iter )->get_date_glib() == date )
					return false;
				else
				if( ( *iter )->get_date_glib() < date && ( *iter ) != chapter )
				{
					iter_insert = iter;
					break;
				}
			}
			else
			if( ( *iter ) != chapter )
			{
				iter_insert = iter;
				break;
			}
		}

		if( iter_insert != end() )
		{
			remove( chapter );
			insert( iter_insert, chapter );
		}
		else
		{
			remove( chapter );
			push_back( chapter );
		}
	}

	chapter->set_date( date );

	return true;
}


// OLD FILE READERS
LIFEO::Result
Database::read_header37()
{
	std::ifstream file( m_path.c_str() );

	if( ! file.is_open() )
	{
		LIFEO::Error( "37: Failed to open database file " + m_path );
		m_path = "";
		return LIFEO::COULD_NOT_START;
	}
	std::string line;

	getline( file, line );

	if( line != "RZNMDB" )
	{
		LIFEO::Error( "37: Corrupt diary file: " + m_path );
		file.close();
		m_path = "";
		return LIFEO::CORRUPT_FILE;
	}

	while( getline( file, line ) )
	{
		switch( line[ 0 ] )
		{
			case 'V':
				if( LIFEO::convert_string( line.substr( 2 ) ) < 34
					||
					LIFEO::convert_string( line.substr( 2 ) ) > 37 )
				{
					std::cout << "37: Incompatible file version" << std::endl;
					file.close();
					return LIFEO::INCOMPATIBLE_FILE;
				}
				break;
			case 'E':
				m_flag_encrypted = line.substr( 2 ) != "no";
				break;
			case '(':
				break;
			case ')':
			case '{':
				// end of header
				BodyPosition = file.tellg();
				file.close();
				m_flag_old = true;
				return LIFEO::SUCCESS;
				break;
		}
	}

	file.close();
	m_path = "";
	return LIFEO::CORRUPT_FILE;
}

LIFEO::Result
Database::parse_db_body_text37( std::stringstream &stream )
{
	std::string line, content( "" );

	int year( 0 ), month( 0 ), day( 0 ), blanklineCount( 0 );
	bool favored( false );

	while( getline( stream, line ) )
	{
		switch( line[ 0 ] )
		{
			case 'Y':    // year
				year = LIFEO::convert_string( line.substr( 2 ) );
				break;
			case 'M':    // month
				month = LIFEO::convert_string( line.substr( 2 ) );
				break;
			case 'D':    // day
				day = LIFEO::convert_string( line.substr( 2 ) );
				break;
			case 'C':    // content
				if( line.size() < 2 )
					continue;

				if( content.size() )  // if not the first line
					content += "\n";

				// subtitle conversion from 37 to 51 format
				if( line.size() == 2 )
					blanklineCount++;
				else
				{
					if( blanklineCount == 2 )
						content += ' ';
					blanklineCount = 0;
				}

				content += line.substr( 2 );
				break;
			case '}':    // end of Entry
			{
				Entry *n_entry =
					new Entry( combine_date( year, month, day ), content );
				if( favored )
					n_entry->make_favored();
				m_entries.push_back( n_entry );
				content.clear();
			}
				break;
			case '{':
				favored = false;
				blanklineCount = 0;
				break;
			case '[':
				favored = true;
				break;
			default:
				LIFEO::Error error1( "Unrecognized line:" );
				LIFEO::Error error2( line );
				return LIFEO::FAILURE;
				break;

		}
	}
	std::cout << m_entries.size() << " entrie(s) converted to new db format." << std::endl;
	return LIFEO::SUCCESS;
}
