/***********************************************************************************

	Copyright (C) 2009 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <gtkmm/table.h>

#include "dialogtheme.hpp"


using namespace LIFEO;


DialogTheme::DialogTheme( Database *ptr2database, TextbufferDiary *buffer )
	:	Gtk::Dialog( _( "Edit Theme" ) ),
	m_ptr2database( ptr2database ),
	m_ptr2buffer( buffer ),
	m_checkbutton( _( "Use Custom Theme" ) ),
	m_label_font( _( "Font:" ), Gtk::ALIGN_LEFT ),
	m_label_colors( _( "<b>Colors:</b>" ), Gtk::ALIGN_LEFT ),
	m_label_bg( _( "Background:" ), Gtk::ALIGN_LEFT ),
	m_label_fg( _( "Text:" ), Gtk::ALIGN_LEFT ),
	m_label_heading( _( "Heading:" ), Gtk::ALIGN_LEFT ),
	m_label_subheading( _( "Subheading:" ), Gtk::ALIGN_LEFT ),
	m_label_highlight( _( "Highlight:" ), Gtk::ALIGN_LEFT ),

	m_fontbutton( buffer->m_ptr2textview->get_style()->get_font().to_string() ),
	m_colorbutton_bg( buffer->m_ptr2textview->get_style()->get_base( Gtk::STATE_NORMAL ) ),
	m_colorbutton_fg( buffer->m_ptr2textview->get_style()->get_text( Gtk::STATE_NORMAL ) ),
	m_colorbutton_heading( buffer->m_tag_heading->property_foreground_gdk() ),
	m_colorbutton_subheading( buffer->m_tag_subheading->property_foreground_gdk() ),
	m_colorbutton_highlight( buffer->m_tag_highlight->property_background_gdk() )
{
	m_label_colors.set_use_markup( true );
	m_table.set_spacings( 3 );
	m_table.set_border_width( 5 );
	m_table.set_homogeneous( true );
	m_table.attach( m_label_font, 0, 1, 0, 1,
					Gtk::FILL | Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND, 20 );
	m_table.attach( m_fontbutton, 1, 2, 0, 1, Gtk::FILL | Gtk::EXPAND  );
	m_table.attach( m_label_colors, 0, 1, 1, 2,
					Gtk::FILL | Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND, 20 );
	m_table.attach(	m_label_bg, 0, 1, 2, 3,
					Gtk::FILL | Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND, 20 );
	m_table.attach( m_colorbutton_bg, 1, 2, 2, 3, Gtk::FILL | Gtk::EXPAND );
	m_table.attach( m_label_fg, 0, 1, 3, 4,
					Gtk::FILL | Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND, 20 );
	m_table.attach( m_colorbutton_fg, 1, 2, 3, 4, Gtk::FILL | Gtk::EXPAND );
	m_table.attach( m_label_heading, 0, 1, 4, 5,
					Gtk::FILL | Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND, 20 );
	m_table.attach( m_colorbutton_heading, 1, 2, 4, 5, Gtk::FILL | Gtk::EXPAND );
	m_table.attach( m_label_subheading, 0, 1, 5, 6,
					Gtk::FILL | Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND, 20 );
	m_table.attach( m_colorbutton_subheading, 1, 2, 5, 6, Gtk::FILL | Gtk::EXPAND );
	m_table.attach( m_label_highlight, 0, 1, 6, 7,
					Gtk::FILL | Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND, 20 );
	m_table.attach( m_colorbutton_highlight, 1, 2, 6, 7, Gtk::FILL | Gtk::EXPAND );

	get_vbox()->pack_start( m_checkbutton, false, false, 5 );
	get_vbox()->pack_start( m_table, true, true );
	get_vbox()->set_spacing( 5 );

	add_button( Gtk::Stock::CLOSE, Gtk::RESPONSE_CLOSE );

	if( m_ptr2database->has_theme() )
		m_checkbutton.set_active();
	else
		m_table.set_sensitive( false );

	m_fontbutton.signal_font_set().connect(
			sigc::mem_fun( *this, &DialogTheme::handle_fontbutton_changed ) );
	m_colorbutton_bg.signal_color_set().connect(
			sigc::mem_fun( *this, &DialogTheme::handle_color_bg_changed ) );
	m_colorbutton_fg.signal_color_set().connect(
			sigc::mem_fun( *this, &DialogTheme::handle_color_fg_changed ) );
	m_colorbutton_heading.signal_color_set().connect(
			sigc::mem_fun( *this, &DialogTheme::handle_color_heading_changed ) );
	m_colorbutton_subheading.signal_color_set().connect(
			sigc::mem_fun( *this, &DialogTheme::handle_color_subheading_changed ) );
	m_colorbutton_highlight.signal_color_set().connect(
			sigc::mem_fun( *this, &DialogTheme::handle_color_highlight_changed ) );

	m_checkbutton.signal_toggled().connect(
			sigc::mem_fun( *this, &DialogTheme::handle_checkbutton_toggled ) );

	m_ptr2buffer->m_ptr2textview->signal_unrealize().connect(
			sigc::bind( sigc::mem_fun( *this, &DialogTheme::response ),
			Gtk::RESPONSE_CANCEL ) );

	set_default_response( Gtk::RESPONSE_CLOSE );
//	set_default_size( 300, -1 );
	show_all();
}

void
DialogTheme::on_response( int response )
{
	// this is safe, isn't it?
	delete this;
}

void
DialogTheme::handle_checkbutton_toggled( void )
{
	if( m_checkbutton.get_active() )
	{
		m_table.set_sensitive( true );
	}
	else
	{
		m_table.set_sensitive( false );

		// FIXME: I know that this is a very dumb method to get the default font:
		Gtk::TextView *textview = new Gtk::TextView;
		textview->show();
		m_fontbutton.set_font_name( textview->get_style()->get_font().to_string() );
		delete textview;
		m_colorbutton_bg.set_color( TextbufferDiary::m_theme_color_base );
		m_colorbutton_fg.set_color( TextbufferDiary::m_theme_color_text );
		m_colorbutton_heading.set_color( TextbufferDiary::m_theme_color_heading );
		m_colorbutton_subheading.set_color( TextbufferDiary::m_theme_color_subheading );
		m_colorbutton_highlight.set_color( TextbufferDiary::m_theme_color_highlight );

		m_ptr2buffer->reset_theme();
		update_database();
	}
}

void
DialogTheme::handle_fontbutton_changed( void )
{
	Pango::FontDescription font( m_fontbutton.get_font_name() );
	m_ptr2buffer->m_ptr2textview->modify_font( font );

	update_database();
}

void
DialogTheme::handle_color_bg_changed( void )
{
	m_ptr2buffer->m_ptr2textview->modify_base( Gtk::STATE_NORMAL,
			m_colorbutton_bg.get_color()  );

	m_ptr2buffer->adjust_colors();

	update_database();
}

void
DialogTheme::handle_color_fg_changed( void )
{
	m_ptr2buffer->m_ptr2textview->modify_text( Gtk::STATE_NORMAL,
			m_colorbutton_fg.get_color()  );

	m_ptr2buffer->adjust_colors();

	update_database();
}

void
DialogTheme::handle_color_heading_changed( void )
{
	m_ptr2buffer->m_tag_heading->property_foreground_gdk() =
			m_colorbutton_heading.get_color();

	update_database();
}

void
DialogTheme::handle_color_subheading_changed( void )
{
	m_ptr2buffer->m_tag_subheading->property_foreground_gdk() =
			m_colorbutton_subheading.get_color();

	update_database();
}

void
DialogTheme::handle_color_highlight_changed( void )
{
	m_ptr2buffer->m_tag_highlight->property_background_gdk() =
			m_colorbutton_highlight.get_color();
	
	update_database();
}

void
DialogTheme::update_database( void )
{
	if( m_checkbutton.get_active() )
	{
		Theme *theme = new Theme;
		theme->font = m_fontbutton.get_font_name();
		theme->color_base = m_colorbutton_bg.get_color().to_string();
		theme->color_text = m_colorbutton_fg.get_color().to_string();
		theme->color_heading = m_colorbutton_heading.get_color().to_string();
		theme->color_subheading = m_colorbutton_subheading.get_color().to_string();
		theme->color_highlight = m_colorbutton_highlight.get_color().to_string();
		m_ptr2database->set_theme( theme );
	}
	else
		m_ptr2database->dismiss_theme();
}

