/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "lifeograph.hpp"
#include "app_window.hpp"
#include "view_entry.hpp"
#include "panel_extra.hpp"


using namespace LIFEO;

// PSEUDO DIARYELEMENT CLASS FOR HEADERS ===========================================================
class ListHeader : public DiaryElement
{
    public:
                                    ListHeader( DiaryElement::Type type )
        : DiaryElement( NULL, ES::VOID ), m_type( type ) {}

        void                        show() {}

        int                         get_size() const
        { return 0; }   // redundant
        DiaryElement::Type          get_type() const
        { return m_type; }

    protected:
        DiaryElement::Type          m_type;
};

static ListHeader s_header( DiaryElement::ET_HEADER ), s_none( DiaryElement::ET_NONE );

// TAG PANEL =======================================================================================
// STATIC MEMBERS
PanelExtra::Colrec*     PanelExtra::colrec;

PanelExtra::PanelExtra( BaseObjectType* cobject, const Glib::RefPtr<Gtk::Builder>& )
:   Gtk::TreeView( cobject ), m_treestore_all_tags( NULL )
{
    Gtk::CellRendererPixbuf* cellr_icon( NULL );
    Gtk::CellRendererPixbuf* cellr_filter( NULL );
    Gtk::CellRendererText* cellr_name( NULL );
    Gtk::TreeViewColumn* column( NULL );

    try
    {
        colrec = new Colrec;

        cellr_icon = Gtk::manage( new Gtk::CellRendererPixbuf );
        cellr_filter = Gtk::manage( new Gtk::CellRendererPixbuf );
        cellr_name = Gtk::manage( new Gtk::CellRendererText );
        column = Gtk::manage( new Gtk::TreeViewColumn( "" ) );

        Lifeograph::builder->get_widget( "button_filter", m_button_filter );

        Lifeograph::builder->get_widget_derived( "entry_search", m_entry_search );
        Lifeograph::builder->get_widget_derived( "entry_replace", m_entry_replace );
        Lifeograph::builder->get_widget( "button_match_prev", m_button_match_prev );
        Lifeograph::builder->get_widget( "button_match_next", m_button_match_next );
        Lifeograph::builder->get_widget( "button_replace", m_button_replace );
        Lifeograph::builder->get_widget( "button_replace_all", m_button_replace_all );
        Lifeograph::builder->get_widget( "grid_replace", m_grid_replace );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create the extra panel" );
    }

    cellr_name->property_ellipsize() = Pango::ELLIPSIZE_END;

    column->pack_start( *cellr_icon, false );
    column->pack_start( *cellr_name );
    column->pack_start( *cellr_filter, false );

    column->add_attribute( cellr_icon->property_pixbuf(), PanelExtra::colrec->icon );
    column->add_attribute( cellr_name->property_markup(), PanelExtra::colrec->name );
    column->add_attribute( cellr_filter->property_pixbuf(), PanelExtra::colrec->filter );

    m_treestore_all_tags = Gtk::TreeStore::create( *PanelExtra::colrec );

    // ALL TAGS
    this->set_model( m_treestore_all_tags );
    column->set_cell_data_func( *cellr_icon,
                                sigc::mem_fun( this, &PanelExtra::cell_data_func_icon ) );
    column->set_cell_data_func( *cellr_name,
                                sigc::mem_fun( this, &PanelExtra::cell_data_func_text ) );
    column->set_cell_data_func( *cellr_filter,
                                sigc::mem_fun( this, &PanelExtra::cell_data_func_filter ) );

    //column->set_sizing( Gtk::TREE_VIEW_COLUMN_FIXED );

    this->append_column( *column );

    this->override_background_color(
            AppWindow::p->get_style_context()->get_background_color( Gtk::STATE_FLAG_NORMAL ),
            Gtk::STATE_FLAG_NORMAL );
    this->override_background_color(
            AppWindow::p->get_style_context()->get_background_color( Gtk::STATE_FLAG_SELECTED ),
            Gtk::STATE_FLAG_SELECTED );

    // SIGNALS
    this->signal_row_activated().connect(
            sigc::mem_fun( this, &PanelExtra::handle_treeview_row_activated ) );

    this->signal_drag_begin().connect_notify(
            sigc::mem_fun( this, &PanelExtra::handle_treeview_drag_begin ) );

    this->signal_row_expanded().connect(
            sigc::mem_fun( this, &PanelExtra::handle_treeview_row_expanded ) );
    this->signal_row_collapsed().connect(
            sigc::mem_fun( this, &PanelExtra::handle_treeview_row_expanded ) );

    m_button_filter->signal_clicked().connect(
            sigc::mem_fun( Diary::d->m_filter_active, &DiaryElement::show ) );
    m_entry_search->signal_changed().connect(
            sigc::mem_fun( this, &PanelExtra::handle_search_string_changed ) );
    m_entry_search->signal_activate().connect(
            sigc::mem_fun( AppWindow::p->panel_diary, &PanelDiary::go_next_match ) );

    m_entry_replace->signal_activate().connect(
            sigc::mem_fun( this, &PanelExtra::replace_match ) );
    m_button_replace->signal_clicked().connect(
            sigc::mem_fun( this, &PanelExtra::replace_match ) );
    m_button_replace_all->signal_clicked().connect(
            sigc::mem_fun( this, &PanelExtra::replace_all_matches ) );

    // ACTIONS
    Lifeograph::create_action(
            false, m_action_focusfilter, "FocusFilter",
            Gtk::Stock::FIND, "", "",     // no grachical interface yet!
            Gtk::AccelKey( GDK_KEY_f, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_entry_search, &Gtk::Widget::grab_focus ) );

    Lifeograph::create_action(
            false, m_action_previous_match, "MatchPrevious", Gtk::Stock::GO_BACK,
// TRANSLATORS: label and tooltip of the go-to-previous-match-of-filtered-text action
            _( "Previous Match" ),
            _( "Go to previous match" ),
            Gtk::AccelKey( GDK_KEY_F3, Gdk::SHIFT_MASK ),
            sigc::mem_fun( AppWindow::p->panel_diary, &PanelDiary::go_prev_match ),
            m_button_match_prev );

    Lifeograph::create_action(
            false, m_action_next_match, "MatchNext", Gtk::Stock::GO_FORWARD,
// TRANSLATORS: label and tooltip of the go-to-next-match-of-filtered-text action
            _( "Next Match" ),
            _( "Go to next match" ),
            Gtk::AccelKey( GDK_KEY_F3, Gdk::ModifierType( 0 ) ),
            sigc::mem_fun( AppWindow::p->panel_diary, &PanelDiary::go_next_match ),
            m_button_match_next );

    // becoming proxy widgets makes them visible:
    m_button_match_prev->set_visible( false );
    m_button_match_next->set_visible( false );
}

void
PanelExtra::handle_login()
{
    if( Diary::d->is_read_only() )
    {
        unset_rows_drag_source();
        unset_rows_drag_dest();
    }
    else
    {
        enable_model_drag_source( Lifeograph::p->drag_targets_tag );
        enable_model_drag_dest( Lifeograph::p->drag_targets_tag, Gdk::ACTION_MOVE );
    }

    populate();
}

void
PanelExtra::handle_logout()
{
    Lifeograph::m_internaloperation++;

    m_treestore_all_tags->clear();

    m_entry_search->set_text( "" );
    m_entry_replace->set_text( "" );

    m_button_match_prev->hide();
    m_button_match_next->hide();
    m_grid_replace->hide();

    Lifeograph::m_internaloperation--;
}

void
PanelExtra::cell_data_func_icon( Gtk::CellRenderer* cell, const Gtk::TreeModel::iterator& iter )
{
    DiaryElement* elem( ( * iter )[ colrec->ptr ] );
    if( elem != NULL )
        cell->set_property( "visible", elem->get_type() != DiaryElement::ET_HEADER );
}

void
PanelExtra::cell_data_func_text( Gtk::CellRenderer* cell, const Gtk::TreeModel::iterator& iter )
{
    DiaryElement* elem( ( * iter )[ colrec->ptr ] );

    if( elem != NULL )
    {
        switch( elem->get_type() )
        {
            case DiaryElement::ET_HEADER:
                //cell->set_property( "background-gdk", Gdk::Color( "#DDDDDD" ) );
                if( Lifeograph::settings.small_lists )
                    cell->set_property( "scale", 1.0 );
                cell->set_property( "ypad", 5 );
                break;
            default:
                //cell->set_property( "background-set", false );
                if( Lifeograph::settings.small_lists )
                    cell->set_property( "scale", 0.9 );
                cell->set_property( "ypad", 0 );
                break;
        }
    }
}

void
PanelExtra::cell_data_func_filter( Gtk::CellRenderer* cell, const Gtk::TreeModel::iterator& iter )
{
    DiaryElement *elem( ( * iter )[ colrec->ptr ] );
    if( elem != NULL )
    {
        if( elem->get_type() == DiaryElement::ET_TAG ||
            elem->get_type() == DiaryElement::ET_UNTAGGED )
        {
            if( dynamic_cast< Tag* >( elem ) == Diary::d->get_filter_tag() )
            {
                cell->set_property( "visible", true );
                cell->set_property( "pixbuf", Lifeograph::icons->filter_16 );
                return;
            }
        }

        cell->set_property( "visible", false );
    }
}

void
PanelExtra::populate()
{
    Lifeograph::m_internaloperation++;

    m_treestore_all_tags->clear();

    Gtk::TreeRow row;

    // TAGS HEADER
    row = * m_treestore_all_tags->append();
    row[ colrec->ptr ] = &s_header;
    row[ colrec->name ] = Glib::ustring::compose( "<b>%1</b>", _( "TAGS" ) );

    if( Diary::d->get_tags()->size() < 1 && Diary::d->m_tag_categories.size() < 1 )
    {
        row = * m_treestore_all_tags->append();
        row[ colrec->ptr ] = &s_none;
        row[ colrec->name ] = Glib::ustring::compose( "<i>%1</i>", _( "None" ) );
    }

    // ROOT TAGS
    for( PoolTags::const_iterator iter( Diary::d->get_tags()->begin() );
         iter != Diary::d->get_tags()->end();
         ++iter )
    {
        Tag *tag( iter->second );
        if( tag->get_category() == NULL )
            add_elem_to_list( tag );
    }

    // CATEGORIES
    for( PoolCategoriesTags::const_iterator iter_category( Diary::d->m_tag_categories.begin() );
         iter_category != Diary::d->m_tag_categories.end();
         ++iter_category )
    {
        CategoryTags *category = iter_category->second;
        row = * add_elem_to_list( category );

        for( CategoryTags::const_iterator iter_tag = category->begin();
             iter_tag != category->end();
             ++iter_tag )
        {
            add_elem_to_list( *iter_tag, &row.children() );
        }

        if( category->get_expanded() )
            this->expand_row( m_treestore_all_tags->get_path( row ), false );
    }

    // EMPTY SEPARATOR ROW
    row = * m_treestore_all_tags->append();
    row[ colrec->ptr ] = &s_none;

    // UNTAGGED PSEUDO TAG
    add_elem_to_list( &Diary::d->m_untagged );

    // EMPTY SEPARATOR ROW
    row = * m_treestore_all_tags->append();
    row[ colrec->ptr ] = &s_none;

    // FILTERS GROUP
    //add_elem_to_list( Diary::d->m_filter_active );

    /*for( FilterVector::iterator iter = )
        add_elem_to_list( *iter );*/

    Lifeograph::m_internaloperation--;
}

inline Gtk::TreeIter
PanelExtra::add_elem_to_list( DiaryElement* elem, const Gtk::TreeNodeChildren* children )
{
    Gtk::TreeRow row = * ( children != NULL ? m_treestore_all_tags->append( *children ) :
                                              m_treestore_all_tags->append() );

    row[ colrec->ptr ] = elem;
    row[ colrec->name ] = elem->get_list_str();
    row[ colrec->icon ] = elem->get_icon();

    elem->m_list_data->treepath = m_treestore_all_tags->get_path( row );

    return row;
}

Gtk::TreeRow
PanelExtra::get_row( const Gtk::TreePath& path )
{
    return( * m_treestore_all_tags->get_iter( path ) );
    // TODO: check validity
}

void
PanelExtra::set_filtered_tag( const Tag* tag )
{
    static const Tag *tag_old( NULL );

    if( tag != tag_old && tag_old != NULL )
    {
        Gtk::TreeRow row = get_row( tag_old->m_list_data->treepath );
        row[ colrec->filter ] = Lifeograph::icons->filter_16;
    }
    if( tag != NULL )
    {
        Gtk::TreeRow row = get_row( tag->m_list_data->treepath );
        row[ colrec->filter ] = Lifeograph::icons->filter_16;
    }

    tag_old = tag;
}

void
PanelExtra::refresh_elem_icon( DiaryElement* elem )
{
    Gtk::TreeRow row = get_row( elem->m_list_data->treepath );
    row[ PanelExtra::colrec->icon ] = elem->get_icon();
}

void
PanelExtra::handle_treeview_drag_begin( const Glib::RefPtr< Gdk::DragContext >& )
{
    Lifeograph::s_flag_dragging = true;
    this->get_selection()->unselect_all();
}

bool
PanelExtra::on_drag_motion( const Glib::RefPtr< Gdk::DragContext >& context,
                                 int x, int y, guint time )
{
    if( ! Lifeograph::s_flag_dragging )
        return false;

    if( ! Lifeograph::s_elem_dragged )
        return false;

    if( Lifeograph::s_elem_dragged->get_type() != DiaryElement::ET_TAG )
        return false;

    bool retval( false );
    DiaryElement *element;
    Gtk::TreePath path;
    Tag *tag_dragged( dynamic_cast< Tag* >( Lifeograph::s_elem_dragged ) );

    // AUTOSCROLL
    Gdk::Rectangle rect;
    get_visible_rect( rect );
    int new_y = rect.get_y();
    if( y < ( rect.get_height() * 0.15 ) )
    {
        if( new_y > 0 )
        {
            new_y -= 5;
            if( new_y < 0 )
                new_y = 0;
            scroll_to_point( -1, new_y );
        }
    }
    else
    if( y > ( rect.get_height() * 0.85 ) )
    {
        if( get_vadjustment()->get_value() < get_vadjustment()->get_upper() )
        {
            new_y += 5;
            scroll_to_point( -1, new_y );
        }
    }

    // EVALUATE THE HOVERED ELEMENT
    if( get_path_at_pos( x, y, path ) )
    {
        Gtk::TreeRow row( * m_treestore_all_tags->get_iter( path ) );
        element = row[ colrec->ptr ];
        if( element != NULL )
        {
            if( element->get_type() == DiaryElement::ET_TAG_CTG )
            {
                CategoryTags *ctg_dropped( dynamic_cast< CategoryTags* >( element ) );
                if( tag_dragged->get_category() != ctg_dropped )
                    retval = true;
            }
            else if( element->get_type() == DiaryElement::ET_TAG )
            {
                Tag *tag_dropped( dynamic_cast< Tag* >( element ) );
                if( tag_dragged->get_category() != tag_dropped->get_category() )
                    retval = true;
            }
        }
    }

    if( retval )
    {
        m_elem_drop_target = element;
        context->drag_status( Gdk::ACTION_MOVE, time );
        get_selection()->select( path );
    }
    else
        get_selection()->unselect_all();

    return retval;
}

bool
PanelExtra::on_drag_drop( const Glib::RefPtr< Gdk::DragContext >& context,
                          int x, int y, guint time )
{
    bool flag_successful( false );

    if( Lifeograph::s_flag_dragging )
    {
        if( Lifeograph::s_elem_dragged != NULL || m_elem_drop_target != NULL )
        {
            if( Lifeograph::s_elem_dragged->get_type() == DiaryElement::ET_TAG )
            {
                Tag *tag( dynamic_cast< Tag* >( Lifeograph::s_elem_dragged ) );
                if( m_elem_drop_target->get_type() == DiaryElement::ET_TAG_CTG )
                {
                    CategoryTags *ctg( dynamic_cast< CategoryTags* >( m_elem_drop_target ) );
                    tag->set_category( ctg );
                    flag_successful = true;
                }
                else if( m_elem_drop_target->get_type() == DiaryElement::ET_TAG )
                {
                    Tag *target_tag( dynamic_cast< Tag* >( m_elem_drop_target ) );
                    tag->set_category( target_tag->get_category() );
                    flag_successful = true;
                }
            }
        }
    }

    if( flag_successful )
    {
        populate();
    }

    context->drag_finish( flag_successful, false, time );
    return flag_successful;
}

bool
PanelExtra::on_button_press_event( GdkEventButton* event )
{
    Lifeograph::s_flag_dragging = false;
    Lifeograph::s_elem_dragged = NULL;

    Gtk::TreePath path;
    if( this->get_path_at_pos( event->x, event->y, path ) )
    {
        Gtk::TreeRow row = * m_treestore_all_tags->get_iter( path );
        Lifeograph::s_elem_dragged = row[ colrec->ptr ];
        if( Lifeograph::s_elem_dragged == NULL )
            return true;    // ignore
        else
        {
            switch( Lifeograph::s_elem_dragged->get_type() )
            {
                case DiaryElement::ET_TAG:
                case DiaryElement::ET_UNTAGGED:
                case DiaryElement::ET_FILTER:
                    break;
                case DiaryElement::ET_TAG_CTG:
                    // TODO: prevent start of a drag action
                    break;
                default:
                    Lifeograph::s_elem_dragged = NULL;
                    return true;
            }
        }
    }
    else
    {
        return true;
    }

    return Gtk::TreeView::on_button_press_event( event );
}

bool
PanelExtra::on_button_release_event( GdkEventButton* event )
{
    if( ! Lifeograph::s_flag_dragging /*&& event->button != 1*/ )
    {
        if( Lifeograph::s_elem_dragged )
            Lifeograph::s_elem_dragged->show();
    }

    get_selection()->unselect_all();

    return Gtk::TreeView::on_button_release_event( event );
}

void
PanelExtra::handle_treeview_row_expanded( const Gtk::TreeIter& iter,
                                          const Gtk::TreePath& path )
{
    if( Lifeograph::m_internaloperation )
        return;

    DiaryElement *element = ( * iter )[ colrec->ptr ];
    if( element )
    {
        if( element->get_type() == DiaryElement::ET_TAG_CTG )
        {
            CategoryTags *category = dynamic_cast< CategoryTags* >( element );
            category->set_expanded( this->row_expanded( path ) );
        }
    }
}

void
PanelExtra::handle_treeview_row_activated( const Gtk::TreePath& path, Gtk::TreeViewColumn* )
{
    Gtk::TreeRow row( * m_treestore_all_tags->get_iter( path ) );
    DiaryElement* element( row[ colrec->ptr ] );

    if( element == NULL )
        return;

    if( element->get_type() == DiaryElement::ET_TAG ||
        element->get_type() == DiaryElement::ET_UNTAGGED )
    {
        const Tag* tag( dynamic_cast< Tag* >( element ) );

        if( tag == Diary::d->get_filter_tag() ) // remove filter if already filtered
            tag = NULL;

        Diary::d->set_filter_tag( tag );
        set_filtered_tag( tag );

        AppWindow::p->m_tag_view->update_button_filter_tag();
        AppWindow::p->panel_diary->update_entry_list();
    }
}

// FILTER & REPLACE
void
PanelExtra::handle_search_string_changed()
{
    if( Lifeograph::m_internaloperation ) return;
    // current entry must be closed here or else it loses...
    // ...changes since it was selected:
    //AppWindow::p->m_entry_view->sync(); update_entry_list() does the syncing

    const std::string filterstring( m_entry_search->get_text().lowercase() );

    Diary::d->set_search_text( filterstring );
    AppWindow::p->panel_diary->update_entry_list();    // syncs current entry

    if( filterstring.size() > 0 )
    {
        if( AppWindow::p->panel_diary->get_entry_count() > 0 )
        {
            AppWindow::p->m_entry_view->get_buffer()->set_search_str( filterstring );
            if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
                AppWindow::p->m_entry_view->get_buffer()->select_searchstr_next();
        }

        m_entry_search->set_tooltip_markup(
                Glib::ustring::compose(
                        _( "Found in <b>%1</b> of <b>%2</b> entrie(s)" ),
                        AppWindow::p->panel_diary->get_entry_count(), Diary::d->get_size() ) );
    }
    else
    {
        AppWindow::p->m_entry_view->get_buffer()->set_search_str( "" );
        m_entry_search->set_tooltip_text( _( "Enter text to be filtered" ) );
    }

    AppWindow::p->panel_diary->update_calendar();

    if( AppWindow::p->panel_diary->get_entry_count() < 1 || filterstring.size() < 1 )
    {
        m_button_match_prev->hide();
        m_button_match_next->hide();
        m_grid_replace->hide();
    }
    else
    {
        m_button_match_prev->show();
        m_button_match_next->show();
        if( ! Diary::d->is_read_only() )
            m_grid_replace->show();
    }
}

void
PanelExtra::replace_match()
{
    if( ! AppWindow::p->m_entry_view->get_buffer()->get_has_selection() )
        AppWindow::p->panel_diary->go_next_match();
    if( AppWindow::p->m_entry_view->get_buffer()->get_has_selection() )
    {
        AppWindow::p->m_entry_view->get_buffer()->erase_selection();
        AppWindow::p->m_entry_view->get_buffer()->insert_at_cursor( m_entry_replace->get_text() );
        AppWindow::p->panel_diary->go_next_match();
    }
}

void
PanelExtra::replace_all_matches()
{
    AppWindow::p->m_entry_view->sync();

    Diary::d->replace_text( m_entry_replace->get_text() );
    // update current entry:
    if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
    {
        Entry *entry = dynamic_cast< Entry* >( AppWindow::p->panel_main->get_cur_elem() );
        AppWindow::p->m_entry_view->get_buffer()->set_richtext( entry );
    }
}
