/* ----------------------------------------------------------------------
LIGGGHTS - LAMMPS Improved for General Granular and Granular Heat
Transfer Simulations

www.liggghts.com | www.cfdem.com
Christoph Kloss, christoph.kloss@cfdem.com

LIGGGHTS is based on LAMMPS
LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
http://lammps.sandia.gov, Sandia National Laboratories
Steve Plimpton, sjplimp@sandia.gov

Copyright (2003) Sandia Corporation. Under the terms of Contract
DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
certain rights in this software. This software is distributed under
the GNU General Public License.

See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#include "string.h"
#include "stdlib.h"
#include "atom.h"
#include "error.h"
#include "memory.h"
#include "modify.h"
#include "math.h"
#include "fix_property_atom.h"
#include "fix_property_global.h"
#include "fix_cfd_coupling.h"
#include "cfd_regionmodel_differential.h"

#define DELTA 10000

using namespace LAMMPS_NS;

/* ---------------------------------------------------------------------- */

CfdRegionmodelDifferential::CfdRegionmodelDifferential(LAMMPS *lmp, int jarg,int narg, char **arg,FixCfdCoupling *fc)  :
  CfdRegionmodelNone(lmp, jarg, narg, arg,fc)
{
    pos_lastup = NULL;
    arraypos_lastc = NULL;

    iarg = jarg;

    if(narg < iarg + 1) error->all("Not enough arguments");

    limsqr = atof(arg[iarg]) * atof(arg[iarg]);
    fprintf(screen,"Set movement limit for CfdRegionmodelDifferential to %f m.",sqrt(limsqr));
    iarg++;

}

/* ---------------------------------------------------------------------- */

CfdRegionmodelDifferential::~CfdRegionmodelDifferential()
{
    if(pos_lastup) modify->delete_fix("pos_lastup");
    if(arraypos_lastc) modify->delete_fix("arraypos_lastc");
}

/* ---------------------------------------------------------------------- */

void CfdRegionmodelDifferential::init()
{
    if(!atom->tag_enable) error->all("Cfd coupling regionmodel needs atom tags");

    char* fixarg[11];
    if(!inRegion)
    {
        fixarg[0]="inRegion";
        fixarg[1]="all";
        fixarg[2]="property/atom";
        fixarg[3]="inRegion";
        fixarg[4]="scalar"; 
        fixarg[5]="yes";    
        fixarg[6]="no";    
        fixarg[7]="no";    
        fixarg[8]="0.";
        modify->add_fix(9,fixarg);
    }

    inRegion = static_cast<FixPropertyAtom*>(modify->find_fix_property("inRegion","property/atom","scalar",1,0,"cfd_regionmodel differential"));

    if(!pos_lastup)
    {
        fixarg[0]="pos_lastup";
        fixarg[1]="all";
        fixarg[2]="property/atom";
        fixarg[3]="pos_lastup";
        fixarg[4]="vector"; 
        fixarg[5]="yes";    
        fixarg[6]="no";    
        fixarg[7]="no";    
        fixarg[8]="0.";
        fixarg[9]="0.";
        fixarg[10]="0.";
        modify->add_fix(11,fixarg);
    }

    pos_lastup = static_cast<FixPropertyAtom*>(modify->find_fix_property("pos_lastup","property/atom","vector",3,0,"cfd_regionmodel differential"));

    if(!outRegion)
    {
        fixarg[0]="outRegion";
        fixarg[1]="all";
        fixarg[2]="property/global";
        fixarg[3]="outRegion";
        fixarg[4]="vector";
        fixarg[5]="0.";
        fixarg[6]="0.";
        modify->add_fix(7,fixarg);
    }

    outRegion = static_cast<FixPropertyGlobal*>(modify->find_fix_property("outRegion","property/global","vector",2,0,"cfd_regionmodel differential"));

    if(!arraypos_lastc)
    {
        fixarg[0]="arraypos_lastc";
        fixarg[1]="all";
        fixarg[2]="property/global";
        fixarg[3]="arraypos_lastc";
        fixarg[4]="vector";
        fixarg[5]="-1.";
        fixarg[6]="-1.";
        modify->add_fix(7,fixarg);
    }

    arraypos_lastc = static_cast<FixPropertyGlobal*>(modify->find_fix_property("arraypos_lastc","property/global","vector",2,0,"cfd_regionmodel differential"));

    special_settings();
}

/* ---------------------------------------------------------------------- */

void CfdRegionmodelDifferential::special_settings()
{
  //values to be transfered to OF
  add_push_property("inRegion","scalar");
  add_push_property("outRegion","globalvector");

  //values to come from OF
  add_pull_property("inRegion","scalar");
}

/* ---------------------------------------------------------------------- */

void CfdRegionmodelDifferential::rm_update()
{
   
   nout = nlocal_last;
   int nlocal = atom->nlocal;

   outRegion->grow(nout,0);

   inregion = inRegion->vector_atom;
   outregion = outRegion->values;
   outRegion->nvalues = nout;
   x_upd = pos_lastup->array_atom;
   arraypos_lastcouple = arraypos_lastc->values;

   double **x = atom->x;

   for(int i = 0; i < nout; i++)
   {
       outregion[i] = 1.;
   }

   for(int i = 0; i < nlocal; i++)
   {
       bool isNew = (x_upd[i][0] == 0.) && (x_upd[i][1] == 0.) && (x_upd[i][2] == 0.);
       double distsqr = (x[i][0] - x_upd[i][0])*(x[i][0] - x_upd[i][0]) + (x[i][1] - x_upd[i][1])*(x[i][1] - x_upd[i][1]) + (x[i][2] - x_upd[i][2])*(x[i][2] - x_upd[i][2]);
       if(!isNew && distsqr < limsqr && inregion[i] == 1.)
         outregion[static_cast<int>(arraypos_lastcouple[i])] = 0.;

       if(!inregion[i])
       {
          if(isNew || distsqr >= limsqr)
          {
              inregion[i] = 1.;
              x_upd[i][0] = x[i][0];
              x_upd[i][1] = x[i][1];
              x_upd[i][2] = x[i][2];
          }
       }
       arraypos_lastcouple[i] = static_cast<double>(i);
   }

   nlocal_last = nlocal;
}
