#!/usr/bin/perl

#
# batch_replace.pl: apply regular expression substitution to multiple files
#
# Copyright (C) 2006 Markus Grabner (grabner@icg.tugraz.at)
#
#	This program is free software; you can redistribute it and/or
#	modify it under the terms of the GNU General Public License as
#	published by the Free Software Foundation, version 2.
#


sub usage {
    print<<EOP;
usage: batch_replace.pl [-b] [-e <code>] [-f <file>] [-h] [-n] [-q] <patterns>

options:
	-b: create backup files (*.bak) for all modified files
	-e <code>: use Perl code <code> to replace strings
	-f <file>: read Perl code from <file> to replace strings
	-h: display this message
	-n: don\'t modify existing files, create new ones (*.new) instead
	-q: quiet operation

<patterns> is a comma-separated list of file patterns that are searched for.
You must use quotes if your pattern contains wildcards that would otherwise
be expanded by the shell (e.g., '"*.c,*.h"').

The Perl code to be executed to replace strings is given either on the command
line ("-e") or in a file ("-f"). The code is typically a substitution operator,
but can be any valid Perl code. The input record separator is empty, i.e., the
whole file is read at once. You must use the 's' option if you want to process
more than one line, e.g. 's/\\/\\*.*?\\*\\///s;' to remove all multi-line
comments.
EOP
    exit @_[0];
}

# process arguments:
$tentative = 0;
$make_backup = 0;
$quiet = 0;
$expr = "";
$file = "";

while(($_ = shift @ARGV) =~ /^-/) {
    if   ($_ eq "-n") { $tentative = 1; }
    elsif($_ eq "-b") { $make_backup = 1; }
    elsif($_ eq "-e") { $expr = shift @ARGV; }
    elsif($_ eq "-f") { $file = shift @ARGV; }
    elsif($_ eq "-q") { $quiet = 1; }
    elsif($_ eq "-h") { usage(0); }
    else              { print "unknown option \"$_\"\n"; usage(1); }
}

usage(1) if ($#ARGV != -1) || ($expr eq "" && $file eq "");
$pattern = $_;
#$replace = shift @ARGV if $expr ne "";
$findargs = join(" -or ", map { "-name \"$_\"" } split(/,/, $pattern));
open(files, "find $findargs |");
$num_matching = 0;
$num_modified = 0;

while(<files>) {
    # read file name:
    chop;
    next if /\.bak$/;
    next if /\.new$/;
    $name = $_;
    $name_new = $name.".new";
    $name_bak = $name.".bak";
    ++$num_matching;

    # read original file:
    open(in, $name);
    local $/;  # empty input record separator -> read whole file at once
    $_ = <in>;
    close(in);

    # apply modifications:
    if($expr ne "") {
	if(!defined eval $expr) {
	    print "couldn't compile \"$expr\"\n";
	    exit 1;
	}
    }
    else {
	if(!defined do $file) {
	    if($@) {
		print "couldn't compile \"$file\"\n";
		exit 1;
	    }
	    else {
		print "couldn't open \"$file\"\n";
		exit 1;
	    }
	}
    }

    # write modified file:
    open(out, ">".$name_new);
    print out $_;
    close(out);

    # check for difference and shuffle file names according to user options:
    if(system("diff --brief $name $name_new > /dev/null")) {
	++$num_modified;
	print "modified: $name\n" unless $quiet;

	if(!$tentative) {
	    if($make_backup) {
		rename($name, $name_bak);
	    }
	    else {
		unlink($name);
	    }

	    rename($name_new, $name);
	}
    }
    else {
	print "matched:  $name\n" unless $quiet;
	unlink($name_new);
    }
}

print "$num_matching files matched\n$num_modified files modified\n" unless $quiet;
