#include <unistd.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <syslog.h>
#include "iface.h"
#include "global_vars.h"
#include "cmd.h"
#include "cmd_lcp3.h"
#include "lists.h"
#include "proc.h"
#include "execute.h"
#include "client.h"
#include "proc_supp.h"
#include "times.h"
#include "lines.h"
#include "../config.h"

#if HAVE_MYSQL
	#include "mysql_backend.h"
#endif

/*------------*/
int call_con_up(struct client_t *who, struct line_t *line);
int call_con_dn(struct client_t *who, struct line_t *line);
int set_offline(struct line_t *line);
void set_line_status(struct line_t *line, int status);
/*------------*/

int call_con_up(struct client_t *who, struct line_t *line)
{
 char *errmsg01 = "line '%s': was already established!";
 char *errmsg02 = "line '%s': unable to execute script_up!";
 segf_indicator = 3200;
 if ( !line )
 {
     syslog(LOG_ERR, "bad programmer: call_con_up(.., NULL) was called!");
     return(-1);
 }
 if ( line->con_stat != CST_DOWN ) { return(-2); }
 set_line_status(line, CST_DIALING);
 segf_indicator = 3201;
 if ( iface_up(server->tcp_socket, line) == 0 )
 {
	 segf_indicator = 3202;
     init_throughput(line);
     set_line_status(line, CST_UP_SERVER);
     if ( line->id == 1 )
         cmd_broadcast(CBR_CON_UP, errmsg01, strlen(errmsg01));
	 segf_indicator = 3203;
    return(0);
 }
 segf_indicator = 3204;
 if ( !(line->script_up[0]) )
 {
	 segf_indicator = 3205;
     syslog(LOG_ERR, errmsg02, line->linename);
     set_line_status(line, CST_DOWN);
 }
 else if ( exec_dont_care_cmdline(line->script_up) )
 { // failed to fork
	 segf_indicator = 3206;
     syslog(LOG_ERR, errmsg02, line->linename);
     set_line_status(line, CST_DOWN);
 }
 else
 {
	 segf_indicator = 3207;
     (struct client_t*)line->con_action.who = who;
     line->con_action.begin = time(NULL);
     server->fast_loop++;
     set_line_status(line, CST_DIALING);
     line->con_action.doing = DOING_OPENING;
 }
 segf_indicator = 3208;
 return(0);
}

int call_con_dn(struct client_t *who, struct line_t *line)
{
 char *errmsg02 = "line '%s': unable to call script_down!";
 pid_t pid;
 if ( !line )
 {
     syslog(LOG_ERR, "bad programmer: call_con_dn(.., NULL) was called!");
     return(-1);
 }
 if ( !(    (line->con_stat == CST_UP_SERVER)
         || (line->con_stat == CST_CLOSE_ERROR) ) )
     return(-2);
 set_line_status(line, CST_CLOSING);
 if ( iface_up(server->tcp_socket, line) == -1 )
 { // already down...
     set_line_status(line, CST_DOWN);
     if ( line->id == 1 )
         cmd_broadcast(CBR_CON_CLOSED, NULL, 0);
     return(0);
 }
 // kill all but last isdn-channels
 if ( (line->used_channels > 1) && (line->con_type == CT_ISDN)
     && (line->isdn_channel_op[0]) )
 {
     int i;
     pid_t pid;
     int ret_status;
     
     for (i=line->used_channels;i>1;i--)
     {
         if ( !( pid = fork()) )
         { // child
		     if (pid == -1) // O-oh! Big problem...
		     {
		         syslog(LOG_ERR, "line '%s': unable to close additional ISDN channels (fork failed)!", line->linename);
		     }
		     else
		     {
		         char *idx;
				 segf_indicator = 401;
				 idx = line->isdn_channel_op + strlen(line->isdn_channel_op);
				 segf_indicator = 402;
                 while ( (*idx != '/') && (idx > line->isdn_channel_op) ) idx--;
				 segf_indicator = 403;
                 if ( *idx == '/' ) idx++;
	                 execlp(line->isdn_channel_op, idx, "remove", NULL);
				 segf_indicator = 404;
                 // returns only on error
                 exit(-1);
		     }
         }
		 segf_indicator = 405;
         waitpid(pid, &ret_status, WUNTRACED);
		 segf_indicator = 406;
         if ( WIFEXITED(ret_status) && (!WEXITSTATUS(ret_status)) )
         {   // script returned 0 --> ok, channel removed
             line->used_channels--;
		 segf_indicator = 407;
		     lcp3_cmd_broadcast(CBR3_LINESTAT, NULL, 0, line);
         }
		 segf_indicator = 408;
     }
	 segf_indicator = 409;
     if ( line->used_channels > 1 )
     {
		 segf_indicator = 410;
         syslog(LOG_ERR, "line '%s': unable to close %d additional isdn channel(s)!", line->linename, line->used_channels-1);
		 segf_indicator = 411;
		 if ( line->id == 1 ) cmd_broadcast(CMD_BADCHANOP, NULL, 0);
		 segf_indicator = 412;
		 lcp3_cmd_queue(CMD3_CHANFAIL, who, NULL, 0);
		 segf_indicator = 413;
     }
 }
 if ( (!(line->script_dn[0])) || ((pid = fork()) == -1) )
 {
     syslog(LOG_ERR, errmsg02, line->linename);
     set_line_status(line, CST_CLOSE_ERROR);
     if ( line->id == 1 )
         cmd_broadcast(CBR_CLOSE_ERROR, NULL, 0);
 }
 else if ( !pid ) // child
 {
     // need's top security, don't pass to exec_dont_care because of
     // cmd_broadcast at the end in this prozess.
     int argc;
     char *argv[20];
	 segf_indicator = 414;
     parse_cmdline(line->script_dn, &argc, argv);
	 segf_indicator = 415;
     execvp(argv[0], argv);
	 segf_indicator = 416;
     // returns only on error
     syslog(LOG_ERR, "%s", errmsg02);
     set_line_status(line, CST_CLOSE_ERROR);
     if ( line->id == 1 )
         cmd_broadcast(CBR_CLOSE_ERROR, NULL, 0);
     exit(-1); // the con-checker will fail and report after timeout...
 }
 else // parent
 { // should work... go on.
     line->con_action.doing = DOING_CLOSING;
     (struct client_t*)line->con_action.who = who;
     line->con_action.begin = time(NULL);
     server->fast_loop++;
     set_line_status(line, CST_CLOSING);
 }
 // parent
// >-----------------//
 return(0);
}


int set_offline(struct line_t *line)
// sets all clients of the line to offline and closes the line if neccessary
// if called with NULL as line, alls clients get set to offline and
// all line get closed.
{
    struct client_t *who;
    int oldstat, oldlinestat;
    int all = 0;
    struct line_t *curline = line;
    if ( !line )
    {
	curline = (struct line_t*)lstlines.first;
	all++;
    }
    while ( curline )
    {
        curline->online = 0;
	oldlinestat = curline->con_stat;
	who = (struct client_t*)cltlist.first;
	while ( who )
	{
	    //20010820-new:
	    if ( line && (who->line != line) )
	    {
		who = (struct client_t*)who->next;
		continue;
	    }
	    oldstat = who->status;
    	    if ( who->status == CLT_ONLINE ) client_log_times(who);
	    if ( who->status != CLT_NOAUTH )
	    {
		who->status = CLT_OFFLINE;
    		who->started = 0;
    		if ( (who->status != oldstat) && (who->type == LCS_CLT_LCP3) )
        	    lcp3_cmd_queue(CMD3_CLIENTSTAT, who, NULL, 0);
	    }
    	    who = (struct client_t*)who->next;
	}
	if ( (iface_up(server->tcp_socket, curline) == 0)
    	    && (curline->con_stat == CST_UP_SERVER) )
	{
    	    call_con_dn(NULL, curline);
	}
// the following shouldn't be neccessary because set_line_status()
// has to be called for every new status:
//	if ( curline->con_stat != oldlinestat )
//	    lcp3_cmd_broadcast(CBR3_LINESTAT, NULL, 0, curline);
	if ( line ) break; // handle only specified line
	curline = (struct line_t*)curline->next;
    }
    return(0);
}

void set_line_status(struct line_t *line, int status)
{
    int old = line->con_stat;
    line->con_stat = status;
    if ( (line->con_stat == CST_UP_USER) && (!line->allow_manually) )
	line->con_stat = CST_UP_SERVER;
    if ( old != line->con_stat )
    {
	lines_check_locks(line);
	lcp3_cmd_broadcast(CBR3_LINESTAT, NULL, 0, line);
    }
}

void proc_con_action_line(struct line_t *line)
{
    static struct in_addr iaddr;
    int len;
     switch ( line->con_action.doing )
     {
         case DOING_NOTHING: break;
         case DOING_OPENING:
	     if (    ((len = iface_up(server->tcp_socket, line)) != 0)
	          && (time(NULL) - line->con_action.begin >= line->con_up_timeout) )
	     { // this is a timeout... cancel
segf_indicator = 17;
	         if ( line->script_esc[0] )
                 {
		     syslog(LOG_INFO, "couldn't estab. con; calling escape-script...");
	             if ( exec_dont_care(line->script_esc) )
		     { // failed to fork
		         syslog(LOG_ERR, "unable to call script_esc! Faking status...");
		         // should detect in the next
		         // loop that all clients went offline...
		         // so set the con to online. (avoid CST_UP_USER)
                         set_line_status(line, CST_CLOSE_ERROR);
		     }
		     else
		     {
                         set_line_status(line, CST_DOWN);
		     }
segf_indicator = 18;
		 }
		 else
		 {
		     syslog(LOG_INFO, "couldn't estab. con; no escape-script specified!");
		 }
segf_indicator = 19;
             if ( ((struct client_t*)line->con_action.who) && (line->id == 1) )
		     	cmd_q((struct client_t*)line->con_action.who, CMD_CON_ERROR, NULL, 0);
			 line->con_action.doing = DOING_NOTHING;
	         set_offline(line);
			 server->fast_loop = 0;
segf_indicator = 20;
	     }
	     else if ( !len )
	     { // con established
	         struct client_t *who = (struct client_t*)cltlist.first;
                 line->used_channels = 1;
                 line->con_up_time = time(NULL);
segf_indicator = 21;
                 init_throughput(line);
segf_indicator = 22;
		 while ( who )
		 {
		     if ( (who->status == CLT_DIALING) && (who->line == line) )
		     {
		         iaddr.s_addr = who->ip;
			 who->status = CLT_ONLINE;
			 who->started = time(NULL);
			 line->online++;
			 syslog(LOG_INFO, "%s:%d online", inet_ntoa(iaddr), ntohs(((struct client_t*)line->con_action.who)->port));
			 // will get clt status update with CBR3_LINESTAT
		         iaddr.s_addr = who->ip;
		         if ( line->client_online[0] )
		             exec_dont_care_param(line->client_online, inet_ntoa(iaddr));
		     }
		     who = (struct client_t*)who->next;
		 }
segf_indicator = 23;
                 set_line_status(line, CST_UP_SERVER);
                 if ( line->id == 1 )
		     cmd_broadcast(CBR_CON_UP, NULL, 0);
		 line->con_action.doing = DOING_NOTHING;
		 server->fast_loop = 0;
segf_indicator = 24;
	     }
	     break;
	 case DOING_CLOSING:
segf_indicator = 25;
	     if (    (!(len = iface_up(server->tcp_socket, line)))
	          && (time(NULL) - line->con_action.begin >= line->con_down_timeout) )
	     { // timeout, unable to close.
segf_indicator = 26;
	         if ( line->id == 1 )
                     cmd_broadcast(CBR_CLOSE_ERROR, NULL, 0);
                 set_line_status(line, CST_CLOSE_ERROR);
		 syslog(LOG_ERR, "CLOSE ERROR, unable to close connection!");
		 line->con_action.doing = DOING_NOTHING;
		 server->fast_loop = 0;
segf_indicator = 27;
		 // might call script_emerg (if exists...) here
		 // or try a system-shutdown:
		 // exec_dont_care(script_shutdown);
	     }
	     else if ( len == -1 )
	     { // ok, con down
segf_indicator = 28;
                 set_line_status(line, CST_DOWN);
		 set_offline(line);
segf_indicator = 29;
	         if ( server->logfile && line->mklog )
		 {
	             time_t now = time(NULL);
		     fprintf(server->logfile, "line: %s '%s' ", line->linename, cor_ctime(&line->con_up_time));
		     fprintf(server->logfile, "'%s' %d %u\n", cor_ctime(&now),
		             (int)(now-line->con_up_time), line->bytes_recvd);
		     fflush(server->logfile);
		 }
		 #if HAVE_MYSQL
            if ( line->mklog && sqlbe_available() )
			{
	             time_t now = time(NULL);
				char qbuf[160];
                 sprintf(qbuf, "INSERT INTO linesrv_con VALUES('%s', '%d', '%d', '%d', '%d')",
                         line->linename, (int)line->con_up_time,
                         (int)now, (int)(now-line->con_up_time),
                         line->bytes_recvd);
					sqlbe_query(qbuf);
            }
		 #endif
segf_indicator = 30;
	         line->used_channels = 1;
		 if ( line->id == 1 )
		     cmd_broadcast(CBR_CON_CLOSED, NULL, 0);
		 line->con_action.doing = DOING_NOTHING;
		 server->fast_loop = 0;
segf_indicator = 31;
	     }
	     break;
	 default:
	     line->con_action.doing = DOING_NOTHING;
	     server->fast_loop = 0;
     }
}

void proc_con_action()
{
    struct line_t *line = (struct line_t*)lstlines.first;
    while ( line )
    {
		segf_indicator = 15;
		proc_con_action_line(line);
		segf_indicator = 16;
		line = (struct line_t*)line->next;
    }
	segf_indicator = 1607;
}
