/*  This file is part of LingoTeach, the Language Teaching program *
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.  
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details.  
 *
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glib/gprintf.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "lingoteach-i18n.h"
#include "about.h"
#include "errors.h"
#include "learningpref.h"
#include "multichoice.h"
#include "onebyone.h"
#include "search.h"
#include "util.h"
#include "gui.h"


/* external global variables */
extern struct lpreferences genLprefs;

/* globally needed stuff */
GtkWidget     *win_main = NULL; /* main window */
GtkAccelGroup *accel_group;     /* accelerator for tooltips */
GtkTooltips   *tooltips;        /* the tooltips */
GtkWidget     *progbar;         /* program bar */
GtkWidget     *box_obo_module;  
GtkWidget     *box_mul_module;
GtkWidget     *box_sab_module;

/* static stuff */
static GtkWidget *statusbar;
static gchar *sab_columns[M_NUMBER] =
  {
    N_("Id"), N_("Type"), N_("Language"), N_("Translation")
  };


/* prototypes */
GtkWidget* get_main_menu   (void);
GtkWidget* create_multichoice (void);
GtkWidget* create_onebyone (void);
GtkWidget* create_searchandbrowse (void);
void on_doc_activate (void);

/*********************
 * private functions *
 *********************/

/*
 * main menu creation 
 */
GtkWidget*
get_main_menu (void)
{
  GtkWidget *menubar_main;
  GtkWidget *file;
  GtkWidget *file_menu;
  GtkWidget *pref;
  GtkWidget *sep;
  GtkWidget *help;
  GtkWidget *help_menu;
  GtkWidget *doc;
  GtkWidget *info;
  GtkWidget *quit;

  menubar_main = gtk_menu_bar_new ();  

  file = gtk_menu_item_new_with_mnemonic (_("_File"));
  gtk_container_add (GTK_CONTAINER (menubar_main), file);

  file_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (file), file_menu);

  pref = gtk_image_menu_item_new_from_stock (GTK_STOCK_PREFERENCES, 
					     accel_group);
  gtk_container_add (GTK_CONTAINER (file_menu), pref);

  sep = gtk_menu_item_new ();
  gtk_container_add (GTK_CONTAINER (file_menu), sep);
  gtk_widget_set_sensitive (sep, FALSE);

  quit = gtk_image_menu_item_new_from_stock (GTK_STOCK_QUIT, accel_group);
  gtk_container_add (GTK_CONTAINER (file_menu), quit);

  help = gtk_menu_item_new_with_mnemonic (_("_Help"));
  gtk_container_add (GTK_CONTAINER (menubar_main), help);

  help_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (help), help_menu);

  doc = gtk_image_menu_item_new_from_stock (GTK_STOCK_HELP, accel_group);
  gtk_container_add (GTK_CONTAINER (help_menu), doc);

  info = gtk_menu_item_new_with_mnemonic (_("_Info"));
  gtk_container_add (GTK_CONTAINER (help_menu), info);

  g_signal_connect (G_OBJECT (pref), "activate",
                    G_CALLBACK (on_preferences1_activate),
                    NULL);
  g_signal_connect (G_OBJECT (quit), "activate",
                    G_CALLBACK (lingoteach_quit),
                    NULL);
  g_signal_connect (G_OBJECT (info), "activate",
                    G_CALLBACK (on_info1_activate),
                    NULL);
  g_signal_connect (G_OBJECT (doc), "activate",
                    G_CALLBACK (on_doc_activate),
                    NULL);

  GLADE_HOOKUP_OBJECT (win_main, file, "menu_file");
  
  gtk_widget_show_all (menubar_main);
  return menubar_main;
}

/* 
 * activates the help
 */
void 
on_doc_activate (void)
{
  gchar  *cmd;
  GError *err;
  gchar  *file;

  if (genLprefs.browser != NULL)
    {
      file = g_build_filename (DOCDIR, "lingoteach-user-manual.html", NULL);
      cmd = g_malloc (strlen (genLprefs.browser) + strlen (file));
      g_sprintf (cmd, "%s %s", genLprefs.browser, file);
      if (!g_spawn_command_line_async (cmd, &err))
	err_error_cb (err->message);
      g_free (cmd);
      g_free (file);
    }
  return;
}

GtkWidget*
create_onebyone (void)
{
  GtkWidget *box_obo_main;
  GtkWidget *box_obo_teach;
  GtkWidget *btn_obo_next;

  box_obo_main = gtk_vbox_new (FALSE, 0);

  /* the one by one teaching box */
  box_obo_teach = create_obo_box ();
  gtk_box_pack_start (GTK_BOX (box_obo_main), box_obo_teach, TRUE, TRUE, 0);

  /* next button in one by one tab */
  btn_obo_next = gtk_button_new_with_mnemonic (_("_Next"));
  gtk_box_pack_start (GTK_BOX (box_obo_main), btn_obo_next, FALSE, FALSE, 0);

  g_signal_connect (G_OBJECT (btn_obo_next), "clicked",
                    G_CALLBACK (on_btn_obo_next_clicked),
                    NULL);

  return box_obo_main;
}

GtkWidget*
create_multichoice (void)
{
  GtkWidget *tbl_mul_main;
  GtkWidget *btn_mul_next;
  GtkWidget *box_mul_teach;

  /* multichoice tab */
  tbl_mul_main = gtk_table_new (2, 3, FALSE);

  /* button box containing the buttons */
  box_mul_teach = create_mul_box ();
  gtk_table_attach (GTK_TABLE (tbl_mul_main), box_mul_teach, 0, 2, 0, 1,
                    (GTK_EXPAND | GTK_FILL), (GTK_EXPAND | GTK_FILL), 0, 0);

  /* 'next' button for multichoice */
  btn_mul_next = gtk_button_new_with_mnemonic (_("_No Idea"));
  gtk_table_attach (GTK_TABLE (tbl_mul_main), btn_mul_next, 0, 1, 2, 3,
                     (GTK_EXPAND | GTK_FILL), 0, 0, 0);

  g_signal_connect (G_OBJECT (btn_mul_next), "clicked",
                    G_CALLBACK (on_btn_mul_next_clicked),
                    NULL);
  
  return tbl_mul_main;
}

GtkWidget*
create_searchandbrowse (void)
{
  GtkWidget *tbl_sab_main;
  GtkWidget *opt_sab_files;
  GtkWidget *win_sab_tree;
  GtkWidget *tree_sab_content;
  GtkWidget *box_sab_btn;
  GtkWidget *lbl_sab_filter;
  GtkWidget *opt_sab_filter;
  GtkWidget *txt_sab_filter;
  GtkWidget *btn_sab_refresh;
  GtkWidget *btn_sab_save;
  GtkWidget *btn_sab_saveas;
  GtkWidget *sep_sab_1;
  GtkWidget *sep_sab_2;
  GtkWidget *btn_sab_add;
  GtkWidget *btn_sab_remove;
  GtkWidget *btn_sab_edit;

  GtkTreeViewColumn *column;
  GtkCellRenderer   *renderer;
 
  tbl_sab_main = gtk_table_new (4, 5, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (tbl_sab_main), 2);

  /* lesson selection for treeview */
  opt_sab_files = gtk_option_menu_new ();
  gtk_table_attach (GTK_TABLE (tbl_sab_main), opt_sab_files, 0, 3, 0, 1,
		    GTK_FILL, (0), 4, 0);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt_sab_files), sab_make_menu ());

  /* layout separator */
  sep_sab_1 = gtk_hseparator_new ();
  gtk_table_attach (GTK_TABLE (tbl_sab_main), sep_sab_1, 0, 3, 1, 2,
                    GTK_FILL, GTK_FILL, 4, 4);
  
  /* scrolled window for treeview */
  win_sab_tree = gtk_scrolled_window_new (NULL, NULL);
  gtk_table_attach (GTK_TABLE (tbl_sab_main), win_sab_tree, 0, 3, 2, 3,
                     (GTK_EXPAND | GTK_FILL), (GTK_EXPAND | GTK_FILL), 4, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (win_sab_tree), 
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);

  /* the tree of the lesson files */
  tree_sab_content = 
    gtk_tree_view_new_with_model (GTK_TREE_MODEL (get_sab_list ()));
  /* gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_sab_content), TRUE); */
  /* gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tree_sab_content), TRUE); */
  gtk_container_add (GTK_CONTAINER (win_sab_tree), tree_sab_content);


  /* the content rows */
  renderer = gtk_cell_renderer_text_new ();
  column   = gtk_tree_view_column_new_with_attributes (sab_columns[M_ID],
						       renderer,
						       "text",
						       M_ID,
						       NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_sab_content), column);
  gtk_tree_view_set_expander_column (GTK_TREE_VIEW (tree_sab_content), column);
  gtk_tree_view_column_set_resizable (column, FALSE);
  gtk_tree_view_column_set_sort_column_id (column, M_ID);

  renderer = gtk_cell_renderer_text_new ();
  column   = gtk_tree_view_column_new_with_attributes (sab_columns[M_TYPE],
						       renderer,
						       "text",
						       M_TYPE,
						       NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_sab_content), column);
  gtk_tree_view_column_set_resizable (column, FALSE);
  gtk_tree_view_column_set_sort_column_id (column, M_TYPE);
  renderer = gtk_cell_renderer_text_new ();
  column   = gtk_tree_view_column_new_with_attributes (sab_columns[M_LANG],
						       renderer,
						       "text",
						       M_LANG,
						       NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_sab_content), column);
  gtk_tree_view_column_set_resizable (column, FALSE);
  gtk_tree_view_column_set_sort_column_id (column, M_LANG);

  renderer = gtk_cell_renderer_text_new ();
  column   = gtk_tree_view_column_new_with_attributes (sab_columns[M_TRANS],
						       renderer,
						       "text",
						       M_TRANS,
						       NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_sab_content), column);
  gtk_tree_view_column_set_resizable (column, FALSE);
  gtk_tree_view_column_set_sort_column_id (column, M_TRANS);

  /* filter label */
  lbl_sab_filter = gtk_label_new (_("Filter"));
  gtk_table_attach (GTK_TABLE (tbl_sab_main), lbl_sab_filter, 0, 1, 3, 4,
                    GTK_SHRINK, GTK_FILL, 4, 2);
  
  /* filter option */
  opt_sab_filter = gtk_option_menu_new ();
  gtk_table_attach (GTK_TABLE (tbl_sab_main), opt_sab_filter, 1, 2, 3, 4,
                    GTK_SHRINK, GTK_FILL, 4, 2);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt_sab_filter), 
			    util_get_menu ((lingchar **) sab_columns, 
					   M_NUMBER));
  
  /* filter entry */
  txt_sab_filter = gtk_entry_new ();
  gtk_table_attach (GTK_TABLE (tbl_sab_main), txt_sab_filter, 2, 3, 3, 4,
                    (GTK_EXPAND | GTK_FILL), GTK_FILL, 4, 2);
  
  /* button box for actions */
  box_sab_btn = gtk_vbox_new (TRUE, 0);
  gtk_table_attach (GTK_TABLE (tbl_sab_main), box_sab_btn, 3, 4, 0, 4,
                    GTK_FILL, (0), 10, 10);

  /* refresh button for treeview */
  btn_sab_refresh = gtk_button_new_from_stock (GTK_STOCK_REFRESH);
  gtk_box_pack_start (GTK_BOX (box_sab_btn), btn_sab_refresh, FALSE, FALSE, 0);
  gtk_tooltips_set_tip (tooltips, btn_sab_refresh, 
			_("Refreshes the list"), NULL);

  /* save button for lesson */
  btn_sab_save = gtk_button_new_from_stock (GTK_STOCK_SAVE);
  gtk_box_pack_start (GTK_BOX (box_sab_btn), btn_sab_save, FALSE, FALSE, 0);
  gtk_tooltips_set_tip (tooltips, btn_sab_save, _("Save the list"), NULL);

  /* save as button for lesson */
  btn_sab_saveas = gtk_button_new_from_stock (GTK_STOCK_SAVE_AS);
  gtk_box_pack_start (GTK_BOX (box_sab_btn), btn_sab_saveas, FALSE, FALSE, 0);
  gtk_tooltips_set_tip (tooltips, btn_sab_saveas, 
			_("Save the list in a new file"), NULL);

  /* layout */
  sep_sab_2 = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (box_sab_btn), sep_sab_2, TRUE, TRUE, 0);

  /* 'editor' button for addition of translations */
  btn_sab_add = gtk_button_new_from_stock (GTK_STOCK_ADD);
  gtk_box_pack_start (GTK_BOX (box_sab_btn), btn_sab_add, FALSE, FALSE, 0);
  gtk_tooltips_set_tip (tooltips, btn_sab_add, _("Add a new entry"), NULL);

  /* 'editor' button for deletion of translations */
  btn_sab_remove = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
  gtk_box_pack_start (GTK_BOX (box_sab_btn), btn_sab_remove, FALSE, FALSE, 0);
  gtk_tooltips_set_tip (tooltips, btn_sab_remove, 
			_("Remove an existing entry"), NULL);

  /* 'editor' button for editing a translation */
  btn_sab_edit = gtk_button_new_with_mnemonic (_("_Edit"));
  gtk_box_pack_start (GTK_BOX (box_sab_btn), btn_sab_edit, FALSE, FALSE, 0);
  gtk_tooltips_set_tip (tooltips, btn_sab_edit, 
			_("Edit a selected entry"), NULL);

  /*
    g_signal_connect (G_OBJECT (opt_sab_files), "changed",
                    G_CALLBACK (on_opt_sab_files_changed),
                    NULL);
  */
  /*
  g_signal_connect (G_OBJECT (tree_sab_content), "row-expanded",
		    G_CALLBACK (on_tree_sab_expand), 
		    NULL);
  */
  g_signal_connect (G_OBJECT (txt_sab_filter), "activate",
		    G_CALLBACK (on_txt_sab_filter_activated),
		    opt_sab_filter);
  
  g_signal_connect_swapped (G_OBJECT (btn_sab_refresh), "clicked",
			    G_CALLBACK (on_btn_sab_refresh_clicked),
			    opt_sab_files);
  g_signal_connect_swapped (G_OBJECT (btn_sab_save), "clicked",
			    G_CALLBACK (on_btn_sab_save_clicked),
			    opt_sab_files);
  g_signal_connect (G_OBJECT (btn_sab_saveas), "clicked",
                    G_CALLBACK (on_btn_sab_saveas_clicked),
                    NULL);
  g_signal_connect_swapped (G_OBJECT (btn_sab_add), "clicked",
			    G_CALLBACK (on_btn_sab_add_clicked),
			    tree_sab_content);
  g_signal_connect_swapped (G_OBJECT (btn_sab_remove), "clicked",
			    G_CALLBACK (on_btn_sab_remove_clicked),
			    tree_sab_content);
  g_signal_connect_swapped (G_OBJECT (btn_sab_edit), "clicked",
			    G_CALLBACK (on_btn_sab_edit_clicked),
			    tree_sab_content);

  GLADE_HOOKUP_OBJECT (win_main, box_sab_btn, "box_sab_btn");
  GLADE_HOOKUP_OBJECT (win_main, opt_sab_files, "opt_sab_files");
  GLADE_HOOKUP_OBJECT (win_main, tree_sab_content, "tree_sab_content");

  return tbl_sab_main;
}

/********************
 * public functions *
 ********************/

/*
 * creates the main window 
*/
GtkWidget*
create_win_main (void)
{
  GtkWidget *box_main;
  GtkWidget *menubar_main;
  GtkWidget *notebook_main;
  GtkWidget *btn_quit;

  GtkWidget *box_obo_main;
  GtkWidget *tbl_mul_main;
  GtkWidget *tbl_sab_main;

  GtkWidget *lbl_onebyone;
  GtkWidget *lbl_multichoice;
  GtkWidget *lbl_search;

  GtkWidget *btn_process_stop;
  GtkWidget *align_process;
  GtkWidget *btn_image;

  GtkWidget *pane_obo;
  GtkWidget *pane_mul;
  GtkWidget *pane_sab;

  tooltips = gtk_tooltips_new ();
  accel_group = gtk_accel_group_new ();

  /* main window */
  win_main = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (win_main), 
			"LingoTeach v" VERSION " - " INTVERSION );
  gtk_window_set_default_size (GTK_WINDOW (win_main), 600, 400);
  gtk_window_set_destroy_with_parent (GTK_WINDOW (win_main), TRUE);

  /* main box */
  box_main = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (win_main), box_main);

  /* menubar */
  menubar_main = get_main_menu ();
  gtk_box_pack_start (GTK_BOX (box_main), menubar_main, FALSE, FALSE, 0);

  /* main notebook */
  notebook_main = gtk_notebook_new ();
  gtk_box_pack_start (GTK_BOX (box_main), notebook_main, TRUE, TRUE, 0);

  /* quit button */
  btn_quit = gtk_button_new_from_stock (GTK_STOCK_QUIT);
  gtk_box_pack_start (GTK_BOX (box_main), btn_quit, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (btn_quit), 2);

  /* one by one tab */
  box_obo_main = create_onebyone ();
  pane_obo = gtk_vpaned_new ();
  gtk_container_set_border_width (GTK_CONTAINER (pane_obo), 2);  
  gtk_paned_pack1 (GTK_PANED (pane_obo), box_obo_main, TRUE, FALSE);
  gtk_container_add (GTK_CONTAINER (notebook_main), pane_obo);

  /* plugin box for onebyone */
  box_obo_module = gtk_hbox_new (TRUE, 2);
  gtk_paned_pack2 (GTK_PANED (pane_obo), box_obo_module, TRUE, FALSE);

  /* tab label one by one */
  lbl_onebyone = gtk_label_new (_("One by One"));
  gtk_notebook_set_tab_label 
    (GTK_NOTEBOOK (notebook_main), 
     gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook_main), 0), 
     lbl_onebyone);
  gtk_label_set_use_markup (GTK_LABEL (lbl_onebyone), TRUE);
  gtk_label_set_justify (GTK_LABEL (lbl_onebyone), GTK_JUSTIFY_LEFT);


  /* multichoice */
  tbl_mul_main = create_multichoice ();
  pane_mul = gtk_vpaned_new ();
  gtk_container_set_border_width (GTK_CONTAINER (pane_mul), 2);  
  gtk_paned_pack1 (GTK_PANED (pane_mul), tbl_mul_main, TRUE, FALSE);
  gtk_container_add (GTK_CONTAINER (notebook_main), pane_mul);

  /* plugin box for multichoice */
  box_mul_module = gtk_hbox_new (TRUE, 2);
  gtk_paned_pack2 (GTK_PANED (pane_mul), box_mul_module, TRUE, FALSE);

  /* tab label for multichoice */
  lbl_multichoice = gtk_label_new (_("Multichoice"));
  gtk_notebook_set_tab_label 
    (GTK_NOTEBOOK (notebook_main), 
     gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook_main), 1), 
     lbl_multichoice);
  gtk_label_set_justify (GTK_LABEL (lbl_multichoice), GTK_JUSTIFY_LEFT);


  /* search and browse tab */
  tbl_sab_main = create_searchandbrowse ();
  pane_sab = gtk_vpaned_new ();
  gtk_container_set_border_width (GTK_CONTAINER (pane_sab), 2);  
  gtk_paned_pack1 (GTK_PANED (pane_sab), tbl_sab_main, TRUE, FALSE);
  gtk_container_add (GTK_CONTAINER (notebook_main), pane_sab);

  /* plugin box for search and browse */
  box_sab_module = gtk_hbox_new (TRUE, 2);
  gtk_paned_pack2 (GTK_PANED (pane_sab), box_sab_module, TRUE, FALSE);

  /* label for search and browse tab */
  lbl_search = gtk_label_new (_("Search and Browse"));
  gtk_notebook_set_tab_label 
    (GTK_NOTEBOOK (notebook_main), 
     gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook_main), 2), 
     lbl_search);
  gtk_label_set_justify (GTK_LABEL (lbl_search), GTK_JUSTIFY_LEFT);
  
  /* status bar for the program */
  statusbar = gtk_statusbar_new ();
  gtk_box_pack_start (GTK_BOX (box_main), statusbar, FALSE, FALSE, 0);
  gtk_statusbar_set_has_resize_grip (GTK_STATUSBAR (statusbar), TRUE);

  /* button for processing interrupt */
  btn_process_stop = gtk_button_new ();
  gtk_widget_hide (btn_process_stop);
  gtk_box_pack_start (GTK_BOX (statusbar), btn_process_stop, FALSE, FALSE, 0);
  gtk_tooltips_set_tip (tooltips, btn_process_stop, 
			_("Stop processing of the current lesson"), NULL);

  /* alignment for the process button */
  align_process = gtk_alignment_new (0.5, 0.5, 0, 0);
  gtk_container_add (GTK_CONTAINER (btn_process_stop), align_process);
  
  /* image for the button */
  btn_image = gtk_image_new_from_stock (GTK_STOCK_STOP, GTK_ICON_SIZE_BUTTON);
  gtk_container_add (GTK_CONTAINER (align_process), btn_image);

  /* progress bar for the search and browse dump */
  progbar = gtk_progress_bar_new ();
  gtk_widget_show (progbar);
  gtk_box_pack_end (GTK_BOX (statusbar), progbar, FALSE, FALSE, 3);
  
  /* signal connectrors */
  g_signal_connect (G_OBJECT (win_main), "destroy", 
                    G_CALLBACK (lingoteach_quit), 
		    NULL);
  g_signal_connect (G_OBJECT (btn_quit), "clicked",
		    G_CALLBACK (lingoteach_quit), 
		    NULL);
  g_signal_connect (G_OBJECT (btn_process_stop), "clicked",
                    G_CALLBACK (on_btn_process_stop_clicked),
                    NULL);

  gtk_window_add_accel_group (GTK_WINDOW (win_main), accel_group);
  gui_new_status ("LingoTeach v" VERSION " - " INTVERSION);

  /* Store pointers to all necessary widgets, for use by lookup_widget(). */
  GLADE_HOOKUP_OBJECT (win_main, btn_quit, "btn_quit");
  GLADE_HOOKUP_OBJECT (win_main, btn_process_stop, "btn_process_stop");

  gtk_widget_show_all (win_main);
  gtk_widget_hide_all (btn_process_stop);

  on_btn_obo_next_clicked ();

  return win_main;
}

/*
 * status message handler 
 */
void
gui_new_status (gchar *status)
{
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, status);
  return;
}

/* 
 * updates several elements of the gui 
 */
void
gui_update (void)
{
  GtkWidget *widget;
  
  /* the search and browse menu */
  if (win_main == NULL)
    return;
  widget = util_lookup_widget (win_main, "opt_sab_files");
  gtk_option_menu_set_menu (GTK_OPTION_MENU (widget), sab_make_menu ());
  return;
}

/* 
 * locks all necessary stuff 
 */ 
void
gui_lock (void)
{
  GtkWidget *widget;

  widget = util_lookup_widget (win_main, "btn_quit");
  gtk_widget_set_sensitive (widget, FALSE);
  widget = util_lookup_widget (win_main, "opt_sab_files");
  gtk_widget_set_sensitive (widget, FALSE);
  widget = util_lookup_widget (win_main, "menu_file");
  gtk_widget_set_sensitive (widget, FALSE);
  widget = util_lookup_widget (win_main, "box_sab_btn");
  gtk_widget_set_sensitive (widget, FALSE);
  widget = util_lookup_widget (win_main, "btn_process_stop");
  gtk_widget_show_all (widget);
  return;
}

/* 
 * unlocks all necessary stuff 
 */ 
void
gui_unlock (void)
{
  GtkWidget *widget;

  widget = util_lookup_widget (win_main, "btn_quit");
  gtk_widget_set_sensitive (widget, TRUE);
  widget = util_lookup_widget (win_main, "opt_sab_files");
  gtk_widget_set_sensitive (widget, TRUE);
  widget = util_lookup_widget (win_main, "box_sab_btn");
  gtk_widget_set_sensitive (widget, TRUE);
  widget = util_lookup_widget (win_main, "menu_file");
  gtk_widget_set_sensitive (widget, TRUE);
  widget = util_lookup_widget (win_main, "btn_process_stop");
  gtk_widget_hide_all (widget);

  return;
}

/* 
 * quits the application 
 */
void 
lingoteach_quit (void)
{
  on_btn_process_stop_clicked (NULL);
  gtk_widget_destroy (win_main);
  gtk_main_quit ();
  return;
}
