/*  this file is part of LingoTeach, the Language Teaching program *
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.  
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details.  
 *
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <string.h>
#include <lingoteach.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "lingoteach-i18n.h"
#include "editor.h"
#include "errors.h"
#include "gui.h"
#include "learningpref.h"
#include "util.h"
#include "search.h"

/* external global variables */
extern lingConfig *settings;
extern lingLesson *lessons;
extern struct lpreferences genLprefs;
extern GtkTooltips  *tooltips;
extern GtkWidget    *progbar;
extern GtkWidget    *win_main;

GtkWidget    *sab_menu     = NULL; /* menu for the lesson selection */
GtkTreeStore *tree_content = NULL; /* internal list for the lesson tree dump */

static lingbool stop = FALSE;
static lingMeaning *cur_tree = NULL;

typedef lingMeaning* (*TreeCmpFunc) (lingMeaning *meaning, const gchar *text);

/* prototypes */
void sab_build_list (GtkOptionMenu *optionmenu);
void save_tree (char *file);
lingbool check_tree (GtkTreeModel *model);
lingMeaning* compare_id (lingMeaning *meaning, gchar *text);
lingMeaning* compare_type (lingMeaning *meaning, gchar *text);
lingMeaning* compare_language  (lingMeaning *meaning, gchar *text);
lingMeaning* compare_translation  (lingMeaning *meaning, gchar *text);
GtkTreeStore* create_tree (GtkTreeStore *tree, lingMeaning *meaning, 
			   const gchar *text, TreeCmpFunc cmp_func);

/*********************
 * private functions *
 *********************/

/*
 * creates the tree of meanings for the tree store 
 */
void
sab_build_list (GtkOptionMenu *optionmenu)
{
  gint          i = 0;
  gint          j;
  gint          tmpId;
  gfloat        step = 0;
  GtkWidget    *t_view  = util_lookup_widget (win_main, "tree_sab_content");
  lingMeaning  *meaning = NULL;
  lingLesson   *tmp = lessons;
  GtkTreeIter   iter;
  GtkTreeIter   iter2;
  gboolean      filled  = FALSE;
  GtkTreeStore *new     = gtk_tree_store_new (M_NUMBER,
					      G_TYPE_INT,
					      G_TYPE_STRING,
					      G_TYPE_STRING,
					      G_TYPE_STRING);

  /* which lesson */
  while (i < gtk_option_menu_get_history (optionmenu))
    {
      i++;
      tmp = tmp->next;
    }

  /* clear the tree */
  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (tree_content), &iter))
    {
      ling_meaning_free_meaning (cur_tree);
      gtk_tree_store_clear (tree_content);
    }
  
  meaning  = ling_lesson_create_tree (tmp);
  cur_tree = meaning;

  gui_lock (); /* lock all necessary stuff */

  gtk_progress_bar_set_text (GTK_PROGRESS_BAR (progbar), _("Processing..."));
  
  /* calculation for the progress bar */      
  j = ling_lesson_get_max_meaning (tmp);

  /* fill the list view */
  while (meaning) 
    {
      if (stop) /* maybe kill the complete event? */
	break;

      /* children */
      if (filled == TRUE)
	{
	  gtk_tree_model_get (GTK_TREE_MODEL (new), &iter,
			      M_ID, &tmpId,
			      -1);
	  if (tmpId == meaning->id)
	    {
	      gtk_tree_store_append (new, &iter2, &iter);
	      gtk_tree_store_set (new, &iter2,
				  M_ID, meaning->id,
				  M_TYPE, "",
				  M_LANG, meaning->language,
				  M_TRANS, meaning->translation,
			      -1);
	      meaning = meaning->next;
	      continue;
	    }
	}
      
      /* parent */
      gtk_tree_store_append (new, &iter, NULL);
      gtk_tree_store_set (new, &iter,
			  M_ID, meaning->id,
			  M_TYPE, meaning->type,
			  M_LANG, meaning->language,
			  M_TRANS, meaning->translation,
			  -1);
      
      gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progbar), step);
      
      while (g_main_context_iteration (NULL, FALSE)); /* non blocking */

      step   += 1 / (1.0 * j);
      meaning = meaning->next;
      filled  = TRUE;
    }

  gtk_tree_view_set_model (GTK_TREE_VIEW (t_view), GTK_TREE_MODEL (new));
  g_object_unref (new);

  gtk_widget_show_all (t_view);

  g_free (tree_content);
  tree_content = new;

  gtk_progress_bar_set_text (GTK_PROGRESS_BAR (progbar), _("Done"));
  gui_unlock (); /* unlock */
  stop = FALSE;

  return;
}


/*
 * saves the current tree into a xml file 
 */
void 
save_tree (char *file)
{
  if (!ling_meaning_save_meanings (cur_tree, file, settings))
    err_error_cb (_("Lesson could not be saved!"));
  return;
}

/*
 * basic checks for the tree operations
 */
lingbool 
check_tree (GtkTreeModel *model)
{
  /* does a model exist? */
  if (model == NULL || cur_tree == NULL)
    {
      err_error_cb (_("Please refresh the tree first!"));
      return FALSE;
    }
  return TRUE;
}

/*
 * compare function for id
 */
lingMeaning*
compare_id (lingMeaning *meaning, gchar *text)
{
  while (meaning)
    if (meaning->id != atoi (text))
      meaning = meaning->next;
    else
      return meaning;
  return NULL;
}

/*
 * compare function for type
 */
lingMeaning*
compare_type (lingMeaning *meaning, gchar *text)
{
  while (meaning)
    if (meaning->type == NULL || g_strrstr (meaning->type, text) == NULL)
      meaning = meaning->next;
    else
      return meaning;
  return NULL;

}

/*
 * compare function for language
 */
lingMeaning*
compare_language (lingMeaning *meaning, gchar *text)
{
  while (meaning)
    if (meaning->language == NULL 
	|| g_strrstr (meaning->language, text) == NULL)
      meaning = meaning->next;
    else
      return meaning;
  return NULL;
}

/*
 * compare function for translation
 */
lingMeaning*
compare_translation (lingMeaning *meaning, gchar *text)
{
  while (meaning)
    if (meaning->translation == NULL 
	  || g_strrstr (meaning->translation, text) == NULL)
      meaning = meaning->next;
    else
      return meaning;
  return NULL;
}

/*
 * creates a tree of the Meanings - internal usage without blocking
 * used for the filters - TreeCmpFunc can be NULL or a function, which
 * returns a lingMeaning, meaning and text are passed to that function,
 * if it is not NULL
 */
GtkTreeStore*
create_tree (GtkTreeStore *tree, lingMeaning *meaning, const gchar *text, 
	     TreeCmpFunc cmp_func)
{
  gint        tmpId;
  gboolean    filled = FALSE;
  GtkTreeIter iter;
  GtkTreeIter iter2;

  /* fill the list view */
  while (meaning)
    {
      /* is a comparision function given, which can be called? */
      if (cmp_func != NULL) 
	{
	  meaning = (*cmp_func)(meaning, text);
	  if (meaning == NULL)
	    return tree;
	}

      /* children */
      if (filled == TRUE)
	{
	  gtk_tree_model_get (GTK_TREE_MODEL (tree), &iter,
			      M_ID, &tmpId,
			      -1);
	  if (tmpId == meaning->id)
	    {
	      gtk_tree_store_append (tree, &iter2, &iter);
	      gtk_tree_store_set (tree, &iter2,
				  M_ID, meaning->id,
				  M_TYPE, "",
				  M_LANG, meaning->language,
				  M_TRANS, meaning->translation,
				  -1);
	      meaning = meaning->next;
	      continue;
	    }
	}
      /* parent */
      gtk_tree_store_append (tree, &iter, NULL);
      gtk_tree_store_set (tree, &iter,
			  M_ID, meaning->id,
			  M_TYPE, meaning->type,
			  M_LANG, meaning->language,
			  M_TRANS, meaning->translation,
			  -1);
      
      meaning = meaning->next;
      filled  = TRUE;
    }
  return tree;
}


/********************
 * public functions *
 ********************/


void 
on_tree_sab_expand (GtkTreeView *tree, GtkTreeIter *iter)
{
  return;
}

void 
on_opt_sab_files_changed (GtkOptionMenu *optionmenu)
{
  sab_build_list (optionmenu);
  return;
}

/*
 * refreshes (or initially loads) the meaning tree of the currently selected
 * lesson
 */
void 
on_btn_sab_refresh_clicked (GtkOptionMenu *optionmenu)
{
  if (lessons != NULL)
    sab_build_list (optionmenu);
  else
    err_error_cb (_("Please load a lesson first!"));
  return;
}

/*
 * saves the meaning tree into the selected lesson
 */
void 
on_btn_sab_save_clicked (GtkOptionMenu *optionmenu)
{
  lingLesson *tmp = lessons;
  char *file;
  gint i = 0;

  if (lessons == NULL)
    {
      err_error_cb (_("Please load a lesson first!"));
      return;
    }

  while (i < gtk_option_menu_get_history (optionmenu))
    {
      i++;
      tmp = tmp->next;
    }

  /* save the lesson */
  file = ling_lesson_return_path (tmp);
  save_tree (file);

  /* reload after saving */
  lessons = ling_lesson_remove_lesson (lessons, tmp);
  lessons = ling_lesson_add_lesson (lessons, file, settings);

  return;
}

/* 
 * saves the meaning tree into a user defined file
 */
void 
on_btn_sab_saveas_clicked (void)
{
  GtkWidget *file_select = util_create_dlg_fileselect (_("Save as"), 
						       (gpointer) save_tree);
  gtk_widget_show (file_select);
  return;
}

/*
 * adds a new meaning entry into the list of existing ones
 */
void 
on_btn_sab_add_clicked (GtkTreeView *tree)
{
  GtkTreeIter       iter;
  GtkTreeIter       iter2;
  gint              id;
  gchar            *lang;
  GtkTreeModel     *model = gtk_tree_view_get_model (tree);
  GtkTreeSelection *selection = gtk_tree_view_get_selection (tree);

  lingMeaning *new = NULL;
  lingMeaning *prev;
  lingMeaning *meaning = cur_tree;

  if (!check_tree (model))
    return;

  /* get the selected one (if any) */
  if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
    {
      gtk_tree_model_get_iter_first (model, &iter);
      while (gtk_tree_model_iter_next (model, &iter))
	iter2 = iter;
      iter = iter2;
    }

  gtk_tree_model_get (model, &iter,
		      M_ID, &id,
		      M_LANG, &lang,
		      -1);

  while (meaning->id != id)   /* search the meaning list */
    {
      prev = meaning;
      meaning = prev->next;
    }
  while (g_strncasecmp (meaning->language, lang, 
			strlen (meaning->language)) != 0)
    {
      prev = meaning;
      meaning = prev->next;
    }
  g_free (lang);
  
  if ((new = add_meaning (meaning)) != NULL)  /* now lets add it */
    {
      /* do we have to add a child to a meaning entry? */
      if (gtk_tree_store_iter_depth (GTK_TREE_STORE (model), &iter) > 0)
	{
	  new->id = meaning->id;
	  
	  gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				       &iter2, NULL, &iter);
	  gtk_tree_store_set (GTK_TREE_STORE (model), &iter2,
			      M_ID, new->id,
			      M_LANG, new->language,
			      M_TRANS, new->translation,
			      -1);
	  cur_tree = ling_meaning_insert_after_meaning (cur_tree, meaning, 
							new);
	}
      else /* no, we do not have to, so lets add a completely new one */
	{
	  meaning = cur_tree;
	  while (meaning->next != NULL)
	    meaning = meaning->next;
	  new->id = meaning->id + 1;

	  gtk_tree_store_append (GTK_TREE_STORE (model), &iter2, NULL);
	  gtk_tree_store_set (GTK_TREE_STORE (model), &iter2,
			      M_ID, new->id,
			      M_LANG, new->language,
			      M_TRANS, new->translation,
			      M_TYPE, new->type,
			      -1);
	  cur_tree = ling_meaning_add_meaning (cur_tree, new);
	}
    }
  return;
}

void 
on_btn_sab_remove_clicked (GtkTreeView *tree)
{
  gchar            *lang;
  gint              id;
  GtkTreeIter       iter;
  GtkTreeModel     *model = gtk_tree_view_get_model (tree);
  GtkTreeSelection *selection = gtk_tree_view_get_selection (tree);

  lingMeaning *prev;
  lingMeaning *meaning = cur_tree;

  if (!check_tree (model))
    return;
  
  /* get the selected one (if any) */
  if (gtk_tree_selection_get_selected (selection, NULL, &iter))
    {
      gtk_tree_model_get (model, &iter,
			  M_ID, &id,
			  M_LANG, &lang,
			  -1);

      while (meaning->id != id)   /* search the meaning list */
	{
	  prev = meaning;
	  meaning = prev->next;
	}
      while (g_strncasecmp (meaning->language, lang, 
			    strlen (meaning->language)) != 0)
	{
	  prev = meaning;
	  meaning = prev->next;
	}
      g_free (lang);
      
      /* found the correct one, now let's remove it */
      if (gtk_tree_model_iter_has_child (model, &iter))
	{
	  gtk_tree_store_remove (GTK_TREE_STORE (model), &iter);
	  while (meaning->id == id)
	    {
	      prev = meaning;
	      if (prev->next == NULL)
		{
		  cur_tree = ling_meaning_free_meaning_1 (cur_tree, prev);
		  break;
		}
	      meaning = prev->next;
	      cur_tree = ling_meaning_free_meaning_1 (cur_tree, prev);
	    }
	}
      else 
	{
	  gtk_tree_store_remove (GTK_TREE_STORE (model), &iter);
	  cur_tree = ling_meaning_free_meaning_1 (cur_tree, meaning);
	}
    }
  else
    err_error_cb (_("Please select a row first!"));
  
  return;
}

/*
 * edits a selected meaning entry
 */
void
on_btn_sab_edit_clicked (GtkTreeView *tree)
{
  
  GtkTreeIter       iter;
  GtkTreeModel     *model = gtk_tree_view_get_model (tree);
  GtkTreeSelection *selection = gtk_tree_view_get_selection (tree);
  lingMeaning      *meaning = cur_tree;
  lingMeaning      *prev;
  lingMeaning      *new = NULL;
  gchar            *lang;
  gint              id;

  if (!check_tree (model))
    return;

  /* get the currently selected one */
  if (gtk_tree_selection_get_selected (selection, NULL, &iter))
    {
      gtk_tree_model_get (model, &iter,
			  M_ID, &id,
			  M_LANG, &lang,
			  -1);
      
      while (meaning->id != id)   /* search the meaning list */
	{
	  prev = meaning;
	  meaning = prev->next;
	}
      while (g_strncasecmp (meaning->language, lang, 
			strlen (meaning->language)) != 0)
	{
	  prev = meaning;
	  meaning = prev->next;
	}
      g_free (lang);

      /* found, now lets edit it */
      if ((new = edit_meaning (meaning)) != NULL)
	{
	  /* modify the list... */
	  cur_tree = ling_meaning_modify_meaning (cur_tree, meaning->id, new);

	  /* ... and show the result in the tree */
	  gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
			      M_LANG, new->language,
			      M_TRANS, new->translation,
			      M_TYPE, new->type,
			      -1);
	}
    }
  else
    err_error_cb (_("Please select a row first!"));
  
  return;
}

/*
 * interrupts the _creation_ of the tree (not of the list!)
 */
void
on_btn_process_stop_clicked (GtkButton *button)
{
  stop = TRUE;
  if (button != NULL)
    gtk_widget_hide_all (GTK_WIDGET (button));
  return;
}

/*
 * creates the lesson menu for the search and browse option list
 */
GtkWidget*
sab_make_menu (void)
{
  GtkWidget  *mitem;
  lingLesson *tmp = lessons;

  if (sab_menu != NULL)
    gtk_widget_destroy (sab_menu);

  sab_menu = gtk_menu_new ();

  if (tmp != NULL)
    do 
      {
	mitem = gtk_menu_item_new_with_label (tmp->type);
	gtk_widget_show (mitem);
	gtk_menu_append (GTK_MENU (sab_menu), mitem);
      }
    while ((tmp = tmp->next));

  gtk_menu_set_active (GTK_MENU (sab_menu), 0);
  return sab_menu;
}

/*
 * creates the tree store for the user interface 
 */
GtkTreeStore*
get_sab_list (void)
{
  if (tree_content == NULL)
    tree_content = gtk_tree_store_new (M_NUMBER,
				       G_TYPE_INT,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING);
  return tree_content;
}

void
on_txt_sab_filter_activated (GtkEntry *entry, GtkOptionMenu *option)
{
  lingMeaning *meaning = cur_tree;
  gint         type = gtk_option_menu_get_history (option);
  const gchar *text = gtk_entry_get_text (entry);
  GtkTreeView *tree = (GtkTreeView *) util_lookup_widget (win_main,
							   "tree_sab_content");
  GtkTreeStore *new = gtk_tree_store_new (M_NUMBER,
					  G_TYPE_INT,
					  G_TYPE_STRING,
					  G_TYPE_STRING,
					  G_TYPE_STRING);
  gtk_tree_view_set_model (tree, NULL);

  /* no filter set */
  if (strcmp (text, "") == 0)
    {
      new = create_tree (new, meaning, text, NULL);
      gtk_tree_view_set_model (tree, GTK_TREE_MODEL (new));
      g_object_unref (new);
      tree_content = new;
      return;
    }

  /* a filter is set, create the tree with the proper comparision function */
  switch (type)
    {
    case M_ID:
      new = create_tree (new, meaning, text, (TreeCmpFunc) compare_id);
      break;
    case M_TYPE:
      new = create_tree (new, meaning, text, (TreeCmpFunc) compare_type);
      break;
    case M_LANG:
      new = create_tree (new, meaning, text, (TreeCmpFunc) compare_language);
      break;
    case M_TRANS:
      new = create_tree (new, meaning, text, (TreeCmpFunc) compare_translation);
      break;
    default:
      g_free (new);
      err_error_cb (_("The column could not be found!"));
      return;
      break;
    }
  gtk_tree_view_set_model (tree, GTK_TREE_MODEL (new));
  g_object_unref (new);
  tree_content = new;
  
  return;
}
