/*
  The osipua library is a library based on oSIP that implements CallLeg and User Agent
  level.
  Copyright (C) 2001  Simon MORLAT simon.morlat@free.fr
  											Aymeric MOIZARD jack@atosc.org
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "resolver.h"
#include "osipmanager.h"
#include "osipua.h"
#include <osip/smsg.h>
#include <osip/smsgtypes.h>
#include <osip/port.h>
#include <osip/fifo.h>


#ifndef __VXWORKS_OS__
#ifdef WIN32
#include <winsock.h>
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#endif
#endif

#ifdef __VXWORKS_OS__
#include <vxWorks.h>
#include <sockLib.h>
#include <inetLib.h>
#include <hostLib.h>
#include <resolvLib.h>
#endif

#ifdef INET6  
#include <sys/types.h>  
#include <sys/socket.h>  
#include <netdb.h>  
#endif

struct hostent *
resolv (char *name)
{
	unsigned long int addr;
	struct hostent *hp = NULL;

	if ((int) (addr = inet_addr (name)) == -1)
	{
		/* The address is not of the form a.b.c.d */
#ifndef __VXWORKS_OS__
		hp = gethostbyname (name);
		/*    fprintf(stdout,"Address is of type name %li\n",addr); */
#endif
#ifdef __VXWORKS_OS__
		char *pbuf = (char *) smalloc (513 * sizeof (char));
		hp = resolvGetHostByName (name, pbuf, 512);
		/* we should check errno */
#endif
	}
	else
	{
#ifndef __VXWORKS_OS__
		hp = gethostbyaddr ((char *) &addr, sizeof (addr), AF_INET);
		/* fprintf(stdout,"Address is of type a.c.d.e %li\n",addr); */
#endif
#ifdef __VXWORKS_OS__
		char *pbuf = (char *) smalloc (513 * sizeof (char));
		/* hp = resolvGetHostByAddr((char *)&addr ,pbuf ,512); */
#endif
	}
	if (hp == NULL)
	{
		osip_trace (OSIP_WARNING,
			    ("error: host information for %s not found\n",
			     name));
		return NULL;
	}

	return hp;
}

void *
resolver_thread (void *p_manager)
{
	async_resolv_t *transevent;
	transaction_t *trans;
	sipevent_t *sipevent;
	sip_t *sipmsg;
	char **name;
	OsipManager *manager = (OsipManager *) p_manager;
	struct hostent *hi;

	osip_trace (OSIP_INFO1,
		    ("Entering name resolution thread.\n"));
	while (manager->resolv_run_cond)
	{
		transevent =
			(async_resolv_t *) fifo_get (manager->resolv_fifo);
		osip_trace (OSIP_INFO1,
			    ("Doing asynchronous name resolution.\n"));

		if (transevent != NULL)
		{
			/* differentiate acks for invite and other requests: DIRTY HACK */
			if (transevent->ackmsg != NULL)
			{
#ifdef INET6  
			        struct addrinfo hints, *res = NULL;  
				char tempname[NI_MAXHOST];  
				name = &transevent->dest;  
				memset(&hints, 0, sizeof(hints));  
				hints.ai_family = PF_UNSPEC;  

				if (getaddrinfo(*name, NULL, &hints, &res) != 0)  
#else
				hi = resolv (*name);
				if (hi != NULL)
#endif
				{
					sfree (*name);
#ifdef INET6
					getnameinfo(res->ai_addr, res->ai_addrlen,  
						    tempname, sizeof(tempname), NULL, 0,  
						    NI_NUMERICHOST);  
					*name = sgetcopy (tempname);  
					freeaddrinfo(res);  
#else
					*name = sgetcopy (inet_ntoa
							  (((struct in_addr *)
							    hi->h_addr)[0]));
#endif
					udp_send (NULL,transevent->ackmsg,*name,transevent->port,def_manager->send_sock);
#ifdef OSIP_RETRANSMISSIONS
					osip_start_ack_retransmissions(manager->config,transevent->dialog,transevent->ackmsg,*name,transevent->port,def_manager->send_sock);
#endif
				}
			}
			else
			{
				trans = transevent->transaction;
				sipevent = transevent->event;
				sipmsg = sipevent->sip;
				sfree (transevent);

				if (MSG_IS_REQUEST (sipmsg))
				{
#ifdef INET6  
				  struct addrinfo hints, *res = NULL;  
				  char namebuf[BUFSIZ];  
				  int error;
#endif
				  int dest_port;

					switch (trans->ctx_type)
					{
					case ICT:
						name = &trans->ict_context->
							destination;
						dest_port = trans->ict_context->
						        port;
						  break;
					case NICT:
						name = &trans->nict_context->
							destination;
						dest_port = trans->nict_context->
						        port;
						break;
					default:
						osip_trace (OSIP_BUG,
							    ("Very strange transaction, aborting"));
						exit (-1);
					}

#ifdef INET6  
					memset(&hints, 0, sizeof(hints));  
					hints.ai_family = PF_UNSPEC;
					error = getaddrinfo(*name, NULL, &hints, &res);
					if (error == 0)
#else
					hi = resolv (*name);

					if (hi != NULL)
#endif
					{
						sfree (*name);
#ifdef INET6
						getnameinfo(res->ai_addr, res->ai_addrlen,  
							    namebuf, sizeof(namebuf), NULL, 0,  
							    NI_NUMERICHOST);
						freeaddrinfo(res);
						*name = sgetcopy (namebuf);  
#else
						*name = sgetcopy (inet_ntoa
								  (((struct
								     in_addr
								     *) hi->
								    h_addr)
								   [0]));
#endif
						ua_transaction_set_destination(trans, sgetcopy (*name), dest_port);
						ua_transaction_execute (trans,sipevent);
					}
					else
					  {
						osip_trace (OSIP_INFO1,
							    ("Could not resolve %s.\n",
							     *name));
					  }
						
				}
				else
				{
					/* we must resolv the via field */
#ifdef INET6  
				        struct addrinfo hints, *res = NULL;  
					char tempname[NI_MAXHOST];  
#endif
					via_t *via;
					via = (via_t *) list_get (sipmsg->
								  vias, 0);
					if (via == NULL)
						osip_trace (OSIP_ERROR,
							    ("Could not get the first via header.\n"));
					name = &via->host;
#ifdef INET6  
					memset(&hints, 0, sizeof(hints));  
					hints.ai_family = PF_UNSPEC;  
					if (getaddrinfo(*name, NULL, &hints, &res) != 0)  
#else
					hi = resolv (*name);
					if (hi != NULL)
#endif
					{
						sfree (*name);
#ifdef INET6  
						getnameinfo(res->ai_addr, res->ai_addrlen,  
							    tempname, sizeof(tempname), NULL, 0,  
							    NI_NUMERICHOST);  
						freeaddrinfo(res);  
						via->host = sgetcopy (tempname);  
#else
						via->host =
							sgetcopy (inet_ntoa
								  (((struct
								     in_addr
								     *) hi->
								    h_addr)
								   [0]));
#endif
						ua_transaction_execute(trans,sipevent);
					}
					else
						osip_trace (OSIP_INFO1,
							    ("Could not resolve %s.\n",
							     *name));
				}
			}
		}
		else
			osip_trace (OSIP_ERROR,
				    ("warning: Null event !\n"));
	}
	osip_trace(OSIP_INFO1,("osipua resolver thread is exiting."));
	return NULL;
}
