/*
  The oRTP library is an RTP (Realtime Transport Protocol - rfc1889) stack.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/***************************************************************************
 *            jitterctl.c
 *
 *  Mon Nov  8 11:53:21 2004
 *  Copyright  2004  Simon MORLAT
 *  Email simon.morlat@linphone.org
 ****************************************************************************/

#include <ortp/rtpsession.h>
#include <ortp/payloadtype.h>
#include <math.h>
#include <stdlib.h>

#define JC_BETA 0.001
#define JC_GAMMA (JC_BETA)


void jitter_control_start(JitterControl *ctl){
	ctl->count=0;
	ctl->delta=0;
	ctl->safe_delay=0;
	ctl->adapt_jitt_comp_time=ctl->jitt_comp_time;
	ctl->low_jitt_thres=ctl->jitt_comp_time;
	ctl->adaptive=TRUE;
}

void jitter_control_set_payload(JitterControl *ctl, PayloadType *pt){
	ctl->corrective_step=(160 * 8000 )/pt->clock_rate; /* This formula got to me after some beers */
}

void jitter_control_stop(JitterControl *ctl)
{
	ctl->adaptive=FALSE;
}

void jitter_control_dump_stats(JitterControl *ctl){
	g_log("oRTP-stats",G_LOG_LEVEL_MESSAGE,"JitterControl:\n\tdelta=%g,safe_delay=%g,count=%i",
			ctl->delta,ctl->safe_delay, ctl->count);
}


/*
 The algorithm computes two values:
	delta: an average of difference between the expected and the socket-received timestamp
	safe_delay: an average of the absolute value of the difference between socket-received timestamp and delta
	delta is used to make clock-slide detection and correction.
	safe_dela is added to the initial jitt_comp_time value. It compensates bursty packets arrival (packets
	not arriving at regular interval ).
*/
void jitter_control_new_packet(JitterControl *ctl, guint32 packet_ts, guint32 cur_str_ts, gint32 * slide, gint32 *safe_delay){
	int diff=packet_ts - cur_str_ts;
	float gap;
	int d;
	//printf("diff=%g\n",diff);
	*slide=0;
	*safe_delay=0;
	ctl->count++;
	ctl->delta= (ctl->delta*(1-JC_BETA)) + ((float)diff*JC_BETA);
	gap=fabs((float)diff - ctl->delta);
	ctl->safe_delay=(ctl->safe_delay*(1-JC_GAMMA)) + (gap*JC_GAMMA);
	d=diff-ctl->olddiff;
	ctl->inter_jitter=ctl->inter_jitter+ (( (float)abs(d) - ctl->inter_jitter)*(1/16.0));
	ctl->olddiff=diff;
	if (ctl->adaptive){
		if (ctl->count%50==0) {
			//jitter_control_dump_stats(ctl);
			if ((2*ctl->safe_delay)>ctl->adapt_jitt_comp_time){
				ctl->low_jitt_thres=ctl->adapt_jitt_comp_time-(ctl->corrective_step/2);
				ctl->adapt_jitt_comp_time+=ctl->corrective_step;
				g_warning("Changing jitter compensation at %i",ctl->adapt_jitt_comp_time);
			}else if ( (ctl->adapt_jitt_comp_time>ctl->jitt_comp_time) && 
				((2*ctl->safe_delay) <= ctl->low_jitt_thres)){
				ctl->adapt_jitt_comp_time-=ctl->corrective_step;
				g_warning("Changing jitter compensation at %i",ctl->adapt_jitt_comp_time);
			}
			if (ctl->delta>ctl->corrective_step){ 
				*slide=ctl->corrective_step;
			}else if (ctl->delta<0){
				*slide=-ctl->corrective_step;
			}
		}
	}
	*safe_delay=ctl->adapt_jitt_comp_time;
	return ;
}


/**
 *rtp_session_set_jitter_compensation:
 *@session: a RtpSession
 *@milisec: the time interval in milisec to be jitter compensed.
 *
 * Sets the time interval for which packet are buffered instead of being delivered to the 
 * application.
 **/
void
rtp_session_set_jitter_compensation (RtpSession * session, gint milisec)
{
	PayloadType *payload = rtp_profile_get_payload (session->profile,
							session->
							payload_type);
	if (payload==NULL){
		g_warning("rtp_session_set_jitter_compensation: cannot set because the payload type is unknown");
		return;
	}
	session->rtp.jittctl.jitt_comp = milisec;
	/* convert in timestamp unit: */
	session->rtp.jittctl.jitt_comp_time =
		(gint) (((double) milisec / 1000.0) * (payload->clock_rate));
	session->rtp.jittctl.adapt_jitt_comp_time=session->rtp.jittctl.jitt_comp_time;
}

void rtp_session_enable_adaptive_jitter_compensation(RtpSession *session, gboolean val){
	if (val) {
		PayloadType *pt;
		jitter_control_start(&session->rtp.jittctl);
		pt=rtp_profile_get_payload(session->profile,session->payload_type);
		if (pt!=NULL)
			jitter_control_set_payload(&session->rtp.jittctl,pt);
	}
	else jitter_control_stop(&session->rtp.jittctl);
}

gboolean rtp_session_adaptive_jitter_compensation_enabled(RtpSession *session){
	return session->rtp.jittctl.adaptive;
}
