/*
 * PQAcompile
 * Steven James <pyro@linuxlabs.com>
 * Linux Labs http://www.linuxlabs.com
 *
 * This is Free software, licensed under the GNU Public License V2.
 *
 * This software compiles html and pnm images into PalmVII PQA.
 * This should run on any system with a port of gcc and many other 
 *	systems as well.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>

//#include <libhtmlparse.h>

#include <libpisock/pi-source.h>
#include <libpisock/pi-socket.h>
#include <libpisock/pi-dlp.h>
#include <libpisock/pi-file.h>


#define TYPE_HTML 4
#define TYPE_IMAGE 5

typedef struct {
	char *tagname;
	char *params;
	unsigned char token_value;
	} PQA_token;

PQA_token token[] = {
	{"b","",0x05},		// bold
};
	

typedef struct {
	char *filename;
	char type;
}	needed_record;

int next_record=0;
needed_record	record_stack[1000];

unsigned char RecordBuffer[65536];		/* oversized to be sure! */

#define PILOT_TIME_DELTA (unsigned)(2082844800)

typedef struct {
	int		urlOffset;
	short	urlLength;
	int		dataOffset;
	short	dataLength;
	char	contentType;
	char	compressionType;
	int		uncompressedSize;
	char	flags;
	char 	reserved;
}	__attribute__ ((packed)) pqa_header;

static char * icon[] = {
"32 32 2 1",
".      c #000000000000",
"*      c #111111111111",
"................................",
"................................",
"..........*...*.................",
"..........*..***................",
"...*......*.*****...............",
"....*.....******.*..............",
".....*..*******.***.............",
"......**.....*.*****.......*....",
"......*.......*******.....*.....",
".....*.........*******...*......",
".....*.........********.........",
".*****.........*....****.**.*...",
".....*.........********.........",
".....*.........*******...*......",
"......*.......*******.....*.....",
"......**.....*.*****.......*....",
".....*..*******.***.............",
"....*.....******.*..............",
"...*......*.*****...............",
"..........*..***................",
"..........*...*.................",
"................................",
"................................",
"................................",
"................................",
"................................",
"................................",
"................................",
"................................",
"................................",
"................................",
"................................"
};

static char * small_icon[] = {
"15 9 2 1",
".      c #000000000000",
"*      c #111111111111",
"........*.......",
"........*.......",
"..*.....*....*..",
"...*..*****.*...",
"....**....**....",
"....*......*....",
"...*........*...",
"...*........*...",
"****........****"
};


//int commentstart () { return 0; }
//int commentend () { return 0; }

//int comments (char *comment) { return 0; }                                       
//int textstart () { return 0; }
//int textend () { return 0; }

//int texts (char *comment) {
//	/* encode text here! */
 //       printf("Text detected: %s\n", comment);
//        return 0;
//}

//int endtag (char *tag) {
//	/* just emit an end tag */
//	*(pos++) = '\0';
//        return 0;
//}

//int decltag (char *tag, struct ArgvTable *args, int numargs) {
//        int i;

//	/* much crunching of args */
//        printf("Declaration tag %s detected\n", tag);
//        if (args) {
//                printf("\twith %d args:\n", numargs);
//                for(i=0; i < numargs; i++) {
//                        printf("\t\t%s=%s\n", args[i].arg, args[i].val);
//                }
//        }
//        return 0;
//}

//int starttag (char *tag, struct ArgvTable *args, int numargs) {
//	int i;
//
//	/* much crunching of args */
//
//        printf("Start tag %s detected\n", tag);
//	if (args) {
//		printf("\twith %d args:\n", numargs);
//		for(i=0; i < numargs; i++) {
//			printf("\t\t%s=%s\n", args[i].arg, args[i].val);
//		}
//	}
//        return 0;
//}
//                               
//int setup_libhtmlparse() 
//{
//        commentCallBack = comments;
//	commentStartCallBack = commentstart;
//	commentEndCallBack = commentend;
//	textCallBack = texts;
//	textStartCallBack = textstart;
//	textEndCallBack = textend;
//	endCallBack = endtag;
//	startCallBack = starttag;
//	declCallBack = decltag;
//
//	return(0);
//}

static time_t
pilot_time_to_unix_time (unsigned long raw_time)
{
  return (time_t)(raw_time - PILOT_TIME_DELTA);
}

static unsigned long
unix_time_to_pilot_time (time_t t)
{
  return (unsigned long)((unsigned long)t + PILOT_TIME_DELTA);
}


int load_bitmap(char *image[], FILE *in) 
{
	char linebuf[1024];
	char *pos;
	char *tmp;
	int line =0;

	
	fgets(linebuf,sizeof(linebuf),in);	/* toss the first line, it only applies to #included images! */

	while(linebuf[0] != '}') {
	
		if(!fgets(linebuf,sizeof(linebuf),in))
			return(-2);

		pos = linebuf;
	
		if(*(pos++) != '\"') 
			return(-1);
		
		tmp = strchr(pos,'\"');
		if(!tmp)
			return(-1);
	
		if(tmp - pos ==1)
			continue;		/* null line! */

		*tmp = '\0';
		image[line++] = strdup(linebuf);
	}

	return(0);
}

int conv_icon_ppm(char *filename, unsigned char *outbuf, char compress, int depth)
{
	char *pos, *outpos, *compbuf;
	unsigned char *token;
	int x, y, bitpix, tmp;
	int bit, size,bytes;
	int byte_per_line;
	int i, j, k, line_in=0;
	char *string;
	unsigned char flag;
	FILE *in;
	char linebuf[1024];
	int skip;

	memset(linebuf,0,sizeof(linebuf));	/* init linebuf */

	if((in = fopen(filename,"r")) == NULL) {
		pos = outbuf;
		*(pos++) = 0;
		*(pos++) = 0;
		return(2);
	}


	if(!fgets(linebuf,sizeof(linebuf),in)) {	/* the identifier */
		pos = outbuf;
		*(pos++) = 0;
		*(pos++) = 0;
		return(2);
	}

	if(linebuf[1] == '6')
		skip=3;
	else 
		skip=1;

	linebuf[0] = '#';

	while(linebuf[0] == '#')
		fgets(linebuf,sizeof(linebuf),in);


	x = atoi(linebuf);
	y = atoi((strchr(linebuf,' ')+1));

	fgets(linebuf,sizeof(linebuf),in);	/* skip the depth */
	line_in= depth+1;

	bytes = depth;
	for(bitpix=0; bytes != 0x01; bytes>>=1, bitpix++);	/* compute bits per pixel [ log2(depth) ] */

	byte_per_line = x/8 * bitpix;
	if(x%8)
		byte_per_line++;

	byte_per_line += byte_per_line%2;	

	pos = outbuf;

	size = byte_per_line *y;
	if(size & 0x01)		/* if odd */
		size+=1;		/* word align it */

	size /=2;	/* words */
	size +=6;	/* allow for header */
	size +=2;

	if(depth == 2) {		// it's an ICON, stick in a length
		*(pos++) = (size>>8) & 0xff;	/* high bits */
		*(pos++) = size & 0xff;			/* low bits */
	}	

	*(pos++) = (x>>8) & 0xff;	/* high bits */
	*(pos++) = x & 0xff;			/* low bits */

	*(pos++) = (y>>8) & 0xff;	/* high bits */
	*(pos++) = y & 0xff;			/* low bits */

	*(pos++) = (byte_per_line>>8) & 0xff;	/* high bits */
	*(pos++) = byte_per_line & 0xff;			/* low bits */

	if(compress)			/* flags */
		*(pos++) = 0x80;
	else		
		*(pos++) = 0x00;

	*(pos++) = 0x00;	/* flags low byte */

	*(pos++) = bitpix;	/* bits per pixel */

	*(pos++) = 0x01;	/* version */

	*(pos++) = 0x00;	/* ? word */
	*(pos++) = 0x00;

	*(pos++) = 0x00;	/* transparent index */
	*(pos++) = 0x00;	/* compression type */

	*(pos++) = 0x00;	/* word, color table entries??!? */
	*(pos++) = 0x00;

	outpos = pos;

	if(compress) {
		compbuf = malloc(y * byte_per_line);

		pos = compbuf;
	}
		
	for(i=0;i<y;i++) {
		memset(linebuf,0,sizeof(linebuf));
		fread(linebuf,skip,x,in);
		token = linebuf;
		for(bytes = 0; bytes <byte_per_line; bytes++) {
			for(bit=0; bit<8; bit+= bitpix) {
				*pos <<= bitpix;
	
				if( (depth ==2) && (*token == 0x00))	/* if ! *token, just finish shifting */
					*pos |= 0x01;

				if( (depth ==4) ) {	/* if ! *token, just finish shifting */

					/* quick and dirty quantization */
					if(*token >0xc0)	
						*pos |= 0x00;
					else if(*token > 0x80)
						*pos |= 0x01;
					else if(*token >0x40)
						*pos |= 0x02;
					else
						*pos |= 0x03;
				}
				token += skip;

			}
			pos++;
			*pos = 0;
		}	/* line done, fetch the next */
	}

	fclose(in);

	if(compress) {	/* now, we compress from the compbuf into the outbuf */
		pos = compbuf;

		/* encode the first line explicitly */
		for(i=0;i<y;i++)	{
			for(j=0;j< (byte_per_line +7)/8 ; j++)	{
				if(j<byte_per_line/8)
					tmp = 8;
				else
					tmp = byte_per_line % 8;

				for(k=0;k<tmp;k++) {
					if(*pos != *(pos-byte_per_line))
						flag |= 0x01;
					flag <<=1;
				}
				flag <<= (8 - tmp);
				*(outpos++) = flag;

				for(k=0; k<tmp; k++) {
					if(flag & 0x80)
						*(outpos++) = *(pos++);
					else
						pos++;

					flag <<=1;
				}
			}
		}

	}	// end if compress

	return((size*2) +2);	/* add 2 for the size word */
}


int conv_icon(char *buffer[], unsigned char *outbuf, char compress)
{
	char *pos, *outpos, *compbuf;
	char *token;
	int x, y, depth, bitpix, tmp;
	int bit, size,bytes;
	int byte_per_line;
	int i, j, k, line_in=0;
	char *string;
	unsigned char flag;

	string = strdup(buffer[0]);

	token = strtok(string," \"");
	x = atoi(token);

	token = strtok(NULL," ");
	y = atoi(token);

	token = strtok(NULL," ");
	depth = atoi(token);

	line_in= depth+1;

	bytes = depth;
	for(bitpix=0; bytes != 0x01; bytes>>=1, bitpix++);	/* compute bits per pixel [ log2(depth) ] */

	byte_per_line = x/8;
	if(x%8)
		byte_per_line++;

	pos = outbuf;

	size = byte_per_line *y;
	if(size & 0x01)		/* if odd */
		size+=1;		/* word align it */

	size /=2;	/* words */
	size +=6;	/* allow for header */
	size +=2;

	*(pos++) = (size>>8) & 0xff;	/* high bits */
	*(pos++) = size & 0xff;			/* low bits */

	*(pos++) = (x>>8) & 0xff;	/* high bits */
	*(pos++) = x & 0xff;			/* low bits */

	*(pos++) = (y>>8) & 0xff;	/* high bits */
	*(pos++) = y & 0xff;			/* low bits */

	*(pos++) = (byte_per_line>>8) & 0xff;	/* high bits */
	*(pos++) = byte_per_line & 0xff;			/* low bits */

	if(compress)			/* flags */
		*(pos++) = 0x80;
	else		
		*(pos++) = 0x00;

	*(pos++) = 0x00;	/* flags low byte */

	*(pos++) = bitpix;	/* bits per pixel */

	*(pos++) = 0x01;	/* version */

	*(pos++) = 0x00;	/* ? word */
	*(pos++) = 0x00;

	*(pos++) = 0x00;	/* transparent index */
	*(pos++) = 0x00;	/* compression type */

	*(pos++) = 0x00;	/* word, color table entries??!? */
	*(pos++) = 0x00;

	outpos = pos;

	if(compress) {
		compbuf = malloc(y * byte_per_line);

		pos = compbuf;
	}
		
	for(i=0;i<y;i++) {
		token = buffer[line_in++];
		for(bytes = 0; bytes <byte_per_line; bytes++) {
			for(bit=0; bit<8; bit+= bitpix) {
				*pos <<= bitpix;
	
				if( (depth ==2) && (*token) && (*(token++) == '*'))	/* if ! *token, just finish shifting */
					*pos |= 0x01;

				if( (depth ==4) && (*token) )	/* if ! *token, just finish shifting */
					switch(*(token++)) {
						default:
						case ' ':
						case '.':
							break;
				
						case '!':
							*pos |= 0x01;
							break;

						case 'X':
							*pos |= 0x02;
							break;

						case '*':
							*pos |= 0x03;
							break;
					}

			}
			pos++;
		}	/* line done, fetch the next */
	}

	if(compress) {	/* now, we compress from the compbuf into the outbuf */
		pos = compbuf;

		/* encode the first line explicitly */
		for(i=0;i<y;i++)	{
			for(j=0;j< (byte_per_line +7)/8 ; j++)	{
				if(j<byte_per_line/8)
					tmp = 8;
				else
					tmp = byte_per_line % 8;

				for(k=0;k<tmp;k++) {
					if(*pos != *(pos-byte_per_line))
						flag |= 0x01;
					flag <<=1;
				}
				flag <<= (8 - tmp);
				*(outpos++) = flag;

				for(k=0; k<tmp; k++) {
					if(flag & 0x80)
						*(outpos++) = *(pos++);
					else
						pos++;

					flag <<=1;
				}
			}
		}

	}	// end if compress

	return((size*2) +2);	/* add 2 for the size word */
}



int conv_icon_globbed(char *buffer, unsigned char *outbuf)
{
	char *pos;
	char *token;
	int x,y,depth,bit,size,bytes;
	int byte_per_line;
	int i;

	pos = buffer+1;

	token = strtok(pos," \"");
	x = atoi(token);

	token = strtok(NULL," \"");
	y = atoi(token);

	strtok(NULL,"\"");	// throw away the rest!
	strtok(NULL,"\"");	// get to start of bit buffer!

	byte_per_line = x/8;
	if(x%8)
		byte_per_line++;

	pos = outbuf;

	size = byte_per_line *y;
	if(size & 0x01)		/* if odd */
		size+=1;		/* word align it */

	size /=2;	/* words */

	*(pos++) = (size>>8) & 0xff;	/* high bits */
	*(pos++) = size & 0xff;			/* low bits */

	for(i=0;i<y;i++) {
		token = strtok(NULL,"\";");	/* get the line! */
		if(*token == ';')	/* this would mean we are short several lines! */
			return(-1);

		for(bytes = 0; bytes <byte_per_line; bytes++) {
			for(bit=0; bit<8; bit++) {
				*pos <<=1;
	
				if((*token) && (*(token++) == '*'))	/* if ! *token, just finish shifting */
					*pos |= 0x01;
			}
			pos++;
		}	/* line done, fetch the next */

		strtok(NULL,"\"}");
	}

	return((size*2) +2);	/* add 2 for the size word */
}

typedef struct attribute_t {
	char *tag;
	char *value;
	struct attribute_t *next;
}	attribute;


attribute *getattrs(char *tag)
{
	attribute *ret = NULL;
	attribute *cur = NULL;

	while(*tag) {
		for(;*tag && (*tag == ' ' || *tag == '\t'); tag++);	/* skip white space */
		if(! *tag)
			return(ret);

		if(!cur) {
			cur =  malloc(sizeof(attribute));
			ret = cur;
		} else {
			cur->next = malloc(sizeof(attribute));
			cur = cur->next;
		}

		cur->next = NULL;

		cur->tag = tag;

		for(;*tag && *tag != ' ' && *tag != '=';tag++);

		if(!*tag)
			return(ret);


		if(*tag == '=') {
			*(tag++) = '\0';	// terminate the attribute
			for(;*tag && (*tag == ' ' || *tag == '\"');tag++);
			cur->value = tag;

			for(;*tag && *tag != ' ' && *tag != '\"';tag++);
			if(!*tag)
				return(ret);

			*(tag++) = '\0';	// terminate the value
		} else
			*(tag++) = '\0';	// terminate the attribute

	}

	return(ret);
}

void destroy_attr(attribute *a)
{
	attribute *next,*cur=a;

	
	while(cur) {
		next = cur->next;
		free(cur);
		cur = next;
	}
}

	
		
attribute *findattr(const char *tag,attribute *attr)
{
	attribute *a = NULL;

	for(a=attr;a && strcasecmp(a->tag,tag);a = a->next);
	return(a);
}
	
	

int compile_tag(char **outpos, char *tag)	
{
	attribute *attrs,*attr;
	char *pos = *outpos;
	char flag1;
	char flag2;
	char *aux, *aux2;
	short tmp;
	int i;
	
	if(*tag == '/') {	/* a closing tag */
		*(pos++) = '\0';
		*outpos = pos;		
		return(0);
	}


	for(aux = tag;*aux && *aux != ' ' && *aux != '\t';aux++);

	if(*aux) {
		*(aux++) = '\0';	/* terminate the tag */
		attrs = getattrs(aux);
	} else
		attrs = NULL;

	if(!strcasecmp(tag,"B"))	{ /* bold */
		*(pos++) = 0x01;
		*(pos++) = 0x05;

	} else if(!strcasecmp(tag,"I"))	{ /* italic */
		*(pos++) = 0x01;
		*(pos++) = 0x06;

	} else if(!strcasecmp(tag,"P")) { /* paragraph */
		*(pos++) = 0x01;
		*(pos++) = 0x08;

		flag1 = 0;
		if(attr = findattr("ALIGN",attrs)) {
			switch(attr->value[0]) {

				case 'l':
					flag1 = 0x00;
					break;
				case 'c':
					flag1 = 0x01;
					break;
				case 'r':
					flag1 = 0x02;
					break;
					
			}
		}
		*(pos++) = flag1;

	} else if(!strcasecmp(tag,"HR")) { /* horizontal rule */
		*(pos++) = 0x01;
		*(pos++) = 0x09;

		flag1 = 0;
		if(findattr("NOSHADE",attrs))
			flag1 |= 0x08;

		if(attr = findattr("ALIGN",attrs)) {
			switch(attr->value[0]) {

				case 'l':
					break;
				case 'c':
					flag1 |= 0x02;
					break;
				case 'r':
					flag1 |= 0x04;
					break;
					
			}
		}

	
		aux = pos++;
		*aux = flag1;

		if(attr = findattr("HEIGHT",attrs))
			tmp = atoi(attr->value);
		else
			tmp = 0;


		if(tmp || (attr = findattr("WIDTH",attrs))) {
			*aux |=1;

			*(pos++) = (tmp>>8) & 0xff;
			*(pos++) = tmp & 0xff;

			if(attr) {
				aux2 = strchr(attr->value,'%');
				tmp = atoi(attr->value);

				if(aux2) {
					*aux |= 0x10;
					*(pos++) = tmp & 0xff;
				} else {
					*(pos++) = (tmp>>8) & 0xff;
					*(pos++) = tmp & 0xff;
				}
			}
		}

					
		
	} else if(!strcasecmp(tag,"H1")) { /* heading 1 */
		*(pos++) = 0x01;
		*(pos++) = 0x0a;

		if(attr = findattr("ALIGN",attrs)) {
			*(pos++) = 0x01;
			
			switch(attr->value[0]) {

				case 'l':
					*(pos++) = 0x00;
					break;
				case 'c':
					*(pos++) = 0x01;
					break;
				case 'r':
					*(pos++) = 0x02;
					break;
					
			}
		} else
			*(pos++) = 0x00;
			


	} else if(!strcasecmp(tag,"BR")) { /* line break */
		*(pos++) = '\r';
	} else if(!strcasecmp(tag,"A")) {	/* link! */
		*(pos++) = 0x01;
		*(pos++) = 0x11;
		
		flag1 = flag2 = 0;
		if(attr = findattr("HREF",attrs)) {
			if(!strncmp(attr->value,"http://",7) || !strncmp(attr->value,"https://",8)) {		/* it's a global, compile it as a full link */
				flag2 |= 0x20;
	
				if(attr->value[4] == 'S')	/* secure */
					flag2 |= 0x10;
	
				*(pos++) = flag1;
				*(pos++) = flag2;
				memcpy(pos,attr->value,strlen(attr->value)+1);	/* copy the href + terminator */
				pos += strlen(attr->value) +1;
			} else {
				/* it's local, add it to the 'to compile' list and encode it as an index */
				/* first, see if it's already included! */

				aux = strchr(attr->value,'#');
				if(aux) 
					*(aux++) ='\0';
					

				for(i=0;i<next_record && strcasecmp(attr->value,record_stack[i].filename);i++);
				
				if(i == next_record) {
					record_stack[next_record].filename = strdup(attr->value);
					record_stack[next_record].type = TYPE_HTML;
					tmp = next_record++;
				} else
					tmp = i;

				flag2 |= 0x04;

				if(aux)
					flag2 |= 0x08;

				*(pos++) = flag1;
				*(pos++) = flag2;

				*(pos++) = (tmp>>8) & 0xff;
				*(pos++) = tmp & 0xff;
				if(aux) {							// if a fragment, append it
					memcpy(pos,aux,strlen(aux)+1);
					pos += strlen(aux)+1;
				}
			}

		}

	} else if(!strcasecmp(tag,"FORM")) {	/* form! */
		*(pos++) = 0x01;
		*(pos++) = 0x21;

		*(pos++) = 0x00;	/* the form index */
		*(pos++) = 0x00;

		attr = findattr("METHOD",attrs);

		if(!attr || strcasecmp(attr->value,"post"))
			flag2 =0;
		else
			flag2=1;

		attr = findattr("ACTION",attrs);

		flag1 = 1;

		if(strncmp(attr->value,"http",4))
			flag1 |= 0x08;				/* mailto or file or something */
		else
			if(attr->value[4] == 's')
				flag1 |= 0x02;


		*(pos++) = flag1;
		*(pos++) = flag2;

		*(pos++) = 0x00;	/* never an enc type */
		memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
		pos += strlen(attr->value) +1;

	} else if(!strcasecmp(tag,"INPUT")) {	/* input! */

		attr = findattr("TYPE",attrs);
		*(pos++) = 0x01;

		switch(attr->value[0]) {
			case 'p':		// password
				*(pos++) = 0x23;
				break;

			case 'c':	// checkbox
				*(pos++) = 0x25;
				break;

			case 's':	// submit
				*(pos++) = 0x26;
				break;
				
				
			case 't':
				if(attr->value[4]) {	// textarea
					attr->value[0] = 'a';
				} else {
					*(pos++) = 0x22;	// text
				}
				break;

			case 'h':					// hidden
				*(pos++) = 0x28;
				break;

			default:
				pos--;					// unknown, skip it for now!
				return(-1);
		}
			

		switch(attr->value[0]) {
			case 'p':
			case 'h':
			case 't':

				if(attr->value[0] != 'h') {
					attr = findattr("SIZE",attrs);
					if(attr)
						tmp = atoi(attr->value);
					else
						tmp = 10;
		
					*(pos++) = (tmp>>8) & 0xff;
					*(pos++) = tmp & 0xff;
					
					attr = findattr("MAXLENGTH",attrs);
					if(attr)
						tmp = atoi(attr->value);
					else
						tmp = 0;
		
					*(pos++) = (tmp>>8) & 0xff;
					*(pos++) = tmp & 0xff;
				}	

				flag1 = 0x01;
	
				if(attr = findattr("VALUE",attrs)) {
					flag1 |= 0x02;
					aux = attr->value;
				} else
					aux = NULL;
	
				*(pos++) = flag1;
				attr = findattr("NAME",attrs);
				
				memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
				pos += strlen(attr->value) +1;
	
				if(aux) {		/* the default value (if any) */
					memcpy(pos,aux,strlen(aux) +1);	/* include the null */
					pos += strlen(aux) +1;
				}

				break;

			case 'c':			// checkbox
				flag1=0;
				aux = 0;
				
				if(attr = findattr("VALUE",attrs)) {
					flag1 |= 0x02;
					aux = attr->value;
				}

				if(findattr("TEXT",attrs))
					flag1 |= 0x08;
				if(findattr("CHECKED",attrs))
					flag1 |= 0x04;

				if(attr = findattr("NAME",attrs))
					flag1 |= 0x01;

				*(pos++) = flag1;

				if(attr) {
					memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
					pos += strlen(attr->value) +1;
				}

				if(aux) {		/* the default value (if any) */
					memcpy(pos,aux,strlen(aux) +1);	/* include the null */
					pos += strlen(aux) +1;
				}

				if(flag1 & 0x08) {
					attr = findattr("TEXT",attrs);
					memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
					pos += strlen(attr->value) +1;
				}

				break;

			case 's':			// submit
				flag1=0;
				aux = 0;
				
				if(attr = findattr("VALUE",attrs)) {
					flag1 |= 0x02;
					aux = attr->value;
				}

				if(attr = findattr("NAME",attrs))
					flag1 |= 0x01;

				*(pos++) = flag1;

				if(attr) {
					memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
					pos += strlen(attr->value) +1;
				}

				if(aux) {		/* the default value (if any) */
					memcpy(pos,aux,strlen(aux) +1);	/* include the null */
					pos += strlen(aux) +1;
				}

				break;

			default:
				break;

		}


	} else if(!strcasecmp(tag,"SELECT")) {	/* select! */
		flag1=0;
		if(findattr("MULTIPLE",attrs))
			flag1 |= 0x02;

		if(attr = findattr("SIZE",attrs)) {
			tmp = atoi(attr->value) -1;
		} else
			tmp = 0;

		if(attr = findattr("NAME",attrs)) {
			flag1 |=0x01;
		}

		*(pos++) = 0x01;
		*(pos++) = 0x2a;
		*(pos++) = flag1;

		*(pos++) = (tmp>>8) & 0xff;
		*(pos++) = tmp & 0xff;

		if(attr) {
			memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
			pos += strlen(attr->value) +1;
		}

	} else if(!strcasecmp(tag,"OPTION")) {	/* option! */
		*(pos++) = 0x01;

		if(!attrs)	/* no attributes, use the simple tag */
			*(pos++) = 0x2b;
		else {
			/* custom option! */
			*(pos++) = 0x2c;

			flag1 = 0;
			if(findattr("SELECTED",attrs))
				flag1 |= 0x01;
	
			if(attr = findattr("VALUE",attrs))
				flag1 |= 0x02;
	
			*(pos++) = flag1;
			if(attr) {
				memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
				pos += strlen(attr->value) +1;
			}
		}

	} else if(!strcasecmp(tag,"TABLE")) {	/* table! */
		*(pos++) = 0x01;
		*(pos++) = 0x2f;

		aux = pos++;		/* placemarker for flags. */
		*aux = 0;

		if(attr = findattr("ALIGN",attrs)) {
			*aux |= 0x01;
			switch(attr->value[0]) {

				case 'l':
					*(pos++) = 0x00;
					break;
				case 'c':
					*(pos++) = 0x01;
					break;
				case 'r':
					*(pos++) = 0x02;
					break;
					
			}
		}

		if(attr = findattr("WIDTH",attrs)) {
			*aux |= 0x02;
			tmp = atoi(attr->value);
			*(pos++) = (tmp>>8) & 0xff;
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("BORDER",attrs)) {
			*aux |= 0x04;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("CELLSPACING",attrs)) {
			*aux |= 0x08;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("CELLPADDING",attrs)) {
			*aux |= 0x10;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

					
	} else if(!strcasecmp(tag,"TR")) {	/* table row! */
		*(pos++) = 0x01;
		*(pos++) = 0x30;

		flag1 = 0;

		if(attr = findattr("ALIGN",attrs)) {
			aux = attr->value;
			flag1 =1;
		}

		if(attr = findattr("VALIGN",attrs))
			flag1 =1;

		*(pos++) = flag1;

		if(flag1) {
			switch(aux[0]) {

				case 'l':
					*(pos++) = 0x00;
					break;
				case 'c':
					*(pos++) = 0x01;
					break;
				case 'r':
					*(pos++) = 0x02;
					break;
					
			}
			
			switch(attr->value[0]) {

				case 't':
					*(pos++) = 0x00;
					break;
				case 'c':
					*(pos++) = 0x01;
					break;
				case 'b':
					*(pos++) = 0x02;
					break;
					
			}
		}
			
	} else if( (!strcasecmp(tag,"TD")) || (!strcasecmp(tag,"TH"))) {	/* table datum or header! */
		*(pos++) = 0x01;

		if(tag[1] == 'D')
			*(pos++) = 0x32;
		else
			*(pos++) = 0x33;

		aux = pos++;

		*aux = 0;

		if(attr = findattr("ALIGN",attrs)) {
			*aux |= 0x01;
			switch(attr->value[0]) {

				case 'l':
					*(pos++) = 0x00;
					break;
				case 'c':
					*(pos++) = 0x01;
					break;
				case 'r':
					*(pos++) = 0x02;
					break;
					
			}
		}
		
		if(attr = findattr("VALIGN",attrs)) {
			*aux |= 0x02;
			switch(attr->value[0]) {

				case 't':
					*(pos++) = 0x00;
					break;
				case 'c':
					*(pos++) = 0x01;
					break;
				case 'b':
					*(pos++) = 0x02;
					break;
					
			}
		}

		if(attr = findattr("COLSPAN",attrs)) {
			*aux |= 0x04;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("ROWSPAN",attrs)) {
			*aux |= 0x08;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("HEIGHT",attrs)) {
			*aux |= 0x10;
			tmp = atoi(attr->value);
			*(pos++) = (tmp>>8) & 0xff;
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("WIDTH",attrs)) {
			*aux |= 0x20;
			tmp = atoi(attr->value);
			*(pos++) = (tmp>>8) & 0xff;
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("NOWRAP",attrs))
			*aux |= 0x40;
		
	} else if(!strcasecmp(tag,"FONT")) {	/* font */
		if(attr = findattr("SIZE",attrs)) {
			*(pos++) = 0x01;
			*(pos++) = 0x04;
	
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		} else
			fprintf(stderr,"ERROR: font tag must have size=\n");
	

	} else if(!strcasecmp(tag,"TABLECAPTION")) {	/* table caption! (not standard html!) */
		*(pos++) = 0x01;
		*(pos++) = 0x31;


		if(findattr("TOP",attrs))
			*(pos++) = 0x01;
		else
			*(pos++) =0;

	} else if(!strcasecmp(tag,"IMG")) {	/* image! */
		*(pos++) = 0x01;
		*(pos++) = 0x34;

		aux = pos++;

		*aux = 0x80;

		/* grab the image */
		/* first, see if it's already included! */
	
		attr = findattr("SRC",attrs);

		for(i=0;i<next_record && strcasecmp(attr->value,record_stack[i].filename);i++);
				
		if(i == next_record) {
			record_stack[next_record].filename = strdup(attr->value);
			record_stack[next_record].type = TYPE_IMAGE;
			tmp = next_record++;
		} else
			tmp = i;

		*(pos++) = (tmp>>8) & 0xff;
		*(pos++) = tmp & 0xff;

		if(attr = findattr("ALT",attrs)) {
			*aux |= 0x40;
			memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
			pos += strlen(attr->value) +1;
		}

		if(attr = findattr("SRC",attrs)) {
			*aux |= 0x20;
			memcpy(pos,attr->value,strlen(attr->value) +1);	/* include the null */
			pos += strlen(attr->value) +1;
		}

		if(attr = findattr("VSPACE",attrs)) {
			*aux |= 0x10;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("HSPACE",attrs)) {
			*aux |= 0x08;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("BORDER",attrs)) {
			*aux |= 0x04;
			tmp = atoi(attr->value);
			*(pos++) = tmp & 0xff;
		}

		if(attr = findattr("ALIGN",attrs)) {
			*aux |= 0x02;
			switch(attr->value[0]) {

				case 'l':
					*(pos++) = 0x00;
					break;
				case 'r':
					*(pos++) = 0x01;
					break;
				case 't':
					*(pos++) = 0x02;
					break;
				case 'm':
					*(pos++) = 0x03;
					break;
				case 'b':
					*(pos++) = 0x04;
					break;
					
			}
		}


	} else if(!strcasecmp(tag,"META")) {	/* one of several meta tags! */

		attr = findattr("NAME",attrs);

		if(!strcasecmp(attr->value,"localicon")) {
			attr = findattr("CONTENT",attrs);

			for(i=0;i<next_record && strcasecmp(attr->value,record_stack[i].filename);i++);
					
			if(i == next_record) {
				record_stack[next_record].filename = strdup(attr->value);
				record_stack[next_record].type = TYPE_IMAGE;
			}

		}
	} else {
		printf("unknown tag <%s>\n",tag);
		return(-1);
	}

	destroy_attr(attrs);
	*outpos = pos;		
	
	return(0);
}

void UpCase(char *buffer,int length)
{
	int i;
	char bInQuot = 0;
	char bInTag =0;
	char bInValue=0;


	for(i=0;i<length;i++)
		switch(buffer[i]) {
			case '<':
				bInTag=1;
				bInQuot =0;
				break;

			case '>':
				bInTag=0;
				break;

			case '=':
				if(bInTag)
					bInValue=1;
				break;

			case '\'':
			case '\"':
				bInQuot = !bInQuot;
				break;

			case ' ':
				bInValue=0;
				break;

			default:
				if(bInTag && !bInQuot && !bInValue)
					buffer[i] = toupper(buffer[i]);
				break;
		}
}

int compile_image(struct pi_file *pf, needed_record *rec, int uid) 
{
	int total_size;
	int length;
	pqa_header *pqa;	
	char *pos;
	int ret;

	
	length = strlen(rec->filename);
	length += length%2;		// make an even word boundary

	pqa = (pqa_header *) RecordBuffer;
	pqa->urlOffset = htonl(sizeof(pqa_header));
	pqa->urlLength = htons(length);
	pqa->dataOffset = htonl(sizeof(pqa_header) + length);
	pqa->dataLength = htons(12);
	pqa->contentType = rec->type;
	pqa->compressionType =0;
	pqa->uncompressedSize = htonl(12);
	pqa->flags =0;
	pqa->reserved =0;

	pos = RecordBuffer + sizeof(pqa_header);

	memcpy(pos,rec->filename,length);		// it is allways safe! rounding adds at most 1 byte which is the null string terminator
	pos += length;

	/* Now, conv_icon into the record_buffer */

	length= conv_icon_ppm(rec->filename,pos, 0, 4);
	pqa->uncompressedSize = htonl(length);
	pqa->dataLength = htons(length);

	total_size = sizeof(pqa_header) + ntohs(pqa->urlLength) + ntohs(pqa->dataLength);

	printf("record size = %u\n",total_size);

	ret = pi_file_append_record(pf,RecordBuffer,total_size,0,0,uid);

	printf("pi_file_append_record = %d\n",ret);
	return(0);
}
	
int compile_html(struct pi_file *pf, needed_record *rec, int uid) 
{
	int total_size;
	int length;
	pqa_header *pqa;	
	char *pos;
	int ret;
	FILE *in;
	char filebuf[65532];
	char *filepos = filebuf;
	int file_length;
	char *auxpos;
	char inpre=0;

	
	if(!(in = fopen(rec->filename,"r"))) {
		return(-1);
	}

	file_length = fread(filebuf,1,sizeof(filebuf),in);
	fclose(in);

	UpCase(filebuf,file_length);


	length = strlen(rec->filename);
	length += length%2;		// make an even word boundary

	pqa = (pqa_header *) RecordBuffer;
	pqa->urlOffset = htonl(sizeof(pqa_header));
	pqa->urlLength = htons(length);
	pqa->dataOffset = htonl(sizeof(pqa_header) + length);
	pqa->dataLength = htons(12);
	pqa->contentType = rec->type;
	pqa->compressionType =0;
	pqa->uncompressedSize = htonl(12);
	pqa->flags =0;
	pqa->reserved =0;

	pos = RecordBuffer + sizeof(pqa_header);

	memcpy(pos,rec->filename,length);		// it is allways safe! rounding adds at most 1 byte which is the null string terminator
	pos += length;

	// now, find the title

	filepos = strstr(filebuf,"<TITLE>");

	if(!filepos) {	// no title!
		length = strlen(rec->filename);
		length += length%2;

		memcpy(pos,rec->filename,length);
		pos += length;
		auxpos = filebuf;
		*(pos++) = '\0';				// null terminate the title
	} else {
		// title found
		filepos = strchr(filepos,'>') +1;	// find the close of the title tag
		auxpos = strchr(filepos,'<');		// the start of the title close tag
		*auxpos = '\0';

		length = strlen(filepos);
		length += length%2;

		memcpy(pos,filepos,length);
		pos += length;
		*(pos++) = '\0';				// null terminate the title
	}

	auxpos +=1;	
	filepos = strstr(auxpos,"<BODY>");
	filepos = strchr(filepos,'>') +1;

	// filepos is now in the body of the html!

	while(strncmp(filepos,"</BODY>",7)) {

// temporary!
		switch(*filepos) {
			case '\n':
				if(inpre) { 
					*(pos++) = '\r';
				 	filepos++;
				} else {
					while(*filepos == ' ' || *filepos == '\n' || *filepos == '\t')
						filepos++;
				}
				break;

			case '&':		/* special char */
				switch(*(++filepos)) {
					case 'l':	/* less than */
						*(pos++) = '<';
						break;
					case 'g':	/* greater than */
						*(pos++) = '>';
						break;
					default:
						break;
				}
				while(*(filepos++) != ';');	/* consume the rest of the special char */

				break;

			case ' ':
			case '\t':
				*(pos++) = ' ';
				filepos++;

				if(!inpre)	{
					while(*filepos == ' ' || *filepos == '\n' || *filepos == '\t')
						filepos++;
				}
				break;
					
		
			case '<':		/* a tag	*/
				auxpos = strchr(filepos,'>');	/* find the end of the tag */
				*(auxpos++) = '\0';			/* null terminate the tag, set auxpos to the point were we resume reading */	
				compile_tag(&pos,++filepos);
				filepos = auxpos;
				break;

			default:
				*(pos++) = *(filepos++);
				break;
		}

	}

	// terminate the document!
	*(pos++) = 0x01;
	*(pos++) = 0x71;

	length = (pos - (char *) RecordBuffer);

	length -= ntohl(pqa->dataOffset);
	length += length%2;		// round to a word

	pqa->dataLength = htons(length);
	pqa->uncompressedSize = htons(length);

	total_size = sizeof(pqa_header) + ntohs(pqa->urlLength) + ntohs(pqa->dataLength);

	printf("record size = %u\n",total_size);

	ret = pi_file_append_record(pf,RecordBuffer,total_size,0,0,uid);

	printf("pi_file_append_record = %d\n",ret);

	return(0);
}

int compile_records(struct pi_file *pf) 
{
	static int record_csm=0;

	while(record_csm < next_record) {
		if(record_stack[record_csm].type == TYPE_HTML)
			compile_html(pf,&(record_stack[record_csm]),record_csm+1);
		else
			compile_image(pf,&(record_stack[record_csm]), record_csm+1);

		record_csm++;
	}

	return(0);
}

int main(int argc, char *argv[])
{
	struct pi_file *pf;
	struct DBInfo info;
	char app_info[1024];
	char *pos, *tmp;
	unsigned short length;
	int ret;
	int i;

	if(argc <5 ) {
		printf("%s <appname> <version> <title> <input file>\n",argv[0]);
		return(0);
	}

	memset(&info,0,sizeof(info));
	info.name[0] =0;
	strcat(info.name,argv[1]);
	memcpy(&(info.type)," aqp",4);
	memcpy(&(info.creator),"rplc",4);

	info.createDate = htonl(unix_time_to_pilot_time (time(NULL)));
	info.modifyDate = htonl(unix_time_to_pilot_time (time(NULL)));
	info.backupDate = htonl(unix_time_to_pilot_time (time(NULL)));

	info.flags = 0x0208;

	pf = pi_file_create(argv[1],&info);	

	strcpy(app_info,"lnch");
	pos = app_info +4;
	
	*(pos++) = 0x00;	/* header ver */
	*(pos++) = 0x03;	
	*(pos++) = 0x80;	/* encoding version */
	*(pos++) = 0x01;	

	length = strlen(argv[2])+1;

	length += length%2;

	length /=2;
	*(pos++) = length >>8;	/* version length */
	*(pos++) = length & 0xff;	

	strcpy(pos,argv[2]);
	pos += length*2;	
	
	
	length = strlen(argv[3])+1;

	length  += length%2;

	length /=2;
	*(pos++) = length >>8;	/* version length */
	*(pos++) = length & 0xff;	

	strcpy(pos,argv[3]);
	pos += length*2;	
	
	pos += conv_icon_ppm("LgIcon.pnm",pos,0,2);
	pos += conv_icon_ppm("SmIcon.pnm",pos,0,2);
	
	pi_file_set_app_info(pf,app_info,pos - app_info);

	/* now compile and append web records */
	for(i=4; i <argc;i++) {
		record_stack[next_record].filename = argv[i];
		tmp = strchr(argv[i],'.');
		
		if(!tmp || *(tmp+1) == 'h' || *(tmp+1) == 'H')
			record_stack[next_record++].type = TYPE_HTML;
		else
			record_stack[next_record++].type = TYPE_IMAGE;
	}

	compile_records(pf);
	
	pi_file_close(pf);

	return(0);
}
