/*
 * SAMSUNG EXYNOS5250 Flattened Device Tree enabled machine
 *
 * Copyright (c) 2012 Samsung Electronics Co., Ltd.
 *		http://www.samsung.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
*/

#include <linux/of_platform.h>
#include <linux/of_fdt.h>
#include <linux/memblock.h>
#include <linux/io.h>
#include <linux/clocksource.h>
#include <linux/dma-mapping.h>
#include <linux/phy.h>
#include <linux/phy_fixed.h>
#include <linux/of.h>
#include <linux/of_address.h>

#include <asm/mach/arch.h>
#include <mach/regs-pmu.h>

#include <plat/cpu.h>
#include <plat/mfc.h>

#include "common.h"

static u64 dma_mask64 = DMA_BIT_MASK(64);
static u64 dma_mask32 = DMA_BIT_MASK(32);

static int exynos5440_platform_notifier(struct notifier_block *nb,
				  unsigned long event, void *__dev)
{
	struct device *dev = __dev;

	if (event != BUS_NOTIFY_ADD_DEVICE)
		return NOTIFY_DONE;

	dev->dma_mask = &dma_mask64;
	dev->coherent_dma_mask = DMA_BIT_MASK(64);

	if (of_device_is_compatible(dev->of_node, "snps,dwmac-3.70a"))
		dev->dma_mask = &dma_mask32;

	return NOTIFY_OK;
}

static struct notifier_block exynos5440_platform_nb = {
	.notifier_call = exynos5440_platform_notifier,
};

static struct fixed_phy_status fixed_phy_status __initdata = {
	.link		= 1,
	.speed		= 1000,
	.duplex		= 1,
};

static void exynos_power_off(void)
{
	struct device_node *np;
	u32 val;
	void __iomem *addr;

	np = of_find_compatible_node(NULL, NULL, "samsung,exynos5440-xmu");
	addr = of_iomap(np, 0) + 0x14;
	val = 0x1;

	__raw_writel(val, addr);
}

static void __init exynos5_dt_machine_init(void)
{
	struct device_node *i2c_np, *gmac_np;
	const char *i2c_compat = "samsung,s3c2440-i2c";
	unsigned int tmp;

	/*
	 * Exynos5's legacy i2c controller and new high speed i2c
	 * controller have muxed interrupt sources. By default the
	 * interrupts for 4-channel HS-I2C controller are enabled.
	 * If node for first four channels of legacy i2c controller
	 * are available then re-configure the interrupts via the
	 * system register.
	 */
	gmac_np = of_find_compatible_node(NULL, NULL, "snps,dwmac-3.70a");

	if (of_find_property(gmac_np, "fixed_phy", NULL))
		tmp = fixed_phy_add(PHY_POLL, 1, &fixed_phy_status);

	for_each_compatible_node(i2c_np, NULL, i2c_compat) {
		if (of_device_is_available(i2c_np)) {
			if (of_alias_get_id(i2c_np, "i2c") < 4) {
				tmp = readl(EXYNOS5_SYS_I2C_CFG);
				writel(tmp & ~(0x1 << of_alias_get_id(i2c_np, "i2c")),
						EXYNOS5_SYS_I2C_CFG);
			}
		}
	}

	if (of_machine_is_compatible("samsung,exynos5440")) {
		bus_register_notifier(&platform_bus_type, &exynos5440_platform_nb);
		pm_power_off = exynos_power_off;
	}

	of_platform_populate(NULL, of_default_bus_match_table, NULL, NULL);
}

static char const *exynos5_dt_compat[] __initdata = {
	"samsung,exynos5250",
	"samsung,exynos5420",
	"samsung,exynos5440",
	NULL
};

static void __init exynos5_reserve(void)
{
#ifdef CONFIG_S5P_DEV_MFC
	struct s5p_mfc_dt_meminfo mfc_mem;

	/* Reserve memory for MFC only if it's available */
	mfc_mem.compatible = "samsung,mfc-v6";
	if (of_scan_flat_dt(s5p_fdt_find_mfc_mem, &mfc_mem))
		s5p_mfc_reserve_mem(mfc_mem.roff, mfc_mem.rsize, mfc_mem.loff,
				mfc_mem.lsize);
#endif
}

DT_MACHINE_START(EXYNOS5_DT, "SAMSUNG EXYNOS5 (Flattened Device Tree)")
	/* Maintainer: Kukjin Kim <kgene.kim@samsung.com> */
	.smp		= smp_ops(exynos_smp_ops),
	.map_io		= exynos_init_io,
	.init_machine	= exynos5_dt_machine_init,
	.init_late	= exynos_init_late,
	.init_time	= exynos_init_time,
	.dt_compat	= exynos5_dt_compat,
	.restart        = exynos5_restart,
	.reserve	= exynos5_reserve,
MACHINE_END
