/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <pthread.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "curse.h"
#include "linuxtrade.h"
#include "optchain.h"
#include "help.h"

static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static STREAMER	Sr;

static char	Symbol[SYMLEN+1];

static int	Cursor;		// Article line number at top of display
static int	MaxCursor;	// Maximum value for above
static int	NumLines;	// Number of lines in pad
static int	NumCall;
static int	NumPut;
static int	ValLivePoll;
static time_t	LastLivePoll;	// Last time we grabbed the live quotes

static char	Exch[2] = "E";
static int	UseSr;

#define	PADLINES	400

typedef struct
{
	char	sym[SYMLEN+1];
	int	x;
} QUOTELOC;

static QUOTELOC	Live[PADLINES][2];

static void
get_live_quotes(void)
{
	int	y;
	int	x;
	int	cnt = 0;

	strcpy(List0, Symbol);
	List0Comments[0] = 0;
	strcpy(StockList0Name, Symbol);
	strcat(StockList0Name, " Option Chains");
	for (y = Cursor; y < Cursor+getmaxy(Win)-2; ++y)
		for (x = 0; x < 2; ++x)
		{
			if (!Live[y][x].sym[0])
				continue;
			(*Sr->send_symbols)(Sr, Live[y][x].sym, TRUE);
			(*Sr->send_symbols_end)(Sr, TRUE, FALSE);
			(*Sr->send_symbols)(Sr, Live[y][x].sym, FALSE);
			(*Sr->send_symbols_end)(Sr, FALSE, FALSE);

			if (cnt < 50)
			{
			       ++cnt;
			       strcat(List0, " ");
			       strcat(List0, Live[y][x].sym);
			}
		}

	time(&LastLivePoll);
}

void
optchain_quote(QUOTE *q)
{
	int	y;
	int	x;

	if (!Win)
		return;

	for (y = Cursor; y < Cursor + getmaxy(Win) - 2 - 1; ++y)
		for (x = 0; x < 2; ++x)
		{
			attr_t	attr;

			if (strcmp(q->sym, Live[y][x].sym))
				continue;

			if (q->last > q->close)
				attr = COLOR_PAIR(1);
			else if (q->last < q->close)
				attr = COLOR_PAIR(2);
			else
				attr = A_NORMAL;

			wattrset(Subwin, attr);
			mvwprintw(Subwin, y, Live[y][x].x,
				"%7.2f %+7.2f", q->last, q->last - q->close);
			wattrset(Subwin, A_NORMAL);
			goto found;
		}
found:

	// TODO: optimize this to do 1 line and move into loop
	copywin(Subwin, Win,
			Cursor, 0,
			1+1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

static void
display_more(void)
{
	mvwaddch(Win, 1, getmaxx(Win)-1,
		Cursor ? ACS_UARROW : ACS_VLINE);

	mvwaddch(Win, getmaxy(Win)-2, getmaxx(Win)-1,
		(Cursor < MaxCursor) ? ACS_DARROW : ACS_VLINE);
}

void
optchain_optchain(OPTCHAIN *oc)
{
	int	x, lx, y;

	if (!Win)
		return;

	if (strcmp(oc->sym, Symbol))
		return;

	// shorten date
	memmove(oc->date+4, oc->date+6, 3);

	if (oc->type == 0)
	{
		if (NumCall == PADLINES)
			return;
		x = 0;
		y = NumCall++;
		lx = 0;
	}
	else if (oc->type == 1)
	{
		if (NumPut == PADLINES)
			return;
		x = getmaxx(Subwin)/2;
		y = NumPut++;
		lx = 1;
	}
	else
		return;

	mvwprintw(Subwin, y, x, "%-5s %s %7.2f",
			oc->optsym, oc->date, oc->price);

	strcpy(Live[y][lx].sym, oc->optsym);
	Live[y][lx].x = getcurx(Subwin) + 1;

	NumLines = max(NumCall, NumPut);

	MaxCursor = NumLines - (getmaxy(Win) - 2 - 1);
	if (MaxCursor < 0)
		MaxCursor = 0;

	if (oc->index == oc->count)
	{
		for (x = 0; x < 11; ++x)
			mvwaddch(Win, getmaxy(Win) - 1,
				(getmaxx(Win)-11)/2 + x, ACS_HLINE);
		//
		// Option quote polls are expensive on the server, because
		// the quick quote feature doesn't work on them, so we punt
		// and issue a streaming add immediately followed by a delete.
		//
		// Issue the first live quote poll, with subsequent ones every
		// 60 seconds.
		//
		get_live_quotes();
		ValLivePoll = 60;
	}

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1+1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

typedef struct
{
	int	lnum;
	double	strike;
	int	coi;
	int	poi;
} MPAIN;

#define	NUMPAIN	128
MPAIN	Pain[NUMPAIN];
int	NumPain;

static void
mpain_reset(void)
{
	NumPain = 0;
}
static void
mpain_data(int lnum, double strike, int coi, int poi)
{
	if (NumPain >= NUMPAIN)
		return;
	Pain[NumPain].lnum = lnum;
	Pain[NumPain].strike = strike;
	Pain[NumPain].coi = coi;
	Pain[NumPain].poi = poi;
	++NumPain;
}
static int
mpain_calc(void)
{
	int		i, j, mvi;
	double		sp1, sp2;
	long long	cv, pv, tv, mv;

	mv = 999999999999;
	mvi = 0;
	for (i = 0; i < NumPain; ++i)
	{
		sp1 = Pain[i].strike;
		cv = pv = 0;
		for (j = 0; j < NumPain; ++j)
		{
			sp2 = Pain[j].strike;
			if (sp1 > sp2)
				cv += Pain[j].coi * (sp1 - sp2);
			if (sp1 < sp2)
				pv += Pain[j].poi * (sp2 - sp1);
		}
		tv = cv + pv;
		if (tv < mv)
		{
			mv = tv;
			mvi = i;
		}
	}

	return mvi;
}

static void *
cboe(void *arg)
{
	char	buf[512];
	FILE	*fp;
	int	rc;
	char	*sym = (char *) arg;
	extern pthread_mutex_t CurseMutex;
	char	*ename;

	int	lyr;
	char	lmm[64];

	pthread_detach(pthread_self());

	sprintf(buf,
		"%s "
		"-dTICKER='%s' "
		"http://quote.cboe.com/QuoteTable.dat"
		, SUCKURL, sym);

	fp = popen(buf, "r");
	if (!fp)
	{
		pthread_mutex_lock(&CurseMutex);
		mvwcenter(Win, getmaxy(Win)/2,
				"Option Chains could not be retrieved");

		goto out;
	}

	// Skip column labels
	// #EMC (NYSE),12.07,pc,
	// #Mar 14 2002 @ 08:59 ET (Data 20 Minutes Delayed),Bid,N/A,Ask,N/A,Size,N/AxN/A,Vol,0,
	// #Calls,Last Sale,Net,Bid,Ask,Vol,Open Int,Puts,Last Sale,Net,Bid,Ask,Vol,Open Int,
	// #02 Mar 5.00 (EMC CA-E),6.20,pc,0,0,0,118,02 Mar 5.00 (EMC OA-E),0,pc,0,0,0,60,

	fgets(buf, sizeof(buf), fp);
	fgets(buf, sizeof(buf), fp);
	fgets(buf, sizeof(buf), fp);

	lyr = 0; strcpy(lmm, "");

	while(fgets(buf, sizeof(buf), fp))
	{
		char		cstr[64], pstr[64];
		double		clast, plast;
		char		cnet[64], pnet[64];
		double		cbid, pbid;
		double		cask, pask;
		int		cvol, pvol;
		int		copen, popen;
		int		cyr, pyr;
		char		cmm[64], pmm[64];
		double		cstrike, pstrike;
		char		cbase[64], pbase[64];
		char		csuf[64], psuf[64];
		char		cexch[64], pexch[64];
		char		csym[64], psym[64];

		rc = sscanf(buf,
			"%[^,],%lf,%[^,],%lf,%lf,%d,%d,"
			"%[^,],%lf,%[^,],%lf,%lf,%d,%d,"
			, cstr, &clast, cnet, &cbid, &cask, &cvol, &copen
			, pstr, &plast, pnet, &pbid, &pask, &pvol, &popen
			);

		if (rc != 14)
			continue;

		rc = sscanf(cstr, "%d %s %lf (%s %[^-]-%[^)]",
				&cyr, cmm, &cstrike, cbase, csuf, cexch);
		if (rc != 6)
			continue;

		rc = sscanf(pstr, "%d %s %lf (%s %[^-]-%[^)]",
				&pyr, pmm, &pstrike, pbase, psuf, pexch);
		if (rc != 6)
			continue;

		if (cyr != pyr || cstrike != pstrike)
			continue;
		if (strcmp(Exch, "*") &&
			(strcmp(Exch, cexch) || strcmp(Exch, pexch)) )
			continue;

		if (lyr != cyr || strcmp(lmm, cmm))
		{
			// New date
			if (lmm[0])
			{
				int	mpi;

				mpi = mpain_calc();
				mvwprintw(Subwin, NumLines, 0,
					"MaxHurt = %.2f\n",
					Pain[mpi].strike);

				if (++NumLines >= PADLINES)
					break;

				if (++NumLines >= PADLINES)
					break;
			}

			mpain_reset();
			lyr = cyr;
			strcpy(lmm, cmm);
		}

		strcpy(csym, cbase); strcat(csym, csuf);
		strcpy(psym, pbase); strcat(psym, psuf);

		mvwprintw(Subwin, NumLines, 0,
			"%s %02d "
			"%6.2f"
			, cmm, cyr, cstrike
			 );

		wprintw(Subwin, " %-5s%7.2f", csym, clast);
		if (strchr(cnet, '-'))
			wattrset(Subwin, COLOR_PAIR(2));
		else if (strchr(cnet, '+'))
			wattrset(Subwin, COLOR_PAIR(1));
		wprintw(Subwin, "%6.6s", cnet);
		wattrset(Subwin, A_NORMAL);
		wprintw(Subwin, "%6d%7d", cvol, copen);

		wprintw(Subwin, " %-5s%7.2f", psym, plast);
		if (strchr(pnet, '-'))
			wattrset(Subwin, COLOR_PAIR(2));
		else if (strchr(pnet, '+'))
			wattrset(Subwin, COLOR_PAIR(1));
		wprintw(Subwin, "%6.6s", pnet);
		wattrset(Subwin, A_NORMAL);
		wprintw(Subwin, "%6d%7d", pvol, popen);

		mpain_data(NumLines, cstrike, copen, popen);

		if (++NumLines >= PADLINES)
			break;
	}

	pclose(fp);

	MaxCursor = NumLines - (getmaxy(Win) - 2 - 1);
	if (MaxCursor < 0)
		MaxCursor = 0;

	pthread_mutex_lock(&CurseMutex);

	wattrset(Win, A_BOLD);
	mvwprintw(Win, 1, 1,
			"DATE   STRIKE"
			" CALLS   LAST   CHG   VOL   OPEN"
			" PUTS    LAST   CHG   VOL   OPEN"
			);
	wattrset(Win, A_NORMAL);

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1+1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);

	switch (Exch[0])
	{
	case 'A':	ename = " (AMEX)"; break;
	case 'E':	ename = " (CBOE)"; break;
	case 'P':	ename = " (PSE)"; break;
	case 'X':	ename = " (PHLX)"; break;
	case '*':	ename = " (all exchanges)"; break;
	default:	ename = ""; break;
	}
	sprintf(buf, "from cboe.com - exchange %s%s", Exch, ename);
	mvwcenter(Win, getmaxy(Win) - 1, buf);

out:
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
	pthread_mutex_unlock(&CurseMutex);

	return (NULL);
}

static void
start_cboe(void)
{
	int	rc;
	pthread_t tid;

	NumLines = 0;
	Cursor = 0;
	MaxCursor = 0;

	werase(Subwin);
	display_more();

	copywin(Subwin, Win,
			Cursor, 0,
			1+1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);

	mvwcenter(Win, getmaxy(Win)/2, "Please wait");
	touchwin(Win);
	update_panels();
	doupdate();

	rc = pthread_create(&tid, NULL, cboe, Symbol);
	if (rc)
		error(1, "Couldn't create inplay thread.\n");

	if (0)
	mvwcenter(Win, getmaxy(Win)/2,
		"Option chains not available with this streamer");
}

void
optchain_poll(void)
{
	time_t	now;

	if (!Win)
		return;

	if (!ValLivePoll)
		return;

	now = time(NULL);

	if (now >= (LastLivePoll + ValLivePoll))
	{
		ValLivePoll = 60;
		get_live_quotes();
	}
}

void
optchain_popup(STOCK *sp, STREAMER sr, int use_sr)
{
	int	cols;
	int	n;

	if (!sr->send_optchain)
		use_sr = FALSE;

	n = LINES - 4 - 2 - NumStock - 12;
	if (n < 20 || use_sr)
		n = 20;

	Win = bestwin(n);
	if (!Win)
		error(1, "Can't create optchain window\n");

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);
	mvwprintw(Win, 0, (cols - strlen(sp->sym) - 14)/2,
			"%s Option Chains", sp->sym);

	Subwin = newpad(PADLINES, getmaxx(Win) - 2);
	if (!Subwin)
		error(1, "Can't create optchain subwindow\n");
	wbkgd(Subwin, Reverse ? A_REVERSE : A_NORMAL);

	Panel = new_panel(Win);

	Sr = sr;

	strncpy(Symbol, sp->sym, SYMLEN);
	Symbol[SYMLEN] = 0;

	memset(Live, 0, sizeof(Live));

	NumCall = NumPut = 0;
	NumLines = 0;
	Cursor = 0;
	MaxCursor = 0;
	LastLivePoll = 0;
	ValLivePoll = 0;
	UseSr = use_sr;

	if (use_sr)
	{
		wattrset(Win, A_BOLD);
		mvwprintw(Win, 1, 1,
				"CALLS DATE    STRIKE    LAST  CHANGE");
		mvwprintw(Win, 1, getmaxx(Win)/2,
				"PUTS  DATE    STRIKE    LAST  CHANGE");
		wattrset(Win, A_NORMAL);

		(*sr->send_optchain)(sr, sp->sym);
		mvwcenter(Win, getmaxy(Win) - 1, "Please wait");
	}
	else
		start_cboe();
}

static void
popdown(void)
{
	hide_panel(Panel);
	update_panels();
	del_panel(Panel);
	delwin(Subwin);
	delwin(Win);
	Win = 0;
	Subwin = 0;
	Sr = NULL;
	Symbol[0] = 0;
}

int
optchain_command(int c, STREAMER sr)
{
	static int	(*handler)(int c, STREAMER sr);
	int		rc;
	MEVENT		m;

	if (handler)
	{
		rc = (*handler)(c, sr);
		if (rc)
			handler = NULL;
		move(LINES-1, CursorX);
		update_panels(); refresh(); // doupdate();
		return 0;
	}

	switch (c)
	{
	case '?':
		if (UseSr)
			beep();
		else
		{
			handler = help_command;
			help_popup("options");
		}
		break;
	case '\f':
		move(LINES-1, CursorX);
		wrefresh(curscr);
		return -1;
	case 'j':
	case KEY_DOWN:
		if (++Cursor > MaxCursor)
		{
			--Cursor;
			beep();
			break;
		}
		break;
	case 'k':
	case KEY_UP:
		if (--Cursor < 0)
		{
			++Cursor;
			beep();
			break;
		}
		break;
	case '-':
	case KEY_PPAGE:
		if (Cursor == 0)
		{
			beep();
			break;
		}
		Cursor -= getmaxy(Win) - 2 - 1;
		if (Cursor < 0)
			Cursor = 0;
		break;
	case '+':
	case ' ':
	case KEY_NPAGE:
		if (Cursor == MaxCursor)
		{
			beep();
			break;
		}
		Cursor += getmaxy(Win) - 2 - 1;
		if (Cursor > MaxCursor)
			Cursor = MaxCursor;
		break;
	case '0':
	case KEY_HOME:
		Cursor = 0;
		break;
	case '$':
	case KEY_END:
		Cursor = MaxCursor;
		break;

	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;

	case KEY_PRINT:
	case KEY_F(12):
	case CTRL('P'):
		print_window(Subwin, NumLines,
				get_rc_value(RcFile, "print_cmd"));
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	case 'E':
	case 'A':
	case 'P':
	case 'X':
	case '8':
	case '*':
		Exch[0] = c;
		start_cboe();
		break;

	case 'O':
		launch_url("pref:optchain_URL", Stock[StockCursor].sym);
		break;

	case 033:
	case 'q':
	case 'x':
		popdown();
		return 2;
	default:
		beep();
		break;
	}

	//
	// If the cursor is in motion, defer a new quote update for 5 seconds
	//
	time(&LastLivePoll);
	ValLivePoll = 5;

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1+1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();

	return (-1);
}
