/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "FSK441Receive.h"

#include "common/SFFT.h"

#include <cmath>
using namespace std;

#include <wx/datetime.h>
#include <wx/debug.h>
#include <wx/log.h>

enum {
	FSK441_SEEKING,
	FSK441_LOCKED
};

CFSK441Receive::CFSK441Receive() :
CReceive(),
m_singleTone(false),
m_id(wxEmptyString),
m_audioSamples(NULL),
m_level(),
m_levels(),
m_samplesCount(0),
m_noise(),
m_burstData(),
m_spectrum(NULL)
{
	m_audioSamples = new double[FSK441_MAX_AUDIO_DATA];

	m_levels[0] = new double[FSK441_MAX_AUDIO_DATA];
	m_levels[1] = new double[FSK441_MAX_AUDIO_DATA];
	m_levels[2] = new double[FSK441_MAX_AUDIO_DATA];
	m_levels[3] = new double[FSK441_MAX_AUDIO_DATA];
}

CFSK441Receive::~CFSK441Receive()
{
	delete[] m_audioSamples;
	delete[] m_levels[0];
	delete[] m_levels[1];
	delete[] m_levels[2];
	delete[] m_levels[3];
}

void CFSK441Receive::run()
{
	::wxLogInfo(wxT("%ld: Starting FSK441Receive"), GetId());

	m_id = createId();

	bool end = false;

	// Reset the state variables
	m_level.clear();
	m_samplesCount = 0;
	m_noise.clear();
	for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++) {
		m_burstData[i].clear();
		m_burstData[i].setRatio(FSK441_MIN_CORRVAL);
	}

	CSFFT sfft(FSK441_FFT_LENGTH, FSK441_BIN0, FSK441_BIN3 + 1);

	int tim   = 0;
	int state = FSK441_SEEKING;

	openSoundDevice();

	while (!isStopped() && !end && m_samplesCount < (FSK441_MAX_AUDIO_DATA - FSK441_SOUNDBUF_LENGTH)) {
		int len = FSK441_SOUNDBUF_LENGTH;
		bool ret = getSoundDevice()->read(m_audioSamples + m_samplesCount, len);

		if (!ret) break;

		for (int i = 0; i < len; i++) {
			double val = m_audioSamples[m_samplesCount + i];

			m_level.addValue(val * val);

			double* bins = sfft.process(val);

			if (tim >= FSK441_FFT_LENGTH) {
				double v[4];
				v[0] = bins[FSK441_BIN0];
				v[1] = bins[FSK441_BIN1];
				v[2] = bins[FSK441_BIN2];
				v[3] = bins[FSK441_BIN3];

				m_levels[0][m_samplesCount + i] = v[0];
				m_levels[1][m_samplesCount + i] = v[1];
				m_levels[2][m_samplesCount + i] = v[2];
				m_levels[3][m_samplesCount + i] = v[3];

				switch (state) {
					case FSK441_SEEKING:
						state = seekBurst(tim, v);
						break;
					case FSK441_LOCKED:
						state = storeBurst(tim, v);
						break;
					default:
						::wxLogError(wxT("Unknown state in FSK441Receive"));
						state = FSK441_SEEKING;
						break;
				}
			}

			tim++;
		}

		m_samplesCount += len;

		end = getEndTime();
	}

	closeSoundDevice();

	// End of the data and still receiving a burst, display the collected data
	if (state == FSK441_LOCKED) {
		decodeBurst();

		showSpectrum(m_spectrum);
		m_spectrum = NULL;
	}

	::wxLogMessage(wxT("%ld: Received %d samples of data"), GetId(), m_samplesCount);

	recordAudio(m_id, m_audioSamples, m_samplesCount);

	getLevels();

	::wxLogInfo(wxT("%ld: Ending FSK441Receive"), GetId());
}

void CFSK441Receive::setSingleTone(bool singleTone)
{
	m_singleTone = singleTone;
}

/*
 * Correlator for the four FSK441 tones held in the bins represented by
 * v0, v1, v2 and v3 with the correlation values returned in c0, c1, c2
 * and c3. If there is zero or negative correlation than a false is
 * returned and the values in c0 to c3 are not normalised, a true
 * indicates a positive correlation with normalised values returned in
 * c0 to c3.
 */
bool CFSK441Receive::correlate(double* v, double* c) const
{
	wxASSERT(v != NULL);
	wxASSERT(c != NULL);

	c[0] = 3.0 * v[0] - v[1] - v[2] - v[3];
	c[1] = 3.0 * v[1] - v[0] - v[2] - v[3];
	c[2] = 3.0 * v[2] - v[0] - v[1] - v[3];
	c[3] = 3.0 * v[3] - v[0] - v[1] - v[2];

	// Not interesting, return false and don't do the expensive sqrt() function
	if (c[0] < FSK441_MIN_CORRVAL && c[1] < FSK441_MIN_CORRVAL && c[2] < FSK441_MIN_CORRVAL && c[3] < FSK441_MIN_CORRVAL)
		return false;

	double div = ::sqrt(12.0 * (v[0] * v[0] + v[1] * v[1] + v[2] * v[2] + v[3] * v[3]));

	c[0] /= div;
	c[1] /= div;
	c[2] /= div;
	c[3] /= div;

	return true;
}

/*
 * This is the normal state when a burst has not been detected. Every
 * fifth FFT reading is processed here and if any data with a suitable
 * correlation is found then its value is stored and the state is set
 * LOCKING. We are at the beginning of a burst.
 */
int CFSK441Receive::seekBurst(int tim, double* v)
{
	wxASSERT(m_spectrum == NULL);
	wxASSERT(v != NULL);

	double c[4];
	bool ret = correlate(v, c);

	// Nothing to do, except save the levels for the noise reference
	if (!ret) {
		m_noise.addValue(v[0]);
		m_noise.addValue(v[1]);
		m_noise.addValue(v[2]);
		m_noise.addValue(v[3]);

		return FSK441_SEEKING;
	}

	bool found = false;

	for (int i = 0; i < 4; i++) {
		if (c[i] >= FSK441_MIN_CORRVAL)
			found = true;
	}

	if (found) {
		m_spectrum = new CFSK441Spectrum();
		m_spectrum->addSpectrum(v[0], v[1], v[2], v[3]);

		for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++) {
			m_burstData[i].setStartBurst();
			m_burstData[i].setStartTime(tim);
		}

		int n = tim % FSK441_SYMBOL_LENGTH;
		m_burstData[n].setData(v, c);

		return FSK441_LOCKED;
	}

	return FSK441_SEEKING;
}

/*
 * We now have a burst of over one symbol in length and a more or
 * less optimum sample location for it. From now on we only process
 * every 25 sample times (one symbol length) and find the best
 * correlation values and store the appropriate symbol in m_data.
 * Once the burst ends then we call decodeBurst() to process and
 * display the burst data.
 */
int CFSK441Receive::storeBurst(int tim, double* v)
{
	wxASSERT(m_spectrum != NULL);
	wxASSERT(v != NULL);

	m_spectrum->addSpectrum(v[0], v[1], v[2], v[3]);

	double c[4];
	bool ret = correlate(v, c);

	// End of the burst, so display the collected data
	if (!ret) {
		m_noise.addValue(v[0]);
		m_noise.addValue(v[1]);
		m_noise.addValue(v[2]);
		m_noise.addValue(v[3]);
	}

	int n = tim % FSK441_SYMBOL_LENGTH;
	m_burstData[n].setData(v, c);

	for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++) {
		if (m_burstData[i].getInBurst())
			return FSK441_LOCKED;
	}

	decodeBurst();

	showSpectrum(m_spectrum);
	m_spectrum = NULL;

	return FSK441_SEEKING;
}

/*
 * This is where the raw burst data is processed. We look for a
 * space character, the synchronisation key in FSK441, and failing
 * that we look for single tone characters. If none is found then
 * we have a very strange burst indeed !
 * The burst data is held as a character string containing the
 * numbers representing the different tones so we can use string
 * manipulation methods on it.
 */
void CFSK441Receive::decodeBurst()
{
	int maxLen = 0;
	int      n = -1;

	for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++) {
		bool valid = m_burstData[i].processBurst(m_singleTone) > 0;
		int    len = m_burstData[i].getLength();

		if (valid && len > maxLen && len >= 9) {	// 20ms
			maxLen = len;
			n      = i;
		}
	}

	if (n == -1)
		return;

	int strength = 99;
	if (m_noise.getCount() > 0)
		strength = m_burstData[n].getStrength(m_noise.getAverage());

	wxString    text = m_burstData[n].getMessage();
	double startTime = double(m_burstData[n].getStartTime()) * 0.0000907;
	int       length = int(double(m_burstData[n].getLength()) * 2.2676);

	CFSK441Message* message = new CFSK441Message(m_id, startTime, length, strength, text);
	receiveMessage(message);
}

void CFSK441Receive::getLevels() const
{
	CFSK441Levels* levels = new CFSK441Levels();

	levels->setAudioData(m_audioSamples, m_samplesCount);

	levels->setSpectrumData(0, m_levels[0], m_samplesCount);
	levels->setSpectrumData(1, m_levels[1], m_samplesCount);
	levels->setSpectrumData(2, m_levels[2], m_samplesCount);
	levels->setSpectrumData(3, m_levels[3], m_samplesCount);

	if (m_level.getMaximum() == (1.0 * 1.0))
		levels->setLevel(Audio_High);
	else if (m_level.getMaximum() < (0.1 * 0.1))
		levels->setLevel(Audio_Low);
	else
		levels->setLevel(Audio_OK);

	showLevels(levels);
}
