/*
 *   Copyright (C) 2003 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT6MBurstData.h"

#include "jt6m/JT6MLookups.h"
#include "jt6m/JT6MDefs.h"

#include <cmath>
using namespace std;

CJT6MBurstData::CJT6MBurstData() :
m_data(),
m_strength(),
m_startTime(0),
m_inBurst(false),
m_ratio(0.0),
m_message(wxEmptyString)
{
}

CJT6MBurstData::~CJT6MBurstData()
{
}

void CJT6MBurstData::clear()
{
	m_data.Empty();

	m_strength.clear();

	m_startTime = 0;

	m_inBurst = false;

	m_message = wxEmptyString;
}

void CJT6MBurstData::setStartBurst()
{
	m_data.Empty();

	m_strength.clear();

	m_startTime = 0;

	m_inBurst = true;

	m_message = wxEmptyString;
}

void CJT6MBurstData::setStartTime(int startTime)
{
	wxASSERT(startTime > 0);

	if (!m_inBurst)
		return;

	m_startTime = startTime;
}

void CJT6MBurstData::setRatio(double ratio)
{
	wxASSERT(ratio > 0.0);

	m_ratio = ratio;
}

void CJT6MBurstData::setData(double* v, double* c)
{
	wxASSERT(v != NULL);
	wxASSERT(c != NULL);

	if (!m_inBurst)
		return;

	m_strength.addValue(v[0]);
	m_strength.addValue(v[1]);
	m_strength.addValue(v[2]);
	m_strength.addValue(v[3]);

	double cMax = -999.0;
	int n = -1;

	for (int i = 0; i < 4; i++) {
		if (c[i] >= m_ratio && c[i] > cMax) {
			cMax  = c[i];
			n     = i;
		}
	}

	switch (n) {
		case 0:
			m_data.Append(wxT("0"));
			break;
		case 1:
			m_data.Append(wxT("1"));
			break;
		case 2:
			m_data.Append(wxT("2"));
			break;
		case 3:
			m_data.Append(wxT("3"));
			break;
		default:
			m_inBurst = false;
			break;
	}
}

bool CJT6MBurstData::getInBurst() const
{
	return m_inBurst;
}

int CJT6MBurstData::getStartTime() const
{
	return m_startTime;
}

int CJT6MBurstData::getLength() const
{
	return m_data.Length();
}

int CJT6MBurstData::processBurst(bool singleTone)
{
	int nMT = m_data.Find(wxT("033"));
	int nST = -1;

	if (nMT == -1 && singleTone) {
		nST = m_data.Find(wxT("000"));

		if (nST == -1) {
			nST = m_data.Find(wxT("111"));

			if (nST == -1) {
				nST = m_data.Find(wxT("222"));

				if (nST == -1)
					nST = m_data.Find(wxT("333"));
			}
		}
	}

	int n;
	if (nMT != -1)
		n = nMT;
	else if (nST != -1)
		n = nST;
	else
		return 0;

	CJT6MLookups lookups;

	for (n %= JT6M_CHAR_LENGTH; n < int(m_data.Length() - 2); n += JT6M_CHAR_LENGTH) {
		long t1, t2, t3;
		m_data.Mid(n + 0, 1).ToLong(&t1);
		m_data.Mid(n + 1, 1).ToLong(&t2);
		m_data.Mid(n + 2, 1).ToLong(&t3);

		wxString text;
		bool     ret;

		if (nMT != -1)
			ret = lookups.lookupMultiTones(t1, t2, t3, text);
		else		// nST != -1
			ret = lookups.lookupSingleTones(t1, t2, t3, text);

		if (ret)
			m_message.Append(text);
		else
			m_message.Append(wxT("*"));
	}

	return m_message.Length();
}

wxString CJT6MBurstData::getMessage() const
{
	return m_message;
}

int CJT6MBurstData::getStrength(double noise) const
{
	wxASSERT(noise != 0.0);

	double strength = 99.0;

	if (m_strength.getCount() > 0)
		strength = 10.0 * ::log10(m_strength.getMaximum() / noise);

	return int(strength + 0.5);
}
