/****************************************************************
 *
 * vdictov: reorder.H
 *
 * Copyright (C) Max Planck Institute 
 * for Human Cognitive and Brain Sciences, Leipzig
 *
 * Author Thomas Arnold, 2002, <lipsia@cbs.mpg.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * $Id: reorder.H 902 2004-07-12 08:37:01Z anwander $
 *
 *****************************************************************/

 /*
 $Log$
 Revision 1.7  2004/07/12 08:37:01  anwander
 added coil identification

 Revision 1.5  2003/11/05 17:29:22  anwander
 corrected slicetime for interleaved mode with odd number of slices

 Revision 1.4  2003/09/17 06:46:38  anwander
 Added support for dti images

 Revision 1.3  2003/09/09 09:26:53  anwander
 corrected slice time and ordering for fmri data
 */


#ifndef REORDER_H_INCLUDED
#define REORDER_H_INCLUDED

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <time.h>
#include <assert.h>

extern "C"
{
   #include <Vlib.h>
   #include <VImage.h>
   #include <option.h>
   #include <mu.h>
}


/*------------------------------------------------------------------------------

MajorAnatomical
===============

Temp   list of functional image with one image per TIMESTEP with voxels of type T
Anat   list of functional image with one image per SLICE with voxels of type T (created)

------------------------------------------------------------------------------*/

template <class T> void MajorAnatomical (VAttrList Temp, VAttrList& Anat)
{
   int       Steps;     /* number of timesteps */
   int       Bands;     /* number of bands     */
   int       Rows;      /* number of rows      */
   int       Columns;   /* number of columns   */
   VRepnKind Repn;      /* type of voxels      */
   VAttrList List;      /* attribute list      */

   VAttrListPosn temp;    /* position in list */
   VAttrListPosn anat;    /* position in list */
   VImage        image;   /* image in list    */

   char* str;          /* attribute value          */
   float duration;     /* time between slices      */
   float repetition;   /* time between repetitions */
   char  attr[65];     /* attribute value          */
   int   sliceMode;    /* sliceArray.ucMode        */

   T* src;    /* source data pointer      */
   T* dest;   /* destination data pointer */

   int x, y, z, t;   /* indices */


   /* get image sizes and attributes */
   for (Steps = 0, VFirstAttr (Temp, &temp); VAttrExists (&temp); VNextAttr (&temp), Steps++);
   VFirstAttr (Temp, &temp);
   VGetAttrValue (&temp, NULL, VImageRepn, &image);
   Bands   = VImageNBands   (image);
   Rows    = VImageNRows    (image);
   Columns = VImageNColumns (image);
   Repn    = VPixelRepn     (image);
   List    = VImageAttrList (image);

   /* get times */
   VGetAttr (List, "duration", NULL, VStringRepn, &str);
   duration = 0; sscanf (str, "%f", &duration);
   VGetAttr (List, "repetition", NULL, VStringRepn, &str);
   repetition = 0; sscanf (str, "%f", &repetition);
   VGetAttr (List, "sliceMode", NULL, VStringRepn, &str);
   sliceMode = 0; sscanf (str, "%d", &sliceMode);

   /* create anatomical images */
   Anat = VCreateAttrList ();
   for (z = 0; z < Bands; z++)
   {
      image = VCreateImage (Steps, Rows, Columns, Repn);
      VImageAttrList (image) = VCopyAttrList (List);
      if ((sliceMode == 0) || (sliceMode == 1))             // ascending slice-order
         sprintf (attr, "%g", z * duration); 
      else if (sliceMode == 2)                              // decending slice-order
         sprintf (attr, "%g", (Bands-z-1) * duration);
      else if (sliceMode == 4) {                            // interleaved slice-order

         // Der vierte Modus heisst "verschachtelt im Atemanhalteintervall" und bedeutet:
         // Die Schichten werden in Schichtpakete aufgeteilt, von denen jedes whrend eines
         // Atemanhalteintervalls (verschachtelt) gemessen wird.
		 
		 //acquisition from inferior to superior
		 //for even number of slices (example = 8)
		 //1 3 5 7 0 2 4 6
		 //
		 //for odd number of slices (example = 9) 
		 //0 2 4 6 8 1 3 5 7

         if (Bands/2. == (int)(Bands/2)) { //even case
         	 //starting position in the second slice (1,3,5, ... 0,2,4,...)
         	 if (z/2. == (int)(z/2))							
         	 	sprintf (attr, "%g", (int)(z/2) * duration + (int)(Bands/2)*duration);
         	 else
         	 	sprintf (attr, "%g", (int)(z/2) * duration);
		 }
		 else { // odd case
         	 //starting position in the first slice (0,2,4,... 1,3,5,..)
         	 if (z/2. == (int)(z/2))							
         	 	sprintf (attr, "%g", (int)(z/2) * duration);
         	 else
         	 	sprintf (attr, "%g", (int)(z/2) * duration + (int)((Bands+1)/2)*duration);
		 }
      }
      VAppendAttr (VImageAttrList (image), "slice_time", NULL, VStringRepn, attr);
      sprintf (attr, " repetition_time=%g packed_data=1 %d", repetition, Steps);
      VAppendAttr (VImageAttrList (image), "MPIL_vista_0", NULL, VStringRepn, attr);
      VAppendAttr (Anat, "image", NULL, VImageRepn, image);
   }

   /* reorder images */
   for (VFirstAttr (Temp, &temp), t = 0; t < Steps; t++, VNextAttr (&temp))
   {
      VGetAttrValue (&temp, NULL, VImageRepn, &image);
      src = (T*) VPixelPtr (image, 0, 0, 0);
      for (VFirstAttr (Anat, &anat), z = 0; z < Bands; z++, VNextAttr (&anat))
      {
         VGetAttrValue (&anat, NULL, VImageRepn, &image);
         dest = (T*) VPixelPtr (image, t, 0, 0);
         for (y = 0; y < Rows; y++)
            for (x = 0; x < Columns; x++)
               *(dest++) = *(src++);
      }
   }

} /* MajorAnatomical */

template <class T> void MajorAnatomical (VAttrList& Temp)
{
   VAttrList Anat;   /* anatomical images */


   /* reorder images */
   MajorAnatomical<T> (Temp, Anat);
   VDestroyAttrList (Temp);
   Temp = Anat;

} /* MajorAnatomical */

/*----------------------------------------------------------------------------*/

#endif
