package Lire::AsciiDlf::Group;

use strict;

# We need to inherit from AsciiDlf::Aggregator first, otherwise
# perl will look into Lire::Aggregator before than
# Lire::AsciiDlf::Aggregator
use base qw/ Lire::AsciiDlf::Aggregator Lire::Group /;

use Carp;

use Lire::AsciiDlf::ReportOperator qw/ group_data_value /;
use Lire::DataTypes qw/ :basic /;

=pod

=head1 NAME

Lire::AsciiDlf::Group - interface to lire:group XML entities

=head1 SYNOPSIS

 use Lire::AsciiDlf::Group;

=head1 DESCRIPTION

Lire::AsciiDlf::Group offers various functions to handle <lire:group>
XML entities. This module is used from
Lire::AsciiDlf::AsciiDlfFactory.

=head1 METHODS AND FUNCTIONS

=cut

#------------------------------------------------------------------------
# Method init_merge($period_start, $period_end)
#
# Method required by Lire::AsciiDlf::ReportOperator
sub init_merge {
    my ( $self, $period_start, $period_end ) = @_;

    # We can't use the array data structure for merging
    # because we aren't seeing the entries sorted.
    $self->{'use_array'} = 0;

    $self->init_common;
    $self->SUPER::init_merge( $period_start, $period_end );
}

#------------------------------------------------------------------------
# Method init_common
#
# Initialization common to both init_report() and init_merge()
sub init_common {
    my ( $self ) = @_;

    if ( defined $self->limit ) {
	my $limit = $self->limit;
	if ( $limit =~ /^\$/ ) {
	    $limit = substr $limit, 1;
	    $limit = $self->report_spec->param( $limit )->value;
	}
	$self->{'limit_view'} = $limit;

	# For better report merging, we add some entries.
	$self->{'limit_num'} = $self->_max_entries();
    }

    $self->{'sort_cmp'} = $self->make_sort_function;
}

#------------------------------------------------------------------------
# Method make_sort_function()
#
# Create a sort function which is used to sort the data according
# to the sort specification.
sub make_sort_function {
    my ( $self ) = @_;

    return unless $self->sort_fields;

    # Build sort function
    my ( $a_dflt, $b_dflt, $schwartzian );
    if (  $self->{'use_array'}) {
	$a_dflt = '$_[0]';
	$b_dflt = '$_[1]';
	$schwartzian = ''; # Not using schwarzian transform
    } else {
	$a_dflt = '$a';
	$b_dflt = '$b';
	$schwartzian = '[1]'; # Using schwartzian transform
    }

    my @sort_ops = ();
    foreach my $f ( @{$self->sort_fields} ) {
	my ($a, $b) = ($a_dflt, $b_dflt );
	if ( $f =~ /^-/ ) {
	    $f = substr $f, 1;
	    ($a,$b) = ($b,$a);
	}

	my $index;	# This will contains the index of the field in the array
	my $summary = ""; # This will be used to get at the summary data
	my $cmp = '<=>'; # Default to numeric comparison
	my $i = 0;
	foreach my $group_field ( @{$self->group_fields} ) {
	    if ( $group_field->name eq $f ) {
		$index = $i;
		if ( is_numeric_type( $group_field->field->type ) ) {
		    $cmp = "<=>";
		} else {
		    $cmp = "cmp";
		}
		last;
	    }
	    $i++;
	}

	# The sort field wasn't found in the group_fields,
	# look into the operators
	unless (defined $index) {
	    $i = @{$self->group_fields};
	    foreach my $op ( @{$self->ops} ) {
		# Skip aggregators
		next if $op->isa( 'Lire::Aggregator' );
		if ( $op->name eq $f ) {
		    $index = $i;
		    last;
		}
		$i++;
	    }
	}

	# Its wasn't found in one of our ReportOperator children, use
	# the summary value of one of our aggregator children
	unless (defined $index ) {
	    $i = @{$self->group_fields};
	    foreach my $op ( @{$self->ops} ) {
		# Only check aggregators
		next unless $op->isa( 'Lire::Aggregator' );
		if ( $op->is_name_defined($f) ) {
		    $index = $i;
		    $summary = $self->get_summary_value_string( $f );
		    last;
		}
		$i++;
	    }
	}
	croak "impossible to find $f value to sort on"
	  unless defined $index;
	if ( $summary ) {
	    push @sort_ops, 'group_data_value(' . $a ."->" . $schwartzian .
	      "[$index]$summary ) $cmp group_data_value( " . 
		$b ."->" . $schwartzian . "[$index]$summary)";
	} else {
	    push @sort_ops, $a ."->" . $schwartzian ."[$index] $cmp " .
	      $b ."->" . $schwartzian . "[$index]";
	}
    }
    my $sort_code = "sub { " . join( " || ", @sort_ops ) . " }";
    my $sort_func = eval $sort_code;
    croak "error compiling sort comparison ($sort_code): $@" if $@;

    $sort_func;
}

#------------------------------------------------------------------------
# Method init_aggregator_data()
#
# Method required by Lire::AsciiDlf::Aggregator
sub init_aggregator_data {
    my ( $self ) = @_;

    # The group datastructure used to hold the operations' data of the
    # group element is an array. It contains the fields value and the
    # op data: 
    # [ group field, group field, ..., op data, op data, op data ]

    if ( $self->{'use_array'}) {
	# To minimize memory, we operate on sorted input by keeping
	# the key in final sorted order and keeping only the minimum
	# needed
	#
	# Structure of the array: we keep in the first element the current 
	# group key, the second element is the current group element data.
	# After this we have the processed group elements data in sort order
	# and up the the limit attribute.
	# [ current_key, current_group_data, sorted_group_data, sorted_group_data, ... ]
	return [];
    } else {
	# For merging we don't see sorted input and keep related keys
	# using a hash.
	return {}
    }
}

sub item_data2sort_key {
   my $key = [];
   foreach my $f ( @{$_[0]} ) {
       push @$key, group_data_value( $f );
   }
   return $key;
}

#------------------------------------------------------------------------
# Method merge_aggregator_data( $group, $data )
#
# Method required by Lire::AsciiDlf::Aggregator
sub merge_aggregator_data {
    my ( $self, $group, $data ) = @_;

    foreach my $e ( $group->entries ) {
	my $key = join( ";", map { $_->{'value'} } $e->names );
	my $key_data = $data->{$key};
	unless ( $key_data ) {
	    $key_data = $data->{$key} = [];

	    my $i = 0;
	    my @names = $e->names;

	    croak "wrong number of names: expected ", 
	      scalar @{$self->group_fields}, " but found ", scalar @names
		if @names != @{$self->group_fields};

	    foreach my $f ( @{$self->group_fields} ) {
		$key_data->[$i] = $names[$i]{'value'};
		$i++;
	    }

	    foreach my $op ( @{$self->ops} ) {
		$key_data->[$i++] = $op->init_group_data();
	    }
	}

	my $i = @{$self->group_fields};
	foreach my $op ( @{$self->ops} ) {
	    my $value = $e->data_by_name( $op->name );
	    my $op_data = $key_data->[$i++];
	    $op->merge_group_data( $value, $op_data )
	      if ( $value );
	}
    }
}

=pod

=head2 binary_insert

Recursive function that uses a binary search to insert 
$item in $array using $cmp as comparison operator.
$first_idx and $max_idx specify the boundaries of the search.
Search ends when $first_idx == $max_idx or when $sort_key
sorts at or before $first_idx or at or after $max_idx

=cut

sub binary_insert {
    my ( $item, $sort_key, $array, $cmp, $first_idx, $max_idx ) = @_;

    my $mid_idx = int( ($max_idx - $first_idx) / 2) + $first_idx;
    my $sort = $cmp->( $sort_key, item_data2sort_key( $array->[$mid_idx] ) ); 
    if ( ($first_idx == $mid_idx && $sort <= 0)  ||
	 ($max_idx  == $mid_idx && $sort >= 0 )  || 
	 $sort == 0
       ) 
    {
	# Search has ended, insert according to sort order
	if ( $sort < 0 ) {
	    # Sort before mid_idx
	    splice( @$array, $mid_idx, 0, $item);
	} else {
	    # Sort right after mid_idx
	    splice( @$array, $mid_idx + 1, 0, $item);
	}
    } else {
	# Recurse
	if ( $sort < 0 ) {
	    binary_insert( $item, $sort_key, $array, $cmp,
			   $first_idx, $mid_idx - 1 );
	} else {
	    binary_insert( $item, $sort_key, $array, $cmp,
			   $mid_idx + 1, $max_idx );
	}
    }
}

sub sort_current_group_element {
    my ( $self, $data ) = @_;

    # Case where @$data is empty
    return unless @$data;

    my $item   = $data->[1];

    # This data item is ended
    my $i = @{$self->group_fields};
    foreach my $op ( @{$self->ops} ) {
	$op->end_group_data( $item->[$i++] );
    }

    if ( $self->{'sort_cmp'}) {
	my $key = item_data2sort_key( $item );
	my $cmp = $self->{'sort_cmp'};
	if ( @$data == 2 ) {
	    push @$data, $item;

	# Small optimization: check for before or at end of array condition
	} elsif ( $cmp->( $key, item_data2sort_key( $data->[2] ) ) <= 0 ) {
	    splice @$data, 2, 0, $item;
	} elsif ( $cmp->( $key, item_data2sort_key( $data->[$#$data])) >= 0 ) {
	    push @$data, $item;
	} else {
	    binary_insert( $item, $key, $data, $cmp, 2, $#$data );
	}
    } else {
	# Push at end
	push @$data, $item;
    }

    # Keep only limit records
    if ( $self->{'limit_num'} ) {
	my $max_count = $self->{'limit_num'} + 2; # last_key, current_element
	splice @$data, $max_count
	  if $max_count < @$data ;
    }
}

#------------------------------------------------------------------------
# Method end_aggregator_data($data)
#
# Method required by Lire::AsciiDlf::Aggregaror
sub end_aggregator_data {
    my ( $self, $data ) = @_;

    if ( $self->{'use_array'} ) {
	$self->sort_current_group_element( $data );

	# Remove last_key, current_element
	splice @$data,0,2;
    } else {
	foreach my $key ( keys %$data ) {
	    my $item = $data->{$key};
	    my $i = @{$self->group_fields};
	    foreach my $op ( @{$self->ops} ) {
		$op->end_group_data( $item->[$i++] );
	    }
	}

	# Sort the keys according to the sort value
	my @sorted_keys;
	if ( $self->sort_fields ) {
	    my $cmp = $self->{'sort_cmp'};
	    # This uses schwartzian transform
	    @sorted_keys =  map { $_->[0] } sort $cmp
	      map { [ $_, item_data2sort_key( $data->{$_} ) ] } keys %$data;
	} else {
	    @sorted_keys = keys %$data;
	}

	# Keep only limit records
	if ( $self->{'limit_num'} ) {
	    my $limit = $self->{'limit_num'};
	    splice @sorted_keys, $limit
	      if ($limit < @sorted_keys );
	}
	
	# Delete unused keys
	%$data = map { $_ => $data->{$_} } @sorted_keys;
	$data->{'_lr_sorted_keys'} = \@sorted_keys;
    }

    $data;
}

#------------------------------------------------------------------------
# Method create_group_entries( $group, $data)
#
# Method required by Lire::AsciiDlf::Aggregator
sub create_group_entries {
    my ( $self, $group, $data ) = @_;

    $group->show( $self->{'limit_view'} )
      if $self->{'limit_view'};

    # Either to the sorted group data
    # or the sorted keys
    my $array_ref;
    if ( $self->{'use_array'} ) {
	$array_ref = $data;
    } else {
	$array_ref = $data->{'_lr_sorted_keys'};
    }

    my $field_count = @{$self->group_fields};
    foreach my $elmnt ( @$array_ref ) {
	my $item;
	if ( $self->{'use_array'} ) {
	    $item = $elmnt;
	} else {
	    $item = $data->{$elmnt};
	}

	my $entry = $group->create_entry;
	my $i = 0;
	while  ( $i < $field_count ) {
	    $entry->add_name( $item->[$i++] );
	}
	foreach my $op ( @{$self->ops} ) {
	    $op->add_entry_value( $entry, $item->[$i++] );
	}
    }
}

# keep perl happy
1;

__END__

=pod

=head1 VERSION

$Id: Group.pm,v 1.47 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001, 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut

