package Lire::AsciiDlf::Records;

use strict;

use base qw/ Lire::AsciiDlf::Aggregator Lire::Records /;

use Carp;

use Lire::Utils qw/tempfile/;

# Implements Lire::AsciiDlf::Aggregator::init_merge()
sub init_merge {
    my ( $self, $period_start, $period_end ) = @_;

    # We use a temporary file to sort the DLF records according to
    # the group id we are using.
    $self->{'fh'} = tempfile();
    $self->{'group_no'}	= 0;

    return;
}

# Implements Lire::AsciiDlf::Aggregator::init_aggregator_data()
sub init_aggregator_data {
    my ( $self ) = @_;

    my $value = $self->{'group_no'}++;

    return $value;
}

# Implements Lire::AsciiDlf::Aggregator::merge_aggregator_data()
sub merge_aggregator_data {
    my ( $self, $group, $data ) = @_;

    croak "value should be of type Lire::Report::Group, not $group\n"
      unless UNIVERSAL::isa( $group, "Lire::Report::Group" );

    my $fh = $self->{'fh'};
    foreach my $e ( $group->entries() ) {
	my @fields = $e->data;
	print $fh join( " ", $data, map { $_->{'value'}} @fields ), "\n";
    }
}

# Implements Lire::AsciiDlf::Aggregator::end_aggregator_data()
sub end_aggregator_data {
    my ( $self, $data ) = @_;

    my $fh = $self->{'fh'};
    print $fh $data, " LIRE_RECORDS_GROUP_DATA_END\n";
}

# Implements Lire::AsciiDlf::Aggregator::create_group_entries()
sub create_group_entries {
    my ( $self, $group, $data ) = @_;

    seek $self->{'fh'}, 0, 0
      or croak "can't seek at the beginning of the temporary DLF file: $!\n";

    my $dlf_fh = $self->{'fh'};
    my $rx = qr/^$data /;
    while (<$dlf_fh>) {
	# Only keep the records matching the group id
	next unless /$rx/;

	chomp;
	my ( $id, @fields) = split / /, $_;

	# Check for end of group data marker
	last if $fields[0] eq 'LIRE_RECORDS_GROUP_DATA_END';

        my $row = {};
        foreach ( my $i=0; $i < @{$self->{'fields'}}; $i++ ) {
            $row->{ $self->{'fields'}[$i] } = $fields[$i];
        }

	$self->create_entry( $group, $row );
    }
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::AsciiDlf::Records -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Records.pm,v 1.21 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
