package Lire::AsciiDlf::Timegroup;

use strict;

# We need to inherit from AsciiDlf::Aggregator first, otherwise
# perl will look into Lire::Aggregator before than
# Lire::AsciiDlf::Aggregator
use base qw/ Lire::AsciiDlf::Aggregator Lire::Timegroup /;

use Carp;

use Time::Local;
use Time::Timezone;

use Lire::DataTypes qw/ :time /;
use Lire::WeekCalculator;

#------------------------------------------------------------------------
# Method init_merge($period_start, $period_end)
#
# Method required by Lire::AsciiDlf::ReportOperator
sub init_merge {
    my ( $self, $period_start, $period_end ) = @_;

    $self->init_common( $period_start, $period_end );
    $self->SUPER::init_merge( $period_start, $period_end );

    return $self;
}

#------------------------------------------------------------------------
# Method init_common( $time, $end_time )
#
# Method that does initialization common to init_merge() and init_report()
#
# $time = starting end of the report's period
# $end_time = ending end of the report's period
sub init_common {
    my ( $self, $time, $end_time ) = @_;

    unless ( $time ) {
        $self->{'_start_time_na'} = 1;
        return;
    }

    my $period = $self->period;
    if ( $period =~ /^\$/ ) {
	$period = substr $period, 1;
	$period = $self->report_spec->param( $period )->value;
    }

    $self->{'period_sec'} = duration2sec( $period );

    # The way to determine the index in the timeslices is different
    # for weekly, monthly and yearly period
    if ( monthly_duration( $period ) ) {
	$self->{'use_month'} = 1;
    } elsif (weekly_duration( $period ) ) {
	$self->{'use_week'} = 1;
    } elsif ( yearly_duration( $period ) ) {
	$self->{'use_year'} = 1;
    } else {
	$self->{'use_sec'} = 1;
    }

    # Calculate the start of the period
    ($self->{'multiple'})   = $period =~ /^(\d+)/;
    if ( $self->{'use_year'} ) {
	$self->{'year_start'} = (localtime $time)[5];
    } elsif ( $self->{'use_month'} ) {
	my ($month,$year) = (localtime $time)[4,5];
	$self->{'month_start'}  = $month;
	$self->{'year_start'}   = $year;
    } elsif ( $self->{'use_week'} ) {
	$self->{'week_calc'} = new Lire::WeekCalculator;

	my $week_start = $self->{'week_calc'}->week_number( $time );
	my $year_start = (localtime $time)[5];
	my $year_end   = (localtime $end_time)[5];
	$self->{'year_start'} = $year_start;
	$self->{'week_start'} = $week_start;

	# Number of weeks spanned in the first year of DLF data
	my $first_year_wk_cnt = $self->{'week_calc'}->last_week_of_year( $year_start ) - $week_start + 1;

	# We need an extra slot in the case that there are orphaned
	# days in the first year.
	$first_year_wk_cnt++ if $week_start == 0;

	# Array which contains offsets from the start of the data
	# array to the element containing the data for week 1 of the
	# year starting $idx after the first year of data. For
	# example, if data starts at week 30 of a year containing 52
	# weeks and the DLF also contains data for the 5 first week of
	# the successive year. This array will contain [ 0, 23 ]
	# Because week 1 of successsive year is contained in element 23 of the
	# data array.
	$self->{'year_wk_idx_offset'} = [ 0 ];

	# Fill starting date of week 1 for other years
	for ( my $year = $year_start + 1; $year <= $year_end; $year++ ) {
	    my $idx = $year - $year_start;
	    if ( $idx == 1 ) {
		$self->{'year_wk_idx_offset'}[$idx] = $first_year_wk_cnt;
	    } else {
		my $prev_year_weeks =
		  $self->{'week_calc'}->last_week_of_year( $year-1);
		$self->{'year_wk_idx_offset'}[$idx] =
		  $prev_year_weeks + $self->{'year_wk_idx_offset'}[$idx-1];
	    }
	}
    } else {
	# The start of the period is offset by the timezone offset.
	# We need only do this once, because UTC doesn't have a standard
	# vs. daylight saving issue. This means that all subsequent period
	# will have the proper localtime value.
	$self->{'tz_offset'} = tz_local_offset( $time );
	$self->{'start'} = int( $time / $self->{'period_sec'})
	  * $self->{'period_sec'};

	# FIXME: Can somebody explain to me why is this working.
	# This was found heuristically and I'm not sure I
	# understand why this works. -- FJL
	$self->{'start'} -= $self->{'tz_offset'}
	  if abs($self->{'tz_offset'}) < $self->{'period_sec'};
    }


    # FIXME: Backward compatibily hack. This should be done
    # in the stylesheets
    if ( $self->{'use_year'} ) {
	$self->{'time_fmt'} = '%Y';
    } elsif ( $self->{'use_week'} ) {
        $self->{'time_fmt'} = $self->{'week_calc'}->strformat;
    } elsif ( $self->{'period_sec'} >= 86400 ) { # 1d
	$self->{'time_fmt'} = '%Y-%m-%d';
    } elsif ( $self->{'period_sec'} >= 60 ) {
	$self->{'time_fmt'} = '           %H:%M';
    } else {
	$self->{'time_fmt'} = '           %H:%M:%S';
    }
}

#------------------------------------------------------------------------
# Method init_slice_data( $idx )
#
# Fill a data structure for $slice at $idx
sub init_slice_data {
    my ($self, $idx) = @_;

    my $start;
    if ( $self->{'use_sec'} ) {
	$start = $idx * $self->{'period_sec'} + $self->{'start'};
    } elsif ( $self->{'use_month'} ) {
	# Since both idx and $month are zero based we can divide by
	# 12 without fear of going to next year when $month = 11
	my $month_off = $idx * $self->{'multiple'};
	my $year_off = int( ($month_off + $self->{'month_start'}) / 12);
	my $year = $self->{'year_start'} + $year_off;
	my $month;
	if ( $year_off == 0 ) {
	    $month = $self->{'month_start'} + $month_off;
	} else {
	    $month = $month_off - (12 - $self->{'month_start'} ) - ($year_off-1) * 12;
	}
	$start = timelocal( 0, 0, 0, 1, $month, $year );
    } elsif ( $self->{'use_week'} ) {
	my $week_off = $idx * $self->{'multiple'};
	my $year_idx = 0;
	my $week_total;

	# Find the year that contains the $week_off-th week from $week_start
	for ( my $i=1; $i< @{$self->{'year_wk_idx_offset'}}; $i++ ) {
	    $week_total += $self->{'year_wk_idx_offset'}[$i];
	    if ( $week_total > $week_off ) {
		$year_idx = $i;
		last;
	    }
	}
	my $year = $self->{'year_start'} + $year_idx;
	my $week;
	if ( $year_idx == 0 ) {
	    $week = $self->{'week_start'} + $week_off;
	} else {
	    # + 1 because week aren't zero indexed
	    $week = $week_off - $self->{'year_wk_idx_offset'}[$year_idx] + 1;
	}

	$start = $self->{'week_calc'}->week_start( $year, $week );
    } else {
	my $year_off = $idx * $self->{'multiple'};
	$start = timelocal( 0, 0, 0, 1, 0, $self->{'year_start'} + $year_off );
    }
    my $data = [ $start ];

    my $i = 1;
    foreach my $op ( @{$self->ops} ) {
	$data->[$i++] = $op->init_group_data();
    }

    $data;
}

#------------------------------------------------------------------------
# Method find_idx($time)
#
# Returns the index in our array which would contains the data
# for $time.
sub find_idx {
    my ( $self, $time ) = @_;

    my $idx;
    if ( $self->{'use_sec'} ) {
	$idx = int( ($time - $self->{'start'}) / $self->{'period_sec'});
    } elsif ( $self->{'use_month'} ) {
	my ($month,$year) = (localtime $time)[4,5];

	if ( $year > $self->{'year_start'}) {
	    my $year_off = $year - $self->{'year_start'};
	    $idx = $month + ( 11 - $self->{'month_start'} ) + ($year_off-1) * 12;
	} else {
	    $idx = $month - $self->{'month_start'};
	}
	$idx = int( $idx / $self->{'multiple'} );
    } elsif ( $self->{'use_week'} ) {
	my ($month,$year) = (localtime $time)[4,5];

	my $week = $self->{'week_calc'}->week_number( $time );

	my $year_idx = $year - $self->{'year_start'};

	# Increase the year when week 1 appears in December
	$year_idx++ if $week == 1 && $month == 11;
	if ($year_idx > 0 ) {
	    $idx = $week + $self->{'year_wk_idx_offset'}[$year_idx] - 1;
	} else {
	    $idx = $week - $self->{'week_start'};
	}
	$idx = int( $idx / $self->{'multiple'} );
    } else {
	my $year = (localtime $time)[5];
	$idx = $year - $self->{'year_start'};
	$idx = int( $idx / $self->{'multiple'} );
    }

    # Negative time index, means that the $info object didn't report
    # correct start_time.
    croak "Lire::AsciiDlf::Timegroup: find_idx: found negative time " .
      "slice index ($idx). Something is seriously broken!\n" if $idx < 0;

    return $idx;
}

#------------------------------------------------------------------------
# Method init_aggregator_data()
#
# Method required by Lire::AsciiDlf::Aggregator
sub init_aggregator_data {
    return [];
}

#------------------------------------------------------------------------
# Method merge_aggregator_data( $group, $data )
#
# Method required by Lire::AsciiDlf::Aggregator
sub merge_aggregator_data {
    my ( $self, $group, $timeslices ) = @_;

    # INVARIANT: entries are assumed to be sorted on timestamp and the
    # length attribute will be identical across all the entries. This
    # is respected by our create_group_entries() method
    my $first = 1;
    foreach my $e ( $group->entries ) {
	my @names = $e->names;
	die "invalid number of names for a rangegroup subreport: ",
	  scalar @names, "\n"
	    if @names != 1;

	my $time = $names[0]{'value'};

	# Check period compatibility $self->{'start'}
	if ( $first) {
	    my $length = $names[0]{'range'};

	    croak "incompatible merge: source period isn't compatible ",
	      "with new period: source=$length;  target=$self->{'period_sec'}\n"
		if $self->{'period_sec'} < $length ||
		  $self->{'period_sec'} % $length;

	    $first = 0;
	}

	my $idx = $self->find_idx( $time );
	my $data = $timeslices->[$idx];
	$timeslices->[$idx] = $data = $self->init_slice_data( $idx )
	  unless defined $data;

	my $i = 1;
	foreach my $op ( @{$self->ops} ) {
	    my $value = $e->data_by_name( $op->name );
	    my $op_data = $data->[$i++];
	    $op->merge_group_data( $value, $op_data )
	      if ( $value );
	}
    }
    return $self;
}

#------------------------------------------------------------------------
# Method end_group_data($data)
#
# Method required by Lire::AsciiDlf::Aggregator
sub end_aggregator_data {
    my ( $self, $timeslices ) = @_;

    # Finalize each timeslice
    # Either create empty one or call end_group_data on them
    my $last_idx = $#$timeslices;
    my $i = 0;
    while ( $i <= $last_idx) {
	if ( $timeslices->[$i]) {
	    my $data = $timeslices->[$i];
	    my $j = 1;
	    foreach my $op ( @{$self->ops} ) {
		$op->end_group_data( $data->[$j++] );
	    }
	} else {
	    # Create empty set
	    my $start;
	    if ( $self->{'use_sec'} ) {
		$start = $i * $self->{'period_sec'} + $self->{'start'};
	    } elsif ($self->{'use_month'} ) {
		my $year_off  = int( ($self->{'month_start'} + $i) / 12);
		my $month     = ($self->{'month_start'} + $i) % 12;
		$start = timelocal( 0, 0, 0, 1, $month,
				    $self->{'year_start'} + $year_off );
	    } elsif ( $self->{'use_week'} ) {
		my $week1_start = 
		  $self->{'week_calc'}->find_year_week1_start_date( $self->{'year_start'} );
		$start = $week1_start +
		  ($self->{'week_start'} + $i - 1) * 86400 * 7;
	    } else {
		$start = timelocal( 0, 0, 0, 1, 0, $self->{'year_start'} + $i );
	    }
	    my $data = [ $start ];

	    my $j = 1;
	    foreach my $op ( @{$self->ops} ) {
		$data->[$j] = $op->init_group_data();
		$op->end_group_data( $data->[$j++] );
	    }

	    $timeslices->[$i] = $data;
	}
	$i++;
    }

    return $self;
}

#------------------------------------------------------------------------
# Method create_group_entries( $group, $data)
#
# Method required by Lire::AsciiDlf::Aggregator
sub create_group_entries {
    my ( $self, $group, $timeslices ) = @_;

    foreach my $tslice ( @$timeslices ) {
        my $row = { $self->name() => $tslice->[0] };
	my $entry = $self->create_entry( $group, $row );

	my $i = 1;
	foreach my $op ( @{$self->ops} ) {
	    $op->add_entry_value( $entry, $tslice->[$i++] );
	}
    }

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::AsciiDlf::Timegroup - handle lire:timegroup in report specifications

=head1 SYNOPSIS

Lire::AsciiDlf::Timegroup handles lire:timegroup in report specifications.

This is a Lire internal module; therefore, this manpage is written for
Lire hackers, not users.

=head1 DESCRIPTION

=head2 year_wk_idx_offset

year_wk_idx_offset is an array which holds the index of the start of each year
in the Timegroup's array.  For example, year_wk_idx_offset might be something
like [ 0, 23, 75 ] for a two years spanning log files starting somewhere in
july.  First year (the year where log begins) will always be zero; i.e. data
related to the first week of the log will go in the 0 index.  Looking at the
second year, week 1 will go at slot 23.  Week 1 of the third year will go into
slot 75 because there was 52 weeks in that year.  All this is done because the
number of weeks in a year isn't constant (unlike the number of months).  Beware
that week 0 is treated as week 53 of the previous year.


=head1 SUBROUTINES

=head2 &update_group_data

update_group_data takes arguments $self, $dlf and $timeslices.  $self is a
Timegroup object. $dlf is a reference to an array containing the DLF records'
fields, ordered as they are represented in the ascii based DLF file:

 $dlf = [ split / /, $line ];

.  $timeslices is a reference to an array containing $data.  $data is a refence
to an array containing the data of the operators of the children of timeslot.
These children are lire:group, lire:avg, lire:sum, etc.  Timeslot is an
aggregator which can contain other aggregators and should contain at least one
group operator, like lire:group, lire:avg, lire:sum, etc.

See also the Lire::AsciiDlf::Timeslot::update_group_data() code.

Something similar is seen in all other aggregators.

During the Lire process, this routine is run like:

 for each dlf record in dlf file {
   for each report which uses lire:timegroup {
      run update_group_data
   }
 }

=head1 THANKS

Edwin Groothuis, for supplying a patch.

=head1 BUGS

This code should be better documented.

=head1 VERSION

$Id: Timegroup.pm,v 1.56 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001, 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut

